/***************************************************************************

     order.h
     -------------
     Copyright (c) 2010 by Andrianov Pavel <andriano@kappa.cs.karelia.ru>

 ***************************************************************************/
/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,	   *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of	   *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the	   *
 *   GNU General Public License for more details.			   *
 * 									   *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the Free Software	   *
 *   Foundation, Inc., 51 Franklin Street,                                 *
 *   Fifth Floor, Boston, MA  02110-1301, USA.                             *
 *                                                                         *
 ***************************************************************************/
#ifndef ORDER_H
#define ORDER_H

#include <QtCore>
#include <QVector>

#include "good.h"
#include "contractor.h"
#include "deliverytype.h"
#include "sellpoint.h"
#include "paytype.h"
#include "pricetype.h"
#include "contract.h"
#include "sellfirm.h"
#include "../database/mdb.h"
#include "../network/soapclient.h"

#include <QStandardItemModel>

/**
 * @class Order model class
 * @short Order model class
 *
 * Class provides functions and variables to work with
 * information about orders
 *
 * @author Andrianov Pavel, Nikiforov Artem
 */
class Order
{
    /// @struct record of the booked good with its quantity
    struct GoodRecord
    {
        Good* good;
        int quantity;
    };

public:
    // Constructors:

    /// Constructor vith blank fields and id equal NEW_ID
    /**
     * @brief Constructor with blank fields
     * Construct the order and set the id to NEW_ID
     * @note if the id equal NEW_ID, it means that this order doesnt exist in database
     */
    Order();

    /// Convert int to QString
    /**
     * @brief Convert int to QString
     * Creates QString and fills it with int
     * @param integer value
     * @return QString, filled with converted value
     */
    QString toQString(int);

    /// Convert double to QString
    /**
     * @brief Convert double to QString
     * Creates QString and fills it with double
     * @param double, number which should be converted
     * @return QString, filled with the converted value
     */
    QString toQString(double);

    /// Returns if the order is a new order or it is already in database
    /**
     * @brief Return if this order is new
     * Returns if the order is new (do not match any order in database)
     * @return bool is the order new
     * @retval true if it is the new order
     * @retval false order was loaded from database
     */
    bool isNew();

    /// Get all goods as model
    /**
     * @brief Get all booked goods in one model
     * @note Columns: id, name, booked quantity
     * @return model
     */
    QStandardItemModel* getGoods();

    // Datebase and network functions

    /// Loads the information from the database
    /**
     * @brief Create the order using loading from database
     * Load order and its linked information (contractor, goods, etc..) from database
     * @param db pointer to the database module object
     * @param id identifier of order in the database
     * @see Mdb::find()
     * @see Order::addGood()
     */
    Order(Mdb* db, int id);

    /// Loads the information from web-service
    /**
     * @brief Create the order using loading from network
     * Load order and its linked information (contractor, goods, etc..) from network
     * @param connection pointer to the connection
     * @param id identifier of order in the database
     */
    //void get(/*QtSoap connection,*/ int id);

    /// Saves the order information to database
    /**
     * @brief Save all order information to database
     * Save order and information about booked goods to database
     * @param db pointer to the database module object
     */
    void set(Mdb* db);

    ///Send this order to web-service
    void set(soapclient* client);

    // Get and set fields

    /// Returns if the order is valid
    /**
     * @brief Is the order valid
     * Checks the validation of the all fields
     * @retval true if valid
     * @retval false if not valid
     */
    bool isValid();

    /// Add good to order
    /**
     * @brief Book good with order
     * @param good pointer to the model object of Good model class
     * @param quantity quantity of booked goods
     */
    void addGood(Good*, int quantity = 1);

    void setId(int);

    /// Remove quantity of booked good from order
    /**
     * @brief Remove quantity of booked good from order
     * Remove quantity from order, if the result quantity will be lesser than 0, unbooks good
     * @param index index of the record about booked good in Order::Goods
     * @param quantity removing quantity
     */
    void removeGood(int index, int quantity = 1);

    /// Remove good from document
    /**
     * @brief Remove booked good from order
     * Unbook good from order
     * @param index index of the record about booked good in Order::Goods
     */
    void removeGood(int index);

    /// Get the number records about of goods, already booked in document
    /**
     * @brief Get the number of booked records
     * Get the number records about of goods, already booked in document
     * @return number of booked records
     */
    int numberOfGoods() const;

    /// Get the good, booked in this position
    /**
     * @brief Get the good, booked in this position
     * Returns the good, booked in this position
     * @param index position of booked record in the Order::Goods
     * @return good pointer to the model object of Good model class
     */
    Good* goodAt (int);

    /// Get the quantity of good, booked in this position
    /**
     * @brief Get quantity the good, booked in this position
     * Returns the quantity of good, booked in this position
     * @param index position of booked record in the Order::Goods
     * @return quantity of booked good
     */
    int quantityOf(int) const;

    /// Returns the id of order
    /**
     * @brief Returns the id of order
     * @return identificator of order in the database
     */
    int getId();

    /// Returns the Contractor of order
    /**
     * @brief Returns the constractor of order
     * @return object, Constractor of order in the database
     */
    Contractor* getContractor();

    /// Set the contractor of order
    /**
     * @brief Sets contractor to the current order
     * @param Contractor*, contractor, parameter, by which contractor is set
     **/
    void setContractor(Contractor* contractor);

    /// Returns the sell point of order
    /**
     * @brief Returns sell point of current order
     * @return SellPoint*, sell point of current order 
     **/
    SellPoint* getSellPoint();

    /// Sets the sell point of order
    /**
     * @brief Sets sell point to the current order
     * @param SellPoint*, sell_point, parameter, by which sell point is set
     **/
    void setSellPoint(SellPoint* sell_point);

    /// Returns the type of pay
    /**
     * @brief Returns the type of pay of current order
     * @return PayType*, pay type of current order
     **/
    PayType* getPayType();

    /// Sets the type of pay
    /**
     * @brief Sets the type of pay of current order
     * @param PayType*, pay_type, pay type of current order
     **/
    void setPayType(PayType* pay_type);

    /// Returns the type of price
    /**
     * @brief Returns the type price type of current order
     * @return PriceType*, PriceType of current order
     **/
    PriceType* getPriceType();

    /// Set the type of price
    /**
     * @brief Sets the type of price of current order
     * @param PriceType*, price_type, price type of current order
     **/
    void setPriceType(PriceType* price_type);

    /// Returns the contract
    /**
     * @brief Returns the contract of current order
     * @return Contract*, contract of current order
     **/
    Contract* getContract();

    /// Set the contract
    /**
     * @brief Sets the contract of current order
     * @param Contract*,contract,  contract of current order
     **/
    void setContract(Contract* contract);

    /// Return the type of delivery
    /**
     * @brief Returns the type of delivery of current order
     * @return DeliveryType*, delivery type of current order
     **/
    DeliveryType* getDeliveryType();

    /// Set the type of delivery
    /**
     * @brief Sets the type of delivery of current order
     * @param DeliveryType*, delivery_type, delivery type of current order
     **/
    void setDeliveryType(DeliveryType* delivery_type);

    /// @brief Set posted value to true
    /**
     * @brief Set posted value to true
     **/
    void setPosted();

    /// @brief Set posted value to false
    /**
     * @brief Set posted value to false
     **/
    void setUnPosted();

    /**
      * @brief Change the posted value to opposite
      * Changes the value to the opposite
      * and returns the new value of property
      * @return new value of posted property
      */
    bool changePosted();

    /// Returns the discount value
    /**
     * @brief Returns the discount value
     * @return double, discount valude
     **/
    double getDiscount();

    /// Sets the discount value
    /**
     * @brief Sets the discount value
     * @param double, discount,  discount valude
     **/
    void setDiscount(double discount);

    /// Returns the date, when order must be ready
    /**
     * @brief Returns the date, when order must be ready
     * @return QDate*, last date, when order must be ready
     **/
    QDate* getDate();

    /// Sets the date
    /**
     * @brief Sets the date, when order must be ready
     * @param QDate*, date, last date, when order must be ready
     **/
    void setDate(QDate* date);

    /// Returns the date when pay must be ready
    /**
     * @brief Returns the date, when pay must be ready
     * @return QDate*, last date, when pay must be ready
     **/
    QDate* getPayUpTo();

    /// Set the pay date
    /**
     * @brief Sets the date, when pay must be ready
     * @param QDate*, date, last date, when pay must be ready
     **/
    void setPayUpTo(QDate* date);

    /// Returns the selling firm
    /**
     * @brief Returns the selling firm
     * @return SellFirm*, the selling firm
     **/
    SellFirm* getSellFirm();

    /// Set the selling firm
    /**
     * @brief Sets the date, when pay must be ready
     * @param SellFirm*, sell_firm, the selling firm 
     **/
    void setSellFirm(SellFirm* sell_firm);

    /// Returns the comment
    /**
     * @brief Returns the comment
     * @return QString, the comment
     **/
    QString getComment();

    /// Sets the comment
    /**
     * @brief Sets the comment
     * @param QString, comment, the comment
     **/
    void setComment(QString comment);

    /// Returns the aternative address
    /**
     * @brief Returns the alternative address
     * @return QString, the alternative address
     **/
    QString getAltAddress();

    /// Sets the altenative address
    /**
     * @brief Sets the alternative address
     * @param QString, alt_address, the alternative address
     **/
    void setAltAddress(QString alt_address);

    /// Get table name
    /**
     * @brief Get table name
     * Get table name in the database, where orders is located
     * @return name of the table
     */
    QString tableName();

    /// Returns tha date as string in some format
    /**
     * @brief Get date in text format
     * @return date to string
     */
    QString getDateString();

private:
    /// Goods, that was booked in this document
    QVector<GoodRecord> Goods;

    /// Identifier of the document in the mobile database
    int id;

    /// Contractor
    Contractor* contractor;

    /// Identifier of the sell point in database
    SellPoint* sell_point;

    /// Identifier of the type of pay in database
    PayType* pay_type;

    /// Identifier of the type of price in database
    PriceType* price_type;

    /// Identifier of the mode of delivery
    DeliveryType* delivery_type;

    /// Identifier if the document is posted or not
    /**
      * @note In the posted documents booked goods
      * decreases the remains of this good in the stores
      */
    bool post;

    /// Discount for the document
    double discount;

    /// Date when the order/sale must be made
    QDate date;

    /// Identifier of the contract in the database
    Contract* contract;

    /// Identifier of sellfirm
    SellFirm* sell_firm;

    /// Comment from seller to that order document
    QString comment;

    /// Due date
    QDate payupto;

    /// Alternative address where to deviver
    QString alt_address;
};

#endif // ORDER_H
