/***************************************************************************

     mdb.h
     -------------
     Copyright (c)last 5.03.2010 by Zabirokhin Denis <denis.zabirokhin@gmail.com>

 ***************************************************************************/
/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,	   *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of	   *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the	   *
 *   GNU General Public License for more details.			   *
 * 									   *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the Free Software	   *
 *   Foundation, Inc., 51 Franklin Street,                                 *
 *   Fifth Floor, Boston, MA  02110-1301, USA.                             *
 *                                                                         *
 ***************************************************************************/

#ifndef MDB_H
#define MDB_H

#include <QtSql>
#include <QSqlDatabase>
#include <QSqlError>
#include <QSqlQuery>
#include <QSqlQueryModel>
#include <QVector>
#include <QDir>

/**
 * @class MDB module
 * @short Mobile Data Base Module
 *
 * Class provides functions to work with sqlite3 database,
 * located in local.db file
 *
 * @note for the proper work the database local.db must be
 * located in the same directory as the application
 *
 * @author
 */
class Mdb
{

#define NEW_ID -1

public:
    /**
     * @short Default constructor
     * Create a connection to database
     */
    Mdb();

    /**
     * @short Default destructor
     * Close the connection to database
     */
    ~Mdb();

    /**
     * @brief status of DB connection
     * Function view ststus of connection to database
     * @return exit code 0 - good, 1 - error
     */
    int status();

    /**
     * @brief Add data to DB
     * Function runs the query to add data to DB
     * @param name QString parameter, the name of Table
     * @param id integer parameter, id of the field in table
     * @param fields QVector parameter, fields and values of this fileds in Table name
     * @return exit code
     */
    int addToDb(const QString name, const QVector<QString> fields);

    /**
     * @brief Add data to DB, existing field
     * Function runs the query to add data to DB
     * @param name QString parameter, the name of Table
     * @param id integer parameter, id of the field in table
     * @return exit code
     */
    int addToDb(const QString name, const QVector<QString> *fieldValue, const QVector<QString> *fieldName, const long id);

    /**
     * @brief Create table and set fields
     * @param name QString parameter, name of new Table
     * @param fields QVector parameter, fields of this table
     * @return exit code
     */
    int createTable(const QString name, const QVector<QString>fieldName, const QVector<QString> fieldType);

    /**
     * @brief Get the value (max id in the table)+1
     * Function runs the query get the unused id value int the table
     * @param name QString parameter, the name of Table
     * @return id as int
     */
    int getUnusedId(const QString name);

    /**
     * @brief Create table and set fields
     * @param name QString parameter, name of new Table
     * @param fields QVector parameter, fields of this table
     * @return exit code
     */
    int deleteTable(const QString name);

    /**
     * @brief Delete data from table by identificator
     * @param name QString parameter, name of new Table
     * @param id long, identificator of data string in DB
     * @return exit code
     */
    int deleteFromDb(const QString name, const long id);

    //TODO:     
    /**
     * @brief Find data in DB by id and return model with founded fields
     * @param table_name QString parameter, name of Table
     * @param id long, identificator of data string in DB
     * @return QSqlQueryModel*
     */
    QSqlQueryModel* find(const QString table_name, const long id);

    /**
     * @brief Find data in DB by fields and return model with founded fields
     * @param table_name QString parameter, name of Table
     * @param fieldName QVector parameter, name of the fields of this table
     * @parem fieldValue QVector parameter, values of the fields of this table
     * @return QSqlQueryModel*
     */
    QSqlQueryModel* find(const QString table_name, const QVector<QString> *fieldName, const QVector<QString> *fieldValue );

    /**
     * @brief Find data in DB by fields and return model with founded rows with colums set by columnName
     * @param table_name QString parameter, name of Table
     * @param fieldName QVector parameter, name of the fields of this table
     * @param fieldEquality QVector parameter, equality of the field value
     * @note e.g. "like", "=", ">="...
     * @param fieldValue QVector parameter, values of the fields of this table
     * @param columnName QVector parameter, names of needed columns in the result model
     * @note if columnName is empty, result return with all fields in the table
     * @return QSqlQueryModel*
     */
    QSqlQueryModel* find(const QString table_name,
                              const QVector<QString> *fieldName,
                              const QVector<QString> *fieldEquality,
                              const QVector<QString> *fieldValue,
                              const QVector<QString> *columnName);


    /**
     * @brief Table validation
     * Validate existing table
     * @param name QString parameter, name table
     * @return code exit
     */
    int isTblValid(const QString name);

    // TODO:   !!! !!!
    QSqlQueryModel* dirty_hacked_exec(const QString);

    protected:
        int exec(const QString);
        /**
          * Connection to database
          */
        QSqlDatabase db;

};

#endif // MDB_H
