/*
 *
 *  Copyright (c) 2010 Zagaia (INdT - Instituto Nokia de Tecnologia/
 *       FUCAPI - Fundação Centro de Análise, Pesquisa e Inovação Tecnológica)
 *
 *  This file is part of TweeGo.
 *
 *  TweeGo is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  TweeGo is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with TweeGo. If not, see <http://www.gnu.org/licenses/>
 *
 */

#include "timelineview.h"

/*!
 * \brief TimelineView constructor.
 */
TimelineView::TimelineView(void)
{
    qDebug() << "+ " << __PRETTY_FUNCTION__;

    _exitAnim = 0;
    _urlShortener = new UrlShortener;
    _ipLocation = new IpLocation;
    m_homeTimeline = new StatusList;
    m_mentionsTimeline = new StatusList;

    m_animating = false;
    m_home = true;
    m_mentions = false;
//    m_dm = false;

    view2 = new QGraphicsView;

    _gpsLocation = new Maemo5GPS;
}

/*!
 * \brief TimelineView destructor.
 */
TimelineView::~TimelineView()
{
    if(this->_topBar)
        delete this->_topBar;
    if(this->m_homeTimeline)
        delete this->m_homeTimeline;
    if(this->m_mentionsTimeline)
        delete this->m_mentionsTimeline;
    if(this->_urlShortener)
        delete this->_urlShortener;
    if(this->_ipLocation)
        delete this->_ipLocation;
    if(view2)
        delete view2;

    if(this->_gpsLocation)
        delete this->_gpsLocation;
}

/*!
 * \brief Creates the TimelineView objects and animations.
 */
void TimelineView::createUi(void)
{
    qDebug() << "+ " << __PRETTY_FUNCTION__;


    _createEntry();
    _createButtons();
    _createTopBar();

    startLoader();

    /* Animations */
    _entryAnim = _createEntryAnimation();
    _entryAnim->start();
    _exitAnim = _createExitAnimation();

    connect(qobject_cast<QObject *>(_exitAnim), SIGNAL(finished()), this,
            SLOT(_exitAnimationFinished()));
    connect(_updateButton, SIGNAL(released()), this, SLOT(_emitPostUpdate()));

    connect(_urlShortener, SIGNAL(newString(QString)), this, SLOT(_handleNewMessage(QString)));
    connect(_logoutButton, SIGNAL(released()), this, SIGNAL(logout()));
    connect(_entryUpdate, SIGNAL(enterPressed()), this, SLOT(_emitPostUpdate()));
    connect(_exposeButton, SIGNAL(released()), this, SLOT(_emitExposeSignal()));
    connect(_topBar, SIGNAL(finished()), this, SLOT(_createTimeline()));
    connect(_topBar, SIGNAL(changeAvatar(QString)), this, SLOT(_emitChangeAvatar(QString)));
    connect(_locationCheckBox, SIGNAL(pressed()), this, SIGNAL(showLocationDialog()));
    connect(_ipLocation, SIGNAL(finished(double, double)), this, SLOT(_showIpLocationInfo(double, double)));

    connect(_homeButton, SIGNAL(released()), this, SLOT(_animateHome()));
    connect(_mentionsButton, SIGNAL(released()), this, SLOT(_animateMentions()));
//    connect(_dmButton, SIGNAL(released()), this, SLOT(_animateDm()));

    connect(_backButton, SIGNAL(released()), this, SLOT(changeToTimeline()));

    connect(m_homeTimeline, SIGNAL(morePressed()), this, SLOT(_moreButtonHome()));
    connect(m_homeTimeline, SIGNAL(createTweetInfo(Tweet*)), this, SLOT(_createTweetInfo(Tweet*)));

    connect(m_mentionsTimeline, SIGNAL(morePressed()), this, SLOT(_moreButtonMentions()));
    connect(m_mentionsTimeline, SIGNAL(createTweetInfo(Tweet*)), this, SLOT(_createTweetInfo(Tweet*)));

}

/*!
 * \brief Creates the timeline.
 */
void TimelineView::_createTimeline()
{
    qDebug() << "+ " << __PRETTY_FUNCTION__;

    emit addHomeTimelineToScene(m_homeTimeline);
    m_atHomeTimeline = true;
}

/*!
 * \brief This slot is called when the a transition back to the login screen is required.
 * This method calls all initial animations backwards.
 */
void TimelineView::_exitAnimationFinished(void)
{
    qDebug() << "+ " << __PRETTY_FUNCTION__;
    emit done();
}

/*!
 * \brief This method creates the entry animation for all objects on the screen.
 * \return anim Entry animation.
 */
QAbstractAnimation *TimelineView::_createEntryAnimation(void)
{
    qDebug() << "+ " << __PRETTY_FUNCTION__;
    QParallelAnimationGroup *anim = new QParallelAnimationGroup(this);

    QEasingCurve easingCurve;
    easingCurve.setType(QEasingCurve::OutBounce);

    /* Buttons */
    _addOpacityAnim(anim, _updateButton, 0);
    _addPosAnim(anim, _updateButton, QPointF(800 - _updateButton->boundingRect().width(), 480 - (_updateButton->boundingRect().height()) - 50),
                QPointF(800 - _updateButton->boundingRect().width(), 480 - (_updateButton->boundingRect().height())),
                easingCurve, 0);

    _addOpacityAnim(anim, _exposeButton, 0);
    _addPosAnim(anim, _exposeButton, QPointF(0, 0 + 50),
                QPointF(0, 0),
                easingCurve, 0);

    _addOpacityAnim(anim, _logoutButton, 0);
    _addPosAnim(anim, _logoutButton, QPointF(800 - _logoutButton->boundingRect().width(), 0 + 50),
                QPointF(800 - _logoutButton->boundingRect().width(), 0),
                easingCurve, 0);

    _addOpacityAnim(anim, _homeButton, 0);
    _addPosAnim(anim, _homeButton, QPointF(_exposeButton->boundingRect().width() + 8 + 49, 0 + 50),
                QPointF(_exposeButton->boundingRect().width() + 8 + 49, 2), easingCurve, 0);
    _homeButton->setTransformOriginPoint(_homeButton->boundingRect().center());
    _homeButton->setScale(1.065);

    _addPosAnim(anim, _mentionsButton, QPointF(_exposeButton->boundingRect().width() + 8 + _homeButton->boundingRect().width() + 6 + 49, 0 + 50),
                QPointF(_exposeButton->boundingRect().width() + 8 + _homeButton->boundingRect().width() + 6 + 49, 2), easingCurve, 0);
    _mentionsButton->setOpacity(0.5);

//    _addPosAnim(anim, _dmButton, QPointF(_exposeButton->boundingRect().width() + 8 + _homeButton->boundingRect().width() + 6 +
//                                         _dmButton->boundingRect().width() + 6, 0 + 50),
//               QPointF(_exposeButton->boundingRect().width() + 8 + _homeButton->boundingRect().width() + 6 +
//                       _dmButton->boundingRect().width() + 6, 2), easingCurve, 0);
//    _dmButton->setOpacity(0.5);

    /* TopBar */
    _addOpacityAnim(anim, _topBar, 0);
    _addPosAnim(anim, _topBar, QPointF(800 - _logoutButton->boundingRect().width() - _topBar->boundingRect().width(), 0 + 50),
                QPointF(800 - _logoutButton->boundingRect().width() - _topBar->boundingRect().width(), 0),
                easingCurve, 0);

     /* Update Entry */
    _addOpacityAnim(anim, _entryUpdate, 0);
    _addPosAnim(anim, _entryUpdate, QPointF(0, 480 - _entryUpdate->boundingRect().height() - 50),
                QPointF(0, 480 - _entryUpdate->boundingRect().height()),easingCurve, 0);

    /* Location CheckBox */
    _addOpacityAnim(anim, _locationCheckBox, 0);
    _addPosAnim(anim, _locationCheckBox, QPointF(_entryUpdate->boundingRect().width() ,480 - (_locationCheckBox->boundingRect().height()) - 50),
                QPointF(_entryUpdate->boundingRect().width() ,480 - (_locationCheckBox->boundingRect().height())),
                easingCurve, 0);

    return anim;
}

/*!
 * \brief This method creates the exit animation for all objects on the screen.
 * \return anim Exit animation.
 * This method gets the entry animation and sets it backwards as the exit animation.
 */
QAbstractAnimation *TimelineView::_createExitAnimation(void)
{
    qDebug() << "+ " << __PRETTY_FUNCTION__;
    QParallelAnimationGroup *anim = qobject_cast<QParallelAnimationGroup *>(_createEntryAnimation());

    anim->setDirection(QAbstractAnimation::Backward);
    return anim;
}

/*!
 * \brief Creates the top bar containing user information.
 */
void TimelineView::_createTopBar()
{
    qDebug() << "+ " << __PRETTY_FUNCTION__;
    _topBar = new TwitterTopBar(this);
}

/*!
 * \brief Creates the buttons on the screen.
 */
void TimelineView::_createButtons()
{
    qDebug() << "+ " << __PRETTY_FUNCTION__;

    _updateButton = new TwitterButton(TwitterButton::Update,"Tweet",this);
    _updateButton->setOpacity(0);

    _exposeButton = new TwitterButton(TwitterButton::Expose, NULL, this);
    _exposeButton->setOpacity(0);
    _exposeButton->setZValue(3.0);

    _logoutButton = new TwitterButton(TwitterButton::Logout, "Logout", this);
    _logoutButton->setOpacity(0);

    _backButton = new TwitterButton(TwitterButton::Back, NULL, this);
    _backButton->setOpacity(1.0);
    _backButton->setPos(QPointF(810, 0));
    _backButton->setZValue(2.0);

    _locationCheckBox = new TwitterCheckBox(TwitterCheckBox::Location,
    QPointF(_entryUpdate->boundingRect().width(), 480 - _updateButton->boundingRect().height()), false, this);
    _locationCheckBox->setOpacity(1);

    _homeButton = new TwitterButton(TwitterButton::Tab, "Home", this);
    _homeButton->setOpacity(0);

    _mentionsButton = new TwitterButton(TwitterButton::Tab, "@", this);
    _mentionsButton->setOpacity(0);

//    _dmButton = new TwitterButton(TwitterButton::Tab, "DM", this);
//    _dmButton->setOpacity(0);

    _tabShadow = new QGraphicsPixmapItem(QPixmap(":images/resources/tab-shadow.png"), this);
    _tabShadow->setOpacity(1);
    _tabShadow->setPos(QPointF(_exposeButton->boundingRect().width() + 1 + 49,
                               _exposeButton->boundingRect().height() - 8));

    homePos = QPointF(_exposeButton->boundingRect().width() + 8 + 49, 2);
    mentionsPos = QPointF(_exposeButton->boundingRect().width() + 8 + _homeButton->boundingRect().width() + 6 + 49, 2);

//    dmPos = QPointF(_exposeButton->boundingRect().width() + 8 + _homeButton->boundingRect().width() + 6 +
//                    _dmButton->boundingRect().width() + 6, 2);
}

/*!
 * \brief Creates the entry to write a new status message.
 */
void TimelineView::_createEntry()
{
     qDebug() << "+ " << __PRETTY_FUNCTION__;
    /* Add the entries to the scene */
    // ID entry
    _entryUpdate = new LineEdit(LineEdit::Long,this);
    _entryUpdate->setOpacity(0);
    _entryUpdate->setPos(QPointF(100,435));
}

/*!
 * \brief Creates tweetinfo object.
 */
void TimelineView::_createTweetInfo(Tweet *tweet)
{
    qDebug() << "+ " << __PRETTY_FUNCTION__;
    _tweetInfo = new TwitterTweetInfo(tweet, m_atHomeTimeline, this);
    _tweetInfo->setZValue(-1.0);

    connect(_tweetInfo, SIGNAL(retweet(QString)), this, SLOT(_handleNewMessage(QString)));
    connect(_tweetInfo, SIGNAL(reply(QString, QString)), this, SLOT(_handleReply(QString, QString)));
    connect(_tweetInfo, SIGNAL(favorite(QString, bool)), this, SLOT(_emitFavorite(QString, bool)));
    connect(_tweetInfo, SIGNAL(del(QString, int)), this, SLOT(_emitDelete(QString, int)));
    connect(_tweetInfo, SIGNAL(unfollow(QString, bool)), this, SLOT(_emitUnfollow(QString, bool)));
    connect(_tweetInfo, SIGNAL(twitpic()), this, SLOT(_openTwitPic()));
    connect(this, SIGNAL(loadMap()), _tweetInfo, SLOT(_loadMap()));

    this->_changeToTweetInfo();
}

/*!
 * \brief Adds opacity animation.
 * \param item the item that will be animated.
 * \param anim the item's animation.
 * \param delay to start each item's animation.
 */
void TimelineView::_addOpacityAnim(QParallelAnimationGroup *anim, TwitterApplicationItem *item, int delay)
{
    QPropertyAnimation *animation = new QPropertyAnimation();

    animation->setTargetObject(item);
    animation->setPropertyName("opacity");
    animation->setStartValue(0.0);
    animation->setEndValue(1.0);
    animation->setDuration(500);
    animation->setEasingCurve(QEasingCurve::Linear);

    if (delay > 0) {
        QSequentialAnimationGroup *delayedAnimation;
        delayedAnimation = new QSequentialAnimationGroup();
        delayedAnimation->addPause(delay);
        delayedAnimation->addAnimation(animation);
        anim->addAnimation(delayedAnimation);
    } else {
        anim->addAnimation(animation);
    }
}

/*!
 * \brief Adds the position animation.
 * \param item the item that will be animated
 * \param anim the item's animation
 * \param delay to start each item's animation
 * \param startValue the start position of the animation
 * \param endValue the end position of the animation
 * \param easing the easing curve for the movimentation animation
 */
void TimelineView::_addPosAnim(QParallelAnimationGroup *anim,
                            TwitterApplicationItem *item,
                            QPointF startValue, QPointF endValue,
                            QEasingCurve easing, int delay)
{
    QPropertyAnimation *animation = new QPropertyAnimation();

    animation->setTargetObject(item);
    animation->setPropertyName("pos");
    animation->setStartValue(startValue);
    animation->setEndValue(endValue);
    animation->setDuration(500);
    animation->setEasingCurve(easing);

    if (delay > 0) {
        QSequentialAnimationGroup *delayedAnimation;
        delayedAnimation = new QSequentialAnimationGroup();
        delayedAnimation->addPause(delay);
        delayedAnimation->addAnimation(animation);
        anim->addAnimation(delayedAnimation);
    } else {
        anim->addAnimation(animation);
    }
}

/*!
 * \brief Adds the position animation.
 * \param item the item that will be animated
 * \param anim the item's animation
 * \param delay to start each item's animation
 * \param startValue the start position of the animation
 * \param endValue the end position of the animation
 * \param easing the easing curve for the movimentation animation
 */
void TimelineView::_addPosAnim(QParallelAnimationGroup *anim,
                            QGraphicsView *view,
                            QPointF startValue, QPointF endValue,
                            QEasingCurve easing, int delay)
{
    QPropertyAnimation *animation = new QPropertyAnimation();

    animation->setTargetObject(view);
    animation->setPropertyName("pos");
    animation->setStartValue(startValue);
    animation->setEndValue(endValue);
    animation->setDuration(500);
    animation->setEasingCurve(easing);

    if (delay > 0) {
        QSequentialAnimationGroup *delayedAnimation;
        delayedAnimation = new QSequentialAnimationGroup();
        delayedAnimation->addPause(delay);
        delayedAnimation->addAnimation(animation);
        anim->addAnimation(delayedAnimation);
    } else {
        anim->addAnimation(animation);
    }
}


/*! \brief Gets the message typed on message edit box.
 *  \return message Message typed.
 */
QString TimelineView::getUpdateMessage(void)
{
    QWidget *widget;

    widget = _entryUpdate->getEntry()->widget();
    return (dynamic_cast<QLineEdit*>(widget))->text();
}

/*! \brief Sets the message in the message edit box.
 *  \param[in] message The user's message.
 */
void TimelineView::setUpdateMessage(const QString &message)
{
    _entryUpdate->setMessage(message);
}

/*!
 * \brief This slot is called when the user pushes the "update" button.
 * This method starts the loader then sends a signal to the controller to try to login.
 */
void TimelineView::_emitPostUpdate()
{
    qDebug() << "+ " << __PRETTY_FUNCTION__;

    if(_entryUpdate->getTextLength() > 140) {
        if( this->containsUrl(this->getUpdateMessage()) > 0 ) {
            _urlShortener->parseUrl(this->getUpdateMessage());
            this->_updateButton->enable(false);
        }
    }
    else {
        startLoader();
        emit postUpdate();
    }
}

/*!
 * \brief Sets the new message on the textbox after the url was shortened.
 * \param newMessage message
 */
void TimelineView::_handleNewMessage(QString newMessage)
{
    this->setUpdateMessage(newMessage);
    this->_updateButton->enable(true);
}

/*! \brief Calls the loader when user clicks on update.
 */
void TimelineView::startLoader()
{
    qDebug() << "+ " << __PRETTY_FUNCTION__;
    _loader = new Loader(this,QPointF(400,435));
    connect(_loader, SIGNAL(finished()), this, SLOT(_deleteLoader()));

    _loader->setZValue(1.0);
    QGraphicsRectItem *bg = _loader->getBackground();
    bg->setOpacity(0.15);

    setEnabled(false);

    _exposeButton->enable(true);
    _backButton->enable(true);
    _loader->startEntryAnimation();
}

/*! \brief Deletes loader object when loader finishes.
 */
void TimelineView::_deleteLoader()
{
    qDebug() << "+ " << __PRETTY_FUNCTION__;
    delete _loader;
    _loader = NULL;
    this->setEnabled(true);
}

/*!
 * \brief This slot is called when there is any problem with the connection.
 * This slot starts the animation for the transition to the login screen.
 */
void TimelineView::connectionError(void)
{
    qDebug() << "+ " << __PRETTY_FUNCTION__;

    _loader->startExitAnimation();
    if (_exitAnim) {
        _exitAnim->start();
    } else {
        emit done();
    }
}

/*!
 * \brief Calls generic error on timelinectrl when an error while shortening the url occurs.
 * \param errorMessage Error message
 */
void TimelineView::urlShorteningError(QString errorMessage)
{
    qDebug() << "+ " << __PRETTY_FUNCTION__;
    emit shorteningError(errorMessage);
}

/*! \brief This slot is called to start loader's exit animation.
 */
void TimelineView::endLoader()
{
    if(_loader)
        _loader->startExitAnimation();
}

/*!
 * \brief Updates the topbar with new information about the user.
 */
void TimelineView::updateTopBar(void)
{
    qDebug() << "+ " << __PRETTY_FUNCTION__;

    _topBar->updateTopBar(false);
//    _topBar->setPos(QPointF(800 - _logoutButton->boundingRect().width() - _topBar->boundingRect().width() -8, 0));
}

/*!
 * \brief Updates Parses the message checking if it has any url.
 * \param message message
 * \return cont Number of urls.
 */
int TimelineView::containsUrl(QString message)
{
    QList<QString> temp = message.split(" ");

        QListIterator<QString> iteratorUrl(temp);
    int cont = 0;
    while (iteratorUrl.hasNext()) {
        QString nextWord = iteratorUrl.next();

        if ( nextWord.contains( QRegExp("http://*") ) ){
            cont++;
        }
    }
    return cont;
}

/*!
 * \brief Starts exit animation to go back to login screen.
 */
void TimelineView::_backToLogin()
{
    if (_exitAnim) {
        _exitAnim->start();
    } else {
        emit done();
    }
}

/*!
 * \brief Emits a DBus signal to Hildon Desktop who manages the app window with.
 * the task swticher when the expose button is pressed.
 */
void TimelineView::_emitExposeSignal()
{
    QDBusConnection bus = QDBusConnection::sessionBus();
    QDBusMessage message = QDBusMessage::createSignal("/", "com.nokia.hildon_desktop", "exit_app_view");
    bus.send(message);
}

/*!
 * \brief Emits an Unfollow signal.
 */
void TimelineView::_emitUnfollow(QString username, bool _unfollow)
{
    startLoader();
    emit unfollow(username, _unfollow);
}

/*!
 * \brief Emits a delete signal.
 */
void TimelineView::_emitDelete(QString tweetId, int posInList)
{
    startLoader();
    emit deleteTweet(tweetId, posInList);
}

/*!
 * \brief Emits a favorite signal.
 * \param isFavorite true if status is favorite and false otherwise.
 */
void TimelineView::_emitFavorite(QString tweetId, bool isFavorite)
{
    startLoader();
    emit favorite(tweetId, isFavorite);
}

/*!
 * \brief Gets geographical goordinates.
 */
void TimelineView::getIPCoordinates()
{
    qDebug() << "+ " << __PRETTY_FUNCTION__;

    QMaemo5InformationBox::information(NULL, tr("Acquiring geographical information through your IP"), QMaemo5InformationBox::DefaultTimeout);

    _ipLocation->getLocation();
}

/*!
 * \brief Shows geographical goordinates.
 */
void TimelineView::_showIpLocationInfo(double latitude, double longitude)
{
    qDebug() << "+ " << __PRETTY_FUNCTION__;

    QMaemo5InformationBox::information(NULL, tr("GOT IT! Latitude = %1, Longitude = %2").arg(latitude).arg(longitude),
                                           QMaemo5InformationBox::DefaultTimeout);
}

/*!
 * \brief Emits a signal to change user's avatar.
 * \param image New image to upload.
 */
void TimelineView::_emitChangeAvatar(QString image)
{
    startLoader();
    emit changeAvatar(image);
}
/*!
 * \brief Starts GPS with ANGSS method
 */
void TimelineView::getGPSCoordinates()
{
    qDebug() << "+ " << __PRETTY_FUNCTION__;

    _gpsLocation->get_agnss();
    QMaemo5InformationBox::information(NULL, tr("Acquiring GPS information"), QMaemo5InformationBox::DefaultTimeout);
    connect(_gpsLocation, SIGNAL(getOK_agnss()), this, SIGNAL(gotGPS()));
}

/*!
 * \brief Unchecks location checkbox
 */
void TimelineView::setLocationCheckboxFalse()
{
    _locationCheckBox->setChecked(false);
}

void TimelineView::_animateHome()
{
    qDebug() << "+ " << __PRETTY_FUNCTION__;

    startLoader();

    if(!m_animating) {
        if(m_mentions) {
            _scaleOutAnim(_mentionsButton);
            _opacityOutAnim(_homeButton);

            _scaleInAnim(_homeButton);
            _opacityInAnim(_mentionsButton);

            emit removeMentionsTimelineFromScene(m_mentionsTimeline);
            emit addHomeTimelineToScene(m_homeTimeline);
            m_atHomeTimeline = true;
        }
//        else
//            if(m_dm) {
//                _scaleOutAnim(_dmButton);
//                _opacityOutAnim(_homeButton);
//
//                _scaleInAnim(_homeButton);
//                _opacityInAnim(_dmButton);
//
////                emit removeDMTimelineFromScene(m_dmTimeline);
//                emit addHomeTimelineToScene(m_homeTimeline);
//                m_atHomeTimeline = true;
//            }
        else
            if(m_home) {
                emit removeHomeTimelineFromScene(m_homeTimeline);
                emit addHomeTimelineToScene(m_homeTimeline);
            }

        m_home = true;
        m_mentions = false;
//        m_dm = false;
    }
}

void TimelineView::_animateMentions()
{
    qDebug() << "+ " << __PRETTY_FUNCTION__;

    startLoader();

    if(!m_animating) {
        if(m_home) {
            _scaleOutAnim(_homeButton);
            _opacityOutAnim(_mentionsButton);

            _scaleInAnim(_mentionsButton);
            _opacityInAnim(_homeButton);

            emit removeHomeTimelineFromScene(m_homeTimeline);
            emit addMentionsTimelineToScene(m_mentionsTimeline);
            m_atHomeTimeline = false;
        }
//        else
//            if(m_dm) {
//                _scaleOutAnim(_dmButton);
//                _opacityOutAnim(_mentionsButton);
//
//                _scaleInAnim(_mentionsButton);
//                _opacityInAnim(_dmButton);
//
////                emit removeDMTimelineFromScene(m_dmTimeline);
//                emit addMentionsTimelineToScene(m_mentionsTimeline);
//                m_atHomeTimeline = false;
//            }
        else
            if(m_mentions) {
                emit removeMentionsTimelineFromScene(m_mentionsTimeline);
                emit addMentionsTimelineToScene(m_mentionsTimeline);
            }

        m_home = false;
        m_mentions = true;
//        m_dm = false;
    }
}

//void TimelineView::_animateDm()
//{
//    qDebug() << "+ " << __PRETTY_FUNCTION__;
//
//    if(!m_animating) {
//        if(m_home) {
//            _scaleOutAnim(_homeButton);
//            _opacityOutAnim(_dmButton);
//
//            _scaleInAnim(_dmButton);
//            _opacityInAnim(_homeButton);
//
//            emit removeHomeTimelineFromScene(m_homeTimeline);
////            emit addDMTimelineToScene(m_dmTimeline);
//            m_atHomeTimeline = false;
//        }
//        else {
//            if(m_mentions) {
//                _scaleOutAnim(_mentionsButton);
//                _opacityOutAnim(_dmButton);
//
//                _scaleInAnim(_dmButton);
//                _opacityInAnim(_mentionsButton);
//
//                emit removeMentionsTimelineFromScene(m_mentionsTimeline);
////                emit addDMTimelineToScene(m_dmTimeline);
//                m_atHomeTimeline = false;
//            }
//        }
//
//        m_home = false;
//        m_mentions = false;
//        m_dm = true;
//    }
//}

void TimelineView::_animationFinished()
{
    qDebug() << "+ " << __PRETTY_FUNCTION__;

    m_animating = false;
}

void TimelineView::_scaleInAnim(TwitterApplicationItem *item)
{
    qDebug() << "+ " << __PRETTY_FUNCTION__;

    QPropertyAnimation *animate = new QPropertyAnimation(item, "scale");
    connect(animate, SIGNAL(finished()), this, SLOT(_animationFinished()));
    item->setTransformOriginPoint(item->boundingRect().center());
    animate->setStartValue(1);
    animate->setEndValue(1.065);
    animate->setDuration(500);
    animate->setEasingCurve(QEasingCurve::OutExpo);
    animate->start(QPropertyAnimation::DeleteWhenStopped);
    m_animating = true;
}

void TimelineView::_scaleOutAnim(TwitterApplicationItem *item)
{
    qDebug() << "+ " << __PRETTY_FUNCTION__;

    QPropertyAnimation *animate = new QPropertyAnimation(item, "scale");
    connect(animate, SIGNAL(finished()), this, SLOT(_animationFinished()));
    item->setTransformOriginPoint(item->boundingRect().center());
    animate->setStartValue(1.065);
    animate->setEndValue(1);
    animate->setDuration(500);
    animate->setEasingCurve(QEasingCurve::OutExpo);
    animate->start(QPropertyAnimation::DeleteWhenStopped);
    m_animating = true;
}

void TimelineView::_opacityInAnim(TwitterApplicationItem *item)
{
    qDebug() << "+ " << __PRETTY_FUNCTION__;

    QPropertyAnimation *animate = new QPropertyAnimation(item, "opacity");
    connect(animate, SIGNAL(finished()), this, SLOT(_animationFinished()));
    animate->setStartValue(1);
    animate->setEndValue(0.5);
    animate->setDuration(500);
    animate->setEasingCurve(QEasingCurve::OutExpo);
    animate->start(QPropertyAnimation::DeleteWhenStopped);
    m_animating = true;
}

void TimelineView::_opacityOutAnim(TwitterApplicationItem *item)
{
    qDebug() << "+ " << __PRETTY_FUNCTION__;

    QPropertyAnimation *animate = new QPropertyAnimation(item, "opacity");
    connect(animate, SIGNAL(finished()), this, SLOT(_animationFinished()));
    animate->setStartValue(0.5);
    animate->setEndValue(1);
    animate->setDuration(500);
    animate->setEasingCurve(QEasingCurve::OutExpo);
    animate->start(QPropertyAnimation::DeleteWhenStopped);
    m_animating = true;
}

void TimelineView::_changeToTweetInfo()
{
    QEasingCurve curve(QEasingCurve::OutCirc);
    QEasingCurve topItemsCurve(QEasingCurve::InCirc);
    curve.setOvershoot(2.5);

    QParallelAnimationGroup *animationGroup = new QParallelAnimationGroup(this);

    // Top items
    _addPosAnim(animationGroup, _homeButton, homePos,
                QPointF(_homeButton->x() - 800, _homeButton->y()), topItemsCurve, 400);
    _addPosAnim(animationGroup, _mentionsButton, mentionsPos,
                QPointF(_mentionsButton->x() - 800, _mentionsButton->y()), topItemsCurve, 200);
//    _addPosAnim(animationGroup, _dmButton, dmPos,
//                QPointF(_dmButton->x() - 800, _dmButton->y()), topItemsCurve, 0);

    // Logout Button
    _addPosAnim(animationGroup, _logoutButton, _logoutButton->pos(),
                QPointF(810, 0), curve, 0);

    // Back Button
    _addPosAnim(animationGroup, _backButton, _backButton->pos(),
                QPointF(800 - _backButton->boundingRect().width(), 0), curve, 0);

    //Timeline
    _addPosAnim(animationGroup, view2, QPointF(0, _logoutButton->boundingRect().height() + 4),
                QPointF(-800, _logoutButton->boundingRect().height() + 4), curve, 0);

    // TweetInfo
    _addPosAnim(animationGroup, _tweetInfo, QPointF(810, _exposeButton->boundingRect().height() + 4),
                QPointF(2, _exposeButton->boundingRect().height() + 4), curve, 0);

    animationGroup->start(QPropertyAnimation::DeleteWhenStopped);

    connect(animationGroup, SIGNAL(finished()), this, SIGNAL(loadMap()));
}

void TimelineView::changeToTimeline()
{
    QEasingCurve curve(QEasingCurve::OutCirc);
    curve.setOvershoot(2.5);

    QParallelAnimationGroup *animationGroup = new QParallelAnimationGroup(this);

    // Top items
    _addPosAnim(animationGroup, _homeButton,  QPointF(_homeButton->x() - 800, _homeButton->y()),
                homePos, curve, 0);
    _addPosAnim(animationGroup, _mentionsButton, QPointF(_mentionsButton->x() - 800, _mentionsButton->y()),
                mentionsPos, curve, 200);
//    _addPosAnim(animationGroup, _dmButton, QPointF(_dmButton->x() - 800, _dmButton->y()),
//                dmPos, curve, 400);

    // Logout Button
    _addPosAnim(animationGroup, _logoutButton, _logoutButton->pos(),
                QPointF(800 - _logoutButton->boundingRect().width(), 0), curve, 0);

    // Back Button
    _addPosAnim(animationGroup, _backButton, _backButton->pos(),
                QPointF(810, 0), curve, 0);

    //Timeline
    _addPosAnim(animationGroup, view2, QPointF(-800, _logoutButton->boundingRect().height() + 4),
                QPointF(0, _logoutButton->boundingRect().height() + 4), curve, 0);

    // TweetInfo
    _addPosAnim(animationGroup, _tweetInfo, QPointF(2, _exposeButton->boundingRect().height() + 4),
                QPointF(810, _exposeButton->boundingRect().height() + 4), curve, 0);

    animationGroup->start(QPropertyAnimation::DeleteWhenStopped);

    connect(animationGroup, SIGNAL(finished()), _tweetInfo, SLOT(deleteLater()));
}

void TimelineView::_moreButtonHome()
{
    startLoader();
    emit moreButtonHome(m_homeTimeline);
}

void TimelineView::_moreButtonMentions()
{
    startLoader();
    emit moreButtonMentions(m_mentionsTimeline);
}

/*!
 * \brief Sets the new message on the textbox after the user presses reply button.
 * \param newMessage message
 */
void TimelineView::_handleReply(QString newMessage, QString statusId)
{
    this->setUpdateMessage(newMessage);
    this->_updateButton->enable(true);

    isReply = true;
    m_statusId = statusId;
}

void TimelineView::_openTwitPic()
{
    m_twitPic = new TwitPic(this);
    connect(m_twitPic, SIGNAL(animExit()), this, SLOT(_closeTwitPic()));
    m_twitPic->setZValue(2.5);
    m_twitPic->download(_tweetInfo->getTwitPicUrl() + "/full");

    QPropertyAnimation *animate = new QPropertyAnimation(m_twitPic, "scale");
    m_twitPic->setTransformOriginPoint(m_twitPic->boundingRect().center());
    animate->setStartValue(0);
    animate->setEndValue(1);
    animate->setDuration(700);
    animate->setEasingCurve(QEasingCurve::OutExpo);
    animate->start(QPropertyAnimation::DeleteWhenStopped);
}

void TimelineView::_closeTwitPic()
{
    QPropertyAnimation *animate = new QPropertyAnimation(m_twitPic, "scale");
    m_twitPic->setTransformOriginPoint(m_twitPic->boundingRect().center());
    animate->setStartValue(1);
    animate->setEndValue(0);
    animate->setDuration(700);
    animate->setEasingCurve(QEasingCurve::OutExpo);
    animate->start(QPropertyAnimation::DeleteWhenStopped);

    connect(animate, SIGNAL(finished()), this, SLOT(_deleteTwitPic()));
}

void TimelineView::_deleteTwitPic()
{
    delete m_twitPic;
}

bool TimelineView::getHome(void)
{
    return this->m_home;
}
