#include <algorithm>
#include "ocrrequest.h"

// ip is the string for the ip of the server.
// port is the port the server is running on.
// path is the path to the file to send to the server for ocr
OCRRequest::OCRRequest(QString ip, qint16 port, QString path)
{
    m_ip = ip;
    m_port = port;
    m_input_file_path = path;
    m_total_recv = 0;
    m_resp_data = 0;
    m_lang = 0;

    // QNetworkAccessManager is used for the HTTP request to the translation service (i.e., Google).
    // Finished SIGNAL is emitted when the HTTP response is finished receiving
    manager = new QNetworkAccessManager(this);
    connect(manager, SIGNAL(finished(QNetworkReply*)),
            this, SLOT(RecvTransResponse(QNetworkReply*)));
    connect(this, SIGNAL(TransFinished(QString)),
            this, SLOT(WrapUp(QString)));
    state = idle;
}

void OCRRequest::SetLang(QString lang){
    m_lang = LangToInt(lang);
}

// Called to initiate connection to OCR server and begin the request
void OCRRequest::MakeOCRRequest(){
    m_need_to_cleanup = false;
    timer.start(120000); //2 minute time out
    connect(&timer, SIGNAL(timeout()), this, SLOT(TimedOut()));
    server_socket = new QTcpSocket(this);
    /* Connect to the server */
    state = connecting_OCR_server;
    connect(server_socket, SIGNAL(connected()),
            this, SLOT(SendOCRReqPkt()));
    connect(server_socket, SIGNAL(error(QAbstractSocket::SocketError)), this, SLOT(ConnectToHostError()));
    server_socket->connectToHost(QHostAddress(m_ip), m_port);
}

void OCRRequest::ConnectToHostError(){
    emit ErrorConnecting(this);
}

void OCRRequest::TimedOut(){
    // Disconnect the error connecting emit, because we dont want to re-emit an error, since this time out will already kill the request.
    disconnect(server_socket, SIGNAL(error(QAbstractSocket::SocketError)), this, SLOT(ConnectToHostError()));
    WrapUp("");
    emit TimedOutSignal(this);
    timer.stop();
}

/* The slot for receiving the http translation response */
void OCRRequest::RecvTransResponse(QNetworkReply *reply)
{
    reply->deleteLater();

    if (reply->error()) {
        printf("Translation failed");
        return;
    }

    QByteArray result = reply->readAll();
    QScriptValue sc;
    QScriptEngine engine;
    sc = engine.evaluate("(" + QString(result) + ")");

    QScriptValue response_data_sc = sc.property("responseData");
    QScriptValue translated_text_sc = response_data_sc.property("translatedText");
    QString translated_text = translated_text_sc.toString();

    /* A little bit of nastiness here to get the encoding right */
    //translated_text = QString::fromUtf8(translated_text.toAscii());
    translated_text = QString::fromUtf8(translated_text.toAscii());
    QTextDocument td;
    td.setHtml(translated_text);
    translated_text = td.toPlainText();

    emit TransFinished(translated_text);
}

void OCRRequest::SetInputPath(QString path){
    m_input_file_path = path;
}

void OCRRequest::WrapUp(QString text){
    //qDebug() << m_resp_data << "\n";
    //qDebug() << text;
    if(m_need_to_cleanup){
        // Need to free the memory allocated to store the .txt results from the OCR server.
        // Don't need to free if this WrapUp() was reached via a retranslation request.
        free(m_resp_data);
    }
}

// SLOT is activated by the SIGNAL emitted when QTcpSocket signals that the connection to the server has been established
void OCRRequest::SendOCRReqPkt(){
    // Connection was established, so no need to emit connection errors anymore.
    disconnect(server_socket, SIGNAL(error(QAbstractSocket::SocketError)), this, SLOT(ConnectToHostError()));
    QFile input_file(m_input_file_path);
    qint64 file_size = input_file.size();

    // Create the request packet to be packed and sent
    request_pkt req;
    req.size = file_size;
    req.lang = m_lang;
    req.req_type = 0;

    state = send_request_pkt;
    connect(server_socket, SIGNAL(bytesWritten(qint64)), this, SLOT(DispatchSend()));
    /* Write the request packet to the output buffer*/
    SendBufferData(server_socket, (char *) &req, sizeof(request_pkt));
}

// Called every time the socket has successfully sent some data out. Check what state the request is in and continue sending or shift states.
void OCRRequest::DispatchSend(){
    if(server_socket->bytesToWrite() != 0){
        // Whatever we are sending, we aren't done yet. No need to do anything, just wait.
        return;
    }else if(state == send_request_pkt && server_socket->bytesToWrite() == 0){
        /* Just finished sending the request pkt. Now need to initiate image sending */
        state = send_image;
        SendFileData(server_socket, m_input_file_path);
    }else if (state == send_image && server_socket->bytesToWrite() == 0){
        /* Just finished sending the image to OCR. Need to await results */
        connect(server_socket, SIGNAL(readyRead()), this, SLOT(DispatchRecv()));
        state = recv_resp_pkt;
    }
}

void OCRRequest::DispatchRecv(){
    if(state == recv_resp_pkt){
        int bytes_to_recv = qMin(qint64(sizeof(result_pkt) - m_total_recv), server_socket->bytesAvailable());
        RecvData(server_socket, (char *) &res_pkt + m_total_recv, bytes_to_recv, 0);
        m_total_recv += bytes_to_recv;

        if(m_total_recv == sizeof(result_pkt)){
            /* Done receiving result_pkt. Now need to recv the response data */
            state = recv_resp_data;
            m_total_recv = 0;
            //Use calloc + 1 to ensure null termination of recvd text file
            m_resp_data = (char *) calloc(res_pkt.size + 1, 1);
            m_need_to_cleanup = true;
            DispatchRecv();
        }

    }else if (state == recv_resp_data){
        int bytes_to_recv = qMin(qint64(res_pkt.size - m_total_recv), server_socket->bytesAvailable());
        RecvData(server_socket, m_resp_data + m_total_recv, bytes_to_recv, 0);
        m_total_recv += bytes_to_recv;

        if(m_total_recv == res_pkt.size){
            state = send_translate;
            /* Done receiving results. Emit signal to declare this. */
            timer.stop();
            emit OCR_finished(QString::fromUtf8(m_resp_data));
        }
    }
}

// Called from cameraUI after it receives notice from cameraUI's SIGNAL SetOCR_Finished (triggered by OCrRequest's OCR_finished signal)
// src and dest specify the source and destination languages
void OCRRequest::Make_Trans_Request(QString text, QString src, QString dest){

    if(src == dest){
        emit TransFinished(" ");
        return;
    }

    QString to_translate(text);
    to_translate = QUrl::toPercentEncoding(to_translate);
    /* Google translate limits requests to 5000 chars. We will stay conservative and use 4800 as the limit */
    to_translate = to_translate.left(4800);

    QString post_data = "v=1.0&q=" + to_translate + "&langpair=" + src + "%7C" + dest + "&key=" + g_API_key;

    QUrl qurl(g_translate_api_url);
    QNetworkRequest http_req(qurl);
    http_req.setRawHeader("Referer", g_http_referrer.toUtf8());

    manager->post(http_req, post_data.toUtf8());
    state = recv_translate;
}

void OCRRequest::Make_Retrans_Request(QString text, QString src, QString dest){
    state = retrans;
    // Don't need to cleanup on WrapUp() slot because we didn't allocate any memory to receive .txt results from OCR server.
    //We are just going back to Google to get a new translation via HTTP.
    m_need_to_cleanup = false;
    Make_Trans_Request(text, src, dest);
}
