/*
 *  Microfeed - Backend for accessing feed-based services (unstable providers)
 *  Copyright (C) 2009 Henrik Hedberg <henrik.hedberg@innologies.fi>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 2 as published by
 *  the Free Software Foundation.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */
 
#include <microfeed-provider/microfeedprovider.h>
#include <microfeed-provider/microfeedpublisher.h>
#include <microfeed-provider/microfeedfeed.h>
#include <microfeed-common/microfeedmain.h>
#include <microfeed-common/microfeedconfiguration.h>
#include <microfeed-common/microfeedthread.h>
#include <microfeed-common/microfeedprotocol.h>
#include <microfeed-common/microfeedmisc.h>
#include <microfeed-provider/microfeedhttp.h>
#include <microfeed-common/microfeedobject.h>

#include "twitter.h"

#include <string.h>
#include <stdio.h>

static MicrofeedPublisher* instantiate_publisher(MicrofeedProvider* provider, const char* publisher_identifier, const char* directory, void* user_data);
static MicrofeedFeed* instantiate_feed(MicrofeedPublisher* publisher, const char* uri, void* user_data);
static void no_more_publishers(MicrofeedProvider* publisher, void* user_data);
static void initialize_settings(MicrofeedPublisher* publisher, void* user_data);
static int update_setting(MicrofeedPublisher* publisher, const char* key, const char* value, void* user_data);
static void* add_timeout(MicrofeedProvider* provider, unsigned long int interval, MicrofeedProviderTimeoutHandler handler, void* data, void* user_data);
static void remove_timeout(MicrofeedProvider* provider, void* timeout_implementation, void* user_data);
static MicrofeedError* download_image(MicrofeedFeed* feed, const char* uid, void** data, size_t* length, void* user_data);

static MicrofeedProviderCallbacks provider_callbacks = {
	.instantiate_publisher = instantiate_publisher,
	.no_more_publishers = no_more_publishers,
	.add_timeout = add_timeout,
	.remove_timeout = remove_timeout
};

static MicrofeedPublisherCallbacks publisher_callbacks = {
	.instantiate_feed = instantiate_feed,
	.initialize_settings = initialize_settings,
	.update_setting = update_setting,
};

static MicrofeedFeedCallbacks images_feed_callbacks = {
	.download_item_data = download_image
};

int main(void) {
	MicrofeedMain* microfeed_main;
	MicrofeedProvider* provider;
	MicrofeedWeakReference* weak_reference;
	
	//dbus_connection_set_change_sigpipe(FALSE);
	dbus_threads_init_default();
	microfeed_thread_init();
	microfeed_http_init(1);

	microfeed_main = microfeed_main_new();
	provider = microfeed_provider_new(PROVIDER_ID_TWITTER, microfeed_main_get_dbus_connection(microfeed_main), &provider_callbacks, microfeed_main);
	weak_reference = microfeed_object_get_weak_reference(provider, MicrofeedProvider);

	do {
		microfeed_main_loop(microfeed_main);
		
		microfeed_object_unref(provider, MicrofeedProvider);
		provider = microfeed_weak_reference_get_object(weak_reference, MicrofeedProvider);
	} while (provider);

	microfeed_weak_reference_unref(weak_reference);

	return 0;
}

static void no_more_publishers(MicrofeedProvider* provider, void* user_data) {
	MicrofeedMain* microfeed_main;
	
	microfeed_main = (MicrofeedMain*)user_data;
	
	microfeed_main_exit(microfeed_main);
}

static MicrofeedPublisher* instantiate_publisher(MicrofeedProvider* provider, const char* publisher_identifier, const char* directory, void* user_data) {
	MicrofeedConfiguration* configuration = NULL;
	MicrofeedPublisher* publisher;
	
	if (!directory) {
		configuration = microfeed_configuration_new();
		directory = microfeed_configuration_get_default_publisher_directory(configuration);
	}
	
	publisher = microfeed_publisher_new(provider, publisher_identifier, directory, &publisher_callbacks, user_data);

	if (configuration) {
		microfeed_configuration_free(configuration);
	}

	return publisher;
}

static MicrofeedFeed* instantiate_feed(MicrofeedPublisher* publisher, const char* uri, void* user_data) {
	MicrofeedFeed* feed = NULL;
	MicrofeedMain* microfeed_main;
	size_t length;
	char* url;
	MicrofeedFeed* images_feed;
	
	microfeed_main = (MicrofeedMain*)user_data;

	if (!strcmp(uri, MICROFEED_FEED_URI_OVERVIEW)) {
		url = strdup("/statuses/friends_timeline.json");
		feed = microfeed_feed_new(publisher, uri, "Overview", MICROFEED_FEED_PERMISSION_ADD, &overview_feed_callbacks, url);
		images_feed = microfeed_publisher_get_or_instantiate_feed(publisher, MICROFEED_FEED_URI_IMAGES);
		microfeed_feed_add_item_data_property(feed, MICROFEED_ITEM_PROPERTY_NAME_USER_IMAGE, images_feed);
		microfeed_feed_add_item_data_property(feed, MICROFEED_ITEM_PROPERTY_NAME_CONTENT_IMAGE, images_feed);
		microfeed_object_unref(images_feed, MicrofeedFeed);
	} else if (!strcmp(uri, MICROFEED_FEED_URI_CONTACTS)) {
		feed = microfeed_feed_new(publisher, uri, "Contacts", MICROFEED_FEED_PERMISSION_ADD, &contacts_callbacks, NULL);
		images_feed = microfeed_publisher_get_or_instantiate_feed(publisher, MICROFEED_FEED_URI_IMAGES);
		microfeed_feed_add_item_data_property(feed, MICROFEED_ITEM_PROPERTY_NAME_USER_IMAGE, images_feed);
		microfeed_object_unref(images_feed, MicrofeedFeed);
	} else if (!strcmp(uri, MICROFEED_FEED_URI_IMAGES)) {
		feed = microfeed_feed_new(publisher, uri, "Images", MICROFEED_FEED_PERMISSION_NONE, &images_feed_callbacks, NULL);
	} else if ((length = microfeed_util_string_starts_with(uri, MICROFEED_FEED_URI_USER_PREFIX))) {
		url = microfeed_util_string_concatenate("/statuses/user_timeline.json?user_id=", uri + strlen(MICROFEED_FEED_URI_USER_PREFIX), NULL);
		feed = microfeed_feed_new(publisher, uri, uri + length, MICROFEED_FEED_PERMISSION_NONE, &feed_callbacks, url);
		images_feed = microfeed_publisher_get_or_instantiate_feed(publisher, MICROFEED_FEED_URI_IMAGES);
		microfeed_feed_add_item_data_property(feed, MICROFEED_ITEM_PROPERTY_NAME_USER_IMAGE, images_feed);
		microfeed_feed_add_item_data_property(feed, MICROFEED_ITEM_PROPERTY_NAME_CONTENT_IMAGE, images_feed);
		microfeed_object_unref(images_feed, MicrofeedFeed);
	}

	return feed;	
}

static void initialize_settings(MicrofeedPublisher* publisher, void* user_data) {
	microfeed_publisher_add_setting(publisher, "service.url", "URL", "", "string", "256", "http://twitter.com/");
	microfeed_publisher_add_setting(publisher, "authentication.login", "Login", "", "string", "128", "");
	microfeed_publisher_add_setting(publisher, "authentication.password", "Password", "", "password", "128", "");
}

static int update_setting(MicrofeedPublisher* publisher, const char* key, const char* value, void* user_data) {
	return 1;
}

static MicrofeedError* download_image(MicrofeedFeed* feed, const char* uid, void** data, size_t* length, void* user_data) {
	char* buffer;
	MicrofeedError* error = NULL;
	MicrofeedHttp* http;

	http = microfeed_http_new();
	if ((buffer = (void*)microfeed_http_get_data(http, uid, length))) {
		*data = malloc(*length);
		memcpy(*data, buffer, *length);
	} else {
		error = microfeed_error_new(MICROFEED_ERROR_CONNECTION_FAILED, "Could not download image");
	}
	microfeed_http_free(http);
	
	return error;	
}

typedef struct {
	unsigned long int interval;
	MicrofeedProviderTimeoutHandler handler;
	void* data;
	MicrofeedTimeout* timeout;
} TimeoutData;

static void timeout_callback(MicrofeedMain* microfeed_main, void* user_data) {
	TimeoutData* timeout_data;
	
	timeout_data = (TimeoutData*)user_data;
	if (timeout_data->handler(timeout_data->data)) {
		timeout_data->timeout = microfeed_main_add_timeout(microfeed_main, timeout_data->interval, timeout_callback, user_data);
	} else {
		microfeed_memory_free(timeout_data);
	}
}

static void* add_timeout(MicrofeedProvider* provider, unsigned long int interval, MicrofeedProviderTimeoutHandler handler, void* data, void* user_data) {
	MicrofeedMain* microfeed_main;
	TimeoutData* timeout_data;

	microfeed_main = (MicrofeedMain*)user_data;
	timeout_data = microfeed_memory_allocate(TimeoutData);
	timeout_data->interval = interval;
	timeout_data->handler = handler;
	timeout_data->data = data;
	timeout_data->timeout = microfeed_main_add_timeout(microfeed_main, interval, timeout_callback, timeout_data);

	return timeout_data;
}

static void remove_timeout(MicrofeedProvider* provider, void* timeout_implementation, void* user_data) {
	MicrofeedMain* microfeed_main;
	TimeoutData* timeout_data;
	
	microfeed_main = (MicrofeedMain*)user_data;
	timeout_data = (TimeoutData*)timeout_implementation;

	microfeed_main_remove_timeout(microfeed_main, timeout_data->timeout);	
	microfeed_memory_free(timeout_data);
}
