/*
 *  Microfeed - Backend for accessing feed-based services (unstable providers)
 *  Copyright (C) 2009 Henrik Hedberg <henrik.hedberg@innologies.fi>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 2 as published by
 *  the Free Software Foundation.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */
 
#define _XOPEN_SOURCE
#define _BSD_SOURCE
#define _GNU_SOURCE

#include <microfeed-provider/microfeedfeed.h>
#include <microfeed-common/microfeedmisc.h>
#include <microfeed-common/microfeedprotocol.h>
#include <microfeed-provider/microfeedhttp.h>
#include <microfeed-provider/microfeedjson.h>

#include "twitter.h"

#include <stdio.h>
#include <string.h>

typedef struct {
	char* nick;
	char* id;
	char* image;
	char* name;
} Contact;

static MicrofeedError* update_feed(MicrofeedFeed* feed, int user_initiated, void* user_data);
static MicrofeedError* modify_item(MicrofeedFeed *feed, MicrofeedItem *existing_item, MicrofeedItem *new_item, void *user_data);

static Contact* contact_new(MicrofeedJson* json);
static void contact_free(Contact* contact);
static char* contact_get_nick(Contact* contact);
static MicrofeedItem* contact_to_item(Contact* contact);

MicrofeedFeedCallbacks contacts_callbacks = {
	NULL, /* destroy */
	update_feed,
	modify_item,
	NULL /* download_item_data */
};

static MicrofeedError* update_feed(MicrofeedFeed* feed, int user_initiated, void* user_data) {
	MicrofeedError* error = NULL;
	char* site;
	MicrofeedStore* contacts;
	int more_contacts;
	int page;
	MicrofeedHttp* http;
	char url_buffer[1024];
	time_t delta;
	MicrofeedJson* json;
	MicrofeedStoreIterator* store_iterator;
	MicrofeedFeedIterator* feed_iterator;
	unsigned int i;
	MicrofeedJson* object;
	Contact* contact;
	const char* created_at;
	time_t timestamp;
	MicrofeedItem* item;
	char* s;

	site = microfeed_publisher_get_setting_value(microfeed_feed_get_publisher(feed), "service.url", "http://twitter.com");
	contacts = microfeed_store_new_sorted((MicrofeedStoreCompareKeysFunction)strcmp, (MicrofeedStoreGetKeyFunction)contact_get_nick);

	http = microfeed_http_new();
	do_authentication(microfeed_feed_get_publisher(feed), http);
	for (page = 1, more_contacts = 1; more_contacts && !error; page++) {
		snprintf(url_buffer, 1024, "%s/statuses/friends.json?page=%d", site, page);
		if ((json = microfeed_http_get_json(http, url_buffer))) {
			if (!(error = check_if_error(json))) {
				if (microfeed_http_get_server_time(http)) {
					delta = microfeed_http_get_server_time(http) - microfeed_http_get_reply_start_time(http);
				} else {
					delta = 0;
				}
				if (microfeed_json_get_type(json, NULL) == MICROFEED_JSON_TYPE_ARRAY) {
					if (microfeed_json_get_size(json) == 0) {
						more_contacts = 0;
					} else {
						for (i = 0; i < microfeed_json_get_size(json); i++) {
							if ((object = microfeed_json_get_object_by_index(json, i)) &&
							    (contact = contact_new(object))) {
								microfeed_store_insert(contacts, contact);
							}
						}
					}
				} else {
					error = microfeed_error_new(MICROFEED_ERROR_COMMUNICATION_FAILED, "Invalid reply (expected JSON array).");
				}
			}
			microfeed_json_free(json);
		} else {
			error = microfeed_error_new(MICROFEED_ERROR_CONNECTION_FAILED, "Could not get the feed.");
		}
	}
	
	if (!error) {
		feed_iterator = microfeed_feed_iterate(feed, NULL, 0);
		for (store_iterator = microfeed_store_iterate(contacts, NULL);
		     (contact = microfeed_store_iterator_get(store_iterator, Contact));
		     microfeed_store_iterator_next(store_iterator)) {
			if (microfeed_feed_iterator_jump_and_remove_previous_items(feed_iterator, contact->nick)) {
				microfeed_feed_iterator_next(feed_iterator);
			} else {
				item = contact_to_item(contact);
				microfeed_feed_replace_item(feed, item);
				microfeed_item_free(item);
			}

			contact_free(contact);
		}
		microfeed_store_iterator_free(store_iterator);
		microfeed_feed_iterator_free(feed_iterator);
	}

	free(site);
	microfeed_store_free(contacts);

	return error;
}

static MicrofeedError* modify_item(MicrofeedFeed *feed, MicrofeedItem *existing_item, MicrofeedItem *new_item, void *user_data) {
	MicrofeedError* error = NULL;
	char* site;
	char* url;
	MicrofeedItem* item;
	MicrofeedHttp* http;
	MicrofeedJson* json;
	Contact* contact;

	site = microfeed_publisher_get_setting_value(microfeed_feed_get_publisher(feed), "service.url", "http://twitter.com");
	if (existing_item && new_item) {		
		error = microfeed_error_new(MICROFEED_ERROR_CANNOT_MODIFY_ITEM, "Cannot modify an existing item.");
	} else if (new_item) {
		url = microfeed_util_string_concatenate(site, "/friendships/create.json?screen_name=", microfeed_item_get_uid(new_item), NULL);
	} else {
		url = microfeed_util_string_concatenate(site, "/friendships/destroy.json?screen_name=", microfeed_item_get_uid(existing_item), NULL);
	}
	free(site);
printf(":%s\n", url);
	if (!error) {
		http = microfeed_http_new();
		do_authentication(microfeed_feed_get_publisher(feed), http);
		if (!(json = microfeed_http_post_json(http, url, ""))) {

			error = microfeed_error_new(MICROFEED_ERROR_COMMUNICATION_FAILED, "Invalid reply when changing friendship status.");
		}
		if (!(error = check_if_error(json))) {
			if (new_item) {
				if ((contact = contact_new(json))) {
					item = contact_to_item(contact);
					microfeed_feed_replace_item(feed, item);
					microfeed_item_free(item);
					contact_free(contact);
				}					
			} else {
				microfeed_feed_remove_item(feed, microfeed_item_get_uid(existing_item));
			}
		}

		microfeed_json_free(json);
		free(url);
	}
	
	return error;
}

static Contact* contact_new(MicrofeedJson* json) {
	Contact* contact = NULL;
	const char* id;
	const char* nick;
	const char* cs;
	
	if ((id = microfeed_json_get_as_string(json, "id")) &&
	    (nick = microfeed_json_get_string(json, "screen_name"))) {
		contact = microfeed_memory_allocate(Contact);
		contact->id = strdup(id);
		contact->nick = strdup(nick);
		if ((cs = microfeed_json_get_string(json, "name"))) {
			contact->name = strdup(cs);
		}
		if ((cs = microfeed_json_get_string(json, "profile_image_url"))) {
			contact->image = strdup(cs);
		}
	}
	
	return contact;
}

static void contact_free(Contact* contact) {
	free(contact->id);
	free(contact->nick);
	free(contact->name);
	free(contact->image);
	free(contact);
}

static char* contact_get_nick(Contact* contact) {
	
	return contact->nick;
}

static MicrofeedItem* contact_to_item(Contact* contact) {
	MicrofeedItem* item;
	
	item = microfeed_item_new(contact->nick, time(NULL));
	microfeed_item_set_property(item, MICROFEED_ITEM_PROPERTY_NAME_USER_NICK, contact->nick);
	microfeed_item_set_property(item, ".twitter.id", contact->id);
	if (contact->name) {
		microfeed_item_set_property(item, MICROFEED_ITEM_PROPERTY_NAME_USER_NAME, contact->name);
	}
	if (contact->image) {
		microfeed_item_set_property(item, MICROFEED_ITEM_PROPERTY_NAME_USER_IMAGE, contact->image);
	}

	return item;
}
