
#include <microfeed-provider/microfeedprovider.h>
#include <microfeed-provider/microfeedpublisher.h>
#include <microfeed-provider/microfeedfeed.h>
#include <microfeed-common/microfeedmain.h>
#include <microfeed-common/microfeedconfiguration.h>
#include <microfeed-common/microfeedthread.h>
#include <microfeed-common/microfeedprotocol.h>
#include <microfeed-common/microfeedmisc.h>
#include <microfeed-provider/microfeedhttp.h>

#include "twitter.h"

#include <string.h>
#include <stdio.h>

static void no_more_publishers(MicrofeedProvider* publisher, void* user_data);
static MicrofeedFeed* create_feed(MicrofeedPublisher* publisher, const char* uri, void* user_data);
static void initialize_settings(MicrofeedPublisher* publisher, void* user_data);
static int update_setting(MicrofeedPublisher* publisher, const char* key, const char* value, void* user_data);
static MicrofeedError* download_image(MicrofeedPublisher* publisher, const char* uid, void** data, size_t* length, void* user_data);
static void* add_timeout(MicrofeedPublisher* publisher, unsigned long int interval, MicrofeedPublisherTimeoutHandler handler, void* data, void* user_data);
static void remove_timeout(MicrofeedPublisher* publisher, void* timeout_implementation, void* user_data);

MicrofeedProviderCallbacks provider_callbacks = {
	no_more_publishers,
	{
		create_feed,
		initialize_settings,
		update_setting,
		download_image,
		NULL,
		add_timeout,
		remove_timeout
	}
};

int main(void) {
	MicrofeedMain* microfeed_main;
	MicrofeedProvider* provider;
	
	//dbus_connection_set_change_sigpipe(FALSE);
	dbus_threads_init_default();
	microfeed_thread_init();
	microfeed_http_init(1);

	microfeed_main = microfeed_main_new();
	provider = microfeed_provider_new(PROVIDER_ID_TWITTER, microfeed_main_get_dbus_connection(microfeed_main), &provider_callbacks, microfeed_main);

	microfeed_main_loop(microfeed_main);

	microfeed_provider_free(provider);

	return 0;
}

static void no_more_publishers(MicrofeedProvider* provider, void* user_data) {
	MicrofeedMain* microfeed_main;
	
	microfeed_main = (MicrofeedMain*)user_data;
	microfeed_main_exit(microfeed_main);
}

static MicrofeedFeed* create_feed(MicrofeedPublisher* publisher, const char* uri, void* user_data) {
	MicrofeedFeed* feed = NULL;
	MicrofeedMain* microfeed_main;
	size_t length;
	char* url;
	
	microfeed_main = (MicrofeedMain*)user_data;

	if (!strcmp(uri, MICROFEED_FEED_URI_OVERVIEW)) {
		url = strdup("/statuses/friends_timeline.json");
		feed = microfeed_feed_new(publisher, uri, "Overview", MICROFEED_FEED_PERMISSION_ADD, &overview_feed_callbacks, url);
	} else if (!strcmp(uri, MICROFEED_FEED_URI_CONTACTS)) {
		feed = microfeed_feed_new(publisher, uri, "Contacts", MICROFEED_FEED_PERMISSION_ADD, &contacts_callbacks, NULL);
	} else if ((length = microfeed_util_string_starts_with(uri, MICROFEED_FEED_URI_USER_PREFIX))) {
		url = microfeed_util_string_concatenate("/statuses/user_timeline.json?user_id=", uri + strlen(MICROFEED_FEED_URI_USER_PREFIX), NULL);
		feed = microfeed_feed_new(publisher, uri, uri + length, MICROFEED_FEED_PERMISSION_NONE, &feed_callbacks, url);
	}

	return feed;	
}

static void initialize_settings(MicrofeedPublisher* publisher, void* user_data) {
	microfeed_publisher_add_setting(publisher, "service.url", "URL", "", "string", "256", "http://twitter.com/");
	microfeed_publisher_add_setting(publisher, "authentication.login", "Login", "", "string", "128", "");
	microfeed_publisher_add_setting(publisher, "authentication.password", "Password", "", "password", "128", "");
}

static int update_setting(MicrofeedPublisher* publisher, const char* key, const char* value, void* user_data) {
	return 1;
}

static MicrofeedError* download_image(MicrofeedPublisher* publisher, const char* uid, void** data, size_t* length, void* user_data) {
	char* buffer;
	MicrofeedError* error = NULL;
	MicrofeedHttp* http;

	http = microfeed_http_new();
	if ((buffer = (void*)microfeed_http_get_data(http, uid, length))) {
		*data = malloc(*length);
		memcpy(*data, buffer, *length);
	} else {
		error = microfeed_error_new(MICROFEED_ERROR_CONNECTION_FAILED, "Could not download image");
	}
	microfeed_http_free(http);
	
	return error;	
}

typedef struct {
	unsigned long int interval;
	MicrofeedPublisherTimeoutHandler handler;
	void* data;
	MicrofeedTimeout* timeout;
} TimeoutData;

static void timeout_callback(MicrofeedMain* microfeed_main, void* user_data) {
	TimeoutData* timeout_data;
	
	timeout_data = (TimeoutData*)user_data;
	timeout_data->handler(timeout_data->data);
	timeout_data->timeout = microfeed_main_add_timeout(microfeed_main, timeout_data->interval, timeout_callback, user_data);
}

static void* add_timeout(MicrofeedPublisher* publisher, unsigned long int interval, MicrofeedPublisherTimeoutHandler handler, void* data, void* user_data) {
	MicrofeedMain* microfeed_main;
	TimeoutData* timeout_data;
	
	microfeed_main = (MicrofeedMain*)user_data;
	timeout_data = microfeed_memory_allocate(TimeoutData);
	timeout_data->interval = interval;
	timeout_data->handler = handler;
	timeout_data->data = data;
	timeout_data->timeout = microfeed_main_add_timeout(microfeed_main, interval, timeout_callback, timeout_data);

	return timeout_data;
}

static void remove_timeout(MicrofeedPublisher* publisher, void* timeout_implementation, void* user_data) {
	MicrofeedMain* microfeed_main;
	TimeoutData* timeout_data;
	
	microfeed_main = (MicrofeedMain*)user_data;
	timeout_data = (TimeoutData*)timeout_implementation;

	microfeed_main_remove_timeout(microfeed_main, timeout_data->timeout);	
}
