
// local includes
#include "recentdialog.h"
#include "ui_recentdialog.h"
#include "maemobarcodewindow.h"

RecentDialog::RecentDialog(MaemoBarcodeWindow *parent) :
    QDialog(parent),
    ui(new Ui::RecentDialog)
{
    ui->setupUi(this);

    barcodeImageFolder = QDir(QDir::homePath() + "/.config/mbarcode/images/");
    if(!barcodeImageFolder.exists()) {
        barcodeImageFolder.mkpath(barcodeImageFolder.path());
    }

    this->parent = parent;

    qDebug() << "RecentDialog::RecentDialog(): loading tillRollModel";
    tillRollModel = new QStandardItemModel(0,1);
    ui->tblTillRoll->setModel(tillRollModel);

    qDebug() << "MainWindow::MainWindow(): Setting tblTillRoll properties";
    ui->tblTillRoll->setProperty("FingerScrollable", true); // apparently this will change for 4.6
    //ui->tillRoll->setItemDelegate(new BarcodeTableItemDelegate);
    ui->tblTillRoll->setSelectionBehavior(QAbstractItemView::SelectColumns);

    //ui->tblTillRoll->verticalHeader()->setResizeMode(QHeaderView::Fixed);
    //ui->tblTillRoll->verticalHeader()->setDefaultSectionSize(300);
    ui->tblTillRoll->verticalHeader()->setResizeMode(QHeaderView::ResizeToContents);
    ui->tblTillRoll->horizontalHeader()->setResizeMode(QHeaderView::ResizeToContents);

    qDebug() << "MainWindow::MainWindow(): Loading tillroll history";
    loadHistory();
}

RecentDialog::~RecentDialog()
{
    saveHistory();

    delete ui;
}

void RecentDialog::changeEvent(QEvent *e)
{
    QDialog::changeEvent(e);
    switch (e->type()) {
    case QEvent::LanguageChange:
        ui->retranslateUi(this);
        break;
    default:
        break;
    }
}

void RecentDialog::on_tblTillRoll_customContextMenuRequested(QPoint pos)
{
    QAction *delAct = new QAction(tr("Delete item"), this);
    connect(delAct, SIGNAL(triggered()), this, SLOT(delItem()));

    QAction *saveAct = new QAction(tr("Save item"), this);
    connect(saveAct, SIGNAL(triggered()), this, SLOT(saveItem()));


    QMenu menu(this);
    menu.addAction(delAct);
    menu.addAction(saveAct);
    menu.exec(QWidget::mapToGlobal(pos));

}


void RecentDialog::loadHistory() {
    // load the history list from the settings class data

    if(!parent->settings->value("save_history",true).toBool())
        return;

    int size = parent->settings->beginReadArray("scan_history");

    qDebug() << "MainWindow::loadHistory(): " << size << " items.";

    tillRollModel->clear();

    QStringList filesInUse;

    for (int i = size-1; i >= 0; --i) {
        parent->settings->setArrayIndex(i);
        QVariantHash itemData = parent->settings->value("barcodeDataHash").toHash();

        QStandardItem *barcodeItem = new QStandardItem(itemData.value("barcodeData").toString());

        barcodeItem->setSizeHint(QSize(300,200)); // make the image 300x200

        tillRollModel->insertColumn(0); // insert at left
        tillRollModel->setItem(0, 0, barcodeItem);


        barcodeItem->setData(itemData);

        // create a widget to hold an image and text
        QWidget *w = new QWidget;
        w->setAutoFillBackground(true); // otherwise we'll see bits of the text contents
        QVBoxLayout *lay = new QVBoxLayout();
        QLabel *imlabel = new QLabel;
        filesInUse << itemData.value("barcodeImageFile").toString();
        QImage barcodeImage(itemData.value("barcodeImageFile").toString()); // load image from file
        QPixmap barcodePixmap = QPixmap::fromImage(barcodeImage);
        imlabel->setPixmap(barcodePixmap);
        lay->addWidget(imlabel);
        //QLabel *txtlabel = new QLabel(itemData.value("barcodeData").toString());
        //txtlabel->setWordWrap(true);
        //txtlabel->setMaximumWidth(300);
        QTextEdit *txtlabel = new QTextEdit(itemData.value("barcodeData").toString());
        txtlabel->setReadOnly(true);
        txtlabel->setFrameStyle(QFrame::NoFrame);
        txtlabel->setBackgroundRole(QPalette::Shadow);
        txtlabel->setMaximumWidth(300);
        lay->addWidget(txtlabel);

        QTextEdit *txtType = new QTextEdit(itemData.value("barcodeType").toString());
        txtType->setReadOnly(true);
        txtType->setFrameStyle(QFrame::NoFrame);
        txtType->setBackgroundRole(QPalette::Shadow);
        txtType->setMaximumWidth(300);
        txtType->setMaximumHeight(50);
        lay->addWidget(txtType);

        QTextEdit *txtDateTime = new QTextEdit(itemData.value("barcodeDateTime").toString());
        txtDateTime->setReadOnly(true);
        txtDateTime->setFrameStyle(QFrame::NoFrame);
        txtDateTime->setBackgroundRole(QPalette::Shadow);
        txtDateTime->setMaximumWidth(300);
        txtDateTime->setMaximumHeight(50);
        lay->addWidget(txtDateTime);


        w->setLayout(lay);
        w->setMaximumWidth(300);

        ui->tblTillRoll->setIndexWidget(barcodeItem->index(),w);
        //ui->tblTillRoll->setColumnWidth(barcodeItem->index().column(),300);
    }

    // Remove all files no longer in use by settings
    QStringList filesInPath = barcodeImageFolder.entryList(QStringList(), QDir::Files);
    foreach(QString fileName, filesInPath) {
        QString filePath = barcodeImageFolder.path() + "/" + fileName;
        if(!filesInUse.contains(filePath)) {
            QFile::remove(filePath);
            qDebug() << "RecentDialog::loadHistory(): File no longer in use, removing" << filePath;
        }
    }


    parent->settings->endArray();

}

void RecentDialog::saveHistory() {
    // save the history list to the settings class data

    if(!parent->settings->value("save_history",true).toBool())
        return;

    qDebug() << "RecentDialog::saveHistory(): " << tillRollModel->columnCount() << " items.";

    parent->settings->beginWriteArray("scan_history");
    for (int i = 0; i < tillRollModel->columnCount(); i++) {
        QStandardItem *item = tillRollModel->item(0,i); // assumes we always use columns!
        QVariantHash itemData = item->data().toHash();

        parent->settings->setArrayIndex(i);

        parent->settings->setValue("barcodeDataHash", itemData);
        //settings->setValue("barcodeData", barcodeData);
        //settings->setValue("barcodeImage", barcodeImage);


    }
    parent->settings->endArray();

    parent->settings->sync(); // should not be necessary to call, it is called by QSettings on destruction and occasionally
    int errorCode = parent->settings->status();
    if(!errorCode) {
        qDebug() << "RecentDialog::saveHistory(): All saved ok!";
    } else {
        qDebug() << "RecentDialog::saveHistory(): Save error: " << parent->settings->status();
    }
}

void RecentDialog::delItem()
{
    // delete the current tillroll item
    tillRollModel->removeColumn ( ui->tblTillRoll->currentIndex().column() );
}

void RecentDialog::saveItem()
{
    // save the current tillroll item to file
    QStandardItem *item = tillRollModel->itemFromIndex(ui->tblTillRoll->currentIndex());

    // save this file
    // pop-up a save dialog
    QString fileName = QFileDialog::getSaveFileName(this, tr("Save File"),
                                                    "/home/user/MyDocs/.images/barcode.jpg",
                                                    tr("JPEG Image (*.jpg)"));

    QFileInfo finfo(fileName);
    //if (finfo.exists())
    //    return; // should at least offer a new filename I suppose...

    // see if the extension survived....
    if(finfo.completeSuffix().isEmpty())
        fileName.append(".jpg");

    QVariantHash itemData = item->data().toHash();
    QPixmap pm = itemData.value("barcodeImage").value<QPixmap>();
    QImage im = pm.toImage().convertToFormat(QImage::Format_RGB32);
    im.save(fileName, 0, 95);
}

void RecentDialog::imageAnalysed(QString barcodeType, QString barcodeData, QImage *im) {

    qDebug() << "RecentDialog::imageAnalysed(): tillRollModel->columnCount()=" << tillRollModel->columnCount();

    // see how many columns are in our table first
    if (tillRollModel->columnCount()>SCAN_HISTORY_LENGTH){
        // too many, so remove one
        qDebug("RecentDialog::imageAnalysed(): Table too long, removing one item");
        tillRollModel->removeColumn( tillRollModel->columnCount()-1 );
    }

    QUuid uuid = QUuid::createUuid();
    QString randomString = uuid.toString();

    QImage savedImg = im->scaledToWidth(300, Qt::SmoothTransformation);

    QString barcodeImageFile = barcodeImageFolder.path() + "/" + randomString + ".png";

    if(!savedImg.save(barcodeImageFile)) {
        qCritical() << "RecentDialog::imageAnalysed(): Could not save image to disk!";
    }

    // IMPORTANT: To change the size, you must also change the iconSize in mainwindow.ui for the table
    //QIcon icon = QIcon(pm);

    tillRollModel->insertColumn(0); // insert at left

    QStandardItem *barcodeItem = new QStandardItem(barcodeData);
    barcodeItem->setSizeHint(QSize(300,200)); // make the image 300x200
    tillRollModel->setItem(0, 0, barcodeItem);

    qDebug() << "RecentDialog::imageAnalysed(): Inserted new item. tillRollModel->columnCount()=" << tillRollModel->columnCount();

    //ui->tblTillRoll->resizeColumnsToContents();

    QDateTime barcodeDateTime = QDateTime::currentDateTimeUtc();

    QVariantHash itemData;
    itemData.insert("barcodeType", barcodeType);
    itemData.insert("barcodeData", barcodeData);
    itemData.insert("barcodeImageFile", barcodeImageFile);
    itemData.insert("barcodeDateTime", barcodeDateTime);


    barcodeItem->setData(itemData);


    // create a widget to hold an image and text
    QWidget *w = new QWidget;
    w->setAutoFillBackground(true); // otherwise we'll see bits of the text contents
    QVBoxLayout *lay = new QVBoxLayout();
    QLabel *imlabel = new QLabel;
    imlabel->setWordWrap(true); // this doesn't work for non-white spaced continuous lines
    QImage barcodeImage(barcodeImageFile);
    QPixmap barcodePixmap = QPixmap::fromImage(barcodeImage);
    imlabel->setPixmap(barcodePixmap); // This does for some reason not work as expected...
    lay->addWidget(imlabel);

    //QLabel *txtlabel = new QLabel(barcodeData);
    QTextEdit *txtlabel = new QTextEdit(itemData.value("barcodeData").toString());
    txtlabel->setReadOnly(true);
    txtlabel->setFrameStyle(QFrame::NoFrame);
    txtlabel->setBackgroundRole(QPalette::Shadow);
    txtlabel->setMaximumWidth(300);

    lay->addWidget(txtlabel);

    QTextEdit *txtType = new QTextEdit(itemData.value("barcodeType").toString());
    txtType->setReadOnly(true);
    txtType->setFrameStyle(QFrame::NoFrame);
    txtType->setBackgroundRole(QPalette::Shadow);
    txtType->setMaximumWidth(300);
    txtType->setMaximumHeight(50);
    lay->addWidget(txtType);

    QTextEdit *txtDateTime = new QTextEdit(itemData.value("barcodeDateTime").toString());
    txtDateTime->setReadOnly(true);
    txtDateTime->setFrameStyle(QFrame::NoFrame);
    txtDateTime->setBackgroundRole(QPalette::Shadow);
    txtDateTime->setMaximumWidth(300);
    txtDateTime->setMaximumHeight(50);
    lay->addWidget(txtDateTime);



    w->setLayout(lay);

    delete(im);

    ui->tblTillRoll->setIndexWidget(barcodeItem->index(),w);
    ui->tblTillRoll->setColumnWidth(barcodeItem->index().column(),300);
}


// we should also allow single items to be deleted using e.g. a long press + context pop-up menu
void RecentDialog::on_actionClear_History_triggered()
{
    while (tillRollModel->columnCount()){
        tillRollModel->removeColumn( tillRollModel->columnCount()-1 ); // remove last one
    }
}

void RecentDialog::on_tblTillRoll_clicked(QModelIndex index)
{
    QStandardItem *item = tillRollModel->itemFromIndex(index);
    QVariantHash itemData = item->data().toHash();
    QString barcodeTypeTmp = itemData.value("barcodeType").toString();
    QString barcodeDataTmp = itemData.value("barcodeData").toString();
    QDateTime barcodeDateTimeTmp = itemData.value("barcodeDateTime").toDateTime();

    qDebug() << "RecentDialog::on_tblTillRoll_clicked(): " << barcodeTypeTmp << barcodeDataTmp << barcodeDateTimeTmp;

    selectedBarcodeData_ = barcodeDataTmp;
    selectedBarcodeType_ = barcodeTypeTmp;
    selectedBarcodeDateTime_ = barcodeDateTimeTmp;

    setResult(QDialog::Accepted);
    hide();
}
