#include <gtk/gtk.h>
#include <hildon/hildon.h>
#include <microfeed-subscriber/microfeedsubscriber.h>
#include <microfeed-common/microfeedprotocol.h>
#include <dbus/dbus.h>
#include <dbus/dbus-glib-lowlevel.h>
#include "mauku.h"
#include "mauku-view.h"
#include "mauku-contacts.h"
#include "mauku-write.h"

static void item_data_received(MicrofeedSubscriber* subscriber, const char* publisher, const char* uri, const char* uid, const void* data, const size_t data_size, void* user_data);

MicrofeedSubscriber* subscriber;

static MaukuView* overview_view = NULL;
static GHashTable* image_caches = NULL;

static MicrofeedSubscriberCallbacks callbacks = {
	NULL, /* error_occured */
	NULL, /* feed_update_started */
	NULL, /* feed_update_ended */
	NULL, /* feed_republishing_started */
	NULL, /* feed_republishing_ended */
	NULL, /* item_added */
	NULL, /* item_changed */
	NULL, /* item_republished */
	NULL, /* item_removed */
	NULL, /* item_status_changed */
	item_data_received
};

GdkPixbuf* image_cache_get_image(const gchar* publisher, const gchar* uid) {
	GdkPixbuf* image = NULL;
	GHashTable* image_cache;
	
	if ((image_cache = (GHashTable*)g_hash_table_lookup(image_caches, publisher))) {
		image = (GdkPixbuf*)g_hash_table_lookup(image_cache, uid);
	}
	
	return image;
}

static void item_data_received(MicrofeedSubscriber* subscriber, const char* publisher, const char* uri, const char* uid, const void* data, const size_t data_size, void* user_data) {
	GdkPixbuf* image = NULL;
	GHashTable* image_cache;
	GdkPixbufLoader* pixbuf_loader;
	
	if (image_caches && (image_cache = (GHashTable*)g_hash_table_lookup(image_caches, publisher))) {
		pixbuf_loader = gdk_pixbuf_loader_new();
		gdk_pixbuf_loader_set_size(pixbuf_loader, 51, 51);
		gdk_pixbuf_loader_write(pixbuf_loader, data, data_size, NULL);
		gdk_pixbuf_loader_close(pixbuf_loader, NULL);
		image = gdk_pixbuf_loader_get_pixbuf(pixbuf_loader);
		g_hash_table_replace(image_cache, g_strdup(uid), g_object_ref(image));
		g_object_unref(pixbuf_loader);
	}
}

static void configured_subscribe(MicrofeedSubscriber* subscriber, const char* publisher, void* user_data) {
	GHashTable* image_cache;

	if (overview_view) {
		mauku_view_add_feed(overview_view, publisher, MICROFEED_FEED_URI_OVERVIEW);
	}
	if (!image_caches) {
		image_caches = g_hash_table_new_full(g_str_hash, g_str_equal, g_free, (GDestroyNotify)g_hash_table_destroy);
	}
	image_cache = g_hash_table_new_full(g_str_hash, g_str_equal, g_free, g_object_unref);
	g_hash_table_replace(image_caches, g_strdup(publisher), image_cache);
	microfeed_subscriber_subscribe_feed(subscriber, publisher, MICROFEED_FEED_URI_IMAGES, &callbacks, NULL, NULL, NULL);
}

static void configured_unsubscribe(MicrofeedSubscriber* subscriber, const char* publisher, const char* uri, MicrofeedSubscriberCallbacks* callbacks, void* callbacks_user_data, void* user_data) {
	GHashTable* image_cache;

	if (overview_view) {
		mauku_view_remove_feed(overview_view, publisher, MICROFEED_FEED_URI_OVERVIEW);
	}
	if (image_caches && (image_cache = (GHashTable*)g_hash_table_lookup(image_caches, publisher))) {
		g_hash_table_destroy(image_cache);
	}	
}


static void add_write_feed(gpointer key, gpointer value, gpointer user_data) {
	MaukuWrite* write;
	const gchar* publisher;

	write = (MaukuWrite*)user_data;
	publisher = (const char*)key;

	mauku_write_add_feed(write, publisher, MICROFEED_FEED_URI_OVERVIEW);
}

void add_current_publishers_to_write(MaukuWrite* write) {
	microfeed_subscriber_handle_configured_subscriptions(subscriber, configured_subscribe, configured_unsubscribe, NULL);
	g_hash_table_foreach(image_caches, add_write_feed, write);
}

static void on_write_button_clicked(gpointer user_data) {
	MaukuWrite* write;

	write = mauku_write_new("New message", NULL, NULL, NULL);
	add_current_publishers_to_write(write);
	mauku_write_show(write);
}

static void add_view_feed(gpointer key, gpointer value, gpointer user_data) {
	MaukuView* view;
	const gchar* publisher;

	view = (MaukuView*)user_data;
	publisher = (const char*)key;

	mauku_view_add_feed(view, publisher, MICROFEED_FEED_URI_OVERVIEW);
}

static void on_overview_button_clicked(gpointer user_data) {
	microfeed_subscriber_handle_configured_subscriptions(subscriber, configured_subscribe, configured_unsubscribe, NULL);
	if (!overview_view) {
		overview_view = mauku_view_new("Overview");
		g_hash_table_foreach(image_caches, add_view_feed, overview_view);
	}
	mauku_view_show(overview_view);
}

static void add_contacts_publisher(gpointer key, gpointer value, gpointer user_data) {
	MaukuContacts* contacts;
	const gchar* publisher;

	contacts = (MaukuContacts*)user_data;
	publisher = (const char*)key;

	mauku_contacts_add_publisher(contacts, publisher);
}

static void on_contact_button_clicked(gpointer user_data) {
	GtkWindow* window;
	MaukuContacts* contacts;
	
	window = GTK_WINDOW(user_data);

	contacts = mauku_contacts_new(window);
	microfeed_subscriber_handle_configured_subscriptions(subscriber, configured_subscribe, configured_unsubscribe, NULL);
	g_hash_table_foreach(image_caches, add_contacts_publisher, contacts);

	mauku_contacts_show(contacts);
}

static void on_about_button_clicked(gpointer user_data) {
	GtkWindow* window;
	GdkPixbuf* pixbuf;

	window = GTK_WINDOW(user_data);

	pixbuf = gdk_pixbuf_new_from_file(IMAGE_DIR "/mauku_logo_60x47.png", NULL);
	gtk_show_about_dialog(window, "name", "Mauku", "logo", pixbuf, "version", VERSION, 
	                      "copyright", "Henrik Hedberg <hhedberg@innologies.fi>", 
			      "website", "http://mauku.henrikhedberg.com/", NULL);
}

static void on_child_exit(GPid pid, gint status, gpointer user_data) {
	microfeed_subscriber_handle_configured_subscriptions(subscriber, configured_subscribe, configured_unsubscribe, NULL);

	g_spawn_close_pid(pid);
}

static void on_publishers_button_clicked(GtkButton* button, gpointer user_data) {
	gchar* args[3];
	GPid pid;
	GtkWidget* message_dialog;

		args[0] = "microfeed-configuration";
	args[1] = SUBSCRIBER_IDENTIFIER;
	args[2] = NULL;
	if (g_spawn_async(NULL, args, NULL, G_SPAWN_DO_NOT_REAP_CHILD | G_SPAWN_SEARCH_PATH, NULL, NULL, &pid, NULL)) {
		g_child_watch_add(pid, on_child_exit, NULL);
	} else {
		hildon_banner_show_information(NULL, NULL, "Could not launch microfeed-configuration.");
	}
}

static void on_special_feeds_button_clicked(GtkButton* button, gpointer user_data) {
	static MaukuContacts* contacts = NULL;
	GtkWindow* window;
	GtkWidget* touch_selector;
	GtkWidget* picker_dialog;
	
	window = GTK_WINDOW(user_data);

	hildon_banner_show_information(NULL, NULL, "Not implemented yet.");
}

static HildonAppMenu* create_menu() {
	HildonAppMenu* app_menu;
	GtkWidget* button;
	
	app_menu = HILDON_APP_MENU(hildon_app_menu_new());
	button = gtk_button_new_with_label("Publishers");
	g_signal_connect_after(button, "clicked", G_CALLBACK(on_publishers_button_clicked), NULL);
	hildon_app_menu_append(app_menu, GTK_BUTTON(button));
	gtk_widget_show_all(GTK_WIDGET(app_menu));
	
	return app_menu;
}


static void create_main_window() {
	GtkWidget* window;
	GtkWidget* table;
	GtkWidget* box;
	GtkWidget* widget;
	GtkWidget* button;

	window = hildon_stackable_window_new();
	gtk_window_set_title(GTK_WINDOW(window), "Mauku");
	g_signal_connect(window, "destroy", G_CALLBACK(gtk_main_quit), NULL);
	hildon_window_set_app_menu(HILDON_WINDOW(window), create_menu());
	hildon_program_add_window(hildon_program_get_instance(), HILDON_WINDOW(window));

	table = gtk_table_new(2, 3, FALSE);
	gtk_container_add(GTK_CONTAINER(window), table);

	button = gtk_event_box_new();
	g_signal_connect_swapped(button, "button-press-event", G_CALLBACK(on_write_button_clicked), window);
	gtk_table_attach(GTK_TABLE(table), button, 0, 1, 0, 1, GTK_EXPAND | GTK_FILL, GTK_EXPAND | GTK_FILL, 16, 16);	
	box = gtk_vbox_new(FALSE, 8);
	gtk_container_add(GTK_CONTAINER(button), box);
	widget = gtk_image_new_from_file(IMAGE_DIR "/mauku_new_message.png");
	gtk_misc_set_alignment(GTK_MISC(widget), 0.5, 1.0);
	gtk_box_pack_start(GTK_BOX(box), widget, TRUE, TRUE, 0);
	widget = gtk_label_new("New message");
	gtk_misc_set_alignment(GTK_MISC(widget), 0.5, 0.0);
	gtk_box_pack_start(GTK_BOX(box), widget, TRUE, TRUE, 0);

	button = gtk_event_box_new();
	g_signal_connect_swapped(button, "button-press-event", G_CALLBACK(on_overview_button_clicked), window);
	gtk_table_attach(GTK_TABLE(table), button, 1, 2, 0, 1, GTK_EXPAND | GTK_FILL, GTK_EXPAND | GTK_FILL, 16, 16);	
	box = gtk_vbox_new(FALSE, 8);
	gtk_container_add(GTK_CONTAINER(button), box);
	widget = gtk_image_new_from_file(IMAGE_DIR "/mauku_overview.png");
	gtk_misc_set_alignment(GTK_MISC(widget), 0.5, 1.0);
	gtk_box_pack_start(GTK_BOX(box), widget, TRUE, TRUE, 0);
	widget = gtk_label_new("Overview");
	gtk_misc_set_alignment(GTK_MISC(widget), 0.5, 0.0);
	gtk_box_pack_start(GTK_BOX(box), widget, TRUE, TRUE, 0);

	button = gtk_event_box_new();
	g_signal_connect_swapped(button, "button-press-event", G_CALLBACK(on_contact_button_clicked), window);
	gtk_table_attach(GTK_TABLE(table), button, 2, 3, 0, 1, GTK_EXPAND | GTK_FILL, GTK_EXPAND | GTK_FILL, 16, 16);	
	box = gtk_vbox_new(FALSE, 8);
	gtk_container_add(GTK_CONTAINER(button), box);
	widget = gtk_image_new_from_file(IMAGE_DIR "/mauku_contacts.png");
	gtk_misc_set_alignment(GTK_MISC(widget), 0.5, 1.0);
	gtk_box_pack_start(GTK_BOX(box), widget, TRUE, TRUE, 0);
	widget = gtk_label_new("Contacts");
	gtk_misc_set_alignment(GTK_MISC(widget), 0.5, 0.0);
	gtk_box_pack_start(GTK_BOX(box), widget, TRUE, TRUE, 0);


	button = gtk_event_box_new();
	g_signal_connect_swapped(button, "button-press-event", G_CALLBACK(on_about_button_clicked), window);
	gtk_table_attach(GTK_TABLE(table), button, 0, 3, 1, 2, GTK_EXPAND | GTK_FILL, 0, 16, 16);	
	box = gtk_hbox_new(FALSE, 8);
	gtk_container_add(GTK_CONTAINER(button), box);
	widget = gtk_image_new_from_file(IMAGE_DIR "/mauku_logo_60x47.png");
	gtk_misc_set_alignment(GTK_MISC(widget), 0.0, 1.0);
	gtk_box_pack_start(GTK_BOX(box), widget, FALSE, FALSE, 0);
	widget = gtk_label_new("");
	gtk_label_set_markup(GTK_LABEL(widget), "<small>Mauku " VERSION "</small>");
	gtk_misc_set_alignment(GTK_MISC(widget), 0.0, 0.95);
	gtk_box_pack_start(GTK_BOX(box), widget, FALSE, FALSE, 8);


#if 0
	vbox = gtk_vbox_new(FALSE, 0);
	gtk_container_add(GTK_CONTAINER(window), vbox);

	button = hildon_button_new_with_text(HILDON_SIZE_FULLSCREEN_WIDTH | HILDON_SIZE_FINGER_HEIGHT, HILDON_BUTTON_ARRANGEMENT_VERTICAL, "New message", NULL);
	g_signal_connect(button, "clicked", G_CALLBACK(on_write_button_clicked), window);
	gtk_box_pack_start(GTK_BOX(vbox), button, FALSE, FALSE, 0);	

	button = hildon_button_new_with_text(HILDON_SIZE_FULLSCREEN_WIDTH | HILDON_SIZE_FINGER_HEIGHT, HILDON_BUTTON_ARRANGEMENT_VERTICAL, "Overview", NULL);
	g_signal_connect(button, "clicked", G_CALLBACK(on_overview_button_clicked), window);
	gtk_box_pack_start(GTK_BOX(vbox), button, FALSE, FALSE, 0);	
	
	button = hildon_button_new_with_text(HILDON_SIZE_FULLSCREEN_WIDTH | HILDON_SIZE_FINGER_HEIGHT, HILDON_BUTTON_ARRANGEMENT_VERTICAL, "Contacts", NULL);
	g_signal_connect(button, "clicked", G_CALLBACK(on_contact_button_clicked), window);
	gtk_box_pack_start(GTK_BOX(vbox), button, FALSE, FALSE, 0);	
/*
	button = hildon_button_new_with_text(HILDON_SIZE_FULLSCREEN_WIDTH | HILDON_SIZE_FINGER_HEIGHT, HILDON_BUTTON_ARRANGEMENT_VERTICAL, "Special feeds", NULL);
	g_signal_connect(button, "clicked", G_CALLBACK(on_special_feeds_button_clicked), window);
	gtk_box_pack_start(GTK_BOX(vbox), button, FALSE, FALSE, 0);	
*/

#endif
	gtk_widget_show_all(window);
}

int main(int argc, char** argv) {
	DBusError error;
	DBusConnection* connection;
	
	hildon_gtk_init(&argc, &argv);
	g_set_application_name("Mauku");

	dbus_error_init(&error);
	connection = dbus_bus_get(DBUS_BUS_SESSION, &error);
	dbus_connection_setup_with_g_main(connection, NULL);

	subscriber = microfeed_subscriber_new(SUBSCRIBER_IDENTIFIER, connection);

	create_main_window();	
	gtk_main();
	

	return 0;
}
