/*
  MathJinni - A simple formular calculator
  Copyright (C) 2007  Tim Teulings

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "UnitConversion.h"

#include <Lum/Base/String.h>

#include <Lum/Def/OneOfMany.h>

#include <Lum/Manager/Behaviour.h>

#include <Lum/Combo.h>
#include <Lum/Label.h>
#include <Lum/String.h>
#include <Lum/TextValue.h>

#include "Util.h"

UnitConversion::UnitConversion()
 : type(new Lum::Model::ULong()),
   types(new Lum::Model::StringTable()),
   src(new Lum::Model::ULong()),
   srcUnits(new Lum::Model::StringTable()),
   dest(new Lum::Model::ULong()),
   destUnits(new Lum::Model::StringTable()),
   value(new Lum::Model::String(L"")),
   result(new Lum::Model::String(L"")),
   variable(parser.AddVariable("x",0.0))
{
  Observe(type);
  Observe(src);
  Observe(dest);
  Observe(value);
}

void UnitConversion::CalcSize()
{
  Lum::Def::OneOfMany         typeDef(Lum::Def::Desc(L"Type"),
                                      type,types);
  Lum::Def::OneOfMany         srcDef(Lum::Def::Desc(L"Source unit"),
                                     src,srcUnits);
  Lum::Def::OneOfMany         destDef(Lum::Def::Desc(L"Destination unit"),
                                      dest,destUnits);

  Lum::Object              *oneOfMany;
  Lum::Label               *label;
  Lum::String              *string;
  Lum::TextValue           *textValue;

  label=new Lum::Label();

  oneOfMany=Lum::Manager::Behaviour::Instance()->GetOneOfManyControl(typeDef);
  oneOfMany->SetFlex(true,false);
  label->AddLabel(L"Type:",oneOfMany);

  oneOfMany=Lum::Manager::Behaviour::Instance()->GetOneOfManyControl(srcDef);
  oneOfMany->SetFlex(true,false);
  label->AddLabel(L"Source Unit:",oneOfMany);

  oneOfMany=Lum::Manager::Behaviour::Instance()->GetOneOfManyControl(destDef);
  oneOfMany->SetFlex(true,false);
  label->AddLabel(L"Destination Unit:",oneOfMany);

  string=new Lum::String();
  string->SetFlex(true,false);
  string->SetAlignment(Lum::String::right);
  string->SetInputTypeHint(Lum::Object::inputTypeSignedFloatNumber);
  string->SetModel(value);
  label->AddLabel(L"Value:",string);

  textValue=new Lum::TextValue();
  textValue->SetFlex(true,false);
  textValue->SetAlignment(Lum::TextValue::right);
  textValue->SetModel(result);
  label->AddLabel(L"Result:",textValue);

  container=label;

  Component::CalcSize();
}

void UnitConversion::Resync(Lum::Base::Model* model, const Lum::Base::ResyncMsg& msg)
{
  if (model==type && !type->IsNull()) {
    size_t idx=0;

    srcUnits->Off();
    destUnits->Off();

    srcUnits->Clear();
    for (size_t i=0; i<units[type->Get()-1]->GetAlternativeCount(); i++) {
      if (idx==0 &&
          units[type->Get()-1]->GetAlternative(i)->GetFrom()==L"x" &&
          units[type->Get()-1]->GetAlternative(i)->GetTo()==L"x") {
        idx=i+1;
      }

      srcUnits->Append(units[type->Get()-1]->GetAlternative(i)->GetName()+
                       L" ("+units[type->Get()-1]->GetAlternative(i)->GetSymbol()+L")");
    }

    destUnits->Clear();
    for (size_t i=0; i<units[type->Get()-1]->GetAlternativeCount(); i++) {
      destUnits->Append(units[type->Get()-1]->GetAlternative(i)->GetName()+
                        L" ("+units[type->Get()-1]->GetAlternative(i)->GetSymbol()+L")");
    }

    if (idx==0) {
      idx=1;
    }

    src->Set(idx);
    dest->Set(idx);

    srcUnits->On();
    destUnits->On();

    Convert();
  }
  else if (model==value || model==src || model==dest) {
    Convert();
  }
}

void UnitConversion::LoadData()
{
  for (size_t i=0; i<units.size(); i++) {
    types->Append(units[i]->GetType());
  }

  if (units.size()>0) {
    type->Set(1);
  }
}

void UnitConversion::LoadConfig(Lum::Config::Node *top)
{
  // no code
}

void UnitConversion::StoreConfig(Lum::Config::Node *top)
{
  // no code
}

void UnitConversion::Convert()
{
  if (value->IsNull() ||
      value->Empty() ||
      type->IsNull() ||
      src->IsNull() ||
      dest->IsNull()) {
    result->Set(L"");
    return;
  }

  Parser::Expression *expression;
  Parser::ReturnCode returnCode;
  size_t             pos;
  Alternative        *srcAlt=units[type->Get()-1]->GetAlternative(src->Get()-1);
  Alternative        *destAlt=units[type->Get()-1]->GetAlternative(dest->Get()-1);
  double             res;

  //
  // Get value to convert
  //

  expression=parser.Parse(Lum::Base::WStringToString(value->Get()),
                          returnCode,pos);
  if (expression==NULL) {
    result->Set(L"<ERROR>");
    return;
  }

  if (!expression->Calculate(res,returnCode)) {
    result->Set(L"<ERROR>");
    delete expression;
    return;
  }

  delete expression;

  //
  // Convert from source unit to standard unit
  //

  expression=parser.Parse(Lum::Base::WStringToString(srcAlt->GetFrom()),
                          returnCode,pos,false);
  if (expression==NULL) {
    result->Set(L"Internal error: src factor has parse error (code "+Lum::Base::NumberToWString((size_t)returnCode)+L")");

    return;
  }

  variable->value=res;

  if (!expression->Calculate(res,returnCode)) {
    result->Set(L"Internal error: src factor has calculation error (code "+Lum::Base::NumberToWString((size_t)returnCode)+L")");
    delete expression;
    return;
  }

  delete expression;

  //
  // Convert from standard unit to target unit
  //

  expression=parser.Parse(Lum::Base::WStringToString(destAlt->GetTo()),
                          returnCode,pos,false);
  if (expression==NULL) {
    result->Set(L"Internal error: dest factor has parse error (code "+Lum::Base::NumberToWString((size_t)returnCode)+L")");
    return;
  }

  variable->value=res;

  if (!expression->Calculate(res,returnCode)) {
    result->Set(L"Internal error: dest factor has calculation error (code "+Lum::Base::NumberToWString((size_t)returnCode)+L")");
    delete expression;
    return;
  }

  delete expression;

  result->Set(DoubleToWStringFloat(res));
}
