#ifndef PLOTTER_H
#define PLOTTER_H

/*
  MathJinni - A simple formular calculator
  Copyright (C) 2007  Tim Teulings

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
*/

#include <vector>

#include <Lum/Model/Table.h>

#include <Lum/OS/Color.h>

#include <Lum/Object.h>
#include <Lum/Table.h>

#include "Parser.h"

class Functions : public Lum::Model::Table
{
private:
  class Function
  {
  public:
    std::wstring        function;
    Parser::Expression  *expression;
    std::vector<double> values;

  public:
    Function(const std::wstring& function, Parser::Expression *expression);
  };

private:
  Parser::Variable      *plotVar;
  std::vector<Function> functions;
  double                top;
  double                bottom;
  double                left;
  double                right;

public:
  Functions(Parser::Variable* plotVar);
  size_t GetRows() const;

  std::wstring GetFunction(size_t row) const;

  Parser::Variable* GetPlotVar() const;
  Parser::Expression* GetExpression(size_t row) const;
  std::vector<double>& GetValue(size_t row);

  void AddFunction(const std::wstring& function, Parser::Expression* expression);
  void SetFunction(size_t row, const std::wstring& function, Parser::Expression* expression);
  void RemoveFunction(size_t row);
  void SetRegion(double top, double bottom,
                 double left, double right);

  double GetTop() const;
  double GetBottom() const;
  double GetLeft() const;
  double GetRight() const;
};

class FunctionsPainter : public Lum::StringTablePainter
{
public:
  std::wstring GetCellData() const
  {
    return dynamic_cast<const Functions*>(GetModel())->GetFunction(GetRow());
  }
};

typedef Lum::Base::Reference<Functions> FunctionsRef;

class Plotter : public Lum::Control
{
private:
  FunctionsRef        model;
  double              top;
  double              bottom;
  double              left;
  double              right;
  bool                areaSelection;
  size_t              assx,assy,asex,asey;
  Lum::OS::ColorRef   backgroundColor;
  Lum::OS::ColorRef   gridAColor;
  Lum::OS::ColorRef   gridBColor;
  Lum::OS::ColorRef   axisColor;
  Lum::OS::ColorRef   lineColor;

private:
  int TransformX(double x);
  double TransformX(size_t x);
  int TransformY(double y);
  double TransformY(size_t y);

  void RedrawSelectionArea(size_t assx, size_t assy,
                           size_t asex, size_t asey,
                           size_t oex, size_t oey);
  void HandleMouseMoveEvent(const Lum::OS::MouseEvent& event);
  void SetRegionFromAreaSelection();

public:
  Plotter();
  ~Plotter();

  bool SetModel(Lum::Base::Model* model);

  void CalcSize();
  void Draw(Lum::OS::DrawInfo* draw,
            int x, int y, size_t w, size_t h);

  void VAutosize();

  bool HandleMouseEvent(const Lum::OS::MouseEvent& event);

  void Resync(Lum::Base::Model* model, const Lum::Base::ResyncMsg& msg);

};

#endif
