# Copyright (C) 2008 by Daniel Martin Yerga
# <dyerga@gmail.com>
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

_version = "MaStory 2.0"

import gtk, hildon
import os
import gobject
import logging

logger = logging.getLogger("utils")
HOME = os.path.expanduser("~")

fhsize = gtk.HILDON_SIZE_FINGER_HEIGHT
horbtn = hildon.BUTTON_ARRANGEMENT_HORIZONTAL
asize = gtk.HILDON_SIZE_AUTO_HEIGHT
thsize = gtk.HILDON_SIZE_THUMB_HEIGHT

def is_treeview_selected(treeview):
    selection = treeview.get_selection()
    if selection.count_selected_rows() == 0:
        show_info_banner(treeview, 'No selected item')
        return False
    else:
        return True

def get_selected_from_treeview(treeview, model, setting):
    selection = treeview.get_selection()
    selected_model, selected_iter = selection.get_selected()
    if selected_iter:
        selected_name = model.get_value(selected_iter, setting)
    return selected_name, selected_iter, selected_model

def show_confirmation(window, msg):
    dialog = hildon.hildon_note_new_confirmation(window, msg)
    dialog.show_all()
    result = dialog.run()
    if result == gtk.RESPONSE_OK:
        dialog.destroy()
        return True

    dialog.destroy()
    return False

def show_info_dialog(window, msg):
    dialog = hildon.hildon_note_new_information(window, msg)
    dialog.show_all()
    dialog.run()
    dialog.destroy()

def show_info_banner(widget, msg):
    hildon.hildon_banner_show_information(widget, 'qgn_note_infoprint', msg)

def show_log_dialog(logfile):
    #Functions for the log
    def close_dialog(widget, dialog):
        dialog.destroy()
    def clear(widget, textview, logfile):
        textview.get_buffer().set_text('')
        f = open(logfile, 'w')
        f.close()
    def save(widget, logfile, dlg):
        import shutil
        filename = file_browser(dlg, gtk.FILE_CHOOSER_ACTION_SAVE,
                    "Save log file", HOME + '/MyDocs/.documents',
                    "mastory-log", "txt", False, False)
        if not filename:
            return

        try:
            shutil.copyfile(logfile, filename)
            show_info_banner(widget, 'Log file saved')
        except:
            logger.exception("Saving log file")
            show_info_banner(widget, 'Error saving the log file')

    def send(widget, dlg, logfile):
        sendtxt = ("You are going to send the log to the developers.\n"
        "This helps the developers to track problems with the application.\n"
        "It doesn't send any personal information (like passwords or similar).")

        dialog = hildon.hildon_note_new_confirmation(dlg, sendtxt)
        dialog.set_button_texts("Send", "Cancel")
        dialog.show_all()
        response = dialog.run()
        if response == gtk.RESPONSE_OK:
            do_pre_send(dlg, logfile)

        dialog.destroy()

    def do_pre_send(dlg, logfile):
        import thread
        hildon.hildon_gtk_window_set_progress_indicator(dlg, 1)
        thread.start_new_thread(_do_send, (dlg, logfile))

    def _do_send(dlg, logfile):
        import pycurl, shutil, random, commands
        try:
            rname = ''
            for i in random.sample('abcdefghijkl123456789', 18):
                rname += i

            rnamepath = HOME + "/.maemowordpy/" + rname
            shutil.copyfile(logfile, rnamepath)

            gtkversion = "%s.%s.%s" % gtk.ver
            if os.path.exists("/etc/maemo_version"):
                mfile = open("/etc/maemo_version", 'r')
                maemoversion = mfile.read()
                mfile.close()
            else:
                maemoversion = ''

            opsystem = ' '.join(os.uname())
            pyversion = os.sys.version
            pid = os.getpid()
            comm = ("awk '/Private_Dirty/{sum+=$2}END{print sum \"kB\"}'"
            " /proc/%s/smaps") % pid
            status, dirtymem = commands.getstatusoutput(comm)

            lfile = open(rnamepath, 'r')
            log = lfile.read()
            lfile.close()

            log = ("%s\nPython version: %s\nGtk version: %s\n"
            "Maemo version: %sOperating system: %s\n"
            "Dirty Memory: %s\nLog:\n%s") % (_version, pyversion, gtkversion,
            maemoversion, opsystem, dirtymem, log)

            lfile = open(rnamepath, 'w')
            lfile.write(log)
            lfile.close()

            url = "http://yerga.net/logs/uploader.php"
            data = [('uploadedfile', (pycurl.FORM_FILE, rnamepath)),]
            mycurl = pycurl.Curl()
            mycurl.setopt(pycurl.URL, url)
            mycurl.setopt(pycurl.HTTPPOST, data)

            mycurl.perform()
            mycurl.close()
            os.remove(rnamepath)

            gtk.gdk.threads_enter()
            show_info_banner(dlg, 'Log sent')
            gtk.gdk.threads_leave()
            hildon.hildon_gtk_window_set_progress_indicator(dlg, 0)
        except:
            logger.exception("Sending log file")
            gtk.gdk.threads_enter()
            show_info_banner(dlg, 'Error sending the log file')
            gtk.gdk.threads_leave()
            hildon.hildon_gtk_window_set_progress_indicator(dlg, 0)

    #Log dialog UI
    dialog = gtk.Dialog(title='Log', parent=None, flags=gtk.WIN_POS_CENTER)

    dialog.set_size_request(600, 350)

    parea = hildon.PannableArea()
    parea.set_property("mov-mode", hildon.MOVEMENT_MODE_BOTH)

    textview = hildon.TextView()
    textview.set_property("editable", False)
    textview.set_property("wrap-mode", gtk.WRAP_WORD)

    log = open(logfile, 'r')
    logtext = log.read()
    log.close()

    textview.get_buffer().set_text(logtext)
    parea.add(textview)

    dialog.vbox.pack_start(parea, True, True, 0)

    hbox = gtk.HBox()

    save_btn = hildon.Button(fhsize, horbtn)
    save_btn.set_title("Save")
    save_btn.connect('clicked', save, logfile, dialog)

    clear_btn = hildon.Button(fhsize, horbtn)
    clear_btn.set_title("Clear")
    clear_btn.connect('clicked', clear, textview, logfile)

    send_btn = hildon.Button(fhsize, horbtn)
    send_btn.set_title('Send')
    send_btn.connect('clicked', send, dialog, logfile)

    hbox.pack_start(save_btn, True, True, 0)
    hbox.pack_start(clear_btn, True, True, 0)
    hbox.pack_start(send_btn, True, True, 0)

    dialog.vbox.pack_start(hbox, False, False, 0)

    dialog.show_all()
    dialog.run()
    dialog.destroy()


def file_browser(window, action, title, folder, filename, file_ext, \
                multiple, addfolder):
    m = hildon.FileSystemModel()
    file_dialog = hildon.FileChooserDialog(window, action, m)
    file_dialog.set_title(title)
    if multiple:
        file_dialog.set_select_multiple(True)
    file_dialog.set_default_response(gtk.RESPONSE_CANCEL)
    file_dialog.set_current_folder(folder)
    if (action==gtk.FILE_CHOOSER_ACTION_SAVE):
        file_dialog.set_current_name(filename)

    if addfolder:
        file_dialog.add_button('Add entire\nfolder', gtk.RESPONSE_YES)

    file_dialog.show_all()
    result = file_dialog.run()

    if result == gtk.RESPONSE_OK:
        if multiple:
            namefile = file_dialog.get_filenames()
        else:
            namefile = file_dialog.get_filename()

        if (action==gtk.FILE_CHOOSER_ACTION_SAVE):
            namefile, extension = os.path.splitext(namefile)
            namefile = namefile + "." + file_ext

    elif result == gtk.RESPONSE_YES:
        namefile = file_browser(window, \
                        gtk.FILE_CHOOSER_ACTION_SELECT_FOLDER, \
                        "Select folder", folder, "", "", True, False)
    else:
        namefile = False
    file_dialog.destroy()

    return namefile

def file_to_pixbuf(filename, mode):
    import gtk.gdk
    new_pixbuf = gtk.gdk.pixbuf_new_from_file(filename)

    ## Scale pixbuf
    w, h = set_thumbnail_size(new_pixbuf, mode)
    image_pixbuf = gtk.gdk.pixbuf_new_from_file_at_size(filename, int(w), int(h))

    del new_pixbuf

    return image_pixbuf

def set_thumbnail_size(pixbuf, mode):
    pix_width = pixbuf.get_width()
    pix_height = pixbuf.get_height()

    ##Calculate thumbnail size for vertical and horizontal images
    if mode == 'small':
        if pix_width > pix_height:
            w = 70
            h = float(pix_height)/float(pix_width) * 70
        elif pix_height > pix_width:
            h = 70
            w = float(pix_width)/float(pix_height) * 70
        elif pix_height == pix_width:
            h = 70
            w = 70
        else:
            print "Pixbuf size can't be calculated."

    elif mode == 'edit':
        if pix_width > pix_height:
            w = 300
            h = float(pix_height)/float(pix_width) * 300
        elif pix_height > pix_width:
            h = 250
            w = float(pix_width)/float(pix_height) * 250
        elif pix_height == pix_width:
            h = 250
            w = 250
        else:
            print "Pixbuf size can't be calculated."

    return w, h

def get_selection_iters(txtview):
    """This function gets the start and end selection
    iters from the text view.  If there is no selection
    the current position of the cursor will be returned.
    Returns - start,end - gtk.TextIter objects"""

    start = None
    end = None

    txtview.grab_focus()
    txtBuffer = txtview.get_buffer()

    #First check to see that the text buffer is valid
    if (not txtBuffer):
        show_info_banner(txtview, "Text buffer not available")
        return start,end

    #Get the selection bounds
    bounds = txtBuffer.get_selection_bounds();
    if (bounds):
        #If there is a selection we are done
        start,end = bounds
    else:
        #There is no selection so just get the cursor mark
        cursor_mark = txtBuffer.get_insert()
        """Set start and end to be gtk.TextIter objercts at the
        position of the cursor mark"""
        start = txtBuffer.get_iter_at_mark(cursor_mark)
        end = txtBuffer.get_iter_at_mark(cursor_mark)

    return start, end

def insert_text(txtview, text):
    """This function inserts text into the text buffer
    self.txtBuffer at the current selection.  If text is
    selected it will be overwritten, otherwise it will simply be
    inserted at the cursor position
    text - The text to be inserted in the buffer
    """

    txtview.grab_focus()
    txtBuffer = txtview.get_buffer()

    start, end = get_selection_iters(txtview);

    if ((not start)or(not end)):
        show_info_banner(txtview, "Error inserting text")
        return;

    #Delete the selected text (start and end will be equal after)
    txtBuffer.delete(start,end)
    #Save a mark at the start position since after we insert
    #the text start will be invalid
    start_mark = txtBuffer.create_mark(None, start, True)
    #Insert, end will be set to the end insert position
    txtBuffer.insert(end,text)
    start = txtBuffer.get_iter_at_mark(start_mark)
    #select the text, use end as the first param so that
    #it will be the cursor position
    txtBuffer.select_range(end,start)
    #delete the start mark
    txtBuffer.delete_mark(start_mark)

def wrap_selection(txtview, start_tag, end_tag):
    """This fucntion is used to wrap the currently selected
    text in the gtk.TextView with start_tag and end_tag. If
    there is no selection start_tag and end_tag will be
    inserted at the cursor position
    start_tag - The text that will go at the start of the
    selection.
    end_tag - The text that will go at the end of the
    selection."""

    txtview.grab_focus()
    txtBuffer = txtview.get_buffer()

    start, end = get_selection_iters(txtview);
    if ((not start)or(not end)):
        show_info_banner(txtview, "Error inserting text")
        return;

    #Create a mark at the start and end
    start_mark = txtBuffer.create_mark(None,start, True)
    end_mark = txtBuffer.create_mark(None, end, False)
    #Insert the start_tag
    txtBuffer.insert(start, start_tag)
    #Get the end iter again
    end = txtBuffer.get_iter_at_mark(end_mark)
    #Insert the end tag
    txtBuffer.insert(end, end_tag)
    #Get the start and end iters
    start = txtBuffer.get_iter_at_mark(start_mark)
    end = txtBuffer.get_iter_at_mark(end_mark)
    #Select the text
    txtBuffer.select_range(end,start)
    #Delete the gtk.TextMark objects
    txtBuffer.delete_mark(start_mark)
    txtBuffer.delete_mark(end_mark)

def preview_img(filename):
    import gc
    view_image_dlg = gtk.Dialog()

    view_image_dlg.set_title('View cached image')

    view_img = gtk.Image()
    view_image_dlg.vbox.pack_start(view_img)

    scale_pix = file_to_pixbuf(filename, 'edit')

    view_img.set_from_pixbuf(scale_pix)
    view_image_dlg.show_all()

    del scale_pix
    gc.collect()

    view_image_dlg.run()
    view_image_dlg.destroy()

#This is a finger friendly number editor
#It's a simple gtk.Entry with two gtk.Button
class FingerNE(gtk.HBox):
    def __init__ (self, orientation):
        gtk.HBox.__init__ (self)

        self.step = 1
        self.minimum = False
        self.maximum = False
        self.entry = hildon.Entry(fhsize)
        self.entry.unset_flags(gtk.CAN_FOCUS)
        self.entry.set_property('editable', False)
        self.entry.set_property('xalign', 0.5)
        self.entry.set_text(str(0))

        self.plus = hildon.Button(thsize, horbtn)
        self.plus.connect('clicked', self.plus_clicked)
        self.plus.connect('pressed', self.press_button, 'more')
        self.plus.connect('released', self.release_button)

        self.minus = hildon.Button(thsize, horbtn)
        self.minus.connect('clicked', self.minus_clicked)
        self.minus.connect('pressed', self.press_button, 'minus')
        self.minus.connect('released', self.release_button)

        self.mybox = self.set_orientation(orientation)

        self.add(self.mybox)

    #Set the image for the buttons, generally + and -
    def set_image_buttons(self, plus_imgfile, minus_imgfile):
        minus_img = gtk.Image()
        minus_img.set_from_file(minus_imgfile)
        self.minus.set_image(minus_img)

        plus_img = gtk.Image()
        plus_img.set_from_file(plus_imgfile)
        self.plus.set_image(plus_img)

        minus_img.show()
        plus_img.show()
        self.minus.show()
        self.plus.show()

    def do_realize (self):
        gtk.HBox.do_realize (self)

    def press_button(self, widget, option):
        self.start_spinning(widget, option)

    def release_button(self, widget):
        self.stop_spinning()

    def stop_spinning(self):
        if (self.timer > 0):
            gobject.source_remove(self.timer)
        self.timer = 0
        self.need_timer = False

    def start_spinning(self, widget, option):
        self.timer_step = self.step
        self.need_timer = True
        self.timer = gobject.timeout_add(50, self.spin_timer, widget, option)

    def spin_timer(self, widget, option):
        is_min = False
        is_max = False
        if self.timer:
            if option == 'more':
                new_value = int(self.get_value()) + self.step
                is_max = self.is_maximum(widget, new_value)
            elif option == 'minus':
                new_value = int(self.get_value()) - self.step
                is_min = self.is_minimum(widget, new_value)

            if is_min or is_max:
                self.stop_spinning()
            else:
                self.timer = gobject.timeout_add(50, self.spin_timer, \
                                                widget, option)

        if self.need_timer:
            self.need_timer = False
            self.timer = gobject.timeout_add(50, self.spin_timer, \
                                            widget, option)

    #When the + button is clicked
    def plus_clicked(self, widget):
        self.is_maximum(widget, self.get_value())

    #When the - button is clicked
    def minus_clicked(self, widget):
        self.is_minimum(widget, self.get_value())

    def is_maximum(self, widget, new_value):
        if self.maximum is not False:
            if int(new_value) >= self.maximum:
                show_info_banner(widget, 'Maximum value reached')
                return True
            else:
                try:
                    self.set_value(int(new_value))
                    return False
                except ValueError:
                    self.set_value(0)
                    show_info_banner(widget, 'Invalid literal value')
                    return True
        else:
            try:
                self.set_value(int(new_value))
                return False
            except ValueError:
                self.set_value(0)
                show_info_banner(widget, 'Invalid literal value')
                return True

    def is_minimum(self, widget, new_value):
        if self.minimum is not False:
            if int(new_value) <= self.minimum:
                show_info_banner(widget, 'Minimum value reached')
                return True
            else:
                try:
                    self.set_value(int(new_value))
                    return False

                except ValueError:
                    self.set_value(self.minimum)
                    show_info_banner(widget, 'Invalid literal value')
                    return True
        else:
            try:
                self.set_value(int(new_value))
                return False

            except ValueError:
                self.set_value(self.minimum)
                show_info_banner(widget, 'Invalid literal value')
                return True

    #Maximum value for the widget
    def set_maximum(self, maximum):
        self.maximum = maximum + 1

    #Minimum value for the widget
    def set_minimum(self, minimum):
        self.minimum = minimum - 1

    #Step for the widget
    def set_step(self, step):
        self.step = step

    #Set the size for the entry and the buttons
    def set_size(self, size):
        self.entry.set_size_request(size[0], size[1])
        self.plus.set_size_request(size[2], size[3])
        self.minus.set_size_request(size[2], size[3])

    #Set the value for the entry
    def set_value(self, value):
        self.entry.set_text(str(value))

    #Return the value for the entry
    def get_value(self):
        value = self.entry.get_text()
        return value

    #Orientation for the widget, vertical or horizontal
    def set_orientation(self, orientation):
        if orientation == 'horizontal':
            mybox = gtk.HBox()
            mybox.pack_start(self.minus, False, False, 0)
            mybox.pack_start(self.entry, False, False, 0)
            mybox.pack_start(self.plus, False, False, 0)

        elif orientation == 'vertical':
            mybox = gtk.VBox()
            mybox.pack_start(self.plus, False, False, 0)
            mybox.pack_start(self.entry, False, False, 0)
            mybox.pack_start(self.minus, False, False, 0)

        return mybox

    def show(self):
        self.mybox.show_all()
