#!/usr/bin/env python
# -*- coding: UTF-8 -*-
#
#    stopcode.py
#
#    This file is part of Mass-Transit
#
#    Copyright (C) 2010 Thibault Cohen
#
#    This program is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program; if not, write to the Free Software
#    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#

from lib import lib_masstransit as masstransit
from PyQt4 import QtCore, QtGui
from module import stm
from datetime import datetime, timedelta

class SchedulePushButton(QtGui.QPushButton):
    def __init__(self, label,schedule_time, line, parent=None):
        """ Schedule Push button keep line and schedule time
        to create an alarm
        """
        QtGui.QPushButton.__init__(self, label, parent)
        self.parent = parent
        self.schedule_time = schedule_time
        self.line = line

        QtCore.QObject.connect( self,
                                QtCore.SIGNAL("clicked()"),
                                self.call_add_alarm,
                                )

    def call_add_alarm(self):
        """ launch alarm creation
        """
        self.parent.add_alarm(self.schedule_time,self.line)

class Ui_StopCode(QtGui.QMainWindow):
    def __init__(self, stop_code, force_update=False, parent=None, root=None):
        """ Ui for bus stop
        """
        QtGui.QMainWindow.__init__(self, parent=parent)
        self.parent = parent
        self.root = root
        self.stop_code = QtCore.QString(stop_code)
        self.setup_ui()

    def setup_ui(self, force_update=False):
        self.scroll_area = QtGui.QScrollArea(self)
        self.widget = QtGui.QWidget(self.scroll_area)
        self.scroll_area.setWidget(self.widget)


        self.setAttribute(QtCore.Qt.WA_Maemo5StackedWindow, True)
        self.setCentralWidget(self.scroll_area)
        self.setWindowTitle("Mass-Transit - Bus Stop Detail")

        grid = QtGui.QGridLayout()

        # Get favorite list
        self.favorite_stop_codes = self.root.config.get_favorites_stop_code()

        if self.stop_code in self.favorite_stop_codes.keys() \
            and force_update == False:
            # Get favorite detail
            self.stop_name,self.schedules = self.root.config.get_favorite_stop_code_detail(self.stop_code)

            now = QtCore.QDateTime.currentDateTime()

            # Check if we need an update
            uptodate = all([t < now for s in self.schedules for t in s[1] if t !=None ])

            # Update schedules
            if uptodate:
                self.stop_name,self.schedules = stm.getBusStopSchedule(unicode(self.stop_code))
                self.root.config.delete_favorite_stop_code(self.stop_code)
                self.root.config.add_favorite_stop_code(self.stop_code,self.stop_name, self.schedules)
        else:
            # Get schedules for non favorite stop bus
            self.stop_name,self.schedules = stm.getBusStopSchedule(unicode(self.stop_code))
        
        # Create first line
        self.stop_name_label = QtGui.QLabel(self.stop_code + " - " + self.stop_name)
        grid.addWidget(self.stop_name_label,0,0,1,3)
        if self.stop_code in self.favorite_stop_codes.keys():
            self.stop_favorite_button = QtGui.QPushButton("Delete from favorites")
        else:
            self.stop_favorite_button = QtGui.QPushButton("Add to favorites")
        grid.addWidget(self.stop_favorite_button,0,3,1,2)
        self.update_button = QtGui.QPushButton("Update")
        grid.addWidget(self.update_button,0,5,1,1)

        # Fill lines
        l = 1
        for schedule in self.schedules:
            line = schedule[0][0]
            direction = schedule[0][1]
            schedule_times = schedule[1]
            group_box = QtGui.QGroupBox("Line %s - %s" % (  line,
                                                            direction,
                                                            )
                                        ) 
            # Group Box
            grid_box = QtGui.QGridLayout()
            t = 0
            # Fill schedule by lines
            for schedule_time in schedule_times:
                if not schedule_time:
                    time_label = QtGui.QLabel(" ")
                    grid_box.addWidget(time_label,0,t)
                    t += 1
                else:
                    schedule_time_label = schedule_time.toString("hh") + \
                            "H" + \
                            schedule_time.toString("mm")
                    time_button = SchedulePushButton(   schedule_time_label,
                                                        schedule_time,
                                                        line,
                                                        self)
                    grid_box.addWidget(time_button,0,t)
                    t += 1

            group_box.setLayout(grid_box)
            grid.addWidget(group_box,l,0,1,6)
            l += 1

        heigth = 100 * ( l + 1 )
        self.widget.setLayout(grid)
        self.widget.resize(790, heigth)

        ## Signals
        ### add/delete favorite
        QtCore.QObject.connect( self.stop_favorite_button,
                                QtCore.SIGNAL("clicked()"),
                                self.edit_favorite
                                )
        ### fetch datas
        QtCore.QObject.connect( self.update_button,
                                QtCore.SIGNAL("clicked()"),
                                self.update_datas
                                )



    def edit_favorite(self):
        """ Add or delete stop bus favorite
        """ 
        if self.stop_favorite_button.text() == "Delete from favorites":
            self.root.config.delete_favorite_stop_code(self.stop_code)
            self.stop_favorite_button.setText("Add to favorites")
            masstransit.banner_notification("Favorite deleted")
            # Update favorites
            self.favorite_stop_codes = self.root.config.get_favorites_stop_code()
        else:
            if not self.stop_code in self.favorite_stop_codes:
                self.root.config.add_favorite_stop_code(self.stop_code,self.stop_name, self.schedules)
                self.stop_favorite_button.setText("Delete from favorites")
                masstransit.banner_notification("Favorite saved")
                # Update favorites
                self.favorite_stop_codes = self.root.config.get_favorites_stop_code()

    def update_datas(self):
        """ Update data from website
        """
        self.setAttribute(QtCore.Qt.WA_Maemo5ShowProgressIndicator, True)
        self.setup_ui(force_update=True)
        self.setAttribute(QtCore.Qt.WA_Maemo5ShowProgressIndicator, False)


    def add_alarm(self, schedule, line):
        minute_list = [unicode(i) + " minutes before" for i in range(1,61)]
        minute_str,bool = QtGui.QInputDialog().getItem(
                                self.root.main_window,
                                "Create alarm",
                                "Select how many minutes before you want to be informed",
                                minute_list,
                                4,
                                False,
                                )
        minute = minute_str.split(" ")[0]
        if not bool:
            return False
        else:
            try:
                minute = int(minute)
            except:
                return False 
        alarm_date = schedule.toPyDateTime() - timedelta(minutes=minute)
        now = datetime.now()
        if alarm_date > now:
            alarm_delta = alarm_date - now
            message = "Bus %s is coming at %sH%s!!!" % (line,
                                            schedule.toPyDateTime().hour,
                                            schedule.toPyDateTime().minute)
            delta_minute = alarm_delta.seconds / 60
            alarm_time = schedule.addSecs(-delta_minute)
            masstransit.add_alarm(message, alarm_delta.seconds)

            masstransit.banner_notification("Alarm added. Bell will ring at %sH%s" % (
                                                alarm_time.toPyDateTime().hour,
                                                alarm_time.toPyDateTime().minute,
                                                )
                                            )
        else:
            masstransit.banner_notification("Too late, you should may be running") 


    def closeEvent(self, event):
        self.root.mode = "favorites"

