/*
 * This file is part of mapper
 *
 * Copyright (C) 2007 Kaj-Michael Lang
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */
#include <config.h>

#include <libgnomevfs/gnome-vfs.h>

#include "utils.h"
#include "gps.h"
#include "map.h"
#include "route.h"
#include "settings.h"
#include "mapper-types.h"
#include "ui-common.h"
#include "file.h"
#include "mapper.h"
#include "track.h"
#include "latlon.h"
#include "path.h"
#include "gpx.h"
#include "dialogs.h"

typedef struct _TrackMarkDialog TrackMarkDialog;
struct _TrackMarkDialog {
	GtkWidget *dialog;
	GtkWidget *pos_label;
	GtkWidget *txt_desc;
};

void 
track_clear(Path *path)
{
GtkWidget *confirm;

confirm = hildon_note_new_confirmation(GTK_WINDOW(mapp.mainwindow), _("Clear the track?"));

if (GTK_RESPONSE_OK == gtk_dialog_run(GTK_DIALOG(confirm))) {
	path_clear(path);
	map_force_redraw();
}
gtk_widget_destroy(confirm);
}

void 
track_show_distance_from(Path *path, Point *point)
{
gchar buffer[80];
gdouble sum;

sum=path_calculate_distance_from(path, point, _gps->data.lat, _gps->data.lon);
g_snprintf(buffer, sizeof(buffer), "%s: %.02f %s", _("Distance"), sum * UNITS_CONVERT[_units], UNITS_TEXT[_units]);
MACRO_BANNER_SHOW_INFO(mapp.mainwindow, buffer);
}

void 
track_show_distance_from_last(Path *path)
{
Point *point;

if (path->head==path->tail) {
	MACRO_BANNER_SHOW_INFO(mapp.mainwindow, _("The current track is empty."));
	return;
}
/* Find last zero point. */
for (point = path->tail; point->unity; point--) {
}
track_show_distance_from(path, point);
}

void 
track_show_distance_from_first(Path *path)
{
if (path->head==path->tail) {
	MACRO_BANNER_SHOW_INFO(mapp.mainwindow, _("The current track is empty."));
	return;
}
track_show_distance_from(path, path->head);
}

/**
 * Add a point to the _track list.
 *
 * If a non-null gps is given, then the current position is appended 
 * to _track with the given time.  If gps is null, then _point_null is 
 * appended to _track with time zero (this produces a "break" in the track).
 */
gboolean
track_add(GpsData *gps)
{
if (!gps) {
	MACRO_PATH_INCREMENT_TAIL(*_track);
	*_track->tail=_point_null;
	return FALSE;
}

/* If gps is available, update the nearest-waypoint data. */
if (gps && _route->head != _route->tail && (gps->newly_fixed ? (route_find_nearest_point(), TRUE) : route_update_nears(TRUE))) {
	/* Nearest waypoint has changed - re-render paths. */
	map_render_paths();
	mapper_info_banner_clear_waypoint_direction();
	MACRO_QUEUE_DRAW_AREA();
}

if (_show_tracks & TRACKS_MASK) {
	gint tx1, ty1, tx2, ty2;

	/* Instead of calling map_render_paths(), we'll draw the new line
	 * ourselves and call gtk_widget_queue_draw_area(). */
	map_render_segment(_gc[COLORABLE_TRACK], _gc[COLORABLE_TRACK_BREAK], 
		_track->tail->unitx, _track->tail->unity, gps->unitx, gps->unity);

	if (_track->tail->unity && _track->tail->unitx) {
		tx1 = unit2x(_track->tail->unitx);
		ty1 = unit2y(_track->tail->unity);
		tx2 = unit2x(gps->unitx);
		ty2 = unit2y(gps->unity);

		/* XXX: This should not be here... */
		gtk_widget_queue_draw_area(mapp.map_widget,
				   MIN(tx1, tx2) - _draw_width, 
				   MIN(ty1, ty2) - _draw_width,
				   abs(tx1 - tx2) + (2 * _draw_width),
				   abs(ty1 - ty2) + (2 * _draw_width));
	}
}

if (_track->tail->unity && _track->tail->unitx) {
	gdouble lat, lon;

	unit2latlon(_track->tail->unitx, _track->tail->unity, lat, lon);
	_track->length += calculate_distance(lat, lon, gps->lat, gps->lon);
	if (_track->points>0)
		_track->avgspeed=(_track->points*_track->avgspeed+gps->speed)/(_track->points+1);
	else
		_track->avgspeed=0.0;
}

MACRO_PATH_INCREMENT_TAIL(*_track);
_track->tail->unitx=gps->unitx;
_track->tail->unity=gps->unity;
_track->tail->time=gps->time;
_track->tail->altitude=gps->altitude;
_track->maxspeed=gps->maxspeed;
_track->points++;

if (G_UNLIKELY(_track->points==1))
	path_insert_mark_text(_track, g_strdup("Start"));

/* Keep the display on if we are moving. */
keep_display_on(&mapper_app);
return TRUE;
}

gboolean 
track_insert_break(Path *path)
{
if (path->tail->unity) {
	guint x1, y1;

	/* To mark a "waypoint" in a track, we'll add a (0, 0) point and then
	 * another instance of the most recent track point. */
	MACRO_PATH_INCREMENT_TAIL(*path);
	*path->tail=_point_null;
	MACRO_PATH_INCREMENT_TAIL(*path);
	*path->tail=_track->tail[-2];

	/* Instead of calling map_render_paths(), we'll just draw the waypoint ourselves. */
	x1=unit2bufx(path->tail->unitx);
	y1=unit2bufy(path->tail->unity);
	map_render_waypoint(x1, y1, _gc[COLORABLE_TRACK_BREAK]);
}
return FALSE;
}

gboolean
track_open(void)
{
gchar *buffer;
guint size;
gboolean r = FALSE;

if (file_open_get_contents(&_track_file_uri, &buffer, &size)) {
	if (gpx_parse(_track, buffer, size, GPX_PATH_NEW)) {
		map_force_redraw();
		MACRO_BANNER_SHOW_INFO(mapp.mainwindow, _("Track Opened"));
		r = TRUE;
	} else {
		popup_error(mapp.mainwindow, _("Error parsing GPX file."));
	}
	g_free(buffer);
}
return r;
}

gboolean
track_save(Path *path)
{
GnomeVFSHandle *handle;
gboolean r = FALSE;

if (file_save(&_track_file_uri, &_track_file_uri, &handle)) {
	if (gpx_write(path, handle)) {
		MACRO_BANNER_SHOW_INFO(mapp.mainwindow, _("Track Saved"));
		r = TRUE;
		track_clear(path);
	} else {
		popup_error(mapp.mainwindow, _("Error writing GPX file."));
	}
	gnome_vfs_close(handle);
}
return r;
}

static gboolean
track_ui_dialog_update_pos_cb(GtkWidget *label)
{
gchar tmp1[16], tmp2[16], *p_latlon;

lat_format(_degformat, _gps->data.lat, tmp1);
lon_format(_degformat, _gps->data.lon, tmp2);
p_latlon=g_strdup_printf("%s, %s", tmp1, tmp2);
gtk_label_set_text(GTK_LABEL(label), p_latlon);
g_free(p_latlon);
return TRUE;
}

static TrackMarkDialog *
track_ui_dialog_mark_new(GtkWindow *mw)
{
TrackMarkDialog *tmd;
GtkWidget *table, *label, *txt_scroll;
tmd=g_slice_new0(TrackMarkDialog);

tmd->dialog=gtk_dialog_new_with_buttons(_("Insert Mark"),
				mw, GTK_DIALOG_MODAL, 
				GTK_STOCK_OK, GTK_RESPONSE_ACCEPT,
				GTK_STOCK_CANCEL, GTK_RESPONSE_REJECT, 
				NULL);

gtk_box_pack_start(GTK_BOX(GTK_DIALOG(tmd->dialog)->vbox), table = gtk_table_new(2, 2, FALSE), TRUE, TRUE, 0);

gtk_table_attach(GTK_TABLE(table), label = gtk_label_new(_("Lat, Lon")), 0, 1, 0, 1, GTK_FILL, 0, 2, 4);
gtk_misc_set_alignment(GTK_MISC(label), 1.f, 0.5f);

gtk_table_attach(GTK_TABLE(table), tmd->pos_label = gtk_label_new(NULL), 1, 2, 0, 1, GTK_FILL, 0, 2, 4);
gtk_misc_set_alignment(GTK_MISC(tmd->pos_label), 0.0f, 0.5f);

gtk_table_attach(GTK_TABLE(table), label = gtk_label_new(_("Description")), 0, 1, 1, 2, GTK_FILL, 0, 2, 4);
gtk_misc_set_alignment(GTK_MISC(label), 1.f, 0.5f);

txt_scroll=gtk_scrolled_window_new(NULL, NULL);
gtk_scrolled_window_set_shadow_type(GTK_SCROLLED_WINDOW(txt_scroll), GTK_SHADOW_IN);
gtk_table_attach(GTK_TABLE(table), txt_scroll, 1, 2, 1, 2, GTK_EXPAND | GTK_FILL, 0, 2, 4);

gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(txt_scroll), GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);

tmd->txt_desc=gtk_text_view_new();
gtk_text_view_set_wrap_mode(GTK_TEXT_VIEW(tmd->txt_desc), GTK_WRAP_WORD);

gtk_container_add(GTK_CONTAINER(txt_scroll), tmd->txt_desc);

gtk_widget_set_size_request(GTK_WIDGET(tmd->dialog), 640, 200);

return tmd;
}

/**
 * Ask for a text description for the current point
 *
 */
gboolean
track_insert_mark(Path *path)
{
static TrackMarkDialog *tmd=NULL;
gboolean ret;
gint s, res;
GtkTextBuffer *tbuf;
GtkTextIter ti1, ti2;

if (!tmd)
	tmd=track_ui_dialog_mark_new(GTK_WINDOW(mapp.mainwindow));

tbuf=gtk_text_view_get_buffer(GTK_TEXT_VIEW(tmd->txt_desc));
gtk_text_buffer_get_start_iter(tbuf, &ti1);
gtk_text_buffer_get_end_iter(tbuf, &ti2);
gtk_text_buffer_select_range(tbuf, &ti1, &ti2);
gtk_widget_grab_focus(tmd->txt_desc);

track_ui_dialog_update_pos_cb(tmd->pos_label);
s=g_timeout_add(1000, (GSourceFunc *)track_ui_dialog_update_pos_cb, tmd->pos_label);

gtk_widget_show_all(tmd->dialog);

ret=FALSE;
res=gtk_dialog_run(GTK_DIALOG(tmd->dialog));
if (res==GTK_RESPONSE_ACCEPT) {
	gtk_text_buffer_get_start_iter(tbuf, &ti1);
	gtk_text_buffer_get_end_iter(tbuf, &ti2);

	if (gtk_text_buffer_get_char_count(tbuf)>0) {
		path_insert_mark_text(path, gtk_text_buffer_get_text(tbuf, &ti1, &ti2, TRUE));
	} else {
		path_insert_mark_text(path, g_strdup("MK"));
	}
	ret=TRUE;
}
g_source_remove(s);
gtk_widget_hide(tmd->dialog);
return ret;
}
