#ifndef _LATLON_H
#define _LATLON_H

#include <glib.h>

/* Int ranges for integerized lat/lon */
#define LATLON_MAX 2147483646
#define LATLON_MIN -2147483646
#define EARTH_RADIUS (3440.06479f)

#define NM_TO_KM (1.85200)
#define NM_TO_MI (1.15077945)
#define NM_TO_NM (1.0)
#define NM_TO_FT (6.0761155)

#define MERCATOR_SPAN (-6.28318377773622f)
#define MERCATOR_TOP (3.14159188886811f)

/** MAX_ZOOM defines the largest map zoom level we will download.
 * (MAX_ZOOM - 1) is the largest map zoom level that the user can zoom to.
 */
#define MAX_ZOOM 16

#define TILE_SIZE_PIXELS (256)
#define TILE_SIZE_P2 (8)

#define WORLD_SIZE_UNITS (2 << (MAX_ZOOM + TILE_SIZE_P2))

#define WORLD_SIZE_UNITS (2 << (MAX_ZOOM + TILE_SIZE_P2))

#define latlon2unit(lat, lon, unitx, unity) { \
	gdouble tmp; \
	unitx = (lon + 180.0) * (WORLD_SIZE_UNITS / 360.0) + 0.5; \
	tmp = sin(lat * (M_PIl / 180.0)); \
	unity = 0.5f + (WORLD_SIZE_UNITS / MERCATOR_SPAN) \
		* (log((1.0 + tmp) / (1.0 - tmp)) * 0.5 - MERCATOR_TOP); \
}

#define unit2latlon(unitx, unity, lat, lon) { \
	(lon) = ((unitx) * (360.0 / WORLD_SIZE_UNITS)) - 180.0; \
	(lat) = (360.0 * (atan(exp(((unity) * (MERCATOR_SPAN / WORLD_SIZE_UNITS)) \
		+ MERCATOR_TOP)))) * (1.0 / M_PIl) - 90.0; \
}

typedef enum {
	DDPDDDDD,
	DD_MMPMMM,
	DD_MM_SSPS,
	DDPDDDDD_NSEW,
	DD_MMPMMM_NSEW,
	DD_MM_SSPS_NSEW,
	DEG_FORMAT_ENUM_COUNT
} DegFormat;
gchar *DEG_FORMAT_TEXT[DEG_FORMAT_ENUM_COUNT];

/** This enum defines the possible units we can use. */
typedef enum {
	UNITS_KM,
	UNITS_MI,
	UNITS_NM,
	UNITS_FT,
	UNITS_ENUM_COUNT
} UnitType;
gchar *UNITS_TEXT[UNITS_ENUM_COUNT];

/* UNITS_CONVERTS, when multiplied, converts from NM. */
#define EARTH_RADIUS (3440.06479f)
gfloat UNITS_CONVERT[UNITS_ENUM_COUNT];

UnitType _units;

#define lat_format(F, A, B) deg_format((F),(A), (B), 'S', 'N')
#define lon_format(F, A, B) deg_format((F),(A), (B), 'W', 'E')

void latlon_init(void);

gint32 lon2mp_int(gdouble lon);
gint32 lat2mp_int(gdouble lat);

gdouble calculate_distance(gdouble lat1, gdouble lon1, gdouble lat2, gdouble lon2);
gdouble calculate_ddistance(gint lat1, gint lon1, gint lat2, gint lon2);
gulong calculate_idistance(gint lat1, gint lon1, gint lat2, gint lon2);
gulong calculate_idistance_cmp(gint lat1, gint lon1, gint lat2, gint lon2);

gdouble calculate_course_rad(gdouble lat1, gdouble lon1, gdouble lat2, gdouble lon2);
gdouble calculate_course(gdouble lat1, gdouble lon1, gdouble lat2, gdouble lon2);

gfloat angle_diff(gfloat a, gfloat b);

gfloat speed_convert(gfloat speed, UnitType t);

gboolean distance_point_to_line(gdouble x, gdouble y, gdouble x1, gdouble y1, gdouble x2, gdouble y2, gdouble *d);

guint32 xy2tile(guint x, guint y);
guint32 lon2x(gdouble lon);
guint32 lat2y(gdouble lat);

gint speed_per_hour_str(gchar *buffer, size_t size, gfloat speed, UnitType t);
gint distance_str(gchar *buffer, size_t size, gdouble lat1, gdouble lon1, gdouble lat2, gdouble lon2, UnitType t);

void deg_format(DegFormat degformat, gdouble coor, gchar *scoor, gchar neg_char, gchar pos_char);

#endif
