/*
 * This file is part of mapper
 *
 * Generic sqlite database helper functions.
 *
 * Copyright (C) 2007-2009 Kaj-Michael Lang
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#include "config.h"

#include <glib.h>
#include <sqlite3.h>

#include "db.h"

#ifdef WITH_DEVICE_MAEMO
#define SQLITE_CACHE_SIZE "6000"
#else
#define SQLITE_CACHE_SIZE "32000"
#endif

sqlite3 *_db=NULL;
gchar *_mapper_db=NULL;

gboolean
db_exec_sql_array(sqlite3 *db, const gchar **sql)
{
gboolean r=TRUE;
const char **d;

d=sql;
while (*d!=NULL) {
	if (db_exec_sql(db, *d)==FALSE)
		r=FALSE;
	d++;
}
return r;
}

/**
 * A simple check to test if a table exists
 *
 */
gboolean 
db_check(sqlite3 *db, const gchar *table)
{
gchar **pszResult;
gint nRow, nColumn;
gchar sql[64];

g_return_val_if_fail(db, FALSE);
g_return_val_if_fail(table, FALSE);

g_snprintf(sql, sizeof(sql),"select count(*) from %s", table);

if (SQLITE_OK != sqlite3_get_table(db, sql, &pszResult, &nRow, &nColumn, NULL))
	return FALSE;

sqlite3_free_table(pszResult);
return TRUE;
}

gboolean
db_exec_sql(sqlite3 *db, const gchar *sql)
{
gint r;

g_return_val_if_fail(db, FALSE);
g_return_val_if_fail(sql, FALSE);

r=sqlite3_exec(db, sql, NULL, NULL, NULL);
if (r!=SQLITE_OK && r!=SQLITE_DONE)
	g_warning("SQL ERROR %d:(%s) %s\n", r, sql, sqlite3_errmsg(db));

return (r==SQLITE_OK || r==SQLITE_DONE) ? TRUE : FALSE;
}

/**
 * Helper that execs a given prepared sql statment and resets clears bindings.
 * return TRUE or FALSE depending on success of the query. 
 *
 */
gboolean
db_exec(sqlite3 *db, sqlite3_stmt *sql)
{
gint r;

g_return_val_if_fail(db, FALSE);
g_return_val_if_fail(sql, FALSE);

r=sqlite3_step(sql);
if (r!=SQLITE_OK && r!=SQLITE_DONE)
	g_printerr("SQL ERROR %d: %s\n", r, sqlite3_errmsg(db));
sqlite3_reset(sql);
sqlite3_clear_bindings(sql);

return (r==SQLITE_OK || r==SQLITE_DONE) ? TRUE : FALSE;
}

/**
 * Close the database connection
 *
 */
gboolean 
db_close(sqlite3 **db)
{
if (db && *db) {
	sqlite3_close(*db);
	*db=NULL;
	return TRUE;
}
return FALSE;
}

gboolean
db_transaction_begin(sqlite3 *db)
{
return db_exec_sql(db, "begin;");
}

gboolean
db_transaction_commit(sqlite3 *db)
{
return db_exec_sql(db, "commit;");
}

/**
 * Connect to given sqlite database
 *
 */
gboolean 
db_connect(sqlite3 **db, const gchar *mapper_db)
{
if (db_close(db))
	g_warning("Re-opening database.");

if (!mapper_db)
	return FALSE;

if (SQLITE_OK!=(sqlite3_open(mapper_db, db))) {
	g_warning("SQL OPEN: %s\n", sqlite3_errmsg(*db));
	sqlite3_close(*db);
	*db=NULL;
	return FALSE;
}

sqlite3_exec(*db, "PRAGMA synchronous=0;", NULL, NULL, NULL);
sqlite3_exec(*db, "PRAGMA page_size=4096;", NULL, NULL, NULL);
sqlite3_exec(*db, "PRAGMA legacy_file_format=false;", NULL, NULL, NULL);
sqlite3_exec(*db, "PRAGMA encoding=\"UTF-8\";", NULL, NULL, NULL);
sqlite3_exec(*db, "PRAGMA cache_size="SQLITE_CACHE_SIZE";", NULL, NULL, NULL);

return TRUE;
}

