/*
 * This file is part of mapper
 *
 * Copyright (C) 2007 Kaj-Michael Lang
 * Copyright (C) 2006-2007 John Costigan.
 *
 * POI and GPS-Info code originally written by Cezary Jackiewicz.
 *
 * Default map data provided by http://www.openstreetmap.org/
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#include <config.h>
#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <strings.h>
#include <stddef.h>
#include <math.h>
#include <errno.h>
#include <sys/wait.h>
#include <glib.h>
#include <glib/gstdio.h>
#include <glib/gi18n.h>
#include <gst/gst.h>
#include <gtk/gtk.h>
#include <fcntl.h>
#include <gdk/gdkkeysyms.h>
#include <dbus/dbus.h>
#include <dbus/dbus-glib.h>
#include <dbus/dbus-glib-lowlevel.h>
#include <libgnomevfs/gnome-vfs.h>
#include <curl/multi.h>
#include <gconf/gconf-client.h>
#include <libintl.h>
#include <locale.h>

#ifdef WITH_HAL
#include <libhal.h>
#endif

#ifdef WITH_HILDON_DBUS_BT
#include <bt-dbus.h>
#define _BLUEZ_DBUS_HEADSET "org.bluez.audio.Headset"
#define _BLUEZ_DBUS_HEADSET_IFACE "interface=org.bluez.audio.Headset"
#else
#define _BLUEZ_DBUS_HEADSET "org.bluez.Headset"
#define _BLUEZ_DBUS_HEADSET_IFACE "interface=org.bluez.Headset"
#endif

#include "mapper.h"
#include "utils.h"
#include "mapper-types.h"
#include "settings.h"
#include "gps.h"
#include "gps-conn.h"
#include "map.h"
#include "map-download.h"
#include "map-poi.h"
#include "route.h"
#include "track.h"
#include "db.h"
#include "osm-db.h"
#include "poi.h"
#include "cb.h"
#include "speak.h"
#include "gpx.h"
#include "config-gconf.h"
#include "dialogs.h"
#include "iap.h"
#include "latlon.h"
#ifdef WITH_OSSO
#include "maemo-osso.h"
#endif

GdkColor COLORABLE_DEFAULT[COLORABLE_ENUM_COUNT] = {
	{0, 0x0000, 0x0000, 0xc000}
	,			/* COLORABLE_MARK */
	{0, 0x6000, 0x6000, 0xf800}
	,			/* COLORABLE_MARK_VELOCITY */
	{0, 0x8000, 0x8000, 0x8000}
	,			/* COLORABLE_MARK_OLD */
	{0, 0xe000, 0x0000, 0x0000}
	,			/* COLORABLE_TRACK */
	{0, 0xa000, 0x0000, 0x0000}
	,			/* COLORABLE_TRACK_MARK */
	{0, 0x7000, 0x0000, 0x0000}
	,			/* COLORABLE_TRACK_BREAK */
	{0, 0x0000, 0xa000, 0x0000}
	,			/* COLORABLE_ROUTE */
	{0, 0x0000, 0x8000, 0x0000}
	,			/* COLORABLE_ROUTE_WAY */
	{0, 0x0000, 0x6000, 0x0000}
	,			/* COLORABLE_ROUTE_BREAK */
	{0, 0xa000, 0x0000, 0xa000}	/* COLORABLE_POI */
};

enum {
	MAPPER_INIT_START=0,
	MAPPER_INIT_MISC,
	MAPPER_INIT_DB,
	MAPPER_INIT_VOICE,
	MAPPER_INIT_UI,
	MAPPER_INIT_GOTO,
	MAPPER_INIT_DONE
} mapper_init_state;

guint mis=MAPPER_INIT_START;

static GtkWidget *init_progress;
static GtkWidget *init_dialog;
static GtkWidget *mw=NULL;

/**
 * Initialize arrays
 */
static void 
variables_init(void)
{
INFO_FONT_TEXT[INFO_FONT_XXSMALL] = "xx-small";
INFO_FONT_TEXT[INFO_FONT_XSMALL] = "x-small";
INFO_FONT_TEXT[INFO_FONT_SMALL] = "small";
INFO_FONT_TEXT[INFO_FONT_MEDIUM] = "medium";
INFO_FONT_TEXT[INFO_FONT_LARGE] = "large";
INFO_FONT_TEXT[INFO_FONT_XLARGE] = "x-large";
INFO_FONT_TEXT[INFO_FONT_XXLARGE] = "xx-large";

CUSTOM_KEY_GCONF[CUSTOM_KEY_UP] = GCONF_KEY_PREFIX "/key_up";
CUSTOM_KEY_GCONF[CUSTOM_KEY_DOWN] = GCONF_KEY_PREFIX "/key_down";
CUSTOM_KEY_GCONF[CUSTOM_KEY_LEFT] = GCONF_KEY_PREFIX "/key_left";
CUSTOM_KEY_GCONF[CUSTOM_KEY_RIGHT] = GCONF_KEY_PREFIX "/key_right";
CUSTOM_KEY_GCONF[CUSTOM_KEY_SELECT] = GCONF_KEY_PREFIX "/key_select";
CUSTOM_KEY_GCONF[CUSTOM_KEY_INCREASE] = GCONF_KEY_PREFIX "/key_increase";
CUSTOM_KEY_GCONF[CUSTOM_KEY_DECREASE] = GCONF_KEY_PREFIX "/key_decrease";
CUSTOM_KEY_GCONF[CUSTOM_KEY_FULLSCREEN] = GCONF_KEY_PREFIX "/key_fullscreen";
CUSTOM_KEY_GCONF[CUSTOM_KEY_ESC] = GCONF_KEY_PREFIX "/key_esc";

CUSTOM_KEY_ICON[CUSTOM_KEY_UP] = HWK_BUTTON_UP;
CUSTOM_KEY_ICON[CUSTOM_KEY_LEFT] = HWK_BUTTON_LEFT;
CUSTOM_KEY_ICON[CUSTOM_KEY_DOWN] = HWK_BUTTON_DOWN;
CUSTOM_KEY_ICON[CUSTOM_KEY_RIGHT] = HWK_BUTTON_RIGHT;
CUSTOM_KEY_ICON[CUSTOM_KEY_SELECT] = HWK_BUTTON_SELECT;
CUSTOM_KEY_ICON[CUSTOM_KEY_INCREASE] = HWK_BUTTON_INCREASE;
CUSTOM_KEY_ICON[CUSTOM_KEY_DECREASE] = HWK_BUTTON_DECREASE;
CUSTOM_KEY_ICON[CUSTOM_KEY_FULLSCREEN] = HWK_BUTTON_VIEW;
CUSTOM_KEY_ICON[CUSTOM_KEY_ESC] = HWK_BUTTON_CANCEL;

CUSTOM_KEY_DEFAULT[CUSTOM_KEY_UP] = CUSTOM_ACTION_PAN_NORTH;
CUSTOM_KEY_DEFAULT[CUSTOM_KEY_LEFT] = CUSTOM_ACTION_PAN_WEST;
CUSTOM_KEY_DEFAULT[CUSTOM_KEY_DOWN] = CUSTOM_ACTION_PAN_SOUTH;
CUSTOM_KEY_DEFAULT[CUSTOM_KEY_RIGHT] = CUSTOM_ACTION_PAN_EAST;
CUSTOM_KEY_DEFAULT[CUSTOM_KEY_SELECT] = CUSTOM_ACTION_TOGGLE_AUTOCENTER;
CUSTOM_KEY_DEFAULT[CUSTOM_KEY_INCREASE] = CUSTOM_ACTION_ZOOM_IN;
CUSTOM_KEY_DEFAULT[CUSTOM_KEY_DECREASE] = CUSTOM_ACTION_ZOOM_OUT;
CUSTOM_KEY_DEFAULT[CUSTOM_KEY_FULLSCREEN] = CUSTOM_ACTION_TOGGLE_FULLSCREEN;
CUSTOM_KEY_DEFAULT[CUSTOM_KEY_ESC] = CUSTOM_ACTION_TOGGLE_TRACKS;

CUSTOM_ACTION_TEXT[CUSTOM_ACTION_PAN_NORTH] = _("Pan North");
CUSTOM_ACTION_TEXT[CUSTOM_ACTION_PAN_WEST] = _("Pan West");
CUSTOM_ACTION_TEXT[CUSTOM_ACTION_PAN_SOUTH] = _("Pan South");
CUSTOM_ACTION_TEXT[CUSTOM_ACTION_PAN_EAST] = _("Pan East");
CUSTOM_ACTION_TEXT[CUSTOM_ACTION_TOGGLE_AUTOCENTER] = _("Toggle Auto-Center");
CUSTOM_ACTION_TEXT[CUSTOM_ACTION_TOGGLE_FULLSCREEN] = _("Toggle Fullscreen");
CUSTOM_ACTION_TEXT[CUSTOM_ACTION_ZOOM_IN] = _("Zoom In");
CUSTOM_ACTION_TEXT[CUSTOM_ACTION_ZOOM_OUT] = _("Zoom Out");
CUSTOM_ACTION_TEXT[CUSTOM_ACTION_TOGGLE_TRACKS] = _("Toggle Tracks");
CUSTOM_ACTION_TEXT[CUSTOM_ACTION_TOGGLE_SCALE] = _("Toggle Scale");
CUSTOM_ACTION_TEXT[CUSTOM_ACTION_TOGGLE_POI] = _("Toggle POIs");
CUSTOM_ACTION_TEXT[CUSTOM_ACTION_CHANGE_REPO] = _("Select Next Repository");
CUSTOM_ACTION_TEXT[CUSTOM_ACTION_ROUTE_DISTNEXT] = _("Show Distance to Next Waypoint");
CUSTOM_ACTION_TEXT[CUSTOM_ACTION_ROUTE_DISTLAST] = _("Show Distance to End of Route");
CUSTOM_ACTION_TEXT[CUSTOM_ACTION_TRACK_BREAK] = _("Insert Track Break");
CUSTOM_ACTION_TEXT[CUSTOM_ACTION_TRACK_DISTLAST] = _("Show Distance from Last Break");
CUSTOM_ACTION_TEXT[CUSTOM_ACTION_TRACK_DISTFIRST] = _("Show Distance from Beginning");
CUSTOM_ACTION_TEXT[CUSTOM_ACTION_TOGGLE_GPS] = _("Toggle GPS");
CUSTOM_ACTION_TEXT[CUSTOM_ACTION_TOGGLE_GPSINFO] = _("Toggle GPS Info");
CUSTOM_ACTION_TEXT[CUSTOM_ACTION_TOGGLE_SPEEDLIMIT] = _("Toggle Speed Limit");

COLORABLE_GCONF[COLORABLE_MARK] = GCONF_KEY_PREFIX "/color_mark";
COLORABLE_GCONF[COLORABLE_MARK_VELOCITY] = GCONF_KEY_PREFIX "/color_mark_velocity";
COLORABLE_GCONF[COLORABLE_MARK_OLD] = GCONF_KEY_PREFIX "/color_mark_old";
COLORABLE_GCONF[COLORABLE_TRACK] = GCONF_KEY_PREFIX "/color_track";
COLORABLE_GCONF[COLORABLE_TRACK_MARK] = GCONF_KEY_PREFIX "/color_track_mark";
COLORABLE_GCONF[COLORABLE_TRACK_BREAK] = GCONF_KEY_PREFIX "/color_track_break";
COLORABLE_GCONF[COLORABLE_ROUTE] = GCONF_KEY_PREFIX "/color_route";
COLORABLE_GCONF[COLORABLE_ROUTE_WAY] = GCONF_KEY_PREFIX "/color_route_way";
COLORABLE_GCONF[COLORABLE_ROUTE_BREAK] = GCONF_KEY_PREFIX "/color_route_break";
COLORABLE_GCONF[COLORABLE_POI] = GCONF_KEY_PREFIX "/color_poi";
}

/**
 * Save state and destroy all non-UI elements created by this program in
 * preparation for exiting.
 */
static void 
mapper_destroy(void)
{
config_save();
map_download_deinit();
map_poi_deinit();

speak_deinit();

gps_disconnect(_gps);
gps_free(_gps);
osm_deinit();
db_close(&_db);
path_free(_track);
path_free(_route);

gnome_vfs_shutdown();
#ifdef WITH_OSSO
maemo_osso_deinitialize(&mapper_app);
#endif
#ifdef WITH_HAL
if (mapper_app.halctx)
	libhal_ctx_shutdown(mapper_app.halctx, NULL);
#endif
curl_global_cleanup();
}

#ifdef WITH_HAL
static void
hal_condition_cb(LibHalContext *ctx, const char *udi, const char *name, const char *detail)
{
g_debug("HAL: [%s]->[%s:%s]", udi, name, detail);
if (strcmp(name, "ButtonPressed")==0 && strcmp(detail, "phone")==0)
	g_idle_add_full(G_PRIORITY_HIGH_IDLE, headphone_button_cb, NULL,NULL);
}

static gboolean
mapper_hal_init(DBusGConnection *gdbc)
{
DBusError error;
DBusConnection *dbc=dbus_g_connection_get_connection(gdbc);

g_return_val_if_fail(dbc, FALSE);

dbus_error_init(&error);

mapper_app.halctx=libhal_ctx_new();
g_return_val_if_fail(mapper_app.halctx, FALSE);

if (!libhal_ctx_set_dbus_connection(mapper_app.halctx, dbc)) {
	g_warning("HAL dbus connection failed");
	return FALSE;
}

if (!libhal_ctx_init(mapper_app.halctx, &error)) {
	g_warning("HAL Init failed");
	mapper_app.halctx=NULL;
	return FALSE;
}

libhal_ctx_set_device_condition(mapper_app.halctx, hal_condition_cb);
libhal_device_add_property_watch(mapper_app.halctx, "/org/freedesktop/Hal/devices/platform_retu_headset_logicaldev_input",NULL);

return TRUE;
}
#endif

static DBusHandlerResult
bt_headset_button_cb(DBusConnection *conn, DBusMessage *msg, gpointer data)
{
g_debug("dbus-bt");
if (dbus_message_is_signal(msg, _BLUEZ_DBUS_HEADSET, "AnswerRequested"))
	g_idle_add_full(G_PRIORITY_HIGH_IDLE, headphone_button_cb, NULL, NULL);
return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
}

static void
mapper_headset_button_init(DBusGConnection *gdbc)
{
DBusConnection *dbc;

g_return_if_fail(gdbc);
dbc=dbus_g_connection_get_connection(gdbc);

dbus_bus_add_match(dbc, _BLUEZ_DBUS_HEADSET_IFACE, NULL);
dbus_connection_add_filter(dbc, bt_headset_button_cb, NULL, NULL);
}

static void 
timezone_init(void)
{
time_t time1;
struct tm time2;

time1 = time(NULL);
localtime_r(&time1, &time2);
_gmtoffset = time2.tm_gmtoff;
}

static gboolean
mapper_init(gpointer data)
{
GError *error = NULL;
gboolean ret=TRUE;
gfloat p=0;
gchar *w="Starting";

switch (mis) {
	case MAPPER_INIT_START:
		curl_global_init(CURL_GLOBAL_NOTHING);
		gnome_vfs_init();
		/* XXX: Load GPS configuration, then create gps */
		_gps=gps_new(GPS_IO_SIMULATION);
		_gps->connection_retry=gps_retry_connection;
		_gps->connection_error=NULL;
		_gps->connection_progress=gps_conn_set_progress;
		_gps->update_location=gps_location_update;
		_gps->update_info=gps_info_update;
		timezone_init();
		gpx_init();
		variables_init();
		latlon_init();
		_track=path_new();
		_route=path_new();
		config_init();
		poi_init();
		/* Initialize D-Bus system connection. */
		if (NULL==(mapper_app.dbus_conn=dbus_g_bus_get(DBUS_BUS_SYSTEM, &error))) {
			g_warning("Failed to open connection to D-Bus: %s.\n", error->message);
			popup_error(NULL, "Failed to connect to D-Bus.");
			error=NULL;
		}
#ifdef WITH_OSSO
		maemo_osso_initialize(&mapper_app);
#endif
		iap_init();
		map_download_init();
		mis=MAPPER_INIT_DB;
		p=0.2;
		w="Config";
	break;
	case MAPPER_INIT_DB:
		if (db_connect(&_db, _mapper_db)) {
			if (poi_init_db(&_db)) {
				if (osm_init(&_db))
					osm_db_enable_mainloop(_db, TRUE);
			}
		}
		mis=MAPPER_INIT_VOICE;
		p=0.4;
		w="Database";
	break;
	case MAPPER_INIT_VOICE:
		if (speak_init("en",_voice_speed,_voice_pitch)==FALSE) {
			g_warning("Espeak init failed");
			popup_error(mapp.mainwindow, "Speech init failed. Disabled.");
		}
		mis=MAPPER_INIT_UI;
		p=0.6;
		w="Speech";
	break;
	case MAPPER_INIT_UI:
		g_assert(mw);
		mapper_app.ui=mapper_ui_init(mw);
		mis=MAPPER_INIT_MISC;
		p=0.8;
		w="UI";
	break;
	case MAPPER_INIT_MISC:
#ifdef WITH_HAL
		mapper_hal_init(mapper_app.dbus_conn);
#endif
		mapper_headset_button_init(mapper_app.dbus_conn);
		mis=MAPPER_INIT_GOTO;
		p=0.9;
		w="Misc";
	break;
	case MAPPER_INIT_GOTO:
		if (map_goto_position(&_home)==FALSE)
			if (map_goto_position(&_dest)==FALSE)
				map_center_unit(_center.unitx, _center.unity);
		mis=MAPPER_INIT_DONE;
		p=1.0;
		w="Done";
	break;
	case MAPPER_INIT_DONE:
		progress_dialog_remove(init_dialog);
#ifdef WITH_OSSO
		maemo_osso_rotation_start(&mapper_app);
#endif
		if (_enable_gps)   
			gps_connect_now(_gps);
		return FALSE;
	break;
}

gtk_progress_bar_set_fraction(GTK_PROGRESS_BAR(init_progress), p);
gtk_progress_bar_set_text(GTK_PROGRESS_BAR(init_progress), w);
g_debug("Init: %s", w);
return ret;
}

gint 
main(gint argc, gchar * argv[])
{
setlocale(LC_ALL, "");
bindtextdomain(GETTEXT_PACKAGE, LOCALEDIR);
bind_textdomain_codeset(GETTEXT_PACKAGE, "UTF-8");
textdomain(GETTEXT_PACKAGE);

g_thread_init(NULL);
g_type_init();
gst_init(&argc, &argv);
gtk_init(&argc, &argv);
g_set_application_name("Mapper");
gtk_window_set_default_icon_name("mapper");
#if 0
g_setenv("PULSE_PROP_media.role", "navigation", TRUE);
#endif

mw=mapper_window_new();

init_progress=gtk_progress_bar_new();
init_dialog=progress_dialog(mw, "Mapper", gtk_label_new(_("Loading, please wait...")), init_progress);
gtk_widget_show_all(init_dialog);
g_idle_add((GSourceFunc)mapper_init, NULL);
gtk_main();
mapper_destroy();

return 0;
}
