#include <config.h>

#ifndef _MAPPER_MAP_H
#define _MAPPER_MAP_H

#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <strings.h>
#include <stddef.h>
#include <locale.h>
#include <math.h>
#include <errno.h>
#include <sys/wait.h>
#include <glib/gstdio.h>
#include <gtk/gtk.h>
#include <dbus/dbus-glib.h>

#include <image-cache/image-cache.h>

#include "utils.h"
#include "mapper-types.h"
#include "osm.h"
#include "map-repo.h"

#define tile2grid(tile) ((tile) << 3)
#define grid2tile(grid) ((grid) >> 3)

#define tile2pixel(tile) ((tile) << 8)
#define pixel2tile(pixel) ((pixel) >> 8)

#define tile2unit(tile) ((tile) << (8 + _zoom))
#define unit2tile(unit) ((unit) >> (8 + _zoom))
#define tile2zunit(tile, zoom) ((tile) << (8 + zoom))
#define unit2ztile(unit, zoom) ((unit) >> (8 + zoom))

#define grid2pixel(grid) ((grid) << 5)
#define pixel2grid(pixel) ((pixel) >> 5)
#define grid2unit(grid) ((grid) << (5 + _zoom))
#define unit2grid(unit) ((unit) >> (5 + _zoom))

#define pixel2unit(pixel) ((pixel) << _zoom)
#define unit2pixel(pixel) ((pixel) >> _zoom)
#define pixel2zunit(pixel, zoom) ((pixel) << (zoom))

#define unit2bufx(unit) (unit2pixel(unit) - tile2pixel(_base_tilex))
#define bufx2unit(x) (pixel2unit(x) + tile2unit(_base_tilex))
#define unit2bufy(unit) (unit2pixel(unit) - tile2pixel(_base_tiley))
#define bufy2unit(y) (pixel2unit(y) + tile2unit(_base_tiley))

#define unit2x(unit) (unit2pixel(unit) - tile2pixel(_base_tilex) - _offsetx)
#define x2unit(x) (pixel2unit(x + _offsetx) + tile2unit(_base_tilex))
#define unit2y(unit) (unit2pixel(unit) - tile2pixel(_base_tiley) - _offsety)
#define y2unit(y) (pixel2unit(y + _offsety) + tile2unit(_base_tiley))

#define leadx2unit(mgps) (mgps.unitx + (_lead_ratio) * pixel2unit(mgps.vel_offsetx))
#define leady2unit(mgps) (mgps.unity + (0.6f*_lead_ratio)*pixel2unit(mgps.vel_offsety))

/* Pans are done two "grids" at a time, or 64 pixels. */
#define PAN_UNITS (grid2unit(2))

#define MACRO_RECALC_CENTER(mgps, center_unitx, center_unity) { \
    switch(_center_mode) \
    { \
        case CENTER_LEAD: \
            center_unitx = leadx2unit(mgps); \
            center_unity = leady2unit(mgps); \
            break; \
        case CENTER_LATLON: \
            center_unitx = mgps.unitx; \
            center_unity = mgps.unity; \
            break; \
        default: \
             center_unitx = _center.unitx; \
             center_unity = _center.unity; \
            ; \
    } \
};

#define MACRO_RECALC_CENTER_BOUNDS() { \
	_min_center.unitx = pixel2unit(grid2pixel(_screen_grids_halfwidth)); \
	_min_center.unity = pixel2unit(grid2pixel(_screen_grids_halfheight)); \
	_max_center.unitx = WORLD_SIZE_UNITS-grid2unit(_screen_grids_halfwidth) - 1; \
	_max_center.unity = WORLD_SIZE_UNITS-grid2unit(_screen_grids_halfheight) - 1; \
}

#define MACRO_QUEUE_DRAW_AREA() \
    gtk_widget_queue_draw_area(mapp.map_widget, 0, 0, _screen_width_pixels, _screen_height_pixels)

typedef enum {
	MAP_MODE_NORMAL=0,
	MAP_MODE_DRAW_TRACK,
	MAP_MODE_DRAW_ROUTE,
	MAP_MODE_SET_ROUTE_FROM,
	MAP_MODE_SET_ROUTE_POINT,
	MAP_MODE_SET_ROUTE_TO,
	MAP_MODES
} MapMode;

/** Possible center modes.  The "WAS" modes imply no current center mode;
 * they only hint at what the last center mode was, so that it can be
 * recalled. */
typedef enum {
	CENTER_WAS_LATLON = -2,
	CENTER_WAS_LEAD = -1,
	CENTER_MANUAL =0,
	CENTER_LEAD = 1,
	CENTER_LATLON = 2
} CenterMode;

/** VARIABLES FOR MAINTAINING STATE OF THE CURRENT VIEW. */

/** The "zoom" level defines the resolution of a pixel, from 0 to MAX_ZOOM.
 * Each pixel in the current view is exactly (1 << _zoom) "units" wide. */

Point _min_center;
Point _max_center;
Point _focus;

/** The "base tile" is the upper-left tile in the pixmap. */
guint _base_tilex;
guint _base_tiley;

guint _zoom;			/* zoom level, from 0 to MAX_ZOOM. */
Point _center;			/* current center location, X. */

/** The "offset" defines the upper-left corner of the visible portion of the
 * 1024x768 pixmap. */
guint _offsetx;
guint _offsety;

/** CACHED SCREEN INFORMATION THAT IS DEPENDENT ON THE CURRENT VIEW. */
guint _screen_grids_halfwidth;
guint _screen_grids_halfheight;
guint _screen_width_pixels;
guint _screen_height_pixels;

guint _focus_unitwidth;
guint _focus_unitheight;
guint _world_size_tiles;

gint _show_tracks;
gboolean _show_scale;
gboolean _show_velvec;
gboolean _show_poi;

guint _lead_ratio;
guint _center_ratio;
guint _draw_width;

guint _key_zoom_new;
guint _key_zoom_timeout_sid;

osm_location map_loc;

guint32 map_poi_id;

ImageCache *map_ic;

CenterMode _center_mode;

/** The widget that provides the visual display of the map. */
GtkWidget *map_new(void);

gboolean map_key_zoom_timeout();

int map_zoom(gint zdir);
guint map_get_zoom(void);
gboolean map_set_zoom(guint zoom);

gboolean map_zoom_in(void);
gboolean map_zoom_out(void);
void map_set_autozoom(gboolean az, gfloat speed);
void map_render_path(Path * path, GdkGC ** gc);
void map_pan(gint delta_unitx, gint delta_unity);
void map_move_mark(void);
void map_set_mark(GpsData *gps);
void map_render_data(void);
gboolean map_render_tile(guint tilex, guint tiley, guint destx, guint desty, gboolean fast_fail);

void map_render_segment(GdkGC *gc_norm, GdkGC *gc_alt, guint unitx1, guint unity1, guint unitx2, guint unity2);
void map_render_waypoint(guint x1, guint y1, GdkGC *gc);
void map_render_paths(void);

void map_force_redraw(void);
GdkPixmap *map_pixmap_get(void);

void map_center_unit(guint new_center_unitx, guint new_center_unity);
void map_center_latlon(gdouble lat, gdouble lon);

gboolean map_goto_position(Position *pos);
gboolean map_update_location_from_center(void);
void map_update_destination(gdouble lat, gdouble lon);
void map_update_location_from_center_idle(void);

gboolean map_dialog_goto_latlon(void);

void map_refresh_mark(void);

gchar *map_tile_key(guint x, guint y, gint z);
gchar *map_tile_file(RepoData *repo, guint x, guint y, gint z, gboolean temp);

void map_menu_check_context(void);

#endif
