/*
 * This file is part of mapper
 *
 * Copyright (C) 2007 Kaj-Michael Lang
 * Copyright (C) 2006-2007 John Costigan.
 *
 * POI and GPS-Info code originally written by Cezary Jackiewicz.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#include <config.h>

#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <strings.h>
#include <stddef.h>
#include <math.h>
#include <errno.h>
#include <sys/wait.h>
#include <glib/gstdio.h>
#include <gtk/gtk.h>
#include <libintl.h>
#include <locale.h>

#include "utils.h"
#include "gps.h"
#include "settings.h"
#include "mapper-types.h"
#include "ui-common.h"
#include "latlon.h"
#include "track.h"
#include "route.h"

#include "gps-panels.h"
#include "gtkgps.h"
#include "gtkcompass.h"

static GtkWidget *text_latlon;
static GtkWidget *text_speed;
static GtkWidget *text_speedmax;
static GtkWidget *text_speedavg;
static GtkWidget *text_alt;
static GtkWidget *text_dop;
static GtkWidget *text_time;
static GtkWidget *text_heading;

GtkWidget *
gps_info_panel(void)
{
GtkWidget *vbox;

vbox=gtk_vbox_new(FALSE, 0);

text_time=gtk_label_new(NULL);
gtk_widget_set_size_request(GTK_WIDGET(text_time), -1, 30);
gtk_box_pack_start(GTK_BOX(vbox), text_time, FALSE, TRUE, 0);

text_latlon=gtk_label_new(NULL);
gtk_widget_set_size_request(GTK_WIDGET(text_latlon), -1, 30);
gtk_box_pack_start(GTK_BOX(vbox), text_latlon, FALSE, TRUE, 0);

text_speed=gtk_label_new(NULL);
gtk_label_set_use_markup(GTK_LABEL(text_speed), TRUE);
gtk_widget_set_size_request(GTK_WIDGET(text_speed), -1, 30);
gtk_box_pack_start(GTK_BOX(vbox), text_speed, FALSE, TRUE, 0);

text_speedmax=gtk_label_new(NULL);
gtk_widget_set_size_request(GTK_WIDGET(text_speedmax), -1, 30);
gtk_box_pack_start(GTK_BOX(vbox), text_speedmax, FALSE, TRUE, 0);

text_speedavg=gtk_label_new(NULL);
gtk_widget_set_size_request(GTK_WIDGET(text_speedavg), -1, 30);
gtk_box_pack_start(GTK_BOX(vbox), text_speedavg, FALSE, TRUE, 0);

text_alt=gtk_label_new(NULL);
gtk_widget_set_size_request(GTK_WIDGET(text_alt), -1, 30);
gtk_box_pack_start(GTK_BOX(vbox), text_alt, FALSE, TRUE, 0);

text_heading=gtk_label_new(NULL);
gtk_widget_set_size_request(GTK_WIDGET(text_heading), -1, 30);
gtk_box_pack_start(GTK_BOX(vbox), text_dop, FALSE, TRUE, 0);

text_dop=gtk_label_new(NULL);
gtk_widget_set_size_request(GTK_WIDGET(text_dop), -1, 30);
gtk_box_pack_start(GTK_BOX(vbox), text_dop, FALSE, TRUE, 0);

gps_info_clear();

return vbox;
}

void
gps_display_data_speed(GtkWidget *widget, gfloat s, gboolean overspeed, const gchar *fsize)
{
gchar buffer[64];
gchar *unit;
gfloat speed=s*UNITS_CONVERT[_units];

switch (_units) {
	case UNITS_MI:
		unit="mph";
	break;
	case UNITS_NM:
		unit="kn";;
	break;
	default:
		unit="km/h";
	break;
	}
g_snprintf(buffer, sizeof(buffer), 
	"<span size=\"%s\" foreground=\"%s\">%.1f</span>\n%s",
	fsize, overspeed ? "#A50000" : "#009500", speed, unit);
gtk_label_set_markup(GTK_LABEL(widget), buffer);
}

void
gps_info_clear(void)
{
gtk_label_set_label(GTK_LABEL(text_latlon), " --- ");
gtk_label_set_label(GTK_LABEL(text_speed), " --- ");
gtk_label_set_label(GTK_LABEL(text_speedavg), " --- ");
gtk_label_set_label(GTK_LABEL(text_speedmax), " --- ");
gtk_label_set_label(GTK_LABEL(text_alt), " --- ");
gtk_label_set_label(GTK_LABEL(text_dop), " --- ");
}

void 
gps_display_data(GpsData *gps)
{
gchar buffer[80];
gchar blat[24];
gchar blon[24];

/* local time */
strftime(buffer, 15, "%X", localtime(&gps->time));
gtk_label_set_label(GTK_LABEL(text_time), buffer);

if (gps->fix==FIX_NOFIX) {
	gps_info_clear();
	return;
}

/* latitude */
lat_format(_degformat, gps->lat, blat);
lon_format(_degformat, gps->lon, blon);
g_snprintf(buffer, sizeof(buffer), "<span size=\"small\">%s %s</span>", blat, blon);
gtk_label_set_markup(GTK_LABEL(text_latlon), buffer);

/* speed */
gps_display_data_speed(text_speed, gps->speed, _speed_excess, "x-large");
gps_display_data_speed(text_speedmax, _track->maxspeed, FALSE, "large");
gps_display_data_speed(text_speedavg, _track->avgspeed, FALSE, "large");

g_snprintf(buffer, 32, "%0.0f°", gps->heading);
gtk_label_set_label(GTK_LABEL(text_heading), buffer);

/* altitude */
switch (_units) {
case UNITS_MI:
case UNITS_NM:
	g_snprintf(buffer, sizeof(buffer), "Alt: %.1f ft", gps->altitude * 3.2808399f);
	break;
default:
	g_snprintf(buffer, sizeof(buffer), "Alt: %.1f m", gps->altitude);
}
gtk_label_set_label(GTK_LABEL(text_alt), buffer);

/* DOP */
g_snprintf(buffer, 32, "%.1f/%.1f/%.1f", gps->hdop, gps->vdop, gps->pdop);
gtk_label_set_label(GTK_LABEL(text_dop), buffer);
}

gboolean
gps_display_details(GpsData *gps)
{
gchar buffer[32];

if (gps->fix==FIX_NOFIX) {
	/* no fix no fun */
	gtk_label_set_label(GTK_LABEL(_sdi_lat), " --- ");
	gtk_label_set_label(GTK_LABEL(_sdi_lon), " --- ");
	gtk_label_set_label(GTK_LABEL(_sdi_spd), " --- ");
	gtk_label_set_label(GTK_LABEL(_sdi_alt), " --- ");
	gtk_label_set_label(GTK_LABEL(_sdi_hea), " --- ");
	gtk_label_set_label(GTK_LABEL(_sdi_tim), " --:--:-- ");
} else {
	/* latitude */
	lat_format(_degformat, gps->lat, buffer);
	gtk_label_set_label(GTK_LABEL(_sdi_lat), buffer);

	/* longitude */
	lon_format(_degformat, gps->lon, buffer);
	gtk_label_set_label(GTK_LABEL(_sdi_lon), buffer);

	/* speed */
	gps_display_data_speed(_sdi_spd, gps->speed, _speed_excess, "x-large");

	/* altitude */
	switch (_units) {
	case UNITS_MI:
	case UNITS_NM:
		g_snprintf(buffer, 32, "%.1f ft", gps->altitude * 3.2808399f);
	break;
	default:
		g_snprintf(buffer, 32, "%.1f m", gps->altitude);
	break;
	}
	gtk_label_set_label(GTK_LABEL(_sdi_alt), buffer);

	/* heading */
	g_snprintf(buffer, 32, "%0.0f° (%0.0f)", gps->heading, fabs(gps->heading-gps->lheading));
	gtk_label_set_label(GTK_LABEL(_sdi_hea), buffer);

	/* local time */
	strftime(buffer, 15, "%X", localtime(&gps->time));
	gtk_label_set_label(GTK_LABEL(_sdi_tim), buffer);
}

/* Sat in view */
g_snprintf(buffer, 32, "%d", gps->satinview);
gtk_label_set_label(GTK_LABEL(_sdi_vie), buffer);

/* Sat in use */
g_snprintf(buffer, 32, "%d", gps->satinuse);
gtk_label_set_label(GTK_LABEL(_sdi_use), buffer);

/* fix */
switch (gps->fix) {
case 2:
case 3:
	g_snprintf(buffer, 32, "%dD fix", gps->fix);
break;
default:
	g_snprintf(buffer, 32, "nofix");
break;
}
gtk_label_set_label(GTK_LABEL(_sdi_fix), buffer);

if (gps->fix ==FIX_NOFIX)
	g_snprintf(buffer, 32, "none");
else {
	switch (gps->fixquality) {
	case 1:
		g_strlcat(buffer,_("SPS"),32);
	break;
	case 2:
		g_strlcat(buffer,_("DGPS"),32);
	break;
	case 3:
		g_strlcat(buffer,_("PPS"),32);
	break;
	case 4:
		g_strlcat(buffer,_("Real Time Kinematic"),32);
	break;
	case 5:
		g_strlcat(buffer,_("Float RTK"),32);
	break;
	case 6:
		g_strlcat(buffer,_("Estimated"),32);
	break;
	case 7:
		g_strlcat(buffer,_("Manual"),32);
	break;
	case 8:
		g_strlcat(buffer,_("Simulation"),32);
	break;
	default:
		g_strlcat(buffer,_("none"),32);
	break;
	}
}
gtk_label_set_label(GTK_LABEL(_sdi_fqu), buffer);

/* max speed */
gps_display_data_speed(_sdi_msp, gps->maxspeed, FALSE, "large");

return TRUE;
}
