/*
 * This file is part of mapper
 *
 * Copyright (C) 2007 Kaj-Michael Lang
 * Copyright (C) 2006-2007 John Costigan.
 *
 * POI and GPS-Info code originally written by Cezary Jackiewicz.
 *
 * Default map data provided by http://www.openstreetmap.org/
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#include "config.h"

#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <strings.h>
#include <stddef.h>
#include <locale.h>
#include <math.h>
#include <errno.h>
#include <sys/wait.h>
#include <glib/gstdio.h>
#include <fcntl.h>
#include <curl/multi.h>
#include <libintl.h>
#include <locale.h>

#include "hildon-mapper.h"

#include "utils.h"
#include "map.h"
#include "osm.h"
#include "db.h"
#include "osm-db.h"
#include "poi.h"
#include "route.h"
#include "gps.h"
#include "mapper-types.h"
#include "ui-common.h"
#include "settings.h"
#include "latlon.h"
#include "gpx.h"
#include "map-download.h"
#include "iap.h"
#include "map-repo.h"

/** Data used during the asynchronous progress update phase of automatic map
 * downloading. */
typedef struct _ProgressUpdateInfo ProgressUpdateInfo;
struct _ProgressUpdateInfo {
	gchar *src_str;
	gchar *dest_str;
	RepoData *repo;
	guint tilex, tiley;
	guint zoom;	/* for refresh. */
	gint retries;		/* if equal to zero, it means we're DELETING maps. */
	guint priority;
	FILE *file;
};

static guint _num_downloads=0;
static guint _curr_download=0;

static GQueue *curl_easy_queue=NULL;
static GTree *pui_tree=NULL;
static GTree *downloading_tree=NULL;
static GHashTable *pui_by_easy=NULL;

static gchar *map_construct_url(guint tilex, guint tiley, guint zoom);
static gboolean map_download_idle_refresh(ProgressUpdateInfo * pui);

static gboolean 
get_next_pui(gpointer key, gpointer value, ProgressUpdateInfo ** data)
{
*data = key;
return TRUE;
}

static gchar *
map_construct_destination_file(const ProgressUpdateInfo *pui, gboolean temp)
{
return map_tile_file(pui->repo, pui->tilex, pui->tiley, pui->zoom, temp);
}

static gint
download_comparefunc(const ProgressUpdateInfo * a, const ProgressUpdateInfo * b, gpointer user_data)
{
gint diff = (a->priority - b->priority);
if (diff)
	return diff;
diff = (a->tilex - b->tilex);
if (diff)
	return diff;
diff = (a->tiley - b->tiley);
if (diff)
	return diff;
diff = (a->zoom - b->zoom);
if (diff)
	return diff;
diff = (a->repo - b->repo);
if (diff)
	return diff;
/* Otherwise, deletes are "greatest" (least priority). */
if (!a->retries)
	return (b->retries ? -1 : 0);
else if (!b->retries)
	return (a->retries ? 1 : 0);
/* Do updates after non-updates (because they'll both be done anyway). */
return (a->retries - b->retries);
}

/**
 * Free a ProgressUpdateInfo data structure that was allocated during the
 * auto-map-download process.
 */
static void 
progress_update_info_free(ProgressUpdateInfo * pui)
{
g_free(pui->src_str);
g_free(pui->dest_str);
pui->src_str=NULL;
pui->dest_str=NULL;
g_slice_free(ProgressUpdateInfo, pui);
}

gboolean 
map_download_timeout(void)
{
static guint destroy_counter = 50;
gint num_transfers = 0, num_msgs = 0;
gint deletes_left = 50;	/* only do 50 deletes at a time. */
CURLMsg *msg;

/* curl should always be available, but check it once here just in case */
if (!_curl_multi) {
	g_warning("MD: curl_multi not set!");
	_curl_sid=0;
	return FALSE;
}

/* User canceled connection request, stop what we are doing */
if (iap_is_canceled()==TRUE) {
	set_action_activate("map_auto_download", FALSE);
	_curl_sid=0;
	return FALSE;
}

/* Lets try again later if we don't have a connection */
if (iap_is_connected()==FALSE) {
	return TRUE;
}

/* Give UI a chance first before we let curl do its thing again */
if (curl_multi_perform(_curl_multi, &num_transfers)==CURLM_CALL_MULTI_PERFORM)
	return TRUE;

while ((msg=curl_multi_info_read(_curl_multi, &num_msgs))) {
	if (msg->msg!=CURLMSG_DONE)
		continue;
	if (msg->easy_handle == _autoroute_data.curl_easy) {
		/* This is the autoroute download. */
		/* Now, parse the autoroute and update the display. */
		if (_autoroute_data.enabled && gpx_parse(_route, _autoroute_data.rdl_data.bytes, _autoroute_data.rdl_data.bytes_read, 0)) {
			/* Find the nearest route point, if we're connected. */
			route_find_nearest_point();
			map_force_redraw();
		}
		/* We're done. Clean up. */
		route_cancel_autoroute(TRUE);
	} else {
		ProgressUpdateInfo *pui;

		pui=g_hash_table_lookup(pui_by_easy, msg->easy_handle);
		g_return_val_if_fail(pui, TRUE);
		g_queue_push_head(curl_easy_queue, msg->easy_handle);
		g_hash_table_remove(pui_by_easy, msg->easy_handle);

		fclose(pui->file);
		if (msg->data.result!=CURLE_OK) {
			g_debug("Error: %s, removing %s", curl_easy_strerror(msg->data.result), pui->dest_str);
			/* Delete temp file so we try again. */
			g_unlink(pui->dest_str);
		} else {
			gchar *dfile;

			dfile=map_construct_destination_file(pui, FALSE);
			if (g_rename(pui->dest_str, dfile)!=0) {
				g_warning("Failed to rename temporary file %s to %s, removing", pui->dest_str, dfile);
				g_unlink(pui->dest_str);
			}
			/* Set the real name so refresh function gets right file */
			g_free(pui->dest_str);
			pui->dest_str=dfile;
		}

		curl_multi_remove_handle(_curl_multi, msg->easy_handle);
		g_idle_add((GSourceFunc)map_download_idle_refresh, pui);
		return TRUE; /* Let the main loop run */
	}
}

/* Up to 1 transfer per tile. */
while (num_transfers < (4*4) && g_tree_nnodes(pui_tree)) {
	ProgressUpdateInfo *pui;

	g_tree_foreach(pui_tree, (GTraverseFunc)get_next_pui, &pui);

	if (pui->retries) {
		/* This is a download. */
		FILE *f;
		g_tree_steal(pui_tree, pui);
		g_tree_insert(downloading_tree, pui, pui);

		pui->src_str=map_construct_url(pui->tilex, pui->tiley, pui->zoom);

		if (!pui->src_str) {
			/* Failed to generate URL. */
			g_debug("Failed to generate tile download URL");
			g_idle_add((GSourceFunc)map_download_idle_refresh, pui);
			continue;
		}

		pui->dest_str=map_construct_destination_file(pui, TRUE);

		/* Check to see if we need to overwrite. */
		if (pui->retries > 0) {
			/* We're not updating - check if file already exists. */
			if (g_file_test(pui->dest_str, G_FILE_TEST_EXISTS)) {
				g_idle_add((GSourceFunc)map_download_idle_refresh, pui);
				continue;
			}
		}

		/* Attempt to open the file for writing. */
		if (!(f=g_fopen(pui->dest_str, "w")) && errno == ENOENT) {
			/* Directory doesn't exist yet - create it, then we'll retry */
			gchar buffer[BUFFER_SIZE];
			g_snprintf(buffer, sizeof(buffer), "%s/%u/%u", pui->repo->cache_dir, pui->zoom, pui->tilex);
			g_mkdir_with_parents(buffer, 0775);
			f=g_fopen(pui->dest_str, "w");
		}

		if (f) {
			CURL *curl_easy;
			pui->file = f;
			curl_easy = g_queue_pop_tail(curl_easy_queue);
			if (!curl_easy) {
				/* Need a new curl_easy. */
				MACRO_CURL_EASY_INIT(curl_easy);
				MACRO_CURL_PROXY(curl_easy);
			}
			if (curl_easy_setopt(curl_easy, CURLOPT_URL, pui->src_str)!=CURLE_OK) {
				g_debug("curl: failed to set url");
				curl_easy_cleanup(curl_easy);
				fclose(f);
				continue;
			}
			if (curl_easy_setopt(curl_easy, CURLOPT_WRITEDATA, f)!=CURLE_OK) {
				g_debug("curl: failed to set write file");
				curl_easy_cleanup(curl_easy);
				fclose(f);
				continue;
			}
			g_hash_table_insert(pui_by_easy, curl_easy, pui);
			curl_multi_add_handle(_curl_multi, curl_easy);
			num_transfers++;
		} else {
			/* Unable to open tile file for writing. */
			gchar buffer[BUFFER_SIZE];
			g_snprintf(buffer, sizeof(buffer), "%s:\n%s", _("Failed to open file for writing"), pui->dest_str);
			MACRO_BANNER_SHOW_INFO(mapp.mainwindow, buffer);
			g_idle_add((GSourceFunc)map_download_idle_refresh, pui);
			continue;
		}
	} else if (--deletes_left) {
		/* This is a delete. */
		gchar *buffer;

		g_tree_steal(pui_tree, pui);
		g_tree_insert(downloading_tree, pui, pui);

		buffer=map_construct_destination_file(pui, FALSE);
		g_unlink(buffer);
		g_free(buffer);
		g_idle_add((GSourceFunc)map_download_idle_refresh, pui);
	} else
		break;
}

if (!(num_transfers || g_tree_nnodes(pui_tree))) {
	/* Destroy curl after 50 counts (5 seconds). */
	if (--destroy_counter) {
		/* Clean up curl. */
		CURL *curr;
		while ((curr = g_queue_pop_tail(curl_easy_queue)))
			curl_easy_cleanup(curr);

		_curl_sid = 0;
		return FALSE;
	}
} else
	destroy_counter = 50;

return TRUE;
}

/**
 * Given a wms uri pattern, compute the coordinate transformation and
 * trimming.
 * 'proj' is used for the conversion
 */
static gchar *
map_convert_wms_to_wms(gint tilex, gint tiley, gint zoomlevel, gchar * uri)
{
gint system_retcode;
gchar cmd[BUFFER_SIZE], srs[BUFFER_SIZE];
gchar *ret = NULL;
FILE *in;
gdouble lon1, lat1, lon2, lat2;

gchar *widthstr = strcasestr(uri, "WIDTH=");
gchar *heightstr = strcasestr(uri, "HEIGHT=");
gchar *srsstr = strcasestr(uri, "SRS=EPSG");
gchar *srsstre = strchr(srsstr, '&');

/* missing: test if found */
strcpy(srs, "epsg");
strncpy(srs + 4, srsstr + 8, 256);
/* missing: test srsstre-srsstr < 526 */
srs[srsstre - srsstr - 4] = 0;
/* convert to lower, as WMC is EPSG and cs2cs is epsg */

gint dwidth = widthstr ? atoi(widthstr + 6) - TILE_SIZE_PIXELS : 0;
gint dheight = heightstr ? atoi(heightstr + 7) - TILE_SIZE_PIXELS : 0;

unit2latlon(tile2zunit(tilex, zoomlevel) - pixel2zunit(dwidth / 2, zoomlevel),
	    tile2zunit(tiley + 1, zoomlevel) + pixel2zunit((dheight + 1) / 2, zoomlevel), lat1, lon1);

unit2latlon(tile2zunit(tilex + 1, zoomlevel) + pixel2zunit((dwidth + 1) / 2, zoomlevel), 
		tile2zunit(tiley, zoomlevel) - pixel2zunit(dheight / 2, zoomlevel), lat2, lon2);

setlocale(LC_NUMERIC, "C");

g_snprintf(cmd, sizeof(cmd),
	 "(echo \"%.6f %.6f\"; echo \"%.6f %.6f\") | "
	 "/usr/bin/cs2cs +proj=longlat +datum=WGS84 +to +init=%s -f %%.6f "
	 " > /tmp/tmpcs2cs ", lon1, lat1, lon2, lat2, srs);
vprintf("Running command: %s\n", cmd);
system_retcode = system(cmd);

if (system_retcode)
	g_printerr("cs2cs returned error code %d\n", WEXITSTATUS(system_retcode));
else if (!(in = g_fopen("/tmp/tmpcs2cs", "r")))
	g_printerr("Cannot open results of conversion\n");
else if (5 != fscanf(in, "%lf %lf %s %lf %lf", &lon1, &lat1, cmd, &lon2, &lat2)) {
	g_printerr("Wrong conversion\n");
	fclose(in);
} else {
	fclose(in);
	ret = g_strdup_printf(uri, lon1, lat1, lon2, lat2);
}

setlocale(LC_NUMERIC, "");

return ret;
}


/**
 * Given the xyz coordinates of our map coordinate system, write the qrst
 * quadtree coordinates to buffer.
 */
static void
map_convert_coords_to_quadtree_string(gint x, gint y, gint zoomlevel,
				      gchar * buffer, const gchar initial,
				      const gchar * const quadrant)
{
gchar *ptr = buffer;
gint n;

if (initial)
	*ptr++ = initial;

for (n = 16 - zoomlevel; n >= 0; n--) {
	gint xbit = (x >> n) & 1;
	gint ybit = (y >> n) & 1;
	*ptr++ = quadrant[xbit + 2 * ybit];
}
*ptr++ = '\0';
}

/**
 * Construct the URL that we should fetch, based on the current URI format.
 * This method works differently depending on if a "%s" string is present in
 * the URI format, since that would indicate a quadtree-based map coordinate
 * system.
 */
static gchar *
map_construct_url(guint tilex, guint tiley, guint zoom)
{
switch (_curr_repo->type) {
case REPOTYPE_XYZ:
	return g_strdup_printf(_curr_repo->url, tilex, tiley, zoom);

case REPOTYPE_XYZ_INV:
	return g_strdup_printf(_curr_repo->url, 17 - zoom, tilex, tiley);

case REPOTYPE_QUAD_QRST:
	{
		gchar location[MAX_ZOOM + 2];
		map_convert_coords_to_quadtree_string(tilex, tiley, zoom, location, 't', "qrts");
		return g_strdup_printf(_curr_repo->url, location);
	}

case REPOTYPE_QUAD_ZERO:
	{
		/* This is a zero-based quadtree URI. */
		gchar location[MAX_ZOOM + 2];
		map_convert_coords_to_quadtree_string(tilex, tiley, zoom, location, '\0', "0123");
		return g_strdup_printf(_curr_repo->url, location);
	}

case REPOTYPE_WMS:
	return map_convert_wms_to_wms(tilex, tiley, zoom, _curr_repo->url);

default:
	return NULL;
}
return NULL;
}

static void 
map_download_progress_clear(void) 
{
gtk_progress_bar_set_fraction(GTK_PROGRESS_BAR(mapp.progress), 0.0);
gtk_progress_bar_set_text(GTK_PROGRESS_BAR(mapp.progress), "");
}

static void 
map_download_progress_set(gint currd, gint numd)
{
gchar buffer[64];

g_snprintf(buffer, sizeof(buffer), "%d/%d", currd, numd);
gtk_progress_bar_set_text(GTK_PROGRESS_BAR(mapp.progress), buffer);	
gtk_progress_bar_set_fraction(GTK_PROGRESS_BAR(mapp.progress), (gdouble)currd/(gdouble)numd);
}

static gboolean 
map_download_idle_refresh(ProgressUpdateInfo * pui)
{
/* Test if download succeeded (only if retries != 0). */
if (!pui->retries || g_file_test(pui->dest_str, G_FILE_TEST_EXISTS)) {
	gchar *key;

	key=map_tile_key(pui->tilex, pui->tiley, pui->zoom);
	image_cache_invalidate(map_ic, key);
	g_free(key);

	if (pui->zoom == _zoom)
		map_force_redraw();
} else { 
	/* Else the download failed. Update retries and maybe try again. */
	if (pui->retries > 0)
		--pui->retries;
	else if (pui->retries < 0)
		++pui->retries;

	if (pui->retries) {
		/* removal automatically calls progress_update_info_free(). */
		g_tree_steal(downloading_tree, pui);
		g_tree_insert(pui_tree, pui, pui);
		if (iap_is_connected() && !_curl_sid)
			_curl_sid=g_idle_add((GSourceFunc)map_download_timeout, NULL);
		/* Don't do anything else. */
		return FALSE;
	} else {
		/* No more retries left - something must be wrong. */
		MACRO_BANNER_SHOW_INFO(mapp.mainwindow, _("Error in download. Check internet connection and/or Map Repository URL Format."));
	}
}

/* removal automatically calls progress_update_info_free(). */
g_tree_remove(downloading_tree, pui);

if (++_curr_download == _num_downloads) {
	_num_downloads = _curr_download = 0;
	map_download_progress_clear();
} else {
	map_download_progress_set(_curr_download, _num_downloads);
}
return FALSE;
}

/**
 * Initiate a download of the given xyz coordinates using the given buffer
 * as the URL.  If the map already exists on disk, or if we are already
 * downloading the map, then this method does nothing.
 */
void 
map_initiate_download(guint tilex, guint tiley, guint zoom, gint retries)
{
ProgressUpdateInfo *pui;

if (!iap_is_connected())
	iap_connect();

pui = g_slice_new(ProgressUpdateInfo);
pui->tilex = tilex;
pui->tiley = tiley;
pui->zoom = zoom;
pui->priority = (abs((gint) tilex - unit2tile(_center.unitx)) + abs((gint) tiley - unit2tile(_center.unity)));
if (!retries)
	pui->priority = -pui->priority;	/* "Negative" makes them lowest pri. */
pui->retries = retries;
pui->repo = _curr_repo;

if (g_tree_lookup(pui_tree, pui) || g_tree_lookup(downloading_tree, pui)) {
	/* Already downloading. */
	g_slice_free(ProgressUpdateInfo, pui);
	return;
}
pui->src_str = NULL;
pui->dest_str = NULL;
pui->file = NULL;

g_tree_insert(pui_tree, pui, pui);
if (iap_is_connected() && !_curl_sid)
	_curl_sid=g_idle_add((GSourceFunc)map_download_timeout, NULL);

if (!_num_downloads++)
	gtk_progress_bar_set_text(GTK_PROGRESS_BAR(mapp.progress), _("Downloading maps..."));
}

void
map_download_init(void)
{
curl_easy_queue=g_queue_new();
pui_tree=g_tree_new_full((GCompareDataFunc)download_comparefunc, NULL, (GDestroyNotify) progress_update_info_free, NULL);
downloading_tree=g_tree_new_full((GCompareDataFunc)download_comparefunc, NULL, (GDestroyNotify) progress_update_info_free, NULL);
pui_by_easy=g_hash_table_new(g_direct_hash, g_direct_equal);
_curl_multi=curl_multi_init();
_curl_sid=0;
/* curl_multi_setopt(_curl_multi, CURLMOPT_PIPELINING, 1); */
}

gboolean
map_download_stop(void)
{
CURL *curr;
CURLMsg *msg;
gint num_msgs;

if (!_curl_multi)
	return FALSE;

/* Clear the tree and create it again */
if (pui_tree) {
	g_tree_destroy(pui_tree);
	pui_tree=g_tree_new_full((GCompareDataFunc)download_comparefunc, NULL, (GDestroyNotify) progress_update_info_free, NULL);
}

/* Close all finished files. */
while ((msg=curl_multi_info_read(_curl_multi, &num_msgs))) {
	if (msg->msg!=CURLMSG_DONE)
		continue;
	/* This is a map download. */
	ProgressUpdateInfo *pui;

	pui=g_hash_table_lookup(pui_by_easy, msg->easy_handle);
	if (!pui)
		continue;

	g_queue_push_head(curl_easy_queue, msg->easy_handle);
	g_hash_table_remove(pui_by_easy, msg->easy_handle);
	fclose(pui->file);
	curl_multi_remove_handle(_curl_multi, msg->easy_handle);
}

g_debug("Stopping downloads");
while ((curr=g_queue_pop_tail(curl_easy_queue)))
	curl_easy_cleanup(curr);
return TRUE;
}

/*
 * Clean up CURL and structures
 */
void
map_download_deinit(void) 
{
if (_curl_multi) {
	gint num_transfers;

	/* Finish up all downloads. */
	while (CURLM_CALL_MULTI_PERFORM == curl_multi_perform(_curl_multi, &num_transfers) || num_transfers) {
		g_debug("NT: %d", num_transfers);
	}

	/* Stop all transfers and cleanup */
	map_download_stop();

	if (pui_tree)
		g_tree_destroy(pui_tree);
	if (downloading_tree)
		g_tree_destroy(downloading_tree);
	if (pui_by_easy)
		g_hash_table_destroy(pui_by_easy);
	if (curl_easy_queue)
		g_queue_free(curl_easy_queue);
	curl_multi_cleanup(_curl_multi);
	_curl_multi = NULL;
}

if (_curl_sid) {
    g_source_remove(_curl_sid);
    _curl_sid=0;
}
}
