/*
 * This file is part of mapper
 *
 * Copyright (C) 2007-2010 Kaj-Michael Lang
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */
#include <config.h>

#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <strings.h>
#include <stddef.h>
#include <locale.h>
#include <math.h>
#include <errno.h>
#include <sys/wait.h>
#include <glib/gstdio.h>
#include <gtk/gtk.h>
#include <fcntl.h>
#include <libgnomevfs/gnome-vfs.h>
#include <curl/multi.h>
#include <glib/gi18n.h>

#include <gdk/gdkx.h>
#include <X11/Xatom.h>

#include "gtkgps.h"
#include "gtkcompass.h"

#include "mapper.h"
#include "hildon-mapper.h"
#include "utils.h"
#include "poi.h"
#include "route.h"
#include "track.h"
#include "settings.h"
#include "gps.h"
#include "map.h"
#include "mapper-types.h"
#include "ui-common.h"
#include "db.h"
#include "gps-panels.h"
#include "gps-conn.h"
#include "filter.h"
#include "dialogs.h"
#include "settings-gui.h"
#include "map-repo.h"
#include "poi-gui.h"

/* Callbacks */
#include "cb.h"

/* Default size */
#define WINDOW_SIZE_X 640
#define WINDOW_SIZE_Y 480

static poi_quick_data tqp;

#ifndef GTK_STOCK_INFO
#define GTK_STOCK_INFO GTK_STOCK_PROPERTIES
#endif

static gboolean mapper_cb_quit(GtkWidget *widget, GdkEvent *event, gpointer data);
static gboolean cb_quick_poi_display(GtkAction *action);

#ifdef WITH_HILDON
#include "ui-menu-hildon.h"
#else
#include "ui-menu-desktop.h"
#endif
#include "ui-actions.h"

/************************************************************************************/

void
set_action_sensitive(const char *name, gboolean sensitive)
{
GtkAction *action=gtk_action_group_get_action(mapp.actions, name);
gtk_action_set_sensitive(action, sensitive);
}

static GtkWidget *
set_action_proxy(const char *name, GtkWidget *widget)
{
GtkAction *action=gtk_action_group_get_action(mapp.actions, name);
#if GTK_CHECK_VERSION(2,18,0)
gtk_activatable_set_related_action(GTK_ACTIVATABLE(widget), action);
#else
gtk_action_connect_proxy(action, widget);
#endif
return widget;
}

void
set_action_activate(const char *name, gboolean active)
{
GtkAction *action=gtk_action_group_get_action(mapp.actions, name);
gtk_toggle_action_set_active(GTK_TOGGLE_ACTION(action), active);
}

#if 0
void
set_radio_action_value(const char *name, gint value)
{
GtkAction *action=gtk_action_group_get_action(mapp.actions, name);
gtk_radio_action_set_current_value(action, value);
}
#endif

/************************************************************************************/

static gboolean
cb_quick_poi_display(GtkAction *action)
{
gtk_notebook_set_current_page(mapp.nb, mapp.nb_qpoi);
return TRUE;
}

static gboolean
mapper_cb_quit(GtkWidget *widget, GdkEvent  *event, gpointer data)
{
if (dialog_confirm(mapp.mainwindow, _("Are sure you'd like to quit ?"))==TRUE) {
	gtk_main_quit();
	return FALSE;
}
return TRUE;
}

static void 
mapper_menu_maps_fill(void)
{
GList *curr;
static gint id=0;
gint i=0;

if (id>0) {
	gtk_ui_manager_remove_ui(mapp.ui, id);
} else {
	id=gtk_ui_manager_new_merge_id(mapp.ui);
}

if (mapp.repo_actions) {
	gtk_ui_manager_remove_action_group(mapp.ui, mapp.repo_actions);
	g_object_unref(mapp.repo_actions);
}

mapp.repo_actions=gtk_action_group_new("map_repos_actions");
g_assert(mapp.repo_actions);
gtk_ui_manager_insert_action_group(mapp.ui, mapp.repo_actions, 0);
gtk_ui_manager_ensure_update(mapp.ui);

for (curr=g_list_last(_repo_list); curr; curr=curr->prev) {
	RepoData *rd = (RepoData *) curr->data;
	GtkAction *action;
	const gchar *name;

	name=g_strdup_printf("MapRepo%u", i++);
	action=gtk_action_new(name, rd->name, NULL, NULL);
	gtk_action_group_add_action(mapp.repo_actions, action);
	g_signal_connect(G_OBJECT(action), "activate", G_CALLBACK(menu_cb_maps_select), curr->data);
	gtk_ui_manager_add_ui(mapp.ui, id, "/MenuBar/map/map_maps", rd->name, name, GTK_UI_MANAGER_MENUITEM, FALSE);
	g_object_unref(action);
}

}

/**
 * Call gtk_window_present() on Mapper.
 */
gboolean 
mapper_window_present(void)
{
gtk_window_present(GTK_WINDOW(mapp.mainwindow));
return FALSE;
}

static GtkMenu *
menu_init_cmenu(void)
{
/* Create needed handles. */
GtkMenu *menu;
GtkWidget *submenu;
GtkWidget *menu_item;

/* Setup the context menu. */
menu=GTK_MENU(gtk_menu_new());

/* Setup the map context menu. */
gtk_menu_append(menu, menu_item	= gtk_menu_item_new_with_label(_("Location")));
gtk_menu_item_set_submenu(GTK_MENU_ITEM(menu_item), submenu = gtk_menu_new());

/* Setup the map context menu. */
gtk_menu_append(submenu, _cmenu_loc_add_poi = gtk_menu_item_new_with_label(_("Add POI...")));
gtk_menu_append(submenu, gtk_separator_menu_item_new());
gtk_menu_append(submenu, _cmenu_loc_show_latlon_item= gtk_menu_item_new_with_label(_("Show Lat/Lon")));
gtk_menu_append(submenu, _cmenu_loc_clip_latlon_item = gtk_menu_item_new_with_label(_("Copy Lat/Lon")));
gtk_menu_append(submenu, gtk_separator_menu_item_new());
gtk_menu_append(submenu, _cmenu_loc_distance_to_item = gtk_menu_item_new_with_label(_("Show Distance to")));
gtk_menu_append(submenu, _cmenu_loc_route_to_item= gtk_menu_item_new_with_label(_("Download Route to...")));
gtk_menu_append(submenu, gtk_separator_menu_item_new());
gtk_menu_append(submenu, _cmenu_loc_add_route= gtk_menu_item_new_with_label(_("Add Route Point")));
gtk_menu_append(submenu, _cmenu_loc_add_way = gtk_menu_item_new_with_label(_("Add Waypoint...")));
gtk_menu_append(submenu, gtk_separator_menu_item_new());
gtk_menu_append(submenu, _cmenu_loc_set_gps = gtk_menu_item_new_with_label(_("Set as GPS Location")));
gtk_menu_append(submenu, _cmenu_loc_set_home = gtk_menu_item_new_with_label(_("Set as Home Location")));
gtk_menu_append(submenu, _cmenu_loc_set_destination = gtk_menu_item_new_with_label(_("Set as Destination")));

/* Setup the waypoint context menu. */
gtk_menu_append(menu, menu_item = gtk_menu_item_new_with_label(_("Waypoint")));
gtk_menu_item_set_submenu(GTK_MENU_ITEM(menu_item), submenu = gtk_menu_new());

gtk_menu_append(submenu, _cmenu_way_show_latlon_item = gtk_menu_item_new_with_label(_("Show Lat/Lon")));
gtk_menu_append(submenu, _cmenu_way_show_desc_item = gtk_menu_item_new_with_label(_("Show Description")));
gtk_menu_append(submenu, _cmenu_way_clip_latlon_item = gtk_menu_item_new_with_label(_("Copy Lat/Lon")));
gtk_menu_append(submenu, _cmenu_way_clip_desc_item = gtk_menu_item_new_with_label(_("Copy Description")));
gtk_menu_append(submenu, gtk_separator_menu_item_new());
gtk_menu_append(submenu, _cmenu_way_distance_to_item = gtk_menu_item_new_with_label(_("Show Distance to")));
gtk_menu_append(submenu, _cmenu_way_route_to_item = gtk_menu_item_new_with_label(_("Download Route to...")));
gtk_menu_append(submenu, _cmenu_way_delete_item = gtk_menu_item_new_with_label(_("Delete...")));
gtk_menu_append(submenu, gtk_separator_menu_item_new());
gtk_menu_append(submenu, _cmenu_way_add_poi = gtk_menu_item_new_with_label(_("Add POI...")));
gtk_widget_set_sensitive(_cmenu_way_add_poi, _db != NULL);
gtk_menu_append(submenu, gtk_separator_menu_item_new());
gtk_menu_append(submenu, _cmenu_way_goto_nextway = gtk_menu_item_new_with_label(_("Go to Next")));

/* Setup the POI context menu. */
gtk_menu_append(menu, _cmenu_poi = gtk_menu_item_new_with_label(_("POI")));
gtk_widget_set_sensitive(_cmenu_poi, _db != NULL);
gtk_menu_item_set_submenu(GTK_MENU_ITEM(_cmenu_poi), _menu_poi = gtk_menu_new());

gtk_menu_append(_menu_poi, _cmenu_poi_info = gtk_menu_item_new_with_label(_("Information...")));
gtk_menu_append(_menu_poi, gtk_separator_menu_item_new());
gtk_menu_append(_menu_poi, _cmenu_poi_edit_poi = gtk_menu_item_new_with_label(_("Edit...")));
gtk_menu_append(_menu_poi, _cmenu_poi_distance_to_item = gtk_menu_item_new_with_label(_("Show Distance to")));
gtk_menu_append(_menu_poi, _cmenu_poi_route_to_item = gtk_menu_item_new_with_label(_("Download Route to...")));
gtk_menu_append(_menu_poi, gtk_separator_menu_item_new());
gtk_menu_append(_menu_poi, _cmenu_poi_add_route = gtk_menu_item_new_with_label(_("Add Route Point")));
gtk_menu_append(_menu_poi, _cmenu_poi_add_way	= gtk_menu_item_new_with_label(_("Add Waypoint...")));
gtk_menu_append(_menu_poi, gtk_separator_menu_item_new());
gtk_menu_append(_menu_poi, _cmenu_poi_goto_nearpoi = gtk_menu_item_new_with_label(_("Go to")));

/* Connect signals for context menu. */
g_signal_connect(G_OBJECT(_cmenu_loc_show_latlon_item), "activate", G_CALLBACK(cmenu_cb_loc_show_latlon), NULL);
g_signal_connect(G_OBJECT(_cmenu_loc_clip_latlon_item), "activate", G_CALLBACK(cmenu_cb_loc_clip_latlon), NULL);
g_signal_connect(G_OBJECT(_cmenu_loc_route_to_item), "activate", G_CALLBACK(cmenu_cb_loc_route_to), NULL);
g_signal_connect(G_OBJECT(_cmenu_loc_distance_to_item), "activate", G_CALLBACK(cmenu_cb_loc_distance_to), NULL);
g_signal_connect(G_OBJECT(_cmenu_loc_add_route), "activate", G_CALLBACK(cmenu_cb_loc_add_route), NULL);
g_signal_connect(G_OBJECT(_cmenu_loc_add_way), "activate", G_CALLBACK(cmenu_cb_loc_add_way), NULL);
g_signal_connect(G_OBJECT(_cmenu_loc_add_poi), "activate", G_CALLBACK(cmenu_cb_loc_add_poi), NULL);
g_signal_connect(G_OBJECT(_cmenu_loc_set_gps), "activate", G_CALLBACK(cmenu_cb_loc_set_gps), NULL);
g_signal_connect(G_OBJECT(_cmenu_loc_set_home), "activate", G_CALLBACK(cmenu_cb_loc_set_home), NULL);
g_signal_connect(G_OBJECT(_cmenu_loc_set_destination), "activate", G_CALLBACK(cmenu_cb_loc_set_destination), NULL);

g_signal_connect(G_OBJECT(_cmenu_way_show_latlon_item), "activate", G_CALLBACK(cmenu_cb_way_show_latlon), NULL);
g_signal_connect(G_OBJECT(_cmenu_way_show_desc_item), "activate", G_CALLBACK(cmenu_cb_way_show_desc), NULL);
g_signal_connect(G_OBJECT(_cmenu_way_clip_latlon_item), "activate", G_CALLBACK(cmenu_cb_way_clip_latlon), NULL);
g_signal_connect(G_OBJECT(_cmenu_way_clip_desc_item), "activate", G_CALLBACK(cmenu_cb_way_clip_desc), NULL);
g_signal_connect(G_OBJECT(_cmenu_way_route_to_item), "activate", G_CALLBACK(cmenu_cb_way_route_to), NULL);
g_signal_connect(G_OBJECT(_cmenu_way_distance_to_item), "activate", G_CALLBACK(cmenu_cb_way_distance_to), NULL);
g_signal_connect(G_OBJECT(_cmenu_way_delete_item), "activate", G_CALLBACK(cmenu_cb_way_delete), NULL);
g_signal_connect(G_OBJECT(_cmenu_way_add_poi), "activate", G_CALLBACK(cmenu_cb_way_add_poi), NULL);
g_signal_connect(G_OBJECT(_cmenu_way_goto_nextway), "activate", G_CALLBACK(menu_cb_goto_nextway), NULL);

g_signal_connect(G_OBJECT(_cmenu_poi_info), "activate", G_CALLBACK(cmenu_cb_poi_show_poi), &map_poi_id);
g_signal_connect(G_OBJECT(_cmenu_poi_edit_poi), "activate", G_CALLBACK(cmenu_cb_poi_edit_poi), &map_poi_id);
g_signal_connect(G_OBJECT(_cmenu_poi_route_to_item), "activate", G_CALLBACK(cmenu_cb_poi_route_to), &map_poi_id);
g_signal_connect(G_OBJECT(_cmenu_poi_distance_to_item), "activate", G_CALLBACK(cmenu_cb_poi_distance_to), &map_poi_id);
g_signal_connect(G_OBJECT(_cmenu_poi_add_route), "activate", G_CALLBACK(cmenu_cb_poi_add_route), &map_poi_id);
g_signal_connect(G_OBJECT(_cmenu_poi_add_way), "activate", G_CALLBACK(cmenu_cb_poi_add_way), &map_poi_id);
g_signal_connect(G_OBJECT(_cmenu_poi_goto_nearpoi), "activate", G_CALLBACK(cmenu_cb_goto_poi), &map_poi_id);

gtk_widget_show_all(GTK_WIDGET(menu));

return menu;
}

static void
mapper_set_landscape_mode(void)
{
#ifdef WITH_MCE
g_debug("Wide/Landscape");
set_action_activate("view_toolbar", TRUE);
#endif
}

static void
mapper_set_portrait_mode(void)
{
#ifdef WITH_MCE
g_debug("Narrow/Portrait");
set_action_activate("view_toolbar", FALSE);
#endif
}

static gboolean
mapper_screen_size_changed(GdkScreen *screen, gpointer data)
{
gint h,w;

h=gdk_screen_get_height(screen);
w=gdk_screen_get_width(screen);

if (h>w)
	mapper_set_portrait_mode();
else
	mapper_set_landscape_mode();

return FALSE;
}

GtkWidget *
mapper_window_new(void)
{
GtkWidget *window;

#ifdef WITH_HILDON
mapper_app.program=HILDON_PROGRAM(hildon_program_get_instance());
hildon_program_set_can_hibernate(mapper_app.program, FALSE);
window=GTK_WIDGET(hildon_window_new());
hildon_program_add_window(mapper_app.program, HILDON_WINDOW(window));

#if HILDON_CHECK_VERSION(2,2,2)
hildon_gtk_window_enable_zoom_keys(GTK_WINDOW(window), TRUE);
hildon_gtk_window_set_portrait_flags(GTK_WINDOW(window), HILDON_PORTRAIT_MODE_SUPPORT);
he_fullscreen_button_new(GTK_WINDOW(window));
#endif

#else
window=gtk_window_new(GTK_WINDOW_TOPLEVEL);
gtk_window_set_title(GTK_WINDOW(window), "Mapper");
gtk_window_set_resizable(GTK_WINDOW(window), TRUE);
gtk_widget_set_size_request(GTK_WIDGET(window), WINDOW_SIZE_X, WINDOW_SIZE_Y);
gtk_window_resize(GTK_WINDOW(window), WINDOW_SIZE_X, WINDOW_SIZE_Y);
#endif

return window;
}

/**
 * Set sensitivity of route menu items
 */
void
route_menu_set_sensitive(gboolean s)
{
set_action_sensitive("route_save", s);
set_action_sensitive("route_dist_next", s);
set_action_sensitive("route_dist_last", s);
set_action_sensitive("route_reset", s);
set_action_sensitive("route_clear", s);
}

/**
 * Create a tree view inside given container
 */
static GtkWidget *
mapper_tree_view_new(GtkWidget *container)
{
GtkWidget *tree_view, *sw;
GtkTreeViewColumn *column;
GtkCellRenderer *renderer;

#ifndef WITH_HILDON_2_2
sw=gtk_scrolled_window_new(NULL, NULL);
gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(sw), GTK_POLICY_NEVER, GTK_POLICY_AUTOMATIC);
tree_view=gtk_tree_view_new();
#else
sw=hildon_pannable_area_new();
tree_view=hildon_gtk_tree_view_new(HILDON_UI_MODE_EDIT);
#endif

gtk_container_add(GTK_CONTAINER(sw), tree_view);

gtk_box_pack_start(GTK_BOX(container), sw, TRUE, TRUE, 0);

gtk_tree_selection_set_mode(gtk_tree_view_get_selection(GTK_TREE_VIEW(tree_view)), GTK_SELECTION_SINGLE);
gtk_tree_view_set_headers_visible(GTK_TREE_VIEW(tree_view), TRUE);

renderer=gtk_cell_renderer_text_new();
column=gtk_tree_view_column_new_with_attributes(_("Location"), renderer, "text", ROUTE_LATLON, NULL);
gtk_tree_view_append_column(GTK_TREE_VIEW(tree_view), column);

renderer=gtk_cell_renderer_text_new();
column=gtk_tree_view_column_new_with_attributes(_("Distance"), renderer, "text", ROUTE_DISTANCE, NULL);
gtk_tree_view_append_column(GTK_TREE_VIEW(tree_view), column);

renderer=gtk_cell_renderer_text_new();
column=gtk_tree_view_column_new_with_attributes(_("Waypoint"), renderer, "text", ROUTE_WAYPOINT, NULL);
gtk_tree_view_append_column(GTK_TREE_VIEW(tree_view), column);

return tree_view;
}

GtkWidget *
mapper_sidebar_button_new(const gchar *action, const gchar *stock_id)
{
GtkWidget *b;

#ifdef WITH_HILDON_2_2
b=hildon_gtk_button_new(HILDON_SIZE_FINGER_HEIGHT);
#else
b=gtk_button_new();
#endif
b=set_action_proxy(action, b);
gtk_button_set_image(GTK_BUTTON(b), gtk_image_new_from_stock(stock_id, GTK_ICON_SIZE_BUTTON));
gtk_button_set_focus_on_click(GTK_BUTTON(b), FALSE);
return b;
}

static GtkWidget *
mapper_info_banner(void)
{
GtkWidget *lhbox, *dhbox, *vbox;

mapp.ib.container=gtk_hbox_new(FALSE, 1);
mapp.ib.location_ref=gtk_label_new("");
mapp.ib.location_intref=gtk_label_new("");
mapp.ib.location_name=gtk_label_new("");
mapp.ib.speed=gtk_label_new("");
mapp.ib.distance=gtk_label_new("");
mapp.ib.time=gtk_label_new("--:--");

mapp.ib.op=gtk_image_new();
mapp.ib.ope=gtk_event_box_new();
gtk_container_add(GTK_CONTAINER(mapp.ib.ope), mapp.ib.op);

vbox=gtk_vbox_new(FALSE, 1);

gtk_box_pack_start(GTK_BOX(mapp.ib.container), mapp.ib.ope, FALSE, FALSE, 0);
gtk_box_pack_start(GTK_BOX(mapp.ib.container), vbox, TRUE, TRUE, 0);

#if 0
gtk_misc_set_alignment(GTK_MISC(mapp.ib.location_intref), 0.f, 0.5f);
gtk_misc_set_alignment(GTK_MISC(mapp.ib.location_ref), 0.f, 0.5f);
gtk_misc_set_alignment(GTK_MISC(mapp.ib.location_name), 0.f, 0.5f);
gtk_misc_set_alignment(GTK_MISC(mapp.ib.speed), 1.f, 0.5f);
gtk_misc_set_alignment(GTK_MISC(mapp.ib.distance), 1.f, 0.5f);
#endif

lhbox=gtk_hbox_new(FALSE, 1);
gtk_box_pack_start(GTK_BOX(lhbox), mapp.ib.location_intref, FALSE, FALSE, 0);
gtk_box_pack_start(GTK_BOX(lhbox), mapp.ib.location_ref, FALSE, FALSE, 0);
gtk_box_pack_start(GTK_BOX(lhbox), mapp.ib.location_name, TRUE, TRUE, 0);

dhbox=gtk_hbox_new(FALSE, 1);
gtk_box_pack_start(GTK_BOX(dhbox), mapp.ib.distance, TRUE, TRUE, 0);
gtk_box_pack_end(GTK_BOX(dhbox), mapp.ib.speed, TRUE, TRUE, 0);

gtk_box_pack_start(GTK_BOX(vbox), lhbox, TRUE, FALSE, 0);
gtk_box_pack_start(GTK_BOX(vbox), dhbox, TRUE, FALSE, 0);

return mapp.ib.container;
}

/**
 * mapper_ui_init:
 *
 * Create the Mapper main UI in given window.
 */
MapperUi *
mapper_ui_init(GtkWidget *mw)
{
GError *error=NULL;
GtkWidget *hbox, *mapvbox, *label, *vbox_main, *vbox;
GdkScreen *screen;
GtkToolItem *item;
const gchar *notes_base_dir=NULL;
Dicto *d;
DictoFormats f=FORMAT_DEFAULT;

mapp.mainwindow=mw;
gtk_container_set_border_width(GTK_CONTAINER(mapp.mainwindow), 0);

mapp.actions=gtk_action_group_new("Actions");
mapp.ui=gtk_ui_manager_new();

gtk_action_group_add_actions(mapp.actions, ui_entries, n_ui_entries, NULL);
gtk_action_group_add_toggle_actions(mapp.actions, ui_toggle_entries, n_ui_toggle_entries, NULL);
gtk_action_group_add_radio_actions(mapp.actions, ui_autocenter_entries, n_ui_autocenter_entries, _center_mode, G_CALLBACK(menu_cb_autocenter),  NULL);

gtk_ui_manager_insert_action_group(mapp.ui, mapp.actions, 0);
g_object_unref(mapp.actions);

/* N810/N900 has a keyboard so set this up on hildon too */
gtk_window_add_accel_group(GTK_WINDOW(mapp.mainwindow), gtk_ui_manager_get_accel_group(mapp.ui));

if (!gtk_ui_manager_add_ui_from_string (mapp.ui, mapper_ui, -1, &error)) {
	g_error("Building menus failed: %s", error->message);
	g_error_free(error);
	return NULL;
}

/* Menu and Toolbar */
mapp.menu_bar=gtk_ui_manager_get_widget (mapp.ui, "/MenuBar");
g_return_val_if_fail(mapp.menu_bar, NULL);

mapp.toolbar=gtk_ui_manager_get_widget (mapp.ui, "/ToolBar");
g_return_val_if_fail(mapp.toolbar, NULL);

/* Toolbar progress item */
item=gtk_tool_item_new();
mapp.progress=gtk_progress_bar_new();
gtk_tool_item_set_expand(item, TRUE);
gtk_progress_bar_set_ellipsize(GTK_PROGRESS_BAR(mapp.progress), PANGO_ELLIPSIZE_END);
gtk_progress_bar_set_pulse_step(GTK_PROGRESS_BAR(mapp.progress), 0.2);
gtk_container_add(GTK_CONTAINER(item),GTK_WIDGET(mapp.progress));
gtk_toolbar_insert(GTK_TOOLBAR(mapp.toolbar), item, -1);

/* Notebook with different views */
mapp.nb=gtk_notebook_new();
gtk_notebook_set_show_tabs(GTK_NOTEBOOK(mapp.nb), TRUE);

vbox_main=gtk_vbox_new(FALSE, 0);
gtk_container_add(GTK_CONTAINER(mapp.mainwindow), vbox_main);

#ifdef WITH_HILDON
/* Create and add widgets and supporting data. */
gtk_box_pack_start(GTK_BOX(vbox_main), GTK_WIDGET(mapp.nb), TRUE, TRUE, 0);
{
	GtkMenu *hmenu;
	hmenu=GTK_MENU(gtk_menu_new());

	gtk_container_foreach(GTK_CONTAINER(mapp.menu_bar), (GtkCallback)gtk_widget_reparent, hmenu);
	hildon_window_set_menu(HILDON_WINDOW(mapp.mainwindow), hmenu);
}
hildon_window_add_toolbar(mapp.mainwindow, GTK_TOOLBAR(mapp.toolbar));
gtk_widget_show_all(GTK_WIDGET(mapp.toolbar));
#else
mapp.status_bar=gtk_statusbar_new();
gtk_box_pack_start(GTK_BOX(vbox_main), mapp.menu_bar, FALSE, FALSE, 0);
gtk_box_pack_start(GTK_BOX(vbox_main), mapp.toolbar, FALSE, FALSE, 0);
gtk_box_pack_start(GTK_BOX(vbox_main), GTK_WIDGET(mapp.nb), TRUE, TRUE, 0);
gtk_box_pack_start(GTK_BOX(vbox_main), mapp.status_bar, FALSE, FALSE, 0);
#endif

hbox=gtk_hbox_new(FALSE, 0);

mapp.ib.container=mapper_info_banner();

/* Map "widget" */
mapp.map_widget=map_new();
gtk_box_pack_start(GTK_BOX(hbox), mapp.map_widget, TRUE, TRUE, 0);

mapvbox=gtk_vbox_new(FALSE, 0);
gtk_box_pack_start(GTK_BOX(mapvbox), hbox, TRUE, TRUE, 0);
gtk_box_pack_start(GTK_BOX(mapvbox), GTK_WIDGET(mapp.ib.container), FALSE, FALSE, 0);
mapp.nb_map=gtk_notebook_append_page(mapp.nb, mapvbox, gtk_label_new("Map"));

/* Heading Tab */
vbox=gtk_vbox_new(FALSE, 0);
mapp.tab_compass=gtk_compass_new(&_gps->data);
gtk_box_pack_start(GTK_BOX(vbox), mapp.tab_compass, TRUE, TRUE, 0);
mapp.nb_heading=gtk_notebook_append_page(mapp.nb, vbox, gtk_label_new("Heading"));

/* Track Tab */
hbox=gtk_hbox_new(FALSE, 0);
vbox=gtk_vbox_new(FALSE, 0);

mapp.track_tree_view=mapper_tree_view_new(hbox);
g_signal_connect(G_OBJECT(mapp.track_tree_view), "row-activated", G_CALLBACK(path_waypoint_activated_cb), NULL);

gtk_box_pack_start(GTK_BOX(hbox), vbox, FALSE, FALSE, 0);
gtk_box_pack_start(GTK_BOX(vbox), mapper_sidebar_button_new("track_open", GTK_STOCK_OPEN), FALSE, FALSE, 0);
gtk_box_pack_start(GTK_BOX(vbox), mapper_sidebar_button_new("track_save", GTK_STOCK_SAVE), FALSE, FALSE, 0);
gtk_box_pack_start(GTK_BOX(vbox), mapper_sidebar_button_new("track_clear", GTK_STOCK_CLEAR), FALSE, FALSE, 0);
gtk_box_pack_start(GTK_BOX(vbox), mapper_sidebar_button_new("track_insert_mark", GTK_STOCK_NEW), FALSE, FALSE, 0);

mapp.nb_track=gtk_notebook_append_page(mapp.nb, hbox, gtk_label_new(_("Track")));

/* Route Tab */
hbox=gtk_hbox_new(FALSE, 0);
vbox=gtk_vbox_new(FALSE, 0);

mapp.route_tree_view=mapper_tree_view_new(hbox);
g_signal_connect(G_OBJECT(mapp.route_tree_view), "row-activated", G_CALLBACK(path_waypoint_activated_cb), NULL);

gtk_box_pack_start(GTK_BOX(hbox), vbox, FALSE, FALSE, 0);
gtk_box_pack_start(GTK_BOX(vbox), mapper_sidebar_button_new("route_download", GTK_STOCK_GO_DOWN), FALSE, FALSE, 0);
gtk_box_pack_start(GTK_BOX(vbox), mapper_sidebar_button_new("route_open", GTK_STOCK_OPEN), FALSE, FALSE, 0);
gtk_box_pack_start(GTK_BOX(vbox), mapper_sidebar_button_new("route_save", GTK_STOCK_SAVE), FALSE, FALSE, 0);
gtk_box_pack_start(GTK_BOX(vbox), mapper_sidebar_button_new("route_clear", GTK_STOCK_CLEAR), FALSE, FALSE, 0);

mapp.nb_route=gtk_notebook_append_page(mapp.nb, hbox, gtk_label_new(_("Route")));

/* Quick POI Tab */
tqp.fixed=FALSE;
tqp.close=FALSE;
tqp.note=mapp.nb;
tqp.tab=mapp.nb_map;
vbox=poi_quick_button_box(&tqp);
mapp.nb_qpoi=gtk_notebook_append_page(mapp.nb, vbox, gtk_label_new("QPOI"));

/* GPS Tab */
hbox=gtk_hpaned_new();
mapp.gps_widget=gps_info_panel();
gtk_widget_set_size_request(GTK_WIDGET(mapp.gps_widget), 180, 0);
gtk_paned_add1(GTK_PANED(hbox), mapp.gps_widget);

vbox=gtk_vbox_new(FALSE, 0);

mapp.gps_sat_view=gtk_gps_new(GTK_GPS_MODE_SKY, &_gps->data);
gtk_box_pack_start(GTK_BOX(vbox), mapp.gps_sat_view, TRUE, TRUE, 0);

mapp.gps_sat_view=gtk_gps_new(GTK_GPS_MODE_SIGNAL, &_gps->data);
gtk_box_pack_start(GTK_BOX(vbox), mapp.gps_sat_view, TRUE, TRUE, 0);

gtk_paned_add2(GTK_PANED(hbox), vbox);
mapp.nb_gps=gtk_notebook_append_page(mapp.nb, hbox, gtk_label_new("GPS"));

/* Audio tab */
label=gtk_label_new("Audio");
#ifndef NOTES_BASEDIR
notes_base_dir=g_get_user_special_dir(G_USER_DIRECTORY_MUSIC);
if (!notes_base_dir)
	notes_base_dir=g_get_home_dir();
#else
notes_base_dir=NOTES_BASEDIR;
#endif

/* Dictation interface */
d=dicto_new_full(notes_base_dir, f, AUDIO_SINK, AUDIO_SRC);
mapp.audio_ui=dicto_widget_new(d);
mapp.nb_audio=gtk_notebook_append_page(mapp.nb, mapp.audio_ui, label);
g_signal_connect(G_OBJECT(d), "recording", G_CALLBACK(dicto_recording_cb), NULL);

mapper_menu_maps_fill();

#ifdef WITH_DEVICE_MAEMO
gtk_widget_tap_and_hold_setup(mapp.map_widget, GTK_WIDGET(menu_init_cmenu()), NULL, 0);
#else
_menu_map=menu_init_cmenu();
#endif

memset(&_autoroute_data, 0, sizeof(_autoroute_data));

gtk_widget_show_all(GTK_WIDGET(vbox_main));
gtk_widget_show_all(GTK_WIDGET(mapp.mainwindow));

/* Initialize our line styles. */
update_gcs();

/* Set defaults for items */
set_action_activate("map_auto_download", _auto_download);
set_action_activate("gps_enabled", _enable_gps);
set_action_sensitive("goto_gps", _enable_gps);

set_action_sensitive("autocenter_latlon", _enable_gps);
set_action_sensitive("autocenter_lead", _enable_gps);

set_action_activate("view_toolbar", TRUE);

set_action_activate("view_scale", _show_scale);
set_action_activate("view_vector", _show_velvec);
set_action_activate("view_poi", _show_poi);

set_action_activate("view_track", _show_tracks & TRACKS_MASK);
set_action_activate("view_route", _show_tracks & ROUTES_MASK);

route_menu_set_sensitive(FALSE);

/* Connect signals. */
g_signal_connect(G_OBJECT(mapp.mainwindow), "delete_event", G_CALLBACK(mapper_cb_quit), NULL);
g_signal_connect(G_OBJECT(mapp.mainwindow), "destroy", G_CALLBACK(gtk_main_quit), NULL);
g_signal_connect(G_OBJECT(mapp.mainwindow), "key_press_event", G_CALLBACK(window_cb_key_press), NULL);
g_signal_connect(G_OBJECT(mapp.mainwindow), "key_release_event", G_CALLBACK(window_cb_key_release), NULL);

screen=gdk_screen_get_default();
if (screen) {
	g_signal_connect(G_OBJECT(screen), "size-changed", G_CALLBACK(mapper_screen_size_changed), &mapp);	
	mapper_screen_size_changed(screen, &mapp);
} else {
	g_warning("No screen ?");
}

gtk_idle_add((GSourceFunc)mapper_window_present, NULL);
return &mapp;
}

#define PAYPAL_URL "https://www.paypal.com/cgi-bin/webscr?cmd=_donations&business=EHQCWLH6E4U5N&lc=FI&item_name=tal%2eorg&item_number=mapper%2eapp&currency_code=EUR&bn=PP%2dDonationsBF%3abtn_donateCC_LG%2egif%3aNonHosted"
#define MAPPER_URL "http://mapper.tal.org/"
#define BUG_URL "https://garage.maemo.org/tracker/?atid=1529&group_id=402&func=browse"

void
mapper_about_dialog(void)
{
gchar *authors[]={
	"Kaj-Michael Lang",
	"John Costigan",
	"Cezary Jackiewicz", NULL
};

#ifdef WITH_HILDON_2_2
he_about_dialog_present(GTK_WINDOW(mapp.mainwindow),
	"Mapper",
	"mapper",
	VERSION,
	"Mapping application",
	"Mapper is free software, GPLv2\n"
	"Copyright 2007-2010 Kaj-Michael Lang\n"
	"Copyright 2006-2007 John Costigan, Cezary Jackiewicz\n\n"
	"Map tiles provided by OpenStreetMap and Cloudmade are\nunder Creative Commons Attribution-Share Alike 2.0 license,\n"
	"See http://creativecommons.org/licenses/by-sa/2.0/legalcode\n\n"
	"This product uses the CloudMade APIs,\nbut is not endorsed or certified by CloudMade.\nhttp://cloudmade.com/",
	MAPPER_URL,
	BUG_URL,
	PAYPAL_URL);
#else
gtk_show_about_dialog(GTK_WINDOW(mapp.mainwindow), 
	"name", "Mapper",
	"comments", "Mapping application",
	"version", VERSION, 
	"copyright", "Kaj-Michael Lang",
	"website", MAPPER_URL,
	"license", 
	"Mapper is GPLv2\n\n"
	"Map tiles provided by OpenStreetMap and Cloudmade are under Creative Commons Attribution-Share Alike 2.0 license,\n"
	"See http://creativecommons.org/licenses/by-sa/2.0/legalcode\n\n"
	"This product uses the CloudMade APIs, but is not endorsed or certified by CloudMade. http://cloudmade.com/",
	"authors", authors,
	NULL);
#endif
}

gboolean
gps_info_update(Gps *gps)
{
g_return_val_if_fail(gps, FALSE);
gps_display_data(&gps->data);
gps_display_data_speed(mapp.ib.speed, gps->data.speed, _speed_excess, "x-large");

return FALSE;
}

gboolean
gps_satellite_update(Gps *gps)
{
g_return_val_if_fail(gps, FALSE);
if (_satdetails_on)
    gps_display_details(&gps->data);

return FALSE;
}

gboolean
gps_location_update(Gps *gps)
{
g_return_val_if_fail(gps, FALSE);

/* Ignore track filter when routing */
route_check_waypoint_announce(&_gps->data);
route_autoroute_check();

if (filter_check(&filter, &gps->data, &map_loc)==TRUE) {
	track_add(&gps->data);
	map_refresh_mark();
}
return FALSE;
}

/**
 * Simple dialog to ask if we should try to reconnect to GPS
 * returns TRUE or FALSE
 */
gboolean
gps_retry_connection(Gps *gps, const gchar *error)
{
GtkWidget *confirm;
gboolean r;

g_return_val_if_fail(gps, FALSE);
confirm=hildon_note_new_confirmation(GTK_WINDOW(mapp.mainwindow), _("Failed to connect to GPS receiver. Retry?"));
r=(GTK_RESPONSE_OK==gtk_dialog_run(GTK_DIALOG(confirm))) ? TRUE : FALSE;
gtk_widget_destroy(confirm);
return r;
}
