/*
 * This file is part of mapper
 *
 * Copyright (C) 2007 Kaj-Michael Lang
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */
#include "config.h"

#include <unistd.h>
#include <string.h>
#include <strings.h>
#include <sys/types.h>
#include <math.h>
#include <glib.h>
#include <glib/gstdio.h>
#include <glib/gi18n.h>
#include <gtk/gtk.h>
#include <sqlite3.h>

#include "osm.h"
#include "latlon.h"
#include "db.h"
#include "osm-db.h"
#include "settings.h"
#include "osm-sql-tables.h"

/* #define DEBUG_OSM */
/* #define DEBUG_OSM_TIME */
#define OSM_CACHE_MAX_ITEMS (512)

#define OSM_DB_PROGRESS_NUM (30000)

/* Node search ranges */
#define OSM_RANGE_START (16384)
#define OSM_RANGE_STEP  (8192)
#define OSM_RANGE_STOP  (65535)

/* Way range for searching again */
#define OSM_RANGE_WAY (20000)

static sqlite3 *osmdb;
static gboolean osm_db_ok;

struct sql_select_stmt {
	sqlite3_stmt *select_way;
	sqlite3_stmt *select_way2;
	sqlite3_stmt *select_way_next_seg;
	sqlite3_stmt *select_way_prev_seg;

	sqlite3_stmt *select_way_nodes;
	sqlite3_stmt *select_way_name;
	sqlite3_stmt *select_way_name_nls;
	sqlite3_stmt *select_way_name_search;
	sqlite3_stmt *select_way_ref;
	sqlite3_stmt *select_place;
	sqlite3_stmt *select_place_near;
	sqlite3_stmt *select_place_search;
};
static struct sql_select_stmt sql;
static GTimer *dbt;
static GtkProgressBar *dbpw=NULL;

static const gchar *osm_tables[]={
	OSM_TABLE_NODES,
	OSM_TABLE_NODE_TAGS,
	OSM_TABLE_WAY,
	OSM_TABLE_WAY_TAGS,
	OSM_TABLE_WAY_UPDATES,
	OSM_TABLE_WAY_N2N,
	OSM_TABLE_WAY_NAMES,
	OSM_TABLE_WAY_PC,
	OSM_TABLE_WAY_NAMES_NLS,
	OSM_TABLE_WAY_REF,
	OSM_TABLE_PLACES,
	OSM_TABLE_POI,
	OSM_TABLE_POI_CATEGORY, 
	NULL};

static const gchar *osm_indexes[]={
	OSM_INDEX_1,
	OSM_INDEX_2,
#if 0
	/* Disable as we don't do any selects on them */
	OSM_INDEX_3,
	OSM_INDEX_4,
#endif
	OSM_INDEX_5,
	OSM_INDEX_6,
	OSM_INDEX_7,
	OSM_INDEX_9,
	OSM_INDEX_10,
	OSM_INDEX_12,
	OSM_INDEX_13,
	OSM_INDEX_14,
	OSM_INDEX_15,
	OSM_INDEX_16,
	NULL };

/* Cache hash tables */
struct osm_place_cache {
	GHashTable *cache;
	guint hit;
	guint miss;
};
static struct osm_place_cache pcache;

struct osm_name_cache {
	GHashTable *cache;
	guint hit;
	guint miss;
};
static struct osm_name_cache ncache;
static struct osm_name_cache rcache;
static struct osm_name_cache ircache;

static guint way_dist_range=OSM_RANGE_WAY;
static gint sid=0;
static gfloat cspeed=0;
static GList *cways=NULL;

#define QUERY_TIMER_START { g_timer_start(dbt); }
#define QUERY_TIMER_STOP(q) { gulong tms; g_timer_stop(dbt); g_debug("Query %s took: %f sec", q, g_timer_elapsed(dbt, &tms)); }

osm_way_node *osm_way_get_prev_node(osm_way *w);
osm_way_node *osm_way_get_next_node(osm_way *w);

/*****************************************************************************/

gboolean
osm_db_create(sqlite3 *db)
{
g_return_val_if_fail(db, FALSE);

return db_exec_sql_array(db, osm_tables) && db_exec_sql_array(db, osm_indexes);
}

/*****************************************************************************/

void
osm_set_way_range(guint sr)
{
way_dist_range=sr;
}

void
osm_set_way_range_from_speed(gfloat speed)
{
cspeed=speed;
osm_set_way_range(speed>54.0 ? OSM_RANGE_WAY : (OSM_RANGE_WAY-lrint((speed/4)*1000)));
}

/*****************************************************************************/

static gboolean
osm_progress_pulse(void)
{
if (!dbpw)
	return FALSE;
gtk_progress_bar_pulse(dbpw);
return TRUE;
}

static int
osm_progress(void *ud)
{
g_debug("SQL: Running gtk mainloop");
#if 1
while (gtk_events_pending())
	gtk_main_iteration();
#else
gtk_main_iteration_do(FALSE);
#endif
return 0;
}

static void
osm_progress_hide(sqlite3 *db)
{
if (!dbpw)
	return;
gtk_progress_bar_set_text(dbpw, "");
gtk_progress_bar_set_fraction(dbpw, 0.0);
}

static void
osm_progress_show(sqlite3 *db)
{
if (!dbpw)
	return;
gtk_progress_bar_set_text(dbpw, _("Searching..."));
gtk_progress_bar_pulse(dbpw);
gtk_main_iteration_do(FALSE);
}

void
osm_progress_set_widget(sqlite3 *db, GtkProgressBar *w)
{
if (dbpw!=NULL && w==NULL) {
	osm_progress_hide(db);
	if (sid!=0)
		g_source_remove(sid);
	sid=0;
	dbpw=NULL;
	return;
}
dbpw=w;
if (w!=NULL) {
	osm_progress_show(db);
	sid=g_timeout_add(330, (GSourceFunc)osm_progress_pulse, NULL);
}
}

/*****************************************************************************/

gboolean
osm_db_prepare(sqlite3 *db)
{
/* Select nearest place inside lat,lon+-range */
DB_PREP(db, "select name,(($LAT-lat)*($LAT-lat))+(($LON-lon)*($LON-lon)) as d,"
					" lat,lon,nid,isin_p,isin_c "
					" from places where type=$TYPE "
					" and lat between $LAT-$RANGE and $LAT+$RANGE "
					" and lon between $LON-$RANGE and $LON+$RANGE "
					" order by d limit 1", 
					sql.select_place_near);

/* Select place name, location, parent-place and type with given ID */
DB_PREP(db, "select name,lat,lon,type,isin_p,isin_c from places where nid=$NID",
					sql.select_place);

/* Search */
DB_PREP(db, "select nid,name,(($LAT-lat)*($LAT-lat))+(($LON-lon)*($LON-lon)) as d,"
					" lat,lon,type,isin_p,isin_c "
					" from places where "
					" name like $NAME order by d limit 500",
		 			sql.select_place_search);

/* Ways */
/* Select nearest ways inside lat,lon+-range */
DB_PREP(db, "select w.wid,type,nodes,flags,"
					"(($LAT-n.ilat)*($LAT-n.ilat))+(($LON-n.ilon)*($LON-n.ilon)) as d,wn.f,wn.t,n.ilat,n.ilon,w.speed "
					" from way as w,way_n2n as wn,nodes as n "
					" where w.wid=wn.wid and wn.f=n.nid "
					" and n.ilat between $LAT-$RANGE and $LAT+$RANGE "
					" and n.ilon between $LON-$RANGE and $LON+$RANGE "
					" and w.type between $WTS and $WTY " 
					" order by type,d",
					sql.select_way2);

/* Search ways on name */
DB_PREP(db, "select w.wid,w.name as name,"
					"(($LAT-ww.lat)*($LAT-ww.lat))+(($LON-ww.lon)*($LON-ww.lon)) as d,ww.lat,ww.lon "
					" from way_names as w,way as ww where "
					" ww.type between $WTS and $WTY and w.wid=ww.wid and w.name like $NAME "
					" and ww.lat between $LAT-$RANGE and $LAT+$RANGE "
					" and ww.lon between $LON-$RANGE and $LON+$RANGE "
					" union "
					" select w.wid,n.name as name,"
					"(($LAT-ww.lat)*($LAT-ww.lat))+(($LON-ww.lon)*($LON-ww.lon)) as d,ww.lat,ww.lon "
					" from way_names as w, way as ww,way_names_nls as n on w.wid=n.wid where "
					" ww.type between $WTS and $WTY and w.wid=ww.wid and n.name like $NAME "
					" and ww.lat between $LAT-$RANGE and $LAT+$RANGE "
					" and ww.lon between $LON-$RANGE and $LON+$RANGE "
					" order by name limit 200",
					sql.select_way_name_search);

DB_PREP(db, "select wn.t,ilat,ilon from way_n2n as wn,nodes where wid=? and wn.f=? and wn.t=nodes.nid limit 1",
		    sql.select_way_next_seg);

DB_PREP(db, "select wn.f,ilat,ilon from way_n2n as wn,nodes where wid=? and wn.t=? and wn.f=nodes.nid limit 1",
		    sql.select_way_prev_seg);

/* Way name */
DB_PREP(db, "select name from way_names where wid=?", sql.select_way_name);
DB_PREP(db, "select name from way_names_nls where wid=?", sql.select_way_name_nls);

/* Way ref and int_ref */
DB_PREP(db, "select ref,int_ref from way_ref where rid=?", sql.select_way_ref);

return TRUE;
}

void
osm_deinit(void)
{
if (osmdb) {
	if (sql.select_way_ref)
		sqlite3_finalize(sql.select_way_ref);
	if (sql.select_way_name)
		sqlite3_finalize(sql.select_way_name);
	if (sql.select_way_name_nls)
		sqlite3_finalize(sql.select_way_name_nls);
	if (sql.select_way_next_seg)
		sqlite3_finalize(sql.select_way_next_seg);
	if (sql.select_way_prev_seg)
		sqlite3_finalize(sql.select_way_prev_seg);
	if (sql.select_way_name_search)
		sqlite3_finalize(sql.select_way_name_search);
	if (sql.select_way2)
		sqlite3_finalize(sql.select_way2);
	if (sql.select_place)
		sqlite3_finalize(sql.select_place);
	if (sql.select_place_near)
		sqlite3_finalize(sql.select_place_near);
}
osmdb=NULL;
osm_db_ok=FALSE;
memset(&sql, 0, sizeof(sql));

if (pcache.cache)
	g_hash_table_destroy(pcache.cache);
pcache.cache=NULL;
pcache.hit=0;
pcache.miss=0;

if (ncache.cache)
	g_hash_table_destroy(ncache.cache);
ncache.cache=NULL;
ncache.hit=0;
ncache.miss=0;

if (rcache.cache)
	g_hash_table_destroy(rcache.cache);
rcache.cache=NULL;
rcache.hit=0;
rcache.miss=0;

if (ircache.cache)
	g_hash_table_destroy(ircache.cache);
ircache.cache=NULL;
ircache.hit=0;
ircache.miss=0;

if (dbt)
	g_timer_destroy(dbt);
}

gboolean
osm_init(sqlite3 **db)
{
osm_db_ok=FALSE;
pcache.cache=g_hash_table_new(g_direct_hash, g_direct_equal);
ncache.cache=g_hash_table_new_full(g_direct_hash, g_direct_equal, NULL, g_free);
rcache.cache=g_hash_table_new_full(g_direct_hash, g_direct_equal, NULL, g_free);
ircache.cache=g_hash_table_new_full(g_direct_hash, g_direct_equal, NULL, g_free);

dbt=g_timer_new();

if (!db || !*db) {
	osmdb=NULL;
	return FALSE;
}

osmdb=*db;
memset(&sql, 0, sizeof(sql));
if (osm_db_create(osmdb)==FALSE) {
	g_printerr("Failed to create OSM tables or indexes: %s", sqlite3_errmsg(osmdb));
	return FALSE;
}

if (osm_db_prepare(osmdb)==FALSE) {
	g_printerr("Failed to prepare OSM SQL statements: %s", sqlite3_errmsg(osmdb));
	return FALSE;
}
osm_db_ok=TRUE;
return TRUE;
}

void
osm_db_enable_mainloop(sqlite3 *db, gboolean eml)
{
g_return_if_fail(db);
if (eml==FALSE)
	sqlite3_progress_handler(db, OSM_DB_PROGRESS_NUM, NULL, NULL);
else
	sqlite3_progress_handler(db, OSM_DB_PROGRESS_NUM, osm_progress, NULL);
}

/*****************************************************************************/

osm_way_node *
osm_way_node_new(guint id, gint lat, gint lon, gint flags)
{
osm_way_node *n=g_slice_new(osm_way_node);

n->id=id;
n->lat=lat;
n->lon=lon;
n->flags=flags;
return n;
}

void
osm_way_node_free(osm_way_node *n)
{
if (n)
	g_slice_free(osm_way_node, n);
}

/**
 * Free way nodes list 
 */
void
osm_way_nodes_free(osm_way *w)
{
GList *iter;

if (!w->nodes) 
	return;

for (iter=w->nodes; iter!=NULL; iter=iter->next)
	g_slice_free(osm_way_node, (osm_way_node*)iter->data);

g_list_free(w->nodes);
}

/**
 * Free a list of osm_way structures
 */
static void
osm_ways_free(GList *w)
{
GList *iter;

for (iter=w; iter!=NULL; iter=iter->next)
	osm_way_free(iter->data);
}

/**
 * Free a osm_way structure
 */
void
osm_way_free(osm_way *w)
{
if (!w)
	return;
osm_way_nodes_free(w);
if (w->name)
	g_free(w->name);
if (w->ref)
	g_free(w->ref);
if (w->int_ref)
	g_free(w->int_ref);
g_slice_free(osm_way, w);
}

/*****************************************************************************/

static void
osm_place_free(osm_place *p)
{
if (p->name)
	g_free(p->name);
g_slice_free(osm_place, p);
}

static gboolean
osm_place_remove(gpointer k, gpointer v, gpointer ud)
{
osm_place_free((osm_place *)v);
return TRUE;
}

static osm_place *
osm_place_new(void)
{
return g_slice_new0(osm_place);
}

static osm_place *
osm_place_cache_lookup(guint32 id)
{
osm_place *r;

r=g_hash_table_lookup(pcache.cache, GINT_TO_POINTER(id));
if (r) 
	pcache.hit++; 
else 
	pcache.miss++;

g_debug("OSM: PCache %d/%d", pcache.hit, pcache.miss);
return r;
}

static void
osm_place_cache_add(osm_place *p)
{
if (osm_place_cache_lookup(p->id)==NULL)
	g_hash_table_insert(pcache.cache, GINT_TO_POINTER(p->id), p);
}

static void
osm_place_cache_gc(void)
{
guint r;

r=g_hash_table_foreach_remove(pcache.cache, osm_place_remove, NULL);
g_debug("OSM: Cache cleared (%d)", r);
pcache.hit=0;
pcache.miss=0;
}

/**
 * Get place with given id and distance to current location
 */
gboolean
osm_place_get(guint32 id, gdouble lat, gdouble lon, osm_place **nr)
{
osm_place *n;

g_return_val_if_fail(sql.select_place, FALSE);
g_return_val_if_fail(*nr, FALSE);

n=osm_place_cache_lookup(id);
if (n) {
	n->dist=calculate_distance(lat, lon, n->lat, n->lon);
	*nr=n;
	return TRUE;
}
n=NULL;

/* XXX: better place for this */
if (g_hash_table_size(pcache.cache)>OSM_CACHE_MAX_ITEMS)
	osm_place_cache_gc();

sqlite3_clear_bindings(sql.select_place);
sqlite3_reset(sql.select_place);

if (SQLITE_OK != sqlite3_bind_int(sql.select_place, 1, id)) {
	g_warning("Failed to bind values for place");
	return FALSE;
}

if (SQLITE_ROW == sqlite3_step(sql.select_place)) {
	const gchar *place;
	gdouble plat, plon, dist;

	n=osm_place_new();
	place=sqlite3_column_text(sql.select_place, 0);
	n->name=g_strdup(place);
	plat=sqlite3_column_int(sql.select_place, 1);
	plon=sqlite3_column_int(sql.select_place, 2);
	n->type=sqlite3_column_int(sql.select_place, 3);
	n->isin_p=sqlite3_column_int(sql.select_place, 4);
/*	n->isin_c=sqlite3_column_int(sql.select_place, 5); */

	n->dist=calculate_distance(lat, lon, plat, plon);

	osm_place_cache_add(n);
	*nr=n;

	return TRUE;
}
return FALSE;
}

/*
 * Way helper 
 *
 */
static GList *
osm_find_nearest_way_nodes(gint lat, gint lon, guint range)
{
GList *ways=NULL;
osm_way *w;
guint wc=0;

g_return_val_if_fail(sql.select_way2, NULL);

sqlite3_reset(sql.select_way2);
sqlite3_clear_bindings(sql.select_way2);

if (SQLITE_OK != sqlite3_bind_int(sql.select_way2, 1, lat) ||
    SQLITE_OK != sqlite3_bind_int(sql.select_way2, 2, lon) ||
    SQLITE_OK != sqlite3_bind_int(sql.select_way2, 3, range) ||
    SQLITE_OK != sqlite3_bind_int(sql.select_way2, 4, WAY_ROAD_START) ||
    SQLITE_OK != sqlite3_bind_int(sql.select_way2, 5, WAY_ROAD_END)) {
	g_warning("Failed to bind values for way");
	return NULL;
}

QUERY_TIMER_START;
while (SQLITE_ROW == sqlite3_step(sql.select_way2)) {
	guint32 dist;
	gint lat, lon;

	wc++;
	w=g_slice_new0(osm_way);
	w->id=sqlite3_column_int(sql.select_way2, 0);
	w->type=sqlite3_column_int(sql.select_way2, 1);
	w->nodecnt=sqlite3_column_int(sql.select_way2, 2);
	w->flags=sqlite3_column_int(sql.select_way2, 3);
	dist=sqlite3_column_int(sql.select_way2, 4);
	w->dist=sqrt((gdouble)dist);
	w->f=sqlite3_column_int(sql.select_way2, 5);
	w->t=sqlite3_column_int(sql.select_way2, 6);

	lat=sqlite3_column_int(sql.select_way2, 7);
	lon=sqlite3_column_int(sql.select_way2, 8);

	w->speed=sqlite3_column_int(sql.select_way2, 9);

	w->node_f=osm_way_node_new(w->f, lat, lon, 0);

	ways=g_list_prepend(ways, w);
}

QUERY_TIMER_STOP("way2");
return ways;
}

/*****************************************************************************/
gboolean 
osm_way_distance(gint lat, gint lon, osm_way_node *f, osm_way_node *t, gdouble *d)
{
g_return_val_if_fail(f, FALSE);
g_return_val_if_fail(t, FALSE);

return distance_point_to_line((gdouble)lon, (gdouble)lat, (gdouble)f->lon, (gdouble)f->lat, (gdouble)t->lon, (gdouble)t->lat, d);
}

/**
 * Search for the nearest way (road)
 * - First search for ways with nearest node
 * - If only one is returned then we are done.
 * - If more than one, go trough the results:
 *   - Load nodes for the way, check prev/next nodes
 *   - Store result if closer than before
 * - Return closest way
 */

#define START_DIST (7500.0)

static GList *
osm_try_to_find_ways(gint lat, gint lon)
{
GList *w;
guint range=OSM_RANGE_START;

while ((w=osm_find_nearest_way_nodes(lat, lon, range))==NULL && range<=OSM_RANGE_STOP) {
	range+=OSM_RANGE_STEP;
}

return w;
}

static osm_way *
osm_iterate_ways(GList *w, gint lat, gint lon)
{
GList *iter;
osm_way *cw=NULL;
gdouble pdist=START_DIST, dist_n, dist_p;

for (iter=w; iter!=NULL; iter=iter->next) {
	gdouble penalty=0.0;
	osm_way_node *wnn;
	osm_way_node *wnp;
	osm_way *way=(osm_way*)iter->data;

	g_return_val_if_fail(way, NULL);

	if ((way->flags & W_NONAME) && (way->flags & W_NOREF))
		continue;

	switch (way->type) {
	case WAY_SERVICE:
	case WAY_FOOTWAY:
	case WAY_CYCLEWAY:
	case WAY_TRACK:
		/* Give a distance penalty for small roads, and even larger penalty if they are nameless */
		penalty=1000+((way->flags & W_NONAME) ? 500 : 0);
	break;
	}

	way->node_t=NULL;

	wnn=osm_way_get_next_node(way);
	if (!wnn)
		goto skip_to_wnp;

	if (osm_way_distance(lat, lon, way->node_f, wnn, &dist_n)==FALSE) {
		osm_way_node_free(wnn);
		dist_n=START_DIST;
	}
	dist_n+=penalty;
	if (dist_n<pdist) { /* Is it near ?*/
		pdist=dist_n;
		cw=way;
		way->distance=dist_n;
		way->node_t=wnn;
	} else {
		dist_n=START_DIST;
	}

#ifdef DEBUG
	g_debug("WayN: #%d (T:%d %d %d), %d nodes, near [%d-%d], D: %.2f (Valid: %s)",
		way->id, way->flags, way->type, way->speed, way->nodecnt, way->f, way->t, dist_n, dist_n==START_DIST ? "F" : "T");
#endif

	skip_to_wnp: ;

	wnp=osm_way_get_prev_node(way);
	if (!wnp)
		continue;
	if (osm_way_distance(lat, lon, way->node_f, wnp, &dist_p)==FALSE) {
		osm_way_node_free(wnp);
		dist_p=START_DIST;
	}
	dist_p+=penalty;
	if (dist_p<pdist) {
		pdist=dist_p;
		cw=way;
		way->distance=dist_n;
		if (way->node_t)
			osm_way_node_free(way->node_t);
		way->node_t=wnp;
	} else {
		dist_p=START_DIST;
	}

#ifdef DEBUG
	g_debug("WayP: #%d (T:%d %d %d), %d nodes, near [%d-%d], D: %.2f (Valid: %s)",
		way->id, way->flags, way->type, way->speed, way->nodecnt, way->f, way->t, dist_p, dist_p==START_DIST ? "F" : "T");
#endif

}
return cw;
}

osm_way *
osm_find_nearest_way(gint lat, gint lon)
{
GList *w;
osm_way *cw=NULL;

/* Check cache */
if (cways) {
	g_debug("Checking way cache");
	cw=osm_iterate_ways(cways, lat, lon);
	if (!cw) {
		g_debug("Not in cache, clearing it");
		osm_ways_free(cways);
		g_list_free(cways);
		cways=NULL;
	} else {
		g_debug("HIT, found (#%d) in way cache", cw->id);
	}
}
if (!cw) {
	w=osm_try_to_find_ways(lat, lon);
	if (!w)
		return NULL;
	cways=w;
	cw=osm_iterate_ways(w, lat, lon);
}

if (!cw)
	return NULL;

if (!cw->name && !(cw->flags & W_NONAME)) {
	osm_way_get_name(cw);
	osm_way_get_name_nls(cw);
}

switch (cw->type) {
	case WAY_MOTORWAY:
	case WAY_TRUNK:
	case WAY_PRIMARY:
	case WAY_SECONDARY:
	case WAY_TERTIARY:
		if (!(cw->flags & W_NOREF)) {
			/* We could skip int_ref if no ref, but unfortunately there are roads with only int_ref */
			osm_way_get_ref(cw);
			osm_way_get_int_ref(cw);
		}
	break;
	case WAY_SERVICE:
		if (!cw->name)
			cw->name=g_strdup("(Service road)");
	break;
	case WAY_TRACK:
		if (!cw->name)
			cw->name=g_strdup("(Track)");
	break;
	case WAY_FOOTWAY:
	case WAY_CYCLEWAY:
		if (!cw->name)
			cw->name=g_strdup("(Path)");
	break;
	default:
		if (!cw->name)
			cw->name=g_strdup("(Unknown)");
	break;
}

#ifdef DEBUG
g_debug("Found: (#%d): [%s]:[%s][%s]:[%d]", cw->id, cw->name, cw->ref, cw->int_ref, cw->speed);
g_debug("T: %d F: %d N#: %d D: %.2f", cw->type, cw->flags, cw->nodecnt, cw->dist);
g_debug("NF#: %d NT#: %d (DC: %.2f)", cw->f, cw->t, cw->distance);
#endif

return cw;
}

/* XXX: These two should be combined to save memory */
/**
 * Get previous node/segment of given way node
 *
 */
osm_way_node *
osm_way_get_prev_node(osm_way *w)
{
g_return_val_if_fail(sql.select_way_prev_seg, NULL);

sqlite3_reset(sql.select_way_prev_seg);
sqlite3_clear_bindings(sql.select_way_prev_seg);

if (SQLITE_OK != sqlite3_bind_int(sql.select_way_prev_seg, 1, w->id) ||
	SQLITE_OK != sqlite3_bind_int(sql.select_way_prev_seg, 2, w->f)  ) {
	g_warning("Failed to bind values for prev seg");
	return NULL;
}

if (SQLITE_ROW == sqlite3_step(sql.select_way_prev_seg)) {
	return osm_way_node_new(
		sqlite3_column_int(sql.select_way_prev_seg, 0),
		sqlite3_column_int(sql.select_way_prev_seg, 1),
		sqlite3_column_int(sql.select_way_prev_seg, 2),
		0);
}

return NULL;
}

/**
 * Get next node/segment of given way node
 *
 */
osm_way_node *
osm_way_get_next_node(osm_way *w)
{
g_return_val_if_fail(sql.select_way_next_seg, NULL);

sqlite3_reset(sql.select_way_next_seg);
sqlite3_clear_bindings(sql.select_way_next_seg);

if (SQLITE_OK != sqlite3_bind_int(sql.select_way_next_seg, 1, w->id) ||
	SQLITE_OK != sqlite3_bind_int(sql.select_way_next_seg, 2, w->f)  ) {
	g_warning("Failed to bind values for next seg");
	return NULL;
}

if (SQLITE_ROW == sqlite3_step(sql.select_way_next_seg)) {
	return osm_way_node_new(
		sqlite3_column_int(sql.select_way_next_seg, 0),
		sqlite3_column_int(sql.select_way_next_seg, 1),
		sqlite3_column_int(sql.select_way_next_seg, 2),
		0);
}

return NULL;
}

/**
 * Get list of nodes for given way
 *
 */
gboolean
osm_way_get_nodes(osm_way *w)
{
if (w->nodes!=NULL)
	return TRUE;

g_return_val_if_fail(sql.select_way_nodes, FALSE);

sqlite3_reset(sql.select_way_nodes);
sqlite3_clear_bindings(sql.select_way_nodes);

if (SQLITE_OK != sqlite3_bind_int(sql.select_way_nodes, 1, w->id)) {
	g_warning("Failed to bind values way nodes");
	return FALSE;
}

while (SQLITE_ROW == sqlite3_step(sql.select_way_nodes)) {
	osm_way_node *n;

	n=g_slice_new(osm_way_node);
	n->id=sqlite3_column_int(sql.select_way_nodes, 0);
	n->lat=sqlite3_column_int(sql.select_way_nodes, 1);
	n->lon=sqlite3_column_int(sql.select_way_nodes, 2);
	w->nodes=g_list_append(w->nodes, n);
}

return (w->nodes==NULL) ? FALSE : TRUE;
}

/**
 * Get way name
 *
 */
gboolean
osm_way_get_name(osm_way *w)
{
gchar *r;

g_return_val_if_fail(w, FALSE);
g_return_val_if_fail(sql.select_way_name, FALSE);

if (w->name)
	return TRUE;

r=g_hash_table_lookup(ncache.cache, GINT_TO_POINTER(w->id));
if (r) {
	ncache.hit++; 
	w->name=g_strdup(r);
} else {
	ncache.miss++;
}

g_debug("OSM: NCache: %d/%d", ncache.hit, ncache.miss);

sqlite3_reset(sql.select_way_name);
sqlite3_clear_bindings(sql.select_way_name);

if (SQLITE_OK != sqlite3_bind_int(sql.select_way_name, 1, w->id)) {
	g_warning("Failed to bind values for way name");
	return FALSE;
}

if (g_hash_table_size(ncache.cache)>OSM_CACHE_MAX_ITEMS)
	g_hash_table_foreach_remove(ncache.cache, gtk_true, NULL);

if (SQLITE_ROW == sqlite3_step(sql.select_way_name)) {
	const gchar *place;
	place=sqlite3_column_text(sql.select_way_name, 0);
	g_hash_table_insert(ncache.cache, GINT_TO_POINTER(w->id), g_strdup(place));
	w->name=g_strdup(place);
}
return FALSE;
}

gboolean
osm_way_get_name_nls(osm_way *w)
{
g_return_val_if_fail(sql.select_way_name_nls, FALSE);

sqlite3_reset(sql.select_way_name_nls);
sqlite3_clear_bindings(sql.select_way_name_nls);

if (SQLITE_OK != sqlite3_bind_int(sql.select_way_name_nls, 1, w->id)) {
	g_warning("Failed to bind values for way name nls");
	return FALSE;
}

while (SQLITE_ROW == sqlite3_step(sql.select_way_name_nls)) {
	const gchar *place;
	gchar *tmp;

	place=sqlite3_column_text(sql.select_way_name_nls, 0);
	tmp=g_strconcat(w->name ? w->name : "", w->name ? " / " : "", place, NULL);
	g_free(w->name);
	w->name=tmp;
}

return TRUE;
}

/**
 * Get Way ref
 *
 */
gboolean
osm_way_get_ref(osm_way *w)
{
gchar *r;

g_return_val_if_fail(sql.select_way_ref, FALSE);

r=g_hash_table_lookup(rcache.cache, GINT_TO_POINTER(w->id));
if (r) {
	rcache.hit++; 
	w->ref=g_strdup(r);
} else {
	rcache.miss++;
}

g_debug("OSM: RCache: %d/%d", rcache.hit, rcache.miss);

sqlite3_reset(sql.select_way_ref);
sqlite3_clear_bindings(sql.select_way_ref);

if (SQLITE_OK != sqlite3_bind_int(sql.select_way_ref, 1, w->id)) {
	g_warning("Failed to bind values for way ref");
	return FALSE;
}

if (g_hash_table_size(rcache.cache)>OSM_CACHE_MAX_ITEMS)
	g_hash_table_foreach_remove(rcache.cache, gtk_true, NULL);

if (SQLITE_ROW == sqlite3_step(sql.select_way_ref)) {
	const gchar *ref, *int_ref;

	ref=sqlite3_column_text(sql.select_way_ref, 0);
	int_ref=sqlite3_column_text(sql.select_way_ref, 1);
	g_hash_table_insert(rcache.cache, GINT_TO_POINTER(w->id), g_strdup(ref));
	/* Put in int_ref cache, so we can pick it up from there directly */
	if (int_ref)
		g_hash_table_insert(ircache.cache, GINT_TO_POINTER(w->id), g_strdup(int_ref));
	
	w->ref=g_strdup(ref);
	return TRUE;
}
return FALSE;
}

/**
 * Get way int_ref
 */
gboolean
osm_way_get_int_ref(osm_way *w)
{
gchar *r;

g_return_val_if_fail(sql.select_way_ref, FALSE);

r=g_hash_table_lookup(ircache.cache, GINT_TO_POINTER(w->id));
if (r) {
	ircache.hit++; 
	w->int_ref=g_strdup(r);
} else {
	ircache.miss++;
}

g_debug("OSM: IRCache: %d/%d", ircache.hit, ircache.miss);

sqlite3_reset(sql.select_way_ref);
sqlite3_clear_bindings(sql.select_way_ref);

if (SQLITE_OK != sqlite3_bind_int(sql.select_way_ref, 1, w->id)) {
	g_warning("Failed to bind values for way int ref");
	return FALSE;
}

if (g_hash_table_size(ircache.cache)>OSM_CACHE_MAX_ITEMS)
	g_hash_table_foreach_remove(ircache.cache, gtk_true, NULL);

if (SQLITE_ROW == sqlite3_step(sql.select_way_ref)) {
	const gchar *int_ref;

	int_ref=sqlite3_column_text(sql.select_way_ref, 1);
	g_hash_table_insert(ircache.cache, GINT_TO_POINTER(w->id), g_strdup(int_ref));
	w->int_ref=g_strdup(int_ref);
	return TRUE;
}
return FALSE;
}

/******************************************************************************/

/**
 * Check if we have moved away from the last known street
 */
gboolean
osm_check_location(osm_location *map_loc, gint lat, gint lon)
{
gdouble dist;
if (map_loc->street && osm_way_distance(lat, lon, map_loc->street->node_f, map_loc->street->node_t, &dist)==TRUE)
	return (dist>(gdouble)way_dist_range) ? FALSE : TRUE;
return FALSE;
}

void
osm_get_location_place_data(osm_location *map_loc)
{
if (!map_loc->valid)
	return;
map_loc->primary=NULL;
map_loc->secondary=NULL;
if (map_loc->street && map_loc->street->isin_p>0)
	osm_place_get(map_loc->street->isin_p, map_loc->lat, map_loc->lon, &map_loc->primary);
if (map_loc->street && map_loc->street->isin_c>0)
	osm_place_get(map_loc->street->isin_c, map_loc->lat, map_loc->lon, &map_loc->secondary);
}

/**
 * Try to figure out where the given lat,lon is. Fills in the given struct,
 * with street, secondary (suburb) and primary (city,town,village) location.
 * Will try to minimize the amount of database access by skipping queries
 * if we haven't moved or if we don't know where we are.
 *
 */
gboolean 
osm_get_location_data(gint lat, gint lon, gfloat heading, osm_location *map_loc)
{
gdouble dist;
gulong d;

/* Check 10 times, if location is not known then assume we are in here-be-dragons land */
if (map_loc->nfcnt>=10) {
	time_t t;

	t=time(NULL)-map_loc->last_valid; 
	if (t<30) {
		g_debug("Here be dragons, retrying in 30 seconds, %d %d", t, map_loc->last_valid);
		return FALSE;
	}
	g_debug("Trying again");
	map_loc->nfcnt=0;
	map_loc->last_valid=time(NULL);
	map_loc->valid=FALSE;
}

if (map_loc->valid==FALSE) {
	map_loc->lat=lat;
	map_loc->lon=lon;
	map_loc->valid=TRUE;
	d=way_dist_range*5;
} else {
	d=calculate_idistance(lat,lon,map_loc->lat,map_loc->lon);
}

/* Check if we are still near the same way as last time */
if (map_loc->street && osm_way_distance(lat, lon, map_loc->street->node_f, map_loc->street->node_t, &dist)==TRUE) {
	/* We are probably on the same way as last time, 
	   but check if the distance is over a limit or our heading has changed significantly.
	*/
	if ((dist>(gdouble)way_dist_range) || (fabs(angle_diff(heading, map_loc->heading))>15.0)) {
		/* We have moved a large amount, check way again */
		map_loc->street=osm_find_nearest_way(lat, lon);
		map_loc->changed=TRUE;
		g_debug("Distance %.2f > %.2f range or angle %f > 15.0, checking location", 
			dist, (gdouble)way_dist_range, fabs(heading-map_loc->heading));
	} else {
		/* We are still on the same way as last time */
		map_loc->changed=FALSE;
		g_debug("*** No change in location: %f %d", dist, way_dist_range);
	}
	map_loc->lat=lat;
	map_loc->lon=lon;
} else {
	/* We didn't know our location, so check it, but only if we have moved */
	if (d>way_dist_range) {
		g_debug("*** Must check location");
		map_loc->street=osm_find_nearest_way(lat, lon);
#if 0
		map_loc->lat=lat;
		map_loc->lon=lon;
#endif
		map_loc->changed=TRUE;
	} 

	if (!map_loc->street) {
		g_debug("*** Street not known");
		map_loc->nfcnt++;
		map_loc->changed=TRUE;
	} else {
		g_debug("*** Street known");
		map_loc->nfcnt=0;
		map_loc->last_valid=time(NULL);
		map_loc->changed=TRUE;
	}
}

if (map_loc->changed==TRUE)
	map_loc->heading=heading;

g_debug("NFC: %d", map_loc->nfcnt);
g_debug("D: %ld %ld", d, (gulong)way_dist_range);

return map_loc->street ? TRUE : FALSE;
}

GtkListStore *
osm_search_store_new(void)
{
return gtk_list_store_new(ITEM_NUM_COLUMNS, 
			G_TYPE_INT,	/* ID */
			G_TYPE_INT,	/*  */
			G_TYPE_DOUBLE,	/* Latitude */
			G_TYPE_DOUBLE,	/* Longitude */
			G_TYPE_DOUBLE,	/* Distance */
			G_TYPE_STRING,	/* Lat/Lon */
			G_TYPE_STRING,	/* Label */
			G_TYPE_STRING,	/* Desc. */
			G_TYPE_STRING,	/* Category */
			G_TYPE_STRING,	/* Dummy */
			G_TYPE_STRING);	/* Dummy */
}

/**
 * osm_place_search
 */
gboolean
osm_place_search(gdouble lat, gdouble lon, gchar *text, GtkListStore **store)
{
GtkTreeIter iter;
gchar *ltext=NULL;
guint rows=0;
gchar tmp1[16], tmp2[16];
gdouble range=6;

g_return_val_if_fail(sql.select_place_search, FALSE);
g_return_val_if_fail(text, FALSE);
g_return_val_if_fail(*store, FALSE);

gtk_list_store_clear(*store);
ltext=g_strdup_printf("%s%%", text);

if (SQLITE_OK != sqlite3_bind_double(sql.select_place_search, 1, lat) ||
	SQLITE_OK != sqlite3_bind_double(sql.select_place_search, 2, lon) ||
	SQLITE_OK != sqlite3_bind_text(sql.select_place_search,   3, ltext, -1, SQLITE_TRANSIENT)) {
		g_warning("Failed to bind values for sql.select_place_search");
		sqlite3_clear_bindings(sql.select_place_search);
		g_free(ltext);
		return FALSE;
}

if (ltext)
	g_free(ltext);

while (SQLITE_ROW == sqlite3_step(sql.select_place_search)) {
	gdouble rlat, rlon, dist;

	rlat=sqlite3_column_double(sql.select_place_search, 3);
	rlon=sqlite3_column_double(sql.select_place_search, 4);
	lat_format(_degformat, rlat, tmp1);
	lon_format(_degformat, rlon, tmp2);
	dist=calculate_distance(lat, lon, rlat, rlon) * UNITS_CONVERT[_units];

	gtk_list_store_append(*store, &iter);
	gtk_list_store_set(*store, &iter,
		ITEM_ID, sqlite3_column_int(sql.select_place_search, 0),
		ITEM_LAT, rlat,
		ITEM_LON, rlon,
		ITEM_DIST, dist,
		ITEM_LATLON, g_strdup_printf("%s, %s", tmp1, tmp2),
		ITEM_LABEL, sqlite3_column_text(sql.select_place_search, 1),
		-1);
	rows++;
}

g_debug("Found: %d places", rows);

sqlite3_reset(sql.select_place_search);
sqlite3_clear_bindings(sql.select_place_search);

return TRUE;
}

/**
 * osm_way_search
 *
 * Search for a street(way) starting with given 'text', next given lat/lon
 *
 */
gboolean
osm_way_search(gdouble lat, gdouble lon, gchar *text, GtkListStore **store)
{
GtkTreeIter iter;
gchar *ltext=NULL;
guint rows=0;
gchar tmp1[16], tmp2[16];
gdouble range=6;

g_return_val_if_fail(sql.select_way_name_search, FALSE);
g_return_val_if_fail(text, FALSE);
g_return_val_if_fail(*store, FALSE);

g_debug("Way Search: [%s] around %.6f %.6f", text, lat, lon);

gtk_list_store_clear(*store);
ltext=g_strdup_printf("%s%%", text);

if (SQLITE_OK != sqlite3_bind_double(sql.select_way_name_search, 1, lat) ||
    SQLITE_OK != sqlite3_bind_double(sql.select_way_name_search, 2, lon) ||
    SQLITE_OK != sqlite3_bind_int(sql.select_way_name_search, 	 3, WAY_ROAD_START) ||
    SQLITE_OK != sqlite3_bind_int(sql.select_way_name_search,    4, WAY_ROAD_END) ||
    SQLITE_OK != sqlite3_bind_double(sql.select_way_name_search, 6, range) ||
	SQLITE_OK != sqlite3_bind_text(sql.select_way_name_search,   5, ltext, -1, SQLITE_TRANSIENT)) {
		g_warning("Failed to bind values for sql.select_way_name_search");
		sqlite3_clear_bindings(sql.select_way_name_search);
		g_free(ltext);
		return FALSE;
}

if (ltext)
	g_free(ltext);

while (SQLITE_ROW == sqlite3_step(sql.select_way_name_search)) {
	gdouble rlat, rlon, dist;

	rlat=sqlite3_column_double(sql.select_way_name_search, 3);
	rlon=sqlite3_column_double(sql.select_way_name_search, 4);
	lat_format(_degformat, rlat, tmp1);
	lon_format(_degformat, rlon, tmp2);
	dist=calculate_distance(lat, lon, rlat, rlon) * UNITS_CONVERT[_units];

	gtk_list_store_append(*store, &iter);
	gtk_list_store_set(*store, &iter,
		ITEM_ID, sqlite3_column_int(sql.select_way_name_search, 0),
		ITEM_LAT, rlat,
		ITEM_LON, rlon,
		ITEM_DIST, dist,
		ITEM_LATLON, g_strdup_printf("%s, %s", tmp1, tmp2),
		ITEM_LABEL, sqlite3_column_text(sql.select_way_name_search, 1),
		-1);

	rows++;
}

g_debug("Found: %d items", rows);

sqlite3_reset(sql.select_way_name_search);
sqlite3_clear_bindings(sql.select_way_name_search);

return TRUE;
}
