/*
 * This file is part of mapper
 *
 * Copyright (C) 2007 Kaj-Michael Lang
 * Copyright (C) 2006-2007 John Costigan.
 *
 * POI and GPS-Info code originally written by Cezary Jackiewicz.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#include <config.h>
#include <glib.h>

static gboolean iap_initialized=FALSE;
static gboolean iap_connecting=FALSE;
static gboolean iap_connected=FALSE;
static gboolean iap_canceled=FALSE;

#ifdef WITH_OSSO
#include <libosso.h>

#ifdef WITH_OSSO_IC
#include <osso-ic.h>
#include <osso-ic-dbus.h>
#include <dbus/dbus.h>
#include <dbus/dbus-glib.h>
#include <dbus/dbus-glib-lowlevel.h>
#endif

#ifdef WITH_CONIC
#include <conic.h>
#endif

#include "iap.h"
#include "map-download.h"
#include "gps.h"
#include "config-gconf.h"

/*************************************************************/

#ifdef WITH_OSSO
void
iap_is_sdk()
{
gchar *sdk;

sdk=getenv("MAEMO_SDK");
if (!sdk)
	return;
g_debug("SDK: Forcing connection as up");
iap_connected=TRUE;
}
#endif

/**
 * Old OSSO IC (OS2006 and OS2007)
 */
#ifdef WITH_OSSO_IC

static void 
iap_callback(struct iap_event_t *event, void *arg)
{
iap_connecting=FALSE;
if (event->type==OSSO_IAP_CONNECTED && !iap_connected) {
	iap_connected=TRUE;
	config_update_proxy();
}
}

static DBusHandlerResult
get_connection_status_signal_cb(DBusConnection * connection, DBusMessage * message, void *user_data)
{
gchar *iap_name=NULL, *iap_nw_type=NULL, *iap_state=NULL;

/* check signal */
if (!dbus_message_is_signal(message, ICD_DBUS_INTERFACE, ICD_STATUS_CHANGED_SIG))
	return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;

if (!dbus_message_get_args(message, NULL,
		   DBUS_TYPE_STRING, &iap_name,
		   DBUS_TYPE_STRING, &iap_nw_type,
		   DBUS_TYPE_STRING, &iap_state,
		   DBUS_TYPE_INVALID))
		return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;

g_debug("IAP: %s", iap_state);
if (!strcmp(iap_state, "CONNECTED")) {
	if (!iap_connected) {
		iap_connected=TRUE;
		config_update_proxy();
	}
} else if (iap_connected) {
	iap_connected=FALSE;
}
return DBUS_HANDLER_RESULT_HANDLED;
}

void
iap_init(void)
{
DBusGConnection *gc;
DBusConnection *dc;
gchar *filter_string;

if (iap_initialized)
	return;

if ((gc=dbus_g_bus_get(DBUS_BUS_SYSTEM, &error))==NULL) {
	g_warning("Failed to open connection to D-Bus: %s.\n", error->message);
	error=NULL;
	return;
}

dc=dbus_g_connection_get_connection(gc);
filter_string=g_strdup_printf("interface=%s", ICD_DBUS_INTERFACE);
dbus_bus_add_match(dc, filter_string, NULL);
g_free(filter_string);

dbus_connection_add_filter(dc, get_connection_status_signal_cb, NULL, NULL);
osso_iap_cb(iap_callback);
iap_initialized=TRUE;
iap_is_sdk();
return;
}

gboolean
iap_connect(void) 
{
if (!iap_initialized)
	return FALSE;

if (!iap_connected && !iap_connecting) {
	iap_connecting=TRUE;
	osso_iap_connect(OSSO_IAP_ANY, OSSO_IAP_REQUESTED_CONNECT, NULL);
	return TRUE;
}
return FALSE;
}

gboolean
iap_is_connected(void)
{
return iap_connected;
}

gboolean
iap_is_canceled(void)
{
return iap_canceled;
}

#endif

/*************************************************************/
/*
 * libconic (OS2008)
 *
 */

#ifdef WITH_CONIC
static ConIcConnection *connection=NULL;

static void 
iap_connection_cb(ConIcConnection *connection, ConIcConnectionEvent *event, gpointer user_data)
{
ConIcConnectionStatus status;
ConIcConnectionError error;

g_return_if_fail(CON_IC_IS_CONNECTION_EVENT(event));

status=con_ic_connection_event_get_status(event);
switch (status) {
	case CON_IC_STATUS_CONNECTED:
		g_debug("IAP: Conn");
		iap_connected=TRUE;
		iap_connecting=FALSE;
		config_update_proxy();
	break;
        case CON_IC_STATUS_DISCONNECTING:
		g_debug("IAP: DisConNg");
		iap_connecting=FALSE;
	break;
        case CON_IC_STATUS_DISCONNECTED:
		g_debug("IAP: DisConn");
		iap_connected=FALSE;
		iap_connecting=FALSE;
		error=con_ic_connection_event_get_error(event);
		switch(error) {
			case CON_IC_CONNECTION_ERROR_USER_CANCELED:
				iap_canceled=TRUE;	
			break;
			default:
				g_debug("IAP: Connection error");
			break;
		}
	break;
	default:
		g_debug("IAP: Huh?");
	break;
}

}

void
iap_init(void)
{
connection=con_ic_connection_new();
g_signal_connect(G_OBJECT(connection), "connection-event", G_CALLBACK(iap_connection_cb), NULL);
g_object_set(G_OBJECT(connection), "automatic-connection-events", TRUE, NULL);
iap_initialized=TRUE;
iap_is_sdk();
}

gboolean
iap_connect(void)
{
if (!iap_initialized)
	return FALSE;
if (iap_connecting || iap_connected)
	return TRUE;
iap_connecting=TRUE;
g_debug("IAP: Connecting");
return con_ic_connection_connect(connection, CON_IC_CONNECT_FLAG_NONE);
}

gboolean
iap_is_connected(void)
{
return iap_connected;
}

gboolean
iap_is_canceled(void)
{
return iap_canceled;
}

#endif

#else
/* !WITH_OSSO */

void
iap_init(void)
{
iap_initialized=TRUE;
return;
}

gboolean
iap_connect(void)
{
return TRUE;
}

gboolean
iap_is_connected(void)
{
return TRUE;
}

gboolean
iap_is_canceled(void)
{
return FALSE;
}

#endif
