/*
 * poi-osm-fetcher.c - OpenStreetMap Point of Interest fetcher
 * Copyright (C) 2010 Collabora Ltd
 * @author Cosimo Alfarano <cosimo.alfarano@collabora.co.uk>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
#include "config.h"
#include "poi-osm-fetcher.h"

#include "poi.h"
#include "conf.h"
#include "mapbuddy-marshallers.h"

#include <string.h>

#include <glib.h>
#include <libsoup/soup.h>

#include <libxml/SAX2.h>

#include <spatialite/sqlite3.h>
#include <spatialite/gaiageo.h>
#include <spatialite.h>

/* FIXME: those should be defined in config.h */
#define SPATIALITE_EMPTY_DB "/usr/share/mapbuddy/spatialite_empty.db"
#define SPATIALITE_USER_CACHE_DIR "/home/user/MyDocs/.maps"
#define SPATIALITE_USER_DB SPATIALITE_USER_CACHE_DIR G_DIR_SEPARATOR_S "spatialite.db"
#define SPATIALITE_POI_TABLE "osm_point_of_interest"
#define SPATIALITE_POI_TAG_TABLE "osm_point_of_interest_tag"
#define OSM_XAPI_URL_BASE "http://www.openstreetmap.org/api/0.6/map"

static void map_buddy_poi_osm_fetcher_start_element (void *ctx,
    const xmlChar *name, const xmlChar **atts);
static void map_buddy_poi_osm_fetcher_end_element (void *ctx,
    const xmlChar *name);

struct PoiOsmGetUserCallbackData {
    SoupSession *session;
    xmlParserCtxtPtr xml_ctx;
    MapBuddyPoiOsmFetcher *fetcher;
    MapBuddyPoiOsmFetcherGetCallBack user_cb;
    gpointer user_data;

    gboolean is_error;
};


/* signal enum */
enum
{
    POI_ADDED,
    FEATURE_REMOVED,
    LAST_SIGNAL
};

static guint signals[LAST_SIGNAL] = { 0 };

/* taken from http://wiki.openstreetmap.org/wiki/Map_features */
static const gchar *all_osm_features[] = {
    "highway",
    "junction",
    "construction",
    "trafic_calming",
    "service",
    "barrier",
    "waterway",
    "railway",
    "aeroway",
    "aerialway",
    "power",
    "man_made",
    "leisure",
    "amenity",
    "office",
    "shop",
    "emergency",
    "tourism",
    "historic",
    "landuse",
    "military",
    "natural",
    "route",
    "boundary",
    "sport",
    "place",
    NULL};

static xmlSAXHandler sax_handle = {
    NULL, /* internalSubset */
    NULL, /* isStandalone */
    NULL, /* hasInternalSubset */
    NULL, /* hasExternalSubset */
    NULL, /* resolveEntity */
    NULL, /* getEntity */
    NULL, /* entityDecl */
    NULL, /* notationDecl */
    NULL, /* attributeDecl */
    NULL, /* elementDecl */
    NULL, /* unparsedEntityDecl */
    NULL, /* setDocumentLocator */
    NULL, /* startDocument */
    NULL, /* endDocument */
    map_buddy_poi_osm_fetcher_start_element, /* startElement */
    map_buddy_poi_osm_fetcher_end_element, /* endElement */
    NULL, /* reference */
    NULL, /* characters */
    NULL, /* ignorableWhitespace */
    NULL, /* processingInstruction */
    NULL, /* comment */
    NULL, /* xmlParserWarning */
    NULL, /* xmlParserError */
    NULL, /* xmlParserError */
    NULL, /* getParameterEntity */
    NULL, /* cdataBlock; */
    NULL, /* externalSubset; */
    1,    /* initialized */
    NULL, /* private, do now use */
    /* below only SAX2 members */
    NULL, /* startElementNs */
    NULL, /* endElementNs */
    NULL  /* xmlStructuredErrorFunc */
};

typedef struct
{
    /* the following three members are for SAX2 context */
    GQueue *path;
    GHashTable *attrs;
    GHashTable *tags;

    /* the SQLite db handle, to be open at init time and closed when destroyed */
    sqlite3 *handle;

    MapBuddyConf *conf;

} MapBuddyPoiOsmFetcherPrivate;

struct _MapBuddyPoiOsmFetcher
{
  GObject base;

  MapBuddyPoiOsmFetcherPrivate *priv;
};

struct _MapBuddyPoiOsmFetcherClass
{
  GObjectClass parent_class;
};

G_DEFINE_TYPE (MapBuddyPoiOsmFetcher, map_buddy_poi_osm_fetcher, G_TYPE_OBJECT);

#define POI_OSM_FETCHER_GET_PRIVATE(obj) \
  (G_TYPE_INSTANCE_GET_PRIVATE((obj), MAP_BUDDY_POI_OSM_FETCHER_TYPE, \
                               MapBuddyPoiOsmFetcherPrivate))


static GHashTable *
load_poi_tags_from_db (MapBuddyPoiOsmFetcher *self,
    gint osm_id)
{
  MapBuddyPoiOsmFetcherPrivate *priv = POI_OSM_FETCHER_GET_PRIVATE (self);
  GHashTable *tags = NULL; /*retval*/
  sqlite3_stmt *stmt = NULL;
  gint sqlret;

  /* we can operate disabling POIs if it's NULL, but better it gets logged as
   * a possible problem */
  g_assert (priv->handle != NULL);

  sqlret = sqlite3_prepare_v2 (priv->handle,
      "SELECT key, value "
      " FROM " SPATIALITE_POI_TAG_TABLE
      " WHERE osm_id=?",
      -1, &stmt, NULL);
  if (sqlret != SQLITE_OK)
    {
      g_debug ("%s: preparing SQL statement error: %s", G_STRFUNC,
          sqlite3_errmsg (priv->handle));
      goto out;
    }

  sqlite3_bind_int (stmt, 1, osm_id);

  tags = g_hash_table_new_full (g_str_hash,
      g_str_equal, g_free, g_free);

  for (sqlret = sqlite3_step (stmt);
      sqlret == SQLITE_ROW;
      sqlret = sqlite3_step (stmt))
    {
      gchar *key = (gchar *) sqlite3_column_text (stmt, 0);
      gchar *value = (gchar *) sqlite3_column_text (stmt, 1);

      g_hash_table_insert (tags, g_strdup (key), g_strdup (value));
    }

  /* probably a partial tag reading has been done, return it anyway */
  if (sqlret != SQLITE_DONE)
    g_debug ("%s: problem reading query result: %s", G_STRFUNC,
        sqlite3_errmsg (priv->handle));

out:
  if (stmt != NULL)
    /* ignore return value, there's nothing more we can do */
    sqlite3_finalize (stmt); /* notify the user about a fail? */

  return tags;
}


/** load_pois_for_feature:
 *
 * loads POIs for @feature from sqlite. It emits 'poi-added' signal on each
 * loaded POI
 *
 * Return %TRUE if POIs for @feature have been sucessfully loaded, %FALSE
 * otherwise
 */
static gboolean
load_pois_for_feature (MapBuddyPoiOsmFetcher *self,
    const gchar *feature)
{
  MapBuddyPoiOsmFetcherPrivate *priv = POI_OSM_FETCHER_GET_PRIVATE (self);
  sqlite3_stmt *stmt = NULL;
  gint sqlret;
  gboolean ret = FALSE;

  g_return_val_if_fail (feature != NULL && feature[0] != '\0', FALSE);
  /* we can operate disabling POIs if it's NULL, but better it gets logged as
   * a possible problem */
  g_return_val_if_fail (priv->handle != NULL, FALSE);

  g_printerr ("%s: getting pois for %s\n", G_STRFUNC, feature);

  sqlret = sqlite3_prepare_v2 (priv->handle,
      "SELECT "
        "osm_id, name, "
        "feature, feature_type, "
        "X(geometry), Y(geometry) "
      " FROM " SPATIALITE_POI_TABLE
      " WHERE feature=?",
      -1, &stmt, NULL);
  if (sqlret != SQLITE_OK)
    {
      g_debug ("preparing SQL statement error: %s", sqlite3_errmsg (priv->handle));
      goto out;
    }

  sqlite3_bind_text (stmt, 1, feature, -1, SQLITE_TRANSIENT);

  for (sqlret = sqlite3_step (stmt);
      sqlret == SQLITE_ROW;
      sqlret = sqlite3_step (stmt))
    {
      MapBuddyPoi *poi;
      GHashTable *tags = NULL;
      gchar *name = NULL;
      gchar *feat = NULL;
      gchar *feat_type = NULL;
      gint id;
      gdouble x,y;

      /* fetch POI data */
      id = sqlite3_column_int (stmt, 0);
      name = (gchar *) sqlite3_column_text (stmt, 1);
      feat = (gchar *) sqlite3_column_text (stmt, 2);
      feat_type = (gchar *) sqlite3_column_text (stmt, 3);
      x = sqlite3_column_double (stmt, 4);
      y = sqlite3_column_double (stmt, 5);

      /* instantiate a POI and emit a poi-added signal */
      poi = map_buddy_poi_new (id, name, x, y);
      map_buddy_poi_set_feature (poi, feat, feat_type);

      tags = load_poi_tags_from_db (self, id);
      map_buddy_poi_set_tags (poi, tags);

      g_signal_emit (self, signals[POI_ADDED], 0, poi);

      g_hash_table_unref (tags);
    }

  if (sqlret != SQLITE_DONE)
    {
      g_debug ("%s: problem reading query result: %s", G_STRFUNC,
          sqlite3_errmsg (priv->handle));
      goto out;
    }

  ret = TRUE;

out:
  if (stmt != NULL)
    /* ignore return value, there's nothing more we can do */
    sqlite3_finalize (stmt); /* notify the user about a fail? */

  return ret;
}


static void
conf_shown_features_changed_cb (MapBuddyConf *obj,
    GSList *added,
    GSList *removed,
    gpointer user_data)
{
  MapBuddyPoiOsmFetcher *self = MAP_BUDDY_POI_OSM_FETCHER (user_data);
  GSList *iter;

  g_return_if_fail (MAP_BUDDY_IS_POI_OSM_FETCHER (self));

  g_debug ("%s: Allowed feature list changed: %d added and %d removed",
      G_STRFUNC, g_slist_length (added), g_slist_length (removed));

  /* feature added: fetch its data from DB and emit poi-added signal */
  for (iter = added; iter != NULL; iter = iter->next)
    {
      gchar *feature = iter->data;

      load_pois_for_feature (self, feature);
    }

  /* feature removed: notify about its removal */
  for (iter = removed; iter != NULL; iter = iter->next)
    {
      gchar *feature = iter->data;

      g_signal_emit (self, signals[FEATURE_REMOVED], 0, feature);
    }

  return;
}


static void
map_buddy_poi_osm_fetcher_dispose (GObject *object)
{
  MapBuddyPoiOsmFetcher *self = MAP_BUDDY_POI_OSM_FETCHER (object);

  if (self->priv->path != NULL)
    {
      g_queue_foreach (self->priv->path, (GFunc) g_free, NULL);
      g_queue_free (self->priv->path);
      self->priv->path = NULL;
    }

  if (self->priv->tags != NULL)
    {
      g_object_unref (self->priv->tags);
      self->priv->tags = NULL;
    }

  if (self->priv->attrs != NULL)
    {
      g_object_unref (self->priv->attrs);
      self->priv->attrs = NULL;
    }

  if (self->priv->conf != NULL)
    {
      g_object_unref (self->priv->conf);
      self->priv->conf = NULL;
    }

  G_OBJECT_CLASS (map_buddy_poi_osm_fetcher_parent_class)->dispose (object);
}


static void
map_buddy_poi_osm_fetcher_finalize (GObject *object)
{
  MapBuddyPoiOsmFetcherPrivate *priv = POI_OSM_FETCHER_GET_PRIVATE (object);

  xmlCleanupParser ();

  if (priv->handle != NULL)
    {
      sqlite3_close (priv->handle);
      priv->handle = NULL;
    }

  G_OBJECT_CLASS (map_buddy_poi_osm_fetcher_parent_class)->finalize (object);
}

static void
map_buddy_poi_osm_fetcher_class_init (MapBuddyPoiOsmFetcherClass *cls)
{
  g_type_class_add_private (cls, sizeof (MapBuddyPoiOsmFetcherPrivate));

  GObjectClass *object_class = G_OBJECT_CLASS (cls);
  object_class->dispose = map_buddy_poi_osm_fetcher_dispose;
  object_class->finalize = map_buddy_poi_osm_fetcher_finalize;

  signals[POI_ADDED] = g_signal_new ("poi-added",
    G_TYPE_FROM_CLASS (object_class),
    G_SIGNAL_RUN_LAST,
    0, NULL, NULL,
    mapbuddy_marshal_VOID__OBJECT,
    G_TYPE_NONE, 1, G_TYPE_OBJECT);

  signals[FEATURE_REMOVED] = g_signal_new ("feature-removed",
    G_TYPE_FROM_CLASS (object_class),
    G_SIGNAL_RUN_LAST,
    0, NULL, NULL,
    g_cclosure_marshal_VOID__STRING,
    G_TYPE_NONE, 1, G_TYPE_STRING);
}

static void
map_buddy_poi_osm_fetcher_init (MapBuddyPoiOsmFetcher *self)
{
  MapBuddyPoiOsmFetcherPrivate *priv = POI_OSM_FETCHER_GET_PRIVATE (self);

  self->priv = priv;

  priv->path = g_queue_new ();
  priv->conf = map_buddy_conf_dup ();

  g_signal_connect (priv->conf, "shown-features-changed",
      G_CALLBACK (conf_shown_features_changed_cb), self);

  xmlInitParser ();
  spatialite_init (0);
}

static gboolean
_open_db (MapBuddyPoiOsmFetcher *self)
{
  gint ret;
  MapBuddyPoiOsmFetcherPrivate *priv = POI_OSM_FETCHER_GET_PRIVATE (self);

  if (!g_file_test (SPATIALITE_USER_DB, G_FILE_TEST_EXISTS))
    {
      /* the user's DB does not exist. Something went wrong at init time */
      g_debug ("opening user's DB %s: file does not exist.",
          SPATIALITE_USER_DB);

      return FALSE;
    }

  ret = sqlite3_open_v2 (SPATIALITE_USER_DB, &priv->handle, SQLITE_OPEN_READWRITE, NULL);
  if (ret != SQLITE_OK)
    {
      g_debug ("cannot open '%s': %s", SPATIALITE_USER_DB,
          sqlite3_errmsg (priv->handle));
      sqlite3_close (priv->handle);
      priv->handle = NULL;

      return FALSE;
    }

  return TRUE;
}

/** load_pois_from_db:
 *
 * loads POIs from sqlite and notifies emitting 'poi-added' signal on each
 * fetched POI
 */
static void
load_pois_from_db (MapBuddyPoiOsmFetcher *self)
{
  MapBuddyPoiOsmFetcherPrivate *priv = POI_OSM_FETCHER_GET_PRIVATE (self);
  const GSList *l;
  const GSList *feat_list;
  GError *e = NULL;

  feat_list = map_buddy_conf_get_shown_features (priv->conf, &e);
  if (e != NULL)
    {
      g_error_free (e);

      goto out;
    }

  for (l = feat_list; l != NULL; l = l->next)
    {
      gboolean ret;

      ret = load_pois_for_feature (self, (gchar *) l->data);
      if (!ret)
        g_debug ("%s: couldn't properly add POIs for %s feature",
            G_STRFUNC, (gchar *) l->data);
    }

out:
  /* feat_list belongs to the conf's cache and cannot be freed */
  return;
}


static gboolean
spatialite_import_tag (MapBuddyPoiOsmFetcher *self,
    gint osm_id,
    const gchar *key,
    const gchar *value)
{
  MapBuddyPoiOsmFetcherPrivate *priv = POI_OSM_FETCHER_GET_PRIVATE (self);
  sqlite3_stmt *stmt = NULL;
  int sqlret;
  gboolean ret = FALSE;

  sqlret = sqlite3_prepare_v2 (priv->handle,
      "INSERT INTO " SPATIALITE_POI_TAG_TABLE " (osm_id, key, value) "
      "VALUES (?, ?, ?)",
      -1, &stmt, NULL);
  if (sqlret != SQLITE_OK)
    {
      g_debug ("%s: preparing SQL statement for POI tag %s=%s (id %d): %s" ,
          G_STRFUNC, key, value, osm_id,
          sqlite3_errmsg (priv->handle));
      goto out;
    }

  sqlite3_bind_int (stmt, 1, osm_id);
  sqlite3_bind_text (stmt, 2, key, -1, SQLITE_TRANSIENT);
  sqlite3_bind_text (stmt, 3, value, -1, SQLITE_TRANSIENT);

  sqlret = sqlite3_step (stmt);
  if (sqlret != SQLITE_DONE)
    {
      g_debug ("%s: inserting SQL statement for tag '%s=%s' (id %d): %s",
          G_STRFUNC, key, value, osm_id,
          sqlite3_errmsg (priv->handle));
      goto out;
    }

  ret = TRUE;

out:
  if (stmt != NULL)
    sqlite3_finalize (stmt);

  return ret;
}

/** spatialite_import:
 *
 * add to sqlite a new MapBuddyPoi instance
 */
static gboolean
spatialite_import (MapBuddyPoiOsmFetcher *self,
    MapBuddyPoi *poi)
{
  MapBuddyPoiOsmFetcherPrivate *priv = POI_OSM_FETCHER_GET_PRIVATE (self);
  GHashTable *tags = NULL;
  GHashTableIter iter;
  gpointer key, value;
  sqlite3_stmt *stmt = NULL;
  unsigned char *blob = NULL;
  int blob_size;
  int e;
  gboolean ret = FALSE;

  g_assert (priv->handle != NULL);

  gaiaMakePoint (map_buddy_poi_get_lat (poi), map_buddy_poi_get_lon (poi),
      map_buddy_poi_get_srid (poi), &blob, &blob_size);
  if (blob == NULL)
    {
      g_debug ("%s: cannot create PostGIS Well-Known-Blob", G_STRFUNC);
      goto out;
    }

  e = sqlite3_prepare_v2 (priv->handle,
      "INSERT INTO " SPATIALITE_POI_TABLE
        " (osm_id, name, feature, feature_type, geometry) "
      "VALUES (?, ?, ?, ?, ?)",
      -1, &stmt, NULL);
  if (e != SQLITE_OK)
    {
      g_debug ("%s: preparing SQL statement for POI named '%s': %s" ,
          G_STRFUNC,
          map_buddy_poi_get_name (poi),
          sqlite3_errmsg (priv->handle));
      goto out;
    }

  sqlite3_bind_int (stmt, 1, map_buddy_poi_get_id (poi));
  sqlite3_bind_text (stmt, 2, map_buddy_poi_get_name (poi), -1,
      SQLITE_TRANSIENT);
  sqlite3_bind_text (stmt, 3, map_buddy_poi_get_feature (poi), -1,
      SQLITE_TRANSIENT);
  sqlite3_bind_text (stmt, 4, map_buddy_poi_get_feature_type (poi), -1,
      SQLITE_TRANSIENT);
  sqlite3_bind_blob (stmt, 5, blob, blob_size, SQLITE_TRANSIENT);

  e = sqlite3_step (stmt);
  if (e != SQLITE_DONE)
    {
      g_debug ("%s: cannot insert SQL row for element named '%s': %s",
          G_STRFUNC,
          map_buddy_poi_get_name (poi),
          sqlite3_errmsg (priv->handle));
      goto out;
    }

  tags = map_buddy_poi_get_tags (poi);

  g_hash_table_iter_init (&iter, tags);
  while (g_hash_table_iter_next (&iter, &key, &value))
    {
      spatialite_import_tag (self, map_buddy_poi_get_id (poi),
          (gchar *) key, (gchar *) value);
    }

  ret = TRUE;

out:
  free (blob);

  if (stmt != NULL)
    sqlite3_finalize (stmt);

  return ret;
}

static void
on_poi_osm_data_fetched (SoupSession *session,
    SoupMessage *msg,
    gpointer user_data)
{
  struct PoiOsmGetUserCallbackData *data = user_data;
  GError *error = NULL;
  int val;

  if (msg->status_code != SOUP_STATUS_OK)
    g_set_error (&error, DATA_FETCH_ERROR_DOMAIN, CANCELLED,
        "data fetch error: %s (%d)", msg->reason_phrase, msg->status_code);

  /* call the user's callback, if present */
  if (data->user_cb != NULL)
    data->user_cb (data->fetcher, (const GError **) &error, data->user_data);

  /* 'close' the xml chunk parser */
  val = xmlParseChunk (data->xml_ctx, NULL, 0, TRUE);
  if (val != 0)
    g_debug ("%s: closing xml chunk parser: xmlParserError=%d", G_STRFUNC, val);

  xmlFreeParserCtxt (data->xml_ctx);

  g_slice_free (struct PoiOsmGetUserCallbackData, data);
  if (error != NULL)
    g_error_free (error);
}

void
on_poi_osm_data_chunk (SoupMessage *msg,
    SoupBuffer  *chunk,
    gpointer user_data)
{
  struct PoiOsmGetUserCallbackData *data = user_data;

  if (data == NULL || data->is_error)
    return;

  if (!data->xml_ctx)
    {
      data->xml_ctx = xmlCreatePushParserCtxt (&sax_handle, data->fetcher,
          chunk->data, chunk->length, NULL);
      if (data->xml_ctx == NULL)
        {
          g_debug ("%s: not recoverable error occurred during "
              "SAX2 push-context creation", G_STRFUNC);

          soup_session_cancel_message (data->session, msg,
              SOUP_STATUS_CANCELLED);

          data->is_error = TRUE;
        }
    }
  else
    {
      int val;

      val = xmlParseChunk (data->xml_ctx, chunk->data, chunk->length, FALSE);
      if (val)
        {
          g_debug ("%s: xmlParseChunk error no. %d", G_STRFUNC, val);

          soup_session_cancel_message (data->session, msg,
              SOUP_STATUS_CANCELLED);

          data->is_error = TRUE;
        }
    }

  return;
}

void
map_buddy_poi_osm_fetcher_get (MapBuddyPoiOsmFetcher *self,
    gdouble left,
    gdouble buttom,
    gdouble right,
    gdouble top,
    MapBuddyPoiOsmFetcherGetCallBack cb,
    gpointer user_data)
{
  SoupSession *session;
  SoupMessage *message;
  struct PoiOsmGetUserCallbackData *data;
  gchar *url;

  session = soup_session_async_new ();

  /* bbox retrieves OSM elements for the specified area */
  url = g_strdup_printf ("%s?bbox=%f,%f,%f,%f",
      OSM_XAPI_URL_BASE, left, buttom, right, top);

  xmlSAX2InitDefaultSAXHandler (&sax_handle, 1);

  g_debug ("%s: GET %s", G_STRFUNC, url);
  message = soup_message_new ("GET", url);
  soup_message_body_set_accumulate (message->response_body, FALSE);

  data = g_slice_new0 (struct PoiOsmGetUserCallbackData);
  data->fetcher = self;
  data->session = session;
  data->user_cb = cb;
  data->user_data = user_data;

  g_signal_connect (message, "got-chunk", G_CALLBACK (on_poi_osm_data_chunk),
      data);

  soup_session_queue_message (session, message, on_poi_osm_data_fetched,
      data);

  g_free (url);
}

MapBuddyPoiOsmFetcher *
map_buddy_poi_osm_fetcher_new (void)
{
  return g_object_new (MAP_BUDDY_POI_OSM_FETCHER_TYPE, NULL);
}


/* Parse the XML attributes for a <node/> element.
 * SAX2 attributes are passed as a plain NULL-terminated array of strings.
 * we need to read the array and extract (k,v) couples */
static void
insert_node_attributes_to_hashtable (GHashTable *t,
    const xmlChar **attrs)
{
  int i;

  if (attrs == NULL)
    return ;

  for (i=0; attrs[i] != NULL; i+=2)
      g_hash_table_insert (t,
          g_strdup ((gchar*) attrs[i]),
          g_strdup ((gchar*) attrs[i+1]));
  return;
}

/* Parse the XML attributes for a <tag k=".." v="..."/> element. k and v are
 * the only attributes for this element.
 * SAX2 attributes are passed as a plain NULL-terminated array of strings.
 * we need to read the array and extract (k,v) couples */
static void
insert_tag_attributes_to_hashtable (GHashTable *t,
    const xmlChar **attrs)
{
  if (attrs == NULL)
    return ;

  /* ie {"k", "highway", "v", "traffic_signals"} 
   * "k" and "v" attrs can be found in any order, but they will be the only
   * two keys */
  if (xmlStrcmp ((xmlChar*) "k", attrs[0]) == 0 &&
      xmlStrcmp ((xmlChar*) "v", attrs[2]) == 0 &&
      attrs[1] != NULL && attrs[3] != NULL)
    {
      g_hash_table_insert (t,
          g_strdup ((gchar*) attrs[1]),
          g_strdup ((gchar*) attrs[3]));
    }
  else if (xmlStrcmp ((xmlChar*) "b", attrs[0]) == 0 &&
      xmlStrcmp ((xmlChar*) "b", attrs[2]) == 0 &&
      attrs[1] != NULL && attrs[3] != NULL)
    {
      g_hash_table_insert (t,
          g_strdup ((gchar*) attrs[3]),
          g_strdup ((gchar*) attrs[1]));
    }

  return;
}

/* extract from the SAX2 generated tags table the feature and feature_type.
 *
 * NOTE: it's a quite expensive method, it iterates all the possible OSM
 * feature and look for them in the table O(n)*O(hash_lookup_complexity) */
static void
map_buddy_poi_osm_fetcher_check_features (MapBuddyPoiOsmFetcher *self,
    gchar **feature, gchar **feature_type)
{
  int i;

  g_return_if_fail (feature != NULL);
  g_return_if_fail (feature_type != NULL);

  for (*feature_type = NULL, i = 0;
      all_osm_features[i] != NULL && *feature_type == NULL;
      i++)
    {
      /* (*feature_type) will be not-NULL only if the lookup will find
       * something */
      *feature_type = g_strdup (
          g_hash_table_lookup (self->priv->tags, all_osm_features[i]));
    }

  if (*feature_type != NULL)
    *feature = g_strdup (all_osm_features[i-1]);
  else
    *feature = NULL;
}

static void
map_buddy_poi_osm_fetcher_end_element (void *ctx,
    const xmlChar *name)
{
  MapBuddyPoiOsmFetcher *self = ctx;
  MapBuddyPoi *poi = NULL;
  gchar *feature = NULL;
  gchar *feature_type = NULL;

  if (xmlStrcmp (name, (xmlChar*) "node") != 0)
    {
      g_free (g_queue_pop_head (self->priv->path));
      /* do not free anything else if not a <node/> */
      return;
    }

  /* WE ARE AT THE END OF A <node/> ELEMENT */

  /* It's important to do all the sanity and data-presence checks here, so
   * that who's going to use the created (MapBuddyPoi*) knows to have all
   * the data in the right way */

  /* it has to have both hash tables non-NULL */
  if (self->priv->tags == NULL || self->priv->attrs == NULL)
    goto out;

  /* it has to have at least a proper name, id and coordinates */
  if ((g_hash_table_lookup (self->priv->tags, "name") == NULL) ||
      (g_hash_table_lookup (self->priv->attrs, "id") == NULL) ||
      (g_hash_table_lookup (self->priv->attrs, "lat") == NULL) ||
      (g_hash_table_lookup (self->priv->attrs, "lon") == NULL))
    goto out;

  map_buddy_poi_osm_fetcher_check_features (self, &feature,
      &feature_type);
  if (feature == NULL || feature_type == NULL)
    goto out;

  g_hash_table_insert (self->priv->tags,
      g_strdup ("mapbuddy:feature"), feature);
  g_hash_table_insert (self->priv->tags,
      g_strdup ("mapbuddy:feature_type"), feature_type);

  poi = map_buddy_poi_new (
      atoi (g_hash_table_lookup (self->priv->attrs, "id")),
      g_hash_table_lookup (self->priv->tags, "name"),
      g_ascii_strtod (g_hash_table_lookup (self->priv->attrs, "lat"), NULL),
      g_ascii_strtod (g_hash_table_lookup (self->priv->attrs, "lon"), NULL));

  map_buddy_poi_set_feature (poi, feature, feature_type);
  map_buddy_poi_set_tags (poi, self->priv->tags);

  if (spatialite_import (self, poi))
    {
      /* notify that we have a POI object ready to use */
      g_debug ("%s: emitting signal for %s", G_STRFUNC,
          (gchar*) g_hash_table_lookup (self->priv->tags, "name"));
      g_signal_emit (self, signals[POI_ADDED], 0, poi);
    }

out:
  if (poi != NULL)
    g_object_unref (poi);

  g_free (g_queue_pop_head (self->priv->path));

  if (self->priv->attrs != NULL)
    g_hash_table_destroy (self->priv->attrs);
  self->priv->attrs = NULL;

  if (self->priv->tags != NULL)
    g_hash_table_destroy (self->priv->tags);
  self->priv->tags = NULL;

  /* feature and feature_type are owned by priv->tags */
}


static void
map_buddy_poi_osm_fetcher_start_element (void *ctx,
    const xmlChar *name,
    const xmlChar **atts)
{
  MapBuddyPoiOsmFetcher *self = ctx;

  if (xmlStrcmp (name, (xmlChar*) "node") == 0)
    {
      /* attrs and tags are set to NULL at end_element CB.
       * it they are not NULL here we are probably considering the wrong
       * node in the SAX tree like <node><node></node></node> which does not
       * make sense for us. Log it! */
      g_return_if_fail (self->priv->attrs == NULL);
      g_return_if_fail (self->priv->tags == NULL);

      /* similar check here, node are child of <osm/> element */
      if (g_strcmp0 (self->priv->path->head->data, "osm") != 0)
        return;

      self->priv->attrs = g_hash_table_new_full (g_str_hash,
          g_str_equal, g_free, g_free);
      self->priv->tags = g_hash_table_new_full (g_str_hash,
          g_str_equal, g_free, g_free);

      insert_node_attributes_to_hashtable (self->priv->attrs, atts);
    }
  else if (xmlStrcmp (name, (xmlChar*) "tag") == 0)
    {
      /* consider a <tag> element only if child of a <node> one */
      if (g_strcmp0 (self->priv->path->head->data, "node") == 0)
        insert_tag_attributes_to_hashtable (self->priv->tags, atts);
    }

  /* finally update the elements' path */
  g_queue_push_head (self->priv->path, g_strdup ((gchar*) name));

  return;
}

void
map_buddy_poi_osm_fetcher_start (MapBuddyPoiOsmFetcher *self)
{
  MapBuddyPoiOsmFetcherPrivate *priv = POI_OSM_FETCHER_GET_PRIVATE (self);

  g_return_if_fail (MAP_BUDDY_IS_POI_OSM_FETCHER (self));
  g_return_if_fail (priv->handle == NULL);

  /* initialize data if it's the first time execution */
  if (!g_file_test (SPATIALITE_USER_CACHE_DIR, G_FILE_TEST_EXISTS))
    {
      GFile *file;
      GError *error = NULL;

      file = g_file_new_for_path (SPATIALITE_USER_CACHE_DIR);

      g_file_make_directory (file, NULL, &error);
      g_object_unref (file);

      if (error != NULL)
        {
          g_debug ("cannot create dir %s: %s",
              SPATIALITE_USER_CACHE_DIR, error->message);
          g_error_free (error);

          /* FIXME: something goes wrong during the cache dir creation, fall back
           * into no spatialite/no POI mode. The PoiOsmFetcher won't start
           * fetching and noone will receive any signal */
          return;
        }
    }

  /* Copy Point Of Interest DB if not present */
  if (!g_file_test (SPATIALITE_USER_DB, G_FILE_TEST_EXISTS))
    {
      gchar *content = NULL;
      gsize length;
      GError *error = NULL;

      g_file_get_contents (SPATIALITE_EMPTY_DB, &content, &length, &error);
      if (error != NULL)
        {
          g_debug ("copying content from %s: %s", SPATIALITE_EMPTY_DB,
              error->message);
          g_error_free (error);
        }
      else
        {
          g_file_set_contents (SPATIALITE_USER_DB, content, length, &error);
          if (error != NULL)
            {
              g_debug ("copying content to %s: %s", SPATIALITE_USER_DB,
                  error->message);
              g_error_free (error);
            }
        }

      g_free (content);
    }

  /* open DB we just initialized */
  if (!_open_db (self))
    return;

  g_assert (priv->handle != NULL);

  load_pois_from_db (self);
}
