/*
 * Copyright (C) 2010 Pierre-Luc Beaudoin <pierre-luc@pierlux.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
#include "config.h"

#include "search.h"
#include "map-buddy-window.h"
#include "merchant.h"
#include "merchant-marker.h"

#include <locale.h>
#include <glib.h>
#include <math.h>
#include <glib/gi18n.h>
#include <rest/rest-proxy.h>
#include <rest/rest-proxy-call.h>
#include <rest/rest-xml-parser.h>

enum {
  COL_NAME,
  COL_LAT,
  COL_LON,
  COL_COUNT
};

static void
place_result_cb (RestProxyCall *call,
                 GError *error,
                 GObject *weak_object,
                 MapBuddyWindow *self)
{
  const gchar *answer;
  gint len;
  gint count = 0;
  RestXmlParser *parser;
  RestXmlNode *root, *n;
  GtkListStore *store;

  map_buddy_window_loading_pop (self);

  if (error)
    {
      g_error ("Cannot make call: %s", error->message);
      g_error_free (error);
      return;
    }

  answer = rest_proxy_call_get_payload (call);
  len = rest_proxy_call_get_payload_length (call);
  parser = rest_xml_parser_new ();

  root = rest_xml_parser_parse_from_data (parser, answer, len);

  /* Extract the result count */
  n = rest_xml_node_find (root, "totalResultsCount");
  if (n)
    count = g_strtod (n->content, NULL);

  if (count == 0)
    {
      GtkWidget *dialog;

      dialog = hildon_note_new_information (GTK_WINDOW (self), _("The search returned no results."));
      gtk_dialog_run (GTK_DIALOG (dialog));

      gtk_widget_destroy (GTK_WIDGET (dialog));
      rest_xml_node_unref (root);
      g_object_unref (parser);
      return;
    }

  n = rest_xml_node_find (root, "geoname");
  store = gtk_list_store_new (COL_COUNT,
                              G_TYPE_STRING,       /* Name */
                              G_TYPE_DOUBLE,       /* Latitude */
                              G_TYPE_DOUBLE);      /* Longitude */


  while (n)
    {
      RestXmlNode *name, *country, *lon, *lat;
      GtkTreeIter iter;
      gchar *display_name;
      gdouble flon, flat;

      name = rest_xml_node_find (n, "name");
      if (!name)
        {
          n = n->next;
          continue;
        }

      country = rest_xml_node_find (n, "countryName");
      if (!country)
        {
          n = n->next;
          continue;
        }

      lon = rest_xml_node_find (n, "lng");
      if (!lon)
        {
          n = n->next;
          continue;
        }

      lat = rest_xml_node_find (n, "lat");
      if (!lat)
        {
          n = n->next;
          continue;
        }

      display_name = g_strdup_printf ("%s, %s", name->content, country->content);

      flon = g_strtod (lon->content, NULL);
      flat = g_strtod (lat->content, NULL);

      /* Create the row item */
      gtk_list_store_append (store, &iter);
      gtk_list_store_set (store, &iter,
                          COL_NAME, display_name,
                          COL_LAT, flat,
                          COL_LON, flon,
                          -1);

      g_free (display_name);

      n = n->next;
    }

  rest_xml_node_unref (root);

  /* Create a picker */
  GtkWidget *selector;
  GtkWidget *dialog;
  GtkCellRenderer *renderer = NULL;
  gint result;
  GtkTreeIter iter;

  selector = hildon_touch_selector_new ();

  renderer = gtk_cell_renderer_text_new ();
  hildon_touch_selector_append_column (HILDON_TOUCH_SELECTOR (selector),
                                       GTK_TREE_MODEL (store),
                                       renderer,
                                       "text", COL_NAME,
                                       NULL);

  dialog = hildon_picker_dialog_new (GTK_WINDOW (self));
  gtk_window_set_title (GTK_WINDOW (dialog), _("Select the correct place"));
  hildon_picker_dialog_set_selector (HILDON_PICKER_DIALOG (dialog),
                                     HILDON_TOUCH_SELECTOR (selector));

  result = gtk_dialog_run (GTK_DIALOG (dialog));
  gtk_widget_hide (dialog);
  if (result != GTK_RESPONSE_OK)
    {
      g_object_unref (store);
      return;
    }

  if (hildon_touch_selector_get_selected (HILDON_TOUCH_SELECTOR (selector),
                                          0, &iter))
    {
      gdouble lat, lon;
      gtk_tree_model_get (GTK_TREE_MODEL (store),
                          &iter,
                          COL_LAT, &lat,
                          COL_LON, &lon,
                          -1);
      champlain_view_set_zoom_level (self->view, 12);
      champlain_view_center_on (self->view, lat, lon);
    }

  g_object_unref (self->call);
  self->call = NULL;

  /* g_object_unref (selector); commented because ofridious critical errors */
  g_object_unref (store);
}

static void
show_details (MapBuddyMerchantMarker *marker,
              MapBuddyWindow *self)
{
  GList *l;

  l = self->merchants;

  while (l != NULL)
    {
      MapBuddyMerchant *merchant;

      merchant = (MapBuddyMerchant*) (l->data);
      if (MAP_BUDDY_MERCHANT_MARKER (merchant->marker) == MAP_BUDDY_MERCHANT_MARKER (marker))
        {
          map_buddy_window_show_details (self, merchant);
          return;
        }

      l = g_list_next (l);
    }
}

static void
merchant_result_cb (RestProxyCall *call,
                    GError *error,
                    GObject *weak_object,
                    MapBuddyWindow *self)
{
  const gchar *answer;
  gint len;
  gint count = 0;
  RestXmlParser *parser;
  RestXmlNode *root, *n;
  gboolean us_results = FALSE;
  gboolean ca_results = FALSE;

  map_buddy_window_loading_pop (self);

  map_buddy_window_clear_merchants (self);

  if (error)
    {
      g_error ("Cannot make call: %s", error->message);
      g_error_free (error);
      return;
    }

  answer = rest_proxy_call_get_payload (call);
  len = rest_proxy_call_get_payload_length (call);
  parser = rest_xml_parser_new ();

  root = rest_xml_parser_parse_from_data (parser, answer, len);

  /* Extract the result count */
  n = rest_xml_node_find (root, "pagination");
  n = rest_xml_node_find (n, "total_entries");
  if (n)
    count = g_strtod (n->content, NULL);

  n = rest_xml_node_find (root, "merchants");
  n = rest_xml_node_find (root, "merchant");

  while (n)
    {
      RestXmlNode *name, *loc, *lon, *lat, *desc,
                  *address, *city, *province, *postal_code,
                  *phone, *url, *country, *country_code;
      gdouble flon, flat;
      MapBuddyMerchant *merchant;

      name = rest_xml_node_find (n, "name");
      if (!name)
        {
          n = n->next;
          count--;
          continue;
        }

      loc = rest_xml_node_find (n, "location");

      lon = rest_xml_node_find (loc, "longitude");
      if (!lon || lon->content == NULL)
        {
          n = n->next;
          count--;
          continue;
        }

      lat = rest_xml_node_find (loc, "latitude");
      if (!lat || lat->content == NULL)
        {
          n = n->next;
          count--;
          continue;
        }

      desc = rest_xml_node_find (n, "description");
      phone = rest_xml_node_find (n, "phone");
      url = rest_xml_node_find (n, "url");

      address = rest_xml_node_find (loc, "street_address");
      city = rest_xml_node_find (loc, "city");
      city = rest_xml_node_find (city, "name");
      province = rest_xml_node_find (loc, "regions");
      province = rest_xml_node_find (province, "province");
      if (!province)
        {
          province = rest_xml_node_find (loc, "regions");
          province = rest_xml_node_find (province, "state");
        }
      postal_code = rest_xml_node_find (loc, "postal_code");

      country_code = rest_xml_node_find (loc, "country");
      country_code = rest_xml_node_find (country_code, "code");
      if (g_strcmp0 (country_code->content, "CA") == 0)
        ca_results = TRUE;
      else if (g_strcmp0 (country_code->content, "US") == 0)
        us_results = TRUE;

      country = rest_xml_node_find (loc, "country");
      country = rest_xml_node_find (country, "name");

      merchant = map_buddy_merchant_new ();

      flon = g_strtod (lon->content, NULL);
      flat = g_strtod (lat->content, NULL);

      merchant->name = g_markup_escape_text (name->content, -1);
      merchant->lat = flat;
      merchant->lon = flon;
      merchant->description = desc && desc->content ? g_strdup (desc->content) : NULL;
      merchant->phone = phone && phone->content ? g_strdup (phone->content) : NULL;
      merchant->url = url && url->content ? g_strdup (url->content) : NULL;
      merchant->address = address && address->content ? g_strdup (address->content) : NULL;
      merchant->city = city && city->content ? g_strdup (city->content) : NULL;
      merchant->province = province && province->content ? g_strdup (province->content) : NULL;
      merchant->postal_code = postal_code && postal_code->content ? g_strdup (postal_code->content) : NULL;
      merchant->country = country && country->content ? g_strdup (country->content) : NULL;
      merchant->country_code = country_code && country_code->content ? g_strdup (country_code->content) : NULL;

      self->merchants = g_list_append (self->merchants, merchant);

      merchant->marker = CHAMPLAIN_MARKER (map_buddy_merchant_marker_new ());
      champlain_marker_set_text (merchant->marker, merchant->name);
      champlain_base_marker_set_position (CHAMPLAIN_BASE_MARKER (merchant->marker), flat, flon);
      champlain_layer_add_marker (self->merchant_layer, CHAMPLAIN_BASE_MARKER (merchant->marker));

      g_signal_connect (G_OBJECT (merchant->marker),
                        "show-details",
                        G_CALLBACK (show_details),
                        self);

      n = n->next;
    }

  if (count == 0)
    {
      GtkWidget *dialog;

      dialog = hildon_note_new_information (GTK_WINDOW (self), _("The search returned no results."));
      gtk_dialog_run (GTK_DIALOG (dialog));

      gtk_widget_destroy (GTK_WIDGET (dialog));
      rest_xml_node_unref (root);
      g_object_unref (parser);
      return;
    }

#if CHAMPLAIN_CHECK_VERSION (0, 4, 3)
  GString *license;

  license = g_string_new (_("Powered by Praized Media\n"));
  if (ca_results)
    g_string_append (license, _("Canadian business listings distributed by YellowPages.ca\u2122"));

  if (us_results) {
    if (ca_results)
      g_string_append (license, "\n");
    g_string_append (license, _("US data provided by Localeze"));
  }

  champlain_view_set_license_text (CHAMPLAIN_VIEW (self->view), license->str);
  g_string_free (license, TRUE);
#endif

  g_object_unref (self->call);
  self->call = NULL;

  rest_xml_node_unref (root);
}


void
search_places (MapBuddyWindow *self,
               const gchar *text)
{
  gchar *locale;
  gchar lang[2];
  GError *error = NULL;

  locale = setlocale (LC_MESSAGES, NULL);
  g_utf8_strncpy (lang, locale, 2);

  if (self->proxy == NULL)
    self->proxy = rest_proxy_new ("http://ws.geonames.org/", FALSE);

  /* Cancel previous call */
  if (self->call)
    g_object_unref (self->call);
  self->call = rest_proxy_new_call (self->proxy);

  rest_proxy_set_user_agent (self->proxy, "Map Buddy/"VERSION);

  rest_proxy_call_set_function (self->call, "search");
  rest_proxy_call_set_method (self->call, "GET");
  rest_proxy_call_add_params (self->call,
      "q", text,
      "maxRows", "15",
      "lang", lang,
      "isNameRequired", "1",
      "featureClass", "A",
      "featureClass", "P",
      NULL);

  if (!rest_proxy_call_async (self->call,
        (RestProxyCallAsyncCallback) place_result_cb,
        G_OBJECT (self->proxy),
        self,
        &error))
    {
      g_error ("Cannot make call: %s", error->message);
      g_error_free (error);
    }

  map_buddy_window_loading_push (self);
}

void
search_merchants (MapBuddyWindow *self,
                  const gchar *text)
{
  GError *error = NULL;
  gchar *slat, *slon, *sradius, *count;
  gdouble lat, lon, radius;
  gint zoom;

  if (self->proxy == NULL)
    self->proxy = rest_proxy_new ("http://api.praized.com/%s/", TRUE);
  rest_proxy_bind (self->proxy, "map-buddy-for-maemo-5"); /* community slug */

  /* Cancel previous call */
  if (self->call)
    g_object_unref (self->call);
  self->call = rest_proxy_new_call (self->proxy);

  rest_proxy_set_user_agent (self->proxy, "Map Buddy/"VERSION);

  g_object_get (self->view,
                "latitude", &lat,
                "longitude", &lon,
                "zoom-level", &zoom,
                NULL);
  if (zoom < 13)
    zoom = 13;

#if CHAMPLAIN_CHECK_VERSION (0, 4, 3)
  ChamplainMapSource *source;

  g_object_get (self->view, "map-source", &source, NULL);
  radius = champlain_map_source_get_meters_per_pixel (source, zoom, lat, lon) * 0.4; /* (800px / 2) / 1000 m/km */
  g_object_unref (source);
#else
  radius = 2;
#endif

  slat = g_strdup_printf ("%f", lat);
  slon = g_strdup_printf ("%f", lon);
  sradius = g_strdup_printf ("%f", radius);
  count = g_strdup_printf ("%d", (int) floor (radius * 10.0));

  rest_proxy_call_set_function (self->call, "merchants.xml");
  rest_proxy_call_set_method (self->call, "GET");
  rest_proxy_call_add_params (self->call,
      "api_key", "90f44d3b365b40aa9203930f698150ea",
      "q", text,
      "lat", slat,
      "long", slon,
      "radius", sradius,
      "per_page", count, /* limit search results */
      NULL);

  if (!rest_proxy_call_async (self->call,
        (RestProxyCallAsyncCallback) merchant_result_cb,
        G_OBJECT (self->proxy),
        self,
        &error))
    {
      g_error ("Cannot make call: %s", error->message);
      g_error_free (error);
    }

  map_buddy_window_loading_push (self);

  g_free (slat);
  g_free (slon);
  g_free (sradius);
  g_free (count);
}
