/*
 * Copyright (C) 2009 Pierre-Luc Beaudoin <pierre-luc@pierlux.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
#include "config.h"

#include "me-marker.h"

#include <clutter/clutter.h>
#include <glib.h>
#include <glib-object.h>
#include <cairo.h>
#include <math.h>
#include <string.h>


struct _MapBuddyMeMarkerPrivate
{
  ClutterActor *point;
  ClutterActor *area;

  gint value;
};

G_DEFINE_TYPE (MapBuddyMeMarker, map_buddy_me_marker, CHAMPLAIN_TYPE_MARKER);

#define MAP_BUDDY_ME_MARKER_GET_PRIVATE(obj)    (G_TYPE_INSTANCE_GET_PRIVATE((obj), MAP_BUDDY_TYPE_ME_MARKER, MapBuddyMeMarkerPrivate))

static void draw_marker (ChamplainMarker *me_marker);

static void
map_buddy_me_marker_finalize (GObject *object)
{
  G_OBJECT_CLASS (map_buddy_me_marker_parent_class)->finalize (object);
}

static void
map_buddy_me_marker_class_init (MapBuddyMeMarkerClass *map_buddyClass)
{
  g_type_class_add_private (map_buddyClass, sizeof (MapBuddyMeMarkerPrivate));

  GObjectClass *object_class = G_OBJECT_CLASS (map_buddyClass);
  object_class->finalize = map_buddy_me_marker_finalize;

  ChamplainMarkerClass *markerClass = CHAMPLAIN_MARKER_CLASS (map_buddyClass);
  markerClass->draw_marker = draw_marker;
}

#define PADDING (RADIUS / 2)

static void
draw_box (cairo_t *cr,
   gint radius)
{
      cairo_arc (cr, radius, radius, radius -1, 0, 2 * M_PI);
      cairo_close_path (cr);
}

static void
draw_area (MapBuddyMeMarker *me_marker,
    gint radius)
{
  MapBuddyMeMarkerPrivate *priv = me_marker->priv;
  ClutterActor *bg = NULL;
  ClutterColor color = {0x2e, 0x54, 0xdf, 0xff};
  guint line_width = 1;
  cairo_t *cr;

  if (priv->area != NULL)
    {
      clutter_container_remove_actor (CLUTTER_CONTAINER (me_marker),
          priv->area);
      g_object_unref (priv->area);
      priv->area = NULL;
    }

  if (radius == 0)
    return;

  bg = clutter_cairo_texture_new (radius * 2, radius * 2);
  cr = clutter_cairo_texture_create (CLUTTER_CAIRO_TEXTURE (bg));

  draw_box (cr, radius);

  cairo_set_source_rgba (cr,
      color.red / 255.0,
      color.green / 255.0,
      color.blue / 255.0,
      0.35 * color.alpha / 255.0);
  cairo_fill_preserve (cr);

  cairo_set_line_width (cr, line_width);
  cairo_set_source_rgba (cr,
      color.red / 255.0,
      color.green / 255.0,
      color.blue / 255.0,
      0.5 * color.alpha / 255.0);
  cairo_stroke (cr);
  cairo_destroy (cr);

  clutter_container_add_actor (CLUTTER_CONTAINER (me_marker), bg);

  priv->area = g_object_ref (bg);

  if (priv->point)
    clutter_actor_raise (CLUTTER_ACTOR (priv->point), CLUTTER_ACTOR (priv->area));
}

static void
draw_marker (ChamplainMarker *me_marker)
{
  MapBuddyMeMarkerPrivate *priv = MAP_BUDDY_ME_MARKER (me_marker)->priv;
  gfloat radius = priv->value;

  draw_area (MAP_BUDDY_ME_MARKER (me_marker), radius);
  clutter_actor_set_position (CLUTTER_ACTOR (priv->point), radius - 6, radius - 6);

  clutter_actor_set_anchor_point (CLUTTER_ACTOR (me_marker), radius, radius);
}

/*@radius: in px */
void
map_buddy_me_marker_set_radius (MapBuddyMeMarker *me_marker,
                       gint value)
{
  MapBuddyMeMarkerPrivate *priv = MAP_BUDDY_ME_MARKER_GET_PRIVATE (me_marker);

  priv->value = value;
#if CHAMPLAIN_CHECK_VERSION (0, 4, 3)
  champlain_marker_queue_redraw (CHAMPLAIN_MARKER (me_marker));
#endif
}

static void
map_buddy_me_marker_init (MapBuddyMeMarker *me_marker)
{
  MapBuddyMeMarkerPrivate *priv = MAP_BUDDY_ME_MARKER_GET_PRIVATE (me_marker); me_marker->priv = priv;

  priv->value = 0;

  priv->point = clutter_texture_new_from_file (PIXMAPDIR "/mapbuddy-me.png", NULL);
  if (priv->point)
   clutter_container_add_actor (CLUTTER_CONTAINER (me_marker), priv->point);
}

ClutterActor *
map_buddy_me_marker_new (void)
{
  return CLUTTER_ACTOR (g_object_new (MAP_BUDDY_TYPE_ME_MARKER, NULL));
}

