/*
 * Copyright (C) 2010 Pierre-Luc Beaudoin <pierre-luc@pierlux.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
#include "config.h"

#include "map-buddy-window.h"
#include "me-marker.h"
#include "sexy-icon-entry.h"
#include "search.h"
#include "merchant.h"
#include "merchant-marker.h"

#include <gtk/gtk.h>
#include <champlain-gtk/champlain-gtk.h>
#include <gdk/gdkkeysyms.h>
#include <glib/gi18n.h>
#include <gdk/gdkx.h>
#include <X11/Xatom.h>
#include <dbus/dbus-glib.h>
#include <libosso-abook/osso-abook.h>

#include <location/location-gps-device.h>
#include <location/location-gpsd-control.h>

static void create_map_area (MapBuddyWindow *self);
static void create_menu (MapBuddyWindow *self);
static void create_toolbar (MapBuddyWindow *self);
static GtkWidget * create_map_selector (void);
static GtkWidget * create_search_selector (void);

static void on_copy_link (GtkButton *button, MapBuddyWindow *self);
static void on_about (GtkButton *button, MapBuddyWindow *self);
static void on_locate_me (GtkButton *button, MapBuddyWindow *self);
static void on_layer (GtkButton *button, MapBuddyWindow *self);
static gboolean on_key_pressed (GtkWindow* window, GdkEventKey *event, MapBuddyWindow *self);
static void on_state_changed (GObject *gobject, GParamSpec *pspec, MapBuddyWindow *self);
static void on_zoom_level_changed (GObject *gobject, GParamSpec *pspec, MapBuddyWindow *self);

static void on_location_error (LocationGPSDControl *control, LocationGPSDControlError error, MapBuddyWindow *self);
static void on_location_changed (LocationGPSDevice *device, MapBuddyWindow *self);
static void on_location_stop (LocationGPSDControl *control, MapBuddyWindow *self);

static gboolean start_location (gpointer data);
static void ungrab_volume_keys (MapBuddyWindow *self);
static void load_state (MapBuddyWindow *self);
static void save_state (MapBuddyWindow *self);
static void show_welcome_dialog (MapBuddyWindow *self);
static void open_url (GtkButton* button, const gchar* url);

G_DEFINE_TYPE (MapBuddyWindow, map_buddy_window, HILDON_TYPE_STACKABLE_WINDOW)

#define SEARCH_PLACES _("Places, city, state or country")
#define SEARCH_PRAIZED _("Merchants, restaurants and businesses")


static void
map_buddy_window_dispose (GObject *object)
{
  MapBuddyWindow *self = MAP_BUDDY_WINDOW (object);

  location_gpsd_control_stop (self->gps_control);
  save_state (self);

  if (self->gps_control)
    {
      g_object_unref (self->gps_control);
      self->gps_control = NULL;
    }

  if (self->gps_device)
    {
      g_object_unref (self->gps_device);
      self->gps_device = NULL;
    }

  G_OBJECT_CLASS (map_buddy_window_parent_class)->dispose (object);
}


static void
map_buddy_window_finalize (GObject *object)
{
  G_OBJECT_CLASS (map_buddy_window_parent_class)->finalize (object);
}


static void
map_buddy_window_class_init (MapBuddyWindowClass *klass)
{
  GObjectClass *object_class = G_OBJECT_CLASS (klass);

  object_class->dispose = map_buddy_window_dispose;
  object_class->finalize = map_buddy_window_finalize;
}


static void
on_window_realize (GtkWidget *widget,
                   MapBuddyWindow *self)
{
  ungrab_volume_keys (self);
}


static void
map_buddy_window_init (MapBuddyWindow *self)
{
  self->show_warning = TRUE;
  self->current_layer = CHAMPLAIN_MAP_SOURCE_OSM_MAPNIK;
  self->current_search = SEARCH_PLACES;
  self->proxy = NULL;
  self->call = NULL;
  self->merchants = NULL;

  g_signal_connect(G_OBJECT (self), "realize",
                   G_CALLBACK (on_window_realize), self);

  /* Create a vbox which will contain our text view: */
  GtkWidget *main_vbox = gtk_vbox_new (FALSE, 0);
  gtk_container_add (GTK_CONTAINER (self), main_vbox);

  /* Create all necessary parts: */
  create_map_area (self);
  create_menu (self);
  create_toolbar (self);

  /* Put the scrolledwindow in the vbox and show it: */
  gtk_box_pack_start (GTK_BOX (main_vbox), self->embed, TRUE, TRUE, 0);
  gtk_widget_show (main_vbox);

  hildon_window_add_toolbar (HILDON_WINDOW (self), GTK_TOOLBAR (self->toolbar));

  /* Connect to zoom in and out buttons */
  g_signal_connect (
      G_OBJECT (self),
      "key-press-event",
      G_CALLBACK (on_key_pressed),
      self);

  self->gps_control = NULL;
  self->gps_control = location_gpsd_control_get_default ();
  self->gps_device = g_object_new (LOCATION_TYPE_GPS_DEVICE, NULL);

  g_object_set (G_OBJECT (self->gps_control),
      "preferred-interval", LOCATION_INTERVAL_30S,
      NULL);

  g_signal_connect (self->gps_control, "error-verbose", G_CALLBACK (on_location_error), self);
  g_signal_connect (self->gps_device, "changed", G_CALLBACK (on_location_changed), self);
  g_signal_connect (self->gps_control, "gpsd-stopped", G_CALLBACK (on_location_stop), self);

  g_timeout_add (1000, start_location, self->gps_control);

  load_state (self);

  if (self->show_warning)
    show_welcome_dialog (self);
}


MapBuddyWindow*
map_buddy_window_new (void)
{
  return g_object_new (MAP_BUDDY_TYPE_WINDOW, NULL);
}


static void
create_map_area (MapBuddyWindow *self)
{
  self->embed = gtk_champlain_embed_new ();
  gtk_widget_show_all (self->embed);

  self->view = gtk_champlain_embed_get_view (GTK_CHAMPLAIN_EMBED (self->embed));

  g_object_set (self->view,
      "scroll-mode", CHAMPLAIN_SCROLL_MODE_KINETIC,
      "min-zoom-level", 2, /* Fill the whole screen */
#if CHAMPLAIN_CHECK_VERSION (0, 4, 3)
      "show-scale", TRUE,
#endif
      NULL);
  g_signal_connect (self->view,
      "notify::state",
      G_CALLBACK (on_state_changed),
      self);
  g_signal_connect (self->view,
      "notify::zoom-level",
      G_CALLBACK (on_zoom_level_changed),
      self);

  self->layer = champlain_layer_new ();
  champlain_view_add_layer (self->view, CHAMPLAIN_LAYER (self->layer));
  champlain_layer_show (self->layer);

  self->merchant_layer = champlain_selection_layer_new ();
  champlain_view_add_layer (self->view, CHAMPLAIN_LAYER (self->merchant_layer));
  champlain_layer_show (self->merchant_layer);

  self->me_marker = map_buddy_me_marker_new ();
  clutter_actor_hide (CLUTTER_ACTOR (self->me_marker));
#if !CHAMPLAIN_CHECK_VERSION (0,4,3)
  /* Hack to force the marker to draw on earlier versions */
  champlain_marker_set_text (CHAMPLAIN_MARKER (self->me_marker), "");
#endif

  champlain_base_marker_set_position (CHAMPLAIN_BASE_MARKER (self->me_marker), 0.0, 0.0);
  clutter_actor_hide (CLUTTER_ACTOR (self->me_marker));
  champlain_layer_add_marker (self->layer, CHAMPLAIN_BASE_MARKER (self->me_marker));

  champlain_view_center_on (self->view, 0, 0);
}

void
map_buddy_window_clear_merchants (MapBuddyWindow *self)
{
  GList *l;

  l = self->merchants;

  if (l == NULL)
    return;

  while (l != NULL)
    {
      MapBuddyMerchant *merchant;

      merchant = (MapBuddyMerchant*) (l->data);

      champlain_layer_remove_marker (CHAMPLAIN_LAYER (self->merchant_layer),
                                     CHAMPLAIN_BASE_MARKER (merchant->marker));

      map_buddy_merchant_free (merchant);
      l = g_list_next (l);
    }

  g_list_free (self->merchants);
  self->merchants = NULL;

#if CHAMPLAIN_CHECK_VERSION (0, 4, 3)
  champlain_view_set_license_text (CHAMPLAIN_VIEW (self->view), NULL);
#endif
}

static void
on_search (SexyIconEntry *entry,
           MapBuddyWindow *self)
{
  const gchar *text;

  text = hildon_entry_get_text (HILDON_ENTRY (entry));

  if (strcmp (self->current_search, SEARCH_PLACES) == 0)
    search_places (self, text);
  else if (strcmp (self->current_search, SEARCH_PRAIZED) == 0)
    search_merchants (self, text);
}

static void
on_search_cleared (SexyIconEntry *entry,
                   MapBuddyWindow *self)
{
  map_buddy_window_clear_merchants (self);

  if (!self->call)
    return;

  g_object_unref (self->call);
  self->call = NULL;
  map_buddy_window_loading_pop (self);
}

static void
on_change_search (SexyIconEntry *entry,
                  SexyIconEntryPosition icon_pos,
                  gint mouse_button,
                  MapBuddyWindow *self)
{
  GtkWidget *selector;
  GtkWidget *dialog;
  gint result;
  gchar *selection;
  gchar *id = NULL;

  if (icon_pos != SEXY_ICON_ENTRY_PRIMARY)
    return;

  selector = create_search_selector ();
  dialog = hildon_picker_dialog_new (GTK_WINDOW (self));
  gtk_window_set_title (GTK_WINDOW (dialog), _("Search for ..."));
  hildon_picker_dialog_set_selector (HILDON_PICKER_DIALOG (dialog),
                                     HILDON_TOUCH_SELECTOR (selector));

  if (strcmp (self->current_search, SEARCH_PLACES) == 0)
    hildon_touch_selector_set_active (HILDON_TOUCH_SELECTOR (selector), 0, 0);
  else if (strcmp (self->current_search, SEARCH_PRAIZED) == 0)
    hildon_touch_selector_set_active (HILDON_TOUCH_SELECTOR (selector), 0, 1);

  result = gtk_dialog_run (GTK_DIALOG (dialog));
  if (result != GTK_RESPONSE_OK)
    {
      gtk_widget_hide (dialog);
      return;
    }

  selection = hildon_touch_selector_get_current_text (HILDON_TOUCH_SELECTOR (selector));
  if (strcmp (selection, _(SEARCH_PLACES)) == 0)
  {
    id = SEARCH_PLACES;
    hildon_entry_set_placeholder (HILDON_ENTRY (self->search_entry), _("Search for a place"));
  }
  else if (strcmp (selection, _(SEARCH_PRAIZED)) == 0)
  {
    id = SEARCH_PRAIZED;
    hildon_entry_set_placeholder (HILDON_ENTRY (self->search_entry), _("Search for a business"));
  }

  self->current_search = id;

  hildon_entry_set_text (HILDON_ENTRY (self->search_entry), "");
  map_buddy_window_clear_merchants (self);

  if (self->proxy)
  {
    g_object_unref (self->proxy);
    self->proxy = NULL;
  }

  gtk_widget_hide (dialog);
}


static void
create_toolbar (MapBuddyWindow *self)
{
  GtkToolItem *entry;
  GdkPixbuf *pixbuf;
  GtkWidget *icon;

  self->toolbar = gtk_toolbar_new ();
  gtk_widget_show (self->toolbar);

  self->center_button = GTK_WIDGET (gtk_tool_button_new (
      gtk_image_new_from_file (PIXMAPDIR "/mapbuddy-locateme.png"),
      _("Center on me")
      ));
  g_signal_connect (
      G_OBJECT (self->center_button),
      "clicked",
      G_CALLBACK (on_locate_me),
      self);
  gtk_toolbar_insert (GTK_TOOLBAR (self->toolbar), GTK_TOOL_ITEM (self->center_button), 0);
  gtk_widget_show_all (self->center_button);

  self->layer_button = GTK_WIDGET (gtk_tool_button_new (
      gtk_image_new_from_file (PIXMAPDIR "/mapbuddy-layer.png"),
      _("Change map")
      ));
  g_signal_connect (
      G_OBJECT (self->layer_button),
      "clicked",
      G_CALLBACK (on_layer),
      self);
  gtk_toolbar_insert (GTK_TOOLBAR (self->toolbar), GTK_TOOL_ITEM (self->layer_button), 1);
  gtk_widget_show_all (self->layer_button);

  /* search entry */
  entry = gtk_tool_item_new ();
  gtk_tool_item_set_expand (entry, TRUE);

  self->search_entry = sexy_icon_entry_new ();
  hildon_entry_set_placeholder (HILDON_ENTRY (self->search_entry), _("Search for a place"));
  hildon_gtk_widget_set_theme_size (self->search_entry, HILDON_SIZE_AUTO);

  pixbuf = gtk_icon_theme_load_icon (gtk_icon_theme_get_default (), "general_search", 48, 0, NULL);
  icon = gtk_image_new_from_pixbuf (pixbuf);
  sexy_icon_entry_set_icon (SEXY_ICON_ENTRY (self->search_entry),
                            SEXY_ICON_ENTRY_PRIMARY,
                            GTK_IMAGE (icon));

  sexy_icon_entry_add_clear_button (SEXY_ICON_ENTRY (self->search_entry));

  g_signal_connect (
      G_OBJECT (self->search_entry),
      "icon-pressed",
      G_CALLBACK (on_change_search),
      self);
  g_signal_connect (
      G_OBJECT (self->search_entry),
      "activate",
      G_CALLBACK (on_search),
      self);
  g_signal_connect (
      G_OBJECT (self->search_entry),
      "cleared",
      G_CALLBACK (on_search_cleared),
      self);

  gtk_container_add (GTK_CONTAINER (entry), self->search_entry);
  gtk_widget_show_all (GTK_WIDGET (entry));
  gtk_toolbar_insert (GTK_TOOLBAR (self->toolbar), GTK_TOOL_ITEM (entry), 2);
}

#define MAP_DRIVING _("Driving")
#define MAP_CYCLE _("Cycling")
#define MAP_TRANSIT _("Public Transit")
#define MAP_TERRAIN _("Terrain")

static GtkWidget *
create_map_selector ()
{
  GtkWidget *selector;

  selector = hildon_touch_selector_new_text();

  hildon_touch_selector_append_text (HILDON_TOUCH_SELECTOR (selector),
                                     MAP_DRIVING);
  hildon_touch_selector_append_text (HILDON_TOUCH_SELECTOR (selector),
                                     MAP_CYCLE);
  hildon_touch_selector_append_text (HILDON_TOUCH_SELECTOR (selector),
                                     MAP_TRANSIT);
  hildon_touch_selector_append_text (HILDON_TOUCH_SELECTOR (selector),
                                     MAP_TERRAIN);

  return selector;
}


static GtkWidget *
create_search_selector ()
{
  GtkWidget *selector;

  selector = hildon_touch_selector_new_text();

  hildon_touch_selector_append_text (HILDON_TOUCH_SELECTOR (selector),
                                     SEARCH_PLACES);
  hildon_touch_selector_append_text (HILDON_TOUCH_SELECTOR (selector),
                                     SEARCH_PRAIZED);

  return selector;
}

static GtkWidget*
add_menu_item (GtkWidget *main_menu,
               const gchar* title,
               GCallback clicked_callback,
               gpointer user_data)
{
  HildonAppMenu *app_menu = HILDON_APP_MENU (main_menu);

  /* Create a button, add it, and return it: */
  GtkWidget *button = hildon_button_new_with_text (HILDON_SIZE_AUTO,
        HILDON_BUTTON_ARRANGEMENT_VERTICAL, title, NULL);
  gtk_widget_show (button);

  g_signal_connect_after (button, "clicked",
      G_CALLBACK (clicked_callback), user_data);

  hildon_app_menu_append (app_menu, GTK_BUTTON (button));

  return button;
}

/* Create the menu items needed: */
static void
create_menu (MapBuddyWindow *self)
{
  GtkWidget *main_menu;

  main_menu = hildon_app_menu_new ();

  /* Create the menu items */
  self->copy_item = add_menu_item (main_menu, _("Copy Map Link"),
      G_CALLBACK (on_copy_link), self);

  add_menu_item (main_menu, _("Donate"),
      G_CALLBACK (open_url), "http://www.pierlux.com/map-buddy/donate/");

  add_menu_item (main_menu, _("About"),
      G_CALLBACK (on_about), self);

  /* Add menu to HildonWindow */
  hildon_window_set_app_menu (
      HILDON_WINDOW (self),
      HILDON_APP_MENU (main_menu));

  /* We need to show menu items */
  gtk_widget_show_all (GTK_WIDGET (main_menu));
}

static void
on_about (GtkButton *button,
          MapBuddyWindow *self)
{
  const char *license[] = {
     N_("This library is free software; you can redistribute it and/or "
        "modify it under the terms of the GNU Library General Public "
        "License as published by the Free Software Foundation; either "
        "version 2 of the License, or (at your option) any later version. \n"),
     N_("This library is distributed in the hope that it will be useful, "
        "but WITHOUT ANY WARRANTY; without even the implied warranty of "
        "MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU "
        "Library General Public License for more details.\n"),
     N_("You should have received a copy of the GNU Library General Public License "
        "along with this library; see the file COPYING.LIB.  If not, write to "
        "the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor, "
        "Boston, MA 02110-1301, USA.\n"),
  };
  char *license_translated;
  gint result;
  GtkWidget *dialog, *area, *pan, *label;
  gchar *text;

  license_translated = g_strconcat (_(license[0]), "\n", _(license[1]), "\n",
      _(license[2]), "\n", NULL);

  dialog = gtk_dialog_new_with_buttons (_("About Map Buddy"),
      GTK_WINDOW (self),
      GTK_DIALOG_MODAL | GTK_DIALOG_DESTROY_WITH_PARENT,
      _("License"),
      GTK_RESPONSE_HELP,
      _("Web Site"),
      GTK_RESPONSE_NO,
      "Praized Media",
      GTK_RESPONSE_APPLY,
      "YellowPages.ca",
      GTK_RESPONSE_ACCEPT,
      "Localeze",
      GTK_RESPONSE_YES,
      NULL);
  gtk_window_set_default_size (GTK_WINDOW (dialog), -1, 400);

  text = g_strdup_printf ("<big><b>%s %s</b></big>\n\n%s\n\n<small>%s</small>",
      _("Map Buddy"),
      VERSION,
      _("Copyright \xc2\xa9 2009 Pierre-Luc Beaudoin"),
      _("A map viewer for Maemo 5.\n\nThe maps are from the OpenStreetMap project and\nmapsforfree.com.\n\nPlace search is provided by geonames.org.\n\nBusinesses search is powered by Praized Media.\nUS data provided by Localeze.\nCanadian business listings distributed by YellowPages.ca™"));

  label = gtk_label_new (NULL);
  gtk_label_set_markup (GTK_LABEL (label), text);
  gtk_widget_show (label);
  pan = hildon_pannable_area_new ();
  gtk_widget_show (pan);
  hildon_pannable_area_add_with_viewport (HILDON_PANNABLE_AREA (pan), label);

  area = gtk_dialog_get_content_area (GTK_DIALOG (dialog));
  gtk_container_add (GTK_CONTAINER (area), pan);

  result = gtk_dialog_run (GTK_DIALOG (dialog));
  gtk_widget_destroy (dialog);

  switch (result)
    {
      case GTK_RESPONSE_ACCEPT:
        open_url (NULL, "http://www.yellowpages.ca");
        break;
      case GTK_RESPONSE_YES:
        open_url (NULL, "http://www.localeze.com");
        break;
      case GTK_RESPONSE_APPLY:
        open_url (NULL, "http://www.praizedmedia.com");
        break;
      case GTK_RESPONSE_NO:
        open_url (NULL, "http://www.pierlux.com/map-buddy/");
        break;
      case GTK_RESPONSE_HELP:
        dialog = gtk_dialog_new ();
        gtk_window_set_default_size (GTK_WINDOW (dialog), -1, 400);
        label = gtk_label_new (NULL);
        gtk_label_set_markup (GTK_LABEL (label), license_translated);
        gtk_label_set_line_wrap (GTK_LABEL (label), TRUE);
        gtk_widget_show (label);
        pan = hildon_pannable_area_new ();
        gtk_widget_show (pan);
        hildon_pannable_area_add_with_viewport (HILDON_PANNABLE_AREA (pan), label);

        area = gtk_dialog_get_content_area (GTK_DIALOG (dialog));
        gtk_container_add (GTK_CONTAINER (area), pan);

        result = gtk_dialog_run (GTK_DIALOG (dialog));
        gtk_widget_destroy (dialog);
        break;
      default:
        break;
    }

  g_free (license_translated);
  g_free (text);
}


#define LEN 255
static void
on_copy_link (GtkButton *button,
              MapBuddyWindow *self)
{
  GtkClipboard *clipboard;
  gdouble lat, lon;
  gchar slat[LEN], slon[LEN];
  gint zoom;
  gchar *url;

  g_object_get (self->view,
                "latitude", &lat,
                "longitude", &lon,
                "zoom-level", &zoom,
                NULL);

  g_ascii_dtostr (slat, LEN, lat);
  g_ascii_dtostr (slon, LEN, lon);

  clipboard = gtk_clipboard_get (GDK_SELECTION_CLIPBOARD);
  url = g_strdup_printf ("http://www.openstreetmap.org/?lat=%s&lon=%s&zoom=%d", slat, slon, zoom);
  gtk_clipboard_set_text (clipboard, url, -1);
}


static gboolean
on_key_pressed (GtkWindow* window,
                GdkEventKey *event,
                MapBuddyWindow *self)
{
  switch (event->keyval)
    {
      case GDK_F7:
        champlain_view_zoom_in (self->view);
        return TRUE;
        break;
      case GDK_F8:
        champlain_view_zoom_out (self->view);
        return TRUE;
        break;
      default:
        break;
    }

  return FALSE;
}


static void
on_locate_me (GtkButton *button,
              MapBuddyWindow *self)
{
  if (self->has_fix)
    champlain_view_center_on (self->view, self->latitude, self->longitude);
  else
    hildon_banner_show_information (GTK_WIDGET (self), NULL, _("Your position is unknown"));
}


static void
on_layer (GtkButton *button,
           MapBuddyWindow *self)
{
  GtkWidget *selector;
  GtkWidget *dialog;
  gint result;
  gchar *selection;
  ChamplainMapSource *source;
  ChamplainMapSourceFactory *factory;
  gchar *id = NULL;

  selector = create_map_selector ();
  dialog = hildon_picker_dialog_new (GTK_WINDOW (self));
  gtk_window_set_title (GTK_WINDOW (dialog), _("Select a map"));
  hildon_picker_dialog_set_selector (HILDON_PICKER_DIALOG (dialog),
                                     HILDON_TOUCH_SELECTOR (selector));

  if (strcmp (self->current_layer, CHAMPLAIN_MAP_SOURCE_OSM_MAPNIK) == 0)
    hildon_touch_selector_set_active (HILDON_TOUCH_SELECTOR (selector), 0, 0);
  else if (strcmp (self->current_layer, CHAMPLAIN_MAP_SOURCE_OSM_CYCLE_MAP) == 0)
    hildon_touch_selector_set_active (HILDON_TOUCH_SELECTOR (selector), 0, 1);
  else if (strcmp (self->current_layer, CHAMPLAIN_MAP_SOURCE_OSM_TRANSPORT_MAP) == 0)
    hildon_touch_selector_set_active (HILDON_TOUCH_SELECTOR (selector), 0, 2);
  else if (strcmp (self->current_layer, CHAMPLAIN_MAP_SOURCE_MFF_RELIEF) == 0)
    hildon_touch_selector_set_active (HILDON_TOUCH_SELECTOR (selector), 0, 3);

  result = gtk_dialog_run (GTK_DIALOG (dialog));
  if (result != GTK_RESPONSE_OK)
    {
      gtk_widget_hide (dialog);
      return;
    }

  selection = hildon_touch_selector_get_current_text (HILDON_TOUCH_SELECTOR (selector));
  if (strcmp (selection, MAP_DRIVING) == 0)
    id = CHAMPLAIN_MAP_SOURCE_OSM_MAPNIK;
  else if (strcmp (selection, MAP_CYCLE) == 0)
    id = CHAMPLAIN_MAP_SOURCE_OSM_CYCLE_MAP;
  else if (strcmp (selection, MAP_TRANSIT) == 0)
    id = CHAMPLAIN_MAP_SOURCE_OSM_TRANSPORT_MAP;
  else if (strcmp (selection, MAP_TERRAIN) == 0)
    id = CHAMPLAIN_MAP_SOURCE_MFF_RELIEF;

  self->current_layer = id;
  factory = champlain_map_source_factory_dup_default ();
  source = champlain_map_source_factory_create (factory, id);

  champlain_view_set_map_source (self->view, source);

  g_object_unref (factory);
  gtk_widget_hide (dialog);
}

static void
on_state_changed (GObject *gobject,
                  GParamSpec *pspec,
                  MapBuddyWindow *self)
{
  ChamplainState state;

  g_object_get (self->view,
      "state", &state, /* Fill the whole screen */
      NULL);

  if (state == CHAMPLAIN_STATE_LOADING)
    map_buddy_window_loading_push (self);
  else
    map_buddy_window_loading_pop (self);
}

/* Found on http://wiki.maemo.org/Documentation/Maemo_5_Developer_Guide/Porting_Software/Porting_Existing_GTK%2B_Application_to_Maemo_5 */
static void
ungrab_volume_keys (MapBuddyWindow *self)
{
    /* Tell maemo-status-volume daemon to ungrab keys */
    unsigned long val = 1; /* ungrab, use 0 to grab */
    Atom atom;
    GdkDisplay *display;
    GdkWindow *window;

    display = gtk_widget_get_display (GTK_WIDGET (self));
    window = gtk_widget_get_window (GTK_WIDGET (self));
    atom = gdk_x11_get_xatom_by_name_for_display (display, "_HILDON_ZOOM_KEY_ATOM");
    XChangeProperty (GDK_DISPLAY_XDISPLAY (display),
                     GDK_WINDOW_XID (window), atom, XA_INTEGER, 32,
                     PropModeReplace, (unsigned char *) &val, 1);
}


static void
on_location_error (LocationGPSDControl *control,
                   LocationGPSDControlError error,
                   MapBuddyWindow *self)
{
  g_debug ("location error: %d\n", error);
  self->has_fix = FALSE;
}

static void
update_me_marker_radius (MapBuddyWindow *self)
{
  gfloat radius = 0; /* in px */

  /* This is necessary because Markers don't have access to the current zoom
   * level */
#if CHAMPLAIN_CHECK_VERSION (0, 4, 3)
  if (self->has_fix)
    {
      ChamplainMapSource *source;
      gint zoom;

      g_object_get (self->view,
          "map-source", &source,
          "zoom-level", &zoom,
          NULL);

      radius = self->pos_error;
      radius /= champlain_map_source_get_meters_per_pixel (source,
          zoom, self->latitude, self->longitude);

      if (radius > 600)
        radius = 0; /* Drawing a too big circle jam the device */

      g_object_unref (source);
    }
#endif

  map_buddy_me_marker_set_radius (MAP_BUDDY_ME_MARKER (self->me_marker),
                                  radius);

}


static void
on_zoom_level_changed (GObject *gobject,
                       GParamSpec *pspec,
                       MapBuddyWindow *self)
{
  update_me_marker_radius (self);
}


static void
on_location_changed (LocationGPSDevice *device,
                     MapBuddyWindow *self)
{
  if (!device)
    return;

  g_debug ("Location changed\n");
  if (device->fix && (device->fix->fields & LOCATION_GPS_DEVICE_LATLONG_SET))
    {
      g_debug ("lat = %f, long = %f, precision = %f\n",
               device->fix->latitude,
               device->fix->longitude,
               device->fix->eph / 100);
      champlain_base_marker_set_position (CHAMPLAIN_BASE_MARKER (self->me_marker),
                                          device->fix->latitude,
                                          device->fix->longitude);

      self->has_fix = TRUE;
      self->latitude = device->fix->latitude;
      self->longitude = device->fix->longitude;
      self->pos_error = device->fix->eph / 100.0;

      update_me_marker_radius (self);
      clutter_actor_show (CLUTTER_ACTOR (self->me_marker));
    }
  else
    {
      g_debug ("GPS: no position\n");
      clutter_actor_hide (CLUTTER_ACTOR (self->me_marker));
      self->has_fix = FALSE;
    }
}


static void
on_location_stop (LocationGPSDControl *control,
                  MapBuddyWindow *self)
{
  self->has_fix = FALSE;
  g_debug ("Stop location\n");
}

static gboolean
start_location (gpointer data)
{
  g_debug ("Start location\n");
  location_gpsd_control_start (LOCATION_GPSD_CONTROL (data));
  return FALSE;
}

static void
save_state (MapBuddyWindow *self)
{
  GKeyFile *file;
  gdouble lat, lon;
  gint zoom;
  gchar *filename;
  gchar *data;
  GError *error = NULL;

  file = g_key_file_new ();

  g_object_get (self->view,
                "latitude", &lat,
                "longitude", &lon,
                "zoom-level", &zoom,
                NULL);
  g_key_file_set_double (file, "position", "latitude", lat);
  g_key_file_set_double (file, "position", "longitude", lon);
  g_key_file_set_integer (file, "position", "zoom", zoom);
  g_key_file_set_string (file, "position", "source", self->current_layer);

  g_key_file_set_boolean (file, "dialogs", "welcome", self->show_warning);
  g_key_file_set_string (file, "search", "source", self->current_search);

  filename = g_build_filename (g_get_user_config_dir (), "mapbuddy.cfg", NULL);
  data = g_key_file_to_data (file, NULL, NULL);

  if (!g_file_set_contents (filename, data, -1, &error))
    {
      g_warning ("Error writing %s: %s", filename, error->message);
      g_error_free (error);
    }

  g_key_file_free (file);
  g_free (data);
  g_free (filename);
}

static void
load_state (MapBuddyWindow *self)
{
  gchar *filename = NULL;
  GKeyFile *file;
  GError *error = NULL;
  gdouble lat, lon;
  gint zoom;
  gchar *current;
  ChamplainMapSource *source;
  ChamplainMapSourceFactory *factory;

  filename = g_build_filename (g_get_user_config_dir (),
                               "mapbuddy.cfg",
                               NULL);

  file = g_key_file_new ();
  if (!g_key_file_load_from_file (file,
                                 filename,
                                 G_KEY_FILE_KEEP_COMMENTS | G_KEY_FILE_KEEP_TRANSLATIONS,
                                 &error))
    {
      g_warning ("Error loading %s: %s", filename, error->message);
      g_error_free (error);
      //g_key_file_free (file);
      return;
    }
  g_free (filename);

  lat = g_key_file_get_double (file, "position", "latitude", &error);
  if (error)
    {
      g_warning ("Failed to retrieve saved stated: %s", error->message);
      g_error_free (error);
      goto clean;
    }

  lon = g_key_file_get_double (file, "position", "longitude", &error);
  if (error)
    {
      g_warning ("Failed to retrieve saved stated: %s", error->message);
      g_error_free (error);
      goto clean;
    }

  zoom = g_key_file_get_integer (file, "position", "zoom", &error);
  if (error)
    {
      g_warning ("Failed to retrieve saved stated: %s", error->message);
      g_error_free (error);
      goto clean;
    }

  self->show_warning = g_key_file_get_boolean (file, "dialogs", "welcome", &error);
  if (error)
    {
      g_warning ("Failed to retrieve saved stated: %s", error->message);
      g_error_free (error);
      goto clean;
    }

  current = g_key_file_get_string (file, "position", "source", &error);
  if (error)
    {
      g_warning ("Failed to retrieve saved stated: %s", error->message);
      g_error_free (error);
      goto clean;
    }
  else
    self->current_layer = current;

  current = g_key_file_get_string (file, "search", "source", &error);
  if (error)
    {
      g_warning ("Failed to retrieve saved stated: %s", error->message);
      g_error_free (error);
      goto clean;
    }
  else
    self->current_search = current;

  factory = champlain_map_source_factory_dup_default ();
  source = champlain_map_source_factory_create (factory, self->current_layer);
  champlain_view_set_map_source (self->view, source);
  g_object_unref (factory);

  if (strcmp (self->current_search, SEARCH_PLACES) == 0)
    hildon_entry_set_placeholder (HILDON_ENTRY (self->search_entry), _("Search for a place"));
  else if (strcmp (self->current_search, SEARCH_PRAIZED) == 0)
    hildon_entry_set_placeholder (HILDON_ENTRY (self->search_entry), _("Search for a business"));

  champlain_view_set_zoom_level (self->view, zoom);
  champlain_view_center_on (self->view, lat, lon);

clean:
  g_key_file_free (file);
}


static void
on_welcome_response (GtkDialog *dialog,
                     gint response_id,
                     MapBuddyWindow *self)
{
  GtkWidget *content_area;
  GList *children, *last;

  content_area = gtk_dialog_get_content_area (GTK_DIALOG (dialog));
  children = gtk_container_get_children (GTK_CONTAINER (content_area));
  last = g_list_last (children);

  self->show_warning = !hildon_check_button_get_active (HILDON_CHECK_BUTTON (last->data));

  gtk_widget_hide (GTK_WIDGET (dialog));
  g_list_free (children);
}


static void
show_welcome_dialog (MapBuddyWindow *self)
{
  GtkWidget *dialog, *label, *content_area, *check;

  dialog = gtk_dialog_new_with_buttons (_("Welcome to Map Buddy"),
      GTK_WINDOW (self),
      GTK_DIALOG_MODAL,
      _("Dismiss"), GTK_RESPONSE_YES,
      NULL
      );

  content_area = gtk_dialog_get_content_area (GTK_DIALOG (dialog));

  label = gtk_label_new (_("This application uses Internet connectivity to download"
      " map data as you go.  This can represent a lot of bandwidth."
      " Take this fact into account when using your mobile Internet access.\n"));
  gtk_label_set_line_wrap (GTK_LABEL (label), TRUE);
  gtk_widget_show (label);
  gtk_container_add (GTK_CONTAINER (content_area), label);

  check = hildon_check_button_new (HILDON_SIZE_AUTO);
  gtk_button_set_label (GTK_BUTTON (check), _("Don't remind me"));
  gtk_widget_show (check);
  gtk_container_add (GTK_CONTAINER (content_area), check);

  g_signal_connect(dialog, "response",
                   G_CALLBACK (on_welcome_response), self);

  gtk_widget_show (dialog);
}

/* Works as a FILO, the window will be in loading state until the count returns
 * to 0 */
void
map_buddy_window_loading_push (MapBuddyWindow *self)
{
  self->loading_count++;
  hildon_gtk_window_set_progress_indicator (GTK_WINDOW (self), 1);
}

void
map_buddy_window_loading_pop (MapBuddyWindow *self)
{
  self->loading_count--;
  if (self->loading_count < 0)
    self->loading_count = 0;

  hildon_gtk_window_set_progress_indicator (GTK_WINDOW (self),
      self->loading_count > 0 ? 1: 0);
}

static void
open_url (GtkButton* button,
          const gchar* url)
{
  DBusGConnection *connection;
  GError *error = NULL;
  DBusGProxy *proxy;

  connection = dbus_g_bus_get (DBUS_BUS_SYSTEM,
                               &error);
  if (connection == NULL)
    {
      g_printerr ("Failed to open connection to bus: %s\n",
                  error->message);
      g_error_free (error);
    }

  /* Create a proxy object for the "bus driver" (name "org.freedesktop.DBus") */
  proxy = dbus_g_proxy_new_for_name (connection,
                                     "com.nokia.osso_browser",
                                     "/com/nokia/osso_browser",
                                     "com.nokia.osso_browser");

  /* Call ListNames method, wait for reply */
  error = NULL;
  if (!dbus_g_proxy_call (proxy, "load_url", &error,
                          G_TYPE_STRING, url,
                          G_TYPE_INVALID,
                          G_TYPE_STRING,
                          NULL,
                          G_TYPE_INVALID))
    {
      /* Just do demonstrate remote exceptions versus regular GError */
      if (error->domain == DBUS_GERROR && error->code == DBUS_GERROR_REMOTE_EXCEPTION)
        g_printerr ("Caught remote method exception %s: %s",
                    dbus_g_error_get_name (error),
                    error->message);
      else
        g_printerr ("Error: %s\n", error->message);
      g_error_free (error);
    }

  g_object_unref (proxy);
  dbus_g_connection_unref (connection);
}


static void
add_to_abook (GtkButton* button,
              MapBuddyMerchant *merchant)
{
  OssoABookContact *contact;
  GtkWidget *dialog;

  contact = osso_abook_contact_new ();
  if (merchant->name)
    osso_abook_contact_set_value (E_CONTACT (contact), EVC_ORG, merchant->name);
  if (merchant->url)
    osso_abook_contact_set_value (E_CONTACT (contact), EVC_URL, merchant->url);
  if (merchant->phone)
    {

      GRegex *reg = g_regex_new ("[\\(\\)\\ \\-]", 0, 0, NULL);
      gchar *phone;

      phone = g_regex_replace (reg, merchant->phone, -1, 0, "", 0, NULL);

      osso_abook_contact_set_value (E_CONTACT (contact), EVC_TEL, phone);

      g_free (phone);
    }
  if (merchant->address &&
      merchant->city &&
      merchant->province &&
      merchant->postal_code &&
      merchant->country)
    {
      EVCardAttribute *attr;

      attr = e_vcard_attribute_new (NULL, EVC_ADR);

      e_vcard_add_attribute_with_values (E_VCARD (contact), attr,
          "", "", /* the first 2 elements are empty */
          merchant->address,
          merchant->city,
          merchant->province,
          merchant->postal_code,
          merchant->country,
          NULL);

      /* attr ownership has been transfered to contact, don't free it */
    }

  dialog = osso_abook_temporary_contact_dialog_new_with_contact (NULL, contact);
  gtk_dialog_run (GTK_DIALOG (dialog));
  gtk_widget_destroy (dialog);

  g_object_unref (contact);
}

static void
place_call (GtkButton* button,
            const gchar* number)
{
  DBusGConnection *connection;
  GError *error = NULL;
  DBusGProxy *proxy;
  gchar *stripped;
  GRegex *reg = g_regex_new ("[\\(\\)\\ \\-]", 0, 0, NULL);
  gchar *phone;

  stripped = g_regex_replace (reg, number, -1, 0, "", 0, NULL);

  connection = dbus_g_bus_get (DBUS_BUS_SYSTEM,
                               &error);
  if (connection == NULL)
    {
      g_printerr ("Failed to open connection to bus: %s\n",
                  error->message);
      g_error_free (error);
    }

  /* Create a proxy object for the "bus driver" (name "org.freedesktop.DBus") */
  proxy = dbus_g_proxy_new_for_name (connection,
                                     "com.nokia.csd",
                                     "/com/nokia/csd/call",
                                     "com.nokia.csd.Call");

  /* Call ListNames method, wait for reply */
  error = NULL;
  if (!dbus_g_proxy_call (proxy, "CreateWith", &error,
                          G_TYPE_STRING, stripped,
                          G_TYPE_UINT, 0,
                          G_TYPE_INVALID,
                          G_TYPE_STRING,
                          NULL,
                          G_TYPE_INVALID))
    {
      /* Just do demonstrate remote exceptions versus regular GError */
      if (error->domain == DBUS_GERROR && error->code == DBUS_GERROR_REMOTE_EXCEPTION)
        g_printerr ("Caught remote method exception %s: %s",
                    dbus_g_error_get_name (error),
                    error->message);
      else
        g_printerr ("Error: %s\n", error->message);
      g_error_free (error);
    }

  g_object_unref (proxy);
  dbus_g_connection_unref (connection);
  g_free (stripped);
}

void
map_buddy_window_show_details (MapBuddyWindow *self,
                               MapBuddyMerchant *merchant)
{
  GtkWidget *win;
  GtkWidget *hbox, *bbox;
  gchar *text;
  GtkWidget *label, *button;
  GString *license;

  win = hildon_stackable_window_new ();
  gtk_window_set_title (GTK_WINDOW (win), _("Merchant detail"));

  license = g_string_new (_("Powered by Praized Media\n"));
  if (g_strcmp0 (merchant->country_code, "CA") == 0)
    g_string_append (license, _("Canadian business listings distributed by YellowPages.ca\u2122"));
  else if (g_strcmp0 (merchant->country_code, "US") == 0)
    g_string_append (license, _("US data provided by Localeze"));

  text = g_strdup_printf ("<big><b>%s</b></big>\n\n%s\n\n%s\n%s (%s)\n%s\n\n<span font=\"8\">%s</span>",
                        merchant->name,
                        merchant->phone,
                        merchant->address,
                        merchant->city,
                        merchant->province,
                        merchant->postal_code,
                        license->str);
  label =  gtk_label_new (NULL);
  gtk_label_set_markup (GTK_LABEL (label), text);

  hbox = gtk_hbox_new (FALSE, 0);
  gtk_box_pack_start (GTK_BOX (hbox),
                      label,
                      TRUE,
                      TRUE,
                      0);

  bbox = gtk_vbutton_box_new ();
  gtk_button_box_set_layout (GTK_BUTTON_BOX (bbox), GTK_BUTTONBOX_CENTER);
  gtk_box_set_spacing (GTK_BOX (bbox), 10);
  gtk_box_pack_start (GTK_BOX (hbox),
                      bbox,
                      TRUE,
                      TRUE,
                      0);

  if (merchant->phone)
    {
      button = hildon_button_new_with_text (HILDON_SIZE_FINGER_HEIGHT,
                                            HILDON_BUTTON_ARRANGEMENT_VERTICAL,
                                            _("Call"),
                                            merchant->phone);
      gtk_widget_show (button);
      g_signal_connect (G_OBJECT (button), "clicked",
                        G_CALLBACK (place_call), merchant->phone);
      gtk_box_pack_start (GTK_BOX (bbox),
                          button,
                          TRUE,
                          TRUE,
                          0);

      button = hildon_button_new_with_text (HILDON_SIZE_FINGER_HEIGHT,
                                            HILDON_BUTTON_ARRANGEMENT_VERTICAL,
                                            _("Add to Address Book"),
                                            NULL);
      gtk_widget_show (button);
      g_signal_connect (G_OBJECT (button), "clicked",
                        G_CALLBACK (add_to_abook), merchant);
      gtk_box_pack_start (GTK_BOX (bbox),
                          button,
                          TRUE,
                          TRUE,
                          0);
    }

  if (merchant->url)
    {
      button = hildon_button_new_with_text (HILDON_SIZE_FINGER_HEIGHT,
                                            HILDON_BUTTON_ARRANGEMENT_VERTICAL,
                                            _("Open Web Site"),
                                            NULL);
      gtk_widget_show (button);
      g_signal_connect (G_OBJECT (button), "clicked",
                        G_CALLBACK (open_url), merchant->url);
      gtk_box_pack_start (GTK_BOX (bbox),
                          button,
                          TRUE,
                          TRUE,
                          0);
    }

  gtk_container_add (GTK_CONTAINER (win),
                     hbox);

  /* This call show the window and also add the window to the stack */
  gtk_widget_show_all (win);
  g_string_free (license, TRUE);
}
