# -*- coding: utf-8 -*-

###########################################################################
#    Maevies
#    Copyright (C) 2010 Simón Pena <spenap@gmail.com>
#
#    This program is free software: you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation, either version 3 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program.  If not, see <http://www.gnu.org/licenses/>.
###########################################################################

import pygtk
import os
from xml.sax import saxutils
pygtk.require('2.0')
import gtk
import hildon
import pango
import gobject
from webkit import WebView

from lib import constants
from lib.asyncworker import AsyncWorker, AsyncItem
from lib.util import image_downloader
from moviemanager import MovieManager, TmdbMovie, TmdbMovieImage, \
    WatcMovie

moviemanager = MovieManager()

class Maevies(hildon.StackableWindow):

    ACTION_SEARCH = 0
    ACTION_ABOUT = 1
    ACTION_THEATERS = 2
    ACTION_FAVORITES = 3

    def __init__(self):
        super(Maevies, self).__init__()
        self.set_title('%(name)s - %(version)s' %
                {'name' : constants.APP_NAME,
                 'version' : constants.APP_VERSION})
        self.connect('delete-event',
                     self.quit)

        self.add(self._create_contents())
        self.set_app_menu(self._create_app_menu())

        self.show_all()

    def _create_button(self, title, subtitle, action):
        box = gtk.VBox()
        box.set_border_width(20)

        button = hildon.Button(gtk.HILDON_SIZE_THUMB_HEIGHT,
                               hildon.BUTTON_ARRANGEMENT_VERTICAL,
                               title, subtitle)
        button.connect('clicked', self._button_clicked, action)

        box.pack_start(button, expand=True, fill=False)

        return box

    def _create_contents(self):
        contents = gtk.HBox()
        contents.set_border_width(60)
        contents.set_homogeneous(True)
        contents.pack_start(self._create_button('On Theaters',
                                                'Movies playing',
                                                self.ACTION_THEATERS),
                            expand=True, fill=True)
        contents.pack_start(self._create_button('Favorites',
                                                'Your saved movies',
                                                self.ACTION_FAVORITES),
                            expand=True, fill=True)
        contents.pack_start(self._create_button('Search',
                                                'Enter a new search',
                                                self.ACTION_SEARCH),
                            expand=True, fill=True)

        return contents;

    def _button_clicked(self, button, action):
        if action == self.ACTION_THEATERS:
            theaters_view = TheatersWindow()
            theaters_view.display_shows()
        elif action == self.ACTION_FAVORITES:
            favorites_window = FavoritesWindow()
        elif action == self.ACTION_SEARCH:
            search_dialog = SearchDialog(self)
            if search_dialog.run() == gtk.RESPONSE_ACCEPT:
                results_window = ResultsWindow()
                results_window.start_search(search_dialog.get_search_term(),
                                            search_dialog.get_search_category())
            search_dialog.destroy()
        elif action == self.ACTION_ABOUT:
            about_dialog = AboutDialog(self)
            about_dialog.set_logo(constants.APP_ICON)
            about_dialog.set_name(constants.APP_NAME)
            about_dialog.set_version(constants.APP_VERSION)
            about_dialog.set_comments(constants.APP_DESCRIPTION)
            about_dialog.set_authors(constants.APP_AUTHORS)
            about_dialog.set_copyright(constants.APP_COPYRIGHT)
            about_dialog.set_license(saxutils.escape(constants.APP_LICENSE))
            about_dialog.run()
            about_dialog.destroy()

    def _create_app_menu(self):
        menu = hildon.AppMenu()

        about = hildon.GtkButton(gtk.HILDON_SIZE_AUTO)
        about.set_label('About')
        about.connect('clicked', self._button_clicked, self.ACTION_ABOUT)

        menu.append(about)

        menu.show_all()

        return menu

    def run(self):
        gtk.main()

    def quit(self, widget, event):
        moviemanager.destroy()
        gtk.main_quit()

class SearchDialog(gtk.Dialog):

    TMDB_SEARCH = 0
    WATC_SEARCH = 1
    search_fields = {TMDB_SEARCH:'TMDb',
                      WATC_SEARCH:'WATC'}

    def __init__(self, parent):
        super(SearchDialog, self).__init__(parent=parent,
                                           flags=gtk.DIALOG_DESTROY_WITH_PARENT)
        self.set_title('Enter search terms')

        self.vbox.pack_start(self._create_contents(), True, False, 0)
        self.add_button(gtk.STOCK_OK, gtk.RESPONSE_ACCEPT)

        self.show_all()

    def _create_contents(self):
        self._search_entry = hildon.Entry(gtk.HILDON_SIZE_FINGER_HEIGHT)
        search_button = self._create_picker_button()

        search_contents = gtk.VBox()

        search_contents.pack_start(self._search_entry,
                                   expand=True, fill=True)
        search_contents.pack_start(search_button,
                                   expand=True, fill=True)

        return search_contents

    def _create_picker_button(self):
        self._picker_button = hildon.PickerButton(gtk.HILDON_SIZE_FINGER_HEIGHT,
                                                  hildon.BUTTON_ARRANGEMENT_HORIZONTAL)
        self._picker_button.set_title('Search on')

        selector = hildon.TouchSelector(text=True)
        selector.set_column_selection_mode(hildon.TOUCH_SELECTOR_SELECTION_MODE_SINGLE)

        for search_method in [self.TMDB_SEARCH, self.WATC_SEARCH]:
            selector.append_text(self.search_fields[search_method])

        self._picker_button.set_selector(selector)
        self._picker_button.set_active(0)

        return self._picker_button

    def get_search_term(self):
        return self._search_entry.get_text()

    def get_search_category(self):
        return self._picker_button.get_active()

class MovieListWindow(hildon.StackableWindow):

    def __init__(self):
        super(MovieListWindow, self).__init__()
        self.add(self._create_contents())
        self.show_all()

    def _create_contents(self):
        content_area = hildon.PannableArea()
        self._movies_view = MoviesView()
        self._movies_view.connect('row-activated', self._row_activated_cb)

        content_area.add(self._movies_view)
        return content_area

    def _row_activated_cb(self, view, path, column):
        movie = view.get_movie_from_path(path)
        if isinstance(movie, TmdbMovie):
            MovieWindow(movie)
        elif isinstance(movie, WatcMovie):
            WatcWindow(movie)

class FavoritesWindow(MovieListWindow):

    NO_FAVORITES = 'Mark a movie as favorite and it will appear here'

    def __init__(self):
        super(FavoritesWindow, self).__init__()
        self.set_title('Favorites')
        if moviemanager._favorites and self._movies_view:
            self._movies_view.add_movies(moviemanager._favorites)

    def _create_contents(self):
        if moviemanager._favorites:
            return super(FavoritesWindow, self)._create_contents()
        no_items_label = gtk.Label(self.NO_FAVORITES)
        no_items_label.set_justify(gtk.JUSTIFY_CENTER)
        return no_items_label

class ResultsWindow(MovieListWindow):

    SEARCHING = 'Searching'
    NO_RESULTS = 'No results found'
    ERROR = 'Unable to perform search'

    def __init__(self):
        super(ResultsWindow, self).__init__()
        self.set_title('Search results')
        moviemanager.response_received_cb = self._response_received_cb

    def _create_contents(self):
        self._content = super(ResultsWindow, self)._create_contents()
        self._background = gtk.Label(self.SEARCHING)
        self._background.set_justify(gtk.JUSTIFY_CENTER)
        return self._background

    def start_search(self, search_term, search_category):
        self._show_banner(self._get_search_message(search_term, search_category))
        hildon.hildon_gtk_window_set_progress_indicator(self, True)
        moviemanager.query(search_term, search_category)

    def _response_received_cb(self, movies, error=None):
        hildon.hildon_gtk_window_set_progress_indicator(self, False)
        if not error and movies:
            self._movies_view.add_movies(movies)
            self.remove(self._background)
            self._content.show_all()
            self.add(self._content)
        else:
            if error:
                self._show_banner(constants.DBUS_ERROR,
                                  constants.TIMEOUT_TIME_MILLIS * 5)
                self._background.set_text(self.ERROR)
            else:
                self._background.set_text(self.NO_RESULTS)

    def _show_banner(self, message, timeout=constants.TIMEOUT_TIME_MILLIS):
        banner = hildon.hildon_banner_show_information_with_markup(self,
                                                                   'ignored',
                                                                   message)
        banner.set_timeout(timeout)

    def _get_search_message(self, search_term, search_category):
        message = ('Searching on <i>%(category)s</i> for <i>%(term)s</i>' %
                   {'category': SearchDialog.search_fields[search_category],
                    'term' : search_term})
        return message


class MoviesView(gtk.TreeView):

    def __init__(self):
        super(MoviesView, self).__init__()
        model = MoviesListStore()
        self.set_model(model)

        movie_image_renderer = gtk.CellRendererPixbuf()
        column = gtk.TreeViewColumn('Image', movie_image_renderer,
                                    pixbuf=model.IMAGE_COLUMN)
        self.append_column(column)

        movie_text_renderer = gtk.CellRendererText()
        movie_text_renderer.set_property('ellipsize', pango.ELLIPSIZE_END)
        column = gtk.TreeViewColumn('Name', movie_text_renderer,
                                    markup=model.INFO_COLUMN)
        self.append_column(column)

        self.show_all()

    def add_movies(self, movie_list):
        model = self.get_model()
        if model:
            model.add(movie_list)

    def get_movie_from_path(self, path):
        model = self.get_model()
        return model[path][model.MOVIE_COLUMN]

class MoviesListStore(gtk.ListStore):

    IMAGE_COLUMN = 0
    INFO_COLUMN = 1
    MOVIE_COLUMN = 2

    def __init__(self):
        super(MoviesListStore, self).__init__(gtk.gdk.Pixbuf,
                                              str,
                                              gobject.TYPE_PYOBJECT)

    def add(self, movies_found):
        self.clear()
        for movie in movies_found:
            row = {self.IMAGE_COLUMN: movie.get_placeholder_image(),
                   self.INFO_COLUMN: movie.get_info(),
                   self.MOVIE_COLUMN: movie,
                  }
            self.append(row.values())

class AboutDialog(gtk.Dialog):

    PADDING = 5

    def __init__(self, parent):
        super(AboutDialog, self).__init__(parent=parent,
                                          flags=gtk.DIALOG_DESTROY_WITH_PARENT)
        self._logo = gtk.Image()
        self._name = ''
        self._name_label = gtk.Label()
        self._version = ''
        self._comments_label = gtk.Label()
        self._copyright_label = gtk.Label()
        self._license_label = gtk.Label()
        _license_alignment = gtk.Alignment(0, 0, 0, 1)
        _license_alignment.add(self._license_label)
        self._license_label.set_line_wrap(True)

        self._writers_caption = gtk.Label()
        self._writers_caption.set_markup('<b>%s</b>' % 'Authors:')
        _writers_caption = gtk.Alignment()
        _writers_caption.add(self._writers_caption)
        self._writers_label = gtk.Label()
        self._writers_contents = gtk.VBox(False, 0)
        self._writers_contents.pack_start(_writers_caption)
        _writers_alignment = gtk.Alignment(0.2, 0, 0, 1)
        _writers_alignment.add(self._writers_label)
        self._writers_contents.pack_start(_writers_alignment)

        _contents = gtk.VBox(False, 0)
        _contents.pack_start(self._logo, False, False, self.PADDING)
        _contents.pack_start(self._name_label, False, False, self.PADDING)
        _contents.pack_start(self._comments_label, False, False, self.PADDING)
        _contents.pack_start(self._copyright_label, False, False, self.PADDING)
        _contents.pack_start(self._writers_contents, False, False, self.PADDING)
        _contents.pack_start(_license_alignment, False, False, self.PADDING)

        _contents_area = hildon.PannableArea()
        _contents_area.add_with_viewport(_contents)
        _contents_area.set_size_request_policy(hildon.SIZE_REQUEST_CHILDREN)
        self.vbox.add(_contents_area)
        self.vbox.show_all()
        self._writers_contents.hide()

    def set_logo(self, logo_path):
        self._logo.set_from_file(logo_path)

    def set_name(self, name):
        self._name = name
        self.set_version(self._version)
        self.set_title('About %s' % self._name)

    def _set_name_label(self, name):
        self._name_label.set_markup('<big>%s</big>' % name)

    def set_version(self, version):
        self._version = version
        self._set_name_label('%s %s' % (self._name, self._version))

    def set_comments(self, comments):
        self._comments_label.set_text(comments)

    def set_copyright(self, copyright):
        self._copyright_label.set_markup('<small>%s</small>' % copyright)

    def set_license(self, license):
        self._license_label.set_markup('<b>%s</b>\n<small>%s</small>' % \
                                       ('License:', license))

    def set_authors(self, authors_list):
        authors = '\n'.join(authors_list)
        self._writers_label.set_text(authors)
        self._writers_contents.show_all()

class MovieWindow(hildon.StackableWindow):

    ACTION_TOGGLE_FAVORITE = 0

    NON_FAVORITE_LABEL = 'Mark as favorite'
    FAVORITE_LABEL = 'Unmark as favorite'

    def _fetch_movie_image(self, movie):
        image = gtk.Image()
        image.set_from_pixbuf(gtk.IconTheme().load_icon('general_video',
                                                        256, 0))
        movie_image = movie.get_image('poster', 'mid')
        if isinstance(movie_image, TmdbMovieImage):
            image_file = os.path.abspath(os.path.join(constants.MVS_CACHE, movie_image.get_id() + '.jpg'))
            if os.path.isfile(image_file):
                image.set_from_pixbuf(gtk.gdk.pixbuf_new_from_file_at_size(image_file,
                                                                       256,
                                                                       256))
            else:
                banner = hildon.hildon_banner_show_information_with_markup(self,
                                                                           'ignored',
                                                                           'Fetching movie poster')
                banner.set_timeout(constants.TIMEOUT_TIME_MILLIS)
                hildon.hildon_gtk_window_set_progress_indicator(self, True)

                async_item = AsyncItem(image_downloader, (movie_image.get_url(),
                                                          os.path.join(constants.MVS_CACHE,
                                                                       movie_image.get_id())),
                                       self._set_fetched_image, (image,))
                self.async_worker.queue.put(async_item)
                self.async_worker.start()

        return image

    def _set_fetched_image(self, image, target, error):
        if not error:
            image_file = os.path.abspath(target)
            image.set_from_pixbuf(gtk.gdk.pixbuf_new_from_file_at_size(image_file,
                                                                       256,
                                                                       256))
        hildon.hildon_gtk_window_set_progress_indicator(self, False)

    def _create_contents(self, movie):
        main_area = hildon.PannableArea()

        main_box = gtk.VBox(False, 20)
        main_box.set_border_width(20)
        upper_content = gtk.HBox(False, 40)
        upper_content.set_border_width(20)

        image = self._fetch_movie_image(movie)

        side_content = gtk.VBox(False, 30)

        for key in movie.fields:
            label = gtk.Label()
            label.set_markup('<b>%(field)s:</b> <small>%(value)s</small>' %
                             {'field' : key,
                              'value' : saxutils.escape(movie.get_value(key))})
            label.set_alignment(constants.LEFT_ALIGNMENT,
                                constants.CENTER_ALIGNMENT)
            side_content.pack_start(label, False, False)

        upper_content.pack_start(image, False, False)
        upper_content.pack_start(side_content, False, False)

        movie_overview = hildon.TextView()
        movie_overview.set_placeholder('Overview')
        movie_overview.set_wrap_mode(gtk.WRAP_WORD)
        movie_overview.get_buffer().set_text(saxutils.escape(movie.get_overview()))

        label = gtk.Label()
        label.set_markup('<b>Overview:</b>')
        label.set_alignment(constants.LEFT_ALIGNMENT,
                            constants.CENTER_ALIGNMENT)

        main_box.pack_start(upper_content, False, False)
        main_box.pack_start(label, False, False)
        main_box.pack_start(movie_overview, False, False)

        main_area.add_with_viewport(main_box)
        main_area.set_size_request_policy(hildon.SIZE_REQUEST_CHILDREN)

        return main_area

    def _create_app_menu(self):
        menu = hildon.AppMenu()

        toggle_favorite = hildon.GtkButton(gtk.HILDON_SIZE_AUTO)
        if self._movie.get_favorite():
            toggle_favorite.set_label(self.FAVORITE_LABEL)
        else:
            toggle_favorite.set_label(self.NON_FAVORITE_LABEL)
        toggle_favorite.connect('clicked',
                                self._button_clicked,
                                self.ACTION_TOGGLE_FAVORITE)

        menu.append(toggle_favorite)

        menu.show_all()

        return menu

    def _button_clicked(self, button, action):
        if action == self.ACTION_TOGGLE_FAVORITE:
            mark_favorite = button.get_label() == self.NON_FAVORITE_LABEL
            if mark_favorite:
                button.set_label(self.FAVORITE_LABEL)
            else:
                button.set_label(self.NON_FAVORITE_LABEL)
            moviemanager.update_favorite(self._movie, mark_favorite)

    def __init__(self, movie):
        self._movie = movie
        super(MovieWindow, self).__init__()
        self.async_worker = AsyncWorker.get_instance()
        self.set_title('Movie info')
        self.add(self._create_contents(movie))
        self.set_app_menu(self._create_app_menu())
        self.show_all()

class WebkitWindow(hildon.StackableWindow):

    def __init__(self):
        super(WebkitWindow, self).__init__()

        self.view = WebView()
        self.view.connect('load-started', self._load_start)
        self.view.connect('load-finished', self._load_finished)
        self.view.set_full_content_zoom(True)

        wbk_settings = self.view.get_settings()
        wbk_settings.set_property('auto-shrink-images', True)

        self.pannable = hildon.PannableArea()
        self.pannable.add(self.view)
        self.pannable.set_size_request_policy(hildon.SIZE_REQUEST_CHILDREN)
        self.pannable.set_property("mov-mode", hildon.MOVEMENT_MODE_BOTH)

        self.add(self.pannable)
        self.show_all()

    def _load_start(self, view, data):
        hildon.hildon_gtk_window_set_progress_indicator(self, True)

    def _load_finished(self, view, data):
        hildon.hildon_gtk_window_set_progress_indicator(self, False)

    def load_url(self, url):
        self.view.open(url)

class TheatersWindow(WebkitWindow):

    def __init__(self):
        super(TheatersWindow, self).__init__()
        self.show_all()

    def display_shows(self):
        self.load_url('http://www.google.com/movies')

class WatcWindow(WebkitWindow):

    def __init__(self, movie):
        super(WatcWindow, self).__init__()
        self.show_all()
        has_stingers = movie.get_stingers()
        if has_stingers == 0:
            stingers = '?'
        elif has_stingers == 1:
            stingers = '*'
        else:
            stingers = ''
        url = ('http://whatsafterthecredits.com/index.php?title=%(title)s_(%(year)s)%(stingers)s' %
                      {'title':movie.get_title().strip(),
                       'year':movie.get_year().strip(),
                       'stingers':stingers})
        print url
        self.load_url(url)

if __name__ == '__main__':
    maevies = Maevies()
    maevies.run()
