#include "mainwindow.h"
#include "ui_mainwindow.h"

MainWindow::MainWindow(QWidget *parent) :
    QMainWindow(parent),
    ui(new Ui::MainWindow),
    sound_(NULL),
    sound2_(NULL),
    channel_(1),
    bpm_(120),
    timesiq1_(4),
    timesiq2_(4),
    started_(false),
    counter_(0)
{
#ifdef Q_WS_MAEMO_5
    QDBusConnection::systemBus().connect(QString(), MCE_SIGNAL_PATH, MCE_SIGNAL_IF,
                                         MCE_DEVICE_ORIENTATION_SIG,
                                         this,
                                         SLOT(orientationChanged(QString)));
#endif

    ui->setupUi(this);

    timer_ = new QTimer(this);

    // Time signatures...
    for (int i = 1 ; i < 17 ; ++i) {
        ui->comboTS1->addItem(QString::number(i,10));
        if ( i == 1 || i == 2 || i == 4 || i == 8 ) { //|| i == 16 ) {
            ui->comboTS2->addItem(QString::number(i,10));
        }
    }

    ui->comboTS1->setCurrentIndex(3);
    ui->comboTS2->setCurrentIndex(2);

    // Load sounds
    sound_ = Mix_LoadWAV("/opt/maetronome/click.wav");
    if(sound_ == NULL) {
        printf("Unable to load WAV file: %s\n", Mix_GetError());
    }
    sound2_ = Mix_LoadWAV("/opt/maetronome/blip.wav");
    if(sound_ == NULL) {
        printf("Unable to load WAV file: %s\n", Mix_GetError());
    }

    // Connects
    connect(ui->action_Quit, SIGNAL(triggered()), this, SLOT(close()));
    connect(ui->horizontalSlider, SIGNAL(valueChanged(int)), this, SLOT(onSliderChanged(int)));
    connect(ui->pushButton, SIGNAL(clicked()), this, SLOT(onBtnClick()));
    connect(timer_, SIGNAL(timeout()), this, SLOT(onTimeout()));
    connect(ui->comboTS1, SIGNAL(currentIndexChanged(int)), this, SLOT(onTimeSigChange(int)));
    connect(ui->comboTS2, SIGNAL(currentIndexChanged(int)), this, SLOT(onTimeSigChange(int)));

    ui->horizontalSlider->setValue(120);

}

MainWindow::~MainWindow()
{
    while(Mix_Playing(channel_) != 0);
    Mix_FreeChunk(sound_);
    Mix_FreeChunk(sound2_);
    Mix_CloseAudio();
    SDL_Quit();
    delete ui;
}

void MainWindow::changeEvent(QEvent *e)
{
    QMainWindow::changeEvent(e);
    switch (e->type()) {
    case QEvent::LanguageChange:
        ui->retranslateUi(this);
        break;
    default:
        break;
    }
}

void MainWindow::onBtnClick() {
    if ( started_ ) {
        counter_ = 0;
        ui->pushButton->setText("Start");
        started_ = false;
        timer_->stop();
    }
    else {
        ui->pushButton->setText("Stop");

        int interval = (60000 / bpm_) / (timesiq2_ / 4.0);
        timer_->start(interval);
        started_ = true;
    }
}

void MainWindow::onTimeout() {
    //Play our sound file, and capture the channel on which it is played
    Mix_HaltChannel(1);
    if (counter_)
      channel_ = Mix_PlayChannel(1, sound_, 0);
    else channel_ = Mix_PlayChannel(1, sound2_, 0);
    if(channel_ == -1) {
        printf("Unable to play WAV file: %s\n", Mix_GetError());
    }

    counter_ = (counter_ + 1) % timesiq1_;
}

// BPM Changed
void MainWindow::onSliderChanged(int value) {
    ui->lcdNumber->display(value);
    bpm_ = ui->lcdNumber->intValue();

    if ( started_ ) {
        int interval = (60000 / bpm_) / (timesiq2_ / 4.0);
        timer_->stop();
        timer_->start(interval);
    }
}

// Time signature changed
void MainWindow::onTimeSigChange(int i) {
    bool bOk1, bOk2;
    int ts1 = ui->comboTS1->currentText().toInt(&bOk1, 10);
    int ts2 = ui->comboTS2->currentText().toInt(&bOk2, 10);
    if ( !ts1 || !ts2 ) {
        return;
    }

    if (started_ && ts2 != timesiq2_ ) {
        int interval = (60000 / bpm_) / (ts2 / 4.0);
        timer_->stop();
        timer_->start(interval);
    }

    timesiq1_ = ts1;
    timesiq2_ = ts2;
    counter_ = 0;
}


#ifdef Q_WS_MAEMO_5
// Portrait/landscape for maemo5
void MainWindow::setLandscape() {
    setAttribute(Qt::WA_Maemo5ForceLandscapeOrientation, true);
}

void MainWindow::setPortrait() {
    setAttribute(Qt::WA_Maemo5ForcePortraitOrientation, true);
}

void MainWindow::orientationChanged(const QString &newOrientation)
{
    if (newOrientation == QLatin1String(MCE_ORIENTATION_PORTRAIT))
        setPortrait();
    else
        setLandscape();
}

bool MainWindow::event(QEvent *ev) {
    switch (ev->type()) {
    case QEvent::WindowActivate:
        QDBusConnection::systemBus().call(
                QDBusMessage::createMethodCall(MCE_SERVICE, MCE_REQUEST_PATH,
                                               MCE_REQUEST_IF,
                                               MCE_ACCELEROMETER_ENABLE_REQ));
        break;
    case QEvent::WindowDeactivate:
        QDBusConnection::systemBus().call(
                QDBusMessage::createMethodCall(MCE_SERVICE, MCE_REQUEST_PATH,
                                               MCE_REQUEST_IF,
                                               MCE_ACCELEROMETER_DISABLE_REQ));
        break;
    default:
        break;
    }

    return QWidget::event(ev);
}
#endif
