/*
* This file is part of MaemoSweeper
*
* Copyright (C) 2005 INdT - Instituto Nokia de Tecnologia
* http://www.indt.org/maemo
*
* This software is free software; you can redistribute it and/or
* modify it under the terms of the GNU Lesser General Public License
* as published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful, but
* WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free Software
* Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA
*
*/ 

#include <stdlib.h>
#include <string.h>

#include "images.h"
#include "globals.h"

// control dimensions
int xPos,yPos, wCtl, hCtl;

// the grid/ minefield dimensions
int sizeX;
int sizeY;

//curent processed position
int indexX, indexY;

unsigned char minefield[DIMENSION];
unsigned char cellstates[DIMENSION];

SDL_Surface *load_image(char *filename, int transparent);
void draw_board(void);
void draw_counter(int marked, int difficulty_offset);

//surfaces
SDL_Surface *background;
SDL_Surface *openfield;
SDL_Surface *setflag;
SDL_Surface *count_number;
SDL_Surface *mine;
SDL_Surface *message_won;
SDL_Surface *message_loose;

/**
***************************************************************
* @brief Checks if the game ended
* 
* This function tests if all the marked cells are hidden mines
* and if all the hidden mines are marked
* 
* @return 			1 if the game ended
* 					0 if not
***************************************************************/
int check_end_game(void){
    int isMineMarked = 0;
    int i,j;

    // for each position
    for (i = 0; i < COLS; i++) {
        for (j = 0; j < ROWS; j++) {
            if ((int)cellstates[i+j*COLS] == MARKED_CELL && 
                    ((int)minefield[(i)+(j)*COLS] == IS_BOMB)) {
                    isMineMarked++;
            }
        }
    }
    if (isMineMarked == marked) {
        return true;    
    }
    else {
        return false;
    }
}

/**
***************************************************************
* @brief Load image in video memory
* 
* This function receives a filename and creates a surface
* which can be transparent or not
* 
* @return 			SDL_Surface if the image was loaded
* 					0 if not
***************************************************************/
SDL_Surface *load_image(char *filename, int transparent) {
    char filepath[256];
    SDL_Surface *img, *img2;
    
    // Load file image
    sprintf(filepath, "%s%s", PIXMAPSPREFIX, filename);
    img = SDL_LoadBMP(filepath);
    if (!img) {
        fprintf(stderr, "File not found: %s\n", filepath);
        return 0;
    }
    
    // Create hardware surface
    img2 = SDL_CreateRGBSurface(SDL_HWSURFACE | (transparent ? SDL_SRCCOLORKEY :
        0), img->w, img->h, 16, 0xF800, 0x7E0, 0x1F, 0);
    if (!img2) {
        SDL_FreeSurface(img);
        fprintf(stderr, "Error creating surface!\n");
        return 0;
    }
    
    // Set color key
    if (transparent) {
        SDL_SetColorKey(img2, SDL_SRCCOLORKEY, 0xF81F);
    }
    SDL_SetAlpha(img2, 0, 0);
    
    // Copy surface
    SDL_BlitSurface(img, NULL, img2, NULL);
    SDL_FreeSurface(img);
    
    return img2;    
}

/**
**********************************************************
* @brief Load surfaces
* 
* This function loads all the game surfaces
* 
* @return 			1 if the images are loaded
* 					0 if some problem occurs
**********************************************************/
int load_surfaces() {

    char filename[128];

    // Load background
	sprintf(filename, "board.bmp");
	background = load_image(filename, false);
	if (!background) {
		return false;
	}

    // Load button open field
	sprintf(filename, "bt_openfield.bmp");
	openfield = load_image(filename, false);
	if (!openfield) {
		return false;
	}

    // Load button set flag
	sprintf(filename, "bt_setflag.bmp");
	setflag = load_image(filename, false);
	if (!setflag) {
		return false;
	}
	// Load count numbers
    sprintf(filename, "numbers.bmp");
    count_number = load_image(filename, false);
	if (!count_number) {
		return false;
	}

    // Load mine fields
    sprintf(filename, "main.bmp");
	mine = load_image(filename, false);
	if (!mine) {
		return false;
	}
       	
    // Load win message
    sprintf(filename, "DIALOG_WON.bmp");
	message_won = load_image(filename, false);
	if (!message_won) {
		return false;
	}

    // Load loose message
    sprintf(filename, "DIALOG_LOOSE.bmp");
	message_loose = load_image(filename, false);
	if (!message_loose) {
		return false;
	}
 
    return true;
}


/**
**********************************************************
* @brief Free surfaces
* 
* This function frees all the game surfaces
* 
**********************************************************/
void free_surfaces() {

	SDL_FreeSurface(count_number);
	SDL_FreeSurface(background);
	SDL_FreeSurface(openfield);
	SDL_FreeSurface(setflag);
	SDL_FreeSurface(mine);
	SDL_FreeSurface(message_won);
	SDL_FreeSurface(message_loose);
}


/**
**********************************************************************
* @brief Draw background
* 
* This function draws all the images
* @param marked             number of the mines marked
* @param click_type         if the user is marking or searching mines
* @param difficulty_offset  difficulty choosen by the user 
**********************************************************************/
void draw_background(int marked, int click_type, int difficulty_offset) {

    SDL_Rect dest;
    dest.y = dest.x = 0;
    dest.h = SCREEN_BOUNDARYY-1;
    dest.w = SCREEN_BOUNDARYX-1;
    
    SDL_BlitSurface(background, NULL, screen, &dest);

    draw_board();
    draw_counter(marked, difficulty_offset);

    if (gState == GAMEOVER_STATE) {
        show_dialog(false);
    }
    if (gState == CLEAREDGAME_STATE) {
        show_dialog(true);
    }

    if (click_type == CLICK_MARK) {
        dest.y = CURSORFLAG_Y;
        dest.x = CURSORFLAG_X;
        dest.h = CURSORFLAG_OFFSETY + CURSORFLAG_Y;
        dest.w = CURSORFLAG_OFFSETX + CURSORFLAG_X;
        SDL_BlitSurface(setflag, NULL, screen, &dest);
    }
    else if (click_type == CLICK_SEARCH) {
        dest.y = CURSOROPEN_Y;
        dest.x = CURSOROPEN_X;
        dest.h = CURSOROPEN_OFFSETY + CURSOROPEN_Y;
        dest.w = CURSOROPEN_OFFSETX + CURSOROPEN_X;
        SDL_BlitSurface(openfield, NULL, screen, &dest);
    }
    SDL_UpdateRect(screen, 0,0,0,0);
}

/**
**********************************************************
* @brief Draws the mines field
* 
* This function draws the mines field
*
* @param difficulty_offset  difficulty choosen by the user 
**********************************************************/
void draw_board(){
    
    int i, j, index;
    SDL_Rect dest2, src;

    src.x = 0;
    src.w = src.h = MINE_SIZE;
    
    // Draw mines
      dest2.w = dest2.h = MINE_SIZE;
      dest2.x = BOARD_OFFSETX;
      dest2.y = BOARD_OFFSETY;

    for (j = 0; j < ROWS; j++) {
        for (i = 0; i < COLS; i++) {
           switch((int)cellstates[i+j*COLS]) {
                case UNREVEALED_CELL:
                    src.y = UNREVEALED_CELL * MINE_SIZE;
                    SDL_BlitSurface(mine, &src, screen, &dest2);
                    break;
                case MARKED_CELL:
                    src.y = MARKED_CELL * MINE_SIZE;
                    SDL_BlitSurface(mine, &src, screen, &dest2);
                    break;                    
                case QUESTIONMARKED_CELL:
                    src.y = QUESTIONMARKED_CELL * MINE_SIZE;
                    SDL_BlitSurface(mine, &src, screen, &dest2);
                    break;
                case REVEALEDBOMB_CELL:
                    src.y = REVEALEDBOMB_CELL * MINE_SIZE;
                    SDL_BlitSurface(mine, &src, screen, &dest2);
                    break;
                case NOTBOMBMARKED_CELL:
                    src.y = NOTBOMBMARKED_CELL * MINE_SIZE;
                    SDL_BlitSurface(mine, &src, screen, &dest2);
                    break;
                case HIDDENBOMB_CELL:
                    src.y = HIDDENBOMB_CELL * MINE_SIZE;
                    SDL_BlitSurface(mine, &src, screen, &dest2);
                    break;
                case QUESTIONMARKEDDOWN_CELL:
                    src.y = QUESTIONMARKEDDOWN_CELL * MINE_SIZE;
                    SDL_BlitSurface(mine, &src, screen, &dest2);
                    break;                    
                case UNREVEALEDDOWN_CELL:
                    src.y = UNREVEALEDDOWN_CELL * MINE_SIZE;
                    SDL_BlitSurface(mine, &src, screen, &dest2);
                    break;
                case REVEALED_CELL:
                    index = (int) minefield[i+j*COLS];
                    if (index == 10){
                        src.y = REVEALEDBOMB_CELL * MINE_SIZE;
                        SDL_BlitSurface(mine, &src, screen, &dest2);
                    }
                    else {
                        src.y = 400 - (index * MINE_SIZE);
                        SDL_BlitSurface(mine, &src, screen, &dest2);
                    }
                    break;                    
            }
            // 1 is the distance between adjacent cells
            dest2.x += MINE_SIZE + 1;
        }
        dest2.x = BOARD_OFFSETX;
        dest2.y += MINE_SIZE + 1; 
    }
}

/**
**********************************************************
* @brief Draws the marks counter
* 
* This function draws the counter
*
* @param marked             number of the mines marked
* @param difficulty_offset  difficulty choosen by the user 
**********************************************************/
void draw_counter(int marked, int difficulty_offset){
    
    int digits[3] = {0,0,0};
    SDL_Rect src, dest;
    int undiscovered_bombs = (BOMBS_NUMBER-4*difficulty_offset)-marked;
    int i;
    
    
    // put the most significant to the right
    digits[0] = ((undiscovered_bombs%1000) / 100);
    // then the middle digit
    digits[1] = ((undiscovered_bombs%100) / 10);
    // finaly the last one
    digits[2] = (undiscovered_bombs%10);
   
    dest.y = COUNTER_OFFSETY;
    dest.h = COUNTER_OFFSETY + COUNT_NUMBER_SIZEY;

    dest.x = COUNTER_OFFSETX;
    dest.w = COUNTER_OFFSETX + COUNT_NUMBER_SIZEX;
    
    src.x = 0;
    src.w = COUNT_NUMBER_SIZEX;

    for (i =0; i<3; i++){
        src.y = digits[i]*COUNT_NUMBER_SIZEY;
        src.h = COUNT_NUMBER_SIZEY;

        SDL_BlitSurface(count_number, &src, screen, &dest);

        dest.x += COUNT_NUMBER_SIZEX;
        dest.w += COUNT_NUMBER_SIZEX;
    }
    SDL_UpdateRect(screen, COUNTER_OFFSETX, COUNTER_OFFSETY,dest.w, dest.h);
    
}

/**
**********************************************************
* @brief Initializes grid
* 
* This function restarts the game arrays and puts the game
* in wait state
* 
**********************************************************/ 
void initialize_grid()
{
    
    memset(minefield, 0, DIMENSION);
    memset(cellstates, 0, DIMENSION);
    
    gState = WAIT_STATE;
    indexX = indexY = -2;
}

/**
**********************************************************
* @brief Spread bombs in the minefield 
* 
* Generate the required number of bombs and spread them 
*
* @param             bombs               number of bombs
* @param             difficulty_offset   level of game
* 
**********************************************************/ 
void spread_mines(int bombs, int difficulty_offset)
{
    int i;
    int bombs_number = bombs-4*difficulty_offset;
    // update the game state to wait for user
    gState = WAIT_STATE;
    memset(minefield, 0, DIMENSION);
    if (get_cellstate(indexX, indexY)>= 0) 
    {
        // keep thet position untouched
        minefield[indexX+indexY*COLS] = 20;
    }
    // generate all the bombs
    for (i = 0; i < bombs_number; i++)
    {
        int minex = 0;
        int miney = 0;
        // try to generate new positions until empty one is reached
        do {
            miney = rand()%ROWS;
            minex = rand()%COLS;
        } while ((int)minefield[minex+miney*COLS] != 0);
        // mark as 'bomb'
        minefield[minex+miney*COLS] = IS_BOMB;
    } 
    if (get_cellstate(indexX, indexY)>= 0) 
        minefield[indexX+indexY*COLS] = 0;
}

/**
**********************************************************
* @brief Finalize the game 
* 
* Show the user the not revealed bombs 
*
* 
**********************************************************/ 
void gameOver()
{   
    int i,j, cellstate;
    gState = GAMEOVER_STATE;
    for (j=0; j < ROWS; j++) {
        for (i=0; i < COLS; i++) {
            cellstate = (int)minefield[i+j*COLS];
            if (cellstates[i+j*COLS] == UNREVEALED_CELL && cellstate != IS_BOMB) {
            }
            if (cellstates[i+j*COLS] == MARKED_CELL)
            {
                if (cellstate!= IS_BOMB) {
                    cellstates[i+j*COLS] = NOTBOMBMARKED_CELL;
                }
            }
            if (cellstates[i+j*COLS] == UNREVEALED_CELL && cellstate == IS_BOMB){
                cellstates[i+j*COLS] = HIDDENBOMB_CELL;
            }
        }
    }
}

/**
**********************************************************
* @brief Shows won/loose dialog
* 
* This function shows a message when the game ends
*
* @param won      1 if the user won
*                 0 if not
**********************************************************/
void show_dialog(int won) {

    SDL_Rect dest;

    dest.y = SCREEN_BOUNDARYY/2 - DIALOG_SIZEY/2;
    dest.h = DIALOG_SIZEY;

    dest.x = SCREEN_BOUNDARYX/2 - DIALOG_SIZEX/2;
    dest.w = DIALOG_SIZEX;

    if (won){
        SDL_BlitSurface(message_won, NULL, screen, &dest);
    }
    else {
        SDL_BlitSurface(message_loose, NULL, screen, &dest);
    }
    SDL_UpdateRect(screen, dest.x,dest.y, dest.w, dest.h);
    
}


/**
**********************************************************
* @brief Fill cells with the number of mines around it
* 
* This function finds the number of the mines 
* around each minefield position and fills the cell 
* with this number, if it is not a bomb
*
**********************************************************/
void fill_minenumber()
{
    int i,j;
    for (i = 0; i < COLS; i++) {
        for (j = 0; j < ROWS; j++) {
            minefield[i+j*COLS] = count_minesaround(i,j);
        }
    }
}

/**
**********************************************************
* @brief Calculates the number of mines around a cell
* 
* This function finds the number of the mines 
* around a given minefield position
*
* @param       x    the x on position 
* @param       y    the y on position 
*
* @return      the number of mines
*
**********************************************************/
int count_minesaround(int x, int y)
{
    // if it is a mine return 10
    if (minefield[x+y*COLS] == 10)
        return 10;
 
    int i, j, count = 0;
    for (i = x-1; i<=x+1; i++){
        for (j = y-1; j <= y+1; j++) {
            if ((i >= 0 && j >= 0) 
                && (i < COLS && j < ROWS)
                && (x!=i || y != j)) {
                    count += ((int)minefield[i+j*COLS] == 10);
            }
        }
    }
    return count;
}

/**
**********************************************************
* @brief Calculates the number of marked cells around a cell
* 
* This function finds the number of the marked cells 
* around a given minefield position
*
* @param       x    the x on position 
* @param       y    the y on position 
*
* @return      the number of marked cells
*
**********************************************************/
int count_markedcells(int x, int y)
{
    int i, j, count = 0;
    for (i = x-1; i<=x+1; i++){
        for (j = y-1; j <= y+1; j++) {
            if ((i >= 0 && j >= 0) 
                && (i < COLS && j < ROWS)
                && (i!=x || j != y)) {
                    count += (int)(get_cellstate(i, j) == MARKED_CELL);
            }
        }
    }
    return count;
}

/**
**********************************************************
* @brief Reveals cells around a position
* 
* This function reveals the content of the cells 
* around a given minefield position
*
* @param       x    the x on position 
* @param       y    the y on position 
*
**********************************************************/
void show_cells_around(int x, int y)
{
    if (x > 0 && y > 0) {
        show_cellcontent(x-1,y-1);
    }
    if (x > 0) {
        show_cellcontent(x-1,y);
    }    
    if (x > 0 && y < (ROWS-1)) {
        show_cellcontent(x-1, y+1);
    }    
    if (y > 0) {
        show_cellcontent(x,y-1);
    }    
    if (y < (ROWS-1)) {
        show_cellcontent(x,y+1);
    }    
    if (x <(COLS-1) && y > 0) {
        show_cellcontent(x+1,y-1);
    }    
    if (x <(COLS-1)) {
        show_cellcontent(x+1,y);
    }    
    if (x <(COLS-1) && y < (ROWS-1)) {
        show_cellcontent(x+1, y+1);
    }
}


/**
**********************************************************
* @brief Change the visual state of the cell
* 
* This function changes the visual of the minefield position
*
* @param       i    the x on position 
* @param       j    the y on position 
*
**********************************************************/
void set_cellstate(int i, int j)
{
    int x, y;
    for (x = i-1; x<=i+1; x++){
        for (y = j-1; y <= j+1; y++) {
            if ((x >= 0 && y >= 0) 
                && (x < COLS && y < ROWS)
                && (x!=i || y != j)) {
                switch (get_cellstate(x,y))
                {
                    case UNREVEALED_CELL:
                        cellstates[x+y*COLS] = UNREVEALEDDOWN_CELL;
                        break;
                    case UNREVEALEDDOWN_CELL:
                        cellstates[x+y*COLS] = UNREVEALED_CELL;
                        break;
                    case QUESTIONMARKED_CELL:
                        cellstates[x+y*COLS] = QUESTIONMARKEDDOWN_CELL;
                        break;
                    case QUESTIONMARKEDDOWN_CELL:
                        cellstates[x+y*COLS] = QUESTIONMARKED_CELL;
                        break;
                }
            }
        }
    }
}

/**
**********************************************************
* @brief Return the visual state of the cell
* 
* This function returns the visual of the minefield position
*
* @param       i    the x on position 
* @param       j    the y on position 
*
* @return      the visual state
*
**********************************************************/
int get_cellstate(int i, int j)
{
    
    if (i < 0 || j < 0 || i >= COLS || j >= ROWS)
        return -1;
    
    switch ((int)cellstates[i+j*COLS])
    {
        case UNREVEALED_CELL:
            return UNREVEALED_CELL;
        case UNREVEALEDDOWN_CELL:
            return UNREVEALEDDOWN_CELL;
        case QUESTIONMARKED_CELL:
            return QUESTIONMARKED_CELL;
        case QUESTIONMARKEDDOWN_CELL:
            return QUESTIONMARKEDDOWN_CELL;
        case MARKED_CELL:
            return MARKED_CELL;
    }
    return REVEALED_CELL;
}


/**
**********************************************************
* @brief Set the visual state of a not revealed cell
* 
* This function returns the visual of the minefield position
*
* @param       i     the x on position 
* @param       j     the y on position 
* @param       bombs the total number of bombs
*
**********************************************************/
void set_flag(int x, int y, int bombs)
{
    switch (get_cellstate(x, y))
    {
    case UNREVEALED_CELL:
        // if it is not revealed, change for marked if 
        //the number of marked cell is less than the numbre of bombs
        if (marked < bombs) {
            cellstates[x+y*COLS] = MARKED_CELL;
            marked++;
        }
        break;
    case MARKED_CELL:
        // if the cell is marked, change for question marked
        cellstates[x+y*COLS] = QUESTIONMARKED_CELL;
        marked--;
        break;
    case QUESTIONMARKED_CELL:
        // if the cell is question marked, change for not revealed
        cellstates[x+y*COLS] = UNREVEALED_CELL;
        break;
    } 
} 


/**
**********************************************************
* @brief Show the content of a cell, reveal it
* 
* This function shows the content of a cell.
* If the cell is a bomb, game is over.
* If the cell is white, show the cells around
*
* @param       x     the x on position 
* @param       y     the y on position 
*
* @return      0 if is not gameover
*              1 if is gameover
*
**********************************************************/
int show_cellcontent(int x, int y)
{
    // check the localtion to be valid
    if (x<0 || x >= COLS || y < 0 || y >= ROWS)
        return 0;
    // revealed or marked so leave as it is
    if ((cellstates[x+y*COLS] == REVEALED_CELL) ||
        (cellstates[x+y*COLS] == MARKED_CELL)||
        (cellstates[x+y*COLS] == QUESTIONMARKED_CELL))
        return 0;
 
   // if game was over: exit
    if (gState == GAMEOVER_STATE)
        return 0;

    // reveal it
    cellstates[x+y*COLS] = REVEALED_CELL;

    // if it is a bomb, game is over
    if (minefield[x+y*COLS] == 10)
    {
        gameOver();
        return 1;
    }

    // if it was an empty cell then try to show cell content around it
    if (minefield[x+y*COLS] == 0){
        show_cells_around(x, y);
    }
    return 0;
}

/**
**********************************************************
* @brief Save GameState
* 
* This function saves the current game
* 
**********************************************************/
void statesave(int marking, int marked, int difficulty) 
{
    FILE *pFile;
    if (gState != GAMEOVER_STATE) {
        pFile = fopen("/tmp/.maemosweeper-save", "wb");
        if(pFile) {
            fwrite(&marking, sizeof(int), 1, pFile);
            fwrite(&marked, sizeof(int), 1, pFile);
            fwrite(&difficulty, sizeof(int), 1, pFile);
            fwrite(minefield, sizeof(char), DIMENSION, pFile);
            fwrite(cellstates, sizeof(char), DIMENSION, pFile);
            fclose(pFile);
        }
    }
}

/**
**********************************************************
* @brief Load GameState
* 
* This function loads the game saved
* 
**********************************************************/
int loadstatesaved(int *marked, int *difficulty) {

    FILE *pFile;
    int marking = 0;
    pFile = fopen("/tmp/.maemosweeper-save", "rb");
    if(pFile) {
        fread(&marking, sizeof(int), 1, pFile);
        fread(marked, sizeof(int), 1, pFile);
        fread(difficulty, sizeof(int), 1, pFile);
        fread(minefield, sizeof(char), DIMENSION, pFile);
        fread(cellstates, sizeof(char), DIMENSION, pFile);
        fclose(pFile);
        remove("/tmp/.maemosweeper-save");
    }    
    return marking;
}
