/*
Copyright (c) 2012, Petr Pehal, Jakub Šplíchal, Rostislav Stríž,

All rights reserved.

Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

    Redistributions of source code must retain the above copyright notice, this list of conditions and the following disclaimer.
    Redistributions in binary form must reproduce the above copyright notice, this list of conditions and the following disclaimer
    in the documentation and/or other materials provided with the distribution.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING,
BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT
SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/
/*************************************************************************************************
**
** file:    profilecontroller.h
** date:    10.12.2011
** authors: Petr Pehal,      xpehal00@stud.fit.vutbr.cz
**          Rostislav Striz, xstriz03@stud.fit.vutbr.cz
**          Jakub Splichal,  xsplic03@stud.fit.vutbr.cz
**
**************************************************************************************************/

#ifndef PROFILERCONTROLLER_H
#define PROFILERCONTROLLER_H

#include <QObject>
#include <QBasicTimer>
#include "profile.h"

/*************************************************************************************************/

class QmlApplicationViewer;
class ConfigManipulator;

/*************************************************************************************************/

#define Profiler	ProfilerController::instance()

/*************************************************************************************************/
// PROFILER CONTROLLER DATA
/*************************************************************************************************/

struct ProfilerControllerData
{
	QmlApplicationViewer *viewer;

	QObjectList          profiles;
	Profile              *currentProfile;

	QObjectList          actionPlugins;
	QObjectList          conditionPlugins;

	ConfigManipulator    *config;

	QBasicTimer          autosaveTimer;
	int                  autosaveInterval;
};

/*************************************************************************************************/
// PROFILER CONTROLLER DATA
/*************************************************************************************************/

class ProfilerController : public QObject
{
	Q_OBJECT

public:
	explicit ProfilerController(QObject *parent = 0);
	~ProfilerController();

	inline static ProfilerController *instance() { return self; }

	inline int autosafeInterval() const { return d->autosaveInterval; }
	void setAutosafeInterval(int ms);

	void timerEvent(QTimerEvent *event);

	bool loadPlugins();

	bool loadProfiles();
	bool saveProfiles();

	inline QmlApplicationViewer *qmlViewer() const { return d->viewer; }
	void registerQmlViewer(QmlApplicationViewer *viewer);

	inline QObjectList profiles() const { return d->profiles; }

	ConditionInterface *conditionPlugin(const QString &name);
	ActionInterface    *actionPlugin(const QString &name);

	// ------------- //
	// QML INTERFACE //
	// ------------- //

	// operace s pluginy
	Q_INVOKABLE void listConditionPlugins();
	Q_INVOKABLE void listActionPlugins();

	Q_INVOKABLE int conditionPluginCount();
	Q_INVOKABLE int actionPluginCount();

	Q_INVOKABLE QUrl actionPluginIcon(int actionPluginIndex);
	Q_INVOKABLE QUrl conditionPluginIcon(int conditionPluginIndex);

	// operace s profily
	Q_INVOKABLE void addProfile(const QString &name);
	Q_INVOKABLE void renameProfile(int profilesIndex, const QString &name);
	Q_INVOKABLE void removeProfile(int profilesIndex);
	Q_INVOKABLE void selectProfile(int profilesIndex);


	Q_INVOKABLE bool verifyProfileRename(const QString &name);

	Q_INVOKABLE QUrl currentProfileConditionPluginQmlGUI(int profileConditionIndex);
	Q_INVOKABLE QUrl currentProfileActionPluginQmlGUI(int profileActionIndex);

	Q_INVOKABLE QUrl conditionPluginQmlGUI(const QString &conditionName);
	Q_INVOKABLE QUrl actionPluginQmlGUI(const QString &actionName);

	Q_INVOKABLE void addCurrentProfileCondition(int pluginConditionIndex);
	Q_INVOKABLE void removeCurrentProfileCondition(int profileConditionIndex);

	Q_INVOKABLE void addCurrentProfileAction(int pluginActionIndex);
	Q_INVOKABLE void removeCurrentProfileAction(int profileActionIndex);

signals:

public slots:

private:
	void loadPluginsCore(const QString &directory);
	void updateAutosafeIntervalChanged();

	void setCurrentProfile(Profile *profile);
	void updateCurrentProfileChagned();

	inline Profile *profile(int index)
	{
		return static_cast<Profile *>(d->profiles.at(index));
	}

	inline ConditionInterface *conditionPlugin(int index)
	{
		return static_cast<ConditionInterface *>(d->conditionPlugins.at(index));
	}

	ActionInterface    *actionPlugin(int index)
	{
		return static_cast<ActionInterface *>(d->actionPlugins.at(index));
	}

	void updateQmlViewerChanged();

	void notifyQmlProfilesChanged();
	void notifyQmlCurrentProfileChanged();

	void initializeQmlPluginGUI(CoreInstance *instance);

private:
	ProfilerControllerData    *d;

	static ProfilerController *self;
};

/*************************************************************************************************/

#endif // PROFILERCONTROLLER_H
