/* File: lyricsDownloader.cpp
 *
 * This file is part of MaeLyrica.
 *
 * Copyright (C) 2012 Marcin Mielniczuk.
 *
 * MaeLyrica is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, version 3 of the License.
 *
 * MaeLyrica is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 *  along with MaeLyrica.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "lyricsDownloader.h"
#include <curl/curl.h>
#include <QDebug>
#include <iostream>
#include <QFile>
#include <QSettings>

using namespace std;

#define DEBUG 1

/////////////////////////////////////////////////////////////////////////////

const QString lyricsDownloader::OMPLyricsRoot("/home/user/.lyrics/");

Settings lyricsDownloader::progSettings;

inline void lyricsDownloader::setData(const string &a, const string &t)
{
    artist = a;
    track = t;
}

size_t lyricsDownloader::write_data_to_var(char *ptr, size_t size, size_t nmemb, void *userdata)
{
    ostringstream * stream = (ostringstream*) userdata;
    size_t count = size * nmemb;
    stream->write(ptr, count);
    return count;
}

QString lyricsDownloader::cleanQString(const QString qstr)
{
    return qstr.toLower().replace("&", "and").remove(QRegExp("\\([^)]*\\)")).remove(QRegExp("[\\W_]"));
}

void lyricsDownloader::cleanTitle()
{
    cleanedTitle = cleanQString( QString::fromStdString(artist));
    cleanedTitle += "-";
    cleanedTitle += cleanQString(QString::fromStdString(track));
}


bool lyricsDownloader::lyricsExistOMP()
{
    lyricsUrlOMP = OMPLyricsRoot + cleanedTitle + ".txt";
    return QFile(lyricsUrlOMP).exists();
}

void lyricsDownloader::saveLyricsMaeLyrica()
{
    QSettings("Marcin Mielniczuk", "MaeLyrica").setValue("lyrics/" + cleanedTitle, lyrics);
}

void lyricsDownloader::saveLyricsOMP()
{
    QFile lyricsFile(lyricsUrlOMP);
    QTextStream lyricsFileStream(&lyricsFile);
    lyricsFileStream << lyrics;
}

///////////////////////////////////////////////////////////////////////////////

const string AZLyricsDownloader::rooturl("http://azlyrics.com/lyrics");
const QRegExp AZLyricsDownloader::endcomment("<!-- end of lyrics -->");
const QRegExp AZLyricsDownloader::startcomment("<!-- start of lyrics -->");



short AZLyricsDownloader::perform()
{
    CURLcode dlret; // return value from curl
    dlret = download(); // it doesn't fail yet to recognize

    if ( parse() != 0)
    {
        if ( dlret != 0 ) // Error while downloading and parsing means that the lyrics are just not there
        {
            qDebug() << "The lyrics are not present in AZLyrics.com database\n";
            return 3;
        }
        qDebug() << "Error while parsing lyrics\n";
        return 2;
    }
    if (dlret != 0) // just error while downloading
    {
        qDebug() << "Error while downloading lyrics\n";
        return 1;
    }

    //qDebug() << lyrics();
    emit lyricsChanged(lyrics);
    return 0;
}

string AZLyricsDownloader::toProviderCode() const
{
    QString artist_www, track_www;
    artist_www = QString::fromStdString(artist).toLower().remove(QRegExp("^the")).remove(QRegExp("\\W"));
    track_www = QString::fromStdString(track).toLower().remove(QRegExp("\\W"));

    return (rooturl + "/" + artist_www.toStdString() + "/" + track_www.toStdString() + ".html");

}

CURLcode AZLyricsDownloader::download()
{
    CURL * handle;
    CURLcode err;
    ostringstream buff;
    lyrics_str = new string;
    handle = curl_easy_init();
    if (! handle) return static_cast<CURLcode>(-1);
    // set verbose if debug on
    curl_easy_setopt( handle, CURLOPT_VERBOSE, DEBUG );
    curl_easy_setopt( handle, CURLOPT_URL, toProviderCode().c_str() ); // set the download url to the generated one
    curl_easy_setopt(handle, CURLOPT_WRITEDATA, &buff);
    curl_easy_setopt(handle, CURLOPT_WRITEFUNCTION, &AZLyricsDownloader::write_data_to_var); // set the function writing to ostringstream
    curl_easy_setopt(handle, CURLOPT_FOLLOWLOCATION, 1); // curl needs to follow redirects with this provider
    curl_easy_setopt(handle, CURLOPT_FAILONERROR, 1);
    err = curl_easy_perform(handle);
    curl_easy_cleanup(handle);

    // copy the contents to text variable
    *lyrics_str = buff.str();
    return err;


}

bool AZLyricsDownloader::parse()
{
    istringstream temp(*lyrics_str);
    delete lyrics_str;
    lyrics_str = 0;
    string line;
    QString lyrics_QStr, line_QStr;
    while ( temp.good() ) // looking for comment about starting lyrics
    {
        getline(temp, line);
        if (QString::fromStdString(line).contains(startcomment)) break;
    }
    if ( !temp.good() ) return 1; // something went wrong, returning blank string
    // get every line until endcomment
    while ( temp.good() )
    {
        getline(temp, line);
        if (( line_QStr = QString::fromStdString(line) ).contains(endcomment)) break;
        lyrics_QStr += line_QStr.remove(QRegExp("<br[ ]{0,1}[/]{0,1}>"));
        lyrics_QStr += "\n";
    }
    if ( !temp.good() ) return 1; // something went wrong, returning blank string

    lyrics = lyrics_QStr; // after parsing all the work will be done using QStrings
    return 0;
}
