###############################################################################
# copyright 2010 Edwin Marshall (aspidites) <aspidties@inbox.com>             #
#                                                                             #
# This file is part of MaeGym                                                 #
#                                                                             #
# MaeGym is free software: you can redistribute it and/or                     #
# modify it under the terms of the GNU General Public License as published by #
# the Free Software Foundation, either version 3 of the License, or           #
# (at your option) any later version.                                         #
#                                                                             #
# MaeGym is distributed in the hope that it will be useful,                   #
# but WITHOUT ANY WARRANTY; without even the implied warranty of              #
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the               #
# GNU General Public License for more details.                                #
#                                                                             #
# You should have received a copy of the GNU General Public License           #
# along with MaeGym.                                                          # 
# If not, see <http://www.gnu.org/licenses/>.                                 #
###############################################################################

# builtin modules
import sys

# qt modules
from PyQt4.QtSql import (QSqlDatabase, QSqlQuery, QSqlQueryModel,
			 QSqlRelation, QSqlRelationalTableModel)
from PyQt4.QtGui import (QProgressDialog)
from PyQt4.QtCore import (Qt)
import sip

# my modules 
from config import Config
from models import ExercisesRelationalModel

reload(sys)
sys.setdefaultencoding('utf-8')

sip.getapi('QVariant')
class Database(object):
    """PyQt version of database."""
    def __init__(self, parent=None):
        self.config = Config()

        self.parent = parent
        self.db = QSqlDatabase.addDatabase('QSQLITE')
        self.db.setDatabaseName(self.config.uses_db_file)
        self.db.open()

        if self.config.general['first-run']:
            self.createProgressBarDialog()
            self.createTables()
            self.fillTables()
            self.progress_bas_dialog.close()
            self.config.general['first-run'] = False

        self.createModels()

    def createProgressBarDialog(self):
        self.progress_bas_dialog = QProgressDialog('Creating database. Please wait..',
                                                   'Stop', 0, 100, self.parent)
        self.progress_bas_dialog.setWindowTitle('MaeGym')
        if self.config.interface['orientation'] == 'Portrait':
            self.progress_bas_dialog.setAttribute(Qt.WA_Maemo5PortraitOrientation)
        elif self.config.interface['orientation'] == 'Landscape':
            self.progress_bas_dialog.setAttribute(Qt.WA_Maemo5LandscapeOrientation)
        elif self.config.interface['orientation'] == 'Auto':
            self.progress_bas_dialog.setAttribute(Qt.WA_Maemo5AutoOrientation)

        self.progress_bas_dialog.setWindowModality(Qt.WindowModal)
        self.progress_bas_dialog.show()


    def createTables(self):
        self.progress_bas_dialog.setValue(0)
        query = QSqlQuery()
        query.exec_("""CREATE TABLE IF NOT EXISTS target_areas (
                       t_id INTEGER PRIMARY KEY AUTOINCREMENT NOT NULL,
                       t_name TEXT COLLATE NOCASE NOT NULL)""")

        self.progress_bas_dialog.setValue(5)

        query.exec_("""CREATE TABLE IF NOT EXISTS frequencies (
                       f_id INTEGER PRIMARY KEY AUTOINCREMENT NOT NULL,
                       f_name TEXT COLLATE NOCASE NOT NULL)""")

        self.progress_bas_dialog.setValue(10)

        query.exec_("""CREATE TABLE IF NOT EXISTS intensities (
                       i_id INTEGER PRIMARY KEY AUTOINCREMENT NOT NULL,
                       i_name TEXT COLLATE NOCASE NOT NULL)""")

        self.progress_bas_dialog.setValue(15)

        query.exec_("""CREATE TABLE IF NOT EXISTS durations (
                       d_id INTEGER PRIMARY KEY AUTOINCREMENT NOT NULL,
                       d_name TEXT COLLATE NOCASE NOT NULL)""")

        self.progress_bas_dialog.setValue(20)

        query.exec_("""CREATE TABLE IF NOT EXISTS equipment (
                       e_id INTEGER PRIMARY KEY AUTOINCREMENT NOT NULL,
                       e_name TEXT COLLATE NOCASE NOT NULL)""")

        self.progress_bas_dialog.setValue(25)

        query.exec_("""CREATE TABLE IF NOT EXISTS exercises (
                       ex_id INTEGER PRIMARY KEY AUTOINCREMENT NOT NULL,
                       ex_name TEXT COLLATE NOCASE NOT NULL,
                       ex_description TEXT COLLATE NOCASE,
                       t_id INTEGER NOT NULL,
                       f_id INTEGER NOT NULL,
                       d_id INTEGER NOT NULL,
                       i_id INTEGER NOT NULL,
                       e_id INTEGER NOT NULL,
                       ex_media TEXT,
                       FOREIGN KEY(t_id) REFERENCES target_areas(t_id),
                       FOREIGN KEY(f_id) REFERENCES frequencies(f_id),
                       FOREIGN KEY(i_id) REFERENCES intensities(i_id),
                       FOREIGN KEY(d_id) REFERENCES durations(d_id)
                       FOREIGN KEY(e_id) REFERENCES equipment(e_id))""")

        self.progress_bas_dialog.setValue(35)

        query.exec_("""CREATE TABLE IF NOT EXISTS sessions(
                       s_id INTEGER PRIMARY KEY AUTOINCREMENT NOT NULL,
                       s_name TEXT COLLATE NOCASE NOT NULL,
                       s_description TEXT COLLATE NOCASE)""")

        query.exec_("""CREATE TABLE IF NOT EXISTS session_exercises(
                       re_id INTEGER PRIMARY KEY AUTOINCREMENT NOT NULL,
                       s_id INTEGER NOT NULL,
                       ex_id INTEGER NOT NULL,
                       FOREIGN KEY(s_id) REFERENCES sessions(s_id),
                       FOREIGN KEY(ex_id) REFERENCES exercises(ex_id))""")

        self.progress_bas_dialog.setValue(40)

        query.exec_("""CREATE TABLE IF NOT EXISTS completed_sessions(
                       cs_id INTEGER PRIMARY KEY AUTOINCREMENT NOT NULL,
                       re_id INTEGER NOT NULL,
                       date DATETIME,
                       FOREIGN KEY(re_id) REFERENCES sessions(re_id))""")

        self.db.commit()

        self.progress_bas_dialog.setValue(50)

        query.exec_("""CREATE TABLE IF NOT EXISTS completed_exercises(
                       cs_id INTEGER PRIMARY KEY AUTOINCREMENT NOT NULL,
                       ex_id INTEGER NOT NULL,
                       goal_frequency TEXT COLLATE NOCASE NOT NULL
                       actual_frequency TEXT COLLATE NOCASE NOT NULL
                       goal_duration TEXT COLLATE NOCASE NOT NULL
                       actual_duration TEXT COLLATE NOCASE NOT NULL
                       goal_intensity TEXT COLLATE NOCASE NOT NULL
                       actual_intensity TEXT COLLATE NOCASE NOT NULL
                       FOREIGN KEY(ex_id) REFERENCES exercises(ex_id))""")

        self.db.commit()

        self.progress_bas_dialog.setValue(55)

    def fillTables(self):
        query = QSqlQuery()
        query.prepare("""INSERT OR IGNORE INTO target_areas
                         (t_name) VALUES (?)""")

        for t_name in ('Abdominals', 'Aerobic', 'Biceps', 'Calves', 'Cardio', 
                       'Chest', 'Fat Burn', 'Forearms', 'Glutes', 
                       'Hamstrings', 'Lats', 'Lower Back', 'Middle Back', 
                       'Neck', 'Other', 'Quadriceps', 'Shoulders', 'Traps', 
                       'Triceps'):
            query.addBindValue(t_name)
            query.exec_()

        self.db.commit()

        self.progress_bas_dialog.setValue(60)

        query.prepare("""INSERT OR IGNORE INTO frequencies
                         (f_name) VALUES (?)""")

        for f_name in ('Circuits', 'Sets', 'Time'):
            query.addBindValue(f_name)
            query.exec_()

        self.db.commit()

        self.progress_bas_dialog.setValue(65)

        query.prepare("""INSERT OR IGNORE INTO durations
                         (d_name) VALUES (?)""")

        for d_name in ('Distance', 'Laps', 'Reps', 'Time'):
            query.addBindValue(d_name)
            query.exec_()

        self.db.commit()

        self.progress_bas_dialog.setValue(75)

        query.prepare("""INSERT OR IGNORE INTO intensities
                         (i_name) VALUES (?)""")

        for i_name in ('Percentage', 'Weight'):
            query.addBindValue(i_name)
            query.exec_()

        self.db.commit()

        self.progress_bas_dialog.setValue(85)

        query.prepare("""INSERT OR IGNORE INTO equipment
                         (e_name) VALUES (?)""")

        for e_name in ('Barbell', 'Cable', 'Dumbbell', 'Exercise Ball', 
                       'Machine', 'None', 'Other'):
            query.addBindValue(e_name)
            query.exec_()

        self.db.commit()

        self.progress_bas_dialog.setValue(100)

    def createModels(self):
        self.target_areas_model = QSqlRelationalTableModel()
        self.target_areas_model.setTable('target_areas')
        self.target_areas_model.sort(1, Qt.AscendingOrder)

        self.frequencies_model = QSqlRelationalTableModel()
        self.frequencies_model.setTable('frequencies')
        self.frequencies_model.sort(1, Qt.AscendingOrder)

        self.durations_model = QSqlRelationalTableModel()
        self.durations_model.setTable('durations')
        self.durations_model.sort(1, Qt.AscendingOrder)

        self.intensities_model = QSqlRelationalTableModel()
        self.intensities_model.setTable('intensities')
        self.intensities_model.sort(1, Qt.AscendingOrder)

        self.equipment_model = QSqlRelationalTableModel()
        self.equipment_model.setTable('equipment')
        self.equipment_model.sort(1, Qt.AscendingOrder)

        self.sessions_model = QSqlRelationalTableModel()
        self.sessions_model.setTable('sessions')
        self.sessions_model.sort(1, Qt.AscendingOrder)

    def exercises_model(self, t_name=None):
        model = ExercisesRelationalModel()
        model.setTable('exercises')
        model.setRelation(3, QSqlRelation('target_areas', 
                                                         't_id', 't_name'))
        model.setRelation(4, QSqlRelation('frequencies', 
                                                         'f_id', 'f_name'))
        model.setRelation(5, QSqlRelation('durations', 
                                                         'd_id', 'd_name'))
        model.setRelation(6, QSqlRelation('intensities', 
                                                         'i_id', 'i_name'))
        model.setRelation(7, QSqlRelation('equipment', 
                                                         'e_id', 'e_name'))
        model.sort(1, Qt.AscendingOrder)

        if t_name:
            model.setFilter("t_name='%s'" % t_name)

        return model

    def session_exercises_model(self, s_name):
        model = QSqlQueryModel()
        QUERY = """SELECT re.re_id, e.ex_name, e.ex_description, e.t_id, 
                          e.f_id, e.d_id, e.i_id, e.e_id, r.s_name, 
                          r.s_description
                   FROM exercises e, sessions r, session_exercises re
                   WHERE r.s_id=re.s_id AND e.ex_id=re.ex_id """

        QUERY += "AND r.s_name='%s'" % s_name

        model.setQuery(QUERY)
        model.sort(1, Qt.AscendingOrder)

        return model

    def editExercise(self, ex_name, ex_description, t_id, 
                    f_id, d_id, i_id, e_id, ex_media, old_name=None):

        query = QSqlQuery()
        if old_name:
            query.prepare("""UPDATE exercises
                             SET ex_name=?, ex_description=?, t_id=?, f_id=?,
                             d_id=?, e_id=?, e_id=?, ex_media=?
                             WHERE ex_name=?""")
        else:
            query.prepare("""INSERT OR IGNORE INTO exercises
                             (ex_name, ex_description, t_id, f_id, 
                              d_id, i_id, e_id, ex_media) 
                             VALUES (?, ?, ?, ?, ?, ?, ?, ?)""")

        query.addBindValue(ex_name)
        query.addBindValue(ex_description)
        query.addBindValue(t_id)
        query.addBindValue(f_id)
        query.addBindValue(d_id)
        query.addBindValue(i_id)
        query.addBindValue(e_id)
        query.addBindValue(ex_media)

        if old_name:
            query.addBindValue(old_name)

        query.exec_()
        self.db.commit()

    def editTargetArea(self, t_name, old_name=None):

        query = QSqlQuery()
        if old_name:
            query.prepare("""UPDATE target_areas
                             SET t_name=?
                             WHERE t_name=?""")
        else:
            query.prepare("""INSERT OR IGNORE INTO target_areas
                             (t_name) 
                             VALUES (?)""")

        query.addBindValue(t_name)

        if old_name:
            query.addBindValue(old_name)

        query.exec_()
        self.db.commit()
        self.target_areas_model.select()

    def editSession(self, s_name, s_description, old_name=None):
        query = QSqlQuery()
        if old_name:
            query.prepare("""UPDATE sessions
                             SET s_name=?
                             WHERE s_name=?""")
        else:
            query.prepare("""INSERT OR IGNORE INTO sessions
                             (s_name, s_description)
                             VALUES(?, ?)""")

        query.addBindValue(s_name)
        query.addBindValue(s_description)

        if old_name:
            query.addBindValue(old_name)

        query.exec_()
        self.db.commit()
        self.sessions_model.select()

    def editSessionExercise(self, s_id, ex_id, old_name=None):
        query = QSqlQuery()
        if old_name:
            pass
        else:
            query.prepare("""INSERT INTO session_exercises
                             (s_id, ex_id)
                             VALUES(?, ?)""")

        query.addBindValue(s_id)
        query.addBindValue(ex_id)

        if old_name:
            pass

        query.exec_()
        self.db.commit()

    def removeSessionExercise(self, re_id):
        query = QSqlQuery()
        
        query.prepare("""DELETE FROM session_exercises
                         WHERE re_id=?""")

        query.addBindValue(re_id)

        query.exec_()
        self.db.commit()

    def getRow(self, field, value):
        query = QSqlQuery()
        if field == 't_name':
            query.prepare("""SELECT COUNT(*)
                             FROM target_areas
                             WHERE t_id<=(SELECT t_id
                                          FROM target_areas
                                          WHERE t_name=?)""")
        elif field == 'f_name':
            query.prepare("""SELECT COUNT(*)
                             FROM frequencies
                             WHERE f_id<=(SELECT f_id
                                          FROM frequencies
                                          WHERE f_name=?)""")
        elif field == 'd_name':
            query.prepare("""SELECT COUNT(*)
                             FROM durations
                             WHERE d_id<=(SELECT d_id
                                          FROM durations
                                          WHERE d_name=?)""")
        elif field == 'i_name':
            query.prepare("""SELECT COUNT(*)
                             FROM intensities
                             WHERE i_id<=(SELECT i_id
                                          FROM intensities
                                          WHERE i_name=?)""")
        elif field == 'e_name':
            query.prepare("""SELECT e_id
                             FROM equipment
                             WHERE e_id<=(SELECT e_id
                                          FROM equipment
                                          WHERE e_name=?)""")

        query.addBindValue(value)
        query.exec_()
        query.next()

        return query.value(0).toInt()[0] - 1

if __name__ == '__main__':
    Database()
