#!/usr/bin/env python
# coding=UTF-8
# 
# Copyright (C) 2010 Stefanos Harhalakis
#
# This file is part of maegirls.
#
# maegirls is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# maegirls is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with maegirls.  If not, see <http://www.gnu.org/licenses/>.
#
# $Id: 0.py 2265 2010-02-21 19:16:26Z v13 $

__version__ = "$Id: 0.py 2265 2010-02-21 19:16:26Z v13 $"

from PyQt4.QtGui import *
from PyQt4.QtCore import *

import sys
import time
import locale

from graph import DaysGraph
import config
import algo

app=None
win=None

class ConfigDialog(QDialog):
    def __init__(self, *args, **kwargs):
	QDialog.__init__(self, *args, **kwargs)

	self.editName=QLineEdit(self)
	self.editCycle=QSpinBox(self)
	self.editCurrent=QSpinBox(self)
	self.editCycle.setRange(10,50)
	self.editCurrent.setRange(1,50)
	self.editCurrent.setWrapping(True)
	self.editCycle.setSuffix(self.tr(" days"))

	self.editCycle.valueChanged.connect(self.slotEditCycleChanged)

	self.l0=QHBoxLayout(self)

	l1=QFormLayout()
	l1.addRow(self.tr("Name:"), self.editName)
	l1.addRow(self.tr("Cycle length:"), self.editCycle)
	l1.addRow(self.tr("Current day in cycle:"), self.editCurrent)

	self.l0.addLayout(l1)

	spacer=QSpacerItem(20, 20, QSizePolicy.Expanding)
	self.l0.addItem(spacer)

	l2=QVBoxLayout()
	self.l0.addLayout(l2)

	self.buttonOk=QPushButton(self)
	self.buttonOk.setText(self.tr("OK"))
	self.buttonOk.clicked.connect(self.slotButOk)
	l2.addWidget(self.buttonOk)

	spacer=QSpacerItem(20, 20, QSizePolicy.Minimum,QSizePolicy.Expanding)
	l2.addItem(spacer)

	self.setWindowTitle(self.tr("Configuration"))

    def slotButOk(self):
	self.name=str(self.editName.text())
	self.cycle=self.editCycle.value()
	self.current=self.editCurrent.value()-1

	self.accept()

    def slotEditCycleChanged(self, value):
	self.editCurrent.setMaximum(value)

    # current starts from 0
    def initValues(self, dt):
	self.dt=dt
	self.editName.setText(dt['name'])
	self.editCycle.setValue(dt['cycle'])
	self.editCurrent.setValue(dt['day0']+1)

class MyMsgDialog(QDialog):
    """
    A Dialog to show a finger-scrollable message

    Typical usage:

    class Koko(MyMsgDialog):
	def __init__(....)
	    MyMsgDialog.__init__(....)


	    self.setWindowTitle("My title")
    
	    l1=QLabel("koko", self.w)
	    self.l.addWidget(l1)
	    ...

	    self.l is a QVBoxLayout. Add everything there.
	    self.w is a QWidget. Use it as parent.

    """
    def __init__(self, *args, **kwargs):
	QDialog.__init__(self, *args, **kwargs)

	# This freaking thing is hard
	# It needs two layouts, one extra widget, the fingerscrollable
	# property set to true *and* setWidgetResizable(True)
	self._mm_l0=QVBoxLayout(self)

	self._mm_q=QScrollArea(self)
	self._mm_q.setWidgetResizable(True)
	self._mm_q.setProperty('FingerScrollable', True)

	self.w=QWidget(self._mm_q)

	self.l=QVBoxLayout(self.w)
	self._mm_q.setWidget(self.w)
	self._mm_l0.addWidget(self._mm_q)

class AboutDialog(MyMsgDialog):
    def __init__(self, *args, **kwargs):
	MyMsgDialog.__init__(self, *args, **kwargs)

	txt=self.tr("""
<html><style>
div.title {
    text-decoration:	underline;
}
</style>
<body>
<p> A program to monitor the women's cycle.  Good for planning (or acting ;-).
Inspired by "MyGirls" app which is (was?) available for Java ME capable phones.

<p style="color: orange;">
WARNING!!! This app is not guaranteed to be accurate or correct!  You cannot 
trust this program, or any program, to give accurate predictions!
The whole women-cycle-thing highly depends on a number of factors that
only a doctor can tell.
<p>
<div class="title">Copyright</div>
<p> Copyright &copy; 2010, Stefanos Harhalakis &lt;v13@v13.gr&gt;

<p> Send comments and bug reports to the above address.

<div class="title">License</div>
<p> This program is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.
<p> This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.
<p> You should have received a copy of the GNU General Public License
along with this program.  If not, see
<a href="http://www.gnu.org/licenses/">http://www.gnu.org/licenses/</a>.
</body>
</html>
	""")

	self.setWindowTitle(self.tr("About MaeGirls"))

	self.ltitle=QLabel("MaeGirls v" + config.version, self.w)
	self.ltitle.setObjectName("title")
	self.ltitle.setSizePolicy(QSizePolicy.Expanding, QSizePolicy.Preferred)
	self.ltitle.setAlignment(Qt.AlignCenter)
	self.l.addWidget(self.ltitle)

	self.label=QLabel(txt, self.w)
	self.label.setWordWrap(True)
	self.label.setTextFormat(Qt.RichText)
	self.label.setAlignment(Qt.AlignJustify)
	self.l.addWidget(self.label)

	self.ltitle.setStyleSheet("""
	QLabel {
	    font-size:	    25pt;
	    color:	    rgb(192,192,192);
	    margin-bottom:  0.5ex;
	    }
	""")

class HelpDialog(MyMsgDialog):
    def __init__(self, *args, **kwargs):
	MyMsgDialog.__init__(self, *args, **kwargs)

	txt=self.tr("""
<p> MaeGirls shows information about women's cycle using some generic
guidelines:  It assumes that ovulation happens 14 days before the start
of the next period and that the period cycle is constant. Also, it assumes
that sperm can live for 4 days, while an egg can live for 2 days.

<p style="color: orange;">
WARNING!!! This is not always correct. There are FAR TOO MANY exceptions
to the above rules!!! You MUST consult a doctor in order to get accurate
predictions!!!

<p> Assuming that you understand the risks of blindly trusting this program,
you become entitled to read the graph as follows:
<p> <span style="color: red">In red:</span> The days that menstruation
happens, assumed to last 5 days.
<p> <span style="color: green">In green:</span> The fertile days as described above.
<p> <span style="color: blue">In blue:</span> The days of PMS
(Premenstrual Syndrome), assumed to last 7 days.

<p> Navigation is easy: Use left-right finger movement to move the calendar
view. Use up-down finger movement to zoom in/out.

<p> This program supports the "monitoring" of the cycle of multiple girls/women.
	""")

	self.setWindowTitle(self.tr("Help"))

	self.label=QLabel(txt, self.w)
	self.label.setWordWrap(True)
	self.label.setTextFormat(Qt.RichText)
	self.label.setAlignment(Qt.AlignJustify)
	self.l.addWidget(self.label)

class GirlsDialog(QDialog):
    def __init__(self, *args, **kwargs):
	QDialog.__init__(self, *args, **kwargs)

	self.l0=QHBoxLayout(self)

	self.lstm=QStringListModel()
	self.lst=QListView(self)
	self.lst.setModel(self.lstm)

	self.lst.setProperty("FingerScrollable", True)

	self.l0.addWidget(self.lst)

	self.buttonNew=QPushButton(self)
	self.buttonNew.setText(self.tr("New"))

	self.buttonSelect=QPushButton(self)
	self.buttonSelect.setText(self.tr("Select"))

	self.buttonDelete=QPushButton(self)
	self.buttonDelete.setText(self.tr("Delete"))

	spacer=QSpacerItem(20, 20, QSizePolicy.Minimum,QSizePolicy.Expanding)

	self.l1=QVBoxLayout()
	self.l0.addLayout(self.l1)
	self.l1.addWidget(self.buttonNew)
	self.l1.addWidget(self.buttonSelect)
	self.l1.addWidget(self.buttonDelete)
	self.l1.addItem(spacer)

	self.buttonNew.clicked.connect(self.slotNew)
	self.buttonDelete.clicked.connect(self.slotDelete)
	self.buttonSelect.clicked.connect(self.slotSelect)

    def _get_selection(self):
	sel=self.lst.selectedIndexes()
	if len(sel)==1:
	    d=sel[0]
	    ret=str(d.data().toString())
	else:
	    ret=None

	return(ret)

    def exec_(self, current):
	# Set data
	girls=config.loadGirls()
	dt=girls.keys()
	dt.sort()
	self.lstm.setStringList(dt)

	self.what=""
	self.which=None

	# Set current selection
	idx=dt.index(current)

	# Either I'm doing something stupid, or this is a QT bug
	# The selection works but isn't shown
	idx2=self.lstm.index(idx, 0)
	self.lst.setCurrentIndex(idx2)
	# Give if focus to show the current selection - is this normal?
	self.lst.setFocus(Qt.OtherFocusReason)

	# Run
	QDialog.exec_(self)

    def slotNew(self):
	self.what="new"
	self.which=None
	self.accept()

    def slotDelete(self):
	self.what="delete"
	self.which=self._get_selection()
	self.accept()
	
    def slotSelect(self):
	self.what="select"
	self.which=self._get_selection()
	self.accept()

class MaeGirls(QMainWindow):
    def __init__(self, algo):
	QMainWindow.__init__(self)

	try:
	    self.setAttribute(Qt.WA_Maemo5AutoOrientation, True)
	except:
	    pass

	self.setupUi(algo)

#	self.dlgConfig=ConfigDialog(self)
#	self.dlgAbout=AboutDialog(self)
#	self.dlgHelp=HelpDialog(self)
	self.dlgConfig=None
	self.dlgAbout=None
	self.dlgHelp=None
	self.dlgGirls=None

	self.algo=algo

    def setupUi(self, algo):
	self.centralwidget=QWidget(self)
	self.setCentralWidget(self.centralwidget)

	self.l0=QVBoxLayout(self.centralwidget)

	self.dg=DaysGraph(algo, self.centralwidget)
	self.l0.addWidget(self.dg)

	# Menu
	self.menuconfig=QAction(self.tr('Configure'), self)
	self.menuconfig.triggered.connect(self.menuConfig)

	self.menureset=QAction(self.tr('Go to today'), self)
	self.menureset.triggered.connect(self.menuReset)

	self.menugirls=QAction(self.tr('Girls'), self)
	self.menugirls.triggered.connect(self.menuGirls)

	self.menuabout=QAction(self.tr('About'), self)
	self.menuabout.triggered.connect(self.menuAbout)

	self.menuhelp=QAction(self.tr('Help'), self)
	self.menuhelp.triggered.connect(self.menuHelp)

	m=self.menuBar()
	m.addAction(self.menureset)
	m.addAction(self.menuconfig)
	m.addAction(self.menugirls)
	m.addAction(self.menuhelp)
	m.addAction(self.menuabout)

    def updateTitle(self):
	txt="MaeGirls - %s" % (self.girl, )
	self.setWindowTitle(txt)

    def setAlgo(self, algo):
	self.dg.setAlgo(algo)

    def setGirl(self, name):
	cfg=config.loadGirl(name)
	self.girl=name
	self.algo.setReference(cfg['day0'], cfg['cycle'])
	self.update()
	self.updateTitle()

    def menuConfig(self):
	if self.dlgConfig==None:
	    self.dlgConfig=ConfigDialog(self)

	dt={
	    'name':	self.girl,
	    'cycle':	self.algo.cycleLength(),
	    'day0':	self.algo.currentDayInCycle()
	    }

	self.dlgConfig.initValues(dt)

	ret=self.dlgConfig.exec_()

	if ret==self.dlgConfig.Accepted:
	    today=algo.today()

	    name=self.dlgConfig.name
	    day0=today-self.dlgConfig.current

	    dt={
		'cycle':	self.dlgConfig.cycle,
		'day0':		day0,
		}

	    config.storeGirl(name, dt)
	    config.setCurrentGirl(name)

	    # If this is a rename, remove the old one
	    if self.girl!=name:
		config.removeGirl(self.girl)

	    self.setGirl(name)

	    self.update()

    def menuGirls(self):
	if self.dlgGirls==None:
	    self.dlgGirls=GirlsDialog(self)

	ret=self.dlgGirls.exec_(self.girl)

	what=self.dlgGirls.what
	which=self.dlgGirls.which
	if what=='new':
	    # Determine a unique name
	    base="newgirl"
	    idx=0
	    name=base
	    while config.girlExists(name):
		idx+=1
		name="%s%d" % (base, idx)
	    # Store this
	    config.newGirl(name)
	    # Set it as current
	    config.setCurrentGirl(name)
	    self.setGirl(name)
	    # Edit it
	    self.menuConfig()
	elif what=='delete' and which!=None:
	    if self.girl==which:
		msg=QMessageBox(self)
		msg.setText(self.tr('You cannot delete the current girl'))
		msg.exec_()
	    else:
		config.removeGirl(which)
	elif what=='select' and which!=None:
	    config.setCurrentGirl(which)
	    self.setGirl(which)

    def menuAbout(self):
	if self.dlgAbout==None:
	    self.dlgAbout=AboutDialog(self)

	ret=self.dlgAbout.exec_()

    def menuHelp(self):
	if self.dlgHelp==None:
	    self.dlgHelp=HelpDialog(self)

	ret=self.dlgHelp.exec_()

    def menuReset(self):
	self.dg.reset()

def init(algo):
    global app
    global win
    global qttr, maetr

    # Create the application
    app=QApplication(sys.argv)

    # This returns the country and *NOT* the required locale
    # This means that if you have'set language==en_GB and countr==Greece
    # it will return el_GR.
    # IOW: If you want english messages this will not work
    #loc=QLocale.system().name()

    loc=locale.setlocale(locale.LC_MESSAGES, '')

    # Load translations
    qttr=QTranslator()
    qttr.load("qt_" + loc,
	QLibraryInfo.location(QLibraryInfo.TranslationsPath))
    app.installTranslator(qttr)

    maetr=QTranslator()
    maetr.load("maegirls_" + loc,
	"/usr/share/maegirls/translations")

    # Install the translation
    app.installTranslator(maetr)

    # One day support portrait mode
    #app.setAttribute(Qt.WA_Maemo5PortraitOrientation, True);

    # Create the main window
    win=MaeGirls(algo)
    win.show()

def setAlgo(algo):
    global win
    win.setAlgo(algo)

def setGirl(name):
    global win
    win.setGirl(name)

def doit():
    global app
    app.exec_()

# vim: set ts=8 sts=4 sw=4 noet formatoptions=r ai nocindent:

