# Maegios (Nagios client for Maemo) - Desktop widget.
#
# Copyright 2009-2010 Vladimir Semenov <lothix@gmail.com>
#
# This file is part of Maegios.
#
# Maegios is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# Maegios is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with Maegios.  If not, see <http://www.gnu.org/licenses/>.
    
import sys, os, time, threading
import gtk, hildon, hildondesktop, pango, cairo, gobject
import socket, select
import gettext
import dbus
import pygst
pygst.require('0.10')
import gst
from gnome import gconf
#import traceback
from nagiosconnector import NagiosConnector, NagiosParser
from application import default_config, UIMaegios, start

gettext.install('maegios')

class ConfigurationServiceListener(threading.Thread):
    def __init__(self, toplevel = None):
        threading.Thread.__init__(self)
        
        self.socket = socket.socket(socket.AF_INET, socket.SOCK_STREAM)
        self.socket.bind(('', 0))
        self.socket.listen(1)

        gconf.client_get_default().set_int('/apps/maegios/widget_port', self.socket.getsockname()[1])
        
        self.quitting = False
        self.toplevel = toplevel
        self.setDaemon(True)
        self.start()

    def shutdown(self):
        if self.quitting:
            return
        self.quitting = True
        self.join()

    def run(self):
        while not self.quitting:
            rr, rw, err = select.select([self.socket],[],[], 1)
            if rr:
                conn, addr = self.socket.accept()
                while True:
                    data = conn.recv(2)
                    if not data: break
                    if str(data) == 'RE': 
                        self.toplevel.emit("reload_configuration")
                    conn.send('OK')
                conn.close()
        self.socket.close()
    
class MaegiosPlugin(hildondesktop.HomePluginItem):
    __gsignals__ = {
        'reload_configuration': (gobject.SIGNAL_RUN_LAST, gobject.TYPE_NONE, ()),
    }
        
    def __init__(self):
        global default_config
        hildondesktop.HomePluginItem.__init__(self)
        
        self.config = default_config

        # read config from gconf
        if not gconf.client_get_default().dir_exists('/apps/maegios'):
            gconf.client_get_default().add_dir('/apps/maegios', gconf.CLIENT_PRELOAD_NONE)
            gconf.client_get_default().set_string('/apps/maegios/url', self.config['url'])
            gconf.client_get_default().set_string('/apps/maegios/username', self.config['username'])
            gconf.client_get_default().set_string('/apps/maegios/password', self.config['password'])
            gconf.client_get_default().set_string('/apps/maegios/interval', self.config['interval'])
            gconf.client_get_default().set_string('/apps/maegios/attempts', self.config['attempts'])
            gconf.client_get_default().set_string('/apps/maegios/is_configured', self.config['is_configured'])
            gconf.client_get_default().set_string('/apps/maegios/notify_banner', self.config['notify_banner'])
            gconf.client_get_default().set_string('/apps/maegios/notify_sound', self.config['notify_sound'])
            gconf.client_get_default().set_string('/apps/maegios/notify_vibration', self.config['notify_vibration'])
            gconf.client_get_default().set_string('/apps/maegios/notify_light', self.config['notify_light'])
            
        if gconf.client_get_default().get_string('/apps/maegios/notify_banner') == None:
            gconf.client_get_default().set_string('/apps/maegios/notify_banner', self.config['notify_banner'])    
        if gconf.client_get_default().get_string('/apps/maegios/notify_sound') == None:
            gconf.client_get_default().set_string('/apps/maegios/notify_sound', self.config['notify_sound'])
        if gconf.client_get_default().get_string('/apps/maegios/notify_vibration') == None:
            gconf.client_get_default().set_string('/apps/maegios/notify_vibration', self.config['notify_vibration'])
        if gconf.client_get_default().get_string('/apps/maegios/notify_light') == None:    
            gconf.client_get_default().set_string('/apps/maegios/notify_light', self.config['notify_light'])                        
                        
                        
        # listen for changes to configuration: (this doesn't work for some unknown reason - why?)
        # implemented using sockets with random port provided by os instead
        #self.gconf_client.notify_add('/apps/maegios', self.reload_configuration)   

        self.alpha_channel = True
        self.set_size_request(90, 110)
        self.set_decorated(0)
        self.set_app_paintable(1)
        #self.set_events(gtk.gdk.ALL_EVENTS_MASK) 
        self.set_colormap(self.get_screen().get_rgba_colormap())
        
        self.images = {'OK': '/usr/share/icons/hicolor/64x64/hildon/maegios-ok-64x64.png', 
                       'WARNING': '/usr/share/icons/hicolor/64x64/hildon/maegios-warn-64x64.png', 
                       'CRITICAL': '/usr/share/icons/hicolor/64x64/hildon/maegios-crit-64x64.png',
                       'DISCONNECTED': '/usr/share/icons/hicolor/64x64/hildon/maegios-disconnected-64x64.png'}

        self.event_box = gtk.EventBox()
        self.event_box.set_visible_window(False)       
        
        self.load_config()
        
        # todo: move this out
        if int(gconf.client_get_default().get_string('/apps/maegios/is_configured')) == 1:
            self.update_widget()
            self.button_release_handler = self.event_box.connect("button_release_event",\
            self.start_maegios)             
        else:
            self.label = gtk.Label("Click to\nconfigure\nMaegios")
            self.label.modify_font(pango.FontDescription("11"))     
            self.label.set_justify(gtk.JUSTIFY_CENTER)
            self.label.show()
            self.event_box.add(self.label)
            self.button_release_handler = self.event_box.connect("button_release_event",\
            self.display_nagios_configuration) 
 
        self.event_box.show_all()
        self.add(self.event_box)

        # set up a refresh timer to query Nagios every refreshInterval minutes
        refreshInterval = int(gconf.client_get_default().get_string('/apps/maegios/interval'))
        self.timer = gobject.timeout_add(refreshInterval * 60000, self.refresh)
     
        self.connect("reload_configuration", self.reload_configuration)
        self.connect("expose-event", self.expose)
        self.connect("screen-changed", self.screen_changed)     
        self.connect("destroy", self.quit)
        self.connect("delete_event", self.quit)
             
        # start configuration service listener thread
        self.thread = ConfigurationServiceListener(self)
          
    def reload_configuration(self, widget = None):
        #print "reload_configuration"
                
        self.load_config()
        self.refresh(True)
        
        if self.event_box.handler_is_connected(self.button_release_handler):
            self.event_box.disconnect(self.button_release_handler) 
            self.event_box.connect("button_release_event", self.start_maegios)         
              
    def quit(self, widget = None, label = None):
        self.thread.shutdown()
                
    def refresh(self, softUpdate = False):
        refreshInterval = int(gconf.client_get_default().get_string('/apps/maegios/interval'))
        self.timer = gobject.timeout_add(refreshInterval * 60000, self.refresh)
        self.update_widget(softUpdate)                
                
    def start_maegios(self, widget, label):
        gtk.gdk.threads_init()
        start(self.hosts)

    def display_nagios_configuration(self, widget, label):
        gtk.gdk.threads_init()
        maegios = UIMaegios()
        maegios.display() 
        dialog = UINagiosConfigurationDialog(maegios)             
        dialog.display(maegios)
        gtk.main()

    def expose(self, widget, event):
        """
        Taken from example provided by hns (Khertan's HomeIP plugin) 
        for Maemo 4 on talk.maemo.org forums.
        """
        cr = widget.window.cairo_create()
        
        if self.alpha_channel == True:
	        cr.set_source_rgba(1.0, 1.0, 1.0, 0.0)
        else:
	        cr.set_source_rgb(1.0, 1.0, 1.0)

        cr.set_operator(cairo.OPERATOR_SOURCE)
        cr.paint()

        width, height = self.allocation[2], self.allocation[3]

        x0 = 0
        y0 = 0

        radius = min(15, width/2, height/2)

        x1 = x0 + width
        y1 = y0 + height

        cr.move_to(x0, y0 + radius)
        cr.arc(x0 + radius, y0 + radius, radius, 3.14, 1.5 * 3.14)
        cr.line_to (x1 - radius, y0)
        cr.arc(x1 - radius, y0 + radius, radius, 1.5 * 3.14, 0.0)
        cr.line_to (x1 , y1 - radius)
        cr.arc(x1 - radius, y1 - radius, radius, 0.0, 0.5 * 3.14)
        cr.line_to (x0 + radius, y1)
        cr.arc(x0 + radius, y1 - radius, radius, 0.5 * 3.14, 3.14)

        cr.close_path()
        cr.set_source_rgba(0.0, 0.0, 0.0, 0.5)
        cr.fill_preserve()

        return False
    
    def screen_changed(self, widget, old_screen = None):
        screen = widget.get_screen()
        colormap = screen.get_rgba_colormap()
        if colormap == None:
	        colormap = screen.get_rgb_colormap()
	        self.alpha_channel = False
        else:
	        self.alpha_channel = True

        widget.set_colormap(colormap)

        return False 

    def update_widget(self, softUpdate = False):
    
        self.image = gtk.Image()  
        self.image.set_size_request(64, 64) 
        
        hosts = self.get_nagios_data()
        self.hosts = hosts
        
        # notification patterns for N900
        ledPatternWarning = 'PatternBatteryCharging'
        ledPatternCritical = 'PatternError'
        vibratePattern = 'PatternChatAndEmail'
        
        # connect to mce service
        try:
            bus = dbus.SystemBus()
            smsobject = bus.get_object('com.nokia.mce', '/com/nokia/mce/request')
            smsiface = dbus.Interface(smsobject, 'com.nokia.mce.request')      
            dbusConn = True
        except:
            dbusConn = False
        
        # deactivate previous led patterns
        if dbusConn:
            smsiface.req_led_pattern_deactivate(dbus.String(ledPatternCritical))
            smsiface.req_led_pattern_deactivate(dbus.String(ledPatternWarning))
        
        if hosts == False:
            # display disconnected icon
            self.image.set_from_file(self.images['DISCONNECTED'])
            hostStatusString = _(u"Disco'd")
            infoString = ""
        else:
            criticalServices = 0
            warningServices = 0       
            for host in hosts:
                for service in host['services']:
                    if service['status'] == "WARNING":
                        warningServices += 1
                    elif service['status'] == "CRITICAL":
                        criticalServices += 1
            
            ledPattern = str()              
            hostStatusString = str()
            if criticalServices > 0:
                self.image.set_from_file(self.images['CRITICAL'])   
                ledPattern = ledPatternCritical
                if criticalServices > 1:
                    hostStatusString = str(criticalServices) + " " + _(u"Alerts")
                    infoString = str(criticalServices) + " " + _(u"Critical Alerts")
                else:
                    hostStatusString = str(criticalServices) + " " + _(u"Alert")   
                    infoString = str(criticalServices) + " " + _(u"Critical Alert")
            elif warningServices > 0:
                ledPattern = ledPatternWarning
                if warningServices > 1:
                    hostStatusString = str(warningServices) + " " + _(u"Alerts")
                    infoString = str(warningServices) + " " + _(u"Warning Alerts")
                else:
                    hostStatusString = str(warningServices) + " " + _(u"Alert")
                    infoString = str(warningServices) + " " + _(u"Warning Alert")
                self.image.set_from_file(self.images['WARNING'])   
            else:
                self.image.set_from_file(self.images['OK'])   
                hostStatusString = _(u"All Good!")
                infoString = ""
        
        self.label = gtk.Label(hostStatusString)
        self.label.set_justify(gtk.JUSTIFY_CENTER)
        self.label.modify_font(pango.FontDescription("11"))    
        
        if len(infoString) > 0:
            # notify
            if int(gconf.client_get_default().get_string('/apps/maegios/notify_banner')) == 1 and softUpdate == False and dbusConn:
                banner = hildon.hildon_banner_show_information(self, '', \
                "Nagios: %s!" % infoString)
                banner.set_timeout(5000)
                
            if int(gconf.client_get_default().get_string('/apps/maegios/notify_sound')) == 1 and softUpdate == False and dbusConn:
                player = gst.element_factory_make('playbin2', 'myplayer')
                player.set_property('uri', 'file:///usr/share/sounds/ui-new_email.wav')
                player.set_state(gst.STATE_PLAYING)
                            
            if int(gconf.client_get_default().get_string('/apps/maegios/notify_vibration')) == 1 and softUpdate == False and dbusConn:
                smsiface.req_vibrator_pattern_activate(dbus.String(vibratePattern))
            
            if int(gconf.client_get_default().get_string('/apps/maegios/notify_light')) == 1 and dbusConn:
                smsiface.req_led_pattern_activate(dbus.String(ledPattern))
                        
        self.image.show_all()
        self.label.show_all()

        self.vbox = gtk.VBox(False, 0)
        if len(self.event_box.get_children()) > 0:
            self.event_box.remove(self.event_box.get_children()[0])      
              
        self.vbox.set_border_width(12)
        self.vbox.pack_start(self.image, False, False, 0)
        self.vbox.pack_start(self.label, False, False, 5)
        self.vbox.show_all()
        self.event_box.add(self.vbox)

    def load_config(self):
        # load maegios configuration from gconf:
        if gconf.client_get_default().dir_exists('/apps/maegios'):
            self.config['url'] = gconf.client_get_default().get_string('/apps/maegios/url')
            self.config['username'] = gconf.client_get_default().get_string('/apps/maegios/username')
            self.config['password'] = gconf.client_get_default().get_string('/apps/maegios/password')
            self.config['interval'] = gconf.client_get_default().get_string('/apps/maegios/interval')
            self.config['attempts'] = gconf.client_get_default().get_string('/apps/maegios/attempts')
            self.config['is_configured'] = gconf.client_get_default().get_string('/apps/maegios/is_configured')
            self.config['notify_banner'] = gconf.client_get_default().get_string('/apps/maegios/notify_banner')           
            self.config['notify_sound'] = gconf.client_get_default().get_string('/apps/maegios/notify_sound')     
            self.config['notify_vibration'] = gconf.client_get_default().get_string('/apps/maegios/notify_vibration')     
            self.config['notify_light'] = gconf.client_get_default().get_string('/apps/maegios/notify_light')      
                  
    def get_nagios_data(self):
        # Load and parse nagios data
        remainingAttempts = int(gconf.client_get_default().get_string('/apps/maegios/attempts'))
        
        while remainingAttempts > 0:
            #print "connection attempts [widget]: %d" % remainingAttempts
            try:
                parser = NagiosParser(only_active = 1)
                connector = NagiosConnector(self)
                parser.feed(connector.get_down_hosts())
                parser.close()
                break
            except:
                remainingAttempts -= 1
                time.sleep(0.1)
            
        if remainingAttempts == 0:
            return False
        else:
            return parser.get_hosts()


