# Maegios (Nagios client for Maemo) - Connector and parser for Nagios.
#
# Copyright 2009-2010 Vladimir Semenov <lothix@gmail.com>
#
# This file is part of Maegios.
#
# Maegios is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# Maegios is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with Maegios.  If not, see <http://www.gnu.org/licenses/>.

import socket, httplib, urllib, base64, htmllib, formatter
from gnome import gconf

class NagiosConnector():
    '''
    Retrieves status information from Nagios installation over HTTP/HTTPS.
    '''
    
    def __init__(self, toplevel = None):
        self.toplevel = toplevel

    def get_down_hosts(self):
        params = urllib.urlencode({'host': 'all', 'servicestatustypes': 28})
        return self.get_data(params)
      
    def get_all_hosts(self):
        params = urllib.urlencode({'host': 'all'})
        return self.get_data(params)

    def get_data(self, params):
        username = str()
        password = str()
        try:
            username = base64.b64decode(gconf.client_get_default().get_string('/apps/maegios/username'))
            password = base64.b64decode(gconf.client_get_default().get_string('/apps/maegios/password'))
        except:
            pass

        base64string = base64.encodestring('%s:%s' % (username, password))[:-1]
        headers = {'Authorization': 'Basic %s' % base64string}
        url = gconf.client_get_default().get_string('/apps/maegios/url').split('/')
        socket.setdefaulttimeout(5)
        if url[0][4] == 's':
            connection = httplib.HTTPSConnection(url[2])
        else:
            connection = httplib.HTTPConnection(url[2])
        connection.request('POST', '/%s/%s' % (url[3], 'cgi-bin/status.cgi'), params, headers)
        response = connection.getresponse()
        data = str()
        if response.status != 200:
            raise Exception     
        else:
            data = response.read()
            if len(data) == 0:
                raise Exception

        connection.close()
        return data
        
class NagiosParser(htmllib.HTMLParser):
    '''
    Parser for HTML from Nagios installation. 
    When is Nagios getting XML webservice?
    
    Tested to work with Nagios versions 2.8, 3.0.6.
    '''
    
    def __init__(self, verbose = 0, only_active = 0):

        htmllib.HTMLParser.__init__(
            self, formatter.NullFormatter())
            
        self.only_active = only_active
        self.hosts = []
        self.active = 1
        self.skipping = 1
        self.parsingHost = 0
        self.parsingServiceName = 0
        self.tdNum = 0
        self.tableDepth = 0
        self.hostTable = 0
        self.tdCount = 0
        self.tables = 0
        self.hostUp = 0
        self.popNext = 0
        self.skipHost = 0

    def start_table(self, attributes):
        self.tableDepth += 1

    def end_table(self):
        self.tableDepth -= 1
        self.tables += 1

    def start_td(self, attributes):
        if self.tableDepth == 1:
            self.tdCount += 1
        if self.tableDepth == 3:
            for name, value in attributes:
                if (name == "class" and \
                   (value == "statusHOSTDOWN" or value == "statusHOSTDOWNACK")) \
                   or \
                   (name == "class" and \
                   (value == "statusEven" or value == "statusOdd")) \
                   or \
                   (name == "class" and \
                   (value == "statusBGCRITICAL" or value == "statusBGWARNING")):
                    #if self.tdCount == 1:
                    if value == "statusHOSTDOWN" or value == "statusHOSTDOWNACK":
                        self.hostUp = 0
                    else:
                        self.hostUp = 1
                    self.skipping = 0
                    self.parsingHost = 1
                    self.hostTable = 1
                    if self.tdCount == 2:
                        self.skipping = 0
                        self.parsingServiceName = 1
                        self.tdNum = 1
   
        if self.hostTable and self.tableDepth == 1:
            self.skipping = 0
            self.tdNum += 1
          
    def start_img(self, attributes):
        if self.only_active:
            for name, value in attributes:
                if name == "alt" and \
                (value == "Notifications for this host have been disabled" or \
                value == "This host problem has been acknowledged"):
                    self.parsingHost = 0
                    self.skipping = 1
                    self.popNext = 1
                if name == "alt" and \
                (value == "Notifications for this service have been disabled" or \
                value == "This service problem has been acknowledged"):
                    self.skipping = 0
                    self.parsingServiceName == 0
                    if len(self.hosts[-1]['services']) > 0:
                        self.hosts[-1]['services'].pop()
            
    def end_td(self):
        if self.tableDepth == 3:
            self.skipping = 1
        
    def end_a(self):
        if self.tableDepth == 3:
            self.parsingServiceName = 0
            self.parsingHost = 0
            self.skipping = 1
        
    def start_tr(self, attributes):
        if self.tableDepth == 1:
            self.tdCount = 0
        
    def end_tr(self):
        pass

    def handle_data(self, data):
        if self.active and not self.skipping and data != '\n' and len(data) and data[0] != '[':
            if self.parsingServiceName == 1:         
                self.hosts[-1]['services'].append({'name': data, 
                                                   'status': '', 
                                                   'info': ''})    
            elif self.tableDepth == 1 and self.tdNum == 2 and \
            len(self.hosts[-1]['services']) > 0:
                self.hosts[-1]['services'][-1]['status'] = data
            elif self.tableDepth == 1 and self.tdNum == 6 and \
            len(self.hosts[-1]['services']) > 0:
                self.hosts[-1]['services'][-1]['info'] = data.rstrip('\xA0')
                self.skipping = 1
            elif self.parsingHost:
                if self.popNext:
                    self.hosts.pop()
                    self.popNext = 0            
                self.hosts.append({'host': {'hostname': data, 'status': \
                'UP' if self.hostUp else 'DOWN'}, 'services': list()})

    def get_hosts(self):
    
        if self.popNext:
            self.hosts.pop()
        
        result = []
        for host in self.hosts:
            if len(host['services']) > 0:
                result.append(host)
        
        return result        

