/*
  maeFat Rev 1.5
  First Version Feb. 5, 2011

  Copyright (C) (2011) Ken Young orrery.moko@gmail.com

  This program is free software; you can redistribute it and/or 
  modify it under the terms of the GNU General Public License 
  as published by the Free Software Foundation; either 
  version 2 of the License, or (at your option) any later 
  version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

  ------------------------------------------------------------------

  This program is a tool for personal weight reduction/management.
  It is similar to the Hacker Diet tools, although it is not a clone
  of that package.   It lacks the features for tracking exercize
  levels which are present in the Hacker Dieat tools, but adds other
  features such as the ability to prdict when a weightloss goal
  will be reached.   It is optimized to make daily entry of weaight
  measurements as quick as possible.

 */

#include <stdio.h>
#include <sys/stat.h>
#include <dirent.h>
#include <locale.h>
#include <fcntl.h>
#include <math.h>
#include <string.h>
#include <ctype.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <unistd.h>
#include <signal.h>
#include <fcntl.h>
#include <errno.h>
#include <sys/time.h>
#include <stdlib.h>
#include <memory.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <cairo.h>
#include <cairo-xlib.h>
#include <pango/pangocairo.h>
#include <gtk/gtk.h>
#include <dbus/dbus.h>
#include <hildon-1/hildon/hildon.h>
#include <libosso.h>
#include "maeFatColors.h"

#define FALSE       (0)
#define ERROR_EXIT (-1)
#define OK_EXIT     (0)

#define M_HALF_PI (M_PI * 0.5)

#define KG_PER_LB (0.45359237)
#define M_PER_IN (2.54e-2)
#define CALORIES_PER_POUND (3500.0) /* Need to propagate this */

#define RENDER_CENTER_X (80)
#define RENDER_CENTER_Y (400)
#define INT_TOKEN     0
#define DOUBLE_TOKEN  1
#define FLOAT_TOKEN   2
#define STRING_TOKEN  3
#define BORDER_FACTOR_X (0.005)
#define BORDER_FACTOR_Y (0.015)
#define LEFT_BORDER   (33)
#define RIGHT_BORDER  (40)
#define TOP_BORDER    (21)
#define BOTTOM_BORDER (48)

unsigned char normalYear[12] = {0, 3, 3, 6, 1, 4, 6, 2, 5, 0, 3, 5};
unsigned char leapYear[2] = {6, 2};
char *homeDir, *userDir, *fileName, *settingsFileName;
char *backupDir = "/media/mmc1/.maeFat";
char *backupFile = "/media/mmc1/.maeFat/data";
char *defaultComment = "Type Comment Here (optional)";
char *monthName[12] = {"January",   "February", "March",    "April",
		       "May",       "June",     "July",     "August",
		       "September", "October",  "November", "December"};
char *dayName[7] = {"Sunday", "Monday", "Tuesday", "Wednesday", "Thursday", "Friday", "Saturday"};
char *maeFatVersion = "1.5";
char scratchString[200];

int debugMessagesOn = FALSE;
#define dprintf if (debugMessagesOn) printf
int nDataPoints = 0;
int goodColor = OR_GREEN;
int badColor  = OR_RED;
int displayHeight, displayWidth, accepted, deleted;
int minSelectorWeight = 50;
int maxSelectorWeight = 300;
int mainBoxInTrendStackable = FALSE;

double monthLengths[12] = {31.0, 28.0, 31.0, 30.0, 31.0, 30.0, 31.0, 31.0, 30.0, 31.0, 30.0, 31.0};
double maxJD = -1.0;
double minJD = 1.0e30;
double maxWeight = -1.0;
double minWeight = 1.0e30;
double plotMaxX, plotMinX, plotMaxY, plotMinY;

typedef struct logEntry {
  double time;           /* The time at which the log entry was made */
  float weight;          /* The reported weight                      */
  char *comment;         /* User-entered comment text                */
  struct logEntry *next; /* Pointer to next entry                    */
  struct logEntry *last; /* Pointer to last entry                    */
} logEntry;

logEntry *logRoot = NULL;
logEntry *lastEntry = NULL;
logEntry *editPtr;

/*
  logCell structures hold information about the weight entries shown
  on the "Log" page.   They are used to implement the edit callback
  capability.
 */
typedef struct logCell {
  GdkPoint        box[4]; /* Coordinates of the little box drawn around this entry */
  logEntry       *ptr;    /* This points to the log entry displayed, which may be edited */
  struct logCell *next;
} logCell;
logCell *logCellRoot = NULL;
logCell *lastLogCell = NULL;
int logDisplayed = FALSE;

/* Values which are stored in the settings file */
double myHeight          =  72.0;
double myTarget          = 150.0;
int weightkg             = FALSE;
int heightcm             = FALSE;
int monthFirst           = FALSE;
int nonjudgementalColors = FALSE;
int hackerDietMode       = FALSE;
int showComments         = FALSE;
int showTarget           = FALSE;
int plotInterval         =   365;
int fitInterval          =    -1;
/* End of settings file variables */

GdkGC *gC[N_COLORS];
GtkWidget *window, *mainBox, *drawingArea, *weightButton, *dataEntryAccept, *dataEditDelete,
  *dataEntryButton, *logButton, *trendButton, *settingsButton,
  *helpButton, *aboutButton, *commentText, *logSelectorButton;
GdkPixmap *pixmap = NULL;
GdkPixmap *cairoPixmap = NULL;

/* Cairo and Pango related globals */

#define N_PANGO_FONTS (4)
#define SMALL_PANGO_FONT       (0)
#define MEDIUM_PANGO_FONT      (1)
#define BIG_PANGO_FONT         (2)
#define SMALL_MONO_PANGO_FONT  (3)
#define BIG_PANGO_FONT_NAME         "Sans Bold 32"
#define MEDIUM_PANGO_FONT_NAME      "Sans Normal 18"
#define SMALL_PANGO_FONT_NAME       "Sans Normal 14"
#define SMALL_MONO_PANGO_FONT_NAME  "Monospace Bold 16"

cairo_t *cairoContext = NULL;

GtkWidget *selector, *dataEntryStackable, *settingsStackable, *logStackable,
  *dateButton, *timeButton, *trendStackable, *dateEditButton, *timeEditButton,
  *dataEditStackable, *heightSpinLabel, *heightSpin, *weightUnitLabel, *heightUnitLabel,
  *dateFormatLabel, *poundsButton, *kgButton, *inButton, *cmButton, *dayButton,
  *monthButton, *nonjudgementalButton, *hackerDietButton, *showCommentsButton,
  *plotFortnightButton, *plotMonthButton, *plotQuarterButton, *plot6MonthButton,
  *plotYearButton, *plotHistoryButton, *targetSpinLabel, *targetSpin, *showTargetButton,
  *fitNothingButton, *fitMonthButton, *fitQuarterButton, *fit6MonthButton,
  *fitYearButton, *fitHistoryButton;

/*   E N D   O F   V A R I A B L E   D E C L A R A T I O N S   */

/*
  C R E A T E  P A N G O  C A I R O  W O R K S P A C E

  Create the structures needed to be able to yse Pango and Cairo to
render antialiased fonts
*/
void createPangoCairoWorkspace(void)
{
  cairoContext = gdk_cairo_create(cairoPixmap);
  if (cairoContext == NULL) {
    fprintf(stderr, "cairo_create returned a NULL pointer\n");
    exit(ERROR_EXIT);
  }
  cairo_translate(cairoContext, RENDER_CENTER_X, RENDER_CENTER_Y);
} /* End of  C R E A T E  P A N G O  C A I R O  W O R K S P A C E */

/*
  R E N D E R  P A N G O  T E X T

  The following function renders a text string using Pango and Cairo, on the
  cairo screen, and copies it to a gtk drawable.   The height and width
  of the area occupied by the text is returned in passed variables.
*/
void renderPangoText(char *theText, unsigned short color, int font,
		     int *width, int *height, GdkDrawable *dest,
		     int x, int y, float angle, int center, int topClip,
		     int background)
{
  static int firstCall = TRUE;
  static PangoFontMap *fontmap = NULL;
  static char *fontName[N_PANGO_FONTS] = {SMALL_PANGO_FONT_NAME, MEDIUM_PANGO_FONT_NAME,
					  BIG_PANGO_FONT_NAME, SMALL_MONO_PANGO_FONT_NAME};
  static PangoContext *pangoContext[N_PANGO_FONTS];
  static cairo_font_options_t *options = NULL;
  static int xC, yC, wC, hC = 0;
  static float currentAngle = 0.0;

  float deltaAngle, cA, sA, fW, fH;
  int pWidth, pHeight, iFont;
  PangoFontDescription *fontDescription;
  PangoLayout *layout;

  if (angle < 0.0)
    angle += M_PI * 2.0;
  else if (angle > M_PI * 2.0)
    angle -= M_PI * 2.0;
  if ((angle > M_HALF_PI) && (angle < M_HALF_PI + M_PI))
    angle -= M_PI;
  if (angle < 0.0)
    angle += M_PI * 2.0;
  if (firstCall) {
    /* Initialize fonts, etc. */
    createPangoCairoWorkspace();
    fontmap = pango_cairo_font_map_get_default();
    if (fontmap == NULL) {
      fprintf(stderr, "pango_cairo_font_map_get_default() returned a NULL pointer\n");
      exit(ERROR_EXIT);
    }
    options = cairo_font_options_create();
    if (options == NULL) {
      fprintf(stderr, "cairo_font_options_create() returned a NULL pointer\n");
      exit(ERROR_EXIT);
    }
    cairo_font_options_set_antialias(options, CAIRO_ANTIALIAS_GRAY);
    cairo_font_options_set_hint_style(options, CAIRO_HINT_STYLE_FULL);
    cairo_font_options_set_hint_metrics(options, CAIRO_HINT_METRICS_ON);
    cairo_font_options_set_subpixel_order(options, CAIRO_SUBPIXEL_ORDER_BGR);
    for (iFont = 0; iFont < N_PANGO_FONTS; iFont++) {
      fontDescription = pango_font_description_new();
      if (fontDescription == NULL) {
	fprintf(stderr, "pango_font_description_new() returned a NULL pointer, font = %d\n", iFont);
	exit(ERROR_EXIT);
      }
      pango_font_description_set_family(fontDescription, (const char*) fontName[iFont]);
      pangoContext[iFont] = pango_cairo_font_map_create_context(PANGO_CAIRO_FONT_MAP(fontmap));
      if (pangoContext[iFont] == NULL) {
	fprintf(stderr, "pango_cairo_font_map_create_context(iFont = %d) returned a NULL pointer\n", iFont);
	exit(ERROR_EXIT);
      }
      pango_context_set_font_description(pangoContext[iFont], fontDescription);
      pango_font_description_free(fontDescription);
      pango_cairo_context_set_font_options(pangoContext[iFont], options);
    }
    firstCall = FALSE;
  } else
    if (font == MEDIUM_PANGO_FONT)
      gdk_draw_rectangle(cairoPixmap, gC[background], TRUE, RENDER_CENTER_X+xC, RENDER_CENTER_Y+yC,
			 wC, hC+7);
    else if (font == BIG_PANGO_FONT)
      gdk_draw_rectangle(cairoPixmap, gC[background], TRUE, RENDER_CENTER_X+xC, RENDER_CENTER_Y+yC,
			 wC, hC+30);
    else
      gdk_draw_rectangle(cairoPixmap, gC[background], TRUE, RENDER_CENTER_X+xC, RENDER_CENTER_Y+yC,
			 wC, hC);
  layout = pango_layout_new(pangoContext[font]);
  if (layout == NULL) {
    fprintf(stderr, "pango_layout_new() returned a NULL pointer\n");
      exit(ERROR_EXIT);
  }
  pango_layout_set_text(layout, theText, -1);
  fontDescription = pango_font_description_from_string(fontName[font]);
  pango_layout_set_font_description (layout, fontDescription);
  pango_font_description_free(fontDescription);
  cairo_set_source_rgb(cairoContext,
		       ((double)orreryColorRGB[color][0])/DOUBLE_MAX16,
		       ((double)orreryColorRGB[color][1])/DOUBLE_MAX16,
			((double)orreryColorRGB[color][2])/DOUBLE_MAX16);
  deltaAngle = angle - currentAngle;
  cairo_rotate(cairoContext, deltaAngle);
  currentAngle = angle;
  pango_cairo_update_layout(cairoContext, layout);
  pango_layout_get_size(layout, &pWidth, &pHeight);
  fW = (float)pWidth/(float)PANGO_SCALE;
  fH = (float)pHeight/(float)PANGO_SCALE;
  *width = (int)(fW+0.5);
  *height = (int)(fH+0.5);
  cairo_move_to(cairoContext, 0.0, 0.0);
  pango_cairo_show_layout(cairoContext, layout);
  g_object_unref(layout);
  cA = cosf(angle); sA = sinf(angle);
  wC = (int)((fW*fabs(cA) + fH*fabs(sA)) + 0.5);
  hC = (int)((fW*fabs(sA) + fH*fabs(cA)) + 0.5);
  if (angle < M_HALF_PI) {
    xC = (int)((-fH*sA) + 0.5);
    yC = 0;
  } else {
    xC = 0;
    yC = (int)((fW*sA) + 0.5);
  }
  if (dest != NULL) {
    if (center) {
      int xM, yM, ys, yd, h;

      xM = (int)((fW*fabs(cA) + fH*fabs(sA)) + 0.5);
      yM = (int)((fW*fabs(sA) + fH*fabs(cA)) + 0.5);
      yd = y-(yM>>1);
      if (yd < topClip) {
	int delta;

	delta = topClip - yd;
	h = hC - delta;
	ys = RENDER_CENTER_Y + yC + delta;
	yd = topClip;
      } else {
	ys = RENDER_CENTER_Y+yC;
	h = hC;
      }
      gdk_draw_drawable(dest, gC[OR_BLUE], cairoPixmap, RENDER_CENTER_X+xC, ys,
			x-(xM>>1), yd, wC, h);
    } else
      gdk_draw_drawable(dest, gC[OR_BLUE], cairoPixmap, RENDER_CENTER_X+xC, RENDER_CENTER_Y+yC,
			x, y-((*height)>>1), wC, hC);
  }
} /* End of  R E N D E R  P A N G O  T E X T */

/*
  C A L C U L A T E  J U L I A N  D A T E

  calculateJulianDate converts the Gregorian Calendar date (dd/mm/yyy)
  to the Julian Date, which it returns.
 */
int calculateJulianDate(int yyyy, int mm, int dd)
{
  int a, y, m;

  a = (14 - mm)/12;
  y = yyyy + 4800 - a;
  m = mm + 12*a - 3;
  return(dd + (153*m + 2)/5 + 365*y +y/4 - y/100 + y/400 -32045);
} /* End of  C A L C U L A T E  J U L I A N  D A T E */

/*
  T J D  T O  D A T E

  Convert the Julian day into Calendar Date as per
  "Astronomical Algorithms" (Meeus)
*/
void tJDToDate(double tJD, int *year, int *month, int *day)
{
  int Z, alpha, A, B, C, D, E;
  double F;

  Z = (int)(tJD+0.5);
  F = tJD + 0.5 - (double)Z;
  if (Z >= 2299161) {
    alpha = (int)(((double)Z - 1867216.25)/36524.25);
    A = Z + 1 + alpha - alpha/4;
  } else
    A = Z;
  B = A + 1524;
  C = (int)(((double)B - 122.1) / 365.25);
  D = (int)(365.25 * (double)C);
  E = (int)(((double)(B - D))/30.6001);
  *day = (int)((double)B - (double)D - (double)((int)(30.6001*(double)E)) + F);
  if (E < 14)
    *month = E - 1;
  else
    *month = E - 13;
  if (*month > 2)
    *year = C - 4716;
  else
    *year = C - 4715;
} /* End of  T J D  T O  D A T E */

/*
  C A L C U L A T E  G O A L  D A T E

  Attempt to calculate the date on which the goal weight will be reached.
  Return TRUE for success, FALSE for failure.  A linear least squares fit is
  done over the interval defined by fitInterval.
 */
int calculateGoalDate(double *jD, int *year, int *month, int *day,
		      int *hour, int *minute, double *slope, double *intercept)
{
  if (fitInterval > 0) {
    int nPoints = 0;
    double interval;
    double Sx, Sy, Sxx, Sxy;
    logEntry *ptr;

    Sx = Sy = Sxx = Sxy = 0.0;
    interval = (double)fitInterval;
    ptr = logRoot;
    while (ptr != NULL) {
      if ((lastEntry->time - ptr->time) <= interval) {
	Sx  += ptr->time;
	Sy  += ptr->weight;
	Sxx += ptr->time * ptr->time;
	Sxy += ptr->time * ptr->weight;
	nPoints++;
      }
      ptr = ptr->next;
    }
    if (nPoints < 2)
      /* We need at least two points to fit a line */
      return(FALSE);
    else {
      double d, n;

      n = (double)nPoints;
      d = n*Sxx - Sx*Sx;
      if (d == 0.0)
	return(FALSE);
      else {
	int yyyy, mmmm, dd, hh, mm;
	double a, b, goalJD;

	b = (n*Sxy - Sx*Sy)/d;
	a = Sy/n - b*Sx/n;
	if (b == 0.0) {
	  goalJD = -1.0;
	  yyyy   = -1;
	  mmmm = dd = hh = mm = 0;
	} else {
	  double wholeJD, fracJD;

	  goalJD = (myTarget - a)/b;
	  wholeJD = (double)((int)goalJD);
	  fracJD = goalJD - wholeJD;
	  hh = (int)(fracJD*24.0);
	  mm = (int)((fracJD - ((double)hh)/24.0)*1440.0 + 0.5);
	  tJDToDate(goalJD, &yyyy, &mmmm, &dd);
	}
	if (jD != NULL)
	  *jD = goalJD;
	if (year != NULL)
	  *year = yyyy;
	if (month != NULL)
	  *month = mmmm;
	if (day != NULL)
	  *day = dd;
	if (hour != NULL)
	  *hour = hh;
	if (minute != NULL)
	  *minute = mm;
	if (slope != NULL)
	  *slope = b;
	if (intercept != NULL)
	  *intercept = a;
	return(TRUE);
      }
    }
  } else
    return(FALSE);
} /* End of  C A L C U L A T E  G O A L  D A T E */

/*
  E N Q U E U E  E N T R Y

  enqueueEntry adds a new log entry to the end of the linked
  list of log entries.   It keeps them in time order, with logRoot
  pointing to the earliest.
*/
void enqueueEntry(double jD, float weight, char *comment)
{
  logEntry *newEntry;

  newEntry = (logEntry *)malloc(sizeof(logEntry));
  if (newEntry == NULL) {
    perror("malloc of newEntry");
    exit(ERROR_EXIT);
  }
  newEntry->time = jD;
  newEntry->weight = weight;
  newEntry->comment = comment;
  newEntry->next = newEntry->last = NULL;
  if (logRoot == NULL)
    /* This is the first entry, just make the root point to it. */
    logRoot = newEntry;
  else {
    logEntry *ptr;

    if (logRoot->time > jD) {
      /* The new entry is the very oldest, insert it at the list's begining. */
      newEntry->next = logRoot;
      logRoot->last = newEntry;
      logRoot = newEntry;
    } else {
      /* Find out where the new entry belongs in the time-ordered list. */
      ptr = logRoot;
      while ((ptr != NULL) && (ptr->time < jD))
	ptr = ptr->next;
      if (ptr == NULL) {
	/* The new entry is the newest, put it at the end. */
	lastEntry->next = newEntry;
	newEntry->last = lastEntry;
      } else {
	/* Insert the new entry inside the list. */
	newEntry->next = ptr;
	newEntry->last = ptr->last;
	(ptr->last)->next = newEntry;
	ptr->last = newEntry;
      }
    }
  }
  /* Make sure lastEntry points to the last entry in the data queue */
  lastEntry = logRoot;
  while (lastEntry->next != NULL)
    lastEntry = lastEntry->next;
  nDataPoints++;
} /* End of  E N Q U E U E  E N T R Y */

/*
  C A L C U L A T E  W E I G H T E D  A V E R A G E 1

  This function calculates the weighted average of the available
  available weight from the first entry until the one pointed
  to by the end parameter passed to it.   If a weight was
  entered more than 100 days ago, it is ignored (to reduce the
  coumputation load for large data sets).
 */
float calculateWeightedAverage1(logEntry *end)
{
  double startTime;
  float ave, weightSum, weight;
  float p = 0.9; /* The weight of each entry is reduced by p per day */
  logEntry *ptr;

  ptr = end;
  startTime = end->time;
  ave = weightSum = 0.0;
  while (ptr != NULL) {
    if ((startTime - ptr->time) < 100.0) {
      weight = powf(p, (float)(startTime - ptr->time));
      ave += weight * ptr->weight;
      weightSum += weight;
    }
    ptr = ptr->last;
  }
  ave /= weightSum;
  return(ave);
} /* End of  C A L C U L A T E  W E I G H T E D  A V E R A G E 1 */

/*
  H A C K E R  D I E T  A V E R A G E

  This routine calculates the weighted average using the "Hacker Diet"
  algorithm.
 */
float hackerDietAverage(logEntry *end)
{
  float ave;
  float p = 0.9;
  logEntry *ptr;

  ave = logRoot->weight;
  ptr = logRoot;
  while (ptr != end) {
    if (ptr->next != NULL)
      ave = ave + (1.0-p)*(ptr->next->weight - ave);
    ptr = ptr->next;
  }
  return(ave);
} /* End of  H A C K E R  D I E T  A V E R A G E */

/*
  C A L C U L A T E  W E I G H T E D  A V E R A G E

  Select which function to use when calculating the weighted
  average of the weight data values.
 */
float calculateWeightedAverage(logEntry *end)
{
  if (hackerDietMode)
    return(hackerDietAverage(end));
  else
    return(calculateWeightedAverage1(end));
} /* End of  C A L C U L A T E  W E I G H T E D  A V E R A G E */

/*
  G E T  L I N E

  Read a line of text from the config file, and strip comments (flagged by #).
*/
int getLine(int fD, int stripComments, char *buffer, int *eOF)
{
  char inChar = (char)0;
  int count = 0;
  int sawComment = FALSE;
  int foundSomething = FALSE;

  buffer[0] = (char)0;
  while ((!(*eOF)) && (inChar != '\n') && (count < 132)) {
    int nChar;

    nChar = read(fD, &inChar, 1);
    if (nChar > 0) {
      foundSomething = TRUE;
      if ((inChar == '#') && stripComments)
        sawComment = TRUE;
      if (!sawComment)
        buffer[count++] = inChar;
    } else {
      *eOF = TRUE;
    }
  }
  if (foundSomething) {
    if (count > 0)
      buffer[count-1] = (char)0;
    return(TRUE);
  } else
    return(FALSE);
} /* End of  G E T  L I N E */

/*
  C O M M A S  T O  P E R I O D S

  Kludge: sscanf does not seem to honor the regionalization settings, in particular
  whether a period or comma is used as the decimal separator for printed floats.
  So I'll just switch commas to periods before passing the string to sscanf.
*/
void commasToPeriods(char *string)
{
  int i;

  if (strlen(string) > 0) {
    for (i = 0; i < strlen(string); i++)
      if (string[i] == '#')
	break;
      else if (string[i] == ',')
	string[i] = '.';
  }
} /* End of C O M M A S  T O  P E R I O D S */

/*
  T O K E N  C H E C K

  Scan "line" for "token".   If found, read the value into
  "value" as an integer, double or float, depending on "type".

  Return TRUE IFF the token is seen.
*/
int tokenCheck(char *line, char *token, int type, void *value)
{
  if (strstr(line, token)) {
    int nRead;

    switch (type) {
    case INT_TOKEN:
      nRead = sscanf(&((char *)strstr(line, token))[strlen(token)+1], "%d", (int *)value);
      if (nRead != 1) {
	fprintf(stderr, "Unable to parse config file line \"%s\"\n", line);
	return(FALSE);
      }
      break;
    case DOUBLE_TOKEN:
      commasToPeriods(&((char *)strstr(line, token))[strlen(token)+1]);
      nRead = sscanf(&((char *)strstr(line, token))[strlen(token)+1], "%lf", (double *)value);
      if (nRead != 1) {
	fprintf(stderr, "Unable to parse config file line \"%s\"\n", line);
	return(FALSE);
      }
      break;
    case FLOAT_TOKEN:
      commasToPeriods(&((char *)strstr(line, token))[strlen(token)+1]);
      nRead = sscanf(&((char *)strstr(line, token))[strlen(token)+1], "%f", (float *)value);
      if (nRead != 1) {
	fprintf(stderr, "Unable to parse config file line \"%s\"\n", line);
	return(FALSE);
      }
      break;
    case STRING_TOKEN:
      nRead = sscanf(&((char *)strstr(line, token))[strlen(token)+1], "%s", (char *)value);
      if (nRead != 1) {
	fprintf(stderr, "Unable to parse config file line \"%s\"\n", line);
	return(FALSE);
      }
      break;
    default:
      fprintf(stderr, "Unrecognized type (%d) passed to tokenCheck\n", type);
    }
    return(TRUE);
  } else
    return(FALSE);
} /* End of  T O K E N  C H E C K */

/*
  R E A D  S E T T I N G S

  Read the settings file and set the variables stored in it.
*/
void readSettings(char *fileName)
{
  int eOF = FALSE;
  int lineNumber = 0;
  int settingsFD;
  char inLine[100];

  settingsFD = open(fileName, O_RDONLY);
  if (settingsFD < 0) {
    /* Not having a settings file is nonfatal - the default values will do. */
    perror("settings");
    return;
  }
  while (!eOF) {
    lineNumber++;
    if (getLine(settingsFD, TRUE, &inLine[0], &eOF))
      if (strlen(inLine) > 0) {
	tokenCheck(inLine, "MY_HEIGHT",          DOUBLE_TOKEN, &myHeight);
	tokenCheck(inLine, "MY_TARGET",          DOUBLE_TOKEN, &myTarget);
	tokenCheck(inLine, "WEIGHT_KG",             INT_TOKEN, &weightkg);
	tokenCheck(inLine, "HEIGHT_CM",             INT_TOKEN, &heightcm);
	tokenCheck(inLine, "MONTH_FIRST",           INT_TOKEN, &monthFirst);
	tokenCheck(inLine, "NONJUDGEMENTAL_COLORS", INT_TOKEN, &nonjudgementalColors);
	tokenCheck(inLine, "HACKER_DIET_MODE",      INT_TOKEN, &hackerDietMode);
	tokenCheck(inLine, "SHOW_COMMENTS",         INT_TOKEN, &showComments);
	tokenCheck(inLine, "SHOW_TARGET",           INT_TOKEN, &showTarget);
	tokenCheck(inLine, "PLOT_INTERVAL",         INT_TOKEN, &plotInterval);
	tokenCheck(inLine, "FIT_INTERVAL",          INT_TOKEN, &fitInterval);
      }
  }
  if (nonjudgementalColors)
    goodColor = badColor = OR_BLUE;
  close(settingsFD);
} /* End of  R E A D  S E T T I N G S */

/*
  C A L C  M A X I M A

  Calculate the maximum and minimum weight and time (JD) values for the
  entire data set from startTJD though all later entries.  The plot
  maxima and minima are also computed.
 */
void calcMaxima(double startTJD, double endTJD)
{
  if (nDataPoints == 1) {
    double aveJD, aveWeight;

    maxJD     = minJD     = logRoot->time;
    maxWeight = minWeight = logRoot->weight;
    aveJD = (maxJD+minJD)*0.5;
    maxJD = aveJD + 5.0;
    minJD = aveJD - 5.0;
    aveWeight = (maxWeight+minWeight)*0.5;
    maxWeight = aveWeight + 5.0;
    minWeight = aveWeight - 5.0;
  } else {
    logEntry *ptr;

    maxJD = -1.0; minJD = 1.0e30; maxWeight = -1.0; minWeight = 1.0e30;
    ptr = logRoot;
    while (ptr != NULL) {
      if (ptr->time > startTJD) {
	if (maxJD < ptr->time)
	  maxJD = ptr->time;
	if (minJD > ptr->time)
	  minJD = ptr->time;
	if (maxWeight < ptr->weight)
	  maxWeight = ptr->weight;
	if (minWeight > ptr->weight)
	  minWeight = ptr->weight;
	if (maxWeight < calculateWeightedAverage(ptr))
	  maxWeight = calculateWeightedAverage(ptr);
	if (minWeight > calculateWeightedAverage(ptr))
	  minWeight = calculateWeightedAverage(ptr);
      }
      ptr = ptr->next;
    }
  }
  if (endTJD > maxJD)
    maxJD = endTJD;
  if ((maxJD - minJD) < 1.0) {
    maxJD += 1.0;
    minJD -= 1.0;
  }
  /*
    Make sure the plot Y axis covers at least 2 weight unit increments,
    so that at least 2 Y axis label values will be plotted.
  */
  if ((maxWeight - minWeight) < 2.0) {
    maxWeight += 1.0;
    minWeight -= 1.0;
  }
  plotMaxX = maxJD + BORDER_FACTOR_X*(maxJD-minJD);
  plotMinX = minJD - BORDER_FACTOR_X*(maxJD-minJD);
  if (showTarget) {
    if (maxWeight < myTarget)
      maxWeight = myTarget;
    else if (minWeight > myTarget)
      minWeight = myTarget;
  }
  plotMaxY = maxWeight + BORDER_FACTOR_Y*(maxWeight-minWeight);
  plotMinY = minWeight - BORDER_FACTOR_Y*(maxWeight-minWeight);
  dprintf("maxWeight = %f minWeight = %f plotMaxY = %f plotMinY = %f\n",
	 maxWeight, minWeight, plotMaxY, plotMinY);
} /* End of  C A L C  M A X I M A */

/*
  R E A D  D A T A

  Read in the date, time and weight data from the data file.
*/
void readData(char *fileName)
{
  int eOF = FALSE;
  int dataFD;
  char inLine[1000];

  nDataPoints = 0;
  dprintf("Trying to open \"%s\"\n", fileName);
  dataFD = open(fileName, O_RDONLY);
  if (dataFD >= 0) {
    while (!eOF) {
      int nRead;
      float newWeight;
      char dateString[100], timeString[100];

      getLine(dataFD, FALSE, &inLine[0], &eOF);
      commasToPeriods(inLine);
      nRead = sscanf(inLine, "%s %s %f", dateString, timeString, &newWeight);
      if (nRead == 3) {
	double dayFraction, fJD;
	int hh, mm, dd, MM, yyyy, jD;
	char *commentPtr;
	char *comment = NULL;

	sscanf(timeString, "%d:%d", &hh, &mm);
	dayFraction = ((double)hh + (double)mm/60.0)/24.0;
	sscanf(dateString,"%d/%d/%d", &dd, &MM, &yyyy);
	jD = calculateJulianDate(yyyy, MM, dd);
	fJD = (double)jD + dayFraction;
	commentPtr = strstr(inLine, "# ");
	if (commentPtr) {
	  commentPtr += 2;
	  dprintf("Comment: \"%s\"\n", commentPtr);
	  comment = malloc(strlen(commentPtr)+1);
	  if (comment)
	    sprintf(comment, "%s", commentPtr);
	}
	enqueueEntry(fJD, newWeight, comment);
      }
    }
    calcMaxima(0.0, lastEntry->time);
    close(dataFD);
  } else
    perror("Data file could not be opened");
} /* End of  R E A D  D A T A */

/*
  S C R E E N  C O O R D I N A T E S

  This function is called with plot coordinates in physical units
  (time and weight) and it returns x, y coordinates in pixels on
  the weight vs time plot.
 */
void screenCoordinates(double xScale, double yScale,
		       double x, double y, int *ix, int *iy)
{
  *ix = (x - plotMinX)*xScale + LEFT_BORDER;
  *iy = displayHeight - ((y - plotMinY)*yScale + BOTTOM_BORDER) - 2;
} /* End of  S C R E E N  C O O R D I N A T E S */

/*
  B M I

  This function is passed a weight, and returns a Body Mass Index
  value.
 */
double bMI(double weight)
{
  double height, tWeight;

  if (weightkg)
    tWeight = weight/KG_PER_LB;
  else
    tWeight = weight;
  if (heightcm)
    height = myHeight/2.54;
  else
    height = myHeight;
  return (tWeight*KG_PER_LB/(height*height*M_PER_IN*M_PER_IN));
} /* End of  B M I */

/*
  W E I G H T

  This function is called with a BMI value, and it returns the
  corresponding weight.
 */
double weight(double bMI)
{
  double height;

  if (heightcm)
    height = myHeight/2.54;
  else
    height = myHeight;
  if (weightkg)
    return (bMI*height*height*M_PER_IN*M_PER_IN);
  else
    return (bMI*height*height*M_PER_IN*M_PER_IN/KG_PER_LB);
} /* End of  W E I G H T */

/*
  C A L C  S T A T S

  This function is passed a number of days, and it calculates the
  weight loss statistics from a period that far in the past, until
  the current time.
 */
int calcStats(int days, 
	      float *startWeight, float *endWeight,
	      float *maxWeight, float *minWeight, float *aveWeight)
{
  int doStart, doEnd, doMax, doMin, doAve;
  int nSamples = 0;
  double earliestTJD, dDays;
  logEntry *ptr;

  if (logRoot == NULL)
    return(FALSE);
  if (days < 0)
    dDays = lastEntry->time - logRoot->time;
  else
    dDays = (double)days;
  if ((lastEntry->time - logRoot->time) < dDays)
    return(FALSE);
  if (startWeight == NULL) doStart = FALSE; else doStart = TRUE;
  if (endWeight   == NULL) doEnd   = FALSE; else doEnd   = TRUE;
  if (maxWeight   == NULL) doMax   = FALSE; else doMax   = TRUE;
  if (minWeight   == NULL) doMin   = FALSE; else doMin   = TRUE;
  if (aveWeight   == NULL) doAve   = FALSE; else doAve   = TRUE;
  if (doMax)
    *maxWeight = -1.0e30;
  if (doMin)
    *minWeight = 1.0e30;
  if (doAve)
    *aveWeight = 0.0;
  earliestTJD = lastEntry->time - dDays;
  ptr = lastEntry;
  while (ptr->time > earliestTJD)
    ptr = ptr->last;
  if (doStart)
    *startWeight = calculateWeightedAverage(ptr);
  if (doEnd)
    *endWeight = calculateWeightedAverage(lastEntry);
  while (ptr != NULL) {
    float trendWeight;

    trendWeight = calculateWeightedAverage(ptr);
    if (doMax)
      if (*maxWeight < trendWeight)
	*maxWeight = trendWeight;
    if (doMin)
      if (*minWeight > trendWeight)
	*minWeight = trendWeight;
    if (doAve) {
      *aveWeight += trendWeight;
      nSamples++;
    }
    ptr = ptr->next;
  }
  if (doAve && (nSamples > 0))
    *aveWeight /= (double)nSamples;
  if (doStart)
    dprintf("Returning startWeight = %f\n", *startWeight);
  if (doEnd)
    dprintf("Returning endWeight = %f\n", *endWeight);
  if (doMax)
    dprintf("Returning maxWeight = %f\n", *maxWeight);
  if (doMin)
    dprintf("Returning minWeight = %f\n", *minWeight);
  if (doAve)
    dprintf("Returning aveWeight = %f\n", *aveWeight);
  return(TRUE);
} /* End of  C A L C  S T A T S */

/*
  M A K E  M O N T H  S T R I N G

  Put the year or abreviated month name into the scratch string
  for printing on the plot.   Return the x coordinate of the
  first day of the month (screen coordinate system).
*/
int makeMonthString(double xScale, double yScale, double jD)
{
  int i, year, month, day, x, y;

  tJDToDate(jD, &year, &month, &day);
  if (month == 1)
    /* For January, print the year, rather than the month name */
    sprintf(scratchString, "%d", year);
  else {
    /* Abbreviate the month name to 3 characters */
    for (i = 0; i < 3; i++)
      scratchString[i] = monthName[month-1][i];
    scratchString[3] = (char)0;
  }
  i = calculateJulianDate(year, month, 1);
  screenCoordinates(xScale, yScale, (double)i, minWeight, &x, &y);
  return(x);
} /* End of  M A K E  M O N T H  S T R I N G */

/*
  R E D R A W  S C R E E N

  Redraw the weight vs time plot.   This is the default display.
 */
void redrawScreen(void)
{
  int nPoints = 0;
  int inc, onlyOneMonthPlotted;
  int gotGoal = FALSE;
  int nCurvePoints = 0;
  int minPlottableY, maxPlottableY, minPlottableX, maxPlottableX;
  int i, x, y, tWidth, tHeight, weightStart, weightEnd, bMIStart, bMIEnd;
  int sYear, sMonth, sDay, eYear, eMonth, eDay, plotCenter;
  float weightNow, weightLastWeek;
  double plotTimeInterval, plotWeightInterval, xScale, yScale, goalDate;
  double labelDate, lastLabelDate;
  logEntry *ptr;
  logEntry *firstPlottable = NULL;
  GdkPoint *dataPoints, *linePoints, *lineTops, box[4];

  /* Clear the entire display */
  gdk_draw_rectangle(pixmap, drawingArea->style->black_gc,
		     TRUE, 0, 0,
		     drawingArea->allocation.width,
		     drawingArea->allocation.height);

  if (logRoot == NULL) {
    renderPangoText("No data has been entered yet", OR_WHITE, BIG_PANGO_FONT,
		    &tWidth, &tHeight, pixmap, displayWidth/2, 50, 0.0, TRUE, 0, OR_BLACK);
    renderPangoText("select the Data Entry menu option", OR_WHITE, BIG_PANGO_FONT,
		    &tWidth, &tHeight, pixmap, displayWidth/2, 100, 0.0, TRUE, 0, OR_BLACK);
    renderPangoText("to enter initial entries.", OR_WHITE, BIG_PANGO_FONT,
		    &tWidth, &tHeight, pixmap, displayWidth/2, 150, 0.0, TRUE, 0, OR_BLACK);
    renderPangoText("Also, use the Settings menu", OR_WHITE, BIG_PANGO_FONT,
		    &tWidth, &tHeight, pixmap, displayWidth/2, 200, 0.0, TRUE, 0, OR_BLACK);
    renderPangoText("to set your height and units.", OR_WHITE, BIG_PANGO_FONT,
		    &tWidth, &tHeight, pixmap, displayWidth/2, 250, 0.0, TRUE, 0, OR_BLACK);
  } else {
    double slope, intercept;
    double earliestTJDToPlot = 0.0;

    earliestTJDToPlot = lastEntry->time - plotInterval;
    if (earliestTJDToPlot < 0.0)
      earliestTJDToPlot = logRoot->time;
    if ((fitInterval > 0) && showTarget &&
	(gotGoal = calculateGoalDate(&goalDate, NULL, NULL, NULL, NULL, NULL, &slope, &intercept)))
      calcMaxima(earliestTJDToPlot-1.0, goalDate);
    else
      calcMaxima(earliestTJDToPlot-1.0, lastEntry->time);
    plotWeightInterval = maxWeight - minWeight;
    dataPoints = (GdkPoint *)malloc(nDataPoints * sizeof(GdkPoint));
    if (dataPoints == NULL) {
      perror("malloc of dataPoints");
      exit(ERROR_EXIT);
    }
    linePoints = (GdkPoint *)malloc(nDataPoints * sizeof(GdkPoint));
    if (linePoints == NULL) {
      perror("malloc of linePoints");
      exit(ERROR_EXIT);
    }
    lineTops = (GdkPoint *)malloc(nDataPoints * sizeof(GdkPoint));
    if (lineTops == NULL) {
      perror("malloc of lineTops");
      exit(ERROR_EXIT);
    }
    xScale = ((double)displayWidth - LEFT_BORDER - RIGHT_BORDER - 1)/(plotMaxX - plotMinX);
    yScale = ((double)displayHeight - TOP_BORDER - BOTTOM_BORDER - 4)/(plotMaxY - plotMinY);

    /* Draw the empty plot box */
    screenCoordinates(xScale, yScale, plotMinX, plotMinY, &box[0].x, &box[0].y);
    minPlottableX = box[0].x;
    minPlottableY = box[0].y;
    screenCoordinates(xScale, yScale, plotMinX, plotMaxY, &box[1].x, &box[1].y);
    maxPlottableY = box[1].y;
    screenCoordinates(xScale, yScale, plotMaxX, plotMaxY, &box[2].x, &box[2].y);
    maxPlottableX = box[2].x;
    screenCoordinates(xScale, yScale, plotMaxX, plotMinY, &box[3].x, &box[3].y);
    gdk_draw_polygon(pixmap, gC[OR_DARK_GREY], TRUE, box, 4);
    gdk_draw_polygon(pixmap, gC[OR_WHITE], FALSE, box, 4);
    /* plotCenter will be used to position the comments, if they are plotted */
    plotCenter = (box[0].y + box[1].y + box[2].y + box[3].y)/4;

    /* Draw the plot title */
    tJDToDate((double)((int)logRoot->time), &sYear, &sMonth, &sDay);
    if (logRoot->next == NULL) {
      eDay = sDay; eMonth = sMonth; eYear = sYear;
      firstPlottable = logRoot;
      onlyOneMonthPlotted = TRUE;
    } else {
      /* Find the first data point which should be plotted. */
      ptr = lastEntry;
      while (ptr != NULL) {
	if (ptr->time >= earliestTJDToPlot)
	  firstPlottable = ptr;
	ptr = ptr->last;
      }
      /* Generate the starting and stopping dates for the plot. */
      tJDToDate((double)((int)firstPlottable->time), &sYear, &sMonth, &sDay);
      if (gotGoal)
	tJDToDate(goalDate, &eYear, &eMonth, &eDay);
      else
	tJDToDate((double)((int)lastEntry->time), &eYear, &eMonth, &eDay);
      if (sMonth == eMonth)
	onlyOneMonthPlotted = TRUE;
      else
	onlyOneMonthPlotted = FALSE;
    }
    if (onlyOneMonthPlotted)
      sprintf(scratchString, "Weight and Trend Data for %s %d", monthName[sMonth-1], sYear);
    else {
      tJDToDate((double)((int)lastEntry->time), &eYear, &eMonth, &eDay);
      if (monthFirst) {
	int temp;
	
	temp = sDay;
	sDay = sMonth;
	sMonth = temp;
	temp = eDay;
	eDay = eMonth;
	eMonth = temp;
      }
      sprintf(scratchString, "Weight and Trend Data for %d/%d/%d through %d/%d/%d",
	      sDay, sMonth, sYear, eDay, eMonth, eYear);
    }
    renderPangoText(scratchString, OR_WHITE, SMALL_PANGO_FONT,
		    &tWidth, &tHeight, pixmap, displayWidth/2, 10, 0.0, TRUE, 0, OR_BLACK);
    renderPangoText(scratchString, OR_WHITE, SMALL_PANGO_FONT,
		    &tWidth, &tHeight, pixmap, displayWidth/2, 10, 0.0, TRUE, 0, OR_BLACK);
    if (showTarget) {
      int x0, y0, x1, y1;

      if (gotGoal) {
	double leftY;

	/* Draw a line showing the least squares fit line */
	leftY = intercept + slope*plotMinX;
	if (leftY > plotMaxY) {
	  double leftX;

	  leftX = (plotMaxY - intercept)/slope;
	  screenCoordinates(xScale, yScale, leftX, plotMaxY, &x0, &y0);
	} else if (leftY < plotMinY) {
	  double leftX;

	  leftX = (plotMinY - intercept)/slope;
	  screenCoordinates(xScale, yScale, leftX, plotMinY, &x0, &y0);
	} else
	  screenCoordinates(xScale, yScale, plotMinX, leftY, &x0, &y0);
	screenCoordinates(xScale, yScale, goalDate, myTarget, &x1, &y1);
	gdk_draw_line(pixmap, gC[OR_PURPLE], x0, y0, x1, y1);
      }
      /* Draw a thick green line at the target weight */
      screenCoordinates(xScale, yScale, plotMinX, myTarget, &x0, &y0);
      screenCoordinates(xScale, yScale, plotMaxX, myTarget, &x1, &y1);
      box[0].x = x0;       box[0].y = y0-1;
      box[1].x = x1;       box[1].y = box[0].y;
      box[2].x = box[1].x; box[2].y = y0+1;
      box[3].x = box[0].x; box[3].y = box[2].y;
      gdk_draw_polygon(pixmap, gC[OR_GREEN], TRUE, box, 4);
    }

    /* Y axis labels */
    ptr = logRoot;
    weightStart = (int)(minWeight + 0.5);
    bMIStart    = (int)(10.0*(int)(bMI(minWeight) - 0.5));
    weightEnd   = (int)(maxWeight + 0.5);
    bMIEnd      = (int)(10.0*(int)(bMI(maxWeight) + 0.5));
    if (weightStart < weightEnd)
      for (i = weightStart; i <= weightEnd; i += 1) {
	int color, stub;
	
	sprintf(scratchString, "%3d", i);
	screenCoordinates(xScale, yScale, logRoot->time, (double)i, &x, &y);
	if ((y <= minPlottableY) && (y >= maxPlottableY)) {
	  if (i % 10 == 0) {
	    color = OR_WHITE;
	    stub = 10;
	    renderPangoText(scratchString, OR_WHITE, SMALL_PANGO_FONT,
			    &tWidth, &tHeight, pixmap, 0, y, 0.0, FALSE, 0, OR_BLACK);
	  } else if (i % 5 == 0) {
	    stub = 8;
	    if (plotWeightInterval < 20.0)
	      renderPangoText(scratchString, OR_WHITE, SMALL_PANGO_FONT,
			      &tWidth, &tHeight, pixmap, 0, y, 0.0, FALSE, 0, OR_BLACK);
	  } else {
	    stub = 5;
	    if (plotWeightInterval < 5.0)
	      renderPangoText(scratchString, OR_WHITE, SMALL_PANGO_FONT,
			      &tWidth, &tHeight, pixmap, 0, y, 0.0, FALSE, 0, OR_BLACK);
	  }
	  gdk_draw_line(pixmap, gC[OR_WHITE], LEFT_BORDER, y, LEFT_BORDER+stub, y);
	}
      }
    if (bMIStart != bMIEnd) {
      int bMIInc;
      
      if ((bMIStart - bMIEnd) <= -150)
	bMIInc = 20;
      else if ((bMIStart - bMIEnd) <= -60)
	bMIInc = 10;
      else if ((bMIStart - bMIEnd) <= -20)
	bMIInc = 5;
      else
	bMIInc = 1;
      for (i = bMIStart; i <= bMIEnd; i += bMIInc) {
	double tWeight;
	int color, stub;

	tWeight = weight(0.1 * (double)i);
	screenCoordinates(xScale, yScale, logRoot->time, tWeight, &x, &y);
	if ((y > TOP_BORDER) && (y < displayHeight-BOTTOM_BORDER)) {
	  sprintf(scratchString, "%4.1f", 0.1*(double)i);
	  if (i % 10 == 0) {
	    if (nonjudgementalColors)
	      color = OR_WHITE;
	    else if (i >= 300)
	      color = OR_RED;
	    else if (i >= 250)
	      color = OR_YELLOW;
	    else if (i <= 160)
	      color = OR_RED;
	    else if (i <= 200)
	      color = OR_YELLOW;
	    else
	      color = OR_GREEN;
	    stub = 10;
	  } else {
	    if (nonjudgementalColors)
	      color = OR_BLUE;
	    else if (i >= 300)
	      color = OR_RED;
	    else if (i >= 250)
	      color = OR_YELLOW;
	    else if (i <= 160)
	      color = OR_RED;
	    else if (i <= 200)
	      color = OR_YELLOW;
	    else
	      color = OR_GREEN;
	    stub = 5;
	  }
	  x = displayWidth-RIGHT_BORDER;
	  renderPangoText(scratchString, color, SMALL_PANGO_FONT,
			  &tWidth, &tHeight, pixmap, x+2, y, 0.0, FALSE, 0, OR_BLACK);
	  gdk_draw_line(pixmap, gC[OR_WHITE], x, y, x-stub, y);
	}
      }
    }

    /* X axis labelling */
    if (onlyOneMonthPlotted) {
      /*
	All the data to be plotted falls within one month, so we should
	show the full month name, and the days within the plot range
      */
      int day, x, y;
      double jD;

      tJDToDate((double)((int)logRoot->time), &sYear, &sMonth, &sDay);
      for (day = 1; day <= monthLengths[sMonth-1]; day++) {
	jD = (double)calculateJulianDate(sYear, sMonth, day);
	screenCoordinates(xScale, yScale, jD, plotMinY, &x, &y);
	if ((x > minPlottableX) && (x < maxPlottableX)) {
	  gdk_draw_line(pixmap, gC[OR_WHITE], x, y, x, y-4);
	  sprintf(scratchString, "%d", day);
	  renderPangoText(scratchString, OR_WHITE, SMALL_PANGO_FONT,
			  &tWidth, &tHeight, pixmap, x, y+14, 0.0, TRUE, 0, OR_BLACK);
	}
      }
    } else {
      /* More than one month covered by the plot */
      x = makeMonthString(xScale, yScale, firstPlottable->time);
      if (gotGoal && (goalDate > maxJD))
	plotTimeInterval = goalDate - minJD;
      else
	plotTimeInterval = maxJD - minJD;
      tJDToDate(ptr->time, &sYear, &sMonth, &sDay);
      if (x < RIGHT_BORDER)
	x = RIGHT_BORDER;
      if (((plotTimeInterval < 5.0) || (sDay < 26)) && ((plotMaxX - plotMinX) < 180.0))
	renderPangoText(scratchString, OR_WHITE, SMALL_PANGO_FONT,
			&tWidth, &tHeight, pixmap, x, displayHeight-35, 0.0, TRUE, 0, OR_BLACK);
      labelDate = firstPlottable->time;
      if (gotGoal)
	lastLabelDate = goalDate;
      else
	lastLabelDate = lastEntry->time;
      for (labelDate = earliestTJDToPlot; labelDate <= lastLabelDate; labelDate += 1.0) {
	int plotIt;
	
	tJDToDate(labelDate, &eYear, &eMonth, &eDay);
	if (sMonth != eMonth) {
	  sMonth = eMonth;
	  x = makeMonthString(xScale, yScale, labelDate);
	  if (plotTimeInterval < 500.0)
	    plotIt = TRUE;
	  else if ((plotTimeInterval >= 500.0) && (plotTimeInterval < 1000.0) && ((sMonth-1) % 2) == 0)
	    plotIt = TRUE;
	  else if ((plotTimeInterval >= 1000.0) && (plotTimeInterval < 1500.0) && ((sMonth-1) % 3) == 0)
	    plotIt = TRUE;
	  else if ((plotTimeInterval >= 1500.0) && (plotTimeInterval < 2000.0) && ((sMonth-1) % 4) == 0)
	    plotIt = TRUE;
	  else if ((plotTimeInterval >= 2000.0) && (plotTimeInterval < 2500.0) && ((sMonth-1) % 6) == 0)
	    plotIt = TRUE;
	  else if ((plotTimeInterval >= 2500.0) && (sMonth == 1)) {
	    if ((plotTimeInterval < 7000.0)
		|| ((plotTimeInterval < 10000.0) && (eYear % 2 == 0))
		|| ((plotTimeInterval < 20000.0) && (eYear % 4 == 0))
		|| (eYear % 10 == 0))
	    plotIt = TRUE;
	  } else
	    plotIt = FALSE;
	  if (plotIt) {
	    renderPangoText(scratchString, OR_WHITE, SMALL_PANGO_FONT,
			    &tWidth, &tHeight, pixmap, x, displayHeight-35, 0.0, TRUE, 0, OR_BLACK);
	    if (eMonth == 1) {
	      /* Draw a little triangle to mark the start of a year */
	      box[0].x = x-5; box[0].y = displayHeight - BOTTOM_BORDER - 2;
	      box[1].x = x+5; box[1].y = box[0].y;
	      box[2].x = x;   box[2].y = box[1].y - 10;
	      gdk_draw_polygon(pixmap, gC[OR_WHITE], TRUE, box, 3);
	    } else {
	      /* Just draw a tickmark for a month */
	      y = displayHeight - BOTTOM_BORDER - 2;
	      gdk_draw_line(pixmap, gC[OR_WHITE], x, y, x, y-10);
	    }
	  }
	}
      }
    } /* end of multimonth plot labelling */

    /* Put together the summary line for the bottom of the plot. */
    if (calcStats(7, &weightLastWeek, &weightNow, NULL, NULL, NULL)) {
      if (weightNow < weightLastWeek) {
	if (weightkg)
	  sprintf(scratchString,
		  "Current Weight %5.1f kg, Last week's weight loss %4.2f kg, Daily deficit %1.0f calories",
		  weightNow, weightLastWeek - weightNow, (weightLastWeek - weightNow)*500.0/KG_PER_LB);
	else
	  sprintf(scratchString,
		  "Current Weight %5.1f lbs, Last week's weight loss %4.2f lbs, Daily deficit %1.0f calories",
		  weightNow, weightLastWeek - weightNow, (weightLastWeek - weightNow)*500.0);
      } else {
	if (weightkg)
	  sprintf(scratchString,
		  "Current Weight %5.1f kg, Last week's weight gain %4.2f kg, Daily excess %1.0f calories",
		  weightNow, -(weightLastWeek - weightNow), -(weightLastWeek - weightNow)*500.0/KG_PER_LB);
	else
	  sprintf(scratchString,
		  "Current Weight %5.1f lbs, Last week's weight gain %4.2f lbs, Daily excess %1.0f calories",
		  weightNow, -(weightLastWeek - weightNow), -(weightLastWeek - weightNow)*500.0);
      }
      renderPangoText(scratchString, OR_WHITE, SMALL_PANGO_FONT,
		      &tWidth, &tHeight, pixmap, displayWidth/2, displayHeight-13, 0.0, TRUE, 0, OR_BLACK);
      renderPangoText(scratchString, OR_WHITE, SMALL_PANGO_FONT,
		      &tWidth, &tHeight, pixmap, displayWidth/2, displayHeight-13, 0.0, TRUE, 0, OR_BLACK);
    }

    ptr = logRoot;
    while (ptr != NULL) {
      if (ptr->time >= earliestTJDToPlot) {
	screenCoordinates(xScale, yScale, ptr->time, ptr->weight,
			  &dataPoints[nPoints].x, &dataPoints[nPoints].y);
	screenCoordinates(xScale, yScale, ptr->time, calculateWeightedAverage(ptr),
			  &linePoints[nCurvePoints].x, &linePoints[nCurvePoints].y);
	if ((linePoints[nCurvePoints].y > maxPlottableY)
	    && (linePoints[nCurvePoints].y < minPlottableY)) {
	  lineTops[nPoints].y = linePoints[nCurvePoints].y;
	  nCurvePoints++;
	} else if (linePoints[nCurvePoints].y < maxPlottableY)
	  lineTops[nPoints].y = maxPlottableY;
	else
	  lineTops[nPoints].y = minPlottableY;
	nPoints++;
      }
      ptr = ptr->next;
    }
    if (showComments) {
      int x, y;
      int firstComment = TRUE;

      ptr = logRoot;
      while (ptr != NULL) {
	if (ptr->time > earliestTJDToPlot) {
	  if (ptr->comment) {
	    screenCoordinates(xScale, yScale, ptr->time, calculateWeightedAverage(ptr), &x, &y);
	    renderPangoText(ptr->comment, OR_PINK, SMALL_PANGO_FONT,
			    &tWidth, &tHeight, pixmap, x, plotCenter,
			    -M_PI*0.5, TRUE, 0, OR_DARK_GREY);
	    if (firstComment) {
	      renderPangoText(ptr->comment, OR_PINK, SMALL_PANGO_FONT,
			      &tWidth, &tHeight, pixmap, x, plotCenter,
			      -M_PI*0.5, TRUE, 0, OR_DARK_GREY);
	      firstComment = FALSE;
	    }
	  }
	}
	ptr = ptr->next;
      }
    }
    gdk_draw_lines(pixmap, gC[OR_YELLOW], linePoints, nCurvePoints);
    if (plotTimeInterval < 14.0)
      inc = 5;
    else if (plotTimeInterval < 100.0)
      inc = 4;
    else if (plotTimeInterval < 200.0)
      inc = 3;
    else if (plotTimeInterval < 300.0)
      inc = 2;
    else
      inc = 1;
    for (i = 0; i < nPoints; i++) {
      box[0].x = dataPoints[i].x;
      box[0].y = dataPoints[i].y + inc;
      box[1].x = dataPoints[i].x + inc;
      box[1].y = dataPoints[i].y;
      box[2].x = dataPoints[i].x;
      box[2].y = dataPoints[i].y - inc;
      box[3].x = dataPoints[i].x - inc;
      box[3].y = dataPoints[i].y;
      gdk_draw_polygon(pixmap, gC[OR_WHITE], TRUE, box, 4);
      if ((plotTimeInterval < 200.0) && !gotGoal){
	if (box[0].y < lineTops[i].y - 2)
	  gdk_draw_line(pixmap, gC[badColor], box[0].x, box[0].y, box[0].x, lineTops[i].y-1);
	if (box[2].y > lineTops[i].y + 2)
	  gdk_draw_line(pixmap, gC[goodColor], box[0].x, box[2].y, box[0].x, lineTops[i].y+1);
      }
    }
    free(dataPoints); free(linePoints); free(lineTops);
  } /* End of logRoot != NULL */
  gdk_draw_drawable(drawingArea->window,
		    drawingArea->style->fg_gc[GTK_WIDGET_STATE (drawingArea)],
		    pixmap,
		    0,0,0,0,
		    displayWidth, displayHeight);
} /* End of  R E D R A W  S C R E E N */

/*
  E X P O S E  E V E N T

  This function is called when a portion of the drawing area
  is exposed.  The plot is put together through calls which
  write on pixmap, which is copied to the drawing area for
  display.
*/
static gboolean exposeEvent(GtkWidget *widget, GdkEventExpose *event)
{
  dprintf("In exposeEvnet()\n");
  gdk_draw_drawable (widget->window,
                     widget->style->fg_gc[GTK_WIDGET_STATE (widget)],
                     pixmap,
                     event->area.x, event->area.y,
                     event->area.x, event->area.y,
                     event->area.width, event->area.height);
  return(FALSE);
} /* End of  E X P O S E  E V E N T */

/*
  C O N F I G U R E  E V E N T

  This function creates and initializes a backing pixmap
  of the appropriate size for the drawing area.
*/
static int configureEvent(GtkWidget *widget, GdkEventConfigure *event)
{
  displayHeight = widget->allocation.height;
  displayWidth  = widget->allocation.width;
  if (!cairoPixmap)
    cairoPixmap = gdk_pixmap_new(widget->window, 1100, 1100, -1);
  if (pixmap)
    g_object_unref(pixmap);
  pixmap = gdk_pixmap_new(widget->window,
			  widget->allocation.width,
			  widget->allocation.height, -1);
  dprintf("In configureEvent, w: %d, h: %d\n",
	  displayWidth, displayHeight);
  redrawScreen();
  return TRUE;
} /* End of  C O N F I G U R E  E V E N T */

/*
  M A K E  G R A P H I C  C O N T E X T S

  Make all the Graphic Contexts the program will need.
*/
static void makeGraphicContexts(GtkWidget *widget)
{
  int stat, i;
  GdkGCValues gCValues;
  GdkColor gColor;
  GdkGCValuesMask gCValuesMask;

  gCValuesMask = GDK_GC_FOREGROUND;

  for (i = 0; i < N_COLORS; i++) {
    gColor.red   = orreryColorRGB[i][0];
    gColor.green = orreryColorRGB[i][1];
    gColor.blue  = orreryColorRGB[i][2];
    stat = gdk_colormap_alloc_color(widget->style->colormap,
				    &gColor,
				    FALSE,
				    TRUE);
    if (stat != TRUE) {
      fprintf(stderr, "Error allocating color %d\n", i);
      exit(ERROR_EXIT);
    }
    gCValues.foreground = gColor;
    gC[i] = gtk_gc_get(widget->style->depth,
		       widget->style->colormap,
		       &gCValues, gCValuesMask
		       );
    if (gC[i] == NULL) {
      fprintf(stderr, "gtk_gc_get failed for color %d\n", i);
      exit(ERROR_EXIT);
    }
  }
} /* End of  M A K E  G R A P H I C  C O N T E X T S */

/*
  C R E A T E  T O U C H  S E L E C T O R

  This function creates the selector for the weight "picker".
*/
static GtkWidget *createTouchSelector(void)
{
  int weight, inc;
  char weightString[10];
  GtkWidget *selector;
  GtkListStore *model;
  GtkTreeIter iter;
  HildonTouchSelectorColumn *column = NULL;

  selector = hildon_touch_selector_new();
  model = gtk_list_store_new(1, G_TYPE_STRING);
  if (logRoot != NULL) {
    if (weightkg)
      minSelectorWeight = (int)(lastEntry->weight) - 35;
    else
      minSelectorWeight = (int)(lastEntry->weight) - 75;
    if (minSelectorWeight < 5)
      minSelectorWeight = 5;
    if (weightkg)
      maxSelectorWeight = minSelectorWeight + 75;
    else
      maxSelectorWeight = minSelectorWeight + 150;
  } else {
    if (weightkg) {
      minSelectorWeight = (int)(50.0*KG_PER_LB);
      maxSelectorWeight = (int)(250.0*KG_PER_LB);
    } else {
      minSelectorWeight = 50;
      maxSelectorWeight = 250;
    }
  }
  if (weightkg)
    inc = 1;
  else
    inc = 2;
  for (weight = minSelectorWeight*10; weight <= maxSelectorWeight*10; weight += inc) {
    sprintf(weightString, "%5.1f", 0.1*(float)weight);
    gtk_list_store_append(model, &iter);
    gtk_list_store_set(model, &iter, 0, weightString, -1);
  }
  column = hildon_touch_selector_append_text_column(HILDON_TOUCH_SELECTOR(selector),
                                                     GTK_TREE_MODEL(model), TRUE);
  g_object_set(G_OBJECT(column), "text-column", 0, NULL);  
  return selector;
} /* End of  C R E A T E  T O U C H  S E L E C T O R */

/*
  W R I T E  F I L E

  Write the data entries to the file.   The function is called
  twice, to write the main file and its backup.
 */
void writeFile(FILE *dataFile)
{
 int day, month, year, hours, minutes;
  double dayFrac, jD;
  logEntry *ptr;

  ptr = logRoot;
  while (ptr != NULL) {
    dayFrac = ptr->time - (int)ptr->time;
    jD = (double)((int)ptr->time);
    hours = (int)(dayFrac*24.0);
    minutes = (int)((dayFrac - ((double)hours)/24.0)*1440.0 + 0.5);
    tJDToDate(jD, &year, &month, &day);
    if (ptr->comment == NULL)
      fprintf(dataFile, "%02d/%02d/%d %02d:%02d %5.1f\n",
	      day, month, year, hours, minutes, ptr->weight);
    else
      fprintf(dataFile, "%02d/%02d/%d %02d:%02d %5.1f # %s\n",
	      day, month, year, hours, minutes, ptr->weight, ptr->comment);
    ptr = ptr->next;
  }
  fclose(dataFile);
} /* End of  W R I T E  F I L E */

/*
  W R I T E  D A T A  F I L E

  Write the main data file and its backup.
 */
void writeDataFile(void)
{
  FILE *dataFile;

  dataFile = fopen(fileName, "w");
  if (dataFile == NULL)
    fprintf(stderr, "Cannot write weight data file (%s)\n", fileName);
  else
    writeFile(dataFile);
  dataFile = fopen(backupFile, "w");
  if (dataFile == NULL)
    fprintf(stderr, "Cannot write backup data file (%s)\n", backupFile);
  else
    writeFile(dataFile);
} /* End of  W R I T E  D A T A  F I L E */

/*
  I N S E R T  Q U E U E  E N T R Y

  Insert a new value into the data queue, and calculate the
  new data and plot extrema.   Write the data file with the
  full list of data.
 */
void insertQueueEntry(double fJD, float weight, char *comment)
{
  enqueueEntry(fJD, weight, comment);
  if (nDataPoints == 1) {
    /* If this was the first entry, the extrema have never been set. */
    maxJD = minJD = fJD;
    maxWeight = minWeight = weight;
  }
  calcMaxima(0.0, lastEntry->time);
  writeDataFile();
} /* End of  I N S E R T  Q U E U E  E N T R Y */

/*
  D A T A  E N T R Y  A C C E P T  C A L L B A C K

  This fuction is called if the accept button is pushed on the
  Data Entry page.   That is the only way to save the new data.
  The Data Entry page will be destroyed.
 */
void dataEntryAcceptCallback(GtkButton *button, gpointer userData)
{
  accepted = TRUE;
  gtk_widget_destroy(dataEntryStackable);
} /* End of  D A T A  E N T R Y  A C C E P T  C A L L B A C K */

/*
  C H E C K  D A T A

  This function reads the values from the widgets on the Data Entry
  page, to put together the values for a new entry in the weight log.
 */
void checkData(void)
{
  int iJD;
  float weight;
  double fJD;
  char *weightResult;
  char *comment = NULL;
  guint year, month, day, hours, minutes;

  if (accepted) {
    hildon_date_button_get_date((HildonDateButton *)dateButton, &year, &month, &day);
    hildon_time_button_get_time((HildonTimeButton *)timeButton, &hours, &minutes);
    month += 1;
    weightResult = (char *)hildon_button_get_value(HILDON_BUTTON(weightButton));
    sscanf(weightResult, "%f", &weight);
    iJD = calculateJulianDate(year, month, day);
    fJD = (double)iJD + ((double)hours)/24.0 + ((double)minutes)/1440.0;
    comment = (char *)gtk_entry_get_text((GtkEntry *)commentText);
    dprintf("Got %d/%d/%d  %02d:%02d %f %f \"%s\"\n",
	   day, month, year, hours, minutes, weight, fJD, comment);
    /* Discard the comment, if it is just the default comment */
    if (!strcmp(defaultComment, comment))
      comment = NULL;
    insertQueueEntry(fJD, weight, comment);
  }
  redrawScreen();
} /* End of  C H E C K  D A T A */

/*
  D A T A  E N T R Y  B U T T O N  C L I C K E D

  This funtion is called if Data Entry is selected fromt the main menu.
  It builds and displays the widgets on the Data Entry page, and
  loads them with the default values.
 */
void dataEntryButtonClicked(GtkButton *button, gpointer userData)
{
  static GtkWidget *weightSelector, *dataEntryTable;

  dprintf("in dataEntryButtonClicked()\n");
  accepted = FALSE;
  dataEntryTable = gtk_table_new(1, 5, FALSE);

  dateButton = hildon_date_button_new(HILDON_SIZE_AUTO, HILDON_BUTTON_ARRANGEMENT_VERTICAL);
  gtk_table_attach(GTK_TABLE(dataEntryTable), dateButton, 0, 1, 0, 1,
		   GTK_EXPAND, GTK_EXPAND, 0, 0);
  
  timeButton = hildon_time_button_new(HILDON_SIZE_AUTO, HILDON_BUTTON_ARRANGEMENT_VERTICAL);
  gtk_table_attach(GTK_TABLE(dataEntryTable), timeButton, 0, 1, 1, 2,
		   GTK_EXPAND, GTK_EXPAND, 0, 0);
  
  dataEntryStackable = hildon_stackable_window_new();
  g_signal_connect(G_OBJECT(dataEntryStackable), "destroy",
		   G_CALLBACK(checkData), NULL);
  weightSelector = createTouchSelector();
  weightButton = hildon_picker_button_new(HILDON_SIZE_AUTO, HILDON_BUTTON_ARRANGEMENT_VERTICAL);
  if (lastEntry == NULL)
    hildon_touch_selector_set_active(HILDON_TOUCH_SELECTOR(weightSelector), 0, 500);
  else {
    if (weightkg)
      hildon_touch_selector_set_active(HILDON_TOUCH_SELECTOR(weightSelector), 0,
				       (int)((10.0*lastEntry->weight)+0.5) - 10*minSelectorWeight);
    else
      hildon_touch_selector_set_active(HILDON_TOUCH_SELECTOR(weightSelector), 0,
				       (int)((5.0*lastEntry->weight)+0.5) - 5*minSelectorWeight);
  }
  if (weightkg)
    hildon_button_set_title(HILDON_BUTTON(weightButton), "Weight (kg)");
  else
    hildon_button_set_title(HILDON_BUTTON(weightButton), "Weight (lbs)");
  hildon_picker_button_set_selector(HILDON_PICKER_BUTTON(weightButton),
				    HILDON_TOUCH_SELECTOR(weightSelector));
  gtk_table_attach(GTK_TABLE(dataEntryTable), weightButton, 0, 1, 2, 3,
		   GTK_EXPAND, GTK_EXPAND, 0, 0);

  commentText = gtk_entry_new();
  gtk_entry_set_text((GtkEntry *)commentText, defaultComment);
  gtk_table_attach(GTK_TABLE(dataEntryTable), commentText, 0, 1, 3, 4,
		   GTK_EXPAND | GTK_FILL, GTK_FILL, 0, 0);

  dataEntryAccept = gtk_button_new_with_label("Enter This Value");
  g_signal_connect (G_OBJECT(dataEntryAccept), "clicked",
		    G_CALLBACK(dataEntryAcceptCallback), NULL);
  gtk_table_attach(GTK_TABLE(dataEntryTable), dataEntryAccept, 0, 1, 4, 5,
		   GTK_EXPAND, GTK_EXPAND, 0, 0);
  
  gtk_container_add(GTK_CONTAINER(dataEntryStackable), dataEntryTable);
  gtk_widget_show_all(dataEntryStackable);
} /* End of  D A T A  E N T R Y  B U T T O N  C L I C K E D */

/*
  T R E N D  S T A C K A B L E  D E S T R O Y E D

  This function is called when the Trends or Log page is exited.   It causes
the program to re-display the main plot page.
 */
void trendStackableDestroyed(void)
{
  /*
    If the Log page is the one which was displayed, we must do the garbage
    collection for the dynamically allocated structures.
   */
  if (logDisplayed) {
    logCell *ptr, *tptr;
    
    ptr = logCellRoot;
    while (ptr != NULL) {
      tptr = ptr;
      ptr  = ptr->next;
      free(tptr);
    }
    logCellRoot = NULL;
    logDisplayed = FALSE;
  }
  gtk_widget_ref(mainBox);
  gtk_container_remove(GTK_CONTAINER(trendStackable), mainBox);
  gtk_container_add(GTK_CONTAINER(window), mainBox);
  gtk_widget_show(mainBox);
  gtk_widget_unref(mainBox);
  mainBoxInTrendStackable = FALSE;
  gdk_draw_drawable(drawingArea->window,
		    drawingArea->style->fg_gc[GTK_WIDGET_STATE (drawingArea)],
		    pixmap,
		    0,0,0,0,
		    displayWidth, displayHeight);
} /* End of  T R E N D  S T A C K A B L E  D E S T R O Y E D */

/*
  D A Y  O F  W E E K

  Return the day of the week (0->6) for year, month, day
*/
int dayOfWeek(int year, int month, int day)
{
  int century, c, weekDay, yy, y, m, leap;

  century = year / 100;
  yy = year - century*100;
  c = 2*(3 - (century % 4));
  y = yy + yy/4;
  if ((year % 400) == 0)
    leap = TRUE;
  else if ((year % 100) == 0)
    leap = FALSE;
  else if ((year % 4) == 0)
    leap = TRUE;
  else
    leap = FALSE;
  if (leap && (month < 3))
    m = (int)leapYear[month-1];
  else
    m = (int)normalYear[month-1];
  weekDay = (c + y + m + day) % 7;
  return(weekDay);
} /* End of  D A Y  O F  W E E K */

/*
  L O G  C A L L B A C K

  This routine builds the page which displays the weight values for a particular
  month.
 */
void logCallback(void)
{
  int year, month, tWidth, tHeight;
  int row = 0;
  int col = 0;
  char *logMonthString;
  char monthString[10];
  logEntry *ptr;
  logCell *cellPtr;
  GdkPoint box[4];

  dprintf("In logCallback()\n");
  logDisplayed = TRUE;
  logMonthString = (char *)hildon_button_get_value(HILDON_BUTTON(logSelectorButton));
  logMonthString[strlen(logMonthString)-1] = (char)0;
  dprintf("\"%s\" was selected.\n", logMonthString);
  sscanf(logMonthString, "%s %d", &monthString[0], &year);
  dprintf("Month Name \"%s\", Year: %d\n", monthString, year);
  for (month = 0; month < 12; month++)
    if (!strcmp(monthString, monthName[month]))
      break;
  month++;
  dprintf("Month number %d\n", month);
  gtk_widget_destroy(logStackable);
  if (!mainBoxInTrendStackable) {
    trendStackable = hildon_stackable_window_new();
    g_signal_connect(G_OBJECT(trendStackable), "destroy",
		     G_CALLBACK(trendStackableDestroyed), NULL);
    gtk_widget_ref(mainBox);
    gtk_widget_hide(mainBox);
    gtk_container_remove(GTK_CONTAINER(window), mainBox);
    gtk_container_add(GTK_CONTAINER(trendStackable), mainBox);
    gtk_widget_unref(mainBox);
    mainBoxInTrendStackable = TRUE;
  }
  gtk_widget_show_all(trendStackable);
  gdk_draw_rectangle(pixmap, drawingArea->style->black_gc,
		     TRUE, 0, 0,
		     drawingArea->allocation.width,
		     drawingArea->allocation.height);
  renderPangoText(logMonthString, OR_WHITE, BIG_PANGO_FONT,
		  &tWidth, &tHeight, pixmap, displayWidth/2, 20, 0.0, TRUE, 0, OR_BLACK);
  renderPangoText(logMonthString, OR_WHITE, BIG_PANGO_FONT,
		  &tWidth, &tHeight, pixmap, displayWidth/2, 20, 0.0, TRUE, 0, OR_BLACK);
  ptr = logRoot;
  while (ptr != NULL) {
    int day, pDay, pMonth, pYear;

    tJDToDate((double)((int)ptr->time), &pYear, &pMonth, &pDay);
    if ((pMonth == month) && (pYear == year)) {
      int i;
      double trend;
      char shortDayName[4];

      day = dayOfWeek(pYear, pMonth, pDay);
      trend = calculateWeightedAverage(ptr);
      for (i = 0; i < 3; i++)
	shortDayName[i] = dayName[day][i];
      shortDayName[3] = (char)0;
      sprintf(scratchString, "%2d", pDay);
      renderPangoText(scratchString, OR_WHITE, MEDIUM_PANGO_FONT,
		      &tWidth, &tHeight, pixmap, 20+col*displayWidth/3, 70+row*30, 0.0, FALSE, 0, OR_BLACK);
      renderPangoText(shortDayName, OR_WHITE, MEDIUM_PANGO_FONT,
		      &tWidth, &tHeight, pixmap, 55+col*displayWidth/3, 70+row*30, 0.0, FALSE, 0, OR_BLACK);
      sprintf(scratchString, "%5.1f", ptr->weight);
      if (nonjudgementalColors)
	renderPangoText(scratchString, OR_BLUE, MEDIUM_PANGO_FONT,
			&tWidth, &tHeight, pixmap, 115+col*displayWidth/3, 70+row*30, 0.0, FALSE, 0, OR_BLACK);
      else if (ptr->weight <= trend)
	renderPangoText(scratchString, goodColor, MEDIUM_PANGO_FONT,
			&tWidth, &tHeight, pixmap, 115+col*displayWidth/3, 70+row*30, 0.0, FALSE, 0, OR_BLACK);
      else
	renderPangoText(scratchString, badColor, MEDIUM_PANGO_FONT,
			&tWidth, &tHeight, pixmap, 115+col*displayWidth/3, 70+row*30, 0.0, FALSE, 0, OR_BLACK);
      sprintf(scratchString, "%5.1f", trend);
      renderPangoText(scratchString, OR_YELLOW, MEDIUM_PANGO_FONT,
		      &tWidth, &tHeight, pixmap, 185+col*displayWidth/3, 70+row*30, 0.0, FALSE, 0, OR_BLACK);
      box[0].x = 17+col*displayWidth/3; box[0].y = 58+row*30;
      box[1].x = box[0].x + 228;        box[1].y = box[0].y;
      box[2].x = box[1].x;              box[2].y = box[1].y + 26;
      box[3].x = box[0].x;              box[3].y = box[2].y;
      gdk_draw_polygon(pixmap, gC[OR_DARK_GREY], FALSE, box, 4);
      box[0].x = 18+col*displayWidth/3; box[0].y = 59+row*30;
      box[1].x = box[0].x + 226;        box[1].y = box[0].y;
      box[2].x = box[1].x;              box[2].y = box[1].y + 24;
      box[3].x = box[0].x;              box[3].y = box[2].y;
      gdk_draw_polygon(pixmap, gC[OR_DARK_GREY], FALSE, box, 4);
      cellPtr = (logCell *)malloc(sizeof(logCell));
      if (cellPtr != NULL) {
	for (i = 0; i < 4; i++) {
	  cellPtr->box[i].x = box[i].x;
	  cellPtr->box[i].y = box[i].y;
	}
	cellPtr->ptr = ptr;
	cellPtr->next = NULL;
	if (logCellRoot == NULL) {
	  logCellRoot = cellPtr;
	} else {
	  lastLogCell->next = cellPtr;
	}
	lastLogCell = cellPtr;
      }
      row++;
      if (row > 10) {
	col++;
	row = 0;
      }
    }
    ptr = ptr->next;
  }
  renderPangoText("Click on an entry, if you wish to edit or delete it.", OR_BLUE, MEDIUM_PANGO_FONT,
		  &tWidth, &tHeight, pixmap, displayWidth/2, displayHeight-20, 0.0, TRUE, 0, OR_BLACK);
} /* End of  L O G  C A L L B A C K */

/*
  L O G  B U T T O N  C L I C K E D

  This function builds and displays a selector which shows every month
  for which data has been entered.
 */
void logButtonClicked(GtkButton *button, gpointer userData)
{
  int lastMonth = -1;
  int lastYear = -1;
  int year, month, day;
  int nMonths = 0;
  char monthString[50];
  logEntry *ptr;
  GtkWidget *logTable, *monthSelector;
  GtkListStore *model;
  GtkTreeIter iter;
  HildonTouchSelectorColumn *column = NULL;

  dprintf("in logButtonClicked()\n");
  if (nDataPoints > 0) {
    logStackable = hildon_stackable_window_new();
    g_signal_connect(G_OBJECT(logStackable), "destroy",
		     G_CALLBACK(logCallback), NULL);
    logTable = gtk_table_new(1, 2, FALSE);
    monthSelector = hildon_touch_selector_new();
    model = gtk_list_store_new(1, G_TYPE_STRING);
    ptr = logRoot;
    while (ptr != NULL) {
      tJDToDate(ptr->time, &year, &month, &day);
      if ((month != lastMonth) || (year != lastYear)) {
	sprintf(monthString, "%s %d\n", monthName[month-1], year);
	gtk_list_store_append(model, &iter);
	gtk_list_store_set(model, &iter, 0, monthString, -1);
	nMonths++;
	lastYear = year;
	lastMonth = month;
      }
      ptr = ptr->next;
    }
    column = hildon_touch_selector_append_text_column(HILDON_TOUCH_SELECTOR(monthSelector),
						      GTK_TREE_MODEL(model), TRUE);
    g_object_set(G_OBJECT(column), "text-column", 0, NULL);  
    logSelectorButton = hildon_picker_button_new(HILDON_SIZE_AUTO, HILDON_BUTTON_ARRANGEMENT_VERTICAL);
    hildon_touch_selector_set_active(HILDON_TOUCH_SELECTOR(monthSelector), 0, nMonths-1);
    hildon_button_set_title(HILDON_BUTTON(logSelectorButton), "Month to examine");
    hildon_picker_button_set_selector(HILDON_PICKER_BUTTON(logSelectorButton),
				      HILDON_TOUCH_SELECTOR(monthSelector));
    hildon_touch_selector_set_active(HILDON_TOUCH_SELECTOR(monthSelector), 0, nMonths);
    gtk_table_attach(GTK_TABLE(logTable), logSelectorButton, 0, 1, 0, 1,
		   GTK_EXPAND, GTK_EXPAND, 0, 0);
    gtk_container_add(GTK_CONTAINER(logStackable), logTable);
    gtk_widget_show_all(logStackable);
  }
} /* End of  L O G  B U T T O N  C L I C K E D */

/*
  T R E N D  B U T T O N  C L I C K E D

  This function builds an displays the page which shows weight trends for
  the last week, fortnight, month etc.
 */
void trendButtonClicked(GtkButton *button, gpointer userData)
{
  int tWidth, tHeight, day, month, year, line;
  int goalYear, goalMonth, goalDay, goalHour, goalMinute;
  int nItems = 0;
  int delta = 20;
  int delta2 = 13;
  float calCon;
  float startWeightW, endWeightW, maxWeightW, minWeightW, aveWeightW;
  float startWeightF, endWeightF, maxWeightF, minWeightF, aveWeightF;
  float startWeightM, endWeightM, maxWeightM, minWeightM, aveWeightM;
  float startWeightQ, endWeightQ, maxWeightQ, minWeightQ, aveWeightQ;
  float startWeightY, endWeightY, maxWeightY, minWeightY, aveWeightY;
  float startWeightH, endWeightH, maxWeightH, minWeightH, aveWeightH;
  double deltaTime, goalJD;
  GdkPoint box[4];

  dprintf("in trendButtonClicked()\n");
  if (weightkg)
    calCon = 1.0/KG_PER_LB;
  else
    calCon = 1.0;
  if (!mainBoxInTrendStackable) {
    trendStackable = hildon_stackable_window_new();
    g_signal_connect(G_OBJECT(trendStackable), "destroy",
		     G_CALLBACK(trendStackableDestroyed), NULL);
    gtk_widget_ref(mainBox);
    gtk_widget_hide(mainBox);
    gtk_container_remove(GTK_CONTAINER(window), mainBox);
    gtk_container_add(GTK_CONTAINER(trendStackable), mainBox);
    gtk_widget_unref(mainBox);
    mainBoxInTrendStackable = TRUE;
  }
  gtk_widget_show_all(trendStackable);
  gdk_draw_rectangle(pixmap, drawingArea->style->black_gc,
		     TRUE, 0, 0,
		     drawingArea->allocation.width,
		     drawingArea->allocation.height);
  renderPangoText("Trend Analysis", OR_WHITE, BIG_PANGO_FONT,
		  &tWidth, &tHeight, pixmap, displayWidth/2, 20, 0.0, TRUE, 0, OR_BLACK);
  renderPangoText("Trend Analysis", OR_WHITE, BIG_PANGO_FONT,
		  &tWidth, &tHeight, pixmap, displayWidth/2, 20, 0.0, TRUE, 0, OR_BLACK);
  if (calcStats(7, &startWeightW, &endWeightW, &maxWeightW, &minWeightW, &aveWeightW))
    nItems++;
  if (calcStats(14, &startWeightF, &endWeightF, &maxWeightF, &minWeightF, &aveWeightF))
    nItems++;
  if (calcStats(30, &startWeightM, &endWeightM, &maxWeightM, &minWeightM, &aveWeightM))
    nItems++;
  if (calcStats(91, &startWeightQ, &endWeightQ, &maxWeightQ, &minWeightQ, &aveWeightQ))
    nItems++;
  if (calcStats(365, &startWeightY, &endWeightY, &maxWeightY, &minWeightY, &aveWeightY))
    nItems++;
  if (calcStats(-1, &startWeightH, &endWeightH, &maxWeightH, &minWeightH, &aveWeightH))
    nItems++;

  box[0].x = 0;            box[0].y = 100-delta;
  box[1].x = displayWidth; box[1].y = 100-delta;
  box[2].x = displayWidth; box[2].y = displayHeight - 15 - (6-nItems)*35 - delta;
  box[3].x = 0;            box[3].y = displayHeight - 15 - (6-nItems)*35 - delta;
  gdk_draw_polygon(pixmap, gC[OR_DARK_GREY], TRUE, box, 4);

  tJDToDate((double)((int)lastEntry->time), &year, &month, &day);
  if (monthFirst) {
    int temp;
    
    temp = day;
    day = month;
    month = temp;
  }
  sprintf(scratchString, "Intervals ending %d/%d/%d", day, month, year);
  renderPangoText(scratchString, OR_WHITE, MEDIUM_PANGO_FONT,
		  &tWidth, &tHeight, pixmap, displayWidth/2, 70-delta2, 0.0, TRUE, 0, OR_BLACK);
  renderPangoText("Last...", OR_WHITE, MEDIUM_PANGO_FONT,
		  &tWidth, &tHeight, pixmap, 5, 140, 0.0, FALSE, 0, OR_DARK_GREY); 
  
  if (!nonjudgementalColors) {
    renderPangoText("Gain", badColor, MEDIUM_PANGO_FONT,
		    &tWidth, &tHeight, pixmap, 134, 125-delta, 0.0, FALSE, 0, OR_DARK_GREY); 
    renderPangoText("/", OR_WHITE, MEDIUM_PANGO_FONT,
		    &tWidth, &tHeight, pixmap, 194, 125-delta, 0.0, FALSE, 0, OR_DARK_GREY);      
    renderPangoText("Loss", goodColor, MEDIUM_PANGO_FONT,
		    &tWidth, &tHeight, pixmap, 209, 125-delta, 0.0, FALSE, 0, OR_DARK_GREY);
    if (weightkg)
      renderPangoText("kg / week", OR_WHITE, MEDIUM_PANGO_FONT,
		      &tWidth, &tHeight, pixmap, 144, 155-delta, 0.0, FALSE, 0, OR_DARK_GREY);
    else
      renderPangoText("pounds / week", OR_WHITE, MEDIUM_PANGO_FONT,
		      &tWidth, &tHeight, pixmap, 118, 155-delta, 0.0, FALSE, 0, OR_DARK_GREY);
  } else
    if (weightkg)
      renderPangoText("kg / week", OR_WHITE, MEDIUM_PANGO_FONT,
		      &tWidth, &tHeight, pixmap, 144, 140-delta, 0.0, FALSE, 0, OR_DARK_GREY);
    else
      renderPangoText("pounds / week", OR_WHITE, MEDIUM_PANGO_FONT,
		      &tWidth, &tHeight, pixmap, 118, 140-delta, 0.0, FALSE, 0, OR_DARK_GREY);

  if (!nonjudgementalColors) {
    renderPangoText("Excess", badColor, MEDIUM_PANGO_FONT,
		    &tWidth, &tHeight, pixmap, 336, 125-delta, 0.0, FALSE, 0, OR_DARK_GREY);      
    renderPangoText("/", OR_WHITE, MEDIUM_PANGO_FONT,
		    &tWidth, &tHeight, pixmap, 419, 125-delta, 0.0, FALSE, 0, OR_DARK_GREY);      
    renderPangoText("Deficit", goodColor, MEDIUM_PANGO_FONT,
		    &tWidth, &tHeight, pixmap, 433, 125-delta, 0.0, FALSE, 0, OR_DARK_GREY);      
    renderPangoText("calories / day", OR_WHITE, MEDIUM_PANGO_FONT,
		    &tWidth, &tHeight, pixmap, 348, 155-delta, 0.0, FALSE, 0, OR_DARK_GREY);
  } else
    renderPangoText("calories / day", OR_WHITE, MEDIUM_PANGO_FONT,
		    &tWidth, &tHeight, pixmap, 348, 140-delta, 0.0, FALSE, 0, OR_DARK_GREY);

  renderPangoText("Weight Trend", OR_WHITE, MEDIUM_PANGO_FONT,
		  &tWidth, &tHeight, pixmap, 601, 125-delta, 0.0, FALSE, 0, OR_DARK_GREY);      
  renderPangoText("Min.     Mean    Max.", OR_WHITE, MEDIUM_PANGO_FONT,
		  &tWidth, &tHeight, pixmap, 566, 155-delta, 0.0, FALSE, 0, OR_DARK_GREY);

  for (line = 0; line < nItems; line++) {
    float temp;

    if (line == (nItems-1)) {
      renderPangoText("History", OR_WHITE, MEDIUM_PANGO_FONT,
		      &tWidth, &tHeight, pixmap, 5, 200+(35*line)-delta, 0.0, FALSE, 0, OR_DARK_GREY);
      deltaTime = lastEntry->time - logRoot->time;
      temp = (startWeightH-endWeightH)/(deltaTime/7.0);
      if (nonjudgementalColors) {
	sprintf(scratchString, "%4.2f", -temp);
	renderPangoText(scratchString, OR_WHITE, MEDIUM_PANGO_FONT,
			&tWidth, &tHeight, pixmap, 170, 200+(35*line)-delta, 0.0, FALSE, 0, OR_DARK_GREY);
      } else {
	sprintf(scratchString, "%4.2f", fabs(temp));
	if (temp < 0.0) 
	  renderPangoText(scratchString, badColor, MEDIUM_PANGO_FONT,
			  &tWidth, &tHeight, pixmap, 170, 200+(35*line)-delta, 0.0, FALSE, 0, OR_DARK_GREY);
	else
	  renderPangoText(scratchString, goodColor, MEDIUM_PANGO_FONT,
			  &tWidth, &tHeight, pixmap, 170, 200+(35*line)-delta, 0.0, FALSE, 0, OR_DARK_GREY);
      }
      temp = calCon*(startWeightH-endWeightH)*(CALORIES_PER_POUND/deltaTime);
      if (nonjudgementalColors) {
	sprintf(scratchString, "%4.0f", -temp);
	renderPangoText(scratchString, OR_WHITE, MEDIUM_PANGO_FONT,
		      &tWidth, &tHeight, pixmap, 380, 200+(35*line)-delta, 0.0, FALSE, 0, OR_DARK_GREY);
      } else {
	sprintf(scratchString, "%4.0f", fabs(temp));
	if (temp < 0.0)
	  renderPangoText(scratchString, badColor, MEDIUM_PANGO_FONT,
			  &tWidth, &tHeight, pixmap, 380, 200+(35*line)-delta, 0.0, FALSE, 0, OR_DARK_GREY);
	else
	  renderPangoText(scratchString, goodColor, MEDIUM_PANGO_FONT,
			  &tWidth, &tHeight, pixmap, 380, 200+(35*line)-delta, 0.0, FALSE, 0, OR_DARK_GREY);
      }
      sprintf(scratchString, "%5.1f    %5.1f    %5.1f", minWeightH, aveWeightH, maxWeightH);
      renderPangoText(scratchString, OR_WHITE, MEDIUM_PANGO_FONT,
		      &tWidth, &tHeight, pixmap, 561, 200+(35*line)-delta, 0.0, FALSE, 0, OR_DARK_GREY);
    } else
      switch (line) {
      case 0:
	renderPangoText("Week", OR_WHITE, MEDIUM_PANGO_FONT,
			&tWidth, &tHeight, pixmap, 5, 200+(35*line)-delta, 0.0, FALSE, 0, OR_DARK_GREY);      
	temp = startWeightW-endWeightW;
	if (nonjudgementalColors) {
	  sprintf(scratchString, "%4.2f", -temp);
	  renderPangoText(scratchString, OR_WHITE, MEDIUM_PANGO_FONT,
			  &tWidth, &tHeight, pixmap, 170, 200+(35*line)-delta, 0.0, FALSE, 0, OR_DARK_GREY);
	} else {
	  sprintf(scratchString, "%4.2f", fabs(temp));
	  if (temp < 0.0) 
	    renderPangoText(scratchString, badColor, MEDIUM_PANGO_FONT,
			    &tWidth, &tHeight, pixmap, 170, 200+(35*line)-delta, 0.0, FALSE, 0, OR_DARK_GREY);
	  else
	    renderPangoText(scratchString, goodColor, MEDIUM_PANGO_FONT,
			    &tWidth, &tHeight, pixmap, 170, 200+(35*line)-delta, 0.0, FALSE, 0, OR_DARK_GREY);
	}
	temp = calCon*(startWeightW-endWeightW)*(CALORIES_PER_POUND/7.0);
	if (nonjudgementalColors) {
	  sprintf(scratchString, "%4.0f", -temp);
	  renderPangoText(scratchString, OR_WHITE, MEDIUM_PANGO_FONT,
			  &tWidth, &tHeight, pixmap, 380, 200+(35*line)-delta, 0.0, FALSE, 0, OR_DARK_GREY);
	} else {
	  sprintf(scratchString, "%4.0f", fabs(temp));
	  if (temp < 0.0)
	    renderPangoText(scratchString, badColor, MEDIUM_PANGO_FONT,
			    &tWidth, &tHeight, pixmap, 380, 200+(35*line)-delta, 0.0, FALSE, 0, OR_DARK_GREY);
	  else
	    renderPangoText(scratchString, goodColor, MEDIUM_PANGO_FONT,
			    &tWidth, &tHeight, pixmap, 380, 200+(35*line)-delta, 0.0, FALSE, 0, OR_DARK_GREY);
	}
	sprintf(scratchString, "%5.1f    %5.1f    %5.1f", minWeightW, aveWeightW, maxWeightW);
	renderPangoText(scratchString, OR_WHITE, MEDIUM_PANGO_FONT,
			&tWidth, &tHeight, pixmap, 561, 200+(35*line)-delta, 0.0, FALSE, 0, OR_DARK_GREY);
	break;
      case 1:
	renderPangoText("Fortnight", OR_WHITE, MEDIUM_PANGO_FONT,
			&tWidth, &tHeight, pixmap, 5, 200+(35*line)-delta, 0.0, FALSE, 0, OR_DARK_GREY);      
	temp = (startWeightF-endWeightF)/2.0;
	if (nonjudgementalColors) {
	  sprintf(scratchString, "%4.2f", -temp);
	  renderPangoText(scratchString, OR_WHITE, MEDIUM_PANGO_FONT,
			  &tWidth, &tHeight, pixmap, 170, 200+(35*line)-delta, 0.0, FALSE, 0, OR_DARK_GREY);
	} else {
	  sprintf(scratchString, "%4.2f", fabs(temp));
	  if (temp < 0.0) 
	    renderPangoText(scratchString, badColor, MEDIUM_PANGO_FONT,
			    &tWidth, &tHeight, pixmap, 170, 200+(35*line)-delta, 0.0, FALSE, 0, OR_DARK_GREY);
	  else
	    renderPangoText(scratchString, goodColor, MEDIUM_PANGO_FONT,
			    &tWidth, &tHeight, pixmap, 170, 200+(35*line)-delta, 0.0, FALSE, 0, OR_DARK_GREY);
	}
	temp = calCon*(startWeightF-endWeightF)*(CALORIES_PER_POUND/17.0);
	if (nonjudgementalColors) {
	  sprintf(scratchString, "%4.0f", -temp);
	  renderPangoText(scratchString, OR_WHITE, MEDIUM_PANGO_FONT,
			  &tWidth, &tHeight, pixmap, 380, 200+(35*line)-delta, 0.0, FALSE, 0, OR_DARK_GREY);
	} else {
	  sprintf(scratchString, "%4.0f", fabs(temp));
	  if (temp < 0.0)
	    renderPangoText(scratchString, badColor, MEDIUM_PANGO_FONT,
			    &tWidth, &tHeight, pixmap, 380, 200+(35*line)-delta, 0.0, FALSE, 0, OR_DARK_GREY);
	  else
	    renderPangoText(scratchString, goodColor, MEDIUM_PANGO_FONT,
			    &tWidth, &tHeight, pixmap, 380, 200+(35*line)-delta, 0.0, FALSE, 0, OR_DARK_GREY);
	}
	sprintf(scratchString, "%5.1f    %5.1f    %5.1f", minWeightF, aveWeightF, maxWeightF);
	renderPangoText(scratchString, OR_WHITE, MEDIUM_PANGO_FONT,
			&tWidth, &tHeight, pixmap, 561, 200+(35*line)-delta, 0.0, FALSE, 0, OR_DARK_GREY);
	break;
      case 2:
	renderPangoText("Month", OR_WHITE, MEDIUM_PANGO_FONT,
			&tWidth, &tHeight, pixmap, 5, 200+(35*line)-delta, 0.0, FALSE, 0, OR_DARK_GREY);      
	temp = (startWeightM-endWeightM)/(30.0/7.0);
	if (nonjudgementalColors) {
	  sprintf(scratchString, "%4.2f", -temp);
	  renderPangoText(scratchString, OR_WHITE, MEDIUM_PANGO_FONT,
			  &tWidth, &tHeight, pixmap, 170, 200+(35*line)-delta, 0.0, FALSE, 0, OR_DARK_GREY);
	} else {
	  sprintf(scratchString, "%4.2f", fabs(temp));
	  if (temp < 0.0) 
	    renderPangoText(scratchString, badColor, MEDIUM_PANGO_FONT,
			    &tWidth, &tHeight, pixmap, 170, 200+(35*line)-delta, 0.0, FALSE, 0, OR_DARK_GREY);
	  else
	    renderPangoText(scratchString, goodColor, MEDIUM_PANGO_FONT,
			    &tWidth, &tHeight, pixmap, 170, 200+(35*line)-delta, 0.0, FALSE, 0, OR_DARK_GREY);
	}
	temp = calCon*(startWeightM-endWeightM)*(CALORIES_PER_POUND/30.0);
	if (nonjudgementalColors) {
	  sprintf(scratchString, "%4.0f", -temp);
	  renderPangoText(scratchString, OR_WHITE, MEDIUM_PANGO_FONT,
			  &tWidth, &tHeight, pixmap, 380, 200+(35*line)-delta, 0.0, FALSE, 0, OR_DARK_GREY);
	} else {
	  sprintf(scratchString, "%4.0f", fabs(temp));
	  if (temp < 0.0)
	    renderPangoText(scratchString, badColor, MEDIUM_PANGO_FONT,
			    &tWidth, &tHeight, pixmap, 380, 200+(35*line)-delta, 0.0, FALSE, 0, OR_DARK_GREY);
	  else
	    renderPangoText(scratchString, goodColor, MEDIUM_PANGO_FONT,
			    &tWidth, &tHeight, pixmap, 380, 200+(35*line)-delta, 0.0, FALSE, 0, OR_DARK_GREY);
	}
	sprintf(scratchString, "%5.1f    %5.1f    %5.1f", minWeightM, aveWeightM, maxWeightM);
	renderPangoText(scratchString, OR_WHITE, MEDIUM_PANGO_FONT,
			&tWidth, &tHeight, pixmap, 561, 200+(35*line)-delta, 0.0, FALSE, 0, OR_DARK_GREY);
	break;
      case 3:
	renderPangoText("Quarter", OR_WHITE, MEDIUM_PANGO_FONT,
			&tWidth, &tHeight, pixmap, 5, 200+(35*line)-delta, 0.0, FALSE, 0, OR_DARK_GREY);      
	temp = (startWeightQ-endWeightQ)/13.0;
	if (nonjudgementalColors) {
	  sprintf(scratchString, "%4.2f", -temp);
	  renderPangoText(scratchString, OR_WHITE, MEDIUM_PANGO_FONT,
			  &tWidth, &tHeight, pixmap, 170, 200+(35*line)-delta, 0.0, FALSE, 0, OR_DARK_GREY);
	} else {
	  sprintf(scratchString, "%4.2f", fabs(temp));
	  if (temp < 0.0) 
	    renderPangoText(scratchString, badColor, MEDIUM_PANGO_FONT,
			    &tWidth, &tHeight, pixmap, 170, 200+(35*line)-delta, 0.0, FALSE, 0, OR_DARK_GREY);
	  else
	    renderPangoText(scratchString, goodColor, MEDIUM_PANGO_FONT,
			    &tWidth, &tHeight, pixmap, 170, 200+(35*line)-delta, 0.0, FALSE, 0, OR_DARK_GREY);
	}
	temp = calCon*(startWeightQ-endWeightQ)*(CALORIES_PER_POUND/91.0);
	if (nonjudgementalColors) {
	  sprintf(scratchString, "%4.0f", -temp);
	  renderPangoText(scratchString, OR_WHITE, MEDIUM_PANGO_FONT,
			  &tWidth, &tHeight, pixmap, 380, 200+(35*line)-delta, 0.0, FALSE, 0, OR_DARK_GREY);
	} else {
	  sprintf(scratchString, "%4.0f", fabs(temp));
	  if (temp < 0.0)
	    renderPangoText(scratchString, badColor, MEDIUM_PANGO_FONT,
			    &tWidth, &tHeight, pixmap, 380, 200+(35*line)-delta, 0.0, FALSE, 0, OR_DARK_GREY);
	  else
	    renderPangoText(scratchString, goodColor, MEDIUM_PANGO_FONT,
			    &tWidth, &tHeight, pixmap, 380, 200+(35*line)-delta, 0.0, FALSE, 0, OR_DARK_GREY);
	}
	sprintf(scratchString, "%5.1f    %5.1f    %5.1f", minWeightQ, aveWeightQ, maxWeightQ);
	renderPangoText(scratchString, OR_WHITE, MEDIUM_PANGO_FONT,
			&tWidth, &tHeight, pixmap, 561, 200+(35*line)-delta, 0.0, FALSE, 0, OR_DARK_GREY);
	break;
      case 4:
	renderPangoText("Year", OR_WHITE, MEDIUM_PANGO_FONT,
			&tWidth, &tHeight, pixmap, 5, 200+(35*line)-delta, 0.0, FALSE, 0, OR_DARK_GREY);      
	temp = (startWeightY-endWeightY)/(365.0/7.0);
	if (nonjudgementalColors) {
	  sprintf(scratchString, "%4.2f", -temp);
	  renderPangoText(scratchString, OR_WHITE, MEDIUM_PANGO_FONT,
			  &tWidth, &tHeight, pixmap, 170, 200+(35*line)-delta, 0.0, FALSE, 0, OR_DARK_GREY);
	} else {
	  sprintf(scratchString, "%4.2f", fabs(temp));
	  if (temp < 0.0) 
	    renderPangoText(scratchString, badColor, MEDIUM_PANGO_FONT,
			    &tWidth, &tHeight, pixmap, 170, 200+(35*line)-delta, 0.0, FALSE, 0, OR_DARK_GREY);
	  else
	    renderPangoText(scratchString, goodColor, MEDIUM_PANGO_FONT,
			    &tWidth, &tHeight, pixmap, 170, 200+(35*line)-delta, 0.0, FALSE, 0, OR_DARK_GREY);
	}
	temp = calCon*(startWeightY-endWeightY)*(CALORIES_PER_POUND/365.0);
	if (nonjudgementalColors) {
	  sprintf(scratchString, "%4.0f", -temp);
	  renderPangoText(scratchString, OR_WHITE, MEDIUM_PANGO_FONT,
			  &tWidth, &tHeight, pixmap, 380, 200+(35*line)-delta, 0.0, FALSE, 0, OR_DARK_GREY);
	} else {
	  sprintf(scratchString, "%4.0f", fabs(temp));
	  if (temp < 0.0)
	    renderPangoText(scratchString, badColor, MEDIUM_PANGO_FONT,
			    &tWidth, &tHeight, pixmap, 380, 200+(35*line)-delta, 0.0, FALSE, 0, OR_DARK_GREY);
	  else
	    renderPangoText(scratchString, goodColor, MEDIUM_PANGO_FONT,
			    &tWidth, &tHeight, pixmap, 380, 200+(35*line)-delta, 0.0, FALSE, 0, OR_DARK_GREY);
	}
	sprintf(scratchString, "%5.1f    %5.1f    %5.1f", minWeightY, aveWeightY, maxWeightY);
	renderPangoText(scratchString, OR_WHITE, MEDIUM_PANGO_FONT,
			&tWidth, &tHeight, pixmap, 561, 200+(35*line)-delta, 0.0, FALSE, 0, OR_DARK_GREY);
	break;
      }
    if (calculateGoalDate(&goalJD, &goalYear, &goalMonth, &goalDay, &goalHour, &goalMinute, NULL, NULL)) {
      if (goalJD < lastEntry->time)
	sprintf(scratchString, "You will never reach your target weight");
      else {
	if (!monthFirst) {
	  int temp;

	  temp = goalDay;
	  goalDay = goalMonth;
	  goalMonth = temp;
	}
	sprintf(scratchString, "You will reach your target weight at %02d:%02d on %d/%d/%d",
		goalHour, goalMinute, goalMonth, goalDay, goalYear);
      }
      renderPangoText(scratchString, OR_WHITE, MEDIUM_PANGO_FONT,
		      &tWidth, &tHeight, pixmap, displayWidth/2, displayHeight-18, 0.0, TRUE, 0, OR_BLACK);
      renderPangoText(scratchString, OR_WHITE, MEDIUM_PANGO_FONT,
		      &tWidth, &tHeight, pixmap, displayWidth/2, displayHeight-18, 0.0, TRUE, 0, OR_BLACK);
    }
  }
} /* End of  T R E N D  B U T T O N  C L I C K E D */

/*
  W R I T E  S E T T I N G S

  Write the user-selectable parameters into the settings file.
 */
void writeSettings(void)
{
  FILE *settings;

  settings = fopen(settingsFileName, "w");
  if (settings == NULL) {
    perror("writing settings");
    return;
  }
  fprintf(settings, "MY_HEIGHT %5.1f\n",          myHeight);
  fprintf(settings, "MY_TARGET %5.1f\n",          myTarget);
  fprintf(settings, "WEIGHT_KG %d\n",             weightkg);
  fprintf(settings, "HEIGHT_CM %d\n",             heightcm);
  fprintf(settings, "MONTH_FIRST %d\n",           monthFirst);
  fprintf(settings, "NONJUDGEMENTAL_COLORS %d\n", nonjudgementalColors);
  fprintf(settings, "HACKER_DIET_MODE %d\n",      hackerDietMode);
  fprintf(settings, "SHOW_COMMENTS %d\n",         showComments);
  fprintf(settings, "SHOW_TARGET %d\n",           showTarget);
  fprintf(settings, "PLOT_INTERVAL %d\n",         plotInterval);
  fprintf(settings, "FIT_INTERVAL %d\n",          fitInterval);
  fclose(settings);
} /* End of  W R I T E  S E T T I N G S */

/*
  C H E C K  S E T T I N G S

  Read the widgets from the Settings page, and set the user-selectable
  variables.
*/
void checkSettings(void)
{
  dprintf("In checkSettings\n");
  myHeight = gtk_spin_button_get_value((GtkSpinButton *)heightSpin);
  myTarget = gtk_spin_button_get_value((GtkSpinButton *)targetSpin);

  if (GTK_TOGGLE_BUTTON(plotFortnightButton)->active)
    plotInterval = 14;
  else if (GTK_TOGGLE_BUTTON(plotMonthButton)->active)
    plotInterval = 30;
  else if (GTK_TOGGLE_BUTTON(plotQuarterButton)->active)
    plotInterval = 91;
  else if (GTK_TOGGLE_BUTTON(plot6MonthButton)->active)
    plotInterval =183;
  else if (GTK_TOGGLE_BUTTON(plotYearButton)->active)
    plotInterval = 365;
  else
    plotInterval = 1000000000;

  if (GTK_TOGGLE_BUTTON(fitNothingButton)->active)
    fitInterval = -1;
  else if (GTK_TOGGLE_BUTTON(fitMonthButton)->active)
    fitInterval = 30;
  else if (GTK_TOGGLE_BUTTON(fitQuarterButton)->active)
    fitInterval = 91;
  else if (GTK_TOGGLE_BUTTON(fit6MonthButton)->active)
    fitInterval =183;
  else if (GTK_TOGGLE_BUTTON(fitYearButton)->active)
    fitInterval = 365;
  else
    fitInterval = 1000000000;

  if (GTK_TOGGLE_BUTTON(kgButton)->active)
    weightkg = TRUE;
  else
    weightkg = FALSE;
  if (GTK_TOGGLE_BUTTON(nonjudgementalButton)->active) {
    nonjudgementalColors = TRUE;
    goodColor = badColor = OR_BLUE;
  } else {
    nonjudgementalColors = FALSE;
    goodColor = OR_GREEN;
    badColor = OR_RED;
  }
  if (GTK_TOGGLE_BUTTON(hackerDietButton)->active)
    hackerDietMode = TRUE;
  else
    hackerDietMode = FALSE;
  if (GTK_TOGGLE_BUTTON(showCommentsButton)->active)
    showComments = TRUE;
  else
    showComments = FALSE;
  if (GTK_TOGGLE_BUTTON(showTargetButton)->active)
    showTarget = TRUE;
  else
    showTarget = FALSE;
  if (GTK_TOGGLE_BUTTON(cmButton)->active)
    heightcm = TRUE;
  else
    heightcm = FALSE;
  if (GTK_TOGGLE_BUTTON(monthButton)->active)
    monthFirst = TRUE;
  else
    monthFirst = FALSE;
  writeSettings();
  redrawScreen();
} /* End of  C H E C K  S E T T I N G S */

/*
  S E T T I N G S  B U T T O N  C L I C K E D

  This function builds and displays the widgets for the Settings page.
 */
void settingsButtonClicked(GtkButton *button, gpointer userData)
{
  static GtkWidget *settingsTable, *firstSeparator, *secondSeparator,
    *thirdSeparator;
  static GSList *weightGroup, *heightGroup, *dateGroup, *plotGroup,
    *fitGroup;
  static GtkObject *heightAdjustment, *targetAdjustment;

  dprintf("in settingsButtonClicked()\n");
  settingsTable = gtk_table_new(3, 13, FALSE);

  heightUnitLabel = gtk_label_new("Height Unit");
  gtk_table_attach(GTK_TABLE(settingsTable), (GtkWidget *)heightUnitLabel, 0, 1, 0, 1,
		   GTK_EXPAND | GTK_FILL, GTK_FILL, 0, 0);
  inButton = gtk_radio_button_new_with_label(NULL, "inches");
  heightGroup = gtk_radio_button_group(GTK_RADIO_BUTTON(inButton));
  cmButton = gtk_radio_button_new_with_label(heightGroup, "centimeters");
  heightGroup = gtk_radio_button_group(GTK_RADIO_BUTTON(cmButton));
  if (heightcm)
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(cmButton), TRUE);
  else
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(inButton), TRUE);
  gtk_table_attach(GTK_TABLE(settingsTable), (GtkWidget *)inButton, 1, 2, 0, 1,
		   GTK_EXPAND | GTK_FILL, GTK_FILL, 0, 0);
  gtk_table_attach(GTK_TABLE(settingsTable), (GtkWidget *)cmButton, 2, 3, 0, 1,
		   GTK_EXPAND | GTK_FILL, GTK_FILL, 0, 0);

  heightSpinLabel = gtk_label_new("Your Height");
  gtk_table_attach(GTK_TABLE(settingsTable), (GtkWidget *)heightSpinLabel, 0, 1, 1, 2,
		   GTK_EXPAND | GTK_FILL, GTK_FILL, 0, 0);
  heightAdjustment = gtk_adjustment_new(myHeight, 10.0, 216.0, 0.5, 1.0, 0.0);
  heightSpin = gtk_spin_button_new((GtkAdjustment *)heightAdjustment, 0.5, 1);
  gtk_spin_button_set_numeric((GtkSpinButton *)heightSpin, TRUE);
  gtk_table_attach(GTK_TABLE(settingsTable), (GtkWidget *)heightSpin, 1, 2, 1, 2,
		   GTK_EXPAND | GTK_FILL, GTK_FILL, 0, 0);

  weightUnitLabel = gtk_label_new("Weight Unit");
  gtk_table_attach(GTK_TABLE(settingsTable), (GtkWidget *)weightUnitLabel, 0, 1, 2, 3,
		   GTK_EXPAND | GTK_FILL, GTK_FILL, 0, 0);
  poundsButton = gtk_radio_button_new_with_label(NULL, "pounds");
  weightGroup = gtk_radio_button_group(GTK_RADIO_BUTTON(poundsButton));
  kgButton = gtk_radio_button_new_with_label(weightGroup, "kilograms");
  weightGroup = gtk_radio_button_group(GTK_RADIO_BUTTON(kgButton));
  if (weightkg)
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(kgButton), TRUE);
  else
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(poundsButton), TRUE);
  gtk_table_attach(GTK_TABLE(settingsTable), (GtkWidget *)poundsButton, 1, 2, 2, 3,
		   GTK_EXPAND | GTK_FILL, GTK_FILL, 0, 0);
  gtk_table_attach(GTK_TABLE(settingsTable), (GtkWidget *)kgButton, 2, 3, 2, 3,
		   GTK_EXPAND | GTK_FILL, GTK_FILL, 0, 0);

  targetSpinLabel = gtk_label_new("Target Weight");
  gtk_table_attach(GTK_TABLE(settingsTable), (GtkWidget *)targetSpinLabel, 0, 1, 3, 4,
		   GTK_EXPAND | GTK_FILL, GTK_FILL, 0, 0);
  targetAdjustment = gtk_adjustment_new(myTarget, 60.0, 250.0, 0.5, 1.0, 0.0);
  targetSpin = gtk_spin_button_new((GtkAdjustment *)targetAdjustment, 0.5, 1);
  gtk_spin_button_set_numeric((GtkSpinButton *)targetSpin, TRUE);
  gtk_table_attach(GTK_TABLE(settingsTable), (GtkWidget *)targetSpin, 1, 2, 3, 4,
		   GTK_EXPAND | GTK_FILL, GTK_FILL, 0, 0);
  showTargetButton = gtk_check_button_new_with_label("Plot Target Weight");
  if (showTarget)
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(showTargetButton), TRUE);
  else
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(showTargetButton), FALSE);
  gtk_table_attach(GTK_TABLE(settingsTable), showTargetButton, 2, 3, 3, 4,
		   GTK_EXPAND | GTK_FILL, GTK_FILL, 0, 0);

  dateFormatLabel = gtk_label_new("Date Format");
  gtk_table_attach(GTK_TABLE(settingsTable), (GtkWidget *)dateFormatLabel, 0, 1, 4, 5,
		   GTK_EXPAND | GTK_FILL, GTK_FILL, 0, 0);
  dayButton = gtk_radio_button_new_with_label(NULL, "dd/mm/yyyy");
  dateGroup = gtk_radio_button_group(GTK_RADIO_BUTTON(dayButton));
  monthButton = gtk_radio_button_new_with_label(dateGroup, "mm/dd/yyyy");
  dateGroup = gtk_radio_button_group(GTK_RADIO_BUTTON(monthButton));
  if (monthFirst)
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(monthButton), TRUE);
  else
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(dayButton), TRUE);
  gtk_table_attach(GTK_TABLE(settingsTable), (GtkWidget *)dayButton, 1, 2, 4, 5,
		   GTK_EXPAND | GTK_FILL, GTK_FILL, 0, 0);
  gtk_table_attach(GTK_TABLE(settingsTable), (GtkWidget *)monthButton, 2, 3, 4, 5,
		   GTK_EXPAND | GTK_FILL, GTK_FILL, 0, 0);

  nonjudgementalButton = gtk_check_button_new_with_label("Nonjudgemental Colors");
  if (nonjudgementalColors)
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(nonjudgementalButton), TRUE);
  else
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(nonjudgementalButton), FALSE);
  gtk_table_attach(GTK_TABLE(settingsTable), nonjudgementalButton, 0, 1, 5, 6,
		   GTK_EXPAND | GTK_FILL, GTK_FILL, 0, 0);
  hackerDietButton = gtk_check_button_new_with_label("Hacker Diet Mode");
  if (hackerDietMode)
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(hackerDietButton), TRUE);
  else
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(hackerDietButton), FALSE);
  gtk_table_attach(GTK_TABLE(settingsTable), hackerDietButton, 1, 2, 5, 6,
		   GTK_EXPAND | GTK_FILL, GTK_FILL, 0, 0);
  showCommentsButton = gtk_check_button_new_with_label("Plot Comments");
  if (showComments)
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(showCommentsButton), TRUE);
  else
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(showCommentsButton), FALSE);
  gtk_table_attach(GTK_TABLE(settingsTable), showCommentsButton, 2, 3, 5, 6,
		   GTK_EXPAND | GTK_FILL, GTK_FILL, 0, 0);

  firstSeparator = gtk_separator_menu_item_new();
  gtk_table_attach(GTK_TABLE(settingsTable), firstSeparator, 0, 3, 6, 7,
		   GTK_EXPAND | GTK_FILL, GTK_FILL, 0, 0);

  /* Set up a radio box button group to select the plot interval */
  plotFortnightButton = gtk_radio_button_new_with_label(NULL, "Plot Last Fortnight");
  plotGroup = gtk_radio_button_group(GTK_RADIO_BUTTON(plotFortnightButton));
  plotMonthButton = gtk_radio_button_new_with_label(plotGroup, "Plot Last Month");
  plotGroup = gtk_radio_button_group(GTK_RADIO_BUTTON(plotMonthButton));
  plotQuarterButton = gtk_radio_button_new_with_label(plotGroup, "Plot Last Quarter");
  plotGroup = gtk_radio_button_group(GTK_RADIO_BUTTON(plotQuarterButton));
  plot6MonthButton = gtk_radio_button_new_with_label(plotGroup, "Plot Last 6 Months");
  plotGroup = gtk_radio_button_group(GTK_RADIO_BUTTON(plot6MonthButton));
  plotYearButton = gtk_radio_button_new_with_label(plotGroup, "Plot Last Year");
  plotGroup = gtk_radio_button_group(GTK_RADIO_BUTTON(plotYearButton));
  plotHistoryButton = gtk_radio_button_new_with_label(plotGroup, "Plot Entire History");
  plotGroup = gtk_radio_button_group(GTK_RADIO_BUTTON(plotHistoryButton));
  switch (plotInterval) {
  case 14:
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(plotFortnightButton), TRUE);
    break;
  case 30:
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(plotMonthButton), TRUE);
    break;
  case 91:
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(plotQuarterButton), TRUE);
    break;
  case 183:
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(plot6MonthButton), TRUE);
    break;
  case 365:
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(plotYearButton), TRUE);
    break;
  default:
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(plotHistoryButton), TRUE);
  }
  gtk_table_attach(GTK_TABLE(settingsTable), plotFortnightButton, 0, 1, 7, 8,
		   GTK_EXPAND | GTK_FILL, GTK_FILL, 0, 0);
  gtk_table_attach(GTK_TABLE(settingsTable), plotMonthButton, 1, 2, 7, 8,
		   GTK_EXPAND | GTK_FILL, GTK_FILL, 0, 0);
  gtk_table_attach(GTK_TABLE(settingsTable), plotQuarterButton, 2, 3, 7, 8,
		   GTK_EXPAND | GTK_FILL, GTK_FILL, 0, 0);
  gtk_table_attach(GTK_TABLE(settingsTable), plot6MonthButton, 0, 1, 8, 9,
		   GTK_EXPAND | GTK_FILL, GTK_FILL, 0, 0);
  gtk_table_attach(GTK_TABLE(settingsTable), plotYearButton, 1, 2, 8, 9,
		   GTK_EXPAND | GTK_FILL, GTK_FILL, 0, 0);
  gtk_table_attach(GTK_TABLE(settingsTable), plotHistoryButton, 2, 3, 8, 9,
		   GTK_EXPAND | GTK_FILL, GTK_FILL, 0, 0);

  secondSeparator = gtk_separator_menu_item_new();
  gtk_table_attach(GTK_TABLE(settingsTable), secondSeparator, 0, 3, 9, 10,
		   GTK_EXPAND | GTK_FILL, GTK_FILL, 0, 0);

  /* Set up a radio box button group to select the least squares fit interval */
  fitNothingButton = gtk_radio_button_new_with_label(NULL, "Don't Fit Data");
  fitGroup = gtk_radio_button_group(GTK_RADIO_BUTTON(fitNothingButton));
  fitMonthButton = gtk_radio_button_new_with_label(fitGroup, "Fit Last Month");
  fitGroup = gtk_radio_button_group(GTK_RADIO_BUTTON(fitMonthButton));
  fitQuarterButton = gtk_radio_button_new_with_label(fitGroup, "Fit Last Quarter");
  fitGroup = gtk_radio_button_group(GTK_RADIO_BUTTON(fitQuarterButton));
  fit6MonthButton = gtk_radio_button_new_with_label(fitGroup, "Fit Last 6 Months");
  fitGroup = gtk_radio_button_group(GTK_RADIO_BUTTON(fit6MonthButton));
  fitYearButton = gtk_radio_button_new_with_label(fitGroup, "Fit Last Year");
  fitGroup = gtk_radio_button_group(GTK_RADIO_BUTTON(fitYearButton));
  fitHistoryButton = gtk_radio_button_new_with_label(fitGroup, "Fit Entire History");
  fitGroup = gtk_radio_button_group(GTK_RADIO_BUTTON(fitHistoryButton));
  switch (fitInterval) {
  case -1:
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(fitNothingButton), TRUE);
    break;
  case 30:
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(fitMonthButton), TRUE);
    break;
  case 91:
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(fitQuarterButton), TRUE);
    break;
  case 183:
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(fit6MonthButton), TRUE);
    break;
  case 365:
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(fitYearButton), TRUE);
    break;
  default:
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(fitHistoryButton), TRUE);
  }
  gtk_table_attach(GTK_TABLE(settingsTable), fitNothingButton, 0, 1, 10, 11,
		   GTK_EXPAND | GTK_FILL, GTK_FILL, 0, 0);
  gtk_table_attach(GTK_TABLE(settingsTable), fitMonthButton, 1, 2, 10, 11,
		   GTK_EXPAND | GTK_FILL, GTK_FILL, 0, 0);
  gtk_table_attach(GTK_TABLE(settingsTable), fitQuarterButton, 2, 3, 10, 11,
		   GTK_EXPAND | GTK_FILL, GTK_FILL, 0, 0);
  gtk_table_attach(GTK_TABLE(settingsTable), fit6MonthButton, 0, 1, 11, 12,
		   GTK_EXPAND | GTK_FILL, GTK_FILL, 0, 0);
  gtk_table_attach(GTK_TABLE(settingsTable), fitYearButton, 1, 2, 11, 12,
		   GTK_EXPAND | GTK_FILL, GTK_FILL, 0, 0);
  gtk_table_attach(GTK_TABLE(settingsTable), fitHistoryButton, 2, 3, 11, 12,
		   GTK_EXPAND | GTK_FILL, GTK_FILL, 0, 0);
  thirdSeparator = gtk_separator_menu_item_new();

  gtk_table_attach(GTK_TABLE(settingsTable), thirdSeparator, 0, 3, 12, 13,
		   GTK_EXPAND | GTK_FILL, GTK_FILL, 0, 0);

  settingsStackable = hildon_stackable_window_new();
  g_signal_connect(G_OBJECT(settingsStackable), "destroy",
		   G_CALLBACK(checkSettings), NULL);
  gtk_container_add(GTK_CONTAINER(settingsStackable), settingsTable);
  gtk_widget_show_all(settingsStackable);
} /* End of  S E T T I N G S  B U T T O N  C L I C K E D */

/*
  H E L P  B U T T O N  C L I C K E D

  This fucntion is called if Help is chosen fromt he main menu.
 */
void helpButtonClicked(GtkButton *button, gpointer userData)
{
  dprintf("in helpButtonClicked()\n");
  /* Send a dbus command to open the maeFat wiki page in the default browser */
  system("/usr/bin/dbus-send --system --type=method_call  --dest=\"com.nokia.osso_browser\" /com/nokia/osso_browser/request com.nokia.osso_browser.load_url string:\"wiki.maemo.org/maeFat\"");
} /* End of  H E L P  B U T T O N  C L I C K E D */

/*
  A B O U T  B U T T O N  C L I C K E D

  This function is called if About is selected from the main menu.
  It just prints a brief message about the program.
 */
void aboutButtonClicked(GtkButton *button, gpointer userData)
{
  static int firstCall = TRUE;
  static GtkTextBuffer *aboutBuffer;
  static GtkWidget *aboutTextWidget, *aboutStackable;

  dprintf("in aboutButtonClicked()\n");
  if (firstCall) {
    aboutBuffer = gtk_text_buffer_new(NULL);
    gtk_text_buffer_set_text(aboutBuffer, "maeFat Version 1.5\nCopyright (C) 2011, Ken Young\n\nThis program helps you keep track of your weight.\nThis is free, open source software, released under GPL version 2.\n\nPlease send comments, questions and feature requests to\norrery.moko@gmail.com", -1);
    aboutTextWidget = hildon_text_view_new(); 
    ((GtkTextView *)aboutTextWidget)->editable =
      ((GtkTextView *)aboutTextWidget)->cursor_visible = FALSE;
    gtk_widget_ref(aboutTextWidget);
    firstCall = FALSE;
  }
  hildon_text_view_set_buffer((HildonTextView *)aboutTextWidget, aboutBuffer);
  aboutStackable = hildon_stackable_window_new();
  gtk_container_add(GTK_CONTAINER(aboutStackable), aboutTextWidget);
  gtk_widget_show_all(aboutStackable);
} /* End of  A B O U T  B U T T O N  C L I C K E D */

/*
  M A K E  H I L D O N  B U T T O N S

  This function builds the main menu for the app.
 */
static void makeHildonButtons(void)
{
  static HildonAppMenu *hildonMenu;
  HildonSizeType buttonSize = HILDON_SIZE_FINGER_HEIGHT | HILDON_SIZE_AUTO_WIDTH;

  hildonMenu = HILDON_APP_MENU(hildon_app_menu_new());

  /* Data Entry Button */
  dataEntryButton = hildon_gtk_button_new(buttonSize);
  gtk_button_set_label(GTK_BUTTON(dataEntryButton), "Data Entry");
  g_signal_connect(G_OBJECT(dataEntryButton), "clicked", G_CALLBACK(dataEntryButtonClicked), NULL);
  hildon_app_menu_append(hildonMenu, GTK_BUTTON(dataEntryButton));
  
  /* Log  Button */
  logButton = hildon_gtk_button_new(buttonSize);
  gtk_button_set_label(GTK_BUTTON(logButton), "Log");
  g_signal_connect(G_OBJECT(logButton), "clicked", G_CALLBACK(logButtonClicked), NULL);
  hildon_app_menu_append(hildonMenu, GTK_BUTTON(logButton));

  /* Trend Button */
  trendButton = hildon_gtk_button_new(buttonSize);
  gtk_button_set_label(GTK_BUTTON(trendButton), "Trend Analysis");
  g_signal_connect(G_OBJECT(trendButton), "clicked", G_CALLBACK(trendButtonClicked), NULL);
  hildon_app_menu_append(hildonMenu, GTK_BUTTON(trendButton));

  /* Settings Button */
  settingsButton = hildon_gtk_button_new(buttonSize);
  gtk_button_set_label(GTK_BUTTON(settingsButton), "Settings");
  g_signal_connect(G_OBJECT(settingsButton), "clicked", G_CALLBACK(settingsButtonClicked), NULL);
  hildon_app_menu_append(hildonMenu, GTK_BUTTON(settingsButton));

  /* Help Button */
  helpButton = hildon_gtk_button_new(buttonSize);
  gtk_button_set_label(GTK_BUTTON(helpButton), "Help");
  g_signal_connect(G_OBJECT(helpButton), "clicked", G_CALLBACK(helpButtonClicked), NULL);
  hildon_app_menu_append(hildonMenu, GTK_BUTTON(helpButton));

  /* About Button */
  aboutButton = hildon_gtk_button_new(buttonSize);
  gtk_button_set_label(GTK_BUTTON(aboutButton), "About");
  g_signal_connect(G_OBJECT(aboutButton), "clicked", G_CALLBACK(aboutButtonClicked), NULL);
  hildon_app_menu_append(hildonMenu, GTK_BUTTON(aboutButton));

  hildon_stackable_window_set_main_menu((HildonStackableWindow *)window, hildonMenu);
  gtk_widget_show_all(GTK_WIDGET(hildonMenu));
} /* End of  M A K E  H I L D O N  B U T T O N S */

/*
  B U T T O N  P R E S S  E V E N T

  This function is called when the screen is pressed
 */
static gboolean buttonPressEvent(GtkWidget *widget, GdkEventButton *event)
{
  dprintf("In buttonPressEvent\n");
  return(TRUE);
} /* End of  B U T T O N  P R E S S  E V E N T */

/*
  D A T A  E D I T  D E L E T E  C A L L B A C K

  This function is called if the delete button is pressed on the log
  edit page.
 */
void dataEditDeleteCallback(GtkButton *button, gpointer userData)
{
  deleted = TRUE;
  gtk_widget_destroy(dataEditStackable);
} /* End of  D A T A  E D I T  D E L E T E  C A L L B A C K */

/*
  E D I T  D A T A

  This function is called when a log entry has been edited.  It gets
  the new values from the widgets on the edit page.
 */
void editData(void)
{
  if (deleted) {
    if (editPtr->last == NULL)
      /* We're deleting the first entry in the list */
      logRoot = editPtr->next;
    else {
      (editPtr->last)->next = editPtr->next;
      if (editPtr->next != NULL)
	/* We're NOT deleting the last entry in the list */
	(editPtr->next)->last = editPtr->last;
      else
	/* We are deleting the last entry in the list */
	lastEntry = editPtr->last;
    }
    free(editPtr);
    nDataPoints--;
    if (nDataPoints == 0)
      logRoot = NULL;
    writeDataFile();
  } else {
    int iJD;
    int commentChanged = FALSE;
    float weight;
    double fJD;
    char *weightResult;
    char *comment = NULL;
    guint year, month, day, hours, minutes;
    
    hildon_date_button_get_date((HildonDateButton *)dateEditButton, &year, &month, &day);
    hildon_time_button_get_time((HildonTimeButton *)timeEditButton, &hours, &minutes);
    month += 1;
    weightResult = (char *)hildon_button_get_value(HILDON_BUTTON(weightButton));
    sscanf(weightResult, "%f", &weight);
    iJD = calculateJulianDate(year, month, day);
    fJD = (double)iJD + ((double)hours)/24.0 + ((double)minutes)/1440.0;
    comment = (char *)gtk_entry_get_text((GtkEntry *)commentText);
    if ((comment != NULL) && (strlen(comment) > 0)) {
      /* Got a comment which is not NULL */
      if (editPtr->comment != NULL) {
	/* The entry already has a comment */
	if (strcmp(editPtr->comment, comment))
	  commentChanged = TRUE;
      } else
	commentChanged = TRUE;
    } else {
	/* Comment is now empty - flag this as a change */
      commentChanged = TRUE;
      comment = NULL;
    }
    dprintf("Got %d/%d/%d  %02d:%02d %f %f \"%s\"\n",
	   day, month, year, hours, minutes, weight, fJD, comment);
    editPtr->time = fJD;
    editPtr->weight = weight;
    if (commentChanged) {
      if (editPtr->comment != NULL)
	free(editPtr->comment);
      if (comment != NULL) {
	editPtr->comment = malloc(strlen(comment)+1);
	if (editPtr->comment != NULL)
	  strcpy(editPtr->comment, comment);
	else
	  perror("malloc of chaged comment");
      } else
	editPtr->comment = comment;
    }
  }
  writeDataFile();
  gtk_widget_destroy(trendStackable);
} /* End of  E D I T  D A T A */

/*
  C R E A T E  E D I T  P A G E

  This function creates and displays the widgets needed for the Edit
  page.
 */
void createEditPage(logEntry *ptr)
{
  double dayFrac;
  int day, month, year, hours, minutes;
  static GtkWidget *weightSelector, *dataEditTable;

  dprintf("in createEditPage\n");
  deleted = FALSE;
  editPtr = ptr;
  dataEditTable = gtk_table_new(1, 5, FALSE);
  dateEditButton = hildon_date_button_new(HILDON_SIZE_AUTO, HILDON_BUTTON_ARRANGEMENT_VERTICAL);
  tJDToDate((double)((int)ptr->time), &year, &month, &day);
  hildon_date_button_set_date((HildonDateButton *)dateEditButton, year, month-1, day);
  gtk_table_attach(GTK_TABLE(dataEditTable), dateEditButton, 0, 1, 0, 1,
		   GTK_EXPAND, GTK_EXPAND, 0, 0);
  
  timeEditButton = hildon_time_button_new(HILDON_SIZE_AUTO, HILDON_BUTTON_ARRANGEMENT_VERTICAL);
  dayFrac = ptr->time - (double)((int)ptr->time);
  hours = (int)(dayFrac*24.0);
  minutes = (int)((dayFrac - ((double)hours)/24.0) * 1440.0 + 0.5);
  hildon_time_button_set_time((HildonTimeButton *)timeEditButton, hours, minutes);
  gtk_table_attach(GTK_TABLE(dataEditTable), timeEditButton, 0, 1, 1, 2,
		   GTK_EXPAND, GTK_EXPAND, 0, 0);
  dataEditStackable = hildon_stackable_window_new();
  g_signal_connect(G_OBJECT(dataEditStackable), "destroy",
		   G_CALLBACK(editData), NULL);
  weightSelector = createTouchSelector();
  weightButton = hildon_picker_button_new(HILDON_SIZE_AUTO, HILDON_BUTTON_ARRANGEMENT_VERTICAL);
  if (weightkg)
    hildon_touch_selector_set_active(HILDON_TOUCH_SELECTOR(weightSelector), 0,
				     (int)((10.0*ptr->weight)+0.5) - 10*minSelectorWeight);
  else
    hildon_touch_selector_set_active(HILDON_TOUCH_SELECTOR(weightSelector), 0,
				     (int)((5.0*ptr->weight)+0.5) - 5*minSelectorWeight);
  if (weightkg)
    hildon_button_set_title(HILDON_BUTTON(weightButton), "Weight (kg)");
  else
    hildon_button_set_title(HILDON_BUTTON(weightButton), "Weight (lbs)");
  hildon_picker_button_set_selector(HILDON_PICKER_BUTTON(weightButton),
				    HILDON_TOUCH_SELECTOR(weightSelector));
  gtk_table_attach(GTK_TABLE(dataEditTable), weightButton, 0, 1, 2, 3,
		   GTK_EXPAND, GTK_EXPAND, 0, 0);
  
  commentText = gtk_entry_new();
  if (ptr->comment != NULL)
    gtk_entry_set_text((GtkEntry *)commentText, ptr->comment);
  gtk_table_attach(GTK_TABLE(dataEditTable), commentText, 0, 1, 3, 4,
		   GTK_EXPAND | GTK_FILL, GTK_FILL, 0, 0);
  dataEditDelete = gtk_button_new_with_label("Delete This Entry");
  g_signal_connect(G_OBJECT(dataEditDelete), "clicked",
		   G_CALLBACK(dataEditDeleteCallback), NULL);
  gtk_table_attach(GTK_TABLE(dataEditTable), dataEditDelete, 0, 1, 4, 5,
		   GTK_EXPAND, GTK_EXPAND, 0, 0);
  gtk_container_add(GTK_CONTAINER(dataEditStackable), dataEditTable);
  gtk_widget_show_all(dataEditStackable);
} /* End of  C R E A T E  E D I T  P A G E */

/*
  B U T T O N  R E L E A S E  E V E N T

  This function is called when a button press in the screen is released.
  It does nothing unless the Log page is displayed.   If the Log page
  is displayed, it checked to see if the press event corredinates
  corespond to one of the weight entry buttons.   If so, it builds an
  edit page, otherwise nothing happens.
 */
static gboolean buttonReleaseEvent(GtkWidget *widget, GdkEventButton *event)
{
  dprintf("In buttonReleaseEvent\n");
  if (logDisplayed) {
    int x, y;
    int found = FALSE;
    logCell *ptr;

    x = event->x; y = event->y;
    dprintf("The log is displayed (%d, %d)\n", x, y);
    /* Loop through all the data cells displayed on the Log page */
    ptr = logCellRoot;
    while ((ptr != NULL) && (!found)) {
      if ((ptr->box[0].x <= x) && (ptr->box[0].y <= y) && (ptr->box[2].x >= x) && (ptr->box[2].y >= y))
	found = TRUE;
      else
	ptr = ptr->next;
    }
    if (found) {
      int day, month, year;

      tJDToDate((double)((int)ptr->ptr->time), &year, &month, &day);
      dprintf("Found it! (%d/%d/%d) (%f)\n", month, day, year, ptr->ptr->weight);
      createEditPage(ptr->ptr);
    }
  }
  return(TRUE);
} /* End of  B U T T O N  R E L E A S E  E V E N T */

int main(int argc, char **argv)
{
  osso_context_t *oSSOContext;

  homeDir = getenv("HOME");
  if (homeDir == NULL) {
    homeDir = malloc(strlen("/home/user")+1);
    if (homeDir == NULL) {
      perror("malloc of backup homeDir");
      exit(ERROR_EXIT);
    }
    sprintf(homeDir, "/home/user");
  }
  userDir = malloc(strlen(homeDir)+strlen("/.maeFat")+1);
  if (userDir == NULL) {
    perror("malloc of userDir");
    exit(ERROR_EXIT);
  }
  sprintf(userDir, "%s/.maeFat", homeDir);
  mkdir(userDir, 0777);
  mkdir(backupDir, 0777);

  fileName = malloc(strlen(userDir)+strlen("/data")+1);
  if (fileName == NULL) {
    perror("malloc of fileName");
    exit(ERROR_EXIT);
  }
  sprintf(fileName, "%s/data", userDir);
  dprintf("The data file path is \"%s\"\n", fileName);
  readData(fileName);

  settingsFileName = malloc(strlen(userDir)+strlen("/settings")+1);
  if (settingsFileName == NULL) {
    perror("malloc of settingsFileName");
    exit(ERROR_EXIT);
  }
  sprintf(settingsFileName, "%s/settings", userDir);
  dprintf("The settings file path is \"%s\"\n", settingsFileName);
  readSettings(settingsFileName);

  oSSOContext = osso_initialize("com.nokia.maefat", maeFatVersion, TRUE, NULL);
  if (!oSSOContext) {
    fprintf(stderr, "oss_initialize call failed\n");
    exit(-1);
  }

  hildon_gtk_init(&argc, &argv);
  /* Initialize main window */
  window = hildon_stackable_window_new();
  gtk_widget_set_size_request (GTK_WIDGET (window), 640, 480);
  gtk_window_set_title (GTK_WINDOW (window), "maeFat");
  g_signal_connect (G_OBJECT (window), "delete_event",
		    G_CALLBACK (gtk_main_quit), NULL);
  mainBox = gtk_vbox_new(FALSE, 0);
  gtk_container_add(GTK_CONTAINER(window), mainBox);
  g_object_ref(mainBox); /* This keeps mainBox from being destroyed when not displayed */
  gtk_widget_show(mainBox);

  /* Configure Main Menu Buttons */
  makeHildonButtons();

  drawingArea = gtk_drawing_area_new();
  gtk_widget_set_size_request (GTK_WIDGET(drawingArea), 640, 480);
  gtk_box_pack_end(GTK_BOX(mainBox), drawingArea, TRUE, TRUE, 0);
  g_signal_connect(G_OBJECT(drawingArea), "expose_event",
		   G_CALLBACK(exposeEvent), NULL);
  g_signal_connect(G_OBJECT(drawingArea), "configure_event",
		   G_CALLBACK(configureEvent), NULL);

  g_signal_connect(G_OBJECT(drawingArea), "button_release_event",
		   G_CALLBACK(buttonReleaseEvent), NULL);
  g_signal_connect(G_OBJECT(drawingArea), "button_press_event",
		   G_CALLBACK(buttonPressEvent), NULL);
  gtk_widget_show(window);
  makeGraphicContexts(window);
  gtk_widget_set_events(drawingArea,
			GDK_EXPOSURE_MASK       | GDK_BUTTON_PRESS_MASK  |
			GDK_BUTTON_RELEASE_MASK | GDK_POINTER_MOTION_MASK);
  gtk_widget_show(drawingArea);
  gtk_main();
  osso_deinitialize(oSSOContext);
  exit(OK_EXIT);
}
