/*
  maeFat Rev 1.1
  Ken Young (orrery.moko@gmail.com)
  First Version Feb. 5, 2011

  Copyright (C) (2011) Ken Young orrery.moko@gmail.com

  This program is free software; you can redistribute it and/or 
  modify it under the terms of the GNU General Public License 
  as published by the Free Software Foundation; either 
  version 2 of the License, or (at your option) any later 
  version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

  This program is a tool for personal weight reduction/management.
  It is closely modelled upon the Hacker Diet tools, although it
  is not a clone of that package.

 */

#include <stdio.h>
#include <sys/stat.h>
#include <dirent.h>
#include <locale.h>
#include <fcntl.h>
#include <math.h>
#include <string.h>
#include <ctype.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <unistd.h>
#include <signal.h>
#include <fcntl.h>
#include <errno.h>
#include <sys/time.h>
#include <stdlib.h>
#include <memory.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <cairo.h>
#include <cairo-xlib.h>
#include <pango/pangocairo.h>
#include <gtk/gtk.h>
#include <dbus/dbus.h>
#include <hildon-1/hildon/hildon.h>
#include <libosso.h>
#include "maeFatColors.h"

#define FALSE       (0)
#define ERROR_EXIT (-1)
#define OK_EXIT     (0)

#define M_HALF_PI (M_PI * 0.5)

#define kg_per_lb (0.45359237)
#define m_per_in (2.54e-2)

int debugMessagesOn = FALSE;
#define dprintf if (debugMessagesOn) printf

typedef struct logEntry {
  double time;           /* The time at which the log entry was made */
  float weight;          /* The reported weight                      */
  char *comment;         /* User-entered comment text                */
  struct logEntry *next; /* Pointer to next entry                    */
  struct logEntry *last; /* Pointer to last entry                    */
} logEntry;

logEntry *logRoot = NULL;
logEntry *lastEntry = NULL;
int nDataPoints = 0;

char *homeDir, *userDir, *fileName, *settingsFileName;
char *backupDir = "/media/mmc1/.maeFat";
char *backupFile = "/media/mmc1/.maeFat/data";

double myHeight          =  72.0;
double myTarget          = 150.0;
int weightkg             = FALSE;
int heightcm             = FALSE;
int monthFirst           = FALSE;
int nonjudgementalColors = FALSE;
int hackerDietMode       = FALSE;
int showComments         = FALSE;
int showTarget           = FALSE;
int plotInterval         =   365;

int goodColor = OR_GREEN;
int badColor  = OR_RED;

char *defaultComment = "Type Comment Here";

char *monthName[12] = {"January",   "February", "March",    "April",
		       "May",       "June",     "July",     "August",
		       "September", "October",  "November", "December"};
char *dayName[7] = {"Sunday", "Monday", "Tuesday", "Wednesday", "Thursday", "Friday", "Saturday"};
double monthLengths[12] = {31.0, 28.0, 31.0, 30.0, 31.0, 30.0, 31.0, 31.0, 30.0, 31.0, 30.0, 31.0};
char *maeFatVersion = "1.1";
char scratchString[200];

int displayHeight, displayWidth;
double maxJD = -1.0;
double minJD = 1.0e30;
double maxWeight = -1.0;
double minWeight = 1.0e30;
double xScale, yScale, plotMaxX, plotMinX, plotMaxY, plotMinY;

GdkGC *gC[N_COLORS];
GtkWidget *window, *mainBox, *drawingArea, *weightButton, *dataEntryCancel, *dataEditDelete,
  *dataEntryButton, *chartButton, *logButton, *trendButton, *utilitiesButton, *settingsButton,
  *helpButton, *aboutButton, *commentText, *logSelectorButton;
GdkPixmap *pixmap = NULL;
GdkPixmap *cairoPixmap = NULL;

GdkFont *smallFont, *bigFont;

/* Cairo and Pango related globals */

#define N_PANGO_FONTS (6)
#define SMALL_PANGO_FONT       (0)
#define MEDIUM_PANGO_FONT      (1)
#define BIG_PANGO_FONT         (2)
#define SMALL_MONO_PANGO_FONT  (3)
#define GREEK_FONT             (4)
#define TINY_GREEK_FONT        (5)
#define BIG_PANGO_FONT_NAME         "Sans Bold 32"
#define MEDIUM_PANGO_FONT_NAME      "Sans Normal 18"
#define SMALL_PANGO_FONT_NAME       "Sans Normal 14"
#define SMALL_MONO_PANGO_FONT_NAME  "Monospace Bold 16"
#define GREEK_FONT_NAME             "Sans 12"
#define TINY_GREEK_FONT_NAME        "Sans 8"

cairo_surface_t *cairoSurface = NULL;
cairo_t *cairoContext = NULL;

int cairoInitialized = FALSE;

/*   E N D   O F   V A R I A B L E   D E C L A R A T I O N S   */

#define RENDER_CENTER_X (80)
#define RENDER_CENTER_Y (400)

/*
  Create the structures needed to be able to yse Pango and Cairo to
render antialiased fonts
*/
void createPangoCairoWorkspace(void)
{
  cairoContext = gdk_cairo_create(cairoPixmap);
  if (cairoContext == NULL) {
    fprintf(stderr, "cairo_create returned a NULL pointer\n");
    exit(ERROR_EXIT);
  }
  cairo_translate(cairoContext, RENDER_CENTER_X, RENDER_CENTER_Y);
  cairoInitialized = TRUE;
}

/*
  The following function renders a text string using Pango and Cairo, on the
  cairo screen, and copies it to a gtk drawable.   The height and width
  of the area occupied by the text is returned in passed variables.
*/
void renderPangoText(char *theText, unsigned short color, int font,
		     int *width, int *height, GdkDrawable *dest,
		     int x, int y, float angle, int center, int topClip,
		     int background)
{
  static int firstCall = TRUE;
  static PangoFontMap *fontmap = NULL;
  static char *fontName[N_PANGO_FONTS] = {SMALL_PANGO_FONT_NAME, MEDIUM_PANGO_FONT_NAME,
					  BIG_PANGO_FONT_NAME, SMALL_MONO_PANGO_FONT_NAME,
					  GREEK_FONT_NAME, TINY_GREEK_FONT_NAME};
  static PangoContext *pangoContext[N_PANGO_FONTS];
  static cairo_font_options_t *options = NULL;
  static int xC, yC, wC, hC = 0;
  static float currentAngle = 0.0;
  float deltaAngle, cA, sA, fW, fH;
  int pWidth, pHeight, iFont;
  PangoFontDescription *fontDescription;
  PangoLayout *layout;

  if (angle < 0.0)
    angle += M_PI * 2.0;
  else if (angle > M_PI * 2.0)
    angle -= M_PI * 2.0;
  if ((angle > M_HALF_PI) && (angle < M_HALF_PI + M_PI))
    angle -= M_PI;
  if (angle < 0.0)
    angle += M_PI * 2.0;
  if (!cairoInitialized)
    createPangoCairoWorkspace();
  if (firstCall) {
    /* Initialize fonts, etc. */
    fontmap = pango_cairo_font_map_get_default();
    if (fontmap == NULL) {
      fprintf(stderr, "pango_cairo_font_map_get_default() returned a NULL pointer\n");
      exit(ERROR_EXIT);
    }
    options = cairo_font_options_create();
    if (options == NULL) {
      fprintf(stderr, "cairo_font_options_create() returned a NULL pointer\n");
      exit(ERROR_EXIT);
    }
    cairo_font_options_set_antialias(options, CAIRO_ANTIALIAS_GRAY);
    cairo_font_options_set_hint_style(options, CAIRO_HINT_STYLE_FULL);
    cairo_font_options_set_hint_metrics(options, CAIRO_HINT_METRICS_ON);
    cairo_font_options_set_subpixel_order(options, CAIRO_SUBPIXEL_ORDER_BGR);
    for (iFont = 0; iFont < N_PANGO_FONTS; iFont++) {
      fontDescription = pango_font_description_new();
      if (fontDescription == NULL) {
	fprintf(stderr, "pango_font_description_new() returned a NULL pointer, font = %d\n", iFont);
	exit(ERROR_EXIT);
      }
      pango_font_description_set_family(fontDescription, (const char*) fontName[iFont]);
      pangoContext[iFont] = pango_cairo_font_map_create_context(PANGO_CAIRO_FONT_MAP(fontmap));
      if (pangoContext[iFont] == NULL) {
	fprintf(stderr, "pango_cairo_font_map_create_context(iFont = %d) returned a NULL pointer\n", iFont);
	exit(ERROR_EXIT);
      }
      pango_context_set_font_description(pangoContext[iFont], fontDescription);
      pango_font_description_free(fontDescription);
      pango_cairo_context_set_font_options(pangoContext[iFont], options);
    }
    firstCall = FALSE;
  } else
    if (font == MEDIUM_PANGO_FONT)
      gdk_draw_rectangle(cairoPixmap, gC[background], TRUE, RENDER_CENTER_X+xC, RENDER_CENTER_Y+yC,
			 wC, hC+7);
    else if (font == BIG_PANGO_FONT)
      gdk_draw_rectangle(cairoPixmap, gC[background], TRUE, RENDER_CENTER_X+xC, RENDER_CENTER_Y+yC,
			 wC, hC+30);
    else
      gdk_draw_rectangle(cairoPixmap, gC[background], TRUE, RENDER_CENTER_X+xC, RENDER_CENTER_Y+yC,
			 wC, hC);
  layout = pango_layout_new(pangoContext[font]);
  if (layout == NULL) {
    fprintf(stderr, "pango_layout_new() returned a NULL pointer\n");
      exit(ERROR_EXIT);
  }
  pango_layout_set_text(layout, theText, -1);
  fontDescription = pango_font_description_from_string(fontName[font]);
  pango_layout_set_font_description (layout, fontDescription);
  pango_font_description_free(fontDescription);
  cairo_set_source_rgb(cairoContext,
		       ((double)orreryColorRGB[color][0])/DOUBLE_MAX16,
		       ((double)orreryColorRGB[color][1])/DOUBLE_MAX16,
			((double)orreryColorRGB[color][2])/DOUBLE_MAX16);
  deltaAngle = angle - currentAngle;
  cairo_rotate(cairoContext, deltaAngle);
  currentAngle = angle;
  pango_cairo_update_layout(cairoContext, layout);
  pango_layout_get_size(layout, &pWidth, &pHeight);
  fW = (float)pWidth/(float)PANGO_SCALE;
  fH = (float)pHeight/(float)PANGO_SCALE;
  *width = (int)(fW+0.5);
  *height = (int)(fH+0.5);
  cairo_move_to(cairoContext, 0.0, 0.0);
  pango_cairo_show_layout(cairoContext, layout);
  g_object_unref(layout);
  cA = cosf(angle); sA = sinf(angle);
  wC = (int)((fW*fabs(cA) + fH*fabs(sA)) + 0.5);
  hC = (int)((fW*fabs(sA) + fH*fabs(cA)) + 0.5);
  if (angle < M_HALF_PI) {
    xC = (int)((-fH*sA) + 0.5);
    yC = 0;
  } else {
    xC = 0;
    yC = (int)((fW*sA) + 0.5);
  }
  if (dest != NULL) {
    if (center) {
      int xM, yM, ys, yd, h;

      xM = (int)((fW*fabs(cA) + fH*fabs(sA)) + 0.5);
      yM = (int)((fW*fabs(sA) + fH*fabs(cA)) + 0.5);
      yd = y-(yM>>1);
      if (yd < topClip) {
	int delta;

	delta = topClip - yd;
	h = hC - delta;
	ys = RENDER_CENTER_Y + yC + delta;
	yd = topClip;
      } else {
	ys = RENDER_CENTER_Y+yC;
	h = hC;
      }
      gdk_draw_drawable(dest, gC[OR_BLUE], cairoPixmap, RENDER_CENTER_X+xC, ys,
			x-(xM>>1), yd, wC, h);
    } else
      gdk_draw_drawable(dest, gC[OR_BLUE], cairoPixmap, RENDER_CENTER_X+xC, RENDER_CENTER_Y+yC,
			x, y-((*height)>>1), wC, hC);
  }
}

/*
  calculateJulianDate converts the Gregorian Calendar date (dd/mm/yyy)
  to the Julian Date, which it returns.
 */
int calculateJulianDate(int dd, int mm, int yyyy)
{
  int a, y, m;

  a = (14 - mm)/12;
  y = yyyy + 4800 - a;
  m = mm + 12*a - 3;
  return(dd + (153*m + 2)/5 + 365*y +y/4 - y/100 + y/400 -32045);
}

/*
  enqueueEntry adds a new log entry to the end of the linked
  list of log entries.
*/
void enqueueEntry(double jD, float weight, char *comment)
{
  logEntry *newEntry;

  newEntry = (logEntry *)malloc(sizeof(logEntry));
  if (newEntry == NULL) {
    perror("malloc of newEntry");
    exit(ERROR_EXIT);
  }
  newEntry->time = jD;
  newEntry->weight = weight;
  newEntry->comment = comment;
  newEntry->next = NULL;
  newEntry->last = lastEntry;
  if (logRoot == NULL)
    logRoot = newEntry;
  else
    lastEntry->next = newEntry;
  lastEntry = newEntry;
}

float calculateWeightedAverage1(logEntry *end)
{
  double startTime;
  float ave, weightSum, weight;
  float p = 0.9;
  logEntry *ptr;

  ptr = end;
  startTime = end->time;
  ave = weightSum = 0.0;
  while (ptr != NULL) {
    if ((startTime - ptr->time) < 100.0) {
      weight = powf(p, (float)(startTime - ptr->time));
      ave += weight * ptr->weight;
      weightSum += weight;
    }
    ptr = ptr->last;
  }
  ave /= weightSum;
  return(ave);
}

float calculateWeightedAverage2(logEntry *end)
{
  float ave;
  float p = 0.9;
  logEntry *ptr;

  ave = logRoot->weight;
  ptr = logRoot;
  while (ptr != end) {
    if (ptr->next != NULL)
      ave = ave + (1.0-p)*(ptr->next->weight - ave);
    ptr = ptr->next;
  }
  return(ave);
}

float calculateWeightedAverage(logEntry *end)
{
  if (hackerDietMode)
    return(calculateWeightedAverage2(end));
  else
    return(calculateWeightedAverage1(end));
}

/*
  Read a line of text from the config file, and strip comments (flagged by #).
*/
int getLine(int fD, int stripComments, char *buffer, int *eOF)
{
  char inChar = (char)0;
  int count = 0;
  int sawComment = FALSE;
  int foundSomething = FALSE;

  buffer[0] = (char)0;
  while ((!(*eOF)) && (inChar != '\n') && (count < 132)) {
    int nChar;

    nChar = read(fD, &inChar, 1);
    if (nChar > 0) {
      foundSomething = TRUE;
      if ((inChar == '#') && stripComments)
        sawComment = TRUE;
      if (!sawComment)
        buffer[count++] = inChar;
    } else {
      *eOF = TRUE;
    }
  }
  if (foundSomething) {
    if (count > 0)
      buffer[count-1] = (char)0;
    return(TRUE);
  } else
    return(FALSE);
}

int tokenCheck(char *line, char *token, int type, void *value)

#define INT_TOKEN     0
#define DOUBLE_TOKEN  1
#define FLOAT_TOKEN   2
#define STRING_TOKEN  3

/*
  Scan "line" for "token".   If found, read the value into
  "value" as an integer, double or float, depending on "type".

  Return TRUE IFF the token is seen.
*/
{
  if (strstr(line, token)) {
    int nRead;

    switch (type) {
    case INT_TOKEN:
      nRead = sscanf(&((char *)strstr(line, token))[strlen(token)+1], "%d", (int *)value);
      if (nRead != 1) {
	fprintf(stderr, "Unable to parse config file line \"%s\"\n", line);
	return(FALSE);
      }
      break;
    case DOUBLE_TOKEN:
      nRead = sscanf(&((char *)strstr(line, token))[strlen(token)+1], "%lf", (double *)value);
      if (nRead != 1) {
	fprintf(stderr, "Unable to parse config file line \"%s\"\n", line);
	return(FALSE);
      }
      break;
    case FLOAT_TOKEN:
      nRead = sscanf(&((char *)strstr(line, token))[strlen(token)+1], "%f", (float *)value);
      if (nRead != 1) {
	fprintf(stderr, "Unable to parse config file line \"%s\"\n", line);
	return(FALSE);
      }
      break;
    case STRING_TOKEN:
      nRead = sscanf(&((char *)strstr(line, token))[strlen(token)+1], "%s", (char *)value);
      if (nRead != 1) {
	fprintf(stderr, "Unable to parse config file line \"%s\"\n", line);
	return(FALSE);
      }
      break;
    default:
      fprintf(stderr, "Unrecognized type (%d) passed to tokenCheck\n", type);
    }
    return(TRUE);
  } else
    return(FALSE);
}

void readSettings(char *fileName)
{
  int eOF = FALSE;
  int lineNumber = 0;
  int settingsFD;
  char inLine[100];

  settingsFD = open(fileName, O_RDONLY);
  if (settingsFD < 0) {
    perror("settings");
    return;
  }
  while (!eOF) {
    lineNumber++;
    if (getLine(settingsFD, TRUE, &inLine[0], &eOF))
      if (strlen(inLine) > 0) {
	tokenCheck(inLine, "MY_HEIGHT",          DOUBLE_TOKEN, &myHeight);
	tokenCheck(inLine, "MY_TARGET",          DOUBLE_TOKEN, &myTarget);
	tokenCheck(inLine, "WEIGHT_KG",             INT_TOKEN, &weightkg);
	tokenCheck(inLine, "HEIGHT_CM",             INT_TOKEN, &heightcm);
	tokenCheck(inLine, "MONTH_FIRST",           INT_TOKEN, &monthFirst);
	tokenCheck(inLine, "NONJUDGEMENTAL_COLORS", INT_TOKEN, &nonjudgementalColors);
	tokenCheck(inLine, "HACKER_DIET_MODE",      INT_TOKEN, &hackerDietMode);
	tokenCheck(inLine, "SHOW_COMMENTS",         INT_TOKEN, &showComments);
	tokenCheck(inLine, "SHOW_TARGET",           INT_TOKEN, &showTarget);
	tokenCheck(inLine, "PLOT_INTERVAL",         INT_TOKEN, &plotInterval);
      }
  }
  if (nonjudgementalColors)
    goodColor = badColor = OR_BLUE;
}

#define BORDER_FACTOR_X (0.005)
#define BORDER_FACTOR_Y (0.015)

void calcMaxima(double startTJD)
{
  if (nDataPoints == 1) {
    double aveJD, aveWeight;
    
    aveJD = (maxJD+minJD)*0.5;
    maxJD = aveJD + 5.0;
    minJD = aveJD - 5.0;
    aveWeight = (maxWeight+minWeight)*0.5;
    maxWeight = aveWeight + 5.0;
    minWeight = aveWeight - 5.0;
  } else {
    logEntry *ptr;

    maxJD = -1.0; minJD = 1.0e30; maxWeight = -1.0; minWeight = 1.0e30;
    ptr = logRoot;
    while (ptr != NULL) {
      if (ptr->time > startTJD) {
	if (maxJD < ptr->time)
	  maxJD = ptr->time;
	if (minJD > ptr->time)
	  minJD = ptr->time;
	if (maxWeight < ptr->weight)
	  maxWeight = ptr->weight;
	if (minWeight > ptr->weight)
	  minWeight = ptr->weight;
	if (maxWeight < calculateWeightedAverage(ptr))
	  maxWeight = calculateWeightedAverage(ptr);
	if (minWeight > calculateWeightedAverage(ptr))
	  minWeight = calculateWeightedAverage(ptr);
      }
      ptr = ptr->next;
    }
  }
  if (maxJD == minJD) {
    maxJD += 1.0;
    minJD -= 1.0;
  }
  if (maxWeight == minWeight) {
    maxWeight += 1.0;
    minWeight -= 1.0;
  }
  plotMaxX = maxJD + BORDER_FACTOR_X*(maxJD-minJD);
  plotMinX = minJD - BORDER_FACTOR_X*(maxJD-minJD);
  if (showTarget) {
    if (maxWeight < myTarget)
      maxWeight = myTarget;
    else if (minWeight > myTarget)
      minWeight = myTarget;
  }
  plotMaxY = maxWeight + BORDER_FACTOR_Y*(maxWeight-minWeight);
  plotMinY = minWeight - BORDER_FACTOR_Y*(maxWeight-minWeight);
}
/*
  Read in the date, time and weight data from the data file.
*/
void readData(char *fileName)
{
  int eOF = FALSE;
  int dataFD;
  char inLine[1000];

  nDataPoints = 0;
  dprintf("Trying to open \"%s\"\n", fileName);
  dataFD = open(fileName, O_RDONLY);
  if (dataFD >= 0) {
    while (!eOF) {
      int nRead;
      float newWeight;
      char dateString[100], timeString[100];

      getLine(dataFD, FALSE, &inLine[0], &eOF);
      nRead = sscanf(inLine, "%s %s %f", dateString, timeString, &newWeight);
      if (nRead == 3) {
	double dayFraction, fJD;
	int hh, mm, dd, MM, yyyy, jD;
	char *commentPtr;
	char *comment = NULL;

	sscanf(timeString, "%d:%d", &hh, &mm);
	dayFraction = ((double)hh + (double)mm/60.0)/24.0;
	sscanf(dateString,"%d/%d/%d", &dd, &MM, &yyyy);
	jD = calculateJulianDate(dd, MM, yyyy);
	fJD = (double)jD + dayFraction;
	commentPtr = strstr(inLine, "# ");
	if (commentPtr) {
	  commentPtr += 2;
	  dprintf("Comment: \"%s\"\n", commentPtr);
	  comment = malloc(strlen(commentPtr)+1);
	  if (comment)
	    sprintf(comment, "%s", commentPtr);
	}
	enqueueEntry(fJD, newWeight, comment);
	nDataPoints++;
      }
    }
    calcMaxima(0.0);
    close(dataFD);
  } else
    perror("Data file could not be openned");
  /*
  if (debugMessagesOn) {
    logEntry *ptr;

    ptr = logRoot;
    while (ptr != NULL) {
      float est1, est2;

      est1 = calculateWeightedAverage1(ptr);
      est2 = calculateWeightedAverage2(ptr);
      dprintf("%f   %f   %f   %f\n", 
	     ptr->time - logRoot->time, ptr->weight, est1, est2);
      ptr = ptr->next;
    }
  }
  */
}

/*
  Convert the Julian day into Calendar Date as per
  "Astronomical Algorithms" (Meeus)
*/
void tJDToDate(double tJD, int *year, int *month, int *day)
{
  int Z, alpha, A, B, C, D, E;
  double F;

  Z = (int)(tJD+0.5);
  F = tJD + 0.5 - (double)Z;
  if (Z >= 2299161) {
    alpha = (int)(((double)Z - 1867216.25)/36524.25);
    A = Z + 1 + alpha - alpha/4;
  } else
    A = Z;
  B = A + 1524;
  C = (int)(((double)B - 122.1) / 365.25);
  D = (int)(365.25 * (double)C);
  E = (int)(((double)(B - D))/30.6001);
  *day = (int)((double)B - (double)D - (double)((int)(30.6001*(double)E)) + F);
  if (E < 14)
    *month = E - 1;
  else
    *month = E - 13;
  if (*month > 2)
    *year = C - 4716;
  else
    *year = C - 4715;
}

#define LEFT_BORDER   (33)
#define RIGHT_BORDER  (40)
#define TOP_BORDER    (21)
#define BOTTOM_BORDER (48)

void screenCoordinates(double x, double y, int *ix, int *iy)
{
  *ix = (x - plotMinX)*xScale + LEFT_BORDER;
  *iy = displayHeight - ((y - plotMinY)*yScale + BOTTOM_BORDER) - 2;
}

double bMI(double weight)
{
  double height, tWeight;

  if (weightkg)
    tWeight = weight/kg_per_lb;
  else
    tWeight = weight;
  if (heightcm)
    height = myHeight/2.54;
  else
    height = myHeight;
  return (tWeight*kg_per_lb/(height*height*m_per_in*m_per_in));
}

double weight(double bMI)
{
  double height;

  if (heightcm)
    height = myHeight/2.54;
  else
    height = myHeight;
  if (weightkg)
    return (bMI*height*height*m_per_in*m_per_in);
  else
    return (bMI*height*height*m_per_in*m_per_in/kg_per_lb);
}

int calcStats(int days, 
	      float *startWeight, float *endWeight,
	      float *maxWeight, float *minWeight, float *aveWeight)
{
  int doStart, doEnd, doMax, doMin, doAve;
  int nSamples = 0;
  double earliestTJD, dDays;
  logEntry *ptr;

  if (logRoot == NULL)
    return(FALSE);
  if (days < 0)
    dDays = lastEntry->time - logRoot->time;
  else
    dDays = (double)days;
  if ((lastEntry->time - logRoot->time) < dDays)
    return(FALSE);
  if (startWeight == NULL) doStart = FALSE; else doStart = TRUE;
  if (endWeight   == NULL) doEnd   = FALSE; else doEnd   = TRUE;
  if (maxWeight   == NULL) doMax   = FALSE; else doMax   = TRUE;
  if (minWeight   == NULL) doMin   = FALSE; else doMin   = TRUE;
  if (aveWeight   == NULL) doAve   = FALSE; else doAve   = TRUE;
  if (doMax)
    *maxWeight = -1.0e30;
  if (doMin)
    *minWeight = 1.0e30;
  if (doAve)
    *aveWeight = 0.0;
  earliestTJD = lastEntry->time - dDays;
  ptr = lastEntry;
  while (ptr->time > earliestTJD)
    ptr = ptr->last;
  if (doStart)
    *startWeight = calculateWeightedAverage(ptr);
  if (doEnd)
    *endWeight = calculateWeightedAverage(lastEntry);
  while (ptr != NULL) {
    float trendWeight;

    trendWeight = calculateWeightedAverage(ptr);
    if (doMax)
      if (*maxWeight < trendWeight)
	*maxWeight = trendWeight;
    if (doMin)
      if (*minWeight > trendWeight)
	*minWeight = trendWeight;
    if (doAve) {
      *aveWeight += trendWeight;
      nSamples++;
    }
    ptr = ptr->next;
  }
  if (doAve && (nSamples > 0))
    *aveWeight /= (double)nSamples;
  if (doStart)
    dprintf("Returning startWeight = %f\n", *startWeight);
  if (doEnd)
    dprintf("Returning endWeight = %f\n", *endWeight);
  if (doMax)
    dprintf("Returning maxWeight = %f\n", *maxWeight);
  if (doMin)
    dprintf("Returning minWeight = %f\n", *minWeight);
  if (doAve)
    dprintf("Returning aveWeight = %f\n", *aveWeight);
  return(TRUE);
}

void redrawScreen(void)
{
  int nPoints = 0;
  int inc;
  int nCurvePoints = 0;
  int minPlottableY, maxPlottableY;
  int i, x, y, tWidth, tHeight, weightStart, weightEnd, bMIStart, bMIEnd;
  int sYear, sMonth, sDay, eYear, eMonth, eDay, plotCenter;
  float weightNow, weightLastWeek;
  double plotTimeInterval, plotWeightInterval;
  logEntry *ptr;
  logEntry *firstPlottable = NULL;
  GdkPoint *dataPoints, *linePoints, *lineTops, box[4];

  if (logRoot == NULL) {
    renderPangoText("No data has been entered yet", OR_WHITE, BIG_PANGO_FONT,
		    &tWidth, &tHeight, pixmap, displayWidth/2, 50, 0.0, TRUE, 0, OR_BLACK);
    renderPangoText("select the Data Entry menu option", OR_WHITE, BIG_PANGO_FONT,
		    &tWidth, &tHeight, pixmap, displayWidth/2, 100, 0.0, TRUE, 0, OR_BLACK);
    renderPangoText("to enter initial entries.", OR_WHITE, BIG_PANGO_FONT,
		    &tWidth, &tHeight, pixmap, displayWidth/2, 150, 0.0, TRUE, 0, OR_BLACK);
    renderPangoText("Also, use the Settings menu", OR_WHITE, BIG_PANGO_FONT,
		    &tWidth, &tHeight, pixmap, displayWidth/2, 200, 0.0, TRUE, 0, OR_BLACK);
    renderPangoText("to set your height and units.", OR_WHITE, BIG_PANGO_FONT,
		    &tWidth, &tHeight, pixmap, displayWidth/2, 250, 0.0, TRUE, 0, OR_BLACK);
  } else {
    double stopTJD = 0.0;

    stopTJD = lastEntry->time - plotInterval;
    calcMaxima(stopTJD);
    plotWeightInterval = maxWeight - minWeight;
    dataPoints = (GdkPoint *)malloc(nDataPoints * sizeof(GdkPoint));
    if (dataPoints == NULL) {
      perror("malloc of dataPoints");
      exit(ERROR_EXIT);
    }
    linePoints = (GdkPoint *)malloc(nDataPoints * sizeof(GdkPoint));
    if (linePoints == NULL) {
      perror("malloc of linePoints");
      exit(ERROR_EXIT);
    }
    lineTops = (GdkPoint *)malloc(nDataPoints * sizeof(GdkPoint));
    if (lineTops == NULL) {
      perror("malloc of lineTops");
      exit(ERROR_EXIT);
    }
    xScale = ((double)displayWidth - LEFT_BORDER - RIGHT_BORDER - 1)/(plotMaxX - plotMinX);
    yScale = ((double)displayHeight - TOP_BORDER - BOTTOM_BORDER - 4)/(plotMaxY - plotMinY);

    /* Draw the empty plot box */
    gdk_draw_rectangle(pixmap, drawingArea->style->black_gc,
		       TRUE, 0, 0,
		       drawingArea->allocation.width,
		       drawingArea->allocation.height);
    screenCoordinates(plotMinX, plotMinY, &box[0].x, &box[0].y);
    minPlottableY = box[0].y;
    screenCoordinates(plotMinX, plotMaxY, &box[1].x, &box[1].y);
    maxPlottableY = box[1].y;
    screenCoordinates(plotMaxX, plotMaxY, &box[2].x, &box[2].y);
    screenCoordinates(plotMaxX, plotMinY, &box[3].x, &box[3].y);
    plotCenter = (box[0].y + box[1].y + box[2].y + box[3].y)/4;
    gdk_draw_polygon(pixmap, gC[OR_DARK_GREY], TRUE, box, 4);
    gdk_draw_polygon(pixmap, gC[OR_WHITE], FALSE, box, 4);

    /* Draw the plot title */
    tJDToDate((double)((int)logRoot->time), &sYear, &sMonth, &sDay);
    if (logRoot->next == NULL) {
      eDay = sDay, eMonth = sMonth; eYear = sYear;
    } else {
      ptr = lastEntry;
      while (ptr != NULL) {
	if (ptr->time > stopTJD) {
	  dprintf("stopTJD = %f, copying %f\n", stopTJD, ptr->time);
	  firstPlottable = ptr;
	} else
	  dprintf("Not copying %f\n", ptr->time);
	ptr = ptr->last;
      }
      tJDToDate((double)((int)firstPlottable->time), &sYear, &sMonth, &sDay);
      tJDToDate((double)((int)lastEntry->time), &eYear, &eMonth, &eDay);
    }
    if (monthFirst) {
      int temp;

      temp = sDay;
      sDay = sMonth;
      sMonth = temp;
      temp = eDay;
      eDay = eMonth;
      eMonth = temp;
    }
    sprintf(scratchString, "Weight and Trend Data for %d/%d/%d through %d/%d/%d",
	    sDay, sMonth, sYear, eDay, eMonth, eYear);
    renderPangoText(scratchString, OR_WHITE, SMALL_PANGO_FONT,
		    &tWidth, &tHeight, pixmap, displayWidth/2, 10, 0.0, TRUE, 0, OR_BLACK);
    renderPangoText(scratchString, OR_WHITE, SMALL_PANGO_FONT,
		    &tWidth, &tHeight, pixmap, displayWidth/2, 10, 0.0, TRUE, 0, OR_BLACK);

    /* Y axis labels */

    ptr = logRoot;
    xScale = ((double)displayWidth - LEFT_BORDER - RIGHT_BORDER - 1)/(plotMaxX - plotMinX);
    yScale = ((double)displayHeight - TOP_BORDER - BOTTOM_BORDER - 4)/(plotMaxY - plotMinY);
    weightStart = (int)(minWeight + 0.5);
    bMIStart    = (int)(10.0*(int)(bMI(minWeight) + 0.5));
    weightEnd   = (int)(maxWeight + 0.5);
    bMIEnd      = (int)(10.0*(int)(bMI(maxWeight) + 0.5));
    if (weightStart >= weightEnd) {
    } else
      for (i = weightStart; i <= weightEnd; i += 1) {
	int color, stub;

	sprintf(scratchString, "%3d", i);
	screenCoordinates(logRoot->time, (double)i, &x, &y);
	if (i % 10 == 0) {
	  color = OR_WHITE;
	  stub = 10;
	  renderPangoText(scratchString, OR_WHITE, SMALL_PANGO_FONT,
			  &tWidth, &tHeight, pixmap, 0, y, 0.0, FALSE, 0, OR_BLACK);
	} else if (i % 5 == 0) {
	  stub = 8;
	  if (plotWeightInterval < 20.0)
	    renderPangoText(scratchString, OR_WHITE, SMALL_PANGO_FONT,
			    &tWidth, &tHeight, pixmap, 0, y, 0.0, FALSE, 0, OR_BLACK);
	} else {
	  stub = 5;
	  if (plotWeightInterval < 5.0)
	    renderPangoText(scratchString, OR_WHITE, SMALL_PANGO_FONT,
			    &tWidth, &tHeight, pixmap, 0, y, 0.0, FALSE, 0, OR_BLACK);
	}
	gdk_draw_line(pixmap, gC[OR_WHITE], LEFT_BORDER, y, LEFT_BORDER+stub, y);
      }
    if (bMIStart == bMIEnd) {
    } else {
      int bMIInc;

      if ((bMIStart - bMIEnd) <= -150)
	bMIInc = 20;
      else if ((bMIStart - bMIEnd) <= -60)
	bMIInc = 10;
      else if ((bMIStart - bMIEnd) <= -20)
	bMIInc = 5;
      else
	bMIInc = 1;
      for (i = bMIStart; i <= bMIEnd; i += bMIInc) {
	double tWeight;
	int color, stub;

	tWeight = weight(0.1 * (double)i);
	screenCoordinates(logRoot->time, tWeight, &x, &y);
	if ((y > TOP_BORDER) && (y < displayHeight-BOTTOM_BORDER)) {
	  sprintf(scratchString, "%4.1f", 0.1*(double)i);
	  if (i % 10 == 0) {
	    if (nonjudgementalColors)
	      color = OR_WHITE;
	    else if (i >= 300)
	      color = OR_RED;
	    else if (i >= 250)
	      color = OR_YELLOW;
	    else if (i <= 160)
	      color = OR_RED;
	    else if (i <= 200)
	      color = OR_YELLOW;
	    else
	      color = OR_GREEN;
	    stub = 10;
	  } else {
	    if (nonjudgementalColors)
	      color = OR_BLUE;
	    else if (i >= 300)
	      color = OR_RED;
	    else if (i >= 250)
	      color = OR_YELLOW;
	    else if (i <= 160)
	      color = OR_RED;
	    else if (i <= 200)
	      color = OR_YELLOW;
	    else
	      color = OR_GREEN;
	    stub = 5;
	  }
	  x = displayWidth-RIGHT_BORDER;
	  renderPangoText(scratchString, color, SMALL_PANGO_FONT,
			  &tWidth, &tHeight, pixmap, x+2, y, 0.0, FALSE, 0, OR_BLACK);
	  gdk_draw_line(pixmap, gC[OR_WHITE], x, y, x-stub, y);
	}
      }
      if (showTarget) {
	int x0, y0, x1, y1;

	screenCoordinates(plotMinX, myTarget, &x0, &y0);
	screenCoordinates(plotMaxX, myTarget, &x1, &y1);
	gdk_draw_line(pixmap, gC[OR_GREEN], x0, y0-1, x1, y1-1);
	gdk_draw_line(pixmap, gC[OR_GREEN], x0, y0,   x1, y1);
	gdk_draw_line(pixmap, gC[OR_GREEN], x0, y0+1, x1, y1+1);
      }
    }
    /* X axis labelling */
    tJDToDate(logRoot->time, &sYear, &sMonth, &sDay);
    if (sMonth == 1)
      sprintf(scratchString, "%d", sYear);
    else {
      for (i = 0; i < 3; i++)
	scratchString[i] = monthName[sMonth-1][i];
      scratchString[3] = (char)0;
    }
    screenCoordinates(logRoot->time, minWeight, &x, &y);
    plotTimeInterval = maxJD - minJD;
    if (x > RIGHT_BORDER)
      renderPangoText(scratchString, OR_WHITE, SMALL_PANGO_FONT,
		      &tWidth, &tHeight, pixmap, x, displayHeight-35, 0.0, TRUE, 0, OR_BLACK);
    else if (plotTimeInterval < 31.0)
      renderPangoText(scratchString, OR_WHITE, SMALL_PANGO_FONT,
		      &tWidth, &tHeight, pixmap, RIGHT_BORDER, displayHeight-35, 0.0, TRUE, 0, OR_BLACK);
    ptr = logRoot;
    while (ptr != NULL) {
      if (ptr->time > stopTJD) {
	int plotIt;

	tJDToDate(ptr->time, &eYear, &eMonth, &eDay);
	if (sMonth != eMonth) {
	  sMonth = eMonth;
	  if (sMonth == 1)
	    sprintf(scratchString, "%d", eYear);
	  else {
	    for (i = 0; i < 3; i++)
	      scratchString[i] = monthName[sMonth-1][i];
	    scratchString[3] = (char)0;
	  }
	  if (plotTimeInterval < 500.0)
	    plotIt = TRUE;
	  else if ((plotTimeInterval >= 500.0) && (plotTimeInterval < 1000.0) && ((sMonth-1) % 2) == 0)
	    plotIt = TRUE;
	  else if ((plotTimeInterval >= 1000.0) && (plotTimeInterval < 1500.0) && ((sMonth-1) % 3) == 0)
	    plotIt = TRUE;
	  else if ((plotTimeInterval >= 1500.0) && (plotTimeInterval < 2000.0) && ((sMonth-1) % 4) == 0)
	    plotIt = TRUE;
	  else if ((plotTimeInterval >= 2000.0) && (plotTimeInterval < 2500.0) && ((sMonth-1) % 6) == 0)
	    plotIt = TRUE;
	  else if ((plotTimeInterval >= 2500.0) && (sMonth == 1))
	    plotIt = TRUE;
	  else
	    plotIt = FALSE;
	  if (plotIt) {
	    screenCoordinates(ptr->time, minWeight, &x, &y);
	    renderPangoText(scratchString, OR_WHITE, SMALL_PANGO_FONT,
			    &tWidth, &tHeight, pixmap, x, displayHeight-35, 0.0, TRUE, 0, OR_BLACK);
	    y = displayHeight - BOTTOM_BORDER;
	    gdk_draw_line(pixmap, gC[OR_WHITE], x, y, x, y-10);
	  }
	}
      }
      ptr = ptr->next;
    }
    if (calcStats(7, &weightLastWeek, &weightNow, NULL, NULL, NULL)) {
      if (weightNow < weightLastWeek) {
	if (weightkg)
	  sprintf(scratchString, "Current Weight %5.1f kg, Last week's weight loss %4.2f kg, Daily deficit %1.0f calories",
		  weightNow, weightLastWeek - weightNow, (weightLastWeek - weightNow)*500.0/kg_per_lb);
	else
	  sprintf(scratchString, "Current Weight %5.1f lbs, Last week's weight loss %4.2f lbs, Daily deficit %1.0f calories",
		  weightNow, weightLastWeek - weightNow, (weightLastWeek - weightNow)*500.0);
      } else {
	if (weightkg)
	  sprintf(scratchString, "Current Weight %5.1f kg, Last week's weight gain %4.2f kg, Daily excess %1.0f calories",
		  weightNow, -(weightLastWeek - weightNow), -(weightLastWeek - weightNow)*500.0/kg_per_lb);
	else
	  sprintf(scratchString, "Current Weight %5.1f lbs, Last week's weight gain %4.2f lbs, Daily excess %1.0f calories",
		  weightNow, -(weightLastWeek - weightNow), -(weightLastWeek - weightNow)*500.0);
      }
      renderPangoText(scratchString, OR_WHITE, SMALL_PANGO_FONT,
		      &tWidth, &tHeight, pixmap, displayWidth/2, displayHeight-13, 0.0, TRUE, 0, OR_BLACK);
      renderPangoText(scratchString, OR_WHITE, SMALL_PANGO_FONT,
		      &tWidth, &tHeight, pixmap, displayWidth/2, displayHeight-13, 0.0, TRUE, 0, OR_BLACK);
    }
    
    ptr = logRoot;
    while (ptr != NULL) {
      if (ptr->time > stopTJD) {
	screenCoordinates(ptr->time, ptr->weight,
			  &dataPoints[nPoints].x, &dataPoints[nPoints].y);
	screenCoordinates(ptr->time, calculateWeightedAverage(ptr),
			  &linePoints[nCurvePoints].x, &linePoints[nCurvePoints].y);
	if ((linePoints[nCurvePoints].y > maxPlottableY)
	    && (linePoints[nCurvePoints].y < minPlottableY)) {
	  lineTops[nPoints].y = linePoints[nCurvePoints].y;
	  nCurvePoints++;
	} else if (linePoints[nCurvePoints].y < maxPlottableY)
	  lineTops[nPoints].y = maxPlottableY;
	else
	  lineTops[nPoints].y = minPlottableY;
	nPoints++;
      }
      ptr = ptr->next;
    }
    if (showComments) {
      int x, y;
      int firstComment = TRUE;

      ptr = logRoot;
      while (ptr != NULL) {
	if (ptr->time > stopTJD) {
	  if (ptr->comment) {
	    screenCoordinates(ptr->time, calculateWeightedAverage(ptr), &x, &y);
	    renderPangoText(ptr->comment, OR_PINK, SMALL_PANGO_FONT,
			    &tWidth, &tHeight, pixmap, x, plotCenter, -M_PI*0.5, TRUE, 0, OR_DARK_GREY);
	    if (firstComment) {
	      renderPangoText(ptr->comment, OR_PINK, SMALL_PANGO_FONT,
			      &tWidth, &tHeight, pixmap, x, plotCenter, -M_PI*0.5, TRUE, 0, OR_DARK_GREY);
	      firstComment = FALSE;
	    }
	  }
	}
	ptr = ptr->next;
      }
    }
    gdk_draw_lines(pixmap, gC[OR_YELLOW], linePoints, nCurvePoints);
    if (plotTimeInterval < 14.0)
      inc = 5;
    else if (plotTimeInterval < 100.0)
      inc = 4;
    else if (plotTimeInterval < 200.0)
      inc = 3;
    else if (plotTimeInterval < 300.0)
      inc = 2;
    else
      inc = 1;
    for (i = 0; i < nPoints; i++) {
      box[0].x = dataPoints[i].x;
      box[0].y = dataPoints[i].y + inc;
      box[1].x = dataPoints[i].x + inc;
      box[1].y = dataPoints[i].y;
      box[2].x = dataPoints[i].x;
      box[2].y = dataPoints[i].y - inc;
      box[3].x = dataPoints[i].x - inc;
      box[3].y = dataPoints[i].y;
      gdk_draw_polygon(pixmap, gC[OR_WHITE], TRUE, box, 4);
      if (plotTimeInterval < 200.0) {
	if (box[0].y < lineTops[i].y - 2)
	  gdk_draw_line(pixmap, gC[badColor], box[0].x, box[0].y, box[0].x, lineTops[i].y-1);
	if (box[2].y > lineTops[i].y + 2)
	  gdk_draw_line(pixmap, gC[goodColor], box[0].x, box[2].y, box[0].x, lineTops[i].y+1);
      }
    }
    free(dataPoints); free(linePoints); free(lineTops);
  }
  gdk_draw_drawable(drawingArea->window,
		    drawingArea->style->fg_gc[GTK_WIDGET_STATE (drawingArea)],
		    pixmap,
		    0,0,0,0,
		    displayWidth, displayHeight);
}

/*
  This function is called when a portion of the drawing area
  is exposed.
*/
static gboolean exposeEvent(GtkWidget *widget, GdkEventExpose *event)
{
  dprintf("In exposeEvnet()\n");
  gdk_draw_drawable (widget->window,
                     widget->style->fg_gc[GTK_WIDGET_STATE (widget)],
                     pixmap,
                     event->area.x, event->area.y,
                     event->area.x, event->area.y,
                     event->area.width, event->area.height);
  return(FALSE);
}

/*
  This function creates and initializes a backing pixmap
  of the appropriate size for the drawing area.
*/
static int configureEvent(GtkWidget *widget, GdkEventConfigure *event)
{
  displayHeight = widget->allocation.height;
  displayWidth  = widget->allocation.width;
  if (!cairoPixmap)
    cairoPixmap = gdk_pixmap_new(widget->window, 1100, 1100, -1);
  if (pixmap)
    g_object_unref(pixmap);
  pixmap = gdk_pixmap_new(widget->window,
			  widget->allocation.width,
			  widget->allocation.height, -1);
  dprintf("In configureEvent, w: %d, h: %d\n",
	  displayWidth, displayHeight);
  redrawScreen();
  return TRUE;
}

/*
  Make all the Graphic Contexts the program will need.
*/
static void makeGraphicContexts(GtkWidget *widget)
{
  int stat, i;
  GdkGCValues gCValues;
  GdkColor gColor;
  GdkGCValuesMask gCValuesMask;

  gCValuesMask = GDK_GC_FOREGROUND;

  for (i = 0; i < N_COLORS; i++) {
    gColor.red   = orreryColorRGB[i][0];
    gColor.green = orreryColorRGB[i][1];
    gColor.blue  = orreryColorRGB[i][2];
    stat = gdk_colormap_alloc_color(widget->style->colormap,
				    &gColor,
				    FALSE,
				    TRUE);
    if (stat != TRUE) {
      fprintf(stderr, "Error allocating color %d\n", i);
      exit(ERROR_EXIT);
    }
    gCValues.foreground = gColor;
    gC[i] = gtk_gc_get(widget->style->depth,
		       widget->style->colormap,
		       &gCValues, gCValuesMask
		       );
    if (gC[i] == NULL) {
      fprintf(stderr, "gtk_gc_get failed for color %d\n", i);
      exit(ERROR_EXIT);
    }
  }
}

/*
  Set up the fonts I need.
*/
static void makeFonts(GtkWidget *widget)
{
  smallFont =
    gdk_font_load("-misc-fixed-medium-r-semicondensed--13-100-100-100-c-60-iso8859-1");
  if (smallFont == NULL) {
    fprintf(stderr, "Error making smallFont\n");
    exit(ERROR_EXIT);
  }
  bigFont =
    gdk_font_load("-misc-fixed-medium-r-semicondensed--13-120-75-75-c-60-iso8859-1");
  if (bigFont == NULL) {
    fprintf(stderr, "Error making bigFont\n");
    exit(ERROR_EXIT);
  }
}

GtkWidget *selector;
int minSelectorWeight = 50;
int maxSelectorWeight = 300;

/*
  This function creates the selector for the weight "picker".
*/
static GtkWidget *createTouchSelector(void)
{
  int weight, inc;
  char weightString[10];
  GtkWidget *selector;
  GtkListStore *model;
  GtkTreeIter iter;
  HildonTouchSelectorColumn *column = NULL;

  selector = hildon_touch_selector_new();
  model = gtk_list_store_new(1, G_TYPE_STRING);
  if (logRoot != NULL) {
    if (weightkg)
      minSelectorWeight = (int)(lastEntry->weight) - 35;
    else
      minSelectorWeight = (int)(lastEntry->weight) - 75;
    if (minSelectorWeight < 5)
      minSelectorWeight = 5;
    if (weightkg)
      maxSelectorWeight = minSelectorWeight + 75;
    else
      maxSelectorWeight = minSelectorWeight + 150;
  } else {
    if (weightkg) {
      minSelectorWeight = (int)(50.0*kg_per_lb);
      maxSelectorWeight = (int)(250.0*kg_per_lb);
    } else {
      minSelectorWeight = 50;
      maxSelectorWeight = 250;
    }
  }
  if (weightkg)
    inc = 1;
  else
    inc = 2;
  for (weight = minSelectorWeight*10; weight <= maxSelectorWeight*10; weight += inc) {
    sprintf(weightString, "%5.1f", 0.1*(float)weight);
    gtk_list_store_append(model, &iter);
    gtk_list_store_set(model, &iter, 0, weightString, -1);
  }
  column = hildon_touch_selector_append_text_column(HILDON_TOUCH_SELECTOR(selector),
                                                     GTK_TREE_MODEL(model), TRUE);
  g_object_set(G_OBJECT(column), "text-column", 0, NULL);  
  return selector;
}

void writeFile(FILE *dataFile)
{
 int day, month, year, hours, minutes;
  double dayFrac, jD;
  logEntry *ptr;

  ptr = logRoot;
  while (ptr != NULL) {
    dayFrac = ptr->time - (int)ptr->time;
    jD = (double)((int)ptr->time);
    hours = (int)(dayFrac*24.0);
    minutes = (int)((dayFrac - ((double)hours)/24.0)*1440.0 + 0.5);
    tJDToDate(jD, &year, &month, &day);
    if (ptr->comment == NULL)
      fprintf(dataFile, "%02d/%02d/%d %02d:%02d %5.1f\n",
	      day, month, year, hours, minutes, ptr->weight);
    else
      fprintf(dataFile, "%02d/%02d/%d %02d:%02d %5.1f # %s\n",
	      day, month, year, hours, minutes, ptr->weight, ptr->comment);
    ptr = ptr->next;
  }
  fclose(dataFile);
}

void writeDataFile(void)
{
  FILE *dataFile;

  dataFile = fopen(fileName, "w");
  if (dataFile == NULL)
    fprintf(stderr, "Cannot write weight data file (%s)\n", fileName);
  else
    writeFile(dataFile);
  dataFile = fopen(backupFile, "w");
  if (dataFile == NULL)
    fprintf(stderr, "Cannot write backup data file (%s)\n", backupFile);
  else
    writeFile(dataFile);
}

void insertQueueEntry(double fJD, float weight, char *comment)
{
  logEntry  *newEntry;

  dprintf("In insertQueueEntry(%f, %f, \"%s\")\n", fJD, weight, comment);
  newEntry = (logEntry *)malloc(sizeof(logEntry));
  if (newEntry == NULL) {
    perror("malloc of newEntry");
    exit(ERROR_EXIT);
  }
  newEntry->time = fJD;
  newEntry->weight = weight;
  newEntry->comment = comment;
  newEntry->next = NULL;
  if (logRoot == NULL) {
    logRoot = newEntry;
    newEntry->last = NULL;
  } else {
    newEntry->last = lastEntry;
    lastEntry->next = newEntry;
  }
  lastEntry = newEntry;
  dprintf("Pre: minJD = %f, maxJD = %f, minWeight = %f maxWeight = %f\n",
	 minJD, maxJD, minWeight, maxWeight);
  nDataPoints++;
  if (nDataPoints == 1) {
    maxJD = minJD = fJD;
    maxWeight = minWeight = weight;
  }
  calcMaxima(0.0);
  dprintf("Post: minJD = %f, maxJD = %f, minWeight = %f maxWeight = %f\n",
	 minJD, maxJD, minWeight, maxWeight);
  writeDataFile();
}

GtkWidget *dataEntryStackable, *settingsStackable, *logStackable;
int cancelled;

void dataEntryCancelCallback(GtkButton *button, gpointer userData)
{
  cancelled = TRUE;
  gtk_widget_destroy(dataEntryStackable);
}

GtkWidget *dateButton, *timeButton;
void checkData(void)
{
  int iJD, i;
  int commentSum = 0;
  int defaultCommentSum = 0;
  float weight;
  double fJD;
  char *weightResult;
  char *comment = NULL;
  guint year, month, day, hours, minutes;

  if (!cancelled) {
    hildon_date_button_get_date((HildonDateButton *)dateButton, &year, &month, &day);
    hildon_time_button_get_time((HildonTimeButton *)timeButton, &hours, &minutes);
    month += 1;
    weightResult = (char *)hildon_button_get_value(HILDON_BUTTON(weightButton));
    sscanf(weightResult, "%f", &weight);
    iJD = calculateJulianDate(day, month, year);
    fJD = (double)iJD + ((double)hours)/24.0 + ((double)minutes)/1440.0;
    comment = (char *)gtk_entry_get_text((GtkEntry *)commentText);
    for (i = 0; i < strlen(comment); i++)
      commentSum += comment[i] << i;
    for (i = 0; i < strlen(defaultComment); i++)
      defaultCommentSum += defaultComment[i] << i;
    dprintf("Got %d/%d/%d  %02d:%02d %f %f \"%s\" %d\n",
	   day, month, year, hours, minutes, weight, fJD, comment, commentSum);
    if (commentSum == defaultCommentSum) {
      dprintf("Discarding non-comment\n");
      comment = NULL;
    }
    insertQueueEntry(fJD, weight, comment);
  }
  redrawScreen();
}

void dataEntryButtonClicked(GtkButton *button, gpointer userData)
{
  static GtkWidget *weightSelector, *dataEntryTable;

  dprintf("in dataEntryButtonClicked()\n");
  cancelled = FALSE;
  dataEntryTable = gtk_table_new(1, 5, FALSE);

  dateButton = hildon_date_button_new(HILDON_SIZE_AUTO, HILDON_BUTTON_ARRANGEMENT_VERTICAL);
  gtk_table_attach(GTK_TABLE(dataEntryTable), dateButton, 0, 1, 0, 1,
		   GTK_EXPAND, GTK_EXPAND, 0, 0);
  
  timeButton = hildon_time_button_new(HILDON_SIZE_AUTO, HILDON_BUTTON_ARRANGEMENT_VERTICAL);
  gtk_table_attach(GTK_TABLE(dataEntryTable), timeButton, 0, 1, 1, 2,
		   GTK_EXPAND, GTK_EXPAND, 0, 0);
  
  dataEntryStackable = hildon_stackable_window_new();
  g_signal_connect(G_OBJECT(dataEntryStackable), "destroy",
		   G_CALLBACK(checkData), NULL);
  weightSelector = createTouchSelector();
  weightButton = hildon_picker_button_new(HILDON_SIZE_AUTO, HILDON_BUTTON_ARRANGEMENT_VERTICAL);
  if (lastEntry == NULL)
    hildon_touch_selector_set_active(HILDON_TOUCH_SELECTOR(weightSelector), 0, 500);
  else {
    if (weightkg)
      hildon_touch_selector_set_active(HILDON_TOUCH_SELECTOR(weightSelector), 0,
				       (int)((10.0*lastEntry->weight)+0.5) - 10*minSelectorWeight);
    else
      hildon_touch_selector_set_active(HILDON_TOUCH_SELECTOR(weightSelector), 0,
				       (int)((5.0*lastEntry->weight)+0.5) - 5*minSelectorWeight);
  }
  if (weightkg)
    hildon_button_set_title(HILDON_BUTTON(weightButton), "Weight (kg)");
  else
    hildon_button_set_title(HILDON_BUTTON(weightButton), "Weight (lbs)");
  hildon_picker_button_set_selector(HILDON_PICKER_BUTTON(weightButton),
				    HILDON_TOUCH_SELECTOR(weightSelector));
  gtk_table_attach(GTK_TABLE(dataEntryTable), weightButton, 0, 1, 2, 3,
		   GTK_EXPAND, GTK_EXPAND, 0, 0);

  commentText = gtk_entry_new();
  gtk_entry_set_text((GtkEntry *)commentText, defaultComment);
  gtk_table_attach(GTK_TABLE(dataEntryTable), commentText, 0, 1, 3, 4,
		   GTK_EXPAND | GTK_FILL, GTK_FILL, 0, 0);

  dataEntryCancel = gtk_button_new_with_label("Cancel Data Entry");
  g_signal_connect (G_OBJECT(dataEntryCancel), "clicked",
		    G_CALLBACK(dataEntryCancelCallback), NULL);
  gtk_table_attach(GTK_TABLE(dataEntryTable), dataEntryCancel, 0, 1, 4, 5,
		   GTK_EXPAND, GTK_EXPAND, 0, 0);
  
  gtk_container_add(GTK_CONTAINER(dataEntryStackable), dataEntryTable);
  gtk_widget_show_all(dataEntryStackable);
}

void chartButtonClicked(GtkButton *button, gpointer userData)
{
  dprintf("in chartButtonClicked()\n");
}

GtkWidget *trendStackable;
int mainBoxInTrendStackable = FALSE;

typedef struct logCell {
  GdkPoint        box[4];
  logEntry       *ptr;
  struct logCell *next;
} logCell;
logCell *logCellRoot = NULL;
logCell *lastLogCell = NULL;
int logDisplayed = FALSE;

void trendStackableDestroyed(void)
{
  if (logDisplayed) {
    logCell *ptr, *tptr;

    ptr = logCellRoot;
    while (ptr != NULL) {
      tptr = ptr;
      ptr  = ptr->next;
      free(tptr);
    }
    logCellRoot = NULL;
    logDisplayed = FALSE;
  }
  gtk_widget_ref(mainBox);
  gtk_container_remove(GTK_CONTAINER(trendStackable), mainBox);
  gtk_container_add(GTK_CONTAINER(window), mainBox);
  gtk_widget_show(mainBox);
  gtk_widget_unref(mainBox);
  mainBoxInTrendStackable = FALSE;
  gdk_draw_drawable(drawingArea->window,
		    drawingArea->style->fg_gc[GTK_WIDGET_STATE (drawingArea)],
		    pixmap,
		    0,0,0,0,
		    displayWidth, displayHeight);
}

unsigned char normalYear[12] = {0, 3, 3, 6, 1, 4, 6, 2, 5, 0, 3, 5};
unsigned char leapYear[2] = {6, 2};

/*
  Return the day of the week (0->6) for year, month, day
*/
int dayOfWeek(int year, int month, int day)
{
  int century, c, weekDay, yy, y, m, leap;

  century = year / 100;
  yy = year - century*100;
  c = 2*(3 - (century % 4));
  y = yy + yy/4;
  if ((year % 400) == 0)
    leap = TRUE;
  else if ((year % 100) == 0)
    leap = FALSE;
  else if ((year % 4) == 0)
    leap = TRUE;
  else
    leap = FALSE;
  if (leap && (month < 3))
    m = (int)leapYear[month-1];
  else
    m = (int)normalYear[month-1];
  weekDay = (c + y + m + day) % 7;
  return(weekDay);
}

void logCallback(void)
{
  int year, month, tWidth, tHeight;
  int row = 0;
  int col = 0;
  char *logMonthString;
  char monthString[10];
  logEntry *ptr;
  logCell *cellPtr;
  GdkPoint box[4];

  dprintf("In logCallback()\n");
  logDisplayed = TRUE;
  logMonthString = (char *)hildon_button_get_value(HILDON_BUTTON(logSelectorButton));
  logMonthString[strlen(logMonthString)-1] = (char)0;
  dprintf("\"%s\" was selected.\n", logMonthString);
  sscanf(logMonthString, "%s %d", &monthString[0], &year);
  dprintf("Month Name \"%s\", Year: %d\n", monthString, year);
  for (month = 0; month < 12; month++)
    if (!strcmp(monthString, monthName[month]))
      break;
  month++;
  dprintf("Month number %d\n", month);
  gtk_widget_destroy(logStackable);
  if (!mainBoxInTrendStackable) {
    trendStackable = hildon_stackable_window_new();
    g_signal_connect(G_OBJECT(trendStackable), "destroy",
		     G_CALLBACK(trendStackableDestroyed), NULL);
    gtk_widget_ref(mainBox);
    gtk_widget_hide(mainBox);
    gtk_container_remove(GTK_CONTAINER(window), mainBox);
    gtk_container_add(GTK_CONTAINER(trendStackable), mainBox);
    gtk_widget_unref(mainBox);
    mainBoxInTrendStackable = TRUE;
  }
  gtk_widget_show_all(trendStackable);
  gdk_draw_rectangle(pixmap, drawingArea->style->black_gc,
		     TRUE, 0, 0,
		     drawingArea->allocation.width,
		     drawingArea->allocation.height);
  renderPangoText(logMonthString, OR_WHITE, BIG_PANGO_FONT,
		  &tWidth, &tHeight, pixmap, displayWidth/2, 20, 0.0, TRUE, 0, OR_BLACK);
  renderPangoText(logMonthString, OR_WHITE, BIG_PANGO_FONT,
		  &tWidth, &tHeight, pixmap, displayWidth/2, 20, 0.0, TRUE, 0, OR_BLACK);
  ptr = logRoot;
  while (ptr != NULL) {
    int day, pDay, pMonth, pYear;

    tJDToDate((double)((int)ptr->time), &pYear, &pMonth, &pDay);
    if ((pMonth == month) && (pYear == year)) {
      int i;
      double trend;
      char shortDayName[4];

      day = dayOfWeek(pYear, pMonth, pDay);
      trend = calculateWeightedAverage(ptr);
      for (i = 0; i < 3; i++)
	shortDayName[i] = dayName[day][i];
      shortDayName[3] = (char)0;
      sprintf(scratchString, "%2d", pDay);
      renderPangoText(scratchString, OR_WHITE, MEDIUM_PANGO_FONT,
		      &tWidth, &tHeight, pixmap, 20+col*displayWidth/3, 70+row*30, 0.0, FALSE, 0, OR_BLACK);
      renderPangoText(shortDayName, OR_WHITE, MEDIUM_PANGO_FONT,
		      &tWidth, &tHeight, pixmap, 55+col*displayWidth/3, 70+row*30, 0.0, FALSE, 0, OR_BLACK);
      sprintf(scratchString, "%5.1f", ptr->weight);
      if (nonjudgementalColors)
	renderPangoText(scratchString, OR_BLUE, MEDIUM_PANGO_FONT,
			&tWidth, &tHeight, pixmap, 115+col*displayWidth/3, 70+row*30, 0.0, FALSE, 0, OR_BLACK);
      else if (ptr->weight <= trend)
	renderPangoText(scratchString, goodColor, MEDIUM_PANGO_FONT,
			&tWidth, &tHeight, pixmap, 115+col*displayWidth/3, 70+row*30, 0.0, FALSE, 0, OR_BLACK);
      else
	renderPangoText(scratchString, badColor, MEDIUM_PANGO_FONT,
			&tWidth, &tHeight, pixmap, 115+col*displayWidth/3, 70+row*30, 0.0, FALSE, 0, OR_BLACK);
      sprintf(scratchString, "%5.1f", trend);
      renderPangoText(scratchString, OR_YELLOW, MEDIUM_PANGO_FONT,
		      &tWidth, &tHeight, pixmap, 185+col*displayWidth/3, 70+row*30, 0.0, FALSE, 0, OR_BLACK);
      box[0].x = 17+col*displayWidth/3; box[0].y = 58+row*30;
      box[1].x = box[0].x + 228;        box[1].y = box[0].y;
      box[2].x = box[1].x;              box[2].y = box[1].y + 26;
      box[3].x = box[0].x;              box[3].y = box[2].y;
      gdk_draw_polygon(pixmap, gC[OR_DARK_GREY], FALSE, box, 4);
      box[0].x = 18+col*displayWidth/3; box[0].y = 59+row*30;
      box[1].x = box[0].x + 226;        box[1].y = box[0].y;
      box[2].x = box[1].x;              box[2].y = box[1].y + 24;
      box[3].x = box[0].x;              box[3].y = box[2].y;
      gdk_draw_polygon(pixmap, gC[OR_DARK_GREY], FALSE, box, 4);
      cellPtr = (logCell *)malloc(sizeof(logCell));
      if (cellPtr != NULL) {
	for (i = 0; i < 4; i++) {
	  cellPtr->box[i].x = box[i].x;
	  cellPtr->box[i].y = box[i].y;
	}
	cellPtr->ptr = ptr;
	cellPtr->next = NULL;
	if (logCellRoot == NULL) {
	  logCellRoot = cellPtr;
	} else {
	  lastLogCell->next = cellPtr;
	}
	lastLogCell = cellPtr;
      }
      row++;
      if (row > 10) {
	col++;
	row = 0;
      }
    }
    ptr = ptr->next;
  }
}

void logButtonClicked(GtkButton *button, gpointer userData)
{
  int lastMonth = -1;
  int lastYear = -1;
  int year, month, day;
  int nMonths = 0;
  char monthString[50];
  logEntry *ptr;
  GtkWidget *logTable, *monthSelector;
  GtkListStore *model;
  GtkTreeIter iter;
  HildonTouchSelectorColumn *column = NULL;

  dprintf("in logButtonClicked()\n");
  if (nDataPoints > 0) {
    logStackable = hildon_stackable_window_new();
    g_signal_connect(G_OBJECT(logStackable), "destroy",
		     G_CALLBACK(logCallback), NULL);
    logTable = gtk_table_new(1, 2, FALSE);
    monthSelector = hildon_touch_selector_new();
    model = gtk_list_store_new(1, G_TYPE_STRING);
    ptr = logRoot;
    while (ptr != NULL) {
      tJDToDate(ptr->time, &year, &month, &day);
      if ((month != lastMonth) || (year != lastYear)) {
	sprintf(monthString, "%s %d\n", monthName[month-1], year);
	gtk_list_store_append(model, &iter);
	gtk_list_store_set(model, &iter, 0, monthString, -1);
	nMonths++;
	lastYear = year;
	lastMonth = month;
      }
      ptr = ptr->next;
    }
    column = hildon_touch_selector_append_text_column(HILDON_TOUCH_SELECTOR(monthSelector),
						      GTK_TREE_MODEL(model), TRUE);
    g_object_set(G_OBJECT(column), "text-column", 0, NULL);  
    logSelectorButton = hildon_picker_button_new(HILDON_SIZE_AUTO, HILDON_BUTTON_ARRANGEMENT_VERTICAL);
    hildon_touch_selector_set_active(HILDON_TOUCH_SELECTOR(monthSelector), 0, nMonths-1);
    hildon_button_set_title(HILDON_BUTTON(logSelectorButton), "Month to examine");
    hildon_picker_button_set_selector(HILDON_PICKER_BUTTON(logSelectorButton),
				      HILDON_TOUCH_SELECTOR(monthSelector));
    hildon_touch_selector_set_active(HILDON_TOUCH_SELECTOR(monthSelector), 0, nMonths);
    gtk_table_attach(GTK_TABLE(logTable), logSelectorButton, 0, 1, 0, 1,
		   GTK_EXPAND, GTK_EXPAND, 0, 0);
    gtk_container_add(GTK_CONTAINER(logStackable), logTable);
    gtk_widget_show_all(logStackable);
  }
}

void trendButtonClicked(GtkButton *button, gpointer userData)
{
  int tWidth, tHeight, day, month, year, line;
  int nItems = 0;
  float calCon;
  float startWeightW, endWeightW, maxWeightW, minWeightW, aveWeightW;
  float startWeightF, endWeightF, maxWeightF, minWeightF, aveWeightF;
  float startWeightM, endWeightM, maxWeightM, minWeightM, aveWeightM;
  float startWeightQ, endWeightQ, maxWeightQ, minWeightQ, aveWeightQ;
  float startWeightY, endWeightY, maxWeightY, minWeightY, aveWeightY;
  float startWeightH, endWeightH, maxWeightH, minWeightH, aveWeightH;
  double deltaTime;
  GdkPoint box[4];

  dprintf("in trendButtonClicked()\n");
  if (weightkg)
    calCon = 1.0/kg_per_lb;
  else
    calCon = 1.0;
  if (!mainBoxInTrendStackable) {
    trendStackable = hildon_stackable_window_new();
    g_signal_connect(G_OBJECT(trendStackable), "destroy",
		     G_CALLBACK(trendStackableDestroyed), NULL);
    gtk_widget_ref(mainBox);
    gtk_widget_hide(mainBox);
    gtk_container_remove(GTK_CONTAINER(window), mainBox);
    gtk_container_add(GTK_CONTAINER(trendStackable), mainBox);
    gtk_widget_unref(mainBox);
    mainBoxInTrendStackable = TRUE;
  }
  gtk_widget_show_all(trendStackable);
  gdk_draw_rectangle(pixmap, drawingArea->style->black_gc,
		     TRUE, 0, 0,
		     drawingArea->allocation.width,
		     drawingArea->allocation.height);
  renderPangoText("Trend Analysis", OR_WHITE, BIG_PANGO_FONT,
		  &tWidth, &tHeight, pixmap, displayWidth/2, 20, 0.0, TRUE, 0, OR_BLACK);
  renderPangoText("Trend Analysis", OR_WHITE, BIG_PANGO_FONT,
		  &tWidth, &tHeight, pixmap, displayWidth/2, 20, 0.0, TRUE, 0, OR_BLACK);
  if (calcStats(7, &startWeightW, &endWeightW, &maxWeightW, &minWeightW, &aveWeightW))
    nItems++;
  if (calcStats(14, &startWeightF, &endWeightF, &maxWeightF, &minWeightF, &aveWeightF))
    nItems++;
  if (calcStats(30, &startWeightM, &endWeightM, &maxWeightM, &minWeightM, &aveWeightM))
    nItems++;
  if (calcStats(91, &startWeightQ, &endWeightQ, &maxWeightQ, &minWeightQ, &aveWeightQ))
    nItems++;
  if (calcStats(365, &startWeightY, &endWeightY, &maxWeightY, &minWeightY, &aveWeightY))
    nItems++;
  if (calcStats(-1, &startWeightH, &endWeightH, &maxWeightH, &minWeightH, &aveWeightH))
    nItems++;

  box[0].x = 0;            box[0].y = 100;
  box[1].x = displayWidth; box[1].y = 100;
  box[2].x = displayWidth; box[2].y = displayHeight - 15 - (6-nItems)*35;
  box[3].x = 0;            box[3].y = displayHeight - 15 - (6-nItems)*35;
  gdk_draw_polygon(pixmap, gC[OR_DARK_GREY], TRUE, box, 4);

  tJDToDate((double)((int)lastEntry->time), &year, &month, &day);
  if (monthFirst) {
    int temp;
    
    temp = day;
    day = month;
    month = temp;
  }
  sprintf(scratchString, "Intervals ending %d/%d/%d", day, month, year);
  renderPangoText(scratchString, OR_WHITE, MEDIUM_PANGO_FONT,
		  &tWidth, &tHeight, pixmap, displayWidth/2, 70, 0.0, TRUE, 0, OR_BLACK);
  renderPangoText("Last...", OR_WHITE, MEDIUM_PANGO_FONT,
		  &tWidth, &tHeight, pixmap, 5, 140, 0.0, FALSE, 0, OR_DARK_GREY); 
  
  if (!nonjudgementalColors) {
    renderPangoText("Gain", badColor, MEDIUM_PANGO_FONT,
		    &tWidth, &tHeight, pixmap, 134, 125, 0.0, FALSE, 0, OR_DARK_GREY); 
    renderPangoText("/", OR_WHITE, MEDIUM_PANGO_FONT,
		    &tWidth, &tHeight, pixmap, 194, 125, 0.0, FALSE, 0, OR_DARK_GREY);      
    renderPangoText("Loss", goodColor, MEDIUM_PANGO_FONT,
		    &tWidth, &tHeight, pixmap, 209, 125, 0.0, FALSE, 0, OR_DARK_GREY);
    if (weightkg)
      renderPangoText("kg / week", OR_WHITE, MEDIUM_PANGO_FONT,
		      &tWidth, &tHeight, pixmap, 144, 155, 0.0, FALSE, 0, OR_DARK_GREY);
    else
      renderPangoText("pounds / week", OR_WHITE, MEDIUM_PANGO_FONT,
		      &tWidth, &tHeight, pixmap, 118, 155, 0.0, FALSE, 0, OR_DARK_GREY);
  } else
    if (weightkg)
      renderPangoText("kg / week", OR_WHITE, MEDIUM_PANGO_FONT,
		      &tWidth, &tHeight, pixmap, 144, 140, 0.0, FALSE, 0, OR_DARK_GREY);
    else
      renderPangoText("pounds / week", OR_WHITE, MEDIUM_PANGO_FONT,
		      &tWidth, &tHeight, pixmap, 118, 140, 0.0, FALSE, 0, OR_DARK_GREY);

  if (!nonjudgementalColors) {
    renderPangoText("Excess", badColor, MEDIUM_PANGO_FONT,
		    &tWidth, &tHeight, pixmap, 336, 125, 0.0, FALSE, 0, OR_DARK_GREY);      
    renderPangoText("/", OR_WHITE, MEDIUM_PANGO_FONT,
		    &tWidth, &tHeight, pixmap, 419, 125, 0.0, FALSE, 0, OR_DARK_GREY);      
    renderPangoText("Deficit", goodColor, MEDIUM_PANGO_FONT,
		    &tWidth, &tHeight, pixmap, 433, 125, 0.0, FALSE, 0, OR_DARK_GREY);      
    renderPangoText("calories / day", OR_WHITE, MEDIUM_PANGO_FONT,
		    &tWidth, &tHeight, pixmap, 348, 155, 0.0, FALSE, 0, OR_DARK_GREY);
  } else
    renderPangoText("calories / day", OR_WHITE, MEDIUM_PANGO_FONT,
		    &tWidth, &tHeight, pixmap, 348, 140, 0.0, FALSE, 0, OR_DARK_GREY);

  renderPangoText("Weight Trend", OR_WHITE, MEDIUM_PANGO_FONT,
		  &tWidth, &tHeight, pixmap, 601, 125, 0.0, FALSE, 0, OR_DARK_GREY);      
  renderPangoText("Min.     Mean    Max.", OR_WHITE, MEDIUM_PANGO_FONT,
		  &tWidth, &tHeight, pixmap, 566, 155, 0.0, FALSE, 0, OR_DARK_GREY);

  for (line = 0; line < nItems; line++) {
    float temp;

    if (line == (nItems-1)) {
      renderPangoText("History", OR_WHITE, MEDIUM_PANGO_FONT,
		      &tWidth, &tHeight, pixmap, 5, 200+(35*line), 0.0, FALSE, 0, OR_DARK_GREY);
      deltaTime = lastEntry->time - logRoot->time;
      temp = (startWeightH-endWeightH)/(deltaTime/7.0);
      if (nonjudgementalColors) {
	sprintf(scratchString, "%4.2f", -temp);
	renderPangoText(scratchString, OR_WHITE, MEDIUM_PANGO_FONT,
			&tWidth, &tHeight, pixmap, 170, 200+(35*line), 0.0, FALSE, 0, OR_DARK_GREY);
      } else {
	sprintf(scratchString, "%4.2f", fabs(temp));
	if (temp < 0.0) 
	  renderPangoText(scratchString, badColor, MEDIUM_PANGO_FONT,
			  &tWidth, &tHeight, pixmap, 170, 200+(35*line), 0.0, FALSE, 0, OR_DARK_GREY);
	else
	  renderPangoText(scratchString, goodColor, MEDIUM_PANGO_FONT,
			  &tWidth, &tHeight, pixmap, 170, 200+(35*line), 0.0, FALSE, 0, OR_DARK_GREY);
      }
      temp = calCon*(startWeightH-endWeightH)*(3500.0/deltaTime);
      if (nonjudgementalColors) {
	sprintf(scratchString, "%4.0f", -temp);
	renderPangoText(scratchString, OR_WHITE, MEDIUM_PANGO_FONT,
		      &tWidth, &tHeight, pixmap, 380, 200+(35*line), 0.0, FALSE, 0, OR_DARK_GREY);
      } else {
	sprintf(scratchString, "%4.0f", fabs(temp));
	if (temp < 0.0)
	  renderPangoText(scratchString, badColor, MEDIUM_PANGO_FONT,
			  &tWidth, &tHeight, pixmap, 380, 200+(35*line), 0.0, FALSE, 0, OR_DARK_GREY);
	else
	  renderPangoText(scratchString, goodColor, MEDIUM_PANGO_FONT,
			  &tWidth, &tHeight, pixmap, 380, 200+(35*line), 0.0, FALSE, 0, OR_DARK_GREY);
      }
      sprintf(scratchString, "%5.1f    %5.1f    %5.1f", minWeightH, aveWeightH, maxWeightH);
      renderPangoText(scratchString, OR_WHITE, MEDIUM_PANGO_FONT,
		      &tWidth, &tHeight, pixmap, 561, 200+(35*line), 0.0, FALSE, 0, OR_DARK_GREY);
    } else
      switch (line) {
      case 0:
	renderPangoText("Week", OR_WHITE, MEDIUM_PANGO_FONT,
			&tWidth, &tHeight, pixmap, 5, 200+(35*line), 0.0, FALSE, 0, OR_DARK_GREY);      
	temp = startWeightW-endWeightW;
	if (nonjudgementalColors) {
	  sprintf(scratchString, "%4.2f", -temp);
	  renderPangoText(scratchString, OR_WHITE, MEDIUM_PANGO_FONT,
			  &tWidth, &tHeight, pixmap, 170, 200+(35*line), 0.0, FALSE, 0, OR_DARK_GREY);
	} else {
	  sprintf(scratchString, "%4.2f", fabs(temp));
	  if (temp < 0.0) 
	    renderPangoText(scratchString, badColor, MEDIUM_PANGO_FONT,
			    &tWidth, &tHeight, pixmap, 170, 200+(35*line), 0.0, FALSE, 0, OR_DARK_GREY);
	  else
	    renderPangoText(scratchString, goodColor, MEDIUM_PANGO_FONT,
			    &tWidth, &tHeight, pixmap, 170, 200+(35*line), 0.0, FALSE, 0, OR_DARK_GREY);
	}
	temp = calCon*(startWeightW-endWeightW)*500.0;
	if (nonjudgementalColors) {
	  sprintf(scratchString, "%4.0f", -temp);
	  renderPangoText(scratchString, OR_WHITE, MEDIUM_PANGO_FONT,
			  &tWidth, &tHeight, pixmap, 380, 200+(35*line), 0.0, FALSE, 0, OR_DARK_GREY);
	} else {
	  sprintf(scratchString, "%4.0f", fabs(temp));
	  if (temp < 0.0)
	    renderPangoText(scratchString, badColor, MEDIUM_PANGO_FONT,
			    &tWidth, &tHeight, pixmap, 380, 200+(35*line), 0.0, FALSE, 0, OR_DARK_GREY);
	  else
	    renderPangoText(scratchString, goodColor, MEDIUM_PANGO_FONT,
			    &tWidth, &tHeight, pixmap, 380, 200+(35*line), 0.0, FALSE, 0, OR_DARK_GREY);
	}
	sprintf(scratchString, "%5.1f    %5.1f    %5.1f", minWeightW, aveWeightW, maxWeightW);
	renderPangoText(scratchString, OR_WHITE, MEDIUM_PANGO_FONT,
			&tWidth, &tHeight, pixmap, 561, 200+(35*line), 0.0, FALSE, 0, OR_DARK_GREY);
	break;
      case 1:
	renderPangoText("Fortnight", OR_WHITE, MEDIUM_PANGO_FONT,
			&tWidth, &tHeight, pixmap, 5, 200+(35*line), 0.0, FALSE, 0, OR_DARK_GREY);      
	temp = (startWeightF-endWeightF)/2.0;
	if (nonjudgementalColors) {
	  sprintf(scratchString, "%4.2f", -temp);
	  renderPangoText(scratchString, OR_WHITE, MEDIUM_PANGO_FONT,
			  &tWidth, &tHeight, pixmap, 170, 200+(35*line), 0.0, FALSE, 0, OR_DARK_GREY);
	} else {
	  sprintf(scratchString, "%4.2f", fabs(temp));
	  if (temp < 0.0) 
	    renderPangoText(scratchString, badColor, MEDIUM_PANGO_FONT,
			    &tWidth, &tHeight, pixmap, 170, 200+(35*line), 0.0, FALSE, 0, OR_DARK_GREY);
	  else
	    renderPangoText(scratchString, goodColor, MEDIUM_PANGO_FONT,
			    &tWidth, &tHeight, pixmap, 170, 200+(35*line), 0.0, FALSE, 0, OR_DARK_GREY);
	}
	temp = calCon*(startWeightF-endWeightF)*250.0;
	if (nonjudgementalColors) {
	  sprintf(scratchString, "%4.0f", -temp);
	  renderPangoText(scratchString, OR_WHITE, MEDIUM_PANGO_FONT,
			  &tWidth, &tHeight, pixmap, 380, 200+(35*line), 0.0, FALSE, 0, OR_DARK_GREY);
	} else {
	  sprintf(scratchString, "%4.0f", fabs(temp));
	  if (temp < 0.0)
	    renderPangoText(scratchString, badColor, MEDIUM_PANGO_FONT,
			    &tWidth, &tHeight, pixmap, 380, 200+(35*line), 0.0, FALSE, 0, OR_DARK_GREY);
	  else
	    renderPangoText(scratchString, goodColor, MEDIUM_PANGO_FONT,
			    &tWidth, &tHeight, pixmap, 380, 200+(35*line), 0.0, FALSE, 0, OR_DARK_GREY);
	}
	sprintf(scratchString, "%5.1f    %5.1f    %5.1f", minWeightF, aveWeightF, maxWeightF);
	renderPangoText(scratchString, OR_WHITE, MEDIUM_PANGO_FONT,
			&tWidth, &tHeight, pixmap, 561, 200+(35*line), 0.0, FALSE, 0, OR_DARK_GREY);
	break;
      case 2:
	renderPangoText("Month", OR_WHITE, MEDIUM_PANGO_FONT,
			&tWidth, &tHeight, pixmap, 5, 200+(35*line), 0.0, FALSE, 0, OR_DARK_GREY);      
	temp = (startWeightM-endWeightM)/(30.0/7.0);
	if (nonjudgementalColors) {
	  sprintf(scratchString, "%4.2f", -temp);
	  renderPangoText(scratchString, OR_WHITE, MEDIUM_PANGO_FONT,
			  &tWidth, &tHeight, pixmap, 170, 200+(35*line), 0.0, FALSE, 0, OR_DARK_GREY);
	} else {
	  sprintf(scratchString, "%4.2f", fabs(temp));
	  if (temp < 0.0) 
	    renderPangoText(scratchString, badColor, MEDIUM_PANGO_FONT,
			    &tWidth, &tHeight, pixmap, 170, 200+(35*line), 0.0, FALSE, 0, OR_DARK_GREY);
	  else
	    renderPangoText(scratchString, goodColor, MEDIUM_PANGO_FONT,
			    &tWidth, &tHeight, pixmap, 170, 200+(35*line), 0.0, FALSE, 0, OR_DARK_GREY);
	}
	temp = calCon*(startWeightM-endWeightM)*116.6666667;
	if (nonjudgementalColors) {
	  sprintf(scratchString, "%4.0f", -temp);
	  renderPangoText(scratchString, OR_WHITE, MEDIUM_PANGO_FONT,
			  &tWidth, &tHeight, pixmap, 380, 200+(35*line), 0.0, FALSE, 0, OR_DARK_GREY);
	} else {
	  sprintf(scratchString, "%4.0f", fabs(temp));
	  if (temp < 0.0)
	    renderPangoText(scratchString, badColor, MEDIUM_PANGO_FONT,
			    &tWidth, &tHeight, pixmap, 380, 200+(35*line), 0.0, FALSE, 0, OR_DARK_GREY);
	  else
	    renderPangoText(scratchString, goodColor, MEDIUM_PANGO_FONT,
			    &tWidth, &tHeight, pixmap, 380, 200+(35*line), 0.0, FALSE, 0, OR_DARK_GREY);
	}
	sprintf(scratchString, "%5.1f    %5.1f    %5.1f", minWeightM, aveWeightM, maxWeightM);
	renderPangoText(scratchString, OR_WHITE, MEDIUM_PANGO_FONT,
			&tWidth, &tHeight, pixmap, 561, 200+(35*line), 0.0, FALSE, 0, OR_DARK_GREY);
	break;
      case 3:
	renderPangoText("Quarter", OR_WHITE, MEDIUM_PANGO_FONT,
			&tWidth, &tHeight, pixmap, 5, 200+(35*line), 0.0, FALSE, 0, OR_DARK_GREY);      
	temp = (startWeightQ-endWeightQ)/13.0;
	if (nonjudgementalColors) {
	  sprintf(scratchString, "%4.2f", -temp);
	  renderPangoText(scratchString, OR_WHITE, MEDIUM_PANGO_FONT,
			  &tWidth, &tHeight, pixmap, 170, 200+(35*line), 0.0, FALSE, 0, OR_DARK_GREY);
	} else {
	  sprintf(scratchString, "%4.2f", fabs(temp));
	  if (temp < 0.0) 
	    renderPangoText(scratchString, badColor, MEDIUM_PANGO_FONT,
			    &tWidth, &tHeight, pixmap, 170, 200+(35*line), 0.0, FALSE, 0, OR_DARK_GREY);
	  else
	    renderPangoText(scratchString, goodColor, MEDIUM_PANGO_FONT,
			    &tWidth, &tHeight, pixmap, 170, 200+(35*line), 0.0, FALSE, 0, OR_DARK_GREY);
	}
	temp = calCon*(startWeightQ-endWeightQ)*(3500.0/91.0);
	if (nonjudgementalColors) {
	  sprintf(scratchString, "%4.0f", -temp);
	  renderPangoText(scratchString, OR_WHITE, MEDIUM_PANGO_FONT,
			  &tWidth, &tHeight, pixmap, 380, 200+(35*line), 0.0, FALSE, 0, OR_DARK_GREY);
	} else {
	  sprintf(scratchString, "%4.0f", fabs(temp));
	  if (temp < 0.0)
	    renderPangoText(scratchString, badColor, MEDIUM_PANGO_FONT,
			    &tWidth, &tHeight, pixmap, 380, 200+(35*line), 0.0, FALSE, 0, OR_DARK_GREY);
	  else
	    renderPangoText(scratchString, goodColor, MEDIUM_PANGO_FONT,
			    &tWidth, &tHeight, pixmap, 380, 200+(35*line), 0.0, FALSE, 0, OR_DARK_GREY);
	}
	sprintf(scratchString, "%5.1f    %5.1f    %5.1f", minWeightQ, aveWeightQ, maxWeightQ);
	renderPangoText(scratchString, OR_WHITE, MEDIUM_PANGO_FONT,
			&tWidth, &tHeight, pixmap, 561, 200+(35*line), 0.0, FALSE, 0, OR_DARK_GREY);
	break;
      case 4:
	renderPangoText("Year", OR_WHITE, MEDIUM_PANGO_FONT,
			&tWidth, &tHeight, pixmap, 5, 200+(35*line), 0.0, FALSE, 0, OR_DARK_GREY);      
	temp = (startWeightY-endWeightY)/(365.0/7.0);
	if (nonjudgementalColors) {
	  sprintf(scratchString, "%4.2f", -temp);
	  renderPangoText(scratchString, OR_WHITE, MEDIUM_PANGO_FONT,
			  &tWidth, &tHeight, pixmap, 170, 200+(35*line), 0.0, FALSE, 0, OR_DARK_GREY);
	} else {
	  sprintf(scratchString, "%4.2f", fabs(temp));
	  if (temp < 0.0) 
	    renderPangoText(scratchString, badColor, MEDIUM_PANGO_FONT,
			    &tWidth, &tHeight, pixmap, 170, 200+(35*line), 0.0, FALSE, 0, OR_DARK_GREY);
	  else
	    renderPangoText(scratchString, goodColor, MEDIUM_PANGO_FONT,
			    &tWidth, &tHeight, pixmap, 170, 200+(35*line), 0.0, FALSE, 0, OR_DARK_GREY);
	}
	temp = calCon*(startWeightY-endWeightY)*(3500.0/365.0);
	if (nonjudgementalColors) {
	  sprintf(scratchString, "%4.0f", -temp);
	  renderPangoText(scratchString, OR_WHITE, MEDIUM_PANGO_FONT,
			  &tWidth, &tHeight, pixmap, 380, 200+(35*line), 0.0, FALSE, 0, OR_DARK_GREY);
	} else {
	  sprintf(scratchString, "%4.0f", fabs(temp));
	  if (temp < 0.0)
	    renderPangoText(scratchString, badColor, MEDIUM_PANGO_FONT,
			    &tWidth, &tHeight, pixmap, 380, 200+(35*line), 0.0, FALSE, 0, OR_DARK_GREY);
	  else
	    renderPangoText(scratchString, goodColor, MEDIUM_PANGO_FONT,
			    &tWidth, &tHeight, pixmap, 380, 200+(35*line), 0.0, FALSE, 0, OR_DARK_GREY);
	}
	sprintf(scratchString, "%5.1f    %5.1f    %5.1f", minWeightY, aveWeightY, maxWeightY);
	renderPangoText(scratchString, OR_WHITE, MEDIUM_PANGO_FONT,
			&tWidth, &tHeight, pixmap, 561, 200+(35*line), 0.0, FALSE, 0, OR_DARK_GREY);
	break;
      }
  }
}

void utilitiesButtonClicked(GtkButton *button, gpointer userData)
{
  dprintf("in utilitiesButtonClicked()\n");
}

void writeSettings(void)
{
  FILE *settings;

  settings = fopen(settingsFileName, "w");
  if (settings == NULL) {
    perror("writing settings");
    return;
  }
  fprintf(settings, "MY_HEIGHT %5.1f\n",          myHeight);
  fprintf(settings, "MY_TARGET %5.1f\n",          myTarget);
  fprintf(settings, "WEIGHT_KG %d\n",             weightkg);
  fprintf(settings, "HEIGHT_CM %d\n",             heightcm);
  fprintf(settings, "MONTH_FIRST %d\n",           monthFirst);
  fprintf(settings, "NONJUDGEMENTAL_COLORS %d\n", nonjudgementalColors);
  fprintf(settings, "HACKER_DIET_MODE %d\n",      hackerDietMode);
  fprintf(settings, "SHOW_COMMENTS %d\n",         showComments);
  fprintf(settings, "SHOW_TARGET %d\n",           showTarget);
  fprintf(settings, "PLOT_INTERVAL %d\n",         plotInterval);
  fclose(settings);
}

GtkWidget *heightSpinLabel, *heightSpin, *weightUnitLabel, *heightUnitLabel,
  *dateFormatLabel, *poundsButton, *kgButton, *inButton, *cmButton, *dayButton,
  *monthButton, *nonjudgementalButton, *hackerDietButton, *showCommentsButton,
  *plotWeekButton, *plotMonthButton, *plotQuarterButton, *plot6MonthButton,
  *plotYearButton, *plotHistoryButton, *targetSpinLabel, *targetSpin, *showTargetButton;

void checkSettings(void)
{
  dprintf("In checkSettings\n");
  myHeight = gtk_spin_button_get_value((GtkSpinButton *)heightSpin);
  myTarget = gtk_spin_button_get_value((GtkSpinButton *)targetSpin);
  if (GTK_TOGGLE_BUTTON(plotWeekButton)->active)
    plotInterval = 14;
  else if (GTK_TOGGLE_BUTTON(plotMonthButton)->active)
    plotInterval = 30;
  else if (GTK_TOGGLE_BUTTON(plotQuarterButton)->active)
    plotInterval = 91;
  else if (GTK_TOGGLE_BUTTON(plot6MonthButton)->active)
    plotInterval =183;
  else if (GTK_TOGGLE_BUTTON(plotYearButton)->active)
    plotInterval = 365;
  else
    plotInterval = 1000000000;
  if (GTK_TOGGLE_BUTTON(kgButton)->active)
    weightkg = TRUE;
  else
    weightkg = FALSE;
  if (GTK_TOGGLE_BUTTON(nonjudgementalButton)->active) {
    nonjudgementalColors = TRUE;
    goodColor = badColor = OR_BLUE;
  } else {
    nonjudgementalColors = FALSE;
    goodColor = OR_GREEN;
    badColor = OR_RED;
  }
  if (GTK_TOGGLE_BUTTON(hackerDietButton)->active)
    hackerDietMode = TRUE;
  else
    hackerDietMode = FALSE;
  if (GTK_TOGGLE_BUTTON(showCommentsButton)->active)
    showComments = TRUE;
  else
    showComments = FALSE;
  if (GTK_TOGGLE_BUTTON(showTargetButton)->active)
    showTarget = TRUE;
  else
    showTarget = FALSE;
  if (GTK_TOGGLE_BUTTON(cmButton)->active)
    heightcm = TRUE;
  else
    heightcm = FALSE;
  if (GTK_TOGGLE_BUTTON(monthButton)->active)
    monthFirst = TRUE;
  else
    monthFirst = FALSE;
  writeSettings();
  redrawScreen();
}

void settingsButtonClicked(GtkButton *button, gpointer userData)
{
  static GtkWidget *settingsTable;
  static GSList *weightGroup, *heightGroup, *dateGroup, *plotGroup;
  static GtkObject *heightAdjustment, *targetAdjustment;

  dprintf("in settingsButtonClicked()\n");
  settingsTable = gtk_table_new(3, 8, FALSE);

  heightUnitLabel = gtk_label_new("Height Unit");
  gtk_table_attach(GTK_TABLE(settingsTable), (GtkWidget *)heightUnitLabel, 0, 1, 0, 1,
		   GTK_EXPAND | GTK_FILL, GTK_FILL, 0, 0);
  inButton = gtk_radio_button_new_with_label(NULL, "inches");
  heightGroup = gtk_radio_button_group(GTK_RADIO_BUTTON(inButton));
  cmButton = gtk_radio_button_new_with_label(heightGroup, "centimeters");
  heightGroup = gtk_radio_button_group(GTK_RADIO_BUTTON(cmButton));
  if (heightcm)
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(cmButton), TRUE);
  else
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(inButton), TRUE);
  gtk_table_attach(GTK_TABLE(settingsTable), (GtkWidget *)inButton, 1, 2, 0, 1,
		   GTK_EXPAND | GTK_FILL, GTK_FILL, 0, 0);
  gtk_table_attach(GTK_TABLE(settingsTable), (GtkWidget *)cmButton, 2, 3, 0, 1,
		   GTK_EXPAND | GTK_FILL, GTK_FILL, 0, 0);

  heightSpinLabel = gtk_label_new("Your Height");
  gtk_table_attach(GTK_TABLE(settingsTable), (GtkWidget *)heightSpinLabel, 0, 1, 1, 2,
		   GTK_EXPAND | GTK_FILL, GTK_FILL, 0, 0);
  heightAdjustment = gtk_adjustment_new(myHeight, 10.0, 216.0, 0.5, 1.0, 0.0);
  heightSpin = gtk_spin_button_new((GtkAdjustment *)heightAdjustment, 0.5, 1);
  gtk_spin_button_set_numeric((GtkSpinButton *)heightSpin, TRUE);
  gtk_table_attach(GTK_TABLE(settingsTable), (GtkWidget *)heightSpin, 1, 2, 1, 2,
		   GTK_EXPAND | GTK_FILL, GTK_FILL, 0, 0);

  weightUnitLabel = gtk_label_new("Weight Unit");
  gtk_table_attach(GTK_TABLE(settingsTable), (GtkWidget *)weightUnitLabel, 0, 1, 2, 3,
		   GTK_EXPAND | GTK_FILL, GTK_FILL, 0, 0);
  poundsButton = gtk_radio_button_new_with_label(NULL, "pounds");
  weightGroup = gtk_radio_button_group(GTK_RADIO_BUTTON(poundsButton));
  kgButton = gtk_radio_button_new_with_label(weightGroup, "kilograms");
  weightGroup = gtk_radio_button_group(GTK_RADIO_BUTTON(kgButton));
  if (weightkg)
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(kgButton), TRUE);
  else
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(poundsButton), TRUE);
  gtk_table_attach(GTK_TABLE(settingsTable), (GtkWidget *)poundsButton, 1, 2, 2, 3,
		   GTK_EXPAND | GTK_FILL, GTK_FILL, 0, 0);
  gtk_table_attach(GTK_TABLE(settingsTable), (GtkWidget *)kgButton, 2, 3, 2, 3,
		   GTK_EXPAND | GTK_FILL, GTK_FILL, 0, 0);

  targetSpinLabel = gtk_label_new("Target Weight");
  gtk_table_attach(GTK_TABLE(settingsTable), (GtkWidget *)targetSpinLabel, 0, 1, 3, 4,
		   GTK_EXPAND | GTK_FILL, GTK_FILL, 0, 0);
  targetAdjustment = gtk_adjustment_new(myTarget, 60.0, 250.0, 0.5, 1.0, 0.0);
  targetSpin = gtk_spin_button_new((GtkAdjustment *)targetAdjustment, 0.5, 1);
  gtk_spin_button_set_numeric((GtkSpinButton *)targetSpin, TRUE);
  gtk_table_attach(GTK_TABLE(settingsTable), (GtkWidget *)targetSpin, 1, 2, 3, 4,
		   GTK_EXPAND | GTK_FILL, GTK_FILL, 0, 0);
  showTargetButton = gtk_check_button_new_with_label("Plot Target Weight");
  if (showTarget)
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(showTargetButton), TRUE);
  else
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(showTargetButton), FALSE);
  gtk_table_attach(GTK_TABLE(settingsTable), showTargetButton, 2, 3, 3, 4,
		   GTK_EXPAND | GTK_FILL, GTK_FILL, 0, 0);

  dateFormatLabel = gtk_label_new("Date Format");
  gtk_table_attach(GTK_TABLE(settingsTable), (GtkWidget *)dateFormatLabel, 0, 1, 4, 5,
		   GTK_EXPAND | GTK_FILL, GTK_FILL, 0, 0);
  dayButton = gtk_radio_button_new_with_label(NULL, "dd/mm/yyyy");
  dateGroup = gtk_radio_button_group(GTK_RADIO_BUTTON(dayButton));
  monthButton = gtk_radio_button_new_with_label(dateGroup, "mm/dd/yyyy");
  dateGroup = gtk_radio_button_group(GTK_RADIO_BUTTON(monthButton));
  if (monthFirst)
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(monthButton), TRUE);
  else
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(dayButton), TRUE);
  gtk_table_attach(GTK_TABLE(settingsTable), (GtkWidget *)dayButton, 1, 2, 4, 5,
		   GTK_EXPAND | GTK_FILL, GTK_FILL, 0, 0);
  gtk_table_attach(GTK_TABLE(settingsTable), (GtkWidget *)monthButton, 2, 3, 4, 5,
		   GTK_EXPAND | GTK_FILL, GTK_FILL, 0, 0);

  nonjudgementalButton = gtk_check_button_new_with_label("Nonjudgemental Colors");
  if (nonjudgementalColors)
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(nonjudgementalButton), TRUE);
  else
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(nonjudgementalButton), FALSE);
  gtk_table_attach(GTK_TABLE(settingsTable), nonjudgementalButton, 0, 1, 5, 6,
		   GTK_EXPAND | GTK_FILL, GTK_FILL, 0, 0);
  hackerDietButton = gtk_check_button_new_with_label("Hacker Diet Mode");
  if (hackerDietMode)
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(hackerDietButton), TRUE);
  else
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(hackerDietButton), FALSE);
  gtk_table_attach(GTK_TABLE(settingsTable), hackerDietButton, 1, 2, 5, 6,
		   GTK_EXPAND | GTK_FILL, GTK_FILL, 0, 0);
  showCommentsButton = gtk_check_button_new_with_label("Plot Comments");
  if (showComments)
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(showCommentsButton), TRUE);
  else
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(showCommentsButton), FALSE);
  gtk_table_attach(GTK_TABLE(settingsTable), showCommentsButton, 2, 3, 5, 6,
		   GTK_EXPAND | GTK_FILL, GTK_FILL, 0, 0);

  plotWeekButton = gtk_radio_button_new_with_label(NULL, "Plot Last 2 Weeks");
  plotGroup = gtk_radio_button_group(GTK_RADIO_BUTTON(plotWeekButton));
  plotMonthButton = gtk_radio_button_new_with_label(plotGroup, "Plot Last Month");
  plotGroup = gtk_radio_button_group(GTK_RADIO_BUTTON(plotMonthButton));
  plotQuarterButton = gtk_radio_button_new_with_label(plotGroup, "Plot Last Quarter");
  plotGroup = gtk_radio_button_group(GTK_RADIO_BUTTON(plotQuarterButton));
  plot6MonthButton = gtk_radio_button_new_with_label(plotGroup, "Plot Last 6 Months");
  plotGroup = gtk_radio_button_group(GTK_RADIO_BUTTON(plot6MonthButton));
  plotYearButton = gtk_radio_button_new_with_label(plotGroup, "Plot Last Year");
  plotGroup = gtk_radio_button_group(GTK_RADIO_BUTTON(plotYearButton));
  plotHistoryButton = gtk_radio_button_new_with_label(plotGroup, "Plot Entire History");
  plotGroup = gtk_radio_button_group(GTK_RADIO_BUTTON(plotHistoryButton));
  switch (plotInterval) {
  case 7:
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(plotWeekButton), TRUE);
    break;
  case 30:
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(plotMonthButton), TRUE);
    break;
  case 91:
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(plotQuarterButton), TRUE);
    break;
  case 183:
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(plot6MonthButton), TRUE);
    break;
  case 365:
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(plotYearButton), TRUE);
    break;
  default:
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(plotHistoryButton), TRUE);
  }
  gtk_table_attach(GTK_TABLE(settingsTable), plotWeekButton, 0, 1, 6, 7,
		   GTK_EXPAND | GTK_FILL, GTK_FILL, 0, 0);
  gtk_table_attach(GTK_TABLE(settingsTable), plotMonthButton, 1, 2, 6, 7,
		   GTK_EXPAND | GTK_FILL, GTK_FILL, 0, 0);
  gtk_table_attach(GTK_TABLE(settingsTable), plotQuarterButton, 2, 3, 6, 7,
		   GTK_EXPAND | GTK_FILL, GTK_FILL, 0, 0);
  gtk_table_attach(GTK_TABLE(settingsTable), plot6MonthButton, 0, 1, 7, 8,
		   GTK_EXPAND | GTK_FILL, GTK_FILL, 0, 0);
  gtk_table_attach(GTK_TABLE(settingsTable), plotYearButton, 1, 2, 7, 8,
		   GTK_EXPAND | GTK_FILL, GTK_FILL, 0, 0);
  gtk_table_attach(GTK_TABLE(settingsTable), plotHistoryButton, 2, 3, 7, 8,
		   GTK_EXPAND | GTK_FILL, GTK_FILL, 0, 0);

  settingsStackable = hildon_stackable_window_new();
  g_signal_connect(G_OBJECT(settingsStackable), "destroy",
		   G_CALLBACK(checkSettings), NULL);
  gtk_container_add(GTK_CONTAINER(settingsStackable), settingsTable);
  gtk_widget_show_all(settingsStackable);
}

void helpButtonClicked(GtkButton *button, gpointer userData)
{
  dprintf("in helpButtonClicked()\n");
  system("/usr/bin/dbus-send --system --type=method_call  --dest=\"com.nokia.osso_browser\" /com/nokia/osso_browser/request com.nokia.osso_browser.load_url string:\"wiki.maemo.org/maeFat\"");
}

void aboutButtonClicked(GtkButton *button, gpointer userData)
{
  static int firstCall = TRUE;
  static GtkTextBuffer *aboutBuffer;
  static GtkWidget *aboutTextWidget, *aboutStackable;

  dprintf("in aboutButtonClicked()\n");
  if (firstCall) {
    aboutBuffer = gtk_text_buffer_new(NULL);
    gtk_text_buffer_set_text(aboutBuffer, "maeFat Version 1.1\nCopyright (C) 2011, Ken Young\n\nThis program helps you keep track of your weight.\nThis is free, open source software, released under GPL version 2.\n\nPlease send comments, questions and feature requests to\norrery.moko@gmail.com", -1);
    aboutTextWidget = hildon_text_view_new(); 
    ((GtkTextView *)aboutTextWidget)->editable =
      ((GtkTextView *)aboutTextWidget)->cursor_visible = FALSE;
    gtk_widget_ref(aboutTextWidget);
    firstCall = FALSE;
  }
  hildon_text_view_set_buffer((HildonTextView *)aboutTextWidget, aboutBuffer);
  aboutStackable = hildon_stackable_window_new();
  gtk_container_add(GTK_CONTAINER(aboutStackable), aboutTextWidget);
  gtk_widget_show_all(aboutStackable);
}

HildonAppMenu *hildonMenu;
static void makeHildonButtons(void)
{
  HildonSizeType buttonSize = HILDON_SIZE_FINGER_HEIGHT | HILDON_SIZE_AUTO_WIDTH;

  hildonMenu = HILDON_APP_MENU(hildon_app_menu_new());

  /* Data Entry Button */
  dataEntryButton = hildon_gtk_button_new(buttonSize);
  gtk_button_set_label(GTK_BUTTON(dataEntryButton), "Data Entry");
  g_signal_connect(G_OBJECT(dataEntryButton), "clicked", G_CALLBACK(dataEntryButtonClicked), NULL);
  hildon_app_menu_append(hildonMenu, GTK_BUTTON(dataEntryButton));
  
  /* Chart Button */
  /*
  chartButton = hildon_gtk_button_new(buttonSize);
  gtk_button_set_label(GTK_BUTTON(chartButton), "Chart");
  g_signal_connect(G_OBJECT(chartButton), "clicked", G_CALLBACK(chartButtonClicked), NULL);
  hildon_app_menu_append(hildonMenu, GTK_BUTTON(chartButton));
  */

  /* Log  Button */
  logButton = hildon_gtk_button_new(buttonSize);
  gtk_button_set_label(GTK_BUTTON(logButton), "Log");
  g_signal_connect(G_OBJECT(logButton), "clicked", G_CALLBACK(logButtonClicked), NULL);
  hildon_app_menu_append(hildonMenu, GTK_BUTTON(logButton));

  /* Trend Button */
  trendButton = hildon_gtk_button_new(buttonSize);
  gtk_button_set_label(GTK_BUTTON(trendButton), "Trend Analysis");
  g_signal_connect(G_OBJECT(trendButton), "clicked", G_CALLBACK(trendButtonClicked), NULL);
  hildon_app_menu_append(hildonMenu, GTK_BUTTON(trendButton));

  /* Utilities Button */
  /*
  utilitiesButton = hildon_gtk_button_new(buttonSize);
  gtk_button_set_label(GTK_BUTTON(utilitiesButton), "Utilities");
  g_signal_connect(G_OBJECT(utilitiesButton), "clicked", G_CALLBACK(utilitiesButtonClicked), NULL);
  hildon_app_menu_append(hildonMenu, GTK_BUTTON(utilitiesButton));
  */

  /* Settings Button */
  settingsButton = hildon_gtk_button_new(buttonSize);
  gtk_button_set_label(GTK_BUTTON(settingsButton), "Settings");
  g_signal_connect(G_OBJECT(settingsButton), "clicked", G_CALLBACK(settingsButtonClicked), NULL);
  hildon_app_menu_append(hildonMenu, GTK_BUTTON(settingsButton));

  /* Help Button */
  helpButton = hildon_gtk_button_new(buttonSize);
  gtk_button_set_label(GTK_BUTTON(helpButton), "Help");
  g_signal_connect(G_OBJECT(helpButton), "clicked", G_CALLBACK(helpButtonClicked), NULL);
  hildon_app_menu_append(hildonMenu, GTK_BUTTON(helpButton));

  /* About Button */
  aboutButton = hildon_gtk_button_new(buttonSize);
  gtk_button_set_label(GTK_BUTTON(aboutButton), "About");
  g_signal_connect(G_OBJECT(aboutButton), "clicked", G_CALLBACK(aboutButtonClicked), NULL);
  hildon_app_menu_append(hildonMenu, GTK_BUTTON(aboutButton));

  hildon_stackable_window_set_main_menu((HildonStackableWindow *)window, hildonMenu);
  gtk_widget_show_all(GTK_WIDGET(hildonMenu));
}

static gboolean buttonPressEvent(GtkWidget *widget, GdkEventButton *event)
{
  dprintf("In buttonPressEvent\n");
  return(TRUE);
}

int deleted;
logEntry *editPtr;
GtkWidget *dateEditButton, *timeEditButton, *dataEditStackable;

void dataEditDeleteCallback(GtkButton *button, gpointer userData)
{
  deleted = TRUE;
  gtk_widget_destroy(dataEditStackable);
}

void editData(void)
{
  if (deleted) {
    if (editPtr->last == NULL)
      logRoot = editPtr;
    else {
      (editPtr->last)->next = editPtr->next;
      (editPtr->next)->last = editPtr->last;
    }
    free(editPtr);
  } else {
    int iJD, i;
    int commentSum = 0;
    int defaultCommentSum = 0;
    float weight;
    double fJD;
    char *weightResult;
    char *comment = NULL;
    guint year, month, day, hours, minutes;
    
    hildon_date_button_get_date((HildonDateButton *)dateEditButton, &year, &month, &day);
    hildon_time_button_get_time((HildonTimeButton *)timeEditButton, &hours, &minutes);
    month += 1;
    weightResult = (char *)hildon_button_get_value(HILDON_BUTTON(weightButton));
    sscanf(weightResult, "%f", &weight);
    iJD = calculateJulianDate(day, month, year);
    fJD = (double)iJD + ((double)hours)/24.0 + ((double)minutes)/1440.0;
    comment = (char *)gtk_entry_get_text((GtkEntry *)commentText);
    for (i = 0; i < strlen(comment); i++)
      commentSum += comment[i] << i;
    for (i = 0; i < strlen(defaultComment); i++)
      defaultCommentSum += defaultComment[i] << i;
    dprintf("Got %d/%d/%d  %02d:%02d %f %f \"%s\" %d\n",
	   day, month, year, hours, minutes, weight, fJD, comment, commentSum);
    if (commentSum == defaultCommentSum) {
      dprintf("Discarding non-comment\n");
      comment = NULL;
    }
    editPtr->time = fJD;
    editPtr->weight = weight;
    if (editPtr->comment != NULL)
      free(editPtr->comment);
    editPtr->comment = comment;
  }
  writeDataFile();
  gtk_widget_destroy(trendStackable);
}

void createEditPage(logEntry *ptr)
{
  double dayFrac;
  int day, month, year, hours, minutes;
  static GtkWidget *weightSelector, *dataEditTable;

  dprintf("in createEditPage\n");
  deleted = FALSE;
  editPtr = ptr;
  dataEditTable = gtk_table_new(1, 5, FALSE);
  dateEditButton = hildon_date_button_new(HILDON_SIZE_AUTO, HILDON_BUTTON_ARRANGEMENT_VERTICAL);
  tJDToDate((double)((int)ptr->time), &year, &month, &day);
  hildon_date_button_set_date((HildonDateButton *)dateEditButton, year, month-1, day);
  gtk_table_attach(GTK_TABLE(dataEditTable), dateEditButton, 0, 1, 0, 1,
		   GTK_EXPAND, GTK_EXPAND, 0, 0);
  
  timeEditButton = hildon_time_button_new(HILDON_SIZE_AUTO, HILDON_BUTTON_ARRANGEMENT_VERTICAL);
  dayFrac = ptr->time - (double)((int)ptr->time);
  hours = (int)(dayFrac*24.0);
  minutes = (int)((dayFrac - ((double)hours)/24.0) * 1440.0 + 0.5);
  hildon_time_button_set_time((HildonTimeButton *)timeEditButton, hours, minutes);
  gtk_table_attach(GTK_TABLE(dataEditTable), timeEditButton, 0, 1, 1, 2,
		   GTK_EXPAND, GTK_EXPAND, 0, 0);
  dataEditStackable = hildon_stackable_window_new();
  g_signal_connect(G_OBJECT(dataEditStackable), "destroy",
		   G_CALLBACK(editData), NULL);
  weightSelector = createTouchSelector();
  weightButton = hildon_picker_button_new(HILDON_SIZE_AUTO, HILDON_BUTTON_ARRANGEMENT_VERTICAL);
  if (weightkg)
    hildon_touch_selector_set_active(HILDON_TOUCH_SELECTOR(weightSelector), 0,
				     (int)((10.0*ptr->weight)+0.5) - 10*minSelectorWeight);
  else
    hildon_touch_selector_set_active(HILDON_TOUCH_SELECTOR(weightSelector), 0,
				     (int)((5.0*ptr->weight)+0.5) - 5*minSelectorWeight);  if (weightkg)
    hildon_button_set_title(HILDON_BUTTON(weightButton), "Weight (kg)");
  else
    hildon_button_set_title(HILDON_BUTTON(weightButton), "Weight (lbs)");
  hildon_picker_button_set_selector(HILDON_PICKER_BUTTON(weightButton),
				    HILDON_TOUCH_SELECTOR(weightSelector));
  gtk_table_attach(GTK_TABLE(dataEditTable), weightButton, 0, 1, 2, 3,
		   GTK_EXPAND, GTK_EXPAND, 0, 0);

  commentText = gtk_entry_new();
  gtk_entry_set_text((GtkEntry *)commentText, ptr->comment);
  gtk_table_attach(GTK_TABLE(dataEditTable), commentText, 0, 1, 3, 4,
		   GTK_EXPAND | GTK_FILL, GTK_FILL, 0, 0);
  dataEditDelete = gtk_button_new_with_label("Delete This Entry");
  g_signal_connect(G_OBJECT(dataEditDelete), "clicked",
		   G_CALLBACK(dataEditDeleteCallback), NULL);
  gtk_table_attach(GTK_TABLE(dataEditTable), dataEditDelete, 0, 1, 4, 5,
		   GTK_EXPAND, GTK_EXPAND, 0, 0);
  gtk_container_add(GTK_CONTAINER(dataEditStackable), dataEditTable);
  gtk_widget_show_all(dataEditStackable);
}

static gboolean buttonReleaseEvent(GtkWidget *widget, GdkEventButton *event)
{
  dprintf("In buttonReleaseEvent\n");
  if (logDisplayed) {
    int x, y;
    int found = FALSE;
    logCell *ptr;

    x = event->x; y = event->y;
    dprintf("The log is displayed (%d, %d)\n", x, y);
    ptr = logCellRoot;
    while ((ptr != NULL) && (!found)) {
      if ((ptr->box[0].x <= x) && (ptr->box[0].y <= y) && (ptr->box[2].x >= x) && (ptr->box[2].y >= y))
	found = TRUE;
      else
	ptr = ptr->next;
    }
    if (found) {
      int day, month, year;

      tJDToDate((double)((int)ptr->ptr->time), &year, &month, &day);
      dprintf("Found it! (%d/%d/%d) (%f)\n", month, day, year, ptr->ptr->weight);
      createEditPage(ptr->ptr);
    }
  }
  return(TRUE);
}

int main(int argc, char **argv)
{
  osso_context_t *oSSOContext;

  homeDir = getenv("HOME");
  if (homeDir == NULL) {
    homeDir = malloc(strlen("/home/user")+1);
    if (homeDir == NULL) {
      perror("malloc of backup homeDir");
      exit(ERROR_EXIT);
    }
    sprintf(homeDir, "/home/user");
  }
  userDir = malloc(strlen(homeDir)+strlen("/.maeFat")+1);
  if (userDir == NULL) {
    perror("malloc of userDir");
    exit(ERROR_EXIT);
  }
  sprintf(userDir, "%s/.maeFat", homeDir);
  mkdir(userDir, 0777);
  mkdir(backupDir, 0777);

  fileName = malloc(strlen(userDir)+strlen("/data")+1);
  if (fileName == NULL) {
    perror("malloc of fileName");
    exit(ERROR_EXIT);
  }
  sprintf(fileName, "%s/data", userDir);
  dprintf("The data file path is \"%s\"\n", fileName);
  readData(fileName);

  settingsFileName = malloc(strlen(userDir)+strlen("/settings")+1);
  if (settingsFileName == NULL) {
    perror("malloc of settingsFileName");
    exit(ERROR_EXIT);
  }
  sprintf(settingsFileName, "%s/settings", userDir);
  dprintf("The settings file path is \"%s\"\n", settingsFileName);
  readSettings(settingsFileName);

  oSSOContext = osso_initialize("com.nokia.maefat", maeFatVersion, TRUE, NULL);
  if (!oSSOContext) {
    fprintf(stderr, "oss_initialize call failed\n");
    exit(-1);
  }

  hildon_gtk_init(&argc, &argv);
  /* Initialize main window */
  window = hildon_stackable_window_new();
  gtk_widget_set_size_request (GTK_WIDGET (window), 640, 480);
  gtk_window_set_title (GTK_WINDOW (window), "maeFat");
  g_signal_connect (G_OBJECT (window), "delete_event",
		    G_CALLBACK (gtk_main_quit), NULL);
  mainBox = gtk_vbox_new(FALSE, 0);
  gtk_container_add(GTK_CONTAINER(window), mainBox);
  g_object_ref(mainBox); /* This keeps mainBox from being destroyed when not displayed */
  gtk_widget_show(mainBox);

  /* Configure Main Menu Buttons */
  makeHildonButtons();

  drawingArea = gtk_drawing_area_new();
  gtk_widget_set_size_request (GTK_WIDGET(drawingArea), 640, 480);
  gtk_box_pack_end(GTK_BOX(mainBox), drawingArea, TRUE, TRUE, 0);
  g_signal_connect(G_OBJECT(drawingArea), "expose_event",
		   G_CALLBACK(exposeEvent), NULL);
  g_signal_connect(G_OBJECT(drawingArea), "configure_event",
		   G_CALLBACK(configureEvent), NULL);

  g_signal_connect(G_OBJECT(drawingArea), "button_release_event",
		   G_CALLBACK(buttonReleaseEvent), NULL);
  g_signal_connect(G_OBJECT(drawingArea), "button_press_event",
		   G_CALLBACK(buttonPressEvent), NULL);
  gtk_widget_show(window);
  makeGraphicContexts(window);
  makeFonts(window);
  gtk_widget_set_events(drawingArea,
			GDK_EXPOSURE_MASK       | GDK_BUTTON_PRESS_MASK  |
			GDK_BUTTON_RELEASE_MASK | GDK_POINTER_MOTION_MASK);
  gtk_widget_show(drawingArea);
  gtk_main();
  osso_deinitialize(oSSOContext);
  exit(OK_EXIT);
}
