
#include <iostream>
#include <string>

#include <CMulticalendar.h>
#include <CCalendar.h>
#include <QStringList>

#include <CalendarErrors.h>

#include <main.cpp>
#include <functions.h>

using namespace std;


void usage_import(){
    cout << "Usage:" << endl;
    cout << "\t--operation import --calendar CAL_NAME --file FILE_NAME [--clear] [--verify]"<< endl;
    cout << endl;
    cout << "\t--clear\t\tClears the calendar before performing import" << endl;
    cout << "\t--verify\tVerifies the input calendar before performing import" << endl;
}

void usage_export(){
    cout << "Usage:" << endl;
    cout << "\t--operation export --calendar CAL_NAME [--file FILE_NAME]" << endl;
    cout << "\tDefaults to CAL_NAME.ics if no FILE_NAME given" << endl;
}

void print_cals(CMulticalendar &mcal_instance){
    /*
     * Prints the list of calendars to std out
     */
    vector <CCalendar *> cals = mcal_instance.getListCalFromMc();

    cout << "Available calendars(" << cals.size() << "):" << endl;

    for (unsigned int i=0; i<cals.size(); i++){
        CCalendar *calendar = cals.at(i);
        unsigned int cal_id = calendar->getCalendarId();

        cout << calendar->getCalendarName() << " (id=" << cal_id << ")" << endl;
    }
    mcal_instance.releaseListCalendars(cals);

}

unsigned int parse_args(QStringList &args){
    //returns an int with the operation id to perform

    //get first instance of "calendar"
    //QString opstring = ARG_OPERATION_LONG;

    int a = args.indexOf(ARG_OPERATION_LONG);

    //no operation specified
    if(a == -1){
        return OPERATION_UNKNOWN;
    }
    cout << "index of operation is:" << a << endl;
    a++; //get next argument

    cout << "args.size is:" << args.size() << endl;
    if(args.size() < (a + 1)){
        return OPERATION_UNKNOWN;
    }

    cout << "operation is: " << qPrintable(args.at(a)) << endl;
//    cout << "operation is: " << args.at(a).toAscii().constData() << endl;

    return arg_to_code(args.at(a));
}

unsigned int arg_to_code(QString arg){

    if(arg == ARG_OP_VAL_EXPORT){
        return OPERATION_EXPORT;
    }

    if(arg == ARG_OP_VAL_IMPORT){
        return OPERATION_IMPORT;
    }

    if(arg == ARG_OP_VAL_LIST){
        return OPERATION_LIST;
    }

    return OPERATION_UNKNOWN;

}

bool parse_calendar(QStringList &args, QString &calendar){

    int a = args.indexOf(ARG_CAL_NAME);

    if(a == -1){
        return false;
    }

    a++;

    if(args.size() < (a + 1)){
        return false;
    }

    calendar = args.at(a);
    return true;
}



bool parse_cmd_arg(const QStringList args, const QString arg, int& pos){
    /*
     * Gets the index of the given command line argument provided
     *
     * Returns:
     *    - true if the argument exists
     *    - false otherwise
     */

    pos = args.indexOf(arg);

    if(pos == -1){
        return false;
    } else {
        return true;
    }
}

bool parse_cmd_arg(const QStringList args, const QString arg){

    int x = -2;

    if(parse_cmd_arg(args, arg, x)){
        return true;
    } else {
        return false;
    }

}



bool parse_cmd_arg(const QStringList args, const QString arg, QString& value){
    /*
     * Gets the value for an argument given on the command line
     *
     * Returns:
     *     - true if the argument exists and a following value was supplied
     *     - false otherwise
     */

    int a = -1;

    if(parse_cmd_arg(args,arg,a)){

        if(args.size() > ++a){
            value = args.at(a);
            return true;
        }
    }

    return false;
}

bool parse_cmd_arg_calendar(const QStringList args, QString & calendar){
    if(parse_cmd_arg(args, ARG_CAL_NAME, calendar)){
        return true;
   } else {
       cout << "Failed to parse the calendar name" << endl;
       return false;
   }
}

bool parse_cmd_arg_filename(const QStringList args, QString & filename){
    if(parse_cmd_arg(args, ARG_FILE_NAME, filename)){
        return true;
   } else {
       cout << "Failed to parse the file name" << endl;
       return false;
   }
}

bool get_calendar_info(string calendar, int & iCalendarId){
    /*
     * Verifies the calendar name and gets the API id
     *
     * Returns:
     *    - true if valid calendar
     *    - false if not a valid calendar
     *
     */

    if(!mcalendar->checkCalendarNameExists(calendar)){
        return false;
    }
    int errcode = 0;
    CCalendar *ccalendar = mcalendar->getCalendarByName(calendar, errcode);
    //cout << "get_calendar_info: errcode:" << errcode << endl;
    int cal_id = ccalendar->getCalendarId();
    //cout << "get_calendar_info: calendar id is: " << cal_id << endl;
    iCalendarId = cal_id;
    return true;

}

bool get_calendar_from_cmd(const QStringList args, QString & calendar, int & iCalendarId){
    /*
     * Parses the command line for a calendar. Verifies the name and gets the id from the API
     *
     */
    if(!parse_cmd_arg_calendar(args, calendar)){
        return false;
    }

    return get_calendar_info(calendar.toAscii().constData(),iCalendarId);

}


bool do_operation(const QStringList args, const QString operation){
    /*
     * Does the operation given
     *
     */
    bool result = false;

    switch (arg_to_code(operation)){

    case OPERATION_LIST:
        result = do_operation_list();
        break;

    case OPERATION_IMPORT:
        result = do_operation_import(args);
        break;

    case OPERATION_EXPORT:
        result = do_operation_export(args);
        break;

    case OPERATION_UNKNOWN:
        cout << "Either the operation is unknown or not specified!" << endl;
        break;
    default:
        cout << "Could not determine operation" << endl;
    }

    return result;

}

bool do_operation_list(){
    CMulticalendar *mcalendar = CMulticalendar::MCInstance();
    print_cals(*mcalendar);
    return true;
}

bool do_operation_import(const QStringList args){
    QString calendar, filename;
    int calendar_id;

    if(!get_calendar_from_cmd(args, calendar, calendar_id)){
        cout << "That is not a valid calendar name" << endl;
        usage_import();
        return false;
    } else {
        cout << "Will import calendar with id:" << calendar_id << endl;
    }

    if(!parse_cmd_arg_filename(args, filename)){
        cout << "file was not specified:" << filename.toAscii().constData() << endl;
        usage_import();
        return false;
    }

    string eventId;
    unsigned int iCountTask = 0, iCountEvent = 0, iCountJournal = 0;
    int iCountDupeTask = 0, iCountDupeEvent = 0, iCountDupeJournal = 0;
    int errcode = 0;

    //mcalendar->setAutocommitOff();//does import override commit settings?
    //mcalendar->clearCalendar(calendar_id, errcode);
    bool result = false;

    if(parse_cmd_arg(args,ARG_VERIFY)){
        cout << "Verifying input file..." << endl;

        result = mcalendar->getICSFileInfo(
                filename.toStdString(),
                iCountEvent,
                iCountTask,
                iCountJournal,
                errcode);


        if(!result) {
            cout << "Failed to parse the input calendar";
            return false;
        } else {
            cout << "Input file verified." << endl;
            cout << "The following items will be imported:" << endl;
            cout << "Number of tasks-->" << iCountTask << endl;
            cout << "Number of events->" << iCountEvent << endl;
            cout << "Number of notes-->:" << iCountJournal << endl;
        }
    }


    /*
     * Test to see if a clear operation was requested
     *
     * It makes sense to clear the calendar if doing a whole
     * calendar import to try to catch "moved events". While iCal
     * has a concept of 'id', this may not always be implemented
     * in a consistent way by a calendar exporter/provider. This
     * clear option provides a work-around. Also, it makes sense
     * to NOT clear by default if the user likes to import just a single
     * new event (or subset of events) into the calenar.
     */
    if(parse_cmd_arg(args,ARG_CLEAR)){
        /*
         * First, clear the calendar contents. While the import API automatically ignores
         * duplicates, if the external calendar has any appointments that have "moved"
         * then there is no logic to figure that out. Clearing the calendar first
         * is a way to handle potentially "moved" appointments.
         */
        mcalendar->clearCalendar(calendar_id, errcode);

        if(errcode == CALENDAR_OPERATION_SUCCESSFUL){
            cout << "Clearing the calendar was successful." << endl;
        } else {
            cout << "The calendar could not be cleared." << endl;
            cout << "Please contact the developer with this error code: " << errcode << endl;
        }

    }

    cout << "Importing calendar..." << endl;
    result = mcalendar->importIcsFileData(
            filename.toStdString(),
            calendar_id,
            eventId,
            iCountDupeTask,
            iCountDupeEvent,
            iCountDupeJournal,
            errcode);

    if(result){
        switch (errcode){
        case CALENDAR_ENTRY_DUPLICATED:
            cout << "Duplicates were found and updated." << endl;
            cout << "Duplicate tasks-->" << iCountDupeTask << endl;
            cout << "Duplicate events->" << iCountDupeEvent << endl;
            cout << "Duplicate notes-->:" << iCountDupeJournal << endl;

            break;

        case CALENDAR_OPERATION_SUCCESSFUL:
            cout << "Import successful." << endl;

            break;

        default:
            cout << "WARN: api result was true, but the errorcode was not SUCCESS. It was " << errcode << endl;
        }

        return true;
    } else {
        cout << "An error occurred during import" << endl;
        cout << "errcode:" << errcode;

        return false;
    }

}

bool do_operation_export(const QStringList args){

    QString calendar, filename;
    int calendar_id;

    if(!get_calendar_from_cmd(args, calendar, calendar_id)){
        cout << "That is not a valid calendar name" << endl;
        usage_export();
        return false;
    } else {
        cout << "will export calendar with id:" << calendar_id << endl;
    }

    if(!parse_cmd_arg_filename(args, filename)){
        cout << "file was not specified:" << filename.toAscii().constData() << endl;
        filename = calendar.append(".ics");
    }

    int errcode = 0;
    int result = 99;
    cout << "calendar will be exported to:" << filename.toAscii().constData() << endl;
    result = mcalendar->exportCalendarEntries(calendar_id, filename.toAscii().constData(),errcode);

    cout << "result:" << result << endl;
    cout << "errcode:" << errcode << endl;

    if(errcode == CALENDAR_OPERATION_SUCCESSFUL){
        return true;
    } else {
        switch (errcode){
        case CALENDAR_FILE_ERROR:
            cout << "Failed to save the file." << endl;
            break;
        }

        return false;
    }
    \
}
