import _midgard as midgard, os

# Try to load one gconf implementation, preferring python-gconf over gnome-python
for gconf_module in ('gconf', 'gnome.gconf'):
    try:
        gconf = __import__(gconf_module)
        break
    except:
        pass

# Keep the connection and configuration as global variables
configuration = midgard.config()
configuration.dbtype = 'SQLite'
configuration.database = 'maecalories'
#configuration.blobdir =  os.path.expanduser('~/.midgard2/blobs/' + configuration.database)
connection = midgard.connection()

class wrapper:
    """Wrap our storage logic to handier package, import this module and use storage.instance"""
    def __init__(self):
        self.gc = gconf.Client()
        self.gc.add_dir('/apps/maecalories', gconf.CLIENT_PRELOAD_NONE)

        # The classes we use, for these we create/update storage
        self.used_classes = [
            'midgard_person',
            'org_maemo_garage_maecalories_pointslog',
            'org_maemo_garage_maecalories_weightlog'
        ]
        self.person = None

        # Need to check this before opening the connection since that will create the file...
        self.dbpath = os.path.expanduser('~/.midgard2/data/' + configuration.database + '.db')
        self.db_exists = os.path.exists(self.dbpath)
        # Connection state tracking
        self.connected = False

    def connect(self):
        """Open connection to database or raise exception on failure"""
        self.connected = connection.open_config(configuration)
        if not self.connected:
            raise Exception('Could not open database connection, reason: %s' % midgard._connection.get_error_string())
        # TODO: Connect to pointslog signals to update caches on changes

    def return_to_pointer(self, function_pointer, return_pointer, function_args=None):
        if function_args is None:
            return_pointer = function_pointer()
            return
        return_pointer = function_pointer(function_args)

    def initialize(self, window=None):
        if not self.connected:
            self.connect()
        self.initialize_db(window)
        self.initialize_person(window)
        self.update_caches()

    def initialize_person(self, window):
        """Loads or creates the person object for user"""
        qb = midgard.query_builder('midgard_person')
        qb.add_constraint('id', '<>', 1) # Skip midgard administrator
        count = qb.count()
        if count == 0:
            # Create new person
            self.person = midgard.mgdschema.midgard_person()
            self.person.firstname = 'User'
            self.person.lastname = 'Maemo'
            print "Calling person.create()"
            self.person.create()
            if self.person.id == 0:
                # Bug http://trac.midgard-project.org/ticket/1409
                if window:
                    import hildon, gtk
                    note = hildon.Note('information', window, "Hit Midgard bug #1409, please restart application")
                    gtk.Dialog.run(note)
                raise Exception("Hit Midgard bug #1409, restart application")
        elif count > 1:
            # Display selector and save choice
            raise Exception('Multiple user handling not implemented')
        else:
            results = qb.execute()
            self.person = results[0]
        print "Got user #%d: %s, %s" % (self.person.id, self.person.lastname, self.person.firstname)

    def initialize_db(self, window):
        # Much quicker than checking each table separately
        if self.db_exists:
            return True

        # If we have window create a modal progress-bar
        if window:
            import hildon, gtk
            bar_step = 1.0 / (len(self.used_classes) + 2)
            bar = gtk.ProgressBar()
            bar.set_fraction(bar_step)
            note = hildon.Note('information', window, 'Initializing database', progressbar=bar)
            # Muck a non-blocking dialog display
            note.set_modal(True)
            note.connect('response', lambda d, r: r)
            note.show()
            while gtk.events_pending():
                gtk.main_iteration()

        # Create call create storage functions via threads so we can keep UI responsive while they work
        import threading
        create_ok = True
        cthread = threading.Thread(name='base storage', target=self.return_to_pointer, args=(midgard.storage.create_base_storage, create_ok))
        if cthread == None:
            raise Exception("could not create thread for create_base_storage()")
        cthread.start()
        while cthread.isAlive():
            if window:
                while gtk.events_pending():
                    gtk.main_iteration()
            print "waiting for %s" % cthread.getName()
            cthread.join(0.1)
        print "waiting (blocking) for %s" % cthread.getName()
        cthread.join()
        print "done"
        if not create_ok:
            raise Exception('Could not create base storage, reason: %s' % midgard._connection.get_error_string())
        if window:
            bar.set_fraction(bar.get_fraction() + bar_step)
            print "set progress bar fraction to %f" % bar.get_fraction()
            while gtk.events_pending():
                gtk.main_iteration()
        for classname in self.used_classes:
            create_ok = True
            cthread = threading.Thread(name=classname + ' storage', target=self.return_to_pointer, args=(midgard.storage.create_class_storage, create_ok), kwargs={ 'function_args': classname, } )
            if cthread == None:
                raise Exception("could not create thread for create_class_storage(%s)" % classname)
            cthread.start()
            while cthread.isAlive():
                if window:
                    while gtk.events_pending():
                        gtk.main_iteration()
                print "waiting for %s" % cthread.getName()
                cthread.join(0.1)
            print "waiting (blocking) for %s" % cthread.getName()
            cthread.join()
            print "done"
            if not create_ok:
                raise Exception('Could not create storage for class' + classname + ', reason: %s' % midgard._connection.get_error_string())
            if window:
                bar.set_fraction(bar.get_fraction() + bar_step)
                print "set progress bar fraction to %f" % bar.get_fraction()
                while gtk.events_pending():
                    gtk.main_iteration()

        if window:
            note.destroy()
            bar.destroy()
            while gtk.events_pending():
                gtk.main_iteration()

        return True

    def get_week(self, now=None):
        """Resolve start and end datetimes for current week"""
        import datetime
        if now == None:
            now = utcstamp()
        sow = self.gc.get_int('/apps/maecalories/settings/week_starts')
        if sow == None:
            self.gc.set_int('/apps/maecalories/settings/week_starts', 0)
            sow = 0
        if sow == now.weekday():
            # Lucky us, week started today
            weekstart = now.replace(hour=0, minute=0, second=1)
        else:
            # Need to do a bit of juggling
            d_one_d = datetime.timedelta(days=1)
            weekstart = now.replace(hour=0, minute=0, second=1)
            while weekstart.weekday() != sow:
                weekstart -= d_one_d
        
        d_one_w = datetime.timedelta(days=6)
        weekend = weekstart.replace(hour=23, minute=59, second=59)
        weekend += d_one_w
        return (weekstart, weekend)

    def get_day(self, now=None):
        """Resolve start and end timestamps for today"""
        import datetime
        if now == None:
            now = utcstamp()
        daystart = now.replace(hour=0, minute=0, second=1)
        dayend = now.replace(hour=23, minute=59, second=59)
        return (daystart, dayend)

    def update_caches(self):
        # Init datetimes we need
        week = self.get_week()
        day = self.get_day()

        # Init related config variables
        ppd = self.gc.get_float('/apps/maecalories/settings/points_per_day')
        if ppd == None:
            self.gc.set_float('/apps/maecalories/settings/points_per_day', 0.0)
            ppd = 0.0
        ppw = ppd * 7
        nppw = self.gc.get_float('/apps/maecalories/settings/npoints_per_week')
        if nppw == None:
            self.gc.set_float('/apps/maecalories/settings/npoints_per_week', 0.0)
            nppw = 0.0

        # Our accumulators
        ppd_c = 0.0
        ppw_c = 0.0
        nppw_c = 0.0

        # Get weeks points log entries
        qb_week = midgard.query_builder('org_maemo_garage_maecalories_pointslog')
        qb_week.add_constraint('time', '>=', dt_to_constraint(week[0]))
        qb_week.add_constraint('time', '<=', dt_to_constraint(week[1]))
        qb_week.add_order('time', 'ASC')
        items = qb_week.execute()
        for item in items:
            # First w need to check that weekly negatives don't max out
            if item.points < 0.0:
                temp_nppw_c = nppw_c + item.points
                #points_tmp = float(item.points)
                if temp_nppw_c < nppw:
                    item.points = (nppw_c - nppw) * -1
                    #print "temp_nppw_c got to %.1f, rewrote item.points from %.1f to %.1f" % (temp_nppw_c, points_tmp, item.points)
                nppw_c += item.points
            # Then add the adjusted points to weekly saldo
            ppw_c += item.points
            # and if item is on today add to daily saldo
            if (    item.time >= day[0]
                and item.time <= day[1]):
                # Item is for today
                ppd_c += item.points

        print "storage.instance.update_caches() reports: ppd_c=%.1f, ppw_c=%.1f, nppw_c=%.1f" % (ppd_c, ppw_c, nppw_c)
        day_points_left = ppd - ppd_c
        week_points_left = ppw - ppw_c
        self.gc.set_string('/apps/maecalories/cache/lastupdated', utcstamp().strftime('%Y-%m-%d %H:%M:%S'))
        self.gc.set_float('/apps/maecalories/cache/week_points_left', week_points_left)
        self.gc.set_float('/apps/maecalories/cache/day_points_left', day_points_left)
        self.gc.set_float('/apps/maecalories/cache/week_points_accumulated', ppw_c)
        self.gc.set_float('/apps/maecalories/cache/day_points_accumulated', ppd_c)

        qb_wt = midgard.query_builder('org_maemo_garage_maecalories_weightlog')
        # Just in case some brainiac manages to post-date a weight log entry
        qb_wt.add_constraint('time', '<=', dt_to_constraint(day[1]))
        qb_wt.add_order('time', 'DESC')
        qb_wt.set_limit(1)
        if qb_wt.count() == 0:
            self.gc.set_string('/apps/maecalories/cache/last_weight_datetime', 'NEVER')
        else:
            self.gc.set_string('/apps/maecalories/cache/last_weight_datetime', qb_wt.execute()[0].time.strftime('%Y-%m-%d %H:%M:%S'))

    def gc_date_cmp(self, key, now=None):
        """Simple helper to check if now is on the same date as given Gconf key. Used mainly to trigger events and notifiers"""
        gc_value = self.gc.get_string(key)
        if gc_value == None:
            return False
        if now == None:
            now = utcstamp()
        now_formatted = now.strftime('%Y-%m-%d %H:%M:%S')
        # Compare just the date part
        if now_formatted[0:10] != gc_value[0:10]:
            return False
        return True

def utcstamp():
    """Helper fuction to return current UTC timestamp as datime"""
    import datetime
    return datetime.datetime.utcnow()

def dt_to_constraint(dt):
    return dt.strftime('%Y-%m-%d %H:%M:%S')

# Use this instantiated wrapper object to access the base storage
instance = wrapper()
