/*
 * Copyright 2010 Felipe Crochik <foss@crochik.com>
 * All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "mainwindow.h"

#ifdef Q_WS_MAEMO_5
    #include <QtGui/QX11Info>
    #include <X11/Xlib.h>
    #include <X11/Xatom.h>
    #include <QMaemo5InformationBox>

    #include "pickselectoraction.h"
#endif

#include <QDebug>
#include <QApplication>
#include <QDesktopWidget>

#include "fullscreenexitbutton.h"

#include "gotodlg.h"
#include "bookmark.h"
#include "bookmarksdlg.h"
#include "bookmarkdlg.h"

#define SETTINGS_VERSION 1

MainWindow::MainWindow(MainWindow *pParent) :
    QMainWindow(),
    m_webPage(this)
{
    m_pBookmarks = NULL;
    m_pGraphicsView = NULL;
    m_pGWebView = NULL;
    m_pWebView = NULL;

#ifdef Q_WS_MAEMO_5
    m_pOrientationSelector = NULL;
    m_pUserAgentSelector = NULL;
#endif

    m_pParent = pParent;

    m_isUsingGraphicsView = false;

    setWindowTitle("Macuco");


#ifdef Q_WS_MAEMO_5
    FullScreenExitButton *pButt = new FullScreenExitButton(this);
    connect(pButt, SIGNAL(clicked()), this, SLOT(showNormal()));

    grabZoomKeys(true);

    setAttribute(Qt::WA_Maemo5StackedWindow);
#endif

#ifdef Q_WS_MAEMO_5
    QString storagePath = "/home/user/.macuco";
#else
    QString storagePath = QDesktopServices::storageLocation(QDesktopServices::CacheLocation) +
            QDir::separator() + "macuco";
            //QApplication::applicationDirPath() + "/cache";
#endif

    qDebug() << "cache location: " << storagePath;
    QWebSettings::enablePersistentStorage(storagePath);

    // disk cache
    // QString storagePath = QWebSettings::offlineWebApplicationCachePath();
    qDebug() << "disk cache: " << storagePath;
    QNetworkDiskCache *pCache = new QNetworkDiskCache(this);
    pCache->setCacheDirectory(storagePath);
    m_webPage.networkAccessManager()->setCache(pCache);

    // m_webPage.setLinkDelegationPolicy(QWebPage::DelegateAllLinks);
    // m_webPage.settings()->setAttribute(QWebSettings::PluginsEnabled, true);
    // m_webPage.setForwardUnsupportedContent(true);
    m_webPage.settings()->setAttribute(QWebSettings::DeveloperExtrasEnabled, true);

    connect(m_webPage.networkAccessManager(), SIGNAL(finished(QNetworkReply*)),
        this, SLOT(finishedRequest(QNetworkReply*)));

    // connect(&m_webPage, SIGNAL(linkClicked(QUrl)), this, SLOT(linkClicked(QUrl)));
    connect(&m_webPage, SIGNAL(unsupportedContent(QNetworkReply*)), this, SLOT(unsupportedContent(QNetworkReply*)));
    connect(&m_webPage, SIGNAL(downloadRequested(QNetworkRequest)), this, SLOT(downloadRequested(QNetworkRequest)));


    if ( m_pParent ) {
        // copy settings from mainwindow
        m_zoomFactor = m_pParent->m_zoomFactor;
        m_pBookmarks = m_pParent->m_pBookmarks;
        setFullScreen(m_pParent->isFullScreen());
        setUserAgent( m_pParent->m_webPage.isIphoneUserAgent() );

#ifdef Q_WS_MAEMO_5
        qDebug() << "New Window: " << m_pParent->isLandscape() << " - " << m_pParent->isAutoRotate();
        setAutoRotate(false);
        setLandscape(m_pParent->isLandscape());
#endif

    } else {
	m_zoomFactor = 1;
        m_pBookmarks = new QList<Bookmark>();

        // load settings
        loadSettings();
    }

    if ( !m_pParent ) {
        setWidget();
        // visitUrl("http://www.crochik.com/macuco");
        on_actionHome_triggered();

    } else {

        m_isUsingGraphicsView = m_pParent->m_isUsingGraphicsView;
        setWidget();
    }

#ifdef Q_WS_X11
#ifndef Q_WS_MAEMO_5
    // meego?
    setGeometry(QRect(QPoint(0,0),QApplication::desktop()->size()));
#endif
#endif

    createMenu();
    updateTitle();
}

MainWindow::~MainWindow() {
    qDebug() << "~MainWindow";

    if ( m_pParent ) {
        qDebug() << ">> not top most";
        m_pBookmarks = NULL;
        return;
    }

    qDebug() << ">> top most window";
    if ( m_pBookmarks ) {delete m_pBookmarks; m_pBookmarks = NULL;}

    // delete children windows
    while ( !m_children.isEmpty() ) delete m_children.takeLast();
}

void MainWindow::createMenu() {
    QMenu *pMenu = menuBar()->addMenu("&File");
    pMenu->addAction("Home", this, SLOT(on_actionHome_triggered()));
    pMenu->addAction("Go To...", this, SLOT(on_actionGo_To_triggered()));

#ifndef Q_WS_MAEMO_5
    pMenu->addSeparator();
    pMenu->addAction("&Exit", this, SLOT(close()));
#endif

    pMenu = menuBar()->addMenu("&Bookmarks");
    pMenu->addAction("Bookmarks", this, SLOT(on_actionBookmarks_triggered()));
    pMenu->addAction("Add Bookmark", this, SLOT(on_actionAdd_Bookmark_triggered()));

#ifdef Q_WS_MAEMO_5
    menuBar()->addAction("Fullscreen", this, SLOT(fullscreen_triggered()));

    m_pOrientationSelector = new PickSelectorAction(this);
    m_pOrientationSelector->setText("Orientation");
    m_pOrientationSelector->items() << "Auto" << "Landscape" << "Portrait";
    connect(m_pOrientationSelector, SIGNAL(selectionChanged(int)), this, SLOT(orientation_selectionChanged(int)));
    menuBar()->addAction(m_pOrientationSelector);

    m_pUserAgentSelector = new PickSelectorAction(this);
    m_pUserAgentSelector->setText("User Agent");
    m_pUserAgentSelector->items() << "iPhone" << "Default";
    connect(m_pUserAgentSelector, SIGNAL(selectionChanged(int)), this, SLOT(userAgent_selectionChanged(int)));
    menuBar()->addAction(m_pUserAgentSelector);
#else
    pMenu = menuBar()->addMenu("&View");
    QAction *pFullscreen = pMenu->addAction("Fullscreen");
    pFullscreen->setCheckable(true);
    connect(pFullscreen, SIGNAL(toggled(bool)), this, SLOT(setFullScreen(bool)));

    pMenu->addSeparator();

    QActionGroup *pActionGroup = new QActionGroup(pMenu);
    QAction *pAct1 = pActionGroup->addAction("iPhone User Agent");
    QAction *pAct2 = pActionGroup->addAction("Default User Agent");
    connect(pActionGroup, SIGNAL(selected(QAction*)), SLOT(onUserAgentChanged(QAction*)));
    pMenu->addActions(pActionGroup->actions());
    pAct1->setCheckable(true);
    pAct2->setCheckable(true);
    if ( m_webPage.isIphoneUserAgent()) pAct1->setChecked(true); else pAct2->setChecked(true);

    #ifdef QT_DEBUG
        pMenu->addSeparator();
        pMenu->addAction("Switch View", this, SLOT(switchView()));
    #endif

#endif

#ifdef QT_DEBUG
    pMenu = menuBar()->addMenu("&About");
    pMenu->addAction("About Qt..", this, SLOT(aboutQt()));
#endif
}

void MainWindow::switchView() {
    m_isUsingGraphicsView = !m_isUsingGraphicsView;
    setWidget();
}

void MainWindow::setWidget() {
    if ( m_isUsingGraphicsView ) {
        qDebug() << "Switch to Graphics View";
        if ( m_pWebView ) {
            qDebug() << ">> delete WebView";
            delete m_pWebView; m_pWebView = NULL;
        }

        if ( !m_pGWebView ) {
            // using graphics view
            m_pGraphicsView = new GraphicsView(this);
            m_pGWebView = new QGraphicsWebView();
            m_pGraphicsView->viewportItem()->setWidget(m_pGWebView);
            m_pGWebView->setInputMethodHints(Qt::ImhNoAutoUppercase | Qt::ImhNoPredictiveText);

            connect(m_pGWebView, SIGNAL(loadStarted()), this, SLOT(showLoading()));
            connect(m_pGWebView, SIGNAL(urlChanged(QUrl)), this, SLOT(changeUrl(QUrl)));
            connect(m_pGWebView, SIGNAL(loadFinished(bool)), this, SLOT(hideLoading(bool)));
        }

        m_pGWebView->setPage(&m_webPage);
        setCentralWidget(m_pGraphicsView);

        if ( !m_pParent ) reload();
        return;
    }

    qDebug() << "Switch to Web View";
    if ( m_pGWebView ) {
        qDebug() << ">> delete Graphics View";
        delete m_pGWebView; m_pGWebView = NULL;
        delete m_pGraphicsView; m_pGraphicsView = NULL;
    }

    if ( !m_pWebView ) {
        // using web view
        m_pWebView = new WebView(this);

        connect(m_pWebView, SIGNAL(loadStarted()), this, SLOT(showLoading()));
        connect(m_pWebView, SIGNAL(urlChanged(QUrl)), this, SLOT(changeUrl(QUrl)));
        connect(m_pWebView, SIGNAL(loadFinished(bool)), this, SLOT(hideLoading(bool)));
    }

    // set web page
    m_webPage.mainFrame()->setScrollBarPolicy(Qt::Vertical, Qt::ScrollBarAlwaysOff);
    m_webPage.mainFrame()->setScrollBarPolicy(Qt::Horizontal, Qt::ScrollBarAlwaysOff);

    m_pWebView->setPage(&m_webPage);
    setCentralWidget(m_pWebView);
}

void MainWindow::aboutQt() {
    QMessageBox::aboutQt(this, "Qt");
}

void MainWindow::onUserAgentChanged(QAction *pAction) {
    qDebug() << pAction->text();

    m_webPage.setIPhoneUserAgent(pAction->text().startsWith("iPhone"));
    updateTitle();
}

void MainWindow::userAgent_selectionChanged(int index) {
    switch ( index ) {
        case 0: // iphone
            m_webPage.setIPhoneUserAgent();
            break;

        case 1: // default
            m_webPage.setIPhoneUserAgent(false);
            break;
    }
}

void MainWindow::orientation_selectionChanged(int index) {
#ifdef Q_WS_MAEMO_5
    switch ( index ) {
        case 0: // auto
            setAutoRotate(true);
            break;

        case 1: // landscape
            setAutoRotate(false);
            setLandscape(true);
            break;

        case 2: // portrait
            setAutoRotate(false);
            setLandscape(false);
            break;
    }
#endif
}

MainWindow* MainWindow::topMost() {
    if ( m_pParent ) return m_pParent->topMost();
    return this;
}

MainWindow* MainWindow::createWindow() {
    if ( m_pParent ) return m_pParent->createWindow();

    // top most
    MainWindow *pChild =new MainWindow(this);
    m_children.append(pChild);

    return pChild;
}

void MainWindow::loadSettings() {
    qDebug() << "load settings";

    Bookmark::registerMeataType();

    QSettings settings("Macuco", "Macuco");
    int version = settings.value("version", 0).toInt();
    if ( version<SETTINGS_VERSION ) {
        qDebug() << "previous version of settings file";
        //...
    }

    (*m_pBookmarks) = settings.value("bookmarks").value< QList<Bookmark> >();
    setFullScreen(settings.value("fullScreen", false).toBool());
    setUserAgent(settings.value("iphoneUserAgent", true).toBool());

#ifdef Q_WS_MAEMO_5
    int orientation = settings.value("orientation", 0).toInt();
    orientation_selectionChanged(orientation);
#endif

    qDebug() << "try to get icons";
    // get page icons
    // (work around for not getting the icons the first time - have to investigate)
    for ( int c=0; c<m_pBookmarks->size(); c++ ) {
        QWebSettings::iconForUrl((*m_pBookmarks)[c].url());
    }

    qDebug() << ">> done";
}

void MainWindow::updateTitle() {
#ifdef Q_WS_MAEMO_5

#else
    QString title = m_currentUrl.isEmpty() ? "Macuco" : (m_isUsingGraphicsView?m_pGWebView->title():m_pWebView->title());
    if ( m_webPage.isIphoneUserAgent() ) title += " - iPhone";

    title += QString(" - %1%").arg(QString::number(m_zoomFactor*100, 'f', 0));
    setWindowTitle(title);
#endif

}

void MainWindow::saveSettings() {
    QSettings settings("Macuco", "Macuco");
    settings.setValue("version", SETTINGS_VERSION);
    settings.setValue("fullScreen", isFullScreen());
    settings.setValue("iphoneUserAgent", m_webPage.isIphoneUserAgent());

#ifdef Q_WS_MAEMO_5
    settings.setValue("orientation", m_pOrientationSelector->selectedIndex());
#endif

    QVariant bookmarks;
    bookmarks.setValue(*m_pBookmarks);
    settings.setValue("bookmarks", bookmarks);
}

void MainWindow::showLoading() {
    m_errors.clear();

#ifdef Q_WS_MAEMO_5
    // http://qt.nokia.com/doc/qt-maemo-4.6/maemo5-windowstates.html
    setAttribute(Qt::WA_Maemo5ShowProgressIndicator, true);
#else
    setCursor(QCursor(Qt::WaitCursor));
    /*
    m_pErrorBox->hide();
    hideOverlayOnPage();

    m_pLoadingLabel->show();
    */
#endif
}

void MainWindow::changeUrl(QUrl url) {
    /*
    QIcon icon = m_webPage.settings()->iconForUrl(url);
    qDebug() << icon.isNull();
    */

    m_currentUrl = url;

    /*
#ifndef Q_WS_MAEMO_5
    m_pToolbar->getInputAddress()->setText( url.toString() );
#endif
    */
}

void MainWindow::visitUrl(QString url) {

    if ( url.isNull() || url.isEmpty() ) return;

    if ( m_isUsingGraphicsView )
        m_pGWebView->load(QUrl::fromUserInput(url));
    else
        m_pWebView->load(QUrl::fromUserInput(url));
}

void MainWindow::reload() {
    if ( m_isUsingGraphicsView )
        m_pGWebView->load(m_currentUrl);
    else
        m_pWebView->load(m_currentUrl);
    /*
#ifndef Q_WS_MAEMO_5
    m_pErrorBox->hide();
    hideOverlayOnPage();
#end
    */
}

void MainWindow::adjustSizes() {
    if ( !m_currentUrl.isEmpty() && m_webPage.isIphoneUserAgent() ) zoomFit();
}

void MainWindow::zoomFit() {
    if ( width() > height() ) {
        setZoom(qreal(width())/480.0);
    } else {
        setZoom(qreal(height())/480.0);
    }
}

void MainWindow::resizeEvent(QResizeEvent* event)
{
    adjustSizes();
    QWidget::resizeEvent(event);
}

void MainWindow::closeEvent(QCloseEvent *event)
{
#ifdef Q_WS_MAEMO_5
    grabZoomKeys(false);
#endif

    if ( !m_pParent ) {
        if (!m_children.isEmpty()) {
            if ( QMessageBox::question(this, "Close Application",
                "Closing this window will close all other macuco windows.\n"
                "Would you like to continue?",
                QMessageBox::Yes, QMessageBox::No
                )!=QMessageBox::Yes
            ) {
                event->ignore();
                return;
            }
        }

        // top most
        saveSettings();
        m_webPage.saveCookies();

    } else {
        qDebug() << "remove from parent list and schedule for deletion";
        m_pParent->m_children.removeOne(this);
        deleteLater();
    }

    event->accept();
}

void MainWindow::zoomIn() {
    setZoom(m_zoomFactor*1.15);
}

void MainWindow::zoomOut() {
    setZoom(m_zoomFactor/1.15);
}

void MainWindow::setZoom(qreal zoomFactor) {
    enableWebkitAutoSize();
    m_zoomFactor = zoomFactor;
    if ( m_zoomFactor>.90 && m_zoomFactor<1.1) m_zoomFactor =1;

    updateTitle();

    if ( m_isUsingGraphicsView ) {
        if ( !m_pGWebView ) return;
        m_pGWebView->setZoomFactor(m_zoomFactor);
        m_pGraphicsView->viewportItem()->resizeWebViewToFrame();
        return;
    }

    if ( !m_pWebView ) return;
    m_pWebView->setZoomFactor(m_zoomFactor);
}

void MainWindow::keyPressEvent(QKeyEvent *event)
{
    switch ( event->key() ) {
        case Qt::Key_F7:
            zoomIn();
            break;

        case Qt::Key_F8:
            zoomOut();
            break;

        case Qt::Key_B:
            if ( (event->modifiers()&Qt::ControlModifier) != Qt::ControlModifier  ) break;
            on_actionBookmarks_triggered();
            break;

        case Qt::Key_G:
            if ( (event->modifiers()&Qt::ControlModifier) != Qt::ControlModifier  ) break;
            on_actionGo_To_triggered();
            break;

        case Qt::Key_H:
            if ( (event->modifiers()&Qt::ControlModifier) != Qt::ControlModifier  ) break;
            on_actionHome_triggered();
            break;

        case Qt::Key_F:
            if ( (event->modifiers()&Qt::ControlModifier) != Qt::ControlModifier  ) break;
            setFullScreen(true);
            break;

        case Qt::Key_O:
            if ( (event->modifiers()&Qt::ControlModifier) != Qt::ControlModifier  ) break;
            setZoom(1);
            break;

        case Qt::Key_I:
            if ( (event->modifiers()&Qt::ControlModifier) != Qt::ControlModifier  ) break;
            zoomFit();
            break;
    }
}

void MainWindow::enableWebkitAutoSize()
{
    m_webPage.mainFrame()->evaluateJavaScript("document.body.style.WebkitTextSizeAdjust = 'auto'");
}

void MainWindow::on_actionHome_triggered() {
    m_zoomFactor = 1.5;
    visitUrl("http://www.crochik.com/macuco");
}

void MainWindow::on_actionBookmarks_triggered()
{
    BookmarksDlg dlg((*m_pBookmarks), this);

#ifdef Q_WS_MAEMO_5
    if ( !isLandscape() ) dlg.setAttribute(Qt::WA_Maemo5PortraitOrientation);
    dlg.setAttribute(Qt::WA_Maemo5StackedWindow);
    dlg.setWindowFlags(Qt::Window);
#endif

    if ( dlg.exec()!= QDialog::Accepted || dlg.m_bookmark.isNull()) return;

    visitUrl(dlg.m_bookmark.url());
}

void MainWindow::setFullScreen(bool full/*=true*/) {
#ifndef Q_WS_MAEMO_5
    /*
    if ( m_pUI->actionFullscreen->isChecked()!=full ) {
        // change status of menu item and wait for event to be triggered
        m_pUI->actionFullscreen->setChecked(full);
        return;
    }
    */
#endif

    if (!full) {
        showNormal();
    } else {
        showFullScreen();
    }
}

bool MainWindow::isFullScreen() {
    return windowState() & Qt::WindowFullScreen;
}

void MainWindow::on_actionGo_To_triggered()
{
    GoToDlg dlg(this);
    dlg.setUrl(m_currentUrl.toString());
    if ( dlg.exec()!=QDialog::Accepted ) return;
    visitUrl(dlg.url());
}


void MainWindow::on_actionAdd_Bookmark_triggered()
{
    BookmarkDlg dlg(this);
    dlg.setUrl(m_currentUrl.toString());
    dlg.setTitle( m_isUsingGraphicsView ? m_pGWebView->title() : m_pWebView->title() );
    dlg.removeBtt()->setVisible(false);

    if ( dlg.exec()!= QDialog::Accepted ) return;

    Bookmark mark;
    mark.setTitle(dlg.title());
    mark.setUrl(dlg.url());
    m_pBookmarks->append(mark);
}

void MainWindow::on_actionLandscape_triggered() {
#ifdef Q_WS_MAEMO_5
    setLandscape(!isLandscape());
#endif
}

void MainWindow::on_actionAuto_Rotate_triggered() {
#ifdef Q_WS_MAEMO_5
    setAutoRotate(!isAutoRotate());
#endif
}

#ifdef Q_WS_MAEMO_5
void MainWindow::fullscreen_triggered() {
    setFullScreen(!isFullScreen());
}

void MainWindow::orientationChanged() {

    QRect screenGeometry = QApplication::desktop()->screenGeometry();

    qDebug() << "Orientation has changed: " << screenGeometry.width() <<", " << screenGeometry.height();

    // setLandscape( screenGeometry.width() > screenGeometry.height(), true );
}

void MainWindow::setLandscape(bool landscape/*=true*/) {
    m_pOrientationSelector->setSelectedIndex(landscape?1:2);

    if ( landscape ) {
        qDebug() << ">> set Landscape";
        //setAttribute(Qt::WA_Maemo5PortraitOrientation, false);
        setAttribute(Qt::WA_Maemo5LandscapeOrientation, true); //

    } else {
        qDebug() << ">> set Portrait";
        //setAttribute(Qt::WA_Maemo5LandscapeOrientation, false);
        setAttribute(Qt::WA_Maemo5PortraitOrientation, true);

    }
}

void MainWindow::setAutoRotate(bool autoRotate /*=true*/) {
    // http://qt.nokia.com/doc/qt-maemo-4.6/maemo5-rotation.html
    if ( autoRotate ) {
        qDebug() << "set autorotate: on";
        setAttribute(Qt::WA_Maemo5AutoOrientation, true);
        if ( m_pOrientationSelector ) m_pOrientationSelector->setSelectedIndex(0);
        return;
    }

    qDebug() << "set autorotate: off";
    setAttribute(Qt::WA_Maemo5AutoOrientation, false);
}

bool MainWindow::isLandscape(){
    return width()>height();
    // return testAttribute(Qt::WA_Maemo5LandscapeOrientation) || !testAttribute(Qt::WA_Maemo5PortraitOrientation);
}

bool MainWindow::isAutoRotate(){
    return testAttribute(Qt::WA_Maemo5AutoOrientation);
}

void MainWindow::grabZoomKeys(bool grab)
{
    if (!winId()) {
        qWarning("Can't grab keys unless we have a window id");
        return;
    }

    unsigned long val = (grab) ? 1 : 0;
    Atom atom = XInternAtom(QX11Info::display(), "_HILDON_ZOOM_KEY_ATOM", False);
    if (!atom) {
        qWarning("Unable to obtain _HILDON_ZOOM_KEY_ATOM. This example will only work "
                 "on a Maemo 5 device!");
        return;
    }

    XChangeProperty (QX11Info::display(),
            winId(),
            atom,
            XA_INTEGER,
            32,
            PropModeReplace,
            reinterpret_cast<unsigned char *>(&val),
            1);

}
#endif


void MainWindow::on_actionUser_Agent_triggered() {
    setUserAgent(!m_webPage.isIphoneUserAgent());
}

void MainWindow::setUserAgent(bool iphone/*=true*/) {
    m_webPage.setIPhoneUserAgent(iphone);

#ifdef Q_WS_MAEMO_5
    if ( m_pUserAgentSelector ) m_pUserAgentSelector->setSelectedIndex(iphone?0:1);
#endif
}

void MainWindow::linkClicked( const QUrl & url ) {
    qDebug() << "Link clicked: " << url.toString();
}

void MainWindow::unsupportedContent(QNetworkReply* pReply) {

    qDebug() << "unsupportedContent: " << pReply->request().url(); // << pReply;
    QList<QByteArray> headerList = pReply->rawHeaderList();
    QString mimeType = pReply->header(QNetworkRequest::ContentTypeHeader).toString();
    qDebug() << ">> mime type: " << mimeType;
    foreach ( QByteArray array, headerList ) {
        QString header(array);
        QString value(pReply->rawHeader(array));
        qDebug() << ">> Header: " << header << value;
    }

}

void MainWindow::downloadRequested (const QNetworkRequest& request) {
}

void MainWindow::finishedRequest(QNetworkReply* reply)
{
    if (reply->error() == QNetworkReply::NoError) return;

    m_errors << reply->errorString();
    qDebug() << "Error(s): " << m_errors;
}

void MainWindow::hideLoading(bool success) {
    setZoom(m_zoomFactor);

#ifdef Q_WS_MAEMO_5
    setAttribute(Qt::WA_Maemo5ShowProgressIndicator, false);

    // use notification
    // ...

    if ( !success ) {
        QString msg("<b>Loading Error</b><br/>");
        if ( m_errors.length()>0 ) msg+=m_errors.join("<br/>"); else msg+=m_currentUrl.toString();
        QMaemo5InformationBox::information(this,
            msg,
            QMaemo5InformationBox::DefaultTimeout
        );
        return;
    }
#else
    setCursor(QCursor(Qt::ArrowCursor));

    if ( !success ) QMessageBox::warning(this, "Loading Error", m_errors.join("\n"));
#endif
}

QWebPage *MainWindow::page() {
    return &m_webPage;
}
