#include "webview.h"
#include "mainwindow.h"

class FlickableTicker: QObject
{
public:
    FlickableTicker(WebView *scroller) {
        m_pScroller = scroller;
    }

    void start(int interval) {
        if (!m_timer.isActive())
            m_timer.start(interval, this);
    }

    void stop() {
        m_timer.stop();
    }

protected:
    void timerEvent(QTimerEvent *event) {
        Q_UNUSED(event);
        m_pScroller->tick();
    }

private:
    WebView *m_pScroller;
    QBasicTimer m_timer;
};

WebView::WebView(QWidget *parent)
        : QWebView(parent)
{
    setAttribute(Qt::WA_OpaquePaintEvent, true);
    setAttribute(Qt::WA_NoSystemBackground, true);

    setMouseTracking(true);
    // Flickable::setAcceptMouseClick(this);

    m_state = Steady;
    m_threshold = 20; // it was 10
    m_pTicker = new FlickableTicker(this);
    m_timeStamp = QTime::currentTime();
}

QPoint WebView::scrollOffset() const {
    QWebFrame *frame = page()->mainFrame();

    int x = frame->evaluateJavaScript("window.scrollX").toInt();
    int y = frame->evaluateJavaScript("window.scrollY").toInt();
    // qDebug() << "Current Scroll Offset: " << x << ", " << y;

    return QPoint(x, y);
}

void WebView::setScrollOffset(const QPoint &offset) {
    QWebFrame *frame = page()->mainFrame();
    if (!frame) {
        qDebug() << "frame not found?!?!";
        return;
    }

    // qDebug() << "Scroll To: " << offset.x() << ", " << offset.y();
    frame->evaluateJavaScript(QString("window.scrollTo(%1,%2);").arg(offset.x()).arg(offset.y()));
}

void WebView::keyReleaseEvent(QKeyEvent *event) {
    if (event->key() == Qt::Key_Down) {
        setScrollOffset( QPoint(0, scrollOffset().y() + 20) );
        return;
    }

    if (event->key() == Qt::Key_Up) {
        int y = scrollOffset().y();
        y = (y-20>0) ? y-20 :0;
        setScrollOffset( QPoint(0, y) );
        return;
    }
}

void WebView::mousePressEvent(QMouseEvent *pEvent) {
    if (m_ignoreList.removeAll(pEvent) || pEvent->button() != Qt::LeftButton) {
        m_state = Steady;
        QWebView::mousePressEvent(pEvent);
        return;
    }

    if ( pEvent->pos().y()<0 ) {
        qDebug() << "!!!! mousePress (bug:ignore event): y<0=" << pEvent->pos().y();
        switch ( m_state ) {
            case Steady:
            case ManualScroll:
            case AutoScroll:
            case Stop:
            case Bug:
            case Pressed:
                qDebug() << "!!! mousePress (bug:y<0): ignore event, new state=Bug";
                m_state = Bug;
                break;
        }

        pEvent->accept();
        // QWebView::mousePressEvent(event);
        return;
    }

    switch (m_state) {
        case Bug:
            qDebug() << "after bug... just start over...";

        case Steady:
            qDebug() << "handleMousePress: Steady - register click";
            pEvent->accept();
            m_state = Pressed;
            m_pressPos = pEvent->pos();
            m_offset = scrollOffset();
            break;

        case AutoScroll:
            qDebug() << "handleMousePress: AutoScroll - stop auto scrolling";
            pEvent->accept();
            m_state = Stop;
            m_speed = QPoint(0, 0);
            m_pressPos = pEvent->pos();
            m_offset = scrollOffset();
            m_pTicker->stop();
            break;

        case ManualScroll: // work around bug
            qDebug() << "handleMousePress: ManualScroll - workaround bug (do nothing)";
            pEvent->accept();
            break;

        default:
            qDebug() << "handleMousePress: error? " << m_state;
            QWebView::mousePressEvent(pEvent);
            break;
    }

}

void WebView::mouseMoveEvent(QMouseEvent *pEvent) {
    if (!(pEvent->buttons() & Qt::LeftButton)) {
        QWebView::mouseMoveEvent(pEvent);
        return;
    }

    if ( pEvent->pos().y()<0 ) {
        switch ( m_state ) {
            case Steady:
            case ManualScroll:
            case AutoScroll:
            case Stop:
            case Bug:
            case Pressed:
                qDebug() << "!!! mouseMove (bug:y<0): ignore event, new state=Bug";
                m_state = Bug;
                break;
        }

        pEvent->accept();
        // QWebView::mouseMoveEvent(event);
        return;
    }

    // if (m_ignoreList.removeAll(event)) return;

    QPoint delta;
    switch (m_state) {
        case Pressed:
        case Stop:
            delta = pEvent->pos() - m_pressPos;
            if ( delta.x() > m_threshold || delta.x() < -m_threshold ||
                delta.y() > m_threshold || delta.y() < -m_threshold )
            {
                qDebug() << "handleMouseMove (Pressed or Stop): start manual scroll";

                m_timeStamp = QTime::currentTime();
                m_state = ManualScroll;
                delta = QPoint(0, 0);
                m_pressPos = pEvent->pos();
                m_offset = scrollOffset(); // ?!?!?!?!?!? felipe
                pEvent->accept();

            } else {
                qDebug() << "handleMouseMove (Pressed or Stop): bellow threshold: " << m_state;

            }
            break;

        case ManualScroll: {
                qDebug() << "handleMouseMove: ManualScroll";
                pEvent->accept();
                delta = pEvent->pos() - m_pressPos;
                QPoint newpos = m_offset - delta;

                qDebug() << ">> Press: " << m_pressPos.y()
                        << ", Event: " << pEvent->pos().y()
                        << ", Delta: " << delta.y()
                        << ", Scroll: " << m_offset.y()
                        << " = " << newpos.y();

                setScrollOffset(newpos); // offset - delta

                if (m_timeStamp.elapsed() > 100) {
                    qDebug() << ">> elapsed()>100?";
                    m_timeStamp = QTime::currentTime();
                    m_speed = delta - delta;
                    delta = delta;
                }
            }
            break;

        case Bug: // after bug (simulate press)
            qDebug() << ">> after bug: simulate mouse press";
            m_state = Pressed;
            m_pressPos = pEvent->pos();
            m_offset = scrollOffset();
            break;

        default:
            qDebug() << "handleMouseMove: error?" << m_state;
            QWebView::mouseMoveEvent(pEvent);
            break;
    }
}

void WebView::mouseReleaseEvent(QMouseEvent *pEvent) {
    if (m_ignoreList.removeAll(pEvent) || pEvent->button() != Qt::LeftButton) {
        QWebView::mouseReleaseEvent(pEvent);
        return;
    }

    if ( pEvent->pos().y()<0 ) {
        switch ( m_state ) {
            case Steady:
            case ManualScroll:
            case AutoScroll:
            case Stop:
            case Bug:
            case Pressed:
                qDebug() << "!!! mouseRelease (bug:y<0): ignore event, new state=Bug";
                m_state = Bug;
                break;
        }

        pEvent->accept();
        // QWebView::mouseReleaseEvent(pEvent);
        return;
    }

    QPoint delta;
    switch (m_state) {
       case Pressed: {
            qDebug() << "handleMouseRelease: Pressed - fire fake events";

            pEvent->accept();
            m_state = Steady;
            QMouseEvent *m_pPressEvent = new QMouseEvent(
                QEvent::MouseButtonPress,
                m_pressPos, Qt::LeftButton,
                Qt::LeftButton, Qt::NoModifier
            );
            QMouseEvent *m_pReleaseEvent = new QMouseEvent(*pEvent);
            m_ignoreList << m_pPressEvent;
            m_ignoreList << m_pReleaseEvent;
            QApplication::postEvent(this, m_pPressEvent);
            QApplication::postEvent(this, m_pReleaseEvent);
            break;
        }

        case ManualScroll:
            qDebug() << "handleMouseRelease: ManualScroll";

            pEvent->accept();
            delta = pEvent->pos() - m_pressPos;
            if (m_timeStamp.elapsed() > 100) {
                qDebug() << ">> elapsed>100: ????";
                m_timeStamp = QTime::currentTime();
                m_speed = delta - delta;
                delta = delta;
            }

            m_offset = scrollOffset();
            m_pressPos = pEvent->pos();

            if (m_speed == QPoint(0, 0)) {
                qDebug() << ">> keep it steady?";
                m_state = Steady;
            } else {
                qDebug() << ">> start Autoscroll";
                m_speed /= 4;
                m_state = AutoScroll;
                m_pTicker->start(20);
            }
            break;

        case Stop:
            qDebug() << "handleMouseRelease: Stop - just stopped? make it steady";
            pEvent->accept();
            m_state = Steady;
            m_offset = scrollOffset();
            break;

        case Bug:
            qDebug() << ">> after bug: keep same (bug) status";
            break;

        default:
            qDebug() << "handleMouseRelease: error? " << m_state;
            QWebView::mouseReleaseEvent(pEvent);
            break;
    }


}

QWebView* WebView::createWindow ( QWebPage::WebWindowType type )
{
#ifndef WEBGRAPHICS
    Q_UNUSED(type);

    MainWindow* new_window = new MainWindow(QString::null);
    new_window->show();

    return new_window->getWebView();
#else
    return 0;
#endif
}

static QPoint deaccelerate(const QPoint &speed, int a = 1, int max = 64)
{
    int x = qBound(-max, speed.x(), max);
    int y = qBound(-max, speed.y(), max);
    x = (x == 0) ? x : (x > 0) ? qMax(0, x - a) : qMin(0, x + a);
    y = (y == 0) ? y : (y > 0) ? qMax(0, y - a) : qMin(0, y + a);
    return QPoint(x, y);
}

void WebView::tick()
{
    if (m_state ==  AutoScroll) {
        m_speed = deaccelerate(m_speed);
        setScrollOffset(m_offset - m_speed);
        m_offset = scrollOffset();
        if (m_speed == QPoint(0, 0)) {
            m_state = Steady;
            m_pTicker->stop();
        }
    } else {
        m_pTicker->stop();
    }
}
