#include "mainwindow.h"
#include "ui_mainwindow.h"
#include "fullscreenexitbutton.h"

#include "gotodlg.h"
#include "bookmark.h"
#include "bookmarksdlg.h"
#include "bookmarkdlg.h"

#ifdef Q_WS_MAEMO_5
    #include <QtGui/QX11Info>
    #include <X11/Xlib.h>
    #include <X11/Xatom.h>
    #include <QMaemo5InformationBox>
#endif

#include <QDebug>

#define SETTINGS_VERSION 1

MainWindow::MainWindow(QWidget *parent) :
    QMainWindow(parent),
    m_pUI(new Ui::MainWindow)
{
    m_pUI->setupUi(this);

    /*
    m_zoomLevels << 30 << 50 << 67 << 80 << 90;
    m_zoomLevels << 100;
    m_zoomLevels << 110 << 120 << 133 << 150 << 170 << 200 << 240 << 300;
    m_currentZoom = 100;
    */

    m_zoomFactor = 1;

    // loading label...
    m_pLoadingLabel = new QLabel(this);
    m_pLoadingLabel->setPixmap( QPixmap(QString::fromUtf8(":/images/spinner.png")) );

#ifdef Q_WS_MAEMO_5
    FullScreenExitButton *pButt = new FullScreenExitButton(this);
    connect(pButt, SIGNAL(clicked()), this, SLOT(showNormal()));

    grabZoomKeys(true);

    setAttribute(Qt::WA_Maemo5StackedWindow);

    m_pUI->actionFullscreen->setCheckable(false);
    connect(m_pUI->actionFullscreen, SIGNAL(triggered()), this, SLOT(fullscreen_triggered()));

#else // !Q_WS_MAEMO_5
    // error box
    m_pErrorBox = new ErrorBox(this);
    m_pErrorBox->hide();
    connect(m_pErrorBox->getRetryButton(), SIGNAL(clicked()), this, SLOT(reload()));

    // tool bar
    m_pToolbar = new Toolbar(this);
    m_pToolbar->hide();

    m_pMiniToolbar = new MinifiedToolbar(this);
    m_pMiniToolbar->show();

    connect(m_pMiniToolbar->getShowToolbarButton(), SIGNAL(clicked()), this, SLOT(showToolbar()));
    connect(m_pToolbar->getHideToolbarButton(), SIGNAL(clicked()), this, SLOT(hideToolbar()));

    connect(m_pToolbar->getBackButton(), SIGNAL(clicked()), m_pUI->webView, SLOT(back()));
    connect(m_pToolbar->getFowardButton(), SIGNAL(clicked()), m_pUI->webView, SLOT(forward()));
    connect(m_pToolbar->getReloadButton(), SIGNAL(clicked()), m_pUI->webView, SLOT(reload()));

    connect(m_pToolbar, SIGNAL(urlChanged(QString)), this, SLOT(visitUrl(QString)));

    m_pUI->actionFullscreen->setCheckable(true);
    connect(m_pUI->actionFullscreen, SIGNAL(toggled(bool)), this, SLOT(setFullScreen(bool)));
#endif

    loadSettings();


    // set web page
    m_webPage.mainFrame()->setScrollBarPolicy(Qt::Vertical, Qt::ScrollBarAlwaysOff);
    m_webPage.mainFrame()->setScrollBarPolicy(Qt::Horizontal, Qt::ScrollBarAlwaysOff);

    connect(m_webPage.networkAccessManager(), SIGNAL(finished(QNetworkReply*)),
        this, SLOT(finishedRequest(QNetworkReply*)));

    // m_webPage.setLinkDelegationPolicy(QWebPage::DelegateAllLinks);
    // m_webPage.settings()->setAttribute(QWebSettings::PluginsEnabled, true);
    // m_webPage.setForwardUnsupportedContent(true);
    m_webPage.settings()->setAttribute(QWebSettings::DeveloperExtrasEnabled, true);

    // offline?
    // QWebSettings::OfflineStorageDatabaseEnabled
    // QWebSettings::OfflineWebApplicationCacheEnabled
    // QWebSettings::enablePersistentStorage(path);

    // connect(&m_webPage, SIGNAL(linkClicked(QUrl)), this, SLOT(linkClicked(QUrl)));
    // connect(&m_webPage, SIGNAL(unsupportedContent(QNetworkReply*)), this, SLOT(unsupportedContent(QNetworkReply*)));

    // set webview
    connect(m_pUI->webView, SIGNAL(loadStarted()), this, SLOT(showLoading()));
    connect(m_pUI->webView, SIGNAL(urlChanged(QUrl)), this, SLOT(changeUrl(QUrl)));
    connect(m_pUI->webView, SIGNAL(loadFinished(bool)), this, SLOT(hideLoading(bool)));

    m_pUI->webView->setPage(&m_webPage);

    m_pUI->webView->setWindowTitle("Macuco");
    on_actionHome_triggered();
}

MainWindow::~MainWindow()
{
}

void MainWindow::loadSettings() {
    Bookmark::registerMeataType();

    QSettings settings("Macuco", "Macuco");

    int version = settings.value("version", 0).toInt();
    if ( version<SETTINGS_VERSION ) {
        qDebug() << "previous version of settings file";
        //...
    }

    m_bookmarks = settings.value("bookmarks").value< QList<Bookmark> >();
    setFullScreen(settings.value("fullScreen", false).toBool());
    setUserAgent(settings.value("iphoneUserAgent", true).toBool());

#ifndef Q_WS_MAEMO_5
    m_pUI->actionLandscape->setVisible(false);
    m_pUI->actionAuto_Rotate->setVisible(false);

#else
    setLandscape(settings.value("isLandscape", true).toBool());
    // setAutoRotate(settings.value("autoRotate", true).toBool());
    setAutoRotate(true);
    m_pUI->actionAuto_Rotate->setVisible(false);
#endif
}

void MainWindow::saveSettings() {
    QSettings settings("Macuco", "Macuco");
    settings.setValue("version", SETTINGS_VERSION);
    settings.setValue("fullScreen", isFullScreen());
    settings.setValue("iphoneUserAgent", m_webPage.isIphoneUserAgent());

#ifdef Q_WS_MAEMO_5
    settings.setValue("isLandscape", isLandscape());
    settings.setValue("autoRotate", isAutoRotate());
#endif

    QVariant bookmarks;
    bookmarks.setValue(m_bookmarks);
    settings.setValue("bookmarks", bookmarks);

}

QWebView* MainWindow::getWebView() {
    return m_pUI->webView;
}

void MainWindow::showLoading() {
    m_errors.clear();

#ifdef Q_WS_MAEMO_5
    // http://qt.nokia.com/doc/qt-maemo-4.6/maemo5-windowstates.html
    setAttribute(Qt::WA_Maemo5ShowProgressIndicator, true);
    if ( isFullScreen() ) m_pLoadingLabel->show();
#else
    m_pErrorBox->hide();
    hideOverlayOnPage();

    m_pLoadingLabel->show();
#endif
}

void MainWindow::changeUrl(QUrl url)
{
    /*
    QIcon icon = m_webPage.settings()->iconForUrl(url);
    qDebug() << icon.isNull();
    */

    m_currentUrl = url;

#ifndef Q_WS_MAEMO_5
    m_pToolbar->getInputAddress()->setText( url.toString() );
#endif
}

void MainWindow::visitUrl(QString urlString)
{
    m_pUI->webView->load(QUrl::fromUserInput(urlString));
}

void MainWindow::hideLoading(bool success) {
    m_pLoadingLabel->hide();

#ifdef Q_WS_MAEMO_5
    setAttribute(Qt::WA_Maemo5ShowProgressIndicator, false);

    // use notification
    // ...

    if ( !success ) {
        QString msg("<b>Loading Error</b><br/>");
        if ( m_errors.length()>1 ) msg+=m_errors.join("<br/>"); else msg+=m_currentUrl.toString();
        QMaemo5InformationBox::information(this,
            msg,
            QMaemo5InformationBox::DefaultTimeout
        );

        /*
        if ( QMessageBox::warning(
            this,
            "Loading Error",
            m_errors.count()>0 ? m_errors.join("\n") : "Unspecified",
            QMessageBox::Retry, QMessageBox::Cancel
        )== QMessageBox::Retry )
        {
            // retry
            reload();
        }
        */
        return;
    }

#else
    hideOverlayOnPage();

    if (success) {
        m_pErrorBox->hide();
    } else {
        showOverlayOnPage();
        m_pErrorBox->show();
    }
#endif
}

void MainWindow::reload()
{
    m_pUI->webView->load(m_currentUrl);

#ifndef Q_WS_MAEMO_5
    m_pErrorBox->hide();
    hideOverlayOnPage();
#endif
}

void MainWindow::adjustSizes() {
    m_pLoadingLabel->setGeometry( width()/2 - 45, height()/2 - 45, 90, 90); //put the label on the center

#ifndef Q_WS_MAEMO_5
    m_pErrorBox->setGeometry( width()/2 - 150, height()/2 - 75, 300, 150); //put the error box on the center
    m_pToolbar->setGeometry( 0, height() - 38, width(), 38 );
    m_pMiniToolbar->setGeometry( 0, height() - 38, 30, 38 );
#endif

    if ( m_webPage.isIphoneUserAgent() ) zoomFit();

}

void MainWindow::zoomFit() {
    if ( width() > height() ) {
        setZoom(qreal(width())/480.0);
    } else {
        setZoom(qreal(height())/480.0);
    }
}

void MainWindow::resizeEvent(QResizeEvent* event)
{
    adjustSizes();
    QWidget::resizeEvent(event);
}

void MainWindow::closeEvent(QCloseEvent *event)
{
#ifdef Q_WS_MAEMO_5
    grabZoomKeys(false);
#endif

    saveSettings();

    m_webPage.saveCookies();
    event->accept();
}

void MainWindow::finishedRequest(QNetworkReply* reply)
{
    if (reply->error() == QNetworkReply::NoError) return;

    m_errors << reply->errorString();
    qDebug() << "Error(s): " << m_errors;
}

void MainWindow::zoomIn() {
    setZoom(m_zoomFactor*1.15);
}

void MainWindow::zoomOut() {
    setZoom(m_zoomFactor/1.15);
}

void MainWindow::setZoom(qreal zoomFactor) {
    disableWebkitAutoSize();
    m_zoomFactor = zoomFactor;
    if ( m_zoomFactor>.90 && m_zoomFactor<1.1) m_zoomFactor =1;

    m_pUI->webView->setZoomFactor(m_zoomFactor);
}

void MainWindow::keyPressEvent(QKeyEvent *event)
{
    switch ( event->key() ) {
        case Qt::Key_F7:
            zoomIn();
            break;

        case Qt::Key_F8:
            zoomOut();
            break;

        case Qt::Key_B:
            if ( (event->modifiers()&Qt::ControlModifier) != Qt::ControlModifier  ) break;
            on_actionBookmarks_triggered();
            break;

        case Qt::Key_G:
            if ( (event->modifiers()&Qt::ControlModifier) != Qt::ControlModifier  ) break;
            on_actionGo_To_triggered();
            break;

        case Qt::Key_H:
            if ( (event->modifiers()&Qt::ControlModifier) != Qt::ControlModifier  ) break;
            on_actionHome_triggered();
            break;

        case Qt::Key_F:
            if ( (event->modifiers()&Qt::ControlModifier) != Qt::ControlModifier  ) break;
            setFullScreen(true);
            break;

        case Qt::Key_O:
            if ( (event->modifiers()&Qt::ControlModifier) != Qt::ControlModifier  ) break;
            setZoom(1);
            break;

        case Qt::Key_I:
            if ( (event->modifiers()&Qt::ControlModifier) != Qt::ControlModifier  ) break;
            zoomFit();
            break;
    }
}

void MainWindow::disableWebkitAutoSize()
{
    m_pUI->webView->page()->mainFrame()->evaluateJavaScript("document.body.style.WebkitTextSizeAdjust = 'auto'");
}

void MainWindow::on_actionHome_triggered() {
    visitUrl("http://www.crochik.com/macuco");
}

void MainWindow::on_actionBookmarks_triggered()
{
    BookmarksDlg dlg(m_bookmarks, this);

#ifdef Q_WS_MAEMO_5
    dlg.setAttribute(Qt::WA_Maemo5StackedWindow);
    dlg.setWindowFlags(Qt::Window);
#endif

    if ( dlg.exec()!= QDialog::Accepted || dlg.m_bookmark.isNull()) return;

    visitUrl(dlg.m_bookmark.url());
}

void MainWindow::setFullScreen(bool full/*=true*/) {
#ifndef Q_WS_MAEMO_5
    if ( m_pUI->actionFullscreen->isChecked()!=full ) {
        // change status of menu item and wait for event to be triggered
        m_pUI->actionFullscreen->setChecked(full);
        return;
    }
#endif

    if (!full) {
        showNormal();
    } else {
        showFullScreen();
    }
}

bool MainWindow::isFullScreen() {
    return windowState() & Qt::WindowFullScreen;
}

void MainWindow::on_actionGo_To_triggered()
{
    GoToDlg dlg(this);
    dlg.setUrl(m_currentUrl.toString());
    if ( dlg.exec()!=QDialog::Accepted ) return;
    visitUrl(dlg.url());
}


void MainWindow::on_actionAdd_Bookmark_triggered()
{
    BookmarkDlg dlg(this);
    dlg.setUrl(m_currentUrl.toString());
    dlg.setTitle(m_pUI->webView->title());
    dlg.removeBtt()->setVisible(false);

    if ( dlg.exec()!= QDialog::Accepted ) return;

    Bookmark mark;
    mark.setTitle(dlg.title());
    mark.setUrl(dlg.url());
    m_bookmarks.append(mark);
}

void MainWindow::on_actionLandscape_triggered() {
#ifdef Q_WS_MAEMO_5
    setLandscape(!isLandscape());
#endif
}

void MainWindow::on_actionAuto_Rotate_triggered() {
#ifdef Q_WS_MAEMO_5
    setAutoRotate(!isAutoRotate());
#endif
}

#ifdef Q_WS_MAEMO_5
void MainWindow::fullscreen_triggered() {
    setFullScreen(!isFullScreen());
}

void MainWindow::orientationChanged() {

    QRect screenGeometry = QApplication::desktop()->screenGeometry();

    qDebug() << "Orientation has changed: " << screenGeometry.width() <<", " << screenGeometry.height();
    setLandscape( screenGeometry.width() > screenGeometry.height(), true );
}

void MainWindow::setLandscape(bool landscape/*=true*/, bool force/*=false*/) {
    if ( !force && landscape==isLandscape() ) return;

    if ( landscape ) {
        qDebug() << ">> set Landscape";
        //setAttribute(Qt::WA_Maemo5PortraitOrientation, false);
        setAttribute(Qt::WA_Maemo5LandscapeOrientation, true);

    } else {
        qDebug() << ">> set Portrait";
        //setAttribute(Qt::WA_Maemo5LandscapeOrientation, false);
        setAttribute(Qt::WA_Maemo5PortraitOrientation, true);

    }
}

void MainWindow::setAutoRotate(bool autoRotate /*=true*/) {
    // http://qt.nokia.com/doc/qt-maemo-4.6/maemo5-rotation.html
    if ( autoRotate ) {
        qDebug() << "set autorotate: on";
        m_pUI->actionAuto_Rotate->setText("Auto Rotate");
        setAttribute(Qt::WA_Maemo5AutoOrientation, true);

        connect(QApplication::desktop(), SIGNAL(resized(int)), this, SLOT(orientationChanged()));
        return;
    }

    qDebug() << "set autorotate: off";
    m_pUI->actionAuto_Rotate->setText("Disable Auto-Rotate");
    setAttribute(Qt::WA_Maemo5AutoOrientation, false);

    disconnect(QApplication::desktop(), SIGNAL(resized(int)), this, SLOT(orientationChanged()));
}

bool MainWindow::isLandscape(){
    return testAttribute(Qt::WA_Maemo5LandscapeOrientation) || !testAttribute(Qt::WA_Maemo5PortraitOrientation);
}

bool MainWindow::isAutoRotate(){
    return testAttribute(Qt::WA_Maemo5AutoOrientation);
}

void MainWindow::grabZoomKeys(bool grab)
{
    if (!winId()) {
        qWarning("Can't grab keys unless we have a window id");
        return;
    }

    unsigned long val = (grab) ? 1 : 0;
    Atom atom = XInternAtom(QX11Info::display(), "_HILDON_ZOOM_KEY_ATOM", False);
    if (!atom) {
        qWarning("Unable to obtain _HILDON_ZOOM_KEY_ATOM. This example will only work "
                 "on a Maemo 5 device!");
        return;
    }

    XChangeProperty (QX11Info::display(),
            winId(),
            atom,
            XA_INTEGER,
            32,
            PropModeReplace,
            reinterpret_cast<unsigned char *>(&val),
            1);

}

#else  // !Q_WS_MAEMO_5
void MainWindow::showOverlayOnPage()
{
    //TODO: is there a better way to put a transparent overlay on the screen?
    QString script =
           "newDiv = document.createElement(\"div\");"
           "newDiv = document.createElement(\"div\");"
           "newDiv.innerHTML = \"<div id='divOverlay' style='width: 100%; height: 100%; background: white;"
           "position: absolute; top: 0; left: 0; opacity: 0.8'></div>\";"
           "my_div = document.getElementById(\"body\");"
           "document.body.appendChild(newDiv);";

    m_webPage.mainFrame()->evaluateJavaScript(script);
}

void MainWindow::hideOverlayOnPage()
{
    QString script = "node = document.getElementById(\"divOverlay\");"
                     "node.parentNode.removeChild(node);";

    m_webPage.mainFrame()->evaluateJavaScript(script);
}

void MainWindow::showToolbar()
{
    m_pToolbar->show();
    m_pMiniToolbar->hide();
}

void MainWindow::hideToolbar()
{
    m_pToolbar->hide();
    m_pMiniToolbar->show();
}

#endif


void MainWindow::on_actionUser_Agent_triggered()
{
    setUserAgent(!m_webPage.isIphoneUserAgent());
    // reload();
}

void MainWindow::setUserAgent(bool iphone/*=true*/) {
    m_webPage.setIPhoneUserAgent(iphone);
    m_pUI->actionUser_Agent->setText(iphone ? "Default User Agent" : "iPhone User Agent");
}

void MainWindow::linkClicked( const QUrl & url ) {
    qDebug() << "Link clicked: " << url.toString();
}

void MainWindow::unsupportedContent(QNetworkReply* pReply) {
    qDebug() << "unsupportedContent: " << pReply;
}
