/*
 * driver.c -  driver vkontakte of MSA program
 * This file is part of MSA program.
 *
 * Copyright (C) 2009 - Sergey Zakharov
 *
 * MSA program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * MSA program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with MSA program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor,
 * Boston, MA  02110-1301  USA
 */

#include <libxml2/libxml/tree.h>

#include "driver.h"
#include "vkontakte.h"
#include "parser.h"
#include "profile.h"
#include "dbus_functions.h"

#ifdef DEBUG
#	define msa_xmlDocDump(F,D) xmlDocDump(F,D)
#else
#	define msa_xmlDocDump(F,D)
#endif

#define TEST_AUTH_RESPONSE "http://vkontakte.ru/api/login_success.html#session={\"mid\":39567995,\"sid\":\"b9991e54a664fe7c00a31f4b4a988d77a968faac3af68646fe615ae3fc\",\"secret\":\"bc711f8ee6\",\"expire\":0,\"sig\":\"7506d1323df4d633176091b8ded68e0c\"}"

#define MSG_NETWORK_ERROR "<error><error_code>"DRV_NETW_ERROR_CODE"</error_code>\
                              <error_msg>"DRV_NETW_ERROR"</error_msg></error>"

#define MSG_NETWORK_SERVER_ERROR "<error><error_code>"DRV_SERVER_ERROR_CODE"</error_code>\
                              <error_msg>"DRV_SERVER_ERROR"</error_msg></error>"

#define MSG_SETTINGS_SAVED "<info><info_code>"DRV_SETT_SAVED_CODE"</info_code>\
                              <info_msg>"DRV_SETT_SAVED"</info_msg></info>"

#define MSG_SETTINGS_SAVED_ERROR "<error><error_code>"DRV_SETT_ERROR_CODE"</error_code>\
                              <error_msg>"DRV_SETT_ERROR"</error_msg></error>"

///FIXME
#define MSG_FILE_DOWNLOAD "<info><info_code>"DRV_FILE_DOWNLOAD_CODE"</info_code>\
                              <info_msg>"DRV_FILE_DOWNLOAD"</info_msg></info>"

#define MSG_FILE_DOWNLOAD_ERROR "<error><error_code>"DRV_FILE_DOWNLOAD_ERROR_CODE"</error_code>\
                              <error_msg>"DRV_FILE_DOWNLOAD_ERROR"</error_msg></error>"


#define XSL_URL DATADIR"/msa/drivers/vkontakte/make_url.xsl"
#define XSL_CONFIG_DIR DATADIR"/msa/drivers/vkontakte/"
#define AUTH_URL "http://vkontakte.ru/login.php?app=1899304&layout=touch&type=browser&settings=15615"
#define NOT_ESCAPE "!\"#$%&'()*+,-./0123456789:;<=>?@ABCDEFGHIJKLMNOPQRSTUVWXYZ[]^_`abcdefghijklmnopqrstuvwxyz{|}~"


#define REGEXP_PARAM "regexp"
#define TITLE_PARAM "title"
#define CLIENT_ID_PARAM "ClientID"
#define ACCOUNT_ID_PARAM "accountID_RET"

#define LOGIN_SUCCESS "^http://vkontakte\\.ru/api/(login_success|login_failure)\\.html.*"
#define TITLE_VKONTAKTE "vkontakte"

#ifdef TEST
#include "../test/CTF/src/ctf.h"
CTF_TEST_DATA_FILE_NAME_DECL;
#endif

int counter = 0;
int max_count = 0;
static int global_req = 0;
static int global_get = 0;
int attempts = 0;
int captch = 0;



/**
 * Allocate memory
 *
 * @param ptr - pointer
 * @param size - size of data
* @return 0 on success or 1 otherwise.
 */
void* myrealloc(void* ptr, size_t size)
{
/* There might be a realloc() out there that doesn't like reallocing
NULL pointers, so we take care of it here */
    if(ptr)
        return realloc(ptr, size);
    else
        return malloc(size);
}

/**
 * Callbak for libcurl for receiving data
 *
 * @param ptr - pointer
 * @param size - size of chunk of data
 * @param nmemb - number of chunks of data
 * @param data - pointer to data
* @return 0 on success or 1 otherwise.
 */
 static size_t
WriteMemoryCallback(void* ptr, size_t size, size_t nmemb, void* data)
{
    size_t realsize = size * nmemb;
    struct MemoryStruct *mem = (struct MemoryStruct *)data;
    mem->memory = myrealloc(mem->memory, mem->size + realsize + 1);
    if (mem->memory) {
        memcpy(&(mem->memory[mem->size]), ptr, realsize);
        mem->size += realsize;
        mem->memory[mem->size] = 0;
    }
    return realsize;
}


/*
 * Process all internal requests in xml request
 *
 * @param pr - profile data
 * @param inner_req - xml request
 * @return 0 on success, 1 otherwise
 */
int process_inner_request(struct ProfileData *pr, xmlDocPtr inner_req)
{   
    int ret_code = 0;
    xmlDocPtr inner_request = NULL, inner_response = NULL;
    xmlNodePtr node = NULL, node_request = NULL, tmp_node = NULL;
    msa_debug("START process_inner_request");
    msa_xmlDocDump(stderr, inner_req);
    while ((node = xpath_get_node("//Request", inner_req)) != NULL){
        node_request = xmlCopyNode(node, 1);
        xmlUnlinkNode(node_request);
        inner_request = xmlNewDoc(BAD_CAST "1.0");
        xmlDocSetRootElement(inner_request, node_request);
        
        if(process_request(pr, inner_request, &inner_response) == 1)
            info_message(DRV_REQ_ERROR_CODE, inner_response, DRV_REQ_ERROR, 1);

        msa_debug("INNER REQUEST\n");
        msa_xmlDocDump(stderr, inner_request);
        msa_debug("INNER RESPONSE\n");
        msa_xmlDocDump(stderr, inner_response);
        //TODO        
        //Check if error code = 6 sleep(1) and try request
        tmp_node =  xpath_get_node("//Response[@function='errorMessage']/Params/string[@name='code']", inner_response);
        if (tmp_node != NULL){
            gchar* code = NULL;
            code = (gchar*)xmlNodeGetContent(tmp_node);
            if(g_strcmp0(code, "6") == 0){
                g_free(code);
                msa_debug("Error: Too many requests per second");
                xmlFreeDoc(inner_response);
                inner_response = NULL;
                msa_debug("Sleep");
                sleep(1);
                if(process_request(pr, inner_request, &inner_response) == 1)
                    info_message(DRV_REQ_ERROR_CODE, inner_response, DRV_REQ_ERROR, 1);
                msa_debug("INNER RESPONSE\n");
                msa_xmlDocDump(stderr, inner_response);
            }              
        }
       xmlFreeDoc(inner_request); 

        if (xpath_get_node("//Response", inner_response) != NULL){
            node = xmlDocGetRootElement(inner_req);
            ret_code = 1;
        }

        tmp_node = xmlCopyNode(xmlDocGetRootElement(inner_response),1);
        xmlFreeDoc(inner_response);
        inner_response = NULL;
        node = xmlReplaceNode(node, tmp_node);
        xmlFreeNode(node);
        if (ret_code)
            break;
    }
    return ret_code;
}

/*
 * Recursively process request 
 *
 * @param pr - profile data
 * @param inner_req - xml request
 * @return 0 on success or 1 otherwise.
 */

int process_request(struct ProfileData *pr, xmlDocPtr request, xmlDocPtr* response)
{   
    msa_debug("\n\nRequest in process");
    msa_xmlDocDump(stdout, request);

    xmlDocPtr inner_req = NULL;
    xmlDocPtr server_response, resp = NULL;
    xmlNodePtr node = NULL, sig_node = NULL, params_node = NULL;
    gchar* parser = NULL;
    gchar* parser_type = NULL;
    gchar *url = NULL;
    gchar *sig = NULL;
    gchar *tmp = NULL;
    gchar *resp_str = NULL;
    gchar *md5_sig = NULL;
    gchar* data = NULL;
    gchar* http_type = NULL;
    gchar* params = NULL;
    gchar* encoded_params = NULL;
    int http_code = 0;
    int resp_code = 0;
    int data_size = 0;

    msa_debug("START process_request");
    msa_xmlDocDump(stdout, request);
    if (xpath_get_node("//Request", request) != NULL) {
        node = xpath_get_node("//Request/parser", request);

        if (node != NULL){
            tmp = (gchar*)xmlNodeGetContent(node);
            parser = g_strconcat(XSL_CONFIG_DIR, tmp, NULL);
            g_free(tmp);
        } else {          
            //use standart parser
            parser = g_strconcat(XSL_CONFIG_DIR, "parser.xsl", NULL);       
        }

        //create inner request
        inner_req = apply_parser(request, parser, pr);
        g_free(parser);
        msa_xmlDocDump(stdout, inner_req);
        
        //unknown request 
        if (xpath_get_node("//InnerRequest", inner_req) == NULL){
            *response = xmlCopyDoc(inner_req, 1);
            xmlFreeDoc(inner_req);
            return 0;
        }
        if ((resp_code = process_inner_request(pr, inner_req)) == 1){
            *response = xmlCopyDoc(inner_req, 1);
            xmlFreeDoc(inner_req);
            return 0;
        }


        node = xpath_get_node("//InnerRequest", inner_req);

        http_type = (gchar*)xmlGetProp(node, BAD_CAST "type");

        if (g_strcmp0(http_type, "post") == 0 || g_strcmp0(http_type, "get") ==0) {
            node = xpath_get_node("//InnerRequest/url", inner_req);
            if (node != NULL) {
                tmp = (gchar*)xmlNodeGetContent(node);
            } else {
                xmlFreeDoc(inner_req);
                g_free(http_type);
                return 1;
            }

            params_node = xpath_get_node("//InnerRequest/params", inner_req);
            if (params_node != NULL){
                params = (gchar*)xmlNodeGetContent(params_node);
                encoded_params = g_uri_escape_string(params, NOT_ESCAPE, 0); 
                g_free(params);               
            }

            sig_node = xpath_get_node("//InnerRequest/sig", inner_req);
            if (sig_node != NULL){
                sig = (gchar*)xmlNodeGetContent(sig_node);
                md5_sig = g_compute_checksum_for_string(G_CHECKSUM_MD5, sig, strlen(sig));
                g_free(sig);

                url = g_strconcat(tmp, encoded_params, "sig=", md5_sig, NULL);
                g_free(md5_sig);
            } else 
                url = g_strconcat(tmp, encoded_params, NULL);
            g_free(tmp);
            if(encoded_params != NULL)
                g_free(encoded_params);

            msa_debug("URL = %s\n", url);

            if(g_strcmp0(http_type, "get") == 0){
                http_code = http_get(url, info_proxy, info_port, NULL, &data, &data_size);
            }
            if(g_strcmp0(http_type, "post") == 0){
                msa_debug("UPLOADING...");
                struct UpFile *file = (struct UpFile *)malloc(sizeof (struct UpFile));
                node = xpath_get_node("//InnerRequest/post_param[@name='name']", inner_req);
                file->name = (gchar*)xmlNodeGetContent(node);
                node = xpath_get_node("//InnerRequest/post_param[@name='file']", inner_req);
                file->file_path = (gchar*)xmlNodeGetContent(node);
                node = xpath_get_node("//InnerRequest/post_param[@name='file_name']", inner_req);
                file->file_name = (gchar*)xmlNodeGetContent(node);
                
                http_code = http_upload_file(url, info_proxy, info_port, NULL, &data, pr, file);
                upfile_remove(file);
                msa_debug(data);
            }
            g_free(http_type);
            g_free(url);

            if (http_code >= 300 && http_code < 500){
                g_free(data);
                data = g_strdup(MSG_NETWORK_ERROR);
            } 
            if (http_code >= 500){
                g_free(data);
                data = g_strdup(MSG_NETWORK_SERVER_ERROR);
            }
        }

        msa_xmlDocDump(stdout, inner_req);

        //Parse server response
        node = xpath_get_node("//InnerRequest/parser", inner_req);
        if (node != NULL){
            parser_type = (gchar*)xmlGetProp(node, BAD_CAST "type");
            tmp = (gchar*)xmlNodeGetContent(node);

            if(http_code < 300){
                if (g_strcmp0(parser_type, "json") == 0) {
                    parse_upload(data, &resp_str);
                    g_free(data);
                    data = g_strdup(resp_str);
                    g_free(resp_str);
                }           
                if (g_strcmp0(parser_type, "settings") == 0) {
                    msa_debug("SETTINGS");
                    if (set_settings(inner_req, pr)){
                        g_free(data);
                        data = g_strdup(MSG_SETTINGS_SAVED_ERROR);
                    } else {
                        g_free(data);
                        data = g_strdup(MSG_SETTINGS_SAVED);                    
                    }
                }           
                if (g_strcmp0(parser_type, "download") == 0) {
                    if (save_file (inner_req, data, data_size)) {
                        g_free(data);
                        data = g_strdup(MSG_FILE_DOWNLOAD_ERROR);
                    } else {
                        g_free(data);
                        data = g_strdup(MSG_FILE_DOWNLOAD);
                    }
                    msa_debug("response = %s",data);                    
                }
                if (g_strcmp0(parser_type, "text") == 0) {
                    close_tags(data);
                }

            }
            server_response = xmlParseDoc((xmlChar*)data);

            parser = g_strconcat(XSL_CONFIG_DIR, tmp, NULL);

            resp = apply_parser(server_response, parser, pr);
 
            g_free(parser);
            xmlFreeDoc(server_response);
            g_free(tmp);
            g_free(parser_type);
            g_free(data);
            xmlFreeDoc(inner_req);
        }
        //Search for requests
        process_inner_request(pr, resp);
        //if(*response != NULL)
        //    xmlFreeDoc(*response);
         *response = xmlCopyDoc(resp, 1);
        xmlFreeDoc(resp);
        return 0;
    }

    return 1;
}


xmlDocPtr apply_parser(xmlDocPtr request, gchar* parser, struct ProfileData *pr)
{
    msa_debug("Aply parser: %s", parser);
    xmlDocPtr proc = NULL; 
    xsltStylesheetPtr cur = NULL;
    gchar *params[7];

    cur = xsltParseStylesheetFile((xmlChar*)parser);

    if(pr != NULL){
        params[0] = g_strdup("MID");
        params[1] = g_strdup_printf("\"%s\"", pr->remixmid);
        params[2] = g_strdup("SID");
        params[3] = g_strdup_printf("\"%s\"", pr->remixsid);
        params[4] = g_strdup("SECRET");
        params[5] = g_strdup_printf("\"%s\"", pr->secret);
        params[6] = NULL;
    } else
        return NULL;
    
    if(cur != NULL && request != NULL) {
        proc = xsltApplyStylesheet(cur, request, (const char**)params);
        xsltFreeStylesheet(cur);
    }

    g_free((gpointer*)params[0]);
    g_free((gpointer*)params[1]);
    g_free((gpointer*)params[2]);
    g_free((gpointer*)params[3]);
    g_free((gpointer*)params[4]);
    g_free((gpointer*)params[5]);

    return proc;
}
/*
 * Parse back url, fill profileData struct
 *
 * @param pr - profile
 * @param back_url - redirect url from server
 * @return 0 if login success, 1 if login failure
 */

int parse_auth(struct ProfileData *pr, char *back_url)
{
    int i = 0;
    gchar **list;
    if(back_url != NULL){
        //msa_debug(back_url);
        if (g_strrstr(back_url, "login_success") != NULL)
        {
            //msa_debug("login_success");
            g_strdelimit(back_url,"\"",' ');
            g_strdelimit(back_url,",{}",':');

            list = g_strsplit(back_url, ":", 0);
            i = 0;
            while(list[i] != NULL && list[i+1] != NULL){
                g_strstrip(list[i]);
                g_strstrip(list[i+1]);
                if(g_strcmp0(list[i], "mid") == 0)
                    pr->remixmid = g_strdup(list[i+1]);
                if(g_strcmp0(list[i], "sid") == 0)
                    pr->remixsid = g_strdup(list[i+1]);
                if(g_strcmp0(list[i], "secret") == 0)
                    pr->secret = g_strdup(list[i+1]);
                i++;
            }
            g_strfreev(list);
        } else {
            msa_debug("login failure");
            return 1;
        }
    } else
        return 1;
return 0;
}
/**
 * Perform request
 *
 * @param reqXml - request
 * @param doc - response
 * @return 0 on success or 1 otherwise.
 */
int vk_req(xmlDocPtr reqXml, xmlDocPtr* doc, const struct msa_module* info)
{
    captch = 0;
    int init_status = 0;
    char* error_code = NULL;
    gchar* func_val = NULL;
    xmlNodePtr node = NULL;

    struct ProfileData *cp = (struct ProfileData *)profile_get_by_id(info->id);
    msa_debug("%s:%d:%s: id=%s, cp->login=%s", __FILE__, __LINE__, __FUNCTION__, info->id, cp->login);
    info_proxy = info->proxy;
    info_port = info->port;

    global_req++;
    
    msa_debug("IN VK_REQ!!! %d\n", global_req);
    counter = 0;
    msa_xmlDocDump(stdout, reqXml);

    func_val = (gchar*)get_req_function(reqXml);
    if (g_strcmp0(func_val, "setSettings") != 0
        && g_strcmp0(func_val, "getSettings") != 0)
        init_status = initialization(cp);
    g_free(func_val);

    attempts = 0;
    if (init_status != 0) {
        msa_debug("init_status = %d", init_status);
        if (init_status == 2){
            info_message(DRV_NETW_ERROR_CODE, *doc, DRV_NETW_ERROR, 1);
            return 0;
        }
        if(init_status == 1) {
            info_message(DRV_AUTH_ERROR_CODE, *doc, DRV_AUTH_ERROR, 1);
            return 0;
        }
                 
    }
    if (process_request(cp, reqXml, doc) == 1){
        info_message(DRV_REQ_ERROR_CODE, *doc, DRV_REQ_ERROR, 1);
        return 0;
    }

    node = xpath_get_node("//Response[@function='errorMessage']/Params/string[@name='code']", *doc);
    if (node != NULL){
        msa_debug("error occured");
        error_code = (gchar *)xmlNodeGetContent(node);
        if(g_strcmp0(error_code, DRV_AUTH_ERROR_CODE) == 0 
            || g_strcmp0(error_code, "3") == 0
            || g_strcmp0(error_code, "4") == 0){
            if (authorize(cp) == 0) {
                xmlFreeDoc(*doc);
                process_request(cp, reqXml, doc);
            }
        }
        g_free(error_code);
            
    }
    return 0;
}

/**
 * Perform authorization
 *
 * @param cp - current profile settings
 * @return 0 on success or 1 otherwise.
 */
int initialization(struct ProfileData *cp)
{
    msa_debug("init START");

    int auth_status = 0;

    if(cp->remixmid == NULL || cp->remixsid == NULL || cp->secret == NULL) {

        g_print("dr_vk: authorization\n");
        auth_status = authorize(cp);
        if (auth_status != 0 ) {
            msa_debug("dr_vk: Could not get remixmid, remixsid, auth_status = %d \n", auth_status); 
            return auth_status;
        }
    }
    msa_debug("init END");
    return 0;
}

/**
 * Get settings from current profile
 *
 * @param doc - response
 * @return 0 on success or 1 otherwise.
 */
int get_settings(xmlDocPtr doc, const struct ProfileData *cp)
{
    msa_debug("get_settings START");

    xmlNodePtr rootNode, childNode;
    
    rootNode = xmlDocGetRootElement(doc);
    rootNode = rootNode->children;
    
    childNode = xmlNewDocNode(doc, NULL, BAD_CAST STRING, BAD_CAST cp->login);
    xmlSetProp(childNode, BAD_CAST NAME, BAD_CAST LOGIN);
    childNode = xmlAddChild(rootNode, childNode);

    childNode = xmlNewDocNode(doc, NULL, BAD_CAST STRING, BAD_CAST cp->password);
    xmlSetProp(childNode, BAD_CAST NAME, BAD_CAST PASSWORD);
    childNode = xmlAddChild(rootNode, childNode);

    msa_debug("get_settings END");
   return 0;
}

/**
 * Set settings in to current profile
 *
 * @param reqXml - request
 * @param cp - current profile
 * @return 0 on success or 1 otherwise.
 */
int set_settings (xmlDocPtr reqXml, struct ProfileData *cp)
{
    msa_debug("set_settings START");
    xmlNodePtr node;

    node = xpath_get_node("//*/Params/string[@name='mid']", reqXml);
    if (node == NULL)
        return 1;
    if (cp->remixmid != NULL)
    {
        g_free(cp->remixmid);
        cp->remixmid = NULL;
    }
    cp->remixmid = (gchar *)xmlNodeGetContent(node);

    node = xpath_get_node("//*/Params/string[@name='sid']", reqXml);
    if (node == NULL)
        return 1;
    if (cp->remixsid != NULL)
    {
        g_free(cp->remixsid);
        cp->remixsid = NULL;
    }
    cp->remixsid = (gchar *)xmlNodeGetContent(node);

    node = xpath_get_node("//*/Params/string[@name='secret']", reqXml);
    if (node == NULL)
        return 1;
    if (cp->secret != NULL)
    {
        g_free(cp->secret);
        cp->secret = NULL;
    }
    cp->secret = (gchar *)xmlNodeGetContent(node);

    return 0;
}

int close_tags(gchar* data)
{
    char* tmp = NULL;

    while((tmp = strstr(data, "<br>")) != NULL)
    {
        int k = 0;
        for(; k < 3; k++)
            *(tmp + k) = ' ';
        *(tmp + k) = '\n';
    }
    return 0;
}
/**
 * Save file to path in reqXml
 *
 * @param reqXml - request
 * @param data - binary data
 * @return 0 on success or 1 otherwise.
 */
int save_file (xmlDocPtr reqXml, char* data, int data_size)
{
    xmlNodePtr node;
    gchar* path = NULL;

    node = xpath_get_node("//*/Params/string[@name='path']", reqXml);
    if (node == NULL)
        return 1;
    path = (gchar *)xmlNodeGetContent(node);
    msa_debug("try save file to \"%s\"", path);
    if (path != NULL)
    {
        if(g_file_set_contents(path, data, data_size, NULL))
            msa_debug("write to file");
        else {
            g_free(path);
            return 1;
        }
    } else 
        return 1;
    g_free(path);
    return 0;
}

/**
 * Authorization (get remixsid and remixmid for future requests)
 *
 * @param cp - current profile
 * @return 0 on success or 1 otherwise.
 */
int authorize(struct ProfileData *cp)
{
    int ret_code = 0;
    gchar* back_url = NULL;
    gchar* port = g_strdup_printf("%d", info_port);

    if (cp == NULL){
        msa_debug("can't authorize, no profile!");
        return 1;
    }
        
#ifdef TEST
    back_url = g_strdup(TEST_AUTH_RESPONSE);
#else  
    g_debug("Try call webauth with %s = %s", CLIENT_ID_PARAM, cp->id);
    char *patterns[] = { REGEXP_PARAM, LOGIN_SUCCESS, TITLE_PARAM, cp->id, CLIENT_ID_PARAM, TITLE_VKONTAKTE, ACCOUNT_ID_PARAM, cp->id};
    char **patt_arr = patterns;
    back_url = webauth(AUTH_URL, info_proxy, port, "", "", patt_arr, 8, cp->id);

#endif
    //msa_debug("back_url = %s", back_url);
    ret_code = parse_auth(cp, back_url);
    if(back_url != NULL)
        g_free(back_url);
    g_free(port);
   return ret_code;
}

/**
 * Get class of request
 *
 * @param reqXml - request
 * @return class of request.
 */
xmlChar* get_req_class(xmlDocPtr reqXml)
{
    xmlNodePtr rootNode = NULL;
    if(reqXml != NULL)
        rootNode = xmlDocGetRootElement(reqXml);
    if (rootNode != NULL)
        return xmlGetProp(rootNode, BAD_CAST CLASS);
    return NULL;
}

/**
 * Get function of request
 *
 * @param reqXml - request
 * @return function of request.
 */
xmlChar* get_req_function(xmlDocPtr reqXml)
{
    xmlNodePtr rootNode = NULL;
    if (reqXml != NULL)
	rootNode = xmlDocGetRootElement(reqXml);
    if (rootNode != NULL)
	return xmlGetProp(rootNode, BAD_CAST FUNCTION);
    return NULL;
}


/**
 * Form INFO or ERROR message
 *
 * @param code - error code
 * @param doc - response (buffer for message)
 * @param message - text of message
 * @param type - 0 for info message or 1 for error message
* @return 0 on success or 1 otherwise.
 */
int info_message(char* code, xmlDocPtr doc, char* message, int type)
{
    xmlNodePtr rootNode, node, childNode;
    rootNode = xmlNewDocNode(doc, NULL, BAD_CAST "Response", NULL);
    xmlNodePtr oldroot = xmlDocSetRootElement(doc, rootNode);
    if (oldroot != NULL)
        xmlFreeNode(oldroot);

    xmlSetProp(rootNode, BAD_CAST CLASS, BAD_CAST "systemMessages");

    if (type == 1)
        xmlSetProp(rootNode, BAD_CAST FUNCTION, BAD_CAST "errorMessage");
    else if (type == 0)
        xmlSetProp(rootNode, BAD_CAST FUNCTION, BAD_CAST "infoMessage");
    else if (type == 2)
        xmlSetProp(rootNode, BAD_CAST FUNCTION, BAD_CAST "captchaMessage");
    else
        return 1;

    node = xmlNewDocNode(doc, NULL, BAD_CAST PARAMS, NULL);
    childNode = xmlAddChild(rootNode, node);

    node = xmlNewDocNode(doc, NULL, BAD_CAST STRING, BAD_CAST DRIVER_NAME);
    xmlSetProp(node, BAD_CAST NAME, BAD_CAST "moduleName");
    node = xmlAddChild(childNode, node);

    node = xmlNewDocNode(doc, NULL, BAD_CAST STRING, BAD_CAST code);
    xmlSetProp(node, BAD_CAST NAME, BAD_CAST "code");
    node = xmlAddChild(childNode, node);

    node = xmlNewDocNode(doc, NULL, BAD_CAST STRING, BAD_CAST message);
    xmlSetProp(node, BAD_CAST NAME, BAD_CAST "text");
    node = xmlAddChild(childNode, node);
    return 0;
}


/**
 * Perform HTTP GET request
 *
 * @param url - request
 * @param proxy - proxy address 
 * @param port - proxy port
 * @param header - buffer for header of response
 * @param body - buffer for body of response
 * @return http server response code 
 */
int http_get(const char* url, const char* proxy, int port, char** header, gchar** body, int* data_size)
{
    global_get++;    
    msa_debug("IN HTTP GET!!! %d\n", global_get);

    CURL* c;
    long code;
    char* type2;
    char* encoded_url = NULL;
    struct MemoryStruct chunk, chunk2;
    
    chunk.memory = NULL; 
    chunk.size = 0;

    chunk2.memory = NULL; 
    chunk2.size = 0;

    c = curl_easy_init();

    msa_debug("%s:%d:%s:url in http_get %s, proxy=%s, port=%d", __FILE__, __LINE__, __FUNCTION__, url, proxy, port);

    curl_easy_setopt(c, CURLOPT_URL, url);

    /* disable callbacks for working in multithreads */
    curl_easy_setopt(c, CURLOPT_NOSIGNAL, 1);

    /* disable using environment variables for proxy */
    //if (proxy == NULL)
	//curl_easy_setopt(c, CURLOPT_PROXY, "");
    //else
	curl_easy_setopt(c, CURLOPT_PROXY, proxy);
    curl_easy_setopt(c, CURLOPT_PROXYPORT, port);
    curl_easy_setopt(c, CURLOPT_TIMEOUT, 120);

    curl_easy_setopt(c, CURLOPT_WRITEFUNCTION, WriteMemoryCallback);
    curl_easy_setopt(c, CURLOPT_HEADERFUNCTION, WriteMemoryCallback);

    curl_easy_setopt(c, CURLOPT_WRITEDATA, (void *)&chunk);
    curl_easy_setopt(c, CURLOPT_WRITEHEADER, (void *)&chunk2);
    msa_debug("send request");

    if (curl_easy_perform(c) != 0) {
        g_warning("Can't perform the request\n");
        curl_easy_cleanup(c);
        return 400;
    }

    free(encoded_url);
    CURLcode ret = curl_easy_getinfo(c, CURLINFO_RESPONSE_CODE, &code );
    msa_debug("CODE %ld", code);

    //type2 = g_malloc(sizeof(char)*50);
    ret = curl_easy_getinfo(c, CURLINFO_CONTENT_TYPE, &type2 );
    msa_debug("TYPE %s", type2);

    if (header != NULL) {
        *header = g_strdup(chunk2.memory);
    }

    msa_debug("HEADER  %s\n", chunk2.memory);
    if(chunk2.memory)
        free(chunk2.memory);

    if (body != NULL) {
        //if (g_strcmp0(type2, "image/jpeg") == 0) {
        //    *body = g_base64_encode((guchar *)chunk.memory, chunk.size);
        //} else {
            *data_size = chunk.size;
    	    *body = chunk.memory;
        //}
    } else 
        if(chunk.memory)
            free(chunk.memory);

    //g_print("CONTENT  %s\n", *body);

    //g_free(type2);
    curl_easy_cleanup(c);

    return code;
}

/**
 * Perform HTTP POST request
 *
 * @param url - request
 * @param proxy - proxy address
 * @param port - proxy port
 * @param post_data - data to send
 * @param header - buffer for header of response
 * @param body - buffer for body of response
 * @param cookies - list of cookies from server
 * @return http server response code
 */

int http_post(const char* url, const char* proxy, int port, char* post_data, char** header, char** body, struct curl_slist** cookies)
{
    CURL* c;
    long code;
    CURLcode res;
    struct MemoryStruct chunk,chunk2;

    chunk.memory = NULL;
    chunk.size = 0;

    chunk2.memory = NULL;
    chunk2.size = 0;


    c = curl_easy_init();
    msa_debug("%s:%d:%s:url in http_post %s, proxy=%s:%d", __FILE__, __LINE__, __FUNCTION__, url, proxy, port);

    curl_easy_setopt(c, CURLOPT_URL, url);

    /* disable callbacks for working in multithreads */
    curl_easy_setopt(c, CURLOPT_NOSIGNAL, 1);

    /* disable using environment variables for proxy */
    //if (proxy == NULL)
	//curl_easy_setopt(c, CURLOPT_PROXY, "");
    //else
	curl_easy_setopt(c, CURLOPT_PROXY, proxy);
    curl_easy_setopt(c, CURLOPT_PROXYPORT, port);
    curl_easy_setopt(c, CURLOPT_TIMEOUT, 120);


    curl_easy_setopt(c, CURLOPT_POST, 1);
    curl_easy_setopt(c, CURLOPT_POSTFIELDS, post_data);
    curl_easy_setopt(c, CURLOPT_POSTFIELDSIZE, strlen(post_data));
    curl_easy_setopt(c, CURLOPT_FOLLOWLOCATION, 1);

    curl_easy_setopt(c, CURLOPT_COOKIEFILE, "");

    curl_easy_setopt(c, CURLOPT_WRITEFUNCTION, WriteMemoryCallback);
    curl_easy_setopt(c, CURLOPT_HEADERFUNCTION, WriteMemoryCallback);

    curl_easy_setopt(c, CURLOPT_WRITEDATA, (void *)&chunk);
    curl_easy_setopt(c, CURLOPT_WRITEHEADER, (void *)&chunk2);
    msa_debug("%s:%d:%s():send request", __FILE__, __LINE__, __FUNCTION__);
    
    CURLcode errornum = curl_easy_perform(c);
    if (errornum != 0) {
        g_warning("%s:%d:%s: Can't perform the request: %s", __FILE__, __LINE__, __FUNCTION__, curl_easy_strerror(errornum));
        curl_easy_cleanup(c);
        return 400;
    }
    
    res = curl_easy_getinfo(c, CURLINFO_COOKIELIST, &*cookies);
    if (res != CURLE_OK) {
        curl_easy_cleanup(c);
        curl_slist_free_all(*cookies);
        *cookies = NULL;
        return 400;
    }
    
    res = curl_easy_getinfo(c, CURLINFO_RESPONSE_CODE, &code );
    msa_debug("CODE %ld", code);
    msa_debug("HEADER\n  %s", chunk2.memory);
    //msa_debug("BODY====\n  %s\n=====BODY_END======", chunk.memory);
    
    
    if (header != NULL) {
        *header = g_strdup(chunk2.memory);
    }
    
    if(chunk2.memory)
        g_free(chunk2.memory);
    
    if (body != NULL) {
        *body = g_strdup(chunk.memory);
        if(chunk.memory) 
	    g_free(chunk.memory);
    }
    
    curl_easy_cleanup(c);
    return code;
}

int http_upload_file(const char* url, const char* proxy, const int port, char** header, gchar** body, struct ProfileData *cp, struct UpFile *file)
{
    CURL *curl;
    CURLcode res;
    long code;
    struct MemoryStruct chunk,chunk2;

    chunk.memory = NULL;
    chunk.size = 0;

    chunk2.memory = NULL;
    chunk2.size = 0;

    struct curl_httppost *formpost=NULL;
    struct curl_httppost *lastptr=NULL;
    struct curl_slist *headerlist=NULL;
    static const char buf[] = "Expect:";

    curl_global_init(CURL_GLOBAL_ALL);

    /* Fill in the file upload field */
    curl_formadd(&formpost,
                  &lastptr,
         CURLFORM_COPYNAME, file->name,
             CURLFORM_FILE, file->file_path,
         CURLFORM_FILENAME, file->file_name,
              CURLFORM_END);

    curl = curl_easy_init();
    /* initalize custom header list (stating that Expect: 100-continue is not
     *      wanted */
    headerlist = curl_slist_append(headerlist, buf);

    if(curl) {
     /* what URL that receives this POST */
        curl_easy_setopt(curl, CURLOPT_URL, url);

     /* only disable 100-continue header if explicitly requested */
        curl_easy_setopt(curl, CURLOPT_HTTPHEADER, headerlist);
        curl_easy_setopt(curl, CURLOPT_HTTPPOST, formpost);
        
        /* disable callbacks for working in multithreads */
	curl_easy_setopt(curl, CURLOPT_NOSIGNAL, 1);

	/* disable using environment variables for proxy */
	//if (proxy == NULL)
	//    curl_easy_setopt(curl, CURLOPT_PROXY, "");
	//else
	    curl_easy_setopt(curl, CURLOPT_PROXY, proxy);
        curl_easy_setopt(curl, CURLOPT_PROXYPORT, port);
        curl_easy_setopt(curl, CURLOPT_FOLLOWLOCATION, 1);

        curl_easy_setopt(curl, CURLOPT_WRITEFUNCTION, WriteMemoryCallback);
        curl_easy_setopt(curl, CURLOPT_HEADERFUNCTION, WriteMemoryCallback);

        curl_easy_setopt(curl, CURLOPT_WRITEDATA, (void *)&chunk);
        curl_easy_setopt(curl, CURLOPT_WRITEHEADER, (void *)&chunk2);

        res = curl_easy_perform(curl);

        res = curl_easy_getinfo(curl, CURLINFO_RESPONSE_CODE, &code );
        msa_debug("CODE %ld", code);
        msa_debug("HEADER\n  %s", chunk2.memory);        
        
        if (header != NULL) {
            *header = g_strdup(chunk2.memory);
        }
        
        if(chunk2.memory)
            g_free(chunk2.memory);
        
        if (body != NULL) {
            *body = g_strdup(chunk.memory);
        if(chunk.memory) 
	        g_free(chunk.memory);
        }
        /* always cleanup */
        curl_easy_cleanup(curl);
        /* then cleanup the formpost chain */
        curl_formfree(formpost);
        /* free slist */
        curl_slist_free_all (headerlist);
    }
   return code;
}

xmlNodePtr xpath_get_node(char* path, xmlDocPtr doc)
{
    xmlXPathObject *Obj = NULL;
    xmlNodePtr node = NULL;
    Obj = xpath(path, doc);
    if(Obj == NULL)
        return NULL;
    if(!xmlXPathNodeSetIsEmpty(Obj->nodesetval))
        node = xmlXPathNodeSetItem(Obj->nodesetval, 0);
    xmlXPathFreeObject(Obj);
    return node;
}

/**
 * @param req - xpath req
 * @param doc - xmlDocPtr  
 * @return xmlXPathObject*  
**/
xmlXPathObject* xpath(gchar* req, xmlDocPtr doc)
{
	xmlXPathObject *Obj;
	xmlXPathContextPtr Ctx;
	
	msa_debug("vk_xpath: %s", req);

    if (doc == NULL || req == NULL)
        return NULL;

	Ctx = xmlXPathNewContext(doc);
	
	if (Ctx == NULL) {
		msa_debug("vk_xpath: error wrong request format!");
		return NULL;	
	}

	Obj = xmlXPathEvalExpression(BAD_CAST req, Ctx);

	msa_debug("vk_xpath: end");

        xmlXPathFreeContext(Ctx);

	return Obj;

}

void upfile_remove(struct UpFile *file)
{
    if (file != NULL){
        if (file->name != NULL)
	    g_free(file->name);
        if (file->file_path != NULL)
	    g_free(file->file_path);
        if (file->file_name != NULL)
	    g_free(file->file_name);

        free(file);
        file = NULL;
    }
}

