/*
 * Copyright (C) 2009 Nokia Corporation.
 *
 * Contact: Marius Vollmer <marius.vollmer@nokia.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

#include <utime.h>
#include "thumbtest-func.h"
#include <QFile>
#include <QUrl>
#include <QDebug>
#include <QDir>
#include <QFile>
#include <QSignalSpy>
#include <QTimer>

using namespace MediaArt;
using namespace Thumbnails;

/*
 wait until spy will reach target number of signals; after that number is reached
 we can continue with verifying test's results. Thanks to that we do not need to wait always
 long time to be sure that tumbler replied - just give there finished spy and that should be
 working.

 maxTime is the maximum time we want to wait.

 NOTE: that macro could be placed only once in every block of the code and it is supposed
       to be put at max. only once in every test slot!
 NOTE: as a maxTime You need to put positive number
 */ 
#define WAIT_FOR_FINISH(spy,target,maxTime) static int _counter = 0; \
                                             while(spy.count()<target && _counter <= maxTime/50) { \
                                                  ++_counter; QTest::qWait(50); \
                                             }

// some defaults image paths etc.
#ifndef DATA_PREFIX
	#define DATA_PREFIX "/usr/share/libthumbnailer0-func-tests/"
#endif

QString imagePath       =  DATA_PREFIX "range.jpeg";
QUrl    imageUrl        =  QUrl("file://" + imagePath);
QString imageMime       =  QString("image/jpeg");

QString thumbnailImage =   DATA_PREFIX "thumbnail.jpeg";
QString thumbnailPathForImage = NULL;
QString brokenImagePath =  DATA_PREFIX "range-broken.jpeg";
QString brokenMovie =      DATA_PREFIX "broken-movie.avi";
QString thumbnailDirectory = QDir::home().absolutePath() + "/.thumbnails/normal/";

static QString stringToMD5(const QString& src);

/*filename variables for copy/remove/rename testing*/
QString myimageUrl =  QString("file://") + DATA_PREFIX "test.jpg";
QString myimagePath =   DATA_PREFIX "test.jpg";
QString myimageMime       =  QString("image/jpeg");
QString myThumbPath = thumbnailDirectory + stringToMD5(myimageUrl) +".jpeg";

static QString
stringToMD5(const QString& src) {
	QByteArray md5 = QCryptographicHash::hash ( src.toUtf8 (),
						    QCryptographicHash::Md5 );

	return QString(md5.toHex());
}


// set particular time modification for file
void
ThumbTest::setModificationTime(uint time, QString file) {
	if(file.isEmpty()) file = imageUrl.path();

	struct utimbuf t = {time, time};
	if(0 != utime (file.toAscii().constData(), &t)) {
		qDebug() << "Failed to change modification time for file:"
		         << file;
	}
}


// create thumbnail for particular URI
void
ThumbTest::createThumbnailForURI(QString uri)
{
	if(uri.isEmpty()) uri = imageUrl.toString();
	QString md5 = stringToMD5(uri);

	bool suc = true;
	suc &= QFile::copy(thumbnailImage, thumbnailDirectory + md5 + ".jpeg");
	if(!suc) {
		qDebug() << "Failed to create thumbnails: " << thumbnailDirectory + md5 + ".jpeg";
		return;
	}

	// set always the same modification time!
	QUrl tmp(uri);
	QFileInfo resourceFile(tmp.path());
	uint t = resourceFile.lastModified().toTime_t(); // get the modification time of the original one
	setModificationTime(t, thumbnailDirectory + md5 + ".jpeg");
}

// it makes particular thumbnail broken
void
ThumbTest::breakThumbnailForURI(QString uri)
{
	if(uri.isEmpty()) uri = imageUrl.toString();
	QString md5 = stringToMD5(uri);

	// making file 15 bytes long will for sure make it broken one ;)
	bool suc = true;
	suc &= QFile::resize (thumbnailDirectory + md5 + ".jpeg", 15);
	if(!suc) qDebug() << "Failed to broking thumbnails: " << thumbnailDirectory + md5 + ".jpeg";

	// set always the same modification time!
	QUrl tmp(uri);
	QFileInfo resourceFile(tmp.path());
	uint t = resourceFile.lastModified().toTime_t(); // get the modification time of the original one
	setModificationTime(t, thumbnailDirectory + md5 + ".jpeg");

}

// delete thumbnail for particular URI
void
ThumbTest::deleteThumbnailForURI(QString uri)
{
	if(uri.isEmpty()) uri = imageUrl.toString();
	QString md5 = stringToMD5(uri);

	QFile::remove(thumbnailDirectory + md5 + ".jpeg");
}

// clean the whole content of the cache
void
ThumbTest::deleteAllThumbnails()
{
	QDir dir;
	QStringList files;
	dir.setPath(thumbnailDirectory);
	files += dir.entryList(QDir::Files | QDir::Readable);

	foreach(QString file, files) {
		QFile::remove(thumbnailDirectory + file);
	}
}

// create thumbnail for particular URI
void
ThumbTest::createThumbnailWithTumbler(QList<QUrl>& images, QStringList& mimes)
{

        QSignalSpy error(thumbnailer, SIGNAL(error(QString, QUrl)));
        QSignalSpy stop(thumbnailer, SIGNAL(finished(int)));
        QSignalSpy ready(thumbnailer, SIGNAL(thumbnail(QUrl,QUrl,QPixmap,QString)));
        QSignalSpy readyDefault(thumbnailer, SIGNAL(defaultThumbnail(QUrl,QUrl,QPixmap,QString)));
        thumbnailer->request(images, mimes);
        WAIT_FOR_FINISH(stop, 1, 8000);
        if(1 == error.count()) {
                QString message = qvariant_cast<QString>(error.at(0).at(0));
                qDebug() << "Could not get the thumbnail because:" << message;
        }
        QCOMPARE(error.count(), 0);
        QCOMPARE(stop.count(), 1);
        QCOMPARE(readyDefault.count(), 1);
        QCOMPARE(ready.count(), 1);
}

// called once at the begging
void ThumbTest::initTestCase() {
	//FUNC_CALLED;
	qRegisterMetaType<MediaArt::Info>();
}
//called once after all tests
void ThumbTest::cleanupTestCase() {
	//FUNC_CALLED;
}

// called before every test start
void ThumbTest::init() {
	//FUNC_CALLED;
	thumbnailer = new Thumbnailer();
}
// called after every test is finished
void ThumbTest::cleanup() {
	//FUNC_CALLED;
	if(thumbnailer) delete thumbnailer; thumbnailer = NULL;
}

/****************************************************************************
 *     TESTS - THUMBNAILER
 ****************************************************************************/

/* --------------
   getSchedulers() - checks if anything is returned from that
 ---------------- */
void ThumbTest::getSchedulers() {
	QStringList list = Thumbnailer::getSchedulers ();
	// there should be at least: default and noop
	QVERIFY(list.size() >= 2);
	//qDebug() << "Supported schedulers are:" << list;
}

/* --------------
   getSchedulers() - checks if the first element is "default" (spec say that
                     it should be always there)
 ---------------- */
void ThumbTest::getSchedulers_checkingFirstItem() {
	QStringList list = Thumbnailer::getSchedulers ();
	QVERIFY(list.size() > 0);
	QCOMPARE(list.at(0), QString("default"));
}

/* --------------
   getSchedulers() - checks if all of returned items are non empty (are valid)
 ---------------- */
void ThumbTest::getSchedulers_nonEmptyItems() {
	QStringList list = Thumbnailer::getSchedulers ();
	foreach(QString item, list) {
		QVERIFY( !(item.isEmpty()) );
	}
}

/* --------------
   getSchedulers() - checks if there is a 'noop' scheduler
 ---------------- */
void ThumbTest::getSchedulers_noopItem() {
	QStringList list = Thumbnailer::getSchedulers ();
	QCOMPARE(Thumbnailer::noopScheduler, QString("noop"));
	QVERIFY(list.contains(Thumbnailer::noopScheduler));
}

/* --------------
   getFlavors() - checks if anything is returned from that
 ---------------- */
void ThumbTest::getFlavors() {
	QStringList list = Thumbnailer::getFlavors ();
	QVERIFY(list.size() > 0);
	//qDebug() << "Supported flavors are:" << list;
}

/* --------------
   getFlavors() - checks if all of returned items are non empty (are valid)
 ---------------- */
void ThumbTest::getFlavors_nonEmptyItems() {
	QStringList list = Thumbnailer::getFlavors ();
	foreach(QString item, list) {
		QVERIFY( !(item.isEmpty()) );
	}
}

/* --------------
   getFlavors() - checks if there is at least "normal" flavor
 ---------------- */
void ThumbTest::getFlavors_normalExists() {
	QStringList list = Thumbnailer::getFlavors ();
	bool normalExists = false;
	foreach(QString item, list) {
		if(QString("normal") == item) {
			normalExists = true;
			break;
		}
	}

	QVERIFY(normalExists);
}

void ThumbTest::getFlavorSize() {
	QSize size = Thumbnailer::getFlavorSize("normal");
	QVERIFY(true == size.isValid());
	QCOMPARE(size, QSize(100, 100));
}

void ThumbTest::getFlavorSize_nonExisting() {
	QSize size = Thumbnailer::getFlavorSize("foobar");
	QVERIFY(false == size.isValid());
}

/* --------------
   getSupported() - checks if getSupported function is returning anything
 ---------------- */
void ThumbTest::getSupported() {
	QMultiHash<QString, QString> supported = Thumbnailer::getSupported();
	QVERIFY(supported.size() != 0);
}

/* --------------
   getSupported_nonEmptySchemes() - checks if every scheme is a non empty string
 ---------------- */
void ThumbTest::getSupported_nonEmptySchemes() {
	QMultiHash<QString, QString> supported = Thumbnailer::getSupported();

	QMultiHash<QString, QString>::iterator i = supported.begin();
	while (i != supported.end()) {
		QVERIFY(  !(   i.key().isEmpty()  )  );
		++i;
	}
}

/* --------------
   getThumbnailPath()
 ---------------- */
void ThumbTest::getThumbnailPath_data() {
	QTest::addColumn<QUrl>("file");
	QTest::addColumn<QUrl>("thumbnail");
	QTest::addColumn<QString>("flavor");

	// test data converted to MD5 by: http://md5online.110mb.com/
	QTest::newRow("path #1")     << QUrl("file:///home/stranger/work/duithumbnailer/trunk/tests/thumbnailer-test-case/range.jpeg")
	                             << QUrl("file://" + QDir::homePath() + "/.thumbnails/normal/063ca94c0bbdf1c2358f6bdca52a8442.jpeg")
	                             << QString("normal");
	QTest::newRow("path #2")     << QUrl("file:///home/jens/photos/me.png")
	                             << QUrl("file://" + QDir::homePath() + "/.thumbnails/screen/c6ee772d9e49320e97ec29a7eb5b1697.jpeg")
	                             << QString("screen");
	QTest::newRow("path #3")     << QUrl("file:///home/jens/long/long/long/long/long/long/long/long/long/long/long/long/long/long/long/long/long/long/long/path/me.png")
	                             << QUrl("file://" + QDir::homePath() + "/.thumbnails/preview/fceac4cd6f2d4ef13ed8ea00df2412fc.jpeg")
	                             << QString("preview");
}
void ThumbTest::getThumbnailPath() {
    QFETCH(QUrl, file);
    QFETCH(QUrl, thumbnail);
    QFETCH(QString, flavor);

    QCOMPARE(Thumbnailer::getThumbnailPath(file, flavor), thumbnail);
}



/**
*	Testing move request
*
*	image for testing is test.jpg, a copy from image_file.jpg
**/

void ThumbTest::move_request() {


        /*moved filename*/
        QString myimageUrlMoved =  QString("file://") + DATA_PREFIX "test_moved.jpg";
        QString myimagePathMoved =   DATA_PREFIX "test_moved.jpg";
        QString md5Moved = stringToMD5(myimageUrlMoved);
		QString myThumbPathMoved= thumbnailDirectory + md5Moved +".jpeg";

        /*create physical file for testing*/
        if(!QFile::exists(myimagePath))
                QFile::copy(imagePath, myimagePath);

        QFile imageFile(myimagePath);

        /*Cleanup old remainings*/
        QFile::remove(myThumbPathMoved);

        /*Create thumbnail for testing*/

        //qDebug() << "Checking if " << myThumbPath << "exists";
        if(!QFile::exists(myThumbPath))
        {
                //qDebug() << "Nope, so create it via tumbler";
                QList<QUrl> images; images << myimageUrl;
                QStringList mimes;  mimes  << myimageMime;
                createThumbnailWithTumbler(images, mimes);
                QCOMPARE(QFile::exists(myThumbPath), true);
                //qDebug() << "Ok, " << myThumbPath << "exists";
        }

        /*rename file physically*/
        imageFile.rename(myimagePathMoved);

        /*call move reaqest*/
        QList<QUrl> from; from << QUrl(myimageUrl);
        QList<QUrl> to; to << QUrl(myimageUrlMoved);
        QSignalSpy ready(thumbnailer, SIGNAL(thumbnail(QUrl,QUrl,QPixmap,QString)));
        thumbnailer->move(from, to);
        WAIT_FOR_FINISH(ready, 1, 1000);

        /*check if new thumbnail was created and the previous deleted */
        //qDebug() << "Checking if " << myThumbPathMoved << "exists";
        QCOMPARE(QFile::exists(myThumbPathMoved), true);
        QCOMPARE(QFile::exists(myThumbPath), false);

        /*cleanup*/
        QFile::remove(myimagePathMoved);
        QFile::remove(myThumbPathMoved);
        QFile::remove(myimagePath);

 }



/**
*	Testing copy request
*
*	image for testing is test.jpg, a copy from image_file.jpg
**/

void ThumbTest::copy_request() {


        /*create physical file for testing*/
        if(!QFile::exists(myimagePath))
                QFile::copy(imagePath, myimagePath);

        /*copied filename*/
        QString myimageUrlCopied =  QString("file://") + DATA_PREFIX "test_copied.jpg";
        QString myimagePathCopied =   DATA_PREFIX "test_copied.jpg";
		QString myThumbPathCopied= thumbnailDirectory + stringToMD5(myimageUrlCopied) +".jpeg";

        QFile imageFile(myimagePath);

        /*create thumbnail*/
        //qDebug() << "Checking if " << myThumbPath << "exists";

        if(!QFile::exists(myThumbPath))
        {
                //qDebug() << "Nope, so create it via tumbler";
                QList<QUrl> images; images << myimageUrl;
                QStringList mimes;  mimes  << myimageMime;
                createThumbnailWithTumbler(images, mimes);
                QCOMPARE(QFile::exists(myThumbPath), true);
                //qDebug() << "Ok, " << myThumbPath << "exists";
        }

        /*cleanup any old remainings*/
        QFile::remove(myThumbPathCopied);

        /*copy a file for testing*/
        if(!QFile::exists(myimagePathCopied))
        QFile::copy(myimagePath,myimagePathCopied);

        /*request copy command*/
        QList<QUrl> from; from << QUrl(myimageUrl);
        QList<QUrl> to; to << QUrl(myimageUrlCopied);
        QSignalSpy ready(thumbnailer, SIGNAL(thumbnail(QUrl,QUrl,QPixmap,QString)));
        thumbnailer->copy(from, to);
        WAIT_FOR_FINISH(ready, 1, 1000);



        /*Check if both thumnails for file and copied file exists*/

        //qDebug() << "Checking if " << myThumbPathCopied << "exists";
        QCOMPARE(QFile::exists(myThumbPathCopied), true);
        QCOMPARE(QFile::exists(myThumbPath), true);
        //qDebug() << "Ok, " << myThumbPathCopied << "exists";

        /*cleanup*/
        QFile::remove(myThumbPath);
        QFile::remove(myThumbPathCopied);
        QFile::remove(myimagePath);
        QFile::remove(myimagePathCopied);
}



/**
*	Testing remove request
*
*	image for testing is test.jpg, a copy from image_file.jpg
**/

void ThumbTest::remove_request() {

        /*create physical file for testing*/
        if(!QFile::exists(myimagePath))
                QFile::copy(imagePath, myimagePath);

        QFile imageFile(myimagePath);

        /*Create thumbnail for testing*/
        //qDebug() << "Checking if " << myThumbPath << "exists";

        if(!QFile::exists(myThumbPath))
        {
                //qDebug() << "Nope, so create it via tumbler";
                QList<QUrl> images; images << myimageUrl;
                QStringList mimes;  mimes  << myimageMime;
                createThumbnailWithTumbler(images, mimes);
                QCOMPARE(QFile::exists(myThumbPath), true);
                //qDebug() << "Ok, " << myThumbPath << "exists";
        }

        /*remove file physically*/
        imageFile.remove();

        /*call remove reaqest*/
        QList<QUrl> removelist; removelist << QUrl(myimageUrl);
        QSignalSpy readyRemove(thumbnailer, SIGNAL(thumbnail(QUrl,QUrl,QPixmap,QString)));
        thumbnailer->remove(removelist);
        WAIT_FOR_FINISH(readyRemove, 1, 1000);

        /*check if the thumbnail was deleted */
        //qDebug() << "Checking if " << myThumbPathMoved << "exists";
        QCOMPARE(QFile::exists(myThumbPath), false);
        //qDebug() << "Ok, " << myThumbPathMoved << "exists";

        /*cleanup*/
        QFile::remove(myimagePath);

 }

/* --------------
   request_oneExistingFile() - requesting one single thumbnail which already exists
 ---------------- */
void ThumbTest::request_oneExistingThumbnail() {
	deleteThumbnailForURI();
	createThumbnailForURI();

	QSignalSpy ready(thumbnailer, SIGNAL(thumbnail(QUrl,QUrl,QPixmap,QString)));
	QSignalSpy readyDefault(thumbnailer, SIGNAL(defaultThumbnail(QUrl,QUrl,QPixmap,QString)));
	QSignalSpy error(thumbnailer, SIGNAL(error(QString, QUrl)));
	QSignalSpy stop(thumbnailer, SIGNAL(finished(int)));

	QList<QUrl> images; images << imageUrl;
	QStringList mimes;  mimes  << imageMime;

	bool noDBusTraffic = thumbnailer->request(images, mimes);

	WAIT_FOR_FINISH(stop, 1, 50);
	QCOMPARE(noDBusTraffic, true);
	QCOMPARE(error.count(), 0);
	QCOMPARE(readyDefault.count(), 0);
	QCOMPARE(ready.count(), 1);
	QCOMPARE(stop.count(), 1);

	deleteThumbnailForURI();
}
/* --------------
   request_oneNonExistingFile() - requesting one single thumbnail which doesn't exist
 ---------------- */
void ThumbTest::request_oneNonExistingThumbnail() {
	deleteThumbnailForURI();

	QSignalSpy ready(thumbnailer, SIGNAL(thumbnail(QUrl,QUrl,QPixmap,QString)));
	QSignalSpy readyDefault(thumbnailer, SIGNAL(defaultThumbnail(QUrl,QUrl,QPixmap,QString)));
	QSignalSpy error(thumbnailer, SIGNAL(error(QString, QUrl)));
	QSignalSpy stop(thumbnailer, SIGNAL(finished(int)));

	QList<QUrl> images; images << imageUrl;
	QStringList mimes;  mimes  << imageMime;

	bool noDBusTraffic = thumbnailer->request(images, mimes);
	QCOMPARE(noDBusTraffic, false);

	WAIT_FOR_FINISH(stop, 1, 3000);
	if(1 == error.count()) {
		QString message = qvariant_cast<QString>(error.at(0).at(0));
		qDebug() << "Could not get the thumbnail because:" << message;
	}
	QCOMPARE(readyDefault.count(), 1);
	QCOMPARE(error.count(), 0);
	QCOMPARE(ready.count(), 1);
	QCOMPARE(stop.count(), 1);

	deleteThumbnailForURI();
}

/* --------------
   request_oneVideoFile() - requesting one single thumbnail from video file
 ---------------- */
void ThumbTest::request_oneVideoFile() {
	QString uri1 = QString("file://") + DATA_PREFIX + QString("short.avi");
	deleteThumbnailForURI(uri1);

	QSignalSpy ready(thumbnailer, SIGNAL(thumbnail(QUrl,QUrl,QPixmap,QString)));
	QSignalSpy readyDefault(thumbnailer, SIGNAL(defaultThumbnail(QUrl,QUrl,QPixmap,QString)));
	QSignalSpy error(thumbnailer, SIGNAL(error(QString, QUrl)));
	QSignalSpy stop(thumbnailer, SIGNAL(finished(int)));
	QSignalSpy dequeued(thumbnailer, SIGNAL(dequeued(QUrl)));

	QList<QUrl> images1; images1 << QUrl(uri1);
	QStringList mimes1;  mimes1  << QString("video/x-msvideo");

	bool noDBusTraffic = thumbnailer->request(images1, mimes1);
	QCOMPARE(noDBusTraffic, false);

	WAIT_FOR_FINISH(stop, 1, 8000);

	// it should work with current video thumbnailer available
	QCOMPARE(error.count(), 0);
	QCOMPARE(stop.count(), 1);
	QCOMPARE(readyDefault.count(), 1);
	QCOMPARE(dequeued.count(), 0);
	QCOMPARE(ready.count(), 1);

	deleteThumbnailForURI(uri1);
}

void
ThumbTest::request_oneNonExistingThumbnail_emptyCache_bug146404() {
	deleteAllThumbnails();

	QSignalSpy ready(thumbnailer, SIGNAL(thumbnail(QUrl,QUrl,QPixmap,QString)));
	QSignalSpy readyDefault(thumbnailer, SIGNAL(defaultThumbnail(QUrl,QUrl,QPixmap,QString)));
	QSignalSpy error(thumbnailer, SIGNAL(error(QString, QUrl)));
	QSignalSpy stop(thumbnailer, SIGNAL(finished(int)));

	QList<QUrl> images; images << imageUrl;
	QStringList mimes;  mimes  << imageMime;

	bool noDBusTraffic = thumbnailer->request(images, mimes);
	QCOMPARE(noDBusTraffic, false);

	WAIT_FOR_FINISH(stop, 1, 3000);
	if(1 == error.count()) {
		QString message = qvariant_cast<QString>(error.at(0).at(0));
		qDebug() << "Could not get the thumbnail because:" << message;
	}
	QCOMPARE(readyDefault.count(), 1);
	QCOMPARE(error.count(), 0);
	QCOMPARE(ready.count(), 1);
	QCOMPARE(stop.count(), 1);

	deleteAllThumbnails();
}

void
ThumbTest::request_oneNonPossibleThumbnail_emptyCache_bug146404() {
	deleteAllThumbnails();

	QSignalSpy ready(thumbnailer, SIGNAL(thumbnail(QUrl,QUrl,QPixmap,QString)));
	QSignalSpy readyDefault(thumbnailer, SIGNAL(defaultThumbnail(QUrl,QUrl,QPixmap,QString)));
	QSignalSpy error(thumbnailer, SIGNAL(error(QString, QUrl)));
	QSignalSpy stop(thumbnailer, SIGNAL(finished(int)));

	QList<QUrl> images; images << QUrl("foo://bar.non.existing.source/path.jpeg");
	QStringList mimes;  mimes  << "image/jpeg";

	bool noDBusTraffic = thumbnailer->request(images, mimes);
	QCOMPARE(noDBusTraffic, false);

	WAIT_FOR_FINISH(stop, 1, 3000);
	if(1 == error.count()) {
		QString message = qvariant_cast<QString>(error.at(0).at(0));
		//qDebug() << "Could not get the thumbnail because:" << message;
	}
	QCOMPARE(readyDefault.count(), 1);
	QCOMPARE(error.count(), 1);
	QCOMPARE(ready.count(), 0);
	QCOMPARE(stop.count(), 1);

	deleteAllThumbnails();
}

/* --------------
   request_noopScheduler() - check if 'noop' scheduler is working properly
 ---------------- */
void
ThumbTest::request_noopScheduler() {
	deleteThumbnailForURI();
	createThumbnailForURI();

	QSignalSpy ready(thumbnailer, SIGNAL(thumbnail(QUrl,QUrl,QPixmap,QString)));
	QSignalSpy readyDefault(thumbnailer, SIGNAL(defaultThumbnail(QUrl,QUrl,QPixmap,QString)));
	QSignalSpy error(thumbnailer, SIGNAL(error(QString, QUrl)));
	QSignalSpy stop(thumbnailer, SIGNAL(finished(int)));

	QList<QUrl> images; images << imageUrl;
	QStringList mimes;  mimes  << imageMime;

	// put there als some weird url for source which is for sure not thumbnailed yet
	images << QUrl("http://remote/source.jpg");
	mimes  << imageMime;

	bool noDBusTraffic = thumbnailer->request(images, mimes, false,
	                                          QString(), Thumbnailer::noopScheduler);
	QCOMPARE(noDBusTraffic, true);

	// this should be fast - no need for long wait
	WAIT_FOR_FINISH(stop, 1, 1200);
	if(0 < error.count()) {
		QString message = qvariant_cast<QString>(error.at(0).at(0));
		//qDebug() << "Could not get the thumbnail because:" << message;
	}
	QCOMPARE(readyDefault.count(), 0);
	QCOMPARE(error.count(), 1);
	QCOMPARE(ready.count(), 1);
	QCOMPARE(stop.count(), 1);

	deleteThumbnailForURI();
}


/* --------------
   request_oneNonExistingFile() - requesting one single thumbnail for file which is broken
 ---------------- */
void ThumbTest::request_oneNonExistingThumbnailForBrokenFile() {
	deleteThumbnailForURI("file://" + brokenImagePath);

	QSignalSpy ready(thumbnailer, SIGNAL(thumbnail(QUrl,QUrl,QPixmap,QString)));
	QSignalSpy readyDefault(thumbnailer, SIGNAL(defaultThumbnail(QUrl,QUrl,QPixmap,QString)));
	QSignalSpy error(thumbnailer, SIGNAL(error(QString, QUrl)));
	QSignalSpy stop(thumbnailer, SIGNAL(finished(int)));

	QList<QUrl> images; images << QUrl("file://" + brokenImagePath);
	QStringList mimes;  mimes  << imageMime;

	bool noDBusTraffic = thumbnailer->request(images, mimes);
	QCOMPARE(noDBusTraffic, false);

	WAIT_FOR_FINISH(stop, 1, 3000);
	if(1 == error.count()) {
		QString message = qvariant_cast<QString>(error.at(0).at(0));
		qDebug() << "OK: Could not get the thumbnail for broken image. Error message:\n" << message;
	}
	if(1 == ready.count()) {
		QUrl uri = qvariant_cast<QUrl>(ready.at(0).at(0));
		qDebug() << "NOK: Ready signal for broken URI: " << uri.toString();
		qDebug() << "Potential Thumbnail path is: " << qvariant_cast<QUrl>(ready.at(0).at(1)).toString();
	}

	QCOMPARE(readyDefault.count(), 1);
	QCOMPARE(ready.count(), 0);
	QCOMPARE(error.count(), 1);
	QCOMPARE(stop.count(), 1);

	deleteThumbnailForURI("file://" + brokenImagePath);
}


/****************************************************************************
 *     TESTS - THUMBNAILER (BUGS)
 ****************************************************************************/
/* --------------
   bug_125965_twoRequestWithError_noCancel() - check issue with two requests
   which are giving error (there should not be any SEG fault). Cases when
   autoCancel is in false mode (default) and in true mode
 ---------------- */
void ThumbTest::bug_125965_twoRequestWithError_data()
{
	QTest::addColumn<bool>("autoCancel");

	QTest::newRow("autoCancel=false") << false;
	QTest::newRow("autoCancel=true")  << true;
}
void ThumbTest::bug_125965_twoRequestWithError()
{
	QFETCH(bool, autoCancel);
	QString uri1 = QString("file://") + DATA_PREFIX + QString("broken-short.avi");
	QString uri2 = QString("file://") + DATA_PREFIX + QString("broken-leike.3gp");
	deleteThumbnailForURI(uri1);
	deleteThumbnailForURI(uri2);

	QSignalSpy ready(thumbnailer, SIGNAL(thumbnail(QUrl,QUrl,QPixmap,QString)));
	QSignalSpy readyDefault(thumbnailer, SIGNAL(defaultThumbnail(QUrl,QUrl,QPixmap,QString)));
	QSignalSpy error(thumbnailer, SIGNAL(error(QString, QUrl)));
	QSignalSpy stop(thumbnailer, SIGNAL(finished(int)));
	QSignalSpy dequeued(thumbnailer, SIGNAL(dequeued(QUrl)));
	// below signal is available only for test unit purpose!!!!
	QSignalSpy remoteStarted((QObject*)thumbnailer->getPrivateData(), SIGNAL(remoteStarted(uint)));

	QList<QUrl> images1; images1 << QUrl(uri1);
	QList<QUrl> images2; images2 << QUrl(uri2);
	QStringList mimes1;  mimes1  << QString("video/x-msvideo");
	QStringList mimes2;  mimes2  << QString("video/3gpp");

	bool noDBusTraffic  = thumbnailer->request(images1, mimes1, true, QString(), QString(), autoCancel);
	/* we will wait for Started signal and make another request only then - to be sure that
	   the first request was started - make as small wait interval as possible to react as fast
	   as possible */
	int counter = 0;
	while(0 == remoteStarted.count() && counter < 200) {
		++counter; QTest::qWait(10);
	}
	// make the second request as fast as possible
	noDBusTraffic &= thumbnailer->request(images2, mimes2, true, QString(), QString(), autoCancel);
	QVERIFY(2000 != counter); // make sure that the first request started
	QCOMPARE(noDBusTraffic, false); // and that request was send to tumbler

	WAIT_FOR_FINISH(stop, 2, 30000);

	// the first request probably will not manage to start at all - it will not get reply from
	// tumbler before 
	QCOMPARE(stop.count(), 2);
	// still we cannot be sure if the first request reached error signal or was dequeued
	// we need to sum those to signals and check if they count to two
	QCOMPARE(dequeued.count() + error.count(), 2);
	QCOMPARE(readyDefault.count(), 2);
	QCOMPARE(ready.count(), 0);

	deleteThumbnailForURI(uri1);
	deleteThumbnailForURI(uri2);
}
// the same as above, but this time no wait for the first request to started - just two request in row
// the first one does not reach tumbler in that case
void ThumbTest::bug_125965_twoRequestWithError_noWait_data()
{
	QTest::addColumn<bool>("autoCancel");

	QTest::newRow("autoCancel=false") << false;
	QTest::newRow("autoCancel=true")  << true;
}
void ThumbTest::bug_125965_twoRequestWithError_noWait()
{
	QFETCH(bool, autoCancel);
	QString uri1 = QString("file://") + DATA_PREFIX + QString("broken-movie.avi");
	QString uri2 = QString("file://") + DATA_PREFIX + QString("broken-leike.3gp");
	deleteThumbnailForURI(uri1);
	deleteThumbnailForURI(uri2);

	QSignalSpy ready(thumbnailer, SIGNAL(thumbnail(QUrl,QUrl,QPixmap,QString)));
	QSignalSpy readyDefault(thumbnailer, SIGNAL(defaultThumbnail(QUrl,QUrl,QPixmap,QString)));
	QSignalSpy error(thumbnailer, SIGNAL(error(QString, QUrl)));
	QSignalSpy stop(thumbnailer, SIGNAL(finished(int)));
	QSignalSpy dequeued(thumbnailer, SIGNAL(dequeued(QUrl)));

	QList<QUrl> images1; images1 << QUrl(uri1);
	QList<QUrl> images2; images2 << QUrl(uri2);
	QStringList mimes1;  mimes1  << QString("video/x-msvideo");
	QStringList mimes2;  mimes2  << QString("video/3gpp");

	bool noDBusTraffic  = thumbnailer->request(images1, mimes1, true, QString(), QString(), autoCancel);
	     noDBusTraffic &= thumbnailer->request(images2, mimes2, true, QString(), QString(), autoCancel);
	QCOMPARE(noDBusTraffic, false);

	WAIT_FOR_FINISH(stop, 2, 30000);

	// the first request probably will not manage to start at all - it will not get reply from
	// tumbler before the second one will cancel it
	QCOMPARE(stop.count(), 2);
	QCOMPARE(readyDefault.count(), 2);
	if(autoCancel) {
		QCOMPARE(dequeued.count(), 1);
		QCOMPARE(error.count(), 1);
	}
	else {
		QCOMPARE(dequeued.count(), 0);
		QCOMPARE(error.count(), 2);
	}
	QCOMPARE(ready.count(), 0);

	deleteThumbnailForURI(uri1);
	deleteThumbnailForURI(uri2);
}


void ThumbTest::bug_143261_multipleRequests() {
	const int testRunNumber = 100;
	QStringList uris;
	QHash<QString, int> counter;
	for(int i=1; i <= 8; ++i) {
		QString uri = QString("file://") + DATA_PREFIX + QString::number(i) + QString(".png");
		uris << uri;
	}

	QSignalSpy ready(thumbnailer, SIGNAL(thumbnail(QUrl,QUrl,QPixmap,QString)));
	QSignalSpy error(thumbnailer, SIGNAL(error(QString, QUrl)));
	QSignalSpy stop(thumbnailer, SIGNAL(finished(int)));

	// run test several times
	for(int i=0; i < testRunNumber; ++i) {
		// remove all thumbnails
		foreach(QString uri, uris) deleteThumbnailForURI(uri);

		// separate request for each of the file
		foreach(QString uri, uris) {
			// preapre list for request
			QList<QUrl> listOfUris; listOfUris << QUrl(uri);
			QStringList listOfMimes; listOfMimes << QString("image/png");

			thumbnailer->request (listOfUris, listOfMimes);
		}
		WAIT_FOR_FINISH(stop, 8, 50000);

		QCOMPARE(stop.count(), 8);
		QCOMPARE(error.count() + ready.count(), 8);
		/*QEXPECT_FAIL("",
		             "Some problems with DBus prevents sometimes Tumbler from sending all ready signals",
		             Continue);*/
		QCOMPARE(ready.count(), 8);
		// increase number of particular uris in the returned signals
		for(int k=0; k < ready.count(); ++k) {
			QUrl uri1 = qvariant_cast<QUrl>(ready.at(k).at(0)); // source
			QUrl uri2 = qvariant_cast<QUrl>(ready.at(k).at(1)); // thumbnail uri
			counter[uri1.toString()]++;
			counter[uri2.toString()]++;
		}
		// go also through errors list
		for(int k=0; k < error.count(); ++k) {
			QUrl uri = qvariant_cast<QUrl>(error.at(k).at(1)); // source
			counter[uri.toString()]++;
		}
		// clear signal spies
		ready.clear();
		stop.clear();
		error.clear();
	}

	// now lets iterate throug the whole counter list
	// every element should have exactly testRunNumber value
	QHashIterator<QString, int> i(counter);
	while (i.hasNext()) {
		i.next();
		QCOMPARE(i.value(), testRunNumber);
	}

	// and there should be 16 elements: 8 sources and 8 thumbnails paths
	QCOMPARE(counter.size(), 16);

	// remove all thumbnails after the whole test
	foreach(QString uri, uris) deleteThumbnailForURI(uri);
}

void ThumbTest::bug_163333_thumbnailerCrashAfterDelete() {
	// prepare data
	QString uri = QString("file://") + DATA_PREFIX + QString("short.avi");
	deleteThumbnailForURI(uri);

	QList<QUrl> urls;
	QStringList mimes;
	for(int i=0; i < 5; ++i) {
		urls << QUrl(uri);
		mimes << QString("video/x-msvideo");
	}

	QSignalSpy stop(thumbnailer, SIGNAL(finished(int)));

	// request video (couple of times to make lots of requests)
	for(int i=0; i< 100; ++i) {
	     thumbnailer->request(urls, mimes);
	}

	// wait only some miliseconds (before request ends)
	QTest::qWait(100);

	// call destructor of thumbnailer (and hope there will not be any crash ;) )
	delete thumbnailer;
	thumbnailer = 0;
}


/****************************************************************************
 *     TESTS - DUI MEDIA ART - see unit test package (nothing to test from
 *	point of view for MediaArt::Info classes )
 ****************************************************************************/
/****************************************************************************
 *     TESTS - DUI MEDIA ART THUMBNAILER
 ****************************************************************************/
/* --------------
   request_oneExistingThumbnailMediaArt() - make a request for media-art's thumbnail.
     We make sure that thumbnails exists before that call, so there should not be
     any dbus traffic and ready signal should be received
 ---------------- */
void ThumbTest::request_oneExistingThumbnailMediaArt() {
	MediaArt::Info album = Album("Queen", "Greatest Hits vol.1");
	album.setMediaArtImage(QPixmap(QSize(51,52)));
	deleteThumbnailForURI(album.potentialPath().toString());
	createThumbnailForURI(album.potentialPath().toString());

	QSignalSpy ready(thumbnailer, SIGNAL(thumbnail(QUrl,QUrl,QPixmap,QString)));
	QSignalSpy readyDefault(thumbnailer, SIGNAL(defaultThumbnail(QUrl,QUrl,QPixmap,QString)));
	QSignalSpy error(thumbnailer, SIGNAL(error(QString, QUrl)));
	QSignalSpy m_ready(thumbnailer, SIGNAL(thumbnail(MediaArt::Info,QUrl,QPixmap,QString)));
	QSignalSpy m_readyDefault(thumbnailer, SIGNAL(defaultThumbnail(MediaArt::Info,QUrl,QPixmap,QString)));
	QSignalSpy m_error(thumbnailer, SIGNAL(error(QString, MediaArt::Info)));
	QSignalSpy stop(thumbnailer, SIGNAL(finished(int)));

	QList<MediaArt::Info> media; media << album;

	bool noDBusTraffic = thumbnailer->request(media);

	WAIT_FOR_FINISH(stop, 1, 500);
	QCOMPARE(noDBusTraffic, true);
	QCOMPARE(error.count(), 0);
	QCOMPARE(m_error.count(), 0);
	QCOMPARE(readyDefault.count(), 0);
	QCOMPARE(m_readyDefault.count(), 0);
	QCOMPARE(ready.count(), 0);
	QCOMPARE(m_ready.count(), 1);
	QCOMPARE(stop.count(), 1);

	deleteThumbnailForURI(album.potentialPath().toString());
}

void ThumbTest::request_oneNonExistingThumbnailMediaArt() {
	MediaArt::Info album = Album("Queen", "Greatest Hits vol.1");
	album.setMediaArtImage(QPixmap(QSize(51,52)));
	deleteThumbnailForURI(album.potentialPath().toString());

	QSignalSpy ready(thumbnailer, SIGNAL(thumbnail(QUrl,QUrl,QPixmap,QString)));
	QSignalSpy readyDefault(thumbnailer, SIGNAL(defaultThumbnail(QUrl,QUrl,QPixmap,QString)));
	QSignalSpy error(thumbnailer, SIGNAL(error(QString, QUrl)));
	QSignalSpy m_ready(thumbnailer, SIGNAL(thumbnail(MediaArt::Info,QUrl,QPixmap,QString)));
	QSignalSpy m_readyDefault(thumbnailer, SIGNAL(defaultThumbnail(MediaArt::Info,QUrl,QPixmap,QString)));
	QSignalSpy m_error(thumbnailer, SIGNAL(error(QString, MediaArt::Info)));
	QSignalSpy stop(thumbnailer, SIGNAL(finished(int)));

	QList<MediaArt::Info> media; media << album;

	bool noDBusTraffic = thumbnailer->request(media);

	WAIT_FOR_FINISH(stop, 1, 500);
	QCOMPARE(noDBusTraffic, false);
	QCOMPARE(error.count(), 0);
	QCOMPARE(m_error.count(), 0);
	QCOMPARE(readyDefault.count(), 0);
	QCOMPARE(m_readyDefault.count(), 1);
	QCOMPARE(ready.count(), 0);
	QCOMPARE(m_ready.count(), 1);
	QCOMPARE(stop.count(), 1);

	deleteThumbnailForURI(album.potentialPath().toString());
}

void ThumbTest::request_oneNonExistingThumbnailNonExistingMediaArt() {
	MediaArt::Info album = Album("Queen", "Greatest Hits vol.1");
	album.remove();
	deleteThumbnailForURI(album.potentialPath().toString());

	QSignalSpy ready(thumbnailer, SIGNAL(thumbnail(QUrl,QUrl,QPixmap,QString)));
	QSignalSpy readyDefault(thumbnailer, SIGNAL(defaultThumbnail(QUrl,QUrl,QPixmap,QString)));
	QSignalSpy error(thumbnailer, SIGNAL(error(QString, QUrl)));
	QSignalSpy m_ready(thumbnailer, SIGNAL(thumbnail(MediaArt::Info,QUrl,QPixmap,QString)));
	QSignalSpy m_readyDefault(thumbnailer, SIGNAL(defaultThumbnail(MediaArt::Info,QUrl,QPixmap,QString)));
	QSignalSpy m_error(thumbnailer, SIGNAL(error(QString, MediaArt::Info)));
	QSignalSpy stop(thumbnailer, SIGNAL(finished(int)));

	QList<MediaArt::Info> media; media << album;

	bool noDBusTraffic = thumbnailer->request(media);

	WAIT_FOR_FINISH(stop, 1, 500);
	QCOMPARE(noDBusTraffic, true); //no source - no need to request
	QCOMPARE(error.count(), 0);
	QCOMPARE(m_error.count(), 1);
	QCOMPARE(readyDefault.count(), 0);
	QCOMPARE(m_readyDefault.count(), 1);
	QCOMPARE(ready.count(), 0);
	QCOMPARE(m_ready.count(), 0);
	QCOMPARE(stop.count(), 1);

	deleteThumbnailForURI(album.potentialPath().toString());
}

void ThumbTest::request_coupleMixedThumbnailsOfMixedMediaArts() {
	// media arts we will be requesting
	MediaArt::Info album   = Album ("Queen", "Greatest Hits vol.2");
	MediaArt::Info artist  = Artist("Queen", "Greatest Hits vol.2");
	MediaArt::Info track   = Track("Queen", "Greatest Hits vol.2", "I want to break free");
	MediaArt::Info podcast = Podcast("Funky House London");
	MediaArt::Info radio   = Radio("Radio ga-ga");

	// test part: make sure that some of them exists, and some not
	  album.setMediaArtImage(QPixmap(QSize(51,51)));
	  track.setMediaArtImage(QPixmap(QSize(51,52)));
	  radio.setMediaArtImage(QPixmap(QSize(51,53)));
	 artist.setMediaArtImage(QPixmap(QSize(51,54)));
	podcast.remove(); // this one does not exist

	// test part: make sure that some media arts have thmbnails and some not
	createThumbnailForURI(album.potentialPath().toString());
	createThumbnailForURI(artist.potentialPath().toString());
	deleteThumbnailForURI(radio.potentialPath().toString());
	deleteThumbnailForURI(track.potentialPath().toString());
	deleteThumbnailForURI(podcast.potentialPath().toString());

	/* now we have:
	4 existing media arts:
		2 of them are arleady thumbnailed
		2 of them do not have thumbnail created
	1 empty media art
	It should give us:
	  4 ready signals
	  3 default signals
	  1 error signal
	*/

	// real code - test case
	// prepare list of media arts to be requested
	QList<MediaArt::Info> media;
	media << album << artist << track << podcast << radio;

	// connect signals - here we are using SignalSpies
	QSignalSpy ready(thumbnailer, SIGNAL(thumbnail(QUrl,QUrl,QPixmap,QString)));
	QSignalSpy readyDefault(thumbnailer, SIGNAL(defaultThumbnail(QUrl,QUrl,QPixmap,QString)));
	QSignalSpy error(thumbnailer, SIGNAL(error(QString, QUrl)));
	QSignalSpy m_ready(thumbnailer, SIGNAL(thumbnail(MediaArt::Info,QUrl,QPixmap,QString)));
	QSignalSpy m_readyDefault(thumbnailer, SIGNAL(defaultThumbnail(MediaArt::Info,QUrl,QPixmap,QString)));
	QSignalSpy m_error(thumbnailer, SIGNAL(error(QString, MediaArt::Info)));
	QSignalSpy stop(thumbnailer, SIGNAL(finished(int)));

	// make a requests - just as simple as requesting standard thumbnails
	thumbnailer->request(media);

	// now You should receive all signals - lets check if everything was as we are expecting
	WAIT_FOR_FINISH(stop, 1, 3000);
	QCOMPARE(error.count(), 0);
	QCOMPARE(m_error.count(), 1);
	QCOMPARE(readyDefault.count(), 0);
	QCOMPARE(m_readyDefault.count(), 3);
	QCOMPARE(ready.count(), 0);
	QCOMPARE(m_ready.count(), 4);
	QCOMPARE(stop.count(), 1);
}

// execute all tests
QTEST_MAIN(ThumbTest)

