/*
 * Copyright (C) 2009 Nokia Corporation.
 *
 * Contact: Marius Vollmer <marius.vollmer@nokia.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

#include "thumbnailer_p.h"
#include "dbus-services.h"
#include <QtCore/QFileInfo>
#include <QSettings>
#include <QByteArray>
#include <QCryptographicHash>
#include <QMutex>

/* **************************************************************************
 *      LOCAL (STATIC) FUNCTIONS:
 * **************************************************************************/
static inline QPixmap
_getPixmap(QUrl uri, bool sendPixmap) {
	if(!sendPixmap) return QPixmap();
	return QPixmap(uri.path());
}

static inline bool isMatchingTimestamp(QDateTime stamp1, QDateTime stamp2)
{
	/* On VFAT the mtime of a file can vary a second up and down */
	return (abs(stamp1.toTime_t() - stamp2.toTime_t()) <= 1);
}

static bool
_doesThumbnailExistAndIsValid ( const QUrl& resource,
                                const QString& flavor,
                                const QString& scheduler )
{
	QString thumbnailPath;
	thumbnailPath = Thumbnailer::getThumbnailPath(resource, flavor).path();

	// for scheduler 'noop' just check if particular thumbnail exists
	// no check for up to date or uri scheme
	if (Thumbnailer::noopScheduler == scheduler) {
		return QFile::exists(thumbnailPath);
	}

	// we can check only files, everything else need to go to Tumbler
	if(resource.scheme() != "file") return false;

	// check if file with thumbnail exist
	QFileInfo thumbnail(thumbnailPath);
	if (!thumbnail.exists()) {
		return false;
	}

	// check if file with thumbnail has the same modification time as
	// source (tumbler force modification time of new thumbnail to be
	// the same as the source modification time)
	QFileInfo resourceFile(resource.path());

	return isMatchingTimestamp (resourceFile.lastModified(), thumbnail.lastModified());
}

static inline QStringList
_convertQUrlListToQStringList(const QList<QUrl>& uris) {
	QStringList strings;
	foreach(QUrl uri, uris) {
		strings << QString::fromUtf8(uri.toEncoded());
		// original
		// strings << uri.toString().replace ('#', "%23");
	}
	return strings;
}

static QString
_calculateMD5(const QUrl& src) {

	// printf ("\n%s\n", QString::fromUtf8(src.toEncoded()).toUtf8().data());
	QByteArray md5 = QCryptographicHash::hash ( QString::fromUtf8(src.toEncoded()).toUtf8(),
						    QCryptographicHash::Md5 );

	//printf ("\n%s\n", QString(md5.toHex()).toUtf8().data());

	return QString(md5.toHex());
}

static const QString _flavorSettingsPath = QString("/etc/xdg/thumbnails/flavors.conf");
static const QString _fileSchema         = QString("file://");
static const QString _dirSeparator       = QDir::separator();
static const QString _homePath           = QDir::toNativeSeparators(QDir::homePath());
static QMutex _mutex (QMutex::Recursive);

/* **************************************************************************
 *      STATIC FUNCTIONS & MEMBERS:
 * **************************************************************************/
// 
const QString Thumbnailer::noopScheduler = "noop";
const QString Thumbnailer::defaultFlavor = "normal";

// 
QUrl
Thumbnailer::getThumbnailPath (const QUrl& uri, const QString& flavor) {
	static QString thumbnailsPathPrefix = _fileSchema + _homePath +
	                                      _thumbnailsCacheDir;
	QString thumbnailName = _calculateMD5 (uri);
	return QUrl (thumbnailsPathPrefix + flavor + _dirSeparator + thumbnailName + ".jpeg" );
}

//
void
Thumbnailer::move (const QList<QUrl>& from, const QList<QUrl>& to) {
	if(from.size() != to.size()) return;
	QStringList _from = _convertQUrlListToQStringList(from);
	QStringList _to = _convertQUrlListToQStringList(to);

	DBusServices::cache()->Move(_from, _to);
}

// 
void
Thumbnailer::copy (const QList<QUrl>& from, const QList<QUrl>& to) {
	if(from.size() != to.size()) return;
	QStringList _from = _convertQUrlListToQStringList(from);
	QStringList _to = _convertQUrlListToQStringList(to);

	DBusServices::cache()->Copy(_from, _to);
}

// 
void
Thumbnailer::remove (const QList<QUrl>& fileUris) {
	QStringList files = _convertQUrlListToQStringList(fileUris);

	DBusServices::cache()->Delete(files);
}

// 
void
Thumbnailer::cleanup (const QString& prefix, uint before) {
	// just call the function from tubmler
	DBusServices::cache()->Cleanup(prefix, before);
}

//
QMultiHash<QString, QString>
Thumbnailer::getSupported () {
	QMultiHash<QString, QString> supported;
	QDBusReply<QStringList>      reply;
	QStringList                  mimes;

	// get reply from Tumbler service (dbus traffic!)
	reply = DBusServices::tumbler()->GetSupported(mimes);
	// if reply ios valid fill out supported hash with data
	if (reply.isValid() && reply.value().size() == mimes.size()) {
		QString scheme;
		for(int i=0; i < reply.value().size(); ++i) {
			supported.insert( reply.value().at(i),
			                  mimes.at(i) );
		}
	}

	return supported;
}

//
QStringList
Thumbnailer::getSchedulers () {
	QDBusPendingReply<QStringList> reply;
	QStringList list;
	// get reply from Tumbler service (dbus traffic!)
	reply = DBusServices::tumbler()->GetSchedulers();
	reply.waitForFinished();
	if (reply.isValid()) {
		// call successed - get values
		list = reply.value();
	}

	// add 'noop' scheduler
	list.append(QString(Thumbnailer::noopScheduler));
	return list;
}

//
QStringList
Thumbnailer::getFlavors () {
	QDBusPendingReply<QStringList> reply;
	// get reply from Tumbler service (dbus traffic!)
	reply = DBusServices::tumbler()->GetFlavors();
	reply.waitForFinished();
	if (reply.isError()) {
		// call failed. return empty QStringList
		return QStringList();
	}
	else {
		return reply.value();
	}
}

QSize
Thumbnailer::getFlavorSize (const QString& flavor) {
	QSettings flavors( _flavorSettingsPath,
	                   QSettings::IniFormat );
	if(!flavors.childGroups().contains(flavor)) return QSize();
	flavors.beginGroup(flavor);
	// negative values means some error
	if(!flavors.contains("Width")) return QSize(-2,-2);
	if(!flavors.contains("Height")) return QSize(-3,-3);

	return QSize( flavors.value("Width").toInt(),
	              flavors.value("Height").toInt() );
}

/* **************************************************************************
 *      CONSTRUCTOR & DESTRUCTOR:
 * **************************************************************************/

//
Thumbnailer::~Thumbnailer () {
	// first cancel all running requests
	cancel (false, false);
	// then delete private stuff
	delete priv;
}

// 
Thumbnailer::Thumbnailer ( QUrl defaultUri,
                           QPixmap defaultPixmap ) : QObject(),
                                                     priv(new ThumbnailerPrivate)
{
	setDefaultURI (defaultUri);
	setDefaultPixmap (defaultPixmap);
	priv->parent = this;

	// connect signals from dbus services to this Thumbnailer object
	QObject::connect( DBusServices::instance(),
	                  SIGNAL(ReadyHandler(uint, const QStringList&)),
	                  priv,
	                  SLOT(ReadyHandler(uint, const QStringList&)));
	QObject::connect( DBusServices::instance(),
	                  SIGNAL(FinishedHandler(uint)),
	                  priv,
	                  SLOT(FinishedHandler(uint)));
	QObject::connect( DBusServices::instance(),
	                  SIGNAL(StartedHandler(uint)),
	                  priv,
	                  SLOT(StartedHandler(uint)));
	QObject::connect( DBusServices::instance(),
	                  SIGNAL(ErrorHandler(uint,const QStringList&,int,const QString&)),
	                  priv,
	                  SLOT(ErrorHandler(uint,const QStringList&,int,const QString&)));
	QObject::connect( DBusServices::instance(),
	                  SIGNAL(tumblerServiceReconnected()),
	                  priv,
	                  SLOT(genericProxyChanged()) );
}


/* **************************************************************************
 *      MAKING & CANCELING THE REQUEST:
 * **************************************************************************/

// 
void Thumbnailer::cancel ( bool sendRemainingSignals, bool unqueueFromTumbler ) {
	_mutex.lock ();

	// stop the timer
	priv->timerStop ();

	// stop the timer
	priv->timerForcedStop ();

	int itemsLeft = priv->requests.size();
	// cancel every request already running

	QHash<uint, ServiceRequest*>::iterator i = priv->requests.begin();
	while (i != priv->requests.end())
	{
		ServiceRequest* request = i.value();

		if(NULL == request) { ++i; continue; }

		if(sendRemainingSignals) {
			// send dequeue signals if requested was dequeued
			if(request->media.isEmpty()) {
				// for normal thumbnails
				foreach (QString uri, request->uris) {
					Q_EMIT dequeued (QUrl(uri));
				}
			}
			else {
				// for Media-Art thumbnails
				foreach (Info mai, request->media) {
					Q_EMIT dequeued (mai);
				}
			}
		}
		// dequeue request from Tumbler Service (only if wee need to)
		if(unqueueFromTumbler) DBusServices::tumbler()->Dequeue (i.key());
		--itemsLeft;
		
		// delete also the already processed request object itself
		delete request;
		request = 0;
		
		i.value()=NULL;

		++i;


		// do not forget to send finish signal
		Q_EMIT finished (itemsLeft +
		                 priv->waitingForReply.size() );
	}
	priv->requests.clear();

	// cancel also every request which is waiting to reply
	itemsLeft = priv->waitingForReply.size();
	QHash<QDBusPendingCallWatcher*, ServiceRequest*>::iterator j = priv->waitingForReply.begin();
	while (j != priv->waitingForReply.end())
	{
		ServiceRequest* request = j.value();
		if(NULL == request) { j++; continue; }
		QDBusPendingCallWatcher* watcher = j.key();

		if(sendRemainingSignals) {
			// send dequeue signals if requested was dequeued
			if(request->media.isEmpty()) {
				// for normal thumbnails
				foreach (QString uri, request->uris) {
					Q_EMIT dequeued (QUrl(uri));
				}
			}
			else {
				// for Media-Art thumbnails
				foreach (Info mai, request->media) {
					Q_EMIT dequeued (mai);
				}
			}
		}
		// move to next request on the pending list
		--itemsLeft;
		
		
		// disconnect watcher (turn it OFF) - delete it
		delete watcher;
		// delete processed request object
		delete request;
		request = 0;
		j.value()=NULL;

		++j;

		// do not forget to send finish signal
		Q_EMIT finished (itemsLeft);
	}
	priv->waitingForReply.clear();

	priv->timerStop ();

	_mutex.unlock ();
}

bool
Thumbnailer::request ( QList<Info> &media,
                       bool sendPixmap,
                       QString flavor,
                       QString scheduler,
                       bool autoCancel,
                       bool sendRemainingSignals )
{
	_mutex.lock ();

	Q_EMIT started();
	if (flavor.isEmpty())    flavor    = _defaultFlavor;
	if (scheduler.isEmpty()) scheduler = _defaultScheduler;
	if (autoCancel) cancel(sendRemainingSignals);

	ServiceRequest* request = new ServiceRequest( this,
	                                              sendPixmap,
	                                              flavor );
	foreach(Info mai, media) {
		if(_doesThumbnailExistAndIsValid(mai.potentialPath(), flavor, scheduler)) {
			// thumbnail exists
			QUrl thumb = getThumbnailPath(mai.potentialPath(), flavor);
			Q_EMIT thumbnail ( mai,
			                   thumb,
			                   _getPixmap(thumb, sendPixmap),
			                   flavor );
		}
		else if(Thumbnailer::noopScheduler == scheduler) {
			// thumbnail does not exist and we do not want to create it
			Q_EMIT error( QString("noop Scheduler:no creation of new thumbnails"),
			              mai,
			              defaultURI(),
			              sendPixmap ? defaultPixmap() : QPixmap() );
		}
		else if(mai.exists()) {
			// thumbnail do not exist, but media art is there
			request->media.append(mai);
			request->uris.append(mai.potentialPath().toString());
			request->mimes.append(QString("image/jpeg"));
			// and send default signal with default values
			Q_EMIT defaultThumbnail ( mai,
			                          defaultURI(),
			                          sendPixmap ? defaultPixmap() : QPixmap(),
			                          flavor );
		}
		else {
			Q_EMIT defaultThumbnail ( mai,
			                          defaultURI(),
			                          sendPixmap ? defaultPixmap() : QPixmap(),
			                          flavor );
			// no thumbnail either media art
			// INFO: at this moment there is no service from which
			// we would be able to download the content, so we can send
			// error signal stright away.
			// 
			// Later, when some service would became available, we can
			// ask MediaArt::Requester to look for that content first.
			Q_EMIT error( QString("Unable to find that Media Art - no source for thumbnail"),
			              mai,
			              defaultURI(),
			              sendPixmap ? defaultPixmap() : QPixmap() );
		}
	}

	// check if we need to initiate the request to tumbler - and do it if Yes
	bool ret = priv->makeRequest(request, flavor, scheduler);

	_mutex.unlock ();

	return ret;
}

// 
bool
Thumbnailer::request ( QList<QUrl> &uris,
                       QStringList &mimeTypes,
                       bool sendPixmap,
                       QString flavor,
                       QString scheduler,
                       bool autoCancel,
                       bool sendRemainingSignals )
{
	_mutex.lock ();

	Q_EMIT started();
	if (flavor.isEmpty())    flavor    = _defaultFlavor;
	if (scheduler.isEmpty()) scheduler = _defaultScheduler;

	if (autoCancel) cancel(sendRemainingSignals);
	if (uris.size() != mimeTypes.size()) {
		// if number of uris and mimes does not match - send error for
		// every URI and finish immediatelly the request!
		foreach(QUrl uri, uris) {
			Q_EMIT error("Non matching number of URIs and MIME types"
			             " in the request!", uri,
			              defaultURI(),
			              sendPixmap ? defaultPixmap() : QPixmap());
		}

		Q_EMIT finished (priv->requests.size() +
		                 priv->waitingForReply.size());

		_mutex.unlock ();

		return false;
	}

	ServiceRequest* request = new ServiceRequest( this,
	                                              sendPixmap,
	                                              flavor );
	QUrl uri;
	QString mime;
	for(int i=0; i < uris.size(); ++i) {
		uri = uris.at(i);
		// check if that thumbnail already exist
		if (_doesThumbnailExistAndIsValid(uri, flavor, scheduler)) {
			// if yes just send signal with it directly
			QUrl thumb = getThumbnailPath(uri, flavor);
			Q_EMIT thumbnail ( uri,
			                   thumb,
			                   _getPixmap(thumb, sendPixmap),
			                   flavor );
		}
		else if (Thumbnailer::noopScheduler == scheduler) {
			// for 'noop' scheduler we just need to ommit the rest
			// of the sources
			Q_EMIT error ( QString("No thumbnail for source but "
			                     "\'noop\' scheduler is used. "
			                     "_NO_ new thumbnails will be generated."),
			               uri,
			               defaultURI(),
			               sendPixmap ? defaultPixmap() : QPixmap() );
		}
		else {
			// otherwiswe add it to request object - this will be 
			// requested later
			// QUrl::toPercentEncoding ( uri)
			request->uris.append(QString::fromUtf8(uri.toEncoded()));
			//QString (QUrl::toPercentEncoding (uri.toString()).constData()));
			// original:
			// request->uris.append(uri.toString().replace ('#', "%23"));
			request->mimes.append(mimeTypes.at(i));
			// and send default signal with default values
			Q_EMIT defaultThumbnail ( uri,
			                          defaultURI(),
			                          sendPixmap ? defaultPixmap() : QPixmap(),
			                          flavor );

		}
	}

	// check if we need to initiate the request to tumbler - and do it if yes
	bool ret = priv->makeRequest(request, flavor, scheduler);

	_mutex.unlock ();

	return ret;
}


/* **************************************************************************
 *      CURRENT SETTINGS OF THE OBJECT:
 * **************************************************************************/

// 
void
Thumbnailer::setDefaultURI (QUrl uri) {
	priv->defaultUri = uri;
}

//
void
Thumbnailer::setDefaultPixmap (QPixmap pixmap) {
	priv->defaultPixmap = pixmap;
}

//
QUrl
Thumbnailer::defaultURI () const {
	return priv->defaultUri;
}

//
QPixmap
Thumbnailer::defaultPixmap () const {
	return priv->defaultPixmap;
}

/* **************************************************************************
 *      TUMBLER REQUEST METHOD REPLIED WATCHER
 * **************************************************************************/
void
ThumbnailerPrivate::requestCallFinished(QDBusPendingCallWatcher* watcher)
{
	_mutex.lock ();

	// get proper request object
	ServiceRequest* request = waitingForReply.value(watcher);
	// get the reply value
	QDBusPendingReply<uint> reply = *watcher;

	// check the reply from thumbnailer service
	if(reply.isValid()) {

		// if dbus call was sucessfull add the request to the requests list
		requests.insert(reply.value(), request);
		// and start the timer to monitor what is going on with request
		timerStart ();
	}
	else {
		// otherwise send error message for all left uris and finish signal
		qDebug() << "Could not request from Tumbler:" << reply.error();
		if(request->media.isEmpty()) {
			foreach(QString uri, request->uris) {
				Q_EMIT parent->error( "Could not request for thumbnail "
				                      "from Tumbler Service!",
				                      QUrl(uri),
				                      parent->defaultURI(),
				                      request->sendPixmap ? parent->defaultPixmap() : QPixmap() );
			}
		}
		else {
			foreach(Info mai, request->media) {
				Q_EMIT parent->error( "Could not request for Media-Art thumbnail "
				                      "from Tumbler Service!",
				                      mai,
				                      parent->defaultURI(),
				                      request->sendPixmap ? parent->defaultPixmap() : QPixmap() );
			}
		}

		Q_EMIT parent->finished (requests.size() +
		                         waitingForReply.size());
		//delete request object itself also
		delete request;

	}
	// remove request from waiting to reply list..
	waitingForReply.remove(watcher);
	// ..and delete watcher object
	delete watcher;

	_mutex.unlock ();

}

/* **************************************************************************
 *      TUMBLER SIGNALS HANDLERS:
 * **************************************************************************/
#define FILTER_SIGNALS(h) /*qDebug() << "Handler:" << __FUNCTION__ << "called";*/\
                          if(!isThisForMe(h)) {\
                               /*qDebug() << "filtering out" << h << "signal";*/\
                               return;\
                          }

//
void
ThumbnailerPrivate::ErrorHandler( uint handle,
                                  const QStringList &failedUris,
                                  int errorCode,
                                  const QString &message )
{
	_mutex.lock ();

	Q_UNUSED(errorCode);
	FILTER_SIGNALS(handle);

	ServiceRequest* request = requests.value(handle);

	if(NULL == request) { _mutex.unlock(); return; }

	if(request->media.isEmpty()) {
		// go through all failed uris
		foreach(QString uri, failedUris) {
			// look for that uri in the uris list of requests
			if(request->uris.contains(uri)) {
				// send signal for that uri
				QUrl resource = QUrl(uri);
				Q_EMIT parent->error ( message,
				                       QUrl(uri),
				                       parent->defaultURI(),
				                       request->sendPixmap ? parent->defaultPixmap() : QPixmap() );
				// and remove it from uris list for that request
				request->uris.removeOne(uri);
			}
		}
	}
	else {
		// go through all failed media arts
		foreach(QString uri, failedUris) {
			// look for that uri in the uris list of requests
			if(request->uris.contains(uri)) {
				// send signal for that uri
				QUrl resource = QUrl(uri);
				// get also media art object (need to look after potentialPath member)
				Info mai;
				foreach(Info tmp, request->media) {
					if(tmp.potentialPath() == resource) {
						mai = tmp; break;
					}
				}
				Q_EMIT parent->error ( message, mai,
				                       parent->defaultURI(),
				                       request->sendPixmap ? parent->defaultPixmap() : QPixmap() );
				// and remove it from uris list for that request
				request->uris.removeOne(uri);
				request->media.removeOne(mai);
			}
		}
	}

	// zero the counter for timer
	request->intervalsSinceLastAction = 0;

	_mutex.unlock ();
}

//
void
ThumbnailerPrivate::FinishedHandler(uint handle)
{
	_mutex.lock ();

	FILTER_SIGNALS(handle);

	// check if there are any remainign uris in the list and send dequeue
	// signal for them

	ServiceRequest* request = requests.value(handle);

	if(NULL == request) { _mutex.unlock (); return; }

	if(request->media.isEmpty()) {
		foreach(QString uri, request->uris) {
			Q_EMIT parent->dequeued(QUrl(uri));
		}
	}
	else {
		foreach(Info mai, request->media) {
			Q_EMIT parent->dequeued(mai);
		}
	}
	// remove request from requests list...
	requests.remove(handle);
	// clear the whole list and delete the request object
	delete request;
	// ..and send finish signal
	Q_EMIT parent->finished (requests.size() +
	                         waitingForReply.size());

	// stop timer if it is not needed anymore
	timerStop ();

	_mutex.unlock ();

}

//
void
ThumbnailerPrivate::ReadyHandler(uint handle, const QStringList &uris)
{
	_mutex.lock ();

	FILTER_SIGNALS(handle);

	// gor through all ready uris
	ServiceRequest* request = requests.value(handle);
	if(NULL == request) { _mutex.unlock (); return; }

	// standard thumbnails
	if(request->media.isEmpty()) {
		foreach(QString uri, uris) {
			if(request->uris.contains(uri)) {
				// and for each of them send ready signal
				QUrl resource = QUrl::fromEncoded(uri.toAscii());
				QUrl thumbnail = Thumbnailer::getThumbnailPath( resource,
				                                                request->flavor );
				Q_EMIT parent->thumbnail ( resource,
				                           thumbnail,
				                           _getPixmap(thumbnail, request->sendPixmap),
				                           request->flavor );
				// ... and remove it from the list
				request->uris.removeOne(uri);
			}
		}
	}
	// thumbnails of media arts
	else {
		foreach(QString uri, uris) {
			if(request->uris.contains(uri)) {
				// and for each of them send ready signal
				QUrl resource = QUrl::fromEncoded(uri.toAscii());
				QUrl thumbnail = Thumbnailer::getThumbnailPath( resource,
				                                                request->flavor );
				// get also media art object (need to look after potentialPath member)
				Info mai;
				foreach(Info tmp, request->media) {
					if(tmp.potentialPath() == resource) {
						mai = tmp; break;
					}
				}
				Q_EMIT parent->thumbnail ( mai,
				                         thumbnail,
				                         _getPixmap(thumbnail, request->sendPixmap),
				                         request->flavor );
				// ... and remove it from the list
				request->uris.removeOne(uri);
				request->media.removeOne(mai);
			}
		}
	}
	// zero the counter for timer
	request->intervalsSinceLastAction = 0;

	_mutex.unlock ();
}

void
ThumbnailerPrivate::StartedHandler(uint handle) {
	FILTER_SIGNALS(handle);
	// with this signal we do not need to do anything in fact (atm)
#ifdef TESTING_CODE
	Q_EMIT remoteStarted(handle);
#else
	Q_UNUSED(handle);
#endif
}
#undef FILTER_SIGNALS

/*! Checks if particular signal is dedicated to that instance of Thumbnailer.
 *
 * Currently all instances of Thumbnailer class in particular thread (those
 * connected with single ThumbnailerGenericProxy object reachable by
 * thumbnailer() method) will get signals from DBus Tumbler Service for all
 * instances. We need to filter out signals not dedicated to particular
 * instance. If signal is dedicated for particular instance that instance will
 * have id of the request in the requests list.
 *
 * \param handle request id number
 */
inline bool
ThumbnailerPrivate::isThisForMe(uint handle) {
	bool ret;

	_mutex.lock ();

	ret = requests.contains(handle);

	_mutex.unlock ();

	return ret;
}

void
ThumbnailerPrivate::genericProxyChanged() {
	// cancel all previous request if connection was resetted
	// but do not try to unqueue from tumbler - it was resetted!
	parent->cancel(true, false);
}

/*!
 *
 */
void
ThumbnailerPrivate::timerEvent (QTimerEvent*)
{
	_mutex.lock ();

	bool had_any = true, valid_iter = false;

	// check if connection is OK
	bool connectionStatus = DBusServices::tumbler()->isValid();
	// go through the all request and check them (only couple of them could hang)
	QHash<uint, ServiceRequest*>::iterator i = requests.begin();
	while (i != requests.end())
	{
		valid_iter = true;

		ServiceRequest* request = i.value();

		if (NULL == request) { ++i; continue; } else { had_any = false; }

		// if processServiceRequest return true we need to remove that
		// request from the list and delete request object itself
		if ( processServiceRequest(request, !connectionStatus) ) {
			i = requests.erase(i);
			// emit finished signal
			Q_EMIT parent->finished (requests.size() +
			                         waitingForReply.size());
			// stop timer if it is possible
			timerStop ();
			delete request;
		}
		else {
			++i;
		}
	}

	// If we only had 0 as values in the requests, then let's clear it to get 
	// rid of it
	if (had_any && valid_iter) {
		requests.clear ();
		// stop timer if it is possible
		timerStop ();
	}

	_mutex.unlock ();
}

void
ThumbnailerPrivate::timerStart () {
	_mutex.lock ();

	// check if it is not already running - if yes, does not need to start
	if(0 != timer) { _mutex.unlock (); return; }
	// check if there is any request running/pending - if not does not
	// need to start it
	if(0 >= requests.size()) { _mutex.unlock (); return; }

	timer =  startTimer (__intervalDuration);

	_mutex.unlock ();
}

//
void
ThumbnailerPrivate::timerStop () {

	_mutex.lock ();

	// check if timer is running - if not does not need to stop it
	if(0 == timer) { _mutex.unlock (); return; }
	// check if all requests are done - if there is at least one running
	// request You should not stop the timer
	if(0 < requests.size()) { _mutex.unlock (); return; }

	// stop the timer immediatelly
	killTimer (timer);
	// mark it as not-running
	timer = 0;

	_mutex.unlock ();
}

//
void
ThumbnailerPrivate::timerForcedStop () {

	_mutex.lock ();

	// check if timer is running - if not does not need to stop it
	if(0 == timer) { _mutex.unlock (); return; }

	// stop the timer immediatelly
	killTimer (timer);
	// mark it as not-running
	timer = 0;

	_mutex.unlock ();
}

//
bool
ThumbnailerPrivate::processServiceRequest ( ServiceRequest* request,
                                            bool forceTimeout )
{
	// increase counter
	request->intervalsSinceLastAction++;
	// check if counter does not cross the limit
	if ( (__maxIntervalsBeforeTimeout < request->intervalsSinceLastAction) ||
	     forceTimeout ) {
		// this request did not respond too long - remove it from the
		// list, send proper error signals
		qDebug() << "DBus (Tumbler) timeout for request from Thumbnails::Thumbnailer! Canceling the request.";
		if(request->media.isEmpty()) {
			foreach(QString uri, request->uris) {
				Q_EMIT parent->error( "DBus (Tumbler) timeout for request from Thumbnails::Thumbnailer! Canceling the request.",
				                      QUrl(uri),
				                      parent->defaultURI(),
				                      request->sendPixmap ? parent->defaultPixmap() : QPixmap() );
			}
		}
		else {
			foreach(Info mai, request->media) {
				Q_EMIT parent->error( "DBus (Tumbler) timeout for request from Thumbnails::Thumbnailer! Canceling the request.",
				                      mai,
				                      parent->defaultURI(),
				                      request->sendPixmap ? parent->defaultPixmap() : QPixmap() );
			}
		}
		// clear the whole list
		request->uris.clear();
		request->media.clear();
		// request had hang - remove it from the list
		return true;
	}
	// request did not hang - everything is O.K.
	return false;
}

bool
ThumbnailerPrivate::makeRequest ( ServiceRequest* request,
                                  const QString& flavor,
                                  const QString& scheduler )
{
	_mutex.lock ();

	if ( 0 == request->uris.size() ||
	     Thumbnailer::noopScheduler == scheduler )
	{
		// all requested images were arleady there, no need for dbus request
		Q_EMIT parent->finished ( requests.size() +
		                          waitingForReply.size() );
		delete request;
		request = 0;

		_mutex.unlock ();

		return true;
	}

	// send request via DBus for the rest of non-existing thumbnails
	QDBusPendingReply<uint> reply;
	reply = DBusServices::tumbler()->Queue ( request->uris,
	                                         request->mimes,
	                                         flavor,
	                                         scheduler, 0 );
	// wait for the reply from DBus call
	QDBusPendingCallWatcher* watcher = new QDBusPendingCallWatcher( reply,
	                                                                this );
	// add this request to the list ''waiting for reply'
	waitingForReply.insert(watcher, request);
	QObject::connect ( watcher,
	                   SIGNAL(finished(QDBusPendingCallWatcher*)),
	                   this,
	                   SLOT(requestCallFinished(QDBusPendingCallWatcher*)));

	// we can already clear mimes array - it will not be needed anymore
	// just calling this here to make it smaller in the memmory
	request->mimes.clear();

	_mutex.unlock ();

	// not all thumbnails were existing - request was done so there is some
	// dbus traffic.. return false for noDBusTraffic

	return false;
}
