/*
 * Copyright (C) 2009 Nokia Corporation.
 *
 * Contact: Marius Vollmer <marius.vollmer@nokia.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

/*
 * thumbnailer.h
 *
 *  Created on: Oct 26, 2009
 * Modified on: Nov 10, 2009
 *     Authors: iridian, wiecheck
 */
#ifndef __THUMBNAILER_H__
#define __THUMBNAILER_H__

#include <QtCore/QObject>
#include <QtCore/QString>
#include <QtCore/QUrl>
#include <QtCore/QMultiHash>
#include <QtGui/QPixmap>
#include <QSize>
#include "mediaartinfo.h"

namespace Thumbnails {

/*
 Private data for each, separate Thumbnailer instance.
 */
struct ThumbnailerPrivate;

/** \brief  Requests and delivers thumbnails for given URI-based resources.
 *
 * Uses the D-Bus service org.freedesktop.thumbnails.Thumbnailer1.
 * For D-Bus API see:
 * \a http://live.gnome.org/ThumbnailerSpec .
 * Thumbnailer tries to be as much similiar to original D-Bus API as
 * possible but in order to make it more efficency some additional
 * functionalities were added.
 * Basic use of that class looks like this (pseudo-code):\n
 * \code
 * //create object with default values
 * Thumbnailer* thumb = new Thumbnailer (QUrl("file:///default/image/path"));
 * \endcode \n
 * Given default value will be returned to You if You will use
 * defaultThumbnail signal. If You do not want to utilize this signal
 * (or You do not want to use parameters of that signal) you can call
 * constructor without any parameters:\n
 * \code
 * Thumbnailer* thumb = new Thumbnailer ();
 * \endcode \n
 * Next You need to create list with uris of documents You want to get thumbnailed
 * and connect signals of Thumbnailer object to get know when thumbnail is ready\n
 * \code
 * // connect signals You are interested in (in this case the simplest way -
 * // just thumbnail signal)
 * QObject::connect ( thumb, SIGNAL(thumbnail(QUrl,QUrl,QPixmap,QString)),
 *                    this, SLOT(thumbnail(QUrl,QUrl,QPixmap)) );
 *
 * // make list with uris and mime types for request
 * QUrl    fileUri ("file:///some/image/path.jpeg");
 * QString fileMimeType = "image/jpeg";
 *
 * QList<QUrl> urisList;
 * QStringList mimeList;
 *
 * urisList << fileUri;      // You can add as much uris as you want ...
 * mimelList << fileMimeType; // ...just remember to add the same number,
 *                            // matching mime types
 * \endcode \n
 * Now it is time to make a request:\n
 * \code
 * // make a request
 * thumb->request (urisList, mimeList);
 * \endcode or
 * \code
 * // make a request and ask Thumbnailer to load every thumbnail in the request
 * thumb->request (urisList, mimeList, true);
 * \endcode \n
 * In the second case You will get QPixmap in thumbnail()'s third
 * parameter, so you do not need to open and load data itself. \n
 * Of course You need to have a slot in Your class which will handle new
 * thumbnail and utilize it. It could be something like this:
 * \code
 * public slot void MyClass::thumbnail ( QUrl fileUri,
 *                                       QUrl thumbnailUri,
 *                                       QPixmap pixmap,
 *                                       QString flavor )
 * {
 *    qDebug() << "Thumbnail for image" << fileUri.toString() <<
 *             << "is located in path:" << thumbnailUri.toString();
 *    // if You made a request with the third parameter set to true You can do:
 *    this->label->setPixmap (pixmap);
 * }
 * \endcode \n
 * You can also use the rest of Thumbnailer's signals to know a little bit
 * more what is happening with Your request. For details see documentation
 * for particular signals and request() method:
 * \code
 * QObject::connect ( thumb, SIGNAL(error(QString,QUrl)),
 *                    this, SLOT(error(QString,QUrl)) );
 * QObject::connect ( thumb, SIGNAL(dequeued(QUrl)),
 *                    this, SLOT(dequeued(QUrl)) );
 * QObject::connect ( thumb, SIGNAL(started()),
 *                    this, SLOT(loadingStarted()) );
 * QObject::connect ( thumb, SIGNAL(finished(int)),
 *                    this, SLOT(loadingFinished(int)) );
 * QObject::connect ( thumb, SIGNAL(defaultThumbnail(QUrl,QUrl,QPixmap,QString)),
 *                    this, SLOT(defaultThumbnail(QUrl,QUrl,QPixmap)) );
 * \endcode
 * \n \n
 * Beside thumbnailing particular resources You can request for Media Art
 * thumbnails. \see MediaArt::Info, Album, Artist, Track, Podcast, Radio
 * documentation
 * 
 * In order to do that just call request method with list of
 * MediaArt::Info objects, not QUrl ones. Then listen for signals exactly
 * the same as for normal thumbnails but with MediaArt::Info parameter in
 * place of QUrl as a source of the requests.
 */
class Thumbnailer : public QObject {
Q_OBJECT
/* **************************************************************************
 *      STATIC FUNCTIONS:
 * **************************************************************************/
public:
	/*!
	 * \name Cache related functions.
	 *
	 * Those function could be used after some resources are moved, copied
	 * or deleted. That should be reflected also in thumbnail's cache.
	 */
	//@{

	/** \brief Update thumbnails cache after some resources were moved.
	 *
	 * Length of \a from and \a to lists should be exactly the same.
	 * \param from list of source resources
	 * \param to list of target places, where source resources were moved
	 */
	static void move    ( const QList<QUrl>& from, const QList<QUrl>& to );

	/** \brief Update thumbnails cache after some resources were copied.
	 *
	 * Length of \a from and \a to lists should be exactly the same.
	 * \param from list of source resources
	 * \param to  list of target places, where source resources were copied
	 */
	static void copy    ( const QList<QUrl>& from, const QList<QUrl>& to );

	/** \brief Update thumbnails cache after some resources were deleted.
	 * \param fileUris URI's list of resources which were deleted
	 */
	static void remove  ( const QList<QUrl>& fileUris );

	/** \brief Request for cleaning up the cache of invalid or old thumbnails
	 * \param prefix the prefix that must match the URIs of original
	 *               thumbnailables of whom the thumbnail must be deleted
	 * \param before filter so that only thumbnails of files that have a
	 *               modified time older than \a before are deleted
	 */
	static void cleanup ( const QString& prefix, unsigned before );
	//@}

	/*!
	 * \name Supported schedulers, schemes, MIMEs types and flavors.
	 *
	 * Developer can check results from those functions to get informations
	 * what are all the possibilities of the Thumbnailer. It is possible to
	 * get informations about supported schedulers and flavors. Also
	 * informations about what kind of MIME types are supported for
	 * particular URI scheme could be retrieved by the developer.
	 */
	//@{

	/** \brief Returns supported MIME types for particular URI schemes.
	 *
	 * If You request for resource which is of MIME type not supported for
	 * its URI scheme the thumbnail will not be created and You will get
	 * error signal for that request.\n
	 * Returned Hash table as a key has QString with scheme and as a value
	 * supported MIME type. There could be more entries with the same
	 * scheme so you should check them all to find out if particular MIME
	 * type is supported.
	 * \return a map between supported URI schemes and MIME types
	 */
	static QMultiHash<QString, QString> getSupported  ();

	/** \brief Returns list of supported schedulers.
	 *
	 * It gives to the developer information what kind of schedulers he can
	 * use. For detail informations about what are schedulers and what are
	 * the differences between them see:
	 * \a http://live.gnome.org/ThumbnailerSpec
	 * \return StringList with supported schedulers
	 */
	static QStringList                  getSchedulers ();

	/** \brief Scheduler 'noop' which returns only already existing thumbnails.
	 * 
	 * This is a Thumbnailer's provided scheduler. You should use it if
	 * You are interested in fast reply from Thumbnailer - it will
	 * return You only those thumbnailes which are already existing and
	 * error signal for rest of them. Also none of the thumbnail will be
	 * updated or checked.
	 */
	static const QString                noopScheduler;

	/** \brief Default flavor which could be used with getThumbnailPath function.
	 *
	 * If user want to get thumbnail path by himself and he want to get default
	 * flavor of thumbnail, He can use this value as a second parameter (flavor)
	 * of getThumbnailPath()
	 */
	static const QString                defaultFlavor;

	/** \brief Returns list of supported flavors.
	 *
	 * Flavor describe the kind of thumbnail. You could want to retrive a
	 * a normal thumbnail or preview. There could be more options like:
	 * large, small, cropped and so on. This function returns to You list
	 * with supported flavors on the system.\n
	 * When You make a request You can ask for particular kind of thumbnail
	 * and if You give a wrong flavor error will occur. The default value
	 * in request() method will always work. If You do not want to take
	 * care about proper flavor value You can also pass empty string as a
	 * flavor parameter to request() method.\n\n
	 * Visit also ThumbnailerSpec page for more details about flavors:\n
	 * \a http://live.gnome.org/ThumbnailerSpec
	 * \return StringList with supported flavors
	 */
	static QStringList                  getFlavors ();

	/** \brief Retrieves a size of particular kind of thumbnails.
	 *
	 * \param flavor flavor we want to get the size of
	 * \return QSize structure which tells what dimensions have the flavor
	 *    If given flavor does not exist it returns invalid QSize.
	 */
	static QSize                        getFlavorSize (const QString& flavor);
	//@}

	/** \brief Returns thumbnail's URI for particular resource.
	 *
	 * This function just construct the uri pointing to file in which there
	 * should be a thumbnail for resource \a uri. It does not create a
	 * thumbnail itself.
	 * \param uri uri of resource of which thumbnail we are asking for
	 * \param flavor defines to what type of thumbnail we want to get the path.
	 *               See getFlavors() and
	 * \returns QUrl pointing to thumbnail of \a uri resource
	 * \a http://live.gnome.org/ThumbnailerSpec for more details about flavors.
	 */
	static QUrl getThumbnailPath ( const QUrl& uri, const QString& flavor );
/* **************************************************************************
 *      CONSTRUCTOR & DESTRUCTOR:
 * **************************************************************************/
public:
	/**
	 * \name Configuration state functions.
	 *
	 * Thumbnailer is not copyable and assigneable, so it should be
	 * always used by pointers (in lists, hashes etc.). You can also create
	 * Thumbnailer and use it by value but remember to not assign it to any
	 * other Thumbnailer, otherwise some request could be lost.\n
	 *
	 * It is strongly adivsed to create one Thumbnailer object per
	 * application/thread and make all request through that one object.
	 * Just thinks about it as a proxy for D-Bus API.
	 */
	//@{
	/** \brief Destruct Thumbnailer object and clean memmory.
	 * 
	 * All request are automatically canceled/dequeued. All remaining
	 * signals are sent before object is deleted but no new thumbnail will
	 * arrive - there will be only dequeued and finished signals.
	 */
	~Thumbnailer ();

	/** \brief Construct new Thumbnailer object with default values.
	 *
	 * Default values for defaultThumbnail() signal.
	 * \param defaultUri default URI which will passed to defaultThumbnail()
	 *                   signal
	 * \param defaultPixmap default QPixmap which will passed to 
	 *                      defaultThumbnail() signal
	 */
	 Thumbnailer ( QUrl    defaultUri    = QUrl(),
	               QPixmap defaultPixmap = QPixmap() );

/* **************************************************************************
 *      CURRENT SETTINGS OF THE OBJECT:
 * **************************************************************************/
public:
	/** \brief Changes the default URI for thumbnails.
	 *
	 * \note When creating a QUrl objects please remember to use
	 * proper constructor/function. For strings with encoded
	 * urls (like those returned to You by for example qttracker)
	 * You need to use QUrl::fromEncoded(QString) function.
	 * \see Comments for QUrl constructor:
	 * http://doc.trolltech.com/4.6/qurl.html#QUrl-2
	 * \see defaultThumbnail()
	 * \param uri new URI for default URI
	 */
	void    setDefaultURI    ( QUrl uri );

	/** \brief Changes the default Pixmap for thumbnails.
	 *
	 * \see defaultThumbnail()
	 * \param pixmap to new Pixmap for default Pixmap
	 */
	void    setDefaultPixmap ( QPixmap pixmap );

	/** \brief Gets current default URI for thumbnails.
	 *
	 * \see defaultThumbnail()
	 * \return current default QUrl in human readable format (not encoded)
	 */
	QUrl    defaultURI    () const;

	/** \brief Gets current default Pixmap for thumbnails.
	 * 
	 * \see defaultThumbnail()
	 * \return current default QPixmap
	 */
	QPixmap defaultPixmap () const;
	//@}

/* **************************************************************************
 *      MAKING & CANCELING THE REQUEST:
 * **************************************************************************/
public:
	/** \brief Cancel all currently running request
	 * 
	 * You can call this function if You are not interested anymore in
	 * retriving thumbnails You had asked before.
	 * \param sendRemainingSignals if You give true here You will get all
	 *                             dequeue signals for resources which
	 *                             were not processed yet; otherwise You
	 *                             will get only finished signal.
	 * \param unqueueFromTumbler if You cancel request by hand You should also
	 *                           unqueue request from Tumbler. If tubmler crashed
	 *                           or was resetted You should not try to unqueue
	 *                           request from new instance of Tumbler
	 */
	void cancel  ( bool sendRemainingSignals = true, bool unqueueFromTumbler = true );
	
	/** \brief Make a request for thumbnails.
	 *
	 * The only required parameters are uris and mimeTypes, which should be
	 * list with the same length.\n
	 * If You want to get pixmap with defaultThumbnail/thumbnail signals
	 * remember to put true as a third parameter and if You want to retrieve
	 * some particular kind of thumbnail please remember to use flavor
	 * argument.\n
	 * By default all previous, not finished request are not canceled. If
	 * You want so You need to be explicit about that and specify autoCancel
	 * parameter as a true.
	 *
	 * \param uris list of resources' uris for which we want to get thumbnails.
	 *             \note When creating a QUrl objects please remember to use
	 *             proper constructor/function. For strings with encoded
	 *             urls (like those returned to You by for example qttracker)
	 *             You need to use QUrl::fromEncoded(QString) function.
	 *             \see Comments for QUrl constructor:
	 *             http://doc.trolltech.com/4.6/qurl.html#QUrl-2
	 * \param mimeTypes list with mime types correlated to \a uris list
	 * \param sendPixmap set to TRUE if You would like to retireve QPixmap 
	 *                   with Ready signal; otherwise FALSE
	 * \param scheduler define what scheduler will be used when requesting 
	 *                  thumbnail from tumbler service if empty string is 
	 *                  given thumbnailer will use its default scheduler
	 * \param flavor define what kind of thumbnail user want to retrieve 
	 *               (normal, preview etc.) if empty string is given
	 *               thumbnailer will use its default flavor
	 * \param autoCancel set this to true if You want automatically cancel
	 *                   all previous, currently running request
	 * \param sendRemainingSignals if \a autoCancel is true, it tells if
	 *                             dequeued will be sent for canceled requests
	 * \return true if there was any request made to tumbler. It means that
	 *         there could be some signals later (but not necessary). false
	 *         means that all thumbnails were ready before request was made
	 *         to Thumbnailer - that tells us that all signals from
	 *         Thumbnailer connected with request were arleady sent.
	 */
	bool request ( QList<QUrl> &uris,
	               QStringList &mimeTypes,
	               bool sendPixmap = false,
	               QString flavor = QString(),
	               QString scheduler = QString(),
	               bool autoCancel = false,
	               bool sendRemainingSignals = true );

	/** \brief Requests for Media Art's thumbnails.
	 *
	 * It is exactly the same method as request() but it
	 * requests for media art's thumbnails. If some media art content is
	 * not available at the call moment, it will also try to fetch that
	 * content in the background.\n
	 * There is no need for mime type here while media art content is always
	 * image/jpeg type.
	 * 
	 * \param media list of media arts for which we want to get thumbnails
	 * \param sendPixmap set to TRUE if You would like to retireve QPixmap
	 *                   with Ready signal; otherwise FALSE
	 * \param scheduler define what scheduler will be used when requesting
	 *                  thumbnail from tumbler service if empty string is
	 *                  given thumbnailer will use its default scheduler
	 * \param flavor define what kind of thumbnail user want to retrieve
	 *               (normal, preview etc.) if empty string is given
	 *               thumbnailer will use its default flavor
	 * \param autoCancel set this to true if You want automatically cancel
	 *                   all previous, currently running request
	 * \param sendRemainingSignals if \a autoCancel is true, it tells if
	 *                             dequeued will be sent for canceled requests
	 * \return true if there was any request made to tumbler (MediaArt 
	 *         Requester). It means that there could be some signals later
	 *         (but not necessary). false means that all thumbnails and MediaArts
	 *         were ready before request was made to Thumbnailer - 
	 *         that tells us that all signals from
	 *         Thumbnailer connected with request were arleady sent.
	 * \see request()
	 */
	bool request ( QList<MediaArt::Info> &media,
	               bool sendPixmap = false,
	               QString flavor = QString(),
	               QString scheduler = QString(),
	               bool autoCancel = false,
	               bool sendRemainingSignals = true );

/* **************************************************************************
 *      SIGNALS:
 * **************************************************************************/
Q_SIGNALS:
	/** \brief New request has started to be processed.
	 *
	 * Sent when new request has started.
	 */
	void started  ();

	/** \brief Some request is done.
	 *
	 * Signals that some request has finished. and there is no other work
	 * related to it. There will not be any other signals connected with
	 * that request.\n
	 * Please note that if You made a couple of requests it does not mean
	 * there will not be ANY other signals. It just means that one of that
	 * requests is done - the other could be still in queue and there
	 * could be some signals for those other requests.\n
	 * If You want to know when all request are done You need to check
	 * \a left parameter. When it is equal 0 it means there is no other
	 * request processed anymore.
	 * \param left says how many other request are still in the queue
	 */
	void finished (int left);

	/** \brief New thumbnail is ready for You.
	 *
	 * Signals that there is new thumbnail ready and could be used by
	 * applications.\n
	 * Every resource You requested for will get individual thumbnail
	 * signal what differs from D-Bus API where some resources could be
	 * grouped and signaled together.
	 * \param uri describe resource for which thumbnail was requested
	 * \param thumbnailUri tells You what is the URI of the new thumbnail
	 * \param pixmap pixmap with new thumbnail (if that
	 *               information was requested; empty pixmap otherwise);
	 * \param flavor tells what type of thumbnail was requested
	 */
	void thumbnail        ( const QUrl& uri,
	                        const QUrl& thumbnailUri,
	                        const QPixmap& pixmap,
	                        const QString& flavor );

	/** \brief New thumbnail for media art is ready for You.
	 * 
	 * Check documentation for standard thumbnail() signal. This one is exactly
	 * the same one, but signals about thumbnail for media art.
	 * 
	 * \param media media art for which we got a thumbnail
	 * \param thumbnailUri tells You what is the URI of the new thumbnail
	 * \param pixmap pixmap with new thumbnail (if that
	 *               information was requested; empty pixmap otherwise);
	 * \param flavor tells what type of thumbnail was requested
	 * \see thumbnail()
	 */
	void thumbnail        ( const MediaArt::Info& media,
	                        const QUrl& thumbnailUri,
	                        const QPixmap& pixmap,
	                        const QString& flavor );

	/** \brief Thumbnail need to be (re)created.
	 *
	 * When You make a request for a thumbnail or resource which does not
	 * have thumbnail already it will take some time to create this
	 * (sometimes the thumbnail need to be recreated also - this is the
	 * same situation). You will get this signal in that case with some
	 * defaults values (which You can set by setDefaultPixmap(),
	 * setDefaultURI() and Thumbnailer()) which You can utilize to show
	 * some infromations to the user that he need to wait a little bit
	 * for real thumbnail.\n
	 * After defaultThumbnail signal You will receive thumbnail signal
	 * with some small delay (depends on current performance of the
	 * platform)\n \n
	 * 
	 * The definition of signal is the same as \see thumbnail() signal.
	 */
	void defaultThumbnail ( const QUrl& uri,
	                        const QUrl& thumbnailUri,
	                        const QPixmap& pixmap,
	                        const QString& flavor );

	/** \brief Media Art's thumbnail is not available yet.
	 *
	 * If media art content is not available at call time, or thumbnail
	 * of existing media art's content is not available then You will receive
	 * this signal. It means that after some time You will receive
	 * thumbnail() signal if it was possible to fetch and to thumbnail the
	 * media art's content or error() signal if media art's content is not
	 * available or there is some problem with it that prevents it to be
	 * thumbnailed.
	 * 
	 * Check definition of thumbnail() signal which is the same as this
	 * signal with only one difference - values of thumbnailUri and pixmap
	 * ar the default ones, not real.
	 */
	void defaultThumbnail ( const MediaArt::Info& media,
	                        const QUrl& thumbnailUri,
	                        const QPixmap& pixmap,
	                        const QString& flavor );

	/** \brief Some request was canceled and removed from queue.
	 *
	 * Signals that particular thumbnail request was dequeued (canceled).
	 * \param uri QUrl of resource for which thumbnailing was canceled
	 */
	void dequeued ( const QUrl& uri );

	/** \brief Some request was canceled and removed from queue.
	 *
	 * Signals that particular media art's thumbnail request was dequeued (canceled).
	 * \param media media art for which thumbnailing was canceled
	 */
	void dequeued ( const MediaArt::Info& media );

	/** \brief Some error occured during the request.
	 *
	 * Signals that there were some error while trying to create thumbnail
	 * of \a uri resource.\n
	 * Every resource You requested for and which failed will get individual
	 * error signal what differs from D-Bus API where some failed requests
	 * could be grouped and signaled together.
	 * \param error message describing the problem
	 * \param uri QUrl of resource for which thumbnailing failed
	 * \param defaultUri QUrl with default thumbnail url
	 * \param defaultPixmap pixmap with default thumbnail
	 */
	void error    ( const QString& error,
	                const QUrl& uri,
	                const QUrl& defaultUri = QUrl(),
	                const QPixmap& defaultPixmap = QPixmap() );

	/** \brief Some error occured during the media art's thumbnail  request.
	 *
	 * Signals that there were some error while trying to create thumbnail
	 * of \a media art.\n
	 * \param error message describing the problem
	 * \param media QUrl of resource for which thumbnailing failed
	 * \param defaultUri QUrl with default thumbnail url
	 * \param defaultPixmap pixmap with default thumbnail
	 * \see error()
	 */
	void error    ( const QString& error,
	                const MediaArt::Info& media,
	                const QUrl& defaultUri = QUrl(),
	                const QPixmap& defaultPixmap = QPixmap() );

/* PRIVATE AREA */
private:
	friend class Thumbnails::ThumbnailerPrivate;
	Thumbnails::ThumbnailerPrivate *priv;
#ifdef TESTING_CODE
// code only for unit testing
public:
	Thumbnails::ThumbnailerPrivate* getPrivateData() { return priv; }
#endif
};

} // end of the Thumbnails namespace


#endif
