/*
 * Copyright (C) 2009 Nokia Corporation.
 *
 * Contact: Marius Vollmer <marius.vollmer@nokia.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */
/*
 * dbus-services.cpp
 *
 *  Created on: Jan 12, 2010
 *     Authors: wiecheck
 */
#include "dbus-services.h"


/*            ------- STATIC DBUS SERVICES OBJECTS ------------             */

static QThreadStorage<ThumbnailerGenericProxy*>      _tumblerService;
static QThreadStorage<ThumbnailerGenericCacheProxy*> _cacheService;
static QThreadStorage<ComNokiaAlbumArtRequester*>    _mediaartService;

DBusServices::DBusServices() : QObject()
{
	// we are not connecting to DBus at the initialization time
	// connection to DBus will tak place while trying access any service
}

DBusServices::~DBusServices()
{
	_tumblerService.setLocalData(0);
	_cacheService.setLocalData(0);
	_mediaartService.setLocalData(0);
}

DBusServices*
DBusServices::instance()
{
	static DBusServices singleton;
	return &singleton;
}


/*            ------- STATIC DBUS SERVICES OBJECTS ------------             */
/*            -------       IMPLEMENTATION         ------------             */
static const char* _thumbnailer_service_g = "org.freedesktop.thumbnails.Thumbnailer1";
static const char* _thumbnailer_path_g    = "/org/freedesktop/thumbnails/Thumbnailer1";
static const char* _cache_service_g       = "org.freedesktop.thumbnails.Cache1";
static const char* _cache_path_g          = "/org/freedesktop/thumbnails/Cache1";
static const char* _mediaart_service_g    = "com.nokia.albumart";
static const char* _mediaart_path_g       = "/com/nokia/albumart/Requester";

ThumbnailerGenericCacheProxy*
DBusServices::cache()
{
	/* check if there is no cache service yet */
        if ( !_cacheService.hasLocalData())
	{
	        _cacheService.setLocalData(
		        new ThumbnailerGenericCacheProxy(
				QString(_cache_service_g),
				QString(_cache_path_g),
				QDBusConnection::sessionBus(),
				instance()
			)
		);	  
	}

	if(!_cacheService.localData()->isValid()) {
	        qDebug() << "No valid DBus Service (Tumbler-Cache)! Error:";
		qDebug() << _cacheService.localData()->lastError().message();
		qDebug() << "Relying on autostart";
	}

	// FIXME The reconnects were done to work around qdbus bugs/issues.
	// None should be necessary anymore
	// // send signal about (re)conneecting to the dbus service
	// Q_EMIT instance()->cacheServiceReconnected ();

	return _cacheService.localData();
}

ThumbnailerGenericProxy*
DBusServices::tumbler()
{
	/* check if there is no tumbler service yet or connection is lost */
        if ( !_tumblerService.hasLocalData())
	{
		_tumblerService.setLocalData(
			new ThumbnailerGenericProxy(
				QString(_thumbnailer_service_g),
				QString(_thumbnailer_path_g),
				QDBusConnection::sessionBus(),
				instance()
			)
		);
		reconnectTumblerService ();
	}

	if(!_tumblerService.localData()->isValid()) {
	        qDebug() << "No valid DBus Service (Tumbler)! Error:";
	        qDebug() << _tumblerService.localData()->lastError().message();
		qDebug() << "Relying on autostart";
	}

	return _tumblerService.localData();
}

void
DBusServices::reconnectTumblerService()
{
	QObject::connect( _tumblerService.localData(),
	                  SIGNAL(Ready(uint,const QStringList&)),
	                  instance(),
	                  SLOT(tumblerReadyHandler(uint, const QStringList&)) );
	QObject::connect( _tumblerService.localData(),
	                  SIGNAL(Finished(uint)),
	                  instance(),
	                  SLOT(tumblerFinishedHandler(uint)) );
	QObject::connect( _tumblerService.localData(),
	                  SIGNAL(Started(uint)),
	                  instance(),
	                  SLOT(tumblerStartedHandler(uint)) );
	QObject::connect( _tumblerService.localData(),
	                  SIGNAL(Error(uint,const QStringList&,int,const QString &)),
	                  instance(),
	                  SLOT(tumblerErrorHandler(uint,const QStringList&,int,const QString&)) );
	Q_EMIT instance()->tumblerServiceReconnected ();
}

ComNokiaAlbumArtRequester*
DBusServices::mediaart()
{
	/* check if there is no media art service yet or connection is lost */
        if ( !_mediaartService.hasLocalData())
	{
		_mediaartService.setLocalData(
			new ComNokiaAlbumArtRequester(
			           QString(_mediaart_service_g),
			           QString(_mediaart_path_g),
			           QDBusConnection::sessionBus(),
			           instance() )
		);
		reconnectMediaArtService ();
	}

	if(!_mediaartService.localData()->isValid()) {
	        qDebug() << "No valid DBus Service (MediaArt)! Error:";
	        qDebug() << _mediaartService.localData()->lastError().message();
		qDebug() << "Relying on autostart";
	}

	return _mediaartService.localData();
}

void
DBusServices::reconnectMediaArtService()
{
	QObject::connect( _mediaartService.localData(),
	                  SIGNAL(Ready(QString&,QString&,QString&,QString&)),
	                  instance(),
	                  SLOT(mediaartReadyHandler(QString&,QString&,QString&,QString&)) );
	QObject::connect( _mediaartService.localData(),
	                  SIGNAL(Finished(uint)),
	                  instance(),
	                  SLOT(mediaartFinishedhHandler(uint)) );
	QObject::connect( _mediaartService.localData(),
	                  SIGNAL(Started(uint)),
	                  instance(),
	                  SLOT(mediaartStartedHandler(uint)) );
	QObject::connect( _mediaartService.localData(),
	                  SIGNAL(Error(uint,int,QString&)),
	                  instance(),
	                  SLOT(mediaartErrorHandler(uint,int,QString&)) );

	Q_EMIT instance()->mediaartServiceReconnected ();
}
