#include "simple.h"
#include <QScrollArea>
#include <QFrame>
#include <QApplication>

using namespace Thumbnails;

#define DATA_PATH "/usr/share/thumbexamples/data/"
static QString defaultPixmapPath  = QString(DATA_PATH "default.png");
static QString dequeuedPixmapPath = QString(DATA_PATH "dequeued.png");
static QString errorPixmapPath    = QString(DATA_PATH "error.png");

int
main(int argc, char* argv[]) {
	qDebug() << "Simple thumbnailing example application";
	QApplication application( argc, argv );

	ApplicationWindow mw(argv[1]);
	mw.showMaximized();

	return application.exec();
}


ApplicationWindow::~ApplicationWindow() {
	if(NULL != thumb) {
		thumb->cancel(false);
		delete thumb;
	}
}

ApplicationWindow::ApplicationWindow(char* path) : QWidget()
{
	this->path = QString(path);
	if(this->path.isEmpty()) this->path = QString("/home/user/MyDocs/.images");

	description = new QLabel("This demo presents all signals from Thumbnailer;)", this);
	directory = new QLineEdit(this->path, this);
	action = new QPushButton("Start", this);
	stopAction = new QPushButton("Cancel", this);
	vbox = new QVBoxLayout();
	hbox = new QHBoxLayout();
	thumbs = new QGridLayout();
	progress = new  QProgressBar();
	thumb = NULL;
	progress->setMinimum(0);
	progress->setMaximum(100);
	progress->setValue(0);

	sendRemaining = new QCheckBox ("Send remainig signals (dequeued)?", this);
	sendRemaining->setChecked(true);

	this->setLayout(vbox);
	vbox->addWidget(description);
	vbox->addWidget(sendRemaining);

	hbox->addWidget(directory);
	hbox->addWidget(action);
	hbox->addWidget(stopAction);

	vbox->addLayout(hbox);

	QScrollArea* scrollArea = new QScrollArea(this);
	QWidget* widget = new QWidget(this);
	widget->setLayout(thumbs);
	widget->show();
	scrollArea->setWidget(widget);
	scrollArea->setWidgetResizable(true);

	vbox->addWidget(scrollArea);
	vbox->addWidget(progress);

	QObject::connect( action, SIGNAL(clicked()),
	                  this, SLOT(loadThumbnails()));
	QObject::connect( stopAction, SIGNAL(clicked()),
	                  this, SLOT(cancelRequest()));

	// load other pixmaps
	dequeuedPixmap.load(dequeuedPixmapPath);
	errorPixmap.load(errorPixmapPath);
}

void ApplicationWindow::cancelRequest() {
	if(NULL == thumb) return;
	qDebug() << "Canceling all requests...";

	bool send = sendRemaining->isChecked();
	thumb->cancel(send);
}

void ApplicationWindow::loadThumbnails() {
	qDebug() << "Running the request...";
	progress->setValue(0);

	qDebug() << "Path:" << path;
	QDir dir(path);
	QStringList filters; filters << "*.jpg" << "*.jpeg" << "*.JPG" << "*.JPEG" << "*.avi";
	dir.setNameFilters(filters);
	QStringList files = dir.entryList(QDir::Files | QDir::Readable);
	qDebug() << "Files: " << files;

	QList<QUrl> uris;
	QStringList mimes;

	foreach(QString file, files) {
		file = "file://" + path + "/" + file;
		uris << QUrl(file);
		if(file.endsWith(".avi")) mimes << "video/x-msvideo";
		else mimes << "image/jpeg";
	}
	progress->setMaximum(uris.size());

	if(NULL == thumb) {
		thumb = new Thumbnailer( QUrl(""),
	                                 QPixmap(defaultPixmapPath) );
		QObject::connect(thumb, SIGNAL(started()), this, SLOT(loadingStarted()));
		QObject::connect(thumb, SIGNAL(finished(int)), this, SLOT(loadingFinished(int)));
		QObject::connect(thumb, SIGNAL(thumbnail(QUrl,QUrl,QPixmap,QString)), this, SLOT(thumbnail(QUrl,QUrl,QPixmap)));
		QObject::connect(thumb, SIGNAL(defaultThumbnail(QUrl,QUrl,QPixmap,QString)), this, SLOT(defaultThumbnail(QUrl,QUrl,QPixmap)));
		QObject::connect(thumb, SIGNAL(error(QString,QUrl)), this, SLOT(error(QString,QUrl)));
		QObject::connect(thumb, SIGNAL(dequeued(QUrl)), this, SLOT(dequeued(QUrl)));
	}

	qDebug() << "thumb->request(uris, mimes, true)";
	bool done = false;
	done = thumb->request (uris, mimes, true); // true - send also a pixmap with the thumbnail
	if(done) {
		qDebug() << "All thumbnails were ready at the request time - no need for dbus traffic!";
	}
}

QLabel* ApplicationWindow::getImageArea(QUrl uri) {
	static const int colNumber = 5;
	static int currentColumn = 0;
	static int currentRow = 0;

	// check if that is already somewhere
	if(!widgets.contains(uri.toString())) {
		// that area does not exist yet so create this
		QLabel* label = new QLabel(this);
		label->setVisible(true);
		widgets.insert(uri.toString(), label);
		thumbs->addWidget(label, currentRow, currentColumn);

		++currentColumn;
		if(currentColumn >= colNumber) {
			currentColumn = 0;
			++currentRow;
		}
	}

	return widgets.value(uri.toString());
}

void ApplicationWindow::loadingStarted() {
	description->setText ("Requesting and proccessing thumbnails...");
}

void ApplicationWindow::loadingFinished(int left) {
	qDebug() << "There are other" << left << "request(s) left.";
	description->setText ("Request finished");
	progress->setValue(progress->maximum());
}

void ApplicationWindow::thumbnail(QUrl uri, QUrl thumb, QPixmap pixmap) {
	Q_UNUSED(thumb);
	getImageArea(uri)->setPixmap(pixmap);
	progress->setValue(progress->value() + 1);
	//qDebug() << "#";
}

void ApplicationWindow::defaultThumbnail(QUrl uri, QUrl thumb, QPixmap pixmap) {
	Q_UNUSED(thumb);
	getImageArea(uri)->setPixmap(pixmap);
	//qDebug() << ".";
}

void ApplicationWindow::dequeued (QUrl uri) {
	getImageArea(uri)->setPixmap(dequeuedPixmap);
}

void ApplicationWindow::error (QString error, QUrl uri) {
	qDebug() << "Error for file:" << uri.toString() << ", error: " << error;
	getImageArea(uri)->setPixmap(errorPixmap);
}
