/*
 * tangle-scroll-action.c
 *
 * This file is part of Tangle Toolkit - A graphical widget library based on Clutter Toolkit
 *
 * (c) 2010 Henrik Hedberg <henrik.hedberg@innologies.fi>
 *
 */

#include "tangle-scroll-action.h"
#include "tangle-vault.h"
#include "tangle-misc.h"

G_DEFINE_TYPE(TangleScrollAction, tangle_scroll_action, CLUTTER_TYPE_ACTION);

enum {
	PROP_0,
	PROP_OFFSET_X,
	PROP_OFFSET_Y,
	PROP_MAX_OFFSET_X,
	PROP_MAX_OFFSET_Y,
	PROP_PAGE_LENGTH_X,
	PROP_PAGE_LENGTH_Y,
	PROP_THRESHOLD_X,
	PROP_THRESHOLD_Y,
	PROP_MAX_OVERSHOOT_X,
	PROP_MAX_OVERSHOOT_Y,
	PROP_CONTINUOUS_X,
	PROP_CONTINUOUS_Y,
	PROP_KINETIC_SCROLLING,
	PROP_DECELERATION,
	PROP_BIND_TO_SCROLL_TRICK
};

enum {
	CLAMP_OFFSET_X,
	CLAMP_OFFSET_Y,
	LAST_SIGNAL
};

typedef enum {
	HORIZONTAL = 0,
	VERTICAL
} Axis;

struct _TangleScrollActionPrivate {
	gfloat offset[2];
	gfloat max_offset[2];
	gfloat page_length[2];
	gfloat threshold[2];
	gfloat max_overshoot[2];
	gboolean continuous[2];
	gdouble deceleration;

	gboolean scrolling_effective[2];

	gfloat start_motion[2];
	guint32 start_motion_time[2];
	gfloat farest_motion[2];
	guint32 farest_motion_time[2];
	
	guint32 previous_motion_time;

	gfloat start_scrolling_motion[2];
	gfloat start_offset[2];

	gboolean going_forward[2];

	gulong captured_event_handler_id;
	ClutterAnimation* animation[2];
	
	guint kinetic_scrolling : 1;
	guint bind_to_scroll_trick : 1;
};

static guint signals[LAST_SIGNAL] = { 0 };

static gboolean on_captured_event(ClutterActor* actor, ClutterEvent* event, gpointer user_data);
static gfloat normalise_offset(TangleScrollAction* scroll_action, Axis axis, gfloat offset, gfloat* delta_return);
static gfloat emit_clamp_offset(TangleScrollAction* scroll_action, Axis axis, gfloat offset);
static void stop_interacting(TangleScrollAction* scrol_actor);
static void on_notify_descendant_dragging(GObject* object, GParamSpec* param_spec, gpointer user_data);
static void on_notify_layout(GObject* object, GParamSpec* param_spec, gpointer user_data);
static gboolean on_button_press_event(ClutterActor* actor, ClutterButtonEvent* event, gpointer user_data);

ClutterAction* tangle_scroll_action_new() {

	return CLUTTER_ACTION(g_object_new(TANGLE_TYPE_SCROLL_ACTION, NULL));
}

void tangle_scroll_action_get_offsets(TangleScrollAction* scroll_action, gfloat* offset_x_return, gfloat* offset_y_return) {
	g_return_if_fail(TANGLE_IS_SCROLL_ACTION(scroll_action));

	if (offset_x_return) {
		*offset_x_return = scroll_action->priv->offset[HORIZONTAL];
	}
	if (offset_y_return) {
		*offset_y_return = scroll_action->priv->offset[VERTICAL];
	}
}

void tangle_scroll_action_set_offsets(TangleScrollAction* scroll_action, gfloat offset_x, gfloat offset_y) {
	g_return_if_fail(TANGLE_IS_SCROLL_ACTION(scroll_action));

	if (scroll_action->priv->offset[HORIZONTAL] != offset_x) {
		scroll_action->priv->offset[HORIZONTAL] = normalise_offset(scroll_action, HORIZONTAL, offset_x, NULL);
		g_object_notify(G_OBJECT(scroll_action), "offset-x");
	}
	if (scroll_action->priv->offset[VERTICAL] != offset_y) {
		scroll_action->priv->offset[VERTICAL] = normalise_offset(scroll_action, VERTICAL, offset_y, NULL);
		g_object_notify(G_OBJECT(scroll_action), "offset-y");
	}
}

void tangle_scroll_action_get_max_offsets(TangleScrollAction* scroll_action, gfloat* max_offset_x_return, gfloat* max_offset_y_return) {
	g_return_if_fail(TANGLE_IS_SCROLL_ACTION(scroll_action));

	if (max_offset_x_return) {
		*max_offset_x_return = scroll_action->priv->max_offset[HORIZONTAL];
	}
	if (max_offset_y_return) {
		*max_offset_y_return = scroll_action->priv->max_offset[VERTICAL];
	}
}

void tangle_scroll_action_set_max_offsets(TangleScrollAction* scroll_action, gfloat max_offset_x, gfloat max_offset_y) {
	gboolean changed = FALSE;
	
	g_return_if_fail(TANGLE_IS_SCROLL_ACTION(scroll_action));

	if (scroll_action->priv->max_offset[HORIZONTAL] != max_offset_x) {
		scroll_action->priv->max_offset[HORIZONTAL] =max_offset_x;
		g_object_notify(G_OBJECT(scroll_action), "max-offset-x");
		changed = TRUE;
	}
	if (scroll_action->priv->max_offset[VERTICAL] != max_offset_y) {
		scroll_action->priv->max_offset[VERTICAL] = max_offset_y;
		g_object_notify(G_OBJECT(scroll_action), "max-offset-y");
		changed = TRUE;
	}
}

void tangle_scroll_action_get_thresholds(TangleScrollAction* scroll_action, gfloat* threshold_x_return, gfloat* threshold_y_return) {
	g_return_if_fail(TANGLE_IS_SCROLL_ACTION(scroll_action));

	if (threshold_x_return) {
		*threshold_x_return = scroll_action->priv->threshold[HORIZONTAL];
	}
	if (threshold_y_return) {
		*threshold_y_return = scroll_action->priv->threshold[VERTICAL];
	}
}

void tangle_scroll_action_set_thresholds(TangleScrollAction* scroll_action, gfloat threshold_x, gfloat threshold_y) {
	g_return_if_fail(TANGLE_IS_SCROLL_ACTION(scroll_action));

	if (scroll_action->priv->threshold[HORIZONTAL] != threshold_x) {
		scroll_action->priv->threshold[HORIZONTAL] =threshold_x;
		g_object_notify(G_OBJECT(scroll_action), "threshold-x");
	}
	if (scroll_action->priv->threshold[VERTICAL] != threshold_y) {
		scroll_action->priv->threshold[VERTICAL] = threshold_y;
		g_object_notify(G_OBJECT(scroll_action), "threshold-y");
	}
}


void tangle_scroll_action_get_max_overshoots(TangleScrollAction* scroll_action, gfloat* max_overshoot_x_return, gfloat* max_overshoot_y_return) {
	g_return_if_fail(TANGLE_IS_SCROLL_ACTION(scroll_action));

	if (max_overshoot_x_return) {
		*max_overshoot_x_return = scroll_action->priv->max_overshoot[HORIZONTAL];
	}
	if (max_overshoot_y_return) {
		*max_overshoot_y_return = scroll_action->priv->max_overshoot[VERTICAL];
	}
}

void tangle_scroll_action_set_max_overshoots(TangleScrollAction* scroll_action, gfloat max_overshoot_x, gfloat max_overshoot_y) {
	g_return_if_fail(TANGLE_IS_SCROLL_ACTION(scroll_action));

	gboolean changed = FALSE;
	
	if (scroll_action->priv->max_overshoot[HORIZONTAL] != max_overshoot_x) {
		scroll_action->priv->max_overshoot[HORIZONTAL] =max_overshoot_x;
		g_object_notify(G_OBJECT(scroll_action), "max-overshoot-x");
		changed = TRUE;
	}
	if (scroll_action->priv->max_overshoot[VERTICAL] != max_overshoot_y) {
		scroll_action->priv->max_overshoot[VERTICAL] = max_overshoot_y;
		g_object_notify(G_OBJECT(scroll_action), "max-overshoot-y");
		changed = TRUE;
	}
}


void tangle_scroll_action_get_continuous(TangleScrollAction* scroll_action, gboolean* continuous_x_return, gboolean* continuous_y_return) {
	g_return_if_fail(TANGLE_IS_SCROLL_ACTION(scroll_action));

	if (continuous_x_return) {
		*continuous_x_return = scroll_action->priv->continuous[HORIZONTAL];
	}
	if (continuous_y_return) {
		*continuous_y_return = scroll_action->priv->continuous[VERTICAL];
	}
}

void tangle_scroll_action_set_continuous(TangleScrollAction* scroll_action, gboolean continuous_x, gboolean continuous_y) {
	g_return_if_fail(TANGLE_IS_SCROLL_ACTION(scroll_action));

	gboolean changed = FALSE;
	
	if (scroll_action->priv->continuous[HORIZONTAL] != continuous_x) {
		scroll_action->priv->continuous[HORIZONTAL] =continuous_x;
		g_object_notify(G_OBJECT(scroll_action), "continuous-x");
		changed = TRUE;
	}
	if (scroll_action->priv->continuous[VERTICAL] != continuous_y) {
		scroll_action->priv->continuous[VERTICAL] = continuous_y;
		g_object_notify(G_OBJECT(scroll_action), "continuous-y");
		changed = TRUE;
	}
}

gboolean tangle_scroll_action_get_kinetic_scrolling(TangleScrollAction* scroll_action) {
	g_return_val_if_fail(TANGLE_IS_SCROLL_ACTION(scroll_action), FALSE);

	return scroll_action->priv->kinetic_scrolling;
}

void tangle_scroll_action_set_kinetic_scrolling(TangleScrollAction* scroll_action, gboolean kinetic_scrolling) {
	g_return_if_fail(TANGLE_IS_SCROLL_ACTION(scroll_action));

	if (scroll_action->priv->kinetic_scrolling != kinetic_scrolling) {
		scroll_action->priv->kinetic_scrolling = kinetic_scrolling;
		g_object_notify(G_OBJECT(scroll_action), "kinetic-scrolling");
	}
}

gdouble tangle_scroll_action_get_deceleration(TangleScrollAction* scroll_action) {
	g_return_val_if_fail(TANGLE_IS_SCROLL_ACTION(scroll_action), 0.0);


	return scroll_action->priv->deceleration;
}

void tangle_scroll_action_set_deceleration(TangleScrollAction* scroll_action, gdouble deceleration) {
	g_return_if_fail(TANGLE_IS_SCROLL_ACTION(scroll_action));

	if (scroll_action->priv->deceleration != deceleration) {
		scroll_action->priv->deceleration = deceleration;
		g_object_notify(G_OBJECT(scroll_action), "deceleration");
	}
}

gboolean tangle_scroll_action_get_bind_to_scroll_trick(TangleScrollAction* scroll_action) {
	g_return_val_if_fail(TANGLE_IS_SCROLL_ACTION(scroll_action), FALSE);

	return scroll_action->priv->bind_to_scroll_trick;
}

void tangle_scroll_action_set_bind_to_scroll_trick(TangleScrollAction* scroll_action, gboolean bind_to_scroll_trick) {
	g_return_if_fail(TANGLE_IS_SCROLL_ACTION(scroll_action));

	if (scroll_action->priv->bind_to_scroll_trick != bind_to_scroll_trick) {
		scroll_action->priv->bind_to_scroll_trick = bind_to_scroll_trick;
		g_object_notify(G_OBJECT(scroll_action), "bind-to-scroll-trick");
	}
}

void tangle_scroll_action_clamp_offset_x(TangleScrollAction* scroll_action) {
	TangleClamp* clamp;
	gfloat offset;
	
	g_return_if_fail(TANGLE_IS_SCROLL_ACTION(scroll_action));

	if ((offset = emit_clamp_offset(scroll_action, HORIZONTAL, scroll_action->priv->offset[HORIZONTAL])) != scroll_action->priv->offset[HORIZONTAL]) {
		tangle_object_animate(G_OBJECT(scroll_action), CLUTTER_EASE_IN_OUT_QUAD, 300, "offset-x", offset, NULL);		
	}
}

void tangle_scroll_action_clamp_offset_y(TangleScrollAction* scroll_action) {
	TangleClamp* clamp;
	gfloat offset;
	
	g_return_if_fail(TANGLE_IS_SCROLL_ACTION(scroll_action));

	if ((offset = emit_clamp_offset(scroll_action, VERTICAL, scroll_action->priv->offset[VERTICAL])) != scroll_action->priv->offset[VERTICAL]) {
		tangle_object_animate(G_OBJECT(scroll_action), CLUTTER_EASE_IN_OUT_QUAD, 300, "offset-y", offset, NULL);		
	}
}

void tangle_scroll_action_clamp_offsets(TangleScrollAction* scroll_action) {
	g_return_if_fail(TANGLE_IS_SCROLL_ACTION(scroll_action));

	tangle_scroll_action_clamp_offset_x(scroll_action);
	tangle_scroll_action_clamp_offset_y(scroll_action);
}

void tangle_scroll_action_bind_to_scroll_trick(TangleScrollAction* scroll_action, TangleScrollTrick* scroll_trick) {
	g_return_if_fail(TANGLE_IS_SCROLL_ACTION(scroll_action));

	tangle_binding_new(G_OBJECT(scroll_action), "offset-x", G_OBJECT(scroll_trick), "offset-x");
	tangle_binding_new(G_OBJECT(scroll_action), "offset-y", G_OBJECT(scroll_trick), "offset-y");
	tangle_binding_new(G_OBJECT(scroll_action), "max-offset-x", G_OBJECT(scroll_trick), "max-offset-x");
	tangle_binding_new(G_OBJECT(scroll_action), "max-offset-y", G_OBJECT(scroll_trick), "max-offset-y");
	tangle_binding_new(G_OBJECT(scroll_action), "page-length-x", G_OBJECT(scroll_trick), "page-length-x");
	tangle_binding_new(G_OBJECT(scroll_action), "page-length-y", G_OBJECT(scroll_trick), "page-length-y");
}

static void tangle_scroll_action_set_actor(ClutterActorMeta* actor_meta, ClutterActor* actor) {
	TangleScrollAction* scroll_action;
	ClutterActor* old_actor;
	TangleLayout* layout;
	TangleTrick* trick;
	
	scroll_action = TANGLE_SCROLL_ACTION(actor_meta);
	
	if ((old_actor = clutter_actor_meta_get_actor(actor_meta))) {
		if (TANGLE_IS_WIDGET(old_actor)) {
			g_signal_handlers_disconnect_by_func(old_actor, G_CALLBACK(on_notify_descendant_dragging), scroll_action);
		}
		g_signal_handlers_disconnect_by_func(old_actor, G_CALLBACK(on_button_press_event), scroll_action);
	}

	if (actor) {
		if (TANGLE_IS_WIDGET(actor)) {
			g_signal_connect(actor, "notify::descendant-dragging", G_CALLBACK(on_notify_descendant_dragging), scroll_action);

			if (scroll_action->priv->bind_to_scroll_trick) {
				if ((layout = tangle_widget_get_layout(TANGLE_WIDGET(actor)))) {
					if ((trick = tangle_layout_get_trick_by_type(layout, TANGLE_TYPE_SCROLL_TRICK))) {
						tangle_scroll_action_bind_to_scroll_trick(scroll_action, TANGLE_SCROLL_TRICK(trick));
					}
				} else {
					g_signal_connect(actor, "notify::layout", G_CALLBACK(on_notify_layout), scroll_action);
				}
			}
		}
		g_signal_connect(actor, "button-press-event", G_CALLBACK(on_button_press_event), scroll_action);
	}
	
	CLUTTER_ACTOR_META_CLASS(tangle_scroll_action_parent_class)->set_actor(actor_meta, actor);
}

static void tangle_scroll_action_clamp_offset_x_impl(TangleScrollAction* scroll_action, TangleClamp* clamp) {
	if (!tangle_clamp_get_clamped_value_set(clamp) && !scroll_action->priv->continuous[HORIZONTAL]) {
		if (scroll_action->priv->offset[HORIZONTAL] < 0.0) {
			tangle_clamp_set_clamped_value(clamp, 0.0);
		} else if (scroll_action->priv->offset[HORIZONTAL] > scroll_action->priv->max_offset[HORIZONTAL]) {
			tangle_clamp_set_clamped_value(clamp, scroll_action->priv->max_offset[HORIZONTAL]);
		}
	}
}

static void tangle_scroll_action_clamp_offset_y_impl(TangleScrollAction* scroll_action, TangleClamp* clamp) {
	if (!tangle_clamp_get_clamped_value_set(clamp) && !scroll_action->priv->continuous[VERTICAL]) {
		if (scroll_action->priv->offset[VERTICAL] < 0.0) {
			tangle_clamp_set_clamped_value(clamp, 0.0);
		} else if (scroll_action->priv->offset[VERTICAL] > scroll_action->priv->max_offset[VERTICAL]) {
			tangle_clamp_set_clamped_value(clamp, scroll_action->priv->max_offset[VERTICAL]);
		}
	}
}

static void tangle_scroll_action_set_property(GObject* object, guint prop_id, const GValue* value, GParamSpec* pspec) {
	TangleScrollAction* scroll_action;
	
	scroll_action = TANGLE_SCROLL_ACTION(object);

	switch (prop_id) {
		case PROP_OFFSET_X:
			tangle_scroll_action_set_offsets(scroll_action, g_value_get_float(value), scroll_action->priv->offset[VERTICAL]);
			break;
		case PROP_OFFSET_Y:
			tangle_scroll_action_set_offsets(scroll_action, scroll_action->priv->offset[HORIZONTAL], g_value_get_float(value));
			break;
		case PROP_MAX_OFFSET_X:
			tangle_scroll_action_set_max_offsets(scroll_action, g_value_get_float(value), scroll_action->priv->max_offset[VERTICAL]);
			break;
		case PROP_MAX_OFFSET_Y:
			tangle_scroll_action_set_max_offsets(scroll_action, scroll_action->priv->max_offset[HORIZONTAL], g_value_get_float(value));
			break;
		case PROP_PAGE_LENGTH_X:
			scroll_action->priv->page_length[HORIZONTAL] = g_value_get_float(value);
			break;
		case PROP_PAGE_LENGTH_Y:
			scroll_action->priv->page_length[VERTICAL] = g_value_get_float(value);
			break;
		case PROP_THRESHOLD_X:
			tangle_scroll_action_set_thresholds(scroll_action, g_value_get_float(value), scroll_action->priv->threshold[VERTICAL]);
			break;
		case PROP_THRESHOLD_Y:
			tangle_scroll_action_set_thresholds(scroll_action, scroll_action->priv->threshold[HORIZONTAL], g_value_get_float(value));
			break;
		case PROP_MAX_OVERSHOOT_X:
			tangle_scroll_action_set_max_overshoots(scroll_action, g_value_get_float(value), scroll_action->priv->max_overshoot[VERTICAL]);
			break;
		case PROP_MAX_OVERSHOOT_Y:
			tangle_scroll_action_set_max_overshoots(scroll_action, scroll_action->priv->max_overshoot[HORIZONTAL], g_value_get_float(value));
			break;
		case PROP_CONTINUOUS_X:
			tangle_scroll_action_set_continuous(scroll_action, g_value_get_boolean(value), scroll_action->priv->continuous[VERTICAL]);
			break;
		case PROP_CONTINUOUS_Y:
			tangle_scroll_action_set_continuous(scroll_action, scroll_action->priv->continuous[HORIZONTAL], g_value_get_boolean(value));
			break;
		case PROP_KINETIC_SCROLLING:
			tangle_scroll_action_set_kinetic_scrolling(scroll_action, g_value_get_boolean(value));
			break;
		case PROP_DECELERATION:
			tangle_scroll_action_set_deceleration(scroll_action, g_value_get_double(value));
			break;
		case PROP_BIND_TO_SCROLL_TRICK:
			tangle_scroll_action_set_bind_to_scroll_trick(scroll_action, g_value_get_boolean(value));
			break;
		default:
			G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
			break;
	}
}

static void tangle_scroll_action_get_property(GObject* object, guint prop_id, GValue* value, GParamSpec* pspec) {
        TangleScrollAction* scroll_action;

	scroll_action = TANGLE_SCROLL_ACTION(object);

        switch (prop_id) {
		case PROP_OFFSET_X:
			g_value_set_float(value, scroll_action->priv->offset[HORIZONTAL]);
			break;
		case PROP_OFFSET_Y:
			g_value_set_float(value, scroll_action->priv->offset[VERTICAL]);
			break;
		case PROP_MAX_OFFSET_X:
			g_value_set_float(value, scroll_action->priv->max_offset[HORIZONTAL]);
			break;
		case PROP_MAX_OFFSET_Y:
			g_value_set_float(value, scroll_action->priv->max_offset[VERTICAL]);
			break;
		case PROP_PAGE_LENGTH_X:
			g_value_set_float(value, scroll_action->priv->page_length[HORIZONTAL]);
			break;
		case PROP_PAGE_LENGTH_Y:
			g_value_set_float(value, scroll_action->priv->page_length[VERTICAL]);
			break;
		case PROP_THRESHOLD_X:
			g_value_set_float(value, scroll_action->priv->threshold[HORIZONTAL]);
			break;
		case PROP_THRESHOLD_Y:
			g_value_set_float(value, scroll_action->priv->threshold[VERTICAL]);
			break;
		case PROP_MAX_OVERSHOOT_X:
			g_value_set_float(value, scroll_action->priv->max_overshoot[HORIZONTAL]);
			break;
		case PROP_MAX_OVERSHOOT_Y:
			g_value_set_float(value, scroll_action->priv->max_overshoot[VERTICAL]);
			break;
		case PROP_CONTINUOUS_X:
			g_value_set_float(value, scroll_action->priv->continuous[HORIZONTAL]);
			break;
		case PROP_CONTINUOUS_Y:
			g_value_set_float(value, scroll_action->priv->continuous[VERTICAL]);
			break;
		case PROP_KINETIC_SCROLLING:
			g_value_set_float(value, scroll_action->priv->kinetic_scrolling);
			break;
		case PROP_DECELERATION:
			g_value_set_float(value, scroll_action->priv->deceleration);
			break;
		case PROP_BIND_TO_SCROLL_TRICK:
			g_value_set_boolean(value, scroll_action->priv->bind_to_scroll_trick);
			break;
	        default:
		        G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
		        break;
        }
}

static void tangle_scroll_action_finalize(GObject* object) {
	G_OBJECT_CLASS(tangle_scroll_action_parent_class)->finalize(object);
}

static void tangle_scroll_action_dispose(GObject* object) {
	G_OBJECT_CLASS(tangle_scroll_action_parent_class)->dispose(object);
}

static void tangle_scroll_action_class_init(TangleScrollActionClass* scroll_action_class) {
	GObjectClass* gobject_class = G_OBJECT_CLASS(scroll_action_class);
	ClutterActorMetaClass* actor_meta_class = CLUTTER_ACTOR_META_CLASS(scroll_action_class);

	gobject_class->finalize = tangle_scroll_action_finalize;
	gobject_class->dispose = tangle_scroll_action_dispose;
	gobject_class->set_property = tangle_scroll_action_set_property;
	gobject_class->get_property = tangle_scroll_action_get_property;

	actor_meta_class->set_actor = tangle_scroll_action_set_actor;

	scroll_action_class->clamp_offset_x = tangle_scroll_action_clamp_offset_x_impl;
	scroll_action_class->clamp_offset_y = tangle_scroll_action_clamp_offset_y_impl;

	/**
	 * TangleScrollAction:offset-x:
	 *
	 * The offset to the left side of the bounding box of the scrolled layout.
	 */
	g_object_class_install_property(gobject_class, PROP_OFFSET_X,
	                                g_param_spec_float("offset-x",
	                                                   "Offset X",
	                                                   "The offset to the left side of the bounding box of the scrolled layout",
	                                                   -G_MAXFLOAT, G_MAXFLOAT, 0.0,
	                                                   G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB));
	/**
	 * TangleScrollAction:offset-y:
	 *
	 * The offset to the top side of the bounding box of the scrolled layout.
	 */
	g_object_class_install_property(gobject_class, PROP_OFFSET_Y,
	                                g_param_spec_float("offset-y",
	                                                   "Offset Y",
	                                                   "The offset to the top side of the bounding box of the scrolled layout",
	                                                   -G_MAXFLOAT, G_MAXFLOAT, 0.0,
	                                                   G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB));
	/**
	 * TangleScrollAction:max-offset-x:
	 *
	 * The maximum value for the :offset-x property.
	 */
	g_object_class_install_property(gobject_class, PROP_MAX_OFFSET_X,
	                                g_param_spec_float("max-offset-x",
	                                                   "Max offset X",
	                                                   "The maximum value for the offset-y property",
	                                                   -G_MAXFLOAT, G_MAXFLOAT, 0.0,
	                                                   G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB));
	/**
	 * TangleScrollAction:max-offset-y:
	 *
	 * The maximum value for the :offset-y property.
	 */
	g_object_class_install_property(gobject_class, PROP_MAX_OFFSET_Y,
	                                g_param_spec_float("max-offset-y",
	                                                   "Max offset Y",
	                                                   "The maximum value for the offset-y property",
	                                                   -G_MAXFLOAT, G_MAXFLOAT, 0.0,
	                                                   G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB));
	/**
	 * TangleScrollTrick:page-length-x:
	 *
	 * The width of the visible area
	 */
	g_object_class_install_property(gobject_class, PROP_PAGE_LENGTH_X,
	                                g_param_spec_float("page-length-x",
	                                "Page Length X",
	                                "The width of the visible area",
	                                0, G_MAXFLOAT, 0,
	                                G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB));
	/**
	 * TangleScrollTrick:page-length-y:
	 *
	 * The height of the visible area
	 */
	g_object_class_install_property(gobject_class, PROP_PAGE_LENGTH_Y,
	                                g_param_spec_float("page-length-y",
	                                "Page Length Y",
	                                "The height of the visible area",
	                                0, G_MAXFLOAT, 0,
	                                G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB));
	/**
	 * TangleScrollAction:threshold-x:
	 *
	 * The horizontal threshold before start scrolling.
	 */
	g_object_class_install_property(gobject_class, PROP_THRESHOLD_X,
	                                g_param_spec_float("threshold-x",
	                                                   "Threshold X",
	                                                   "The horizontal threshold before start scrolling",
	                                                   -G_MAXFLOAT, G_MAXFLOAT, 0.0,
	                                                   G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB));
	/**
	 * TangleScrollAction:threshold-y:
	 *
	 * The vertical threshold before start scrolling.
	 */
	g_object_class_install_property(gobject_class, PROP_THRESHOLD_Y,
	                                g_param_spec_float("threshold-y",
	                                                   "Threshold Y",
	                                                   "The vertical threshold before start scrolling",
	                                                   -G_MAXFLOAT, G_MAXFLOAT, 0.0,
	                                                   G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB));
	/**
	 * TangleScrollAction:max-overshoot-x:
	 *
	 * The horizontal maximum of overshooting.
	 */
	g_object_class_install_property(gobject_class, PROP_MAX_OVERSHOOT_X,
	                                g_param_spec_float("max-overshoot-x",
	                                                   "Max overshoot X",
	                                                   "The horizontal maximum of overshooting",
	                                                   0.0, G_MAXFLOAT, 0.0,
	                                                   G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB));
	/**
	 * TangleScrollAction:max-overshoot-y:
	 *
	 * The vertical maximum of overshooting.
	 */
	g_object_class_install_property(gobject_class, PROP_MAX_OVERSHOOT_Y,
	                                g_param_spec_float("max-overshoot-y",
	                                                   "Max overshoot Y",
	                                                   "The vertical maximum of overshooting",
	                                                   0.0, G_MAXFLOAT, 0.0,
	                                                   G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB));
	/**
	 * TangleScrollAction:continuous-x:
	 *
	 * Whether to connect the left and right edges of the scrolled layout to enable infinite horizontal scrolling.
	 * Setting this to TRUE sets also :repeat-x to TRUE as a side effect.
	 */
	g_object_class_install_property(gobject_class, PROP_CONTINUOUS_X,
	                                g_param_spec_boolean("continuous-x",
	                                                    "Continuous X",
	                                                    "Whether to connect the left and right edges of the wrapped actor to enable infinite horizontal scrolling",
	                                                    FALSE,
	                                                    G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB));
	/**
	 * TangleScrollAction:continuous-y:
	 *
	 * Whether to connect the top and bottom edges of the scrolled layout to enable infinite vertical scrolling.
	 * Setting this to TRUE sets also :repeat-y to TRUE as a side effect.
	 */
	g_object_class_install_property(gobject_class, PROP_CONTINUOUS_Y,
	                                g_param_spec_boolean("continuous-y",
	                                                     "Continuous Y",
	                                                     "Whether to connect the top and bottom edges of the wrapped actor to enable infinite vertical scrolling",
	                                                     FALSE,
	                                                     G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB));
	/**
	 * TangleScrollAction:kinetic-scrolling:
	 *
	 * Whether kinetic scrolling is enabled or not.
	 */
	g_object_class_install_property(gobject_class, PROP_KINETIC_SCROLLING,
	                                g_param_spec_boolean("kinetic-scrolling",
	                                                     "Kinetic scrolling",
	                                                     "Whether kinetic scrolling is enabled or not",
	                                                     TRUE,
	                                                     G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB));
	/**
	 * TangleScrollAction:deceleration:
	 *
	 * Deceleration of kinetic scrolling.
	 */
	g_object_class_install_property(gobject_class, PROP_DECELERATION,
	                                g_param_spec_double("deceleration",
	                                                    "Deceleration",
	                                                    "Deceleration of kinetic scrolling",
	                                                    0, G_MAXDOUBLE, 0.7,
	                                                    G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB));
	/**
	 * TangleScrollAction:deceleration:
	 *
	 * Whether this action is automatically binded to an existing #TangleScrollTrick (if exists) when added into a #TangleWidget.
	 */
	g_object_class_install_property(gobject_class, PROP_BIND_TO_SCROLL_TRICK,
	                                g_param_spec_boolean("bind-to-scroll-trick",
	                                                     "Bind to scroll trick",
	                                                     "Whether this action is automatically binded to an existing scroll trick",
	                                                     FALSE,
	                                                     G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB));

	signals[CLAMP_OFFSET_X] = g_signal_new("clamp-offset-x",
	                                       G_TYPE_FROM_CLASS(gobject_class),
					       G_SIGNAL_RUN_LAST,
					       G_STRUCT_OFFSET(TangleScrollActionClass, clamp_offset_x),
					       NULL, NULL,
					       g_cclosure_marshal_VOID__OBJECT,
					       G_TYPE_NONE, 1,
					       TANGLE_TYPE_CLAMP);
	signals[CLAMP_OFFSET_Y] = g_signal_new("clamp-offset-y",
	                                       G_TYPE_FROM_CLASS(gobject_class),
					       G_SIGNAL_RUN_LAST,
					       G_STRUCT_OFFSET(TangleScrollActionClass, clamp_offset_y),
					       NULL, NULL,
					       g_cclosure_marshal_VOID__OBJECT,
					       G_TYPE_NONE, 1,
					       TANGLE_TYPE_CLAMP);

	g_type_class_add_private (gobject_class, sizeof (TangleScrollActionPrivate));
}

static void tangle_scroll_action_init(TangleScrollAction* scroll_action) {
	scroll_action->priv = G_TYPE_INSTANCE_GET_PRIVATE(scroll_action, TANGLE_TYPE_SCROLL_ACTION, TangleScrollActionPrivate);
	scroll_action->priv->kinetic_scrolling = TRUE;
	scroll_action->priv->deceleration = 0.7;
}

static void on_overshooting_animation_completed(ClutterAnimation* animation, gpointer user_data) {
	TangleVault* vault;
	TangleScrollAction* scroll_action;
	const gchar* property;
	gfloat value;
	
	vault = TANGLE_VAULT(user_data);
	tangle_vault_get(vault, 3, TANGLE_TYPE_SCROLL_ACTION, &scroll_action, G_TYPE_STRING, &property, G_TYPE_FLOAT, &value);
	
	tangle_object_animate(G_OBJECT(scroll_action), CLUTTER_EASE_IN_OUT_QUAD, 350, property, value, NULL);
}

static void handle_overshooting_animation(TangleScrollAction* scroll_action, ClutterAnimation* animation, const gchar* property, gfloat value) {
	TangleVault* vault;

	vault = tangle_vault_new(3, TANGLE_TYPE_SCROLL_ACTION, scroll_action, G_TYPE_STRING, property, G_TYPE_FLOAT, value);
	tangle_signal_connect_vault_flags(animation, "completed", G_CALLBACK(on_overshooting_animation_completed), vault, G_CONNECT_AFTER);
}

#define ABS_F(x) ((x) < 0 ? -(x) : (x))

static void handle_event(TangleScrollAction* scroll_action, ClutterActor* actor, Axis axis, gfloat coordinate, gboolean release, guint event_time) {
	ClutterBackend* backend;
	gfloat d;
	gdouble v, a;
	guint t;
	gfloat delta = 0.0;
	gfloat offset, clamped_offset;

	backend = clutter_get_default_backend();
	d = coordinate - scroll_action->priv->start_scrolling_motion[axis];

	if (scroll_action->priv->scrolling_effective[axis] ||
	    (!(TANGLE_IS_WIDGET(actor) && tangle_widget_get_descendant_dragging(TANGLE_WIDGET(actor))) &&
	     (ABS_F(d) > scroll_action->priv->threshold[axis]))) {

		if (!scroll_action->priv->scrolling_effective[axis] ||
		    event_time - scroll_action->priv->previous_motion_time > clutter_backend_get_double_click_time(backend)) {
			if (!scroll_action->priv->scrolling_effective[axis]) {
				
				if (TANGLE_IS_ACTOR(actor)) {
					tangle_actor_set_interacting(TANGLE_ACTOR(actor), TRUE);
					tangle_actor_set_dragging(TANGLE_ACTOR(actor), TRUE);
				}

				scroll_action->priv->scrolling_effective[axis] = TRUE;
				scroll_action->priv->start_scrolling_motion[axis] = coordinate;
				scroll_action->priv->start_offset[axis] = scroll_action->priv->offset[axis];
				d = 0;
			}
			scroll_action->priv->going_forward[axis] = scroll_action->priv->start_motion[axis] < coordinate;
			scroll_action->priv->start_motion[axis] = scroll_action->priv->farest_motion[axis] = coordinate;
			scroll_action->priv->start_motion_time[axis] = scroll_action->priv->farest_motion_time[axis] = event_time;

		} else {
			if ((scroll_action->priv->going_forward[axis] && coordinate > scroll_action->priv->farest_motion[axis]) ||
			    (!scroll_action->priv->going_forward[axis] && coordinate < scroll_action->priv->farest_motion[axis])) {
				scroll_action->priv->farest_motion[axis] = coordinate;
				scroll_action->priv->farest_motion_time[axis] = event_time;
			} else if (ABS_F(coordinate - scroll_action->priv->farest_motion[axis]) > clutter_backend_get_double_click_distance(backend)) {				
				scroll_action->priv->start_motion[axis] = scroll_action->priv->farest_motion[axis];
				scroll_action->priv->start_motion_time[axis] = scroll_action->priv->farest_motion_time[axis];
				scroll_action->priv->farest_motion[axis] = coordinate;
				scroll_action->priv->farest_motion_time[axis] = event_time;
				scroll_action->priv->going_forward[axis] = scroll_action->priv->start_motion[axis] < coordinate;
			}
		}

		if (axis == HORIZONTAL) {
			tangle_scroll_action_set_offsets(scroll_action, scroll_action->priv->start_offset[axis] - d, scroll_action->priv->offset[VERTICAL]);
		} else {
			tangle_scroll_action_set_offsets(scroll_action, scroll_action->priv->offset[HORIZONTAL], scroll_action->priv->start_offset[axis] - d);
		}
		
		if (release && scroll_action->priv->kinetic_scrolling &&
		    event_time - scroll_action->priv->previous_motion_time < clutter_backend_get_double_click_time(backend)) {
			if (event_time - scroll_action->priv->start_motion_time[axis] > 0) {
				v = (gdouble)(coordinate - scroll_action->priv->start_motion[axis]) / (event_time - scroll_action->priv->start_motion_time[axis]);
				offset = normalise_offset(scroll_action, axis, scroll_action->priv->offset[axis] + (scroll_action->priv->going_forward[axis] ? -1 : 1) * v * v * 1000 / scroll_action->priv->deceleration / 2, &delta);
				if ((clamped_offset = emit_clamp_offset(scroll_action, axis, offset)) != offset) {
					a = v * v * 1000 / ABS_F(clamped_offset - scroll_action->priv->offset[axis]) / 2;
				} else {
					a = scroll_action->priv->deceleration;
				}

				t = ABS_F(v * 1000.0 / a);

				if (t > 0) {
					d = (scroll_action->priv->going_forward[axis] ? -1 : 1) * v * v * 1000 / a / 2;
					if (scroll_action->priv->continuous[axis]) {
						scroll_action->priv->animation[axis] = tangle_object_animate(G_OBJECT(scroll_action), CLUTTER_EASE_OUT_QUAD, t,
													           (axis == HORIZONTAL ? "offset-x" : "offset-y"), scroll_action->priv->offset[axis] + d,
													           NULL);
						g_object_add_weak_pointer(G_OBJECT(scroll_action->priv->animation[axis]), (gpointer*)&scroll_action->priv->animation[axis]);
						g_signal_connect_swapped(scroll_action->priv->animation[axis], "completed", G_CALLBACK((axis == HORIZONTAL ? tangle_scroll_action_clamp_offset_x : tangle_scroll_action_clamp_offset_y)), scroll_action);
					} else {
						if (scroll_action->priv->offset[axis] + d < -scroll_action->priv->max_overshoot[axis]) {
							t *= -(scroll_action->priv->offset[axis] + scroll_action->priv->max_overshoot[axis]) / d;
							d = -(scroll_action->priv->offset[axis] + scroll_action->priv->max_overshoot[axis]);
						} else if (scroll_action->priv->offset[axis] + d > scroll_action->priv->max_offset[axis] + scroll_action->priv->max_overshoot[axis]) {
							t *= (scroll_action->priv->max_offset[axis] - scroll_action->priv->offset[axis] + scroll_action->priv->max_overshoot[axis]) / d;
							d = (scroll_action->priv->max_offset[axis] - scroll_action->priv->offset[axis] + scroll_action->priv->max_overshoot[axis]);					
						}

						if (t > 0) {
							scroll_action->priv->animation[axis] = tangle_object_animate(G_OBJECT(scroll_action), CLUTTER_EASE_OUT_QUAD, t,
														  (axis == HORIZONTAL ? "offset-x" : "offset-y"), scroll_action->priv->offset[axis] + d,
														  NULL);
							g_object_add_weak_pointer(G_OBJECT(scroll_action->priv->animation[axis]), (gpointer*)&scroll_action->priv->animation[axis]);
							g_signal_connect_swapped(scroll_action->priv->animation[axis], "completed", G_CALLBACK((axis == HORIZONTAL ? tangle_scroll_action_clamp_offset_x : tangle_scroll_action_clamp_offset_y)), scroll_action);
							if (scroll_action->priv->offset[axis] + d < 0.0) {
								handle_overshooting_animation(scroll_action, scroll_action->priv->animation[axis], (axis == HORIZONTAL ? "offset-x" : "offset-y"), 0.0);
							} else if (scroll_action->priv->offset[axis] + d > scroll_action->priv->max_offset[axis]) {
								handle_overshooting_animation(scroll_action, scroll_action->priv->animation[axis], (axis == HORIZONTAL ? "offset-x" : "offset-y"), scroll_action->priv->max_offset[axis]);
							}					
						} else if (axis == HORIZONTAL) {
							tangle_scroll_action_clamp_offset_x(scroll_action);
						} else {
							tangle_scroll_action_clamp_offset_y(scroll_action);
						}
					}
				} else if (axis == HORIZONTAL) {
					tangle_scroll_action_clamp_offset_x(scroll_action);
				} else {
					tangle_scroll_action_clamp_offset_y(scroll_action);
				}
			}
		} else if (release) {
			tangle_scroll_action_clamp_offsets(scroll_action);
		}

		scroll_action->priv->previous_motion_time = event_time;
	} else if (release) {
		if (axis == HORIZONTAL) {
			tangle_scroll_action_clamp_offset_x(scroll_action);
		} else {
			tangle_scroll_action_clamp_offset_y(scroll_action);
		}	
	}
}

static gboolean on_captured_event(ClutterActor* stage, ClutterEvent* event, gpointer user_data) {
	TangleScrollAction* scroll_action;
	gfloat x, y;
	ClutterActor* actor;

	if (event->type == CLUTTER_MOTION || event->type == CLUTTER_BUTTON_RELEASE) {
		scroll_action = TANGLE_SCROLL_ACTION(user_data);	
		
		g_object_ref(scroll_action);
		
		clutter_event_get_coords(event, &x, &y);
		actor = clutter_actor_meta_get_actor(CLUTTER_ACTOR_META(scroll_action));
		
		handle_event(scroll_action, actor, HORIZONTAL, x, event->type == CLUTTER_BUTTON_RELEASE, clutter_event_get_time(event));
		handle_event(scroll_action, actor, VERTICAL, y, event->type == CLUTTER_BUTTON_RELEASE, clutter_event_get_time(event));

		if (event->type == CLUTTER_BUTTON_RELEASE) {
			stop_interacting(scroll_action);
		}

		g_object_unref(scroll_action);
	}

	return FALSE;
}

static gfloat normalise_offset(TangleScrollAction* scroll_action, Axis axis, gfloat offset, gfloat* delta_return) {
	guint multiplier;
	gfloat length;
	
	if (scroll_action->priv->continuous[axis]) {
		if (offset < -scroll_action->priv->max_offset[axis]) {
			length = scroll_action->priv->max_offset[axis] + scroll_action->priv->page_length[axis];
			multiplier = (guint)(-offset / length) + 1;
			offset += multiplier * length;
			if (delta_return) {
				*delta_return = -multiplier * length;
			}
		} else if (offset > 2 * scroll_action->priv->max_offset[axis]) {
			length = scroll_action->priv->max_offset[axis] + scroll_action->priv->page_length[axis];
			multiplier = (guint)(offset / length);
			offset -= multiplier * length;
			if (delta_return) {
				*delta_return = multiplier * length;
			}
		}
	} else {
		if (offset < -scroll_action->priv->max_overshoot[axis]) {
			if (delta_return) {
				*delta_return = -scroll_action->priv->max_overshoot[axis] - offset;
			}
			offset = -scroll_action->priv->max_overshoot[axis];
		} else if (offset > scroll_action->priv->max_offset[axis] + scroll_action->priv->max_overshoot[axis]) {
			if (delta_return) {
				*delta_return = scroll_action->priv->max_offset[axis] + scroll_action->priv->max_overshoot[axis] - offset;
			}
			offset = scroll_action->priv->max_offset[axis] + scroll_action->priv->max_overshoot[axis];
		}
	}
	
	return offset;
}

static gfloat emit_clamp_offset(TangleScrollAction* scroll_action, Axis axis, gfloat offset) {
	TangleClamp* clamp;
	gfloat offset_adjustment = 0.0;
	
	if (offset < 0.0) {
		offset_adjustment = scroll_action->priv->max_offset[axis];
		offset += offset_adjustment;
	} else if (offset > scroll_action->priv->max_offset[axis]) {
		offset_adjustment = -scroll_action->priv->max_offset[axis];
		offset += offset_adjustment;
	}

	clamp = tangle_clamp_new((axis == HORIZONTAL ? TANGLE_CLAMP_HORIZONTAL : TANGLE_CLAMP_VERTICAL), offset);
	g_signal_emit(scroll_action, signals[(axis == HORIZONTAL ? CLAMP_OFFSET_X : CLAMP_OFFSET_Y)], 0, clamp);
	if (tangle_clamp_get_clamped_value_set(clamp)) {
		offset = tangle_clamp_get_clamped_value(clamp);
	}
	g_object_unref(clamp);

	offset -= offset_adjustment;

	return offset;
}

static void stop_interacting(TangleScrollAction* scroll_action) {
	ClutterActor* actor;
	
	actor = clutter_actor_meta_get_actor(CLUTTER_ACTOR_META(scroll_action));
	if (scroll_action->priv->captured_event_handler_id) {
		g_signal_handler_disconnect(clutter_actor_get_stage(actor), scroll_action->priv->captured_event_handler_id);
		scroll_action->priv->captured_event_handler_id = 0;
		if (TANGLE_IS_ACTOR(actor)) {
			tangle_actor_set_interacting(TANGLE_ACTOR(actor), FALSE);
			tangle_actor_set_dragging(TANGLE_ACTOR(actor), FALSE);
		}
	}
}

static void on_notify_descendant_dragging(GObject* object, GParamSpec* param_spec, gpointer user_data) {
	if (tangle_widget_get_descendant_dragging(TANGLE_WIDGET(object))) {
		stop_interacting(TANGLE_SCROLL_ACTION(user_data));
	}
}

static void on_notify_layout(GObject* object, GParamSpec* param_spec, gpointer user_data) {
	TangleScrollAction* scroll_action;
	TangleLayout* layout;
	TangleTrick* trick;
		
	scroll_action = TANGLE_SCROLL_ACTION(user_data);

	if ((layout = tangle_widget_get_layout(TANGLE_WIDGET(object))) && (trick = tangle_layout_get_trick_by_type(layout, TANGLE_TYPE_SCROLL_TRICK))) {
		tangle_scroll_action_bind_to_scroll_trick(scroll_action, TANGLE_SCROLL_TRICK(trick));
		
		g_signal_handlers_disconnect_by_func(object, G_CALLBACK(on_notify_layout), user_data);
	}
}

static gboolean on_button_press_event(ClutterActor* actor, ClutterButtonEvent* event, gpointer user_data) {
	TangleScrollAction* scroll_action;
		
	scroll_action = TANGLE_SCROLL_ACTION(user_data);

	if (!scroll_action->priv->captured_event_handler_id) {
		if (scroll_action->priv->animation[HORIZONTAL]) {
			tangle_object_stop_animation(G_OBJECT(scroll_action), scroll_action->priv->animation[HORIZONTAL]);
		}
		if (scroll_action->priv->animation[VERTICAL]) {
			tangle_object_stop_animation(G_OBJECT(scroll_action), scroll_action->priv->animation[VERTICAL]);
		}

		scroll_action->priv->start_scrolling_motion[HORIZONTAL] = event->x;
		scroll_action->priv->start_scrolling_motion[VERTICAL] = event->y;
		scroll_action->priv->scrolling_effective[HORIZONTAL] = scroll_action->priv->scrolling_effective[VERTICAL] = FALSE;
		scroll_action->priv->captured_event_handler_id = g_signal_connect(clutter_actor_get_stage(actor), "captured-event", G_CALLBACK(on_captured_event), scroll_action);
	}

	return FALSE;
}
